tPathFunc = {}

-- ****************************************************************
// tPathMap
-- ****************************************************************

-- table of node points for environmental navigation
tPathFunc.tPathMap = {}

-- table of visual node point objects
tPathFunc.tPathObjects = {}

//-----------------------------------------------------------------
// GetNumRows
//-----------------------------------------------------------------
function tPathFunc.GetNumRows() return(table.getn(PetAI.tPathFunc.tPathMap)) end

//-----------------------------------------------------------------
// GetNumCols
//-----------------------------------------------------------------
function tPathFunc.GetNumCols()    return(table.getn(PetAI.tPathFunc.tPathMap[1]))    end

//-----------------------------------------------------------------
// IsBlockerNode
//-----------------------------------------------------------------
function tPathFunc.IsBlockerNode(tNode) return(tNode[2] < 0) end

-- ****************************************************************
-- Helper Functions
-- ****************************************************************

//-----------------------------------------------------------------
// GetNodeIndex
//-----------------------------------------------------------------
function tPathFunc.GetNodeIndex(tNode)

    if (PetAI.tPathFunc.tReversePathMap == nil) then
        PetAI.tPathFunc.tReversePathMap = {}

        for r=1,PetAI.tPathFunc.GetNumRows() do
            for c=1,PetAI.tPathFunc.GetNumCols() do
                PetAI.tPathFunc.tReversePathMap[PetAI.tPathFunc.tPathMap[r][c]] = {r, c}
            end
        end
    end

    return(PetAI.tPathFunc.tReversePathMap[tNode] or PetAI.tPathFunc.GetNodeIndexBruteForce(tNode))
end

function tPathFunc.GetNodeIndexBruteForce(tNode)

    for r=1,PetAI.tPathFunc.GetNumRows() do
        for c=1,PetAI.tPathFunc.GetNumCols() do
            if (PetAI.tPathFunc.PointsEqual(tNode,PetAI.tPathFunc.tPathMap[r][c])) then return({r,c}) end
        end
    end
    return(nil)
end

//-----------------------------------------------------------------
// DistanceBetweenTwoPoints
//-----------------------------------------------------------------
function tPathFunc.DistanceBetweenTwoPoints(tA,tB)
    MyAssert(tA != nil and tB != nil)

    local fX = tA[1]-tB[1]
    local fY = tA[2]-tB[2]
    local fZ = tA[3]-tB[3]
    return(math.sqrt(fX^2 + fY^2 + fZ^2))
end

//-----------------------------------------------------------------
// PointsEqual
//-----------------------------------------------------------------
function tPathFunc.PointsEqual(tA,tB)
    return(((tA[1] == tB[1]) and (tA[2] == tB[2])) and (tA[3] == tB[3]))
end

//-----------------------------------------------------------------
// AddPoints
//-----------------------------------------------------------------
function tPathFunc.AddPoints(tA,tB)
    local tC = {0,0,0}
    for i in ipairs(tA) do tC[i] = tA[i] + tB[i] end
    return(tC)
end

//-----------------------------------------------------------------
// GetPointListString
//-----------------------------------------------------------------
function tPathFunc.GetPointListString(tPointTable)

  MyAssert(tPointTable != nil)

    local szPointList = ""
    for i in ipairs(tPointTable) do
        szPointList = szPointList .. "," .. tPointTable[i][1] .. "," .. tPointTable[i][2] .. "," .. tPointTable[i][3]
    end
    return(szPointList)
end

//-----------------------------------------------------------------
// PrintPoint
//-----------------------------------------------------------------
function tPathFunc.PrintPoint(tPoint)
    if (tPoint == nil) then print("nil")
    else print("{" .. tPoint[1] .. "," .. tPoint[2] .. "," .. tPoint[3] .. "}") end
end

-- ****************************************************************
-- Curve Point Rendering Functionality
-- ****************************************************************

//-----------------------------------------------------------------
// tCurvePoints
//-----------------------------------------------------------------
tPathFunc.tCurvePoints = {}

//-----------------------------------------------------------------
// AddCurvePoint
//-----------------------------------------------------------------
function tPathFunc.AddCurvePoint(fX,fY,fZ)
  local cCurvePoint = Cx3DObject()
  cCurvePoint:Load("Objects/dbg_pathmarker.lua")
  cCurvePoint:SetPosition(fX, fY, fZ)
  cCurvePoint:SetVisible(1)
  cCurvePoint:Link(Room.m_cSceneManager)
    table.insert(PetAI.tPathFunc.tCurvePoints,cCurvePoint)
end

//-----------------------------------------------------------------
// RemoveAllCurvePoints
//-----------------------------------------------------------------
function tPathFunc.RemoveAllCurvePoints()
    for i in ipairs(PetAI.tPathFunc.tCurvePoints) do Room.RemoveObject(PetAI.tPathFunc.tCurvePoints[i]) end
end

-- ****************************************************************
-- Pathfinding Functionality
-- ****************************************************************

//-----------------------------------------------------------------
// GetRoomPetIsIn
//-----------------------------------------------------------------
function tPathFunc.GetRoomPetIsIn()

  local x,y,z = pet:GetPosition()
  local tClosestNode = PetAI.tPathFunc.FindClosestPathNode(x,y,z)

  for k,v in pairs(PetAI.tPathFunc.tRooms) do
    for i = 1,v.iNumRows do
      for j = 1,v.iNumCols do
        if (PetAI.tPathFunc.PointsEqual(tClosestNode,v.tNodes[i][j])) then
          return(k)
        end
      end
    end
  end
end

//-----------------------------------------------------------------
// GetClosestNodeToPet
//-----------------------------------------------------------------
function tPathFunc.GetClosestNodeToPet()
  local x,y,z = pet:GetPosition()
  return(PetAI.tPathFunc.FindClosestPathNode(x,y,z,PetAI.szLocation))
end

//-----------------------------------------------------------------
// GetDoorwayNode
//-----------------------------------------------------------------
function tPathFunc.GetDoorwayNode(szStartRoom,szEndRoom)
  if (szEndRoom == "Bathroom") then
    return(PetAI.tPathFunc.tPathMap[37][16])
  elseif (szEndRoom == "Kitchen") then
    if (szStartRoom == "Bathroom") then return(PetAI.tPathFunc.tPathMap[37][16])
    elseif (szStartRoom == "LivingRoom") then return(PetAI.tPathFunc.tPathMap[37][33])
    elseif (szStartRoom == "Backyard") then return(PetAI.tPathFunc.tPathMap[37][33]) end
  elseif (szEndRoom == "LivingRoom") then
    if (szStartRoom == "Bathroom") then return(PetAI.tPathFunc.tPathMap[37][33])
    elseif (szStartRoom == "Kitchen") then return(PetAI.tPathFunc.tPathMap[37][33])
    elseif (szStartRoom == "Backyard") then return(PetAI.tPathFunc.tPathMap[37][51]) end
  elseif (szEndRoom == "Backyard") then
   return(PetAI.tPathFunc.tPathMap[37][49])
  end
end

//-----------------------------------------------------------------
// GeneratePathMap
//-----------------------------------------------------------------
function tPathFunc.GeneratePathMap(szRoom)

    PetAI.tPathFunc.tPathMap = {}
    PetAI.tPathFunc.tPathObjects = {}

    -- specify top/left and bottom/right corners of map
    local tStartPoint = {605,0,-455}
    local tEndPoint = {-490,0,105}

    -- initialize x and z
    local iX = tStartPoint[1]
    local iZ = tStartPoint[3]

    -- initialize row counter
    local iRow = 1

  local iSpacing = 15

    -- while within the
    while not (iZ > tEndPoint[3]) do
        iX = tStartPoint[1]
        PetAI.tPathFunc.tPathMap[iRow] = {}
        while not (iX < tEndPoint[1]) do
            table.insert(PetAI.tPathFunc.tPathMap[iRow],{iX,0,iZ})
            iX = iX - iSpacing
        end
        iZ = iZ + iSpacing
        iRow = iRow + 1
    end

  PetAI.tPathFunc.tDisableSet = {
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,0,1,1,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,0,1,1,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,0,1,1,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,1,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,1,},
{1,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,1,},
{1,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,1,},
{1,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,1,},
{1,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,1,},
{1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,1,},
{1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,2,1,1,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,1,},
{1,2,0,0,0,0,0,0,0,0,0,0,1,2,1,1,1,2,0,0,0,0,0,0,0,0,0,0,0,2,2,1,1,1,2,0,0,0,0,0,0,0,0,0,0,0,0,2,1,1,2,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,1,},
{1,2,2,0,0,0,0,0,0,0,0,0,0,2,1,1,1,2,2,0,0,0,0,0,0,0,0,0,0,2,2,1,1,1,2,2,0,0,0,0,0,0,0,0,0,0,2,2,1,1,2,2,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,2,1,},
{1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,},
{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,},
}

  PetAI.tPathFunc.tTransNodes = {}

  for r in ipairs(PetAI.tPathFunc.tDisableSet) do
    for c in ipairs(PetAI.tPathFunc.tDisableSet[r]) do
      if (PetAI.tPathFunc.tDisableSet[r][c] == 1) then PetAI.tPathFunc.tPathMap[r][c][2] = -1
      elseif (PetAI.tPathFunc.tDisableSet[r][c] == 2) then
        table.insert(PetAI.tPathFunc.tTransNodes,{r,c})
      end
    end
  end

  PetAI.tPathFunc.DisableTransNodes()

  PetAI.tPathFunc.ConfigureTagPoints()

    PetAI.tPathFunc.tRooms = {}

    -- ****************************************************************
  -- Bathroom Nodes
  -- ****************************************************************

    PetAI.tPathFunc.tRooms.Bathroom = {}
    PetAI.tPathFunc.tRooms.Bathroom.iRowOffset = 0
    PetAI.tPathFunc.tRooms.Bathroom.iColOffset = 0
    PetAI.tPathFunc.tRooms.Bathroom.tNodes = {}
    for r=1,38 do
        PetAI.tPathFunc.tRooms.Bathroom.tNodes[r] = {}
        for c=1,16 do
            table.insert(PetAI.tPathFunc.tRooms.Bathroom.tNodes[r],c,PetAI.tPathFunc.tPathMap[r][c+PetAI.tPathFunc.tRooms.Bathroom.iColOffset])
        end
    end
    PetAI.tPathFunc.tRooms.Bathroom.iNumRows = table.getn(PetAI.tPathFunc.tRooms.Bathroom.tNodes)
    PetAI.tPathFunc.tRooms.Bathroom.iNumCols = table.getn(PetAI.tPathFunc.tRooms.Bathroom.tNodes[1])
    PetAI.tPathFunc.tRooms.Bathroom.GetHomePosition = function() return(PetAI.tPathFunc.tPathMap[33][8]) end
    PetAI.tPathFunc.tRooms.Bathroom.GetAttentionNode = function()
        local iRow = PetAI.tPathFunc.tRooms.Bathroom.iRowOffset + PetAI.tPathFunc.tRooms.Bathroom.iNumRows-2
        local iCol = PetAI.tPathFunc.tRooms.Bathroom.iColOffset + math.floor(PetAI.tPathFunc.tRooms.Bathroom.iNumCols/2)
        return(PetAI.tPathFunc.tPathMap[iRow][iCol])
    end
    PetAI.tPathFunc.tRooms.Bathroom.GetRunAwayNode = function() return(PetAI.tPathFunc.tPathMap[25][12]) end
    PetAI.tPathFunc.tRooms.Bathroom.GetPeeNode = function() return(PetAI.tPathFunc.tPathMap[28][10]) end
    PetAI.tPathFunc.tRooms.Bathroom.GetLeftBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][4]) end
    PetAI.tPathFunc.tRooms.Bathroom.GetRightBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][13]) end

  -- ****************************************************************
  -- Kitchen Nodes
  -- ****************************************************************

    PetAI.tPathFunc.tRooms.Kitchen = {}
    PetAI.tPathFunc.tRooms.Kitchen.iRowOffset = 0
    PetAI.tPathFunc.tRooms.Kitchen.iColOffset = 16
    PetAI.tPathFunc.tRooms.Kitchen.tNodes = {}
    for r=1,38 do
        PetAI.tPathFunc.tRooms.Kitchen.tNodes[r] = {}
        for c=1,17 do
            table.insert(PetAI.tPathFunc.tRooms.Kitchen.tNodes[r],c,PetAI.tPathFunc.tPathMap[r][c+PetAI.tPathFunc.tRooms.Kitchen.iColOffset])
        end
    end
    PetAI.tPathFunc.tRooms.Kitchen.iNumRows = table.getn(PetAI.tPathFunc.tRooms.Kitchen.tNodes)
    PetAI.tPathFunc.tRooms.Kitchen.iNumCols = table.getn(PetAI.tPathFunc.tRooms.Kitchen.tNodes[1])
    PetAI.tPathFunc.tRooms.Kitchen.GetHomePosition = function() return(PetAI.tPathFunc.tPathMap[33][23]) end
    PetAI.tPathFunc.tRooms.Kitchen.GetAttentionNode = function()
        local iRow = PetAI.tPathFunc.tRooms.Kitchen.iRowOffset + PetAI.tPathFunc.tRooms.Kitchen.iNumRows-2
        local iCol = PetAI.tPathFunc.tRooms.Kitchen.iColOffset + math.floor(PetAI.tPathFunc.tRooms.Kitchen.iNumCols/2) + 1
        return(PetAI.tPathFunc.tPathMap[iRow][iCol])
    end
    PetAI.tPathFunc.tRooms.Kitchen.GetRunAwayNode = function() return(PetAI.tPathFunc.tPathMap[28][29]) end
    PetAI.tPathFunc.tRooms.Kitchen.GetPeeNode = function() return(PetAI.tPathFunc.tPathMap[33][23]) end
    PetAI.tPathFunc.tRooms.Kitchen.GetLeftBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][20]) end
    PetAI.tPathFunc.tRooms.Kitchen.GetRightBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][29]) end

  -- ****************************************************************
    -- Living Room Nodes
  -- ****************************************************************

    PetAI.tPathFunc.tRooms.LivingRoom = {}
    PetAI.tPathFunc.tRooms.LivingRoom.iRowOffset = 0
    PetAI.tPathFunc.tRooms.LivingRoom.iColOffset = 33
    PetAI.tPathFunc.tRooms.LivingRoom.tNodes = {}
    for r=1,38 do
        PetAI.tPathFunc.tRooms.LivingRoom.tNodes[r] = {}
        for c=1,16 do
            table.insert(PetAI.tPathFunc.tRooms.LivingRoom.tNodes[r],c,PetAI.tPathFunc.tPathMap[r][c+PetAI.tPathFunc.tRooms.LivingRoom.iColOffset])
        end
    end
    PetAI.tPathFunc.tRooms.LivingRoom.iNumRows = table.getn(PetAI.tPathFunc.tRooms.LivingRoom.tNodes)
    PetAI.tPathFunc.tRooms.LivingRoom.iNumCols = table.getn(PetAI.tPathFunc.tRooms.LivingRoom.tNodes[1])
    PetAI.tPathFunc.tRooms.LivingRoom.GetHomePosition = function() return(PetAI.tPathFunc.tPathMap[33][42]) end
    PetAI.tPathFunc.tRooms.LivingRoom.GetAttentionNode = function()
        local iRow = PetAI.tPathFunc.tRooms.LivingRoom.iRowOffset + PetAI.tPathFunc.tRooms.LivingRoom.iNumRows-2
        local iCol = PetAI.tPathFunc.tRooms.LivingRoom.iColOffset + math.floor(PetAI.tPathFunc.tRooms.LivingRoom.iNumCols/2) + 1
        return(PetAI.tPathFunc.tPathMap[iRow][iCol])
    end
    PetAI.tPathFunc.tRooms.LivingRoom.GetRunAwayNode = function() return(PetAI.tPathFunc.tPathMap[27][45]) end
    PetAI.tPathFunc.tRooms.LivingRoom.GetPeeNode = function() return(PetAI.tPathFunc.tPathMap[28][44]) end
    PetAI.tPathFunc.tRooms.LivingRoom.GetLeftBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][37]) end
    PetAI.tPathFunc.tRooms.LivingRoom.GetRightBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][46]) end


    -- ****************************************************************
    -- Backyard Nodes
    -- ****************************************************************

    PetAI.tPathFunc.tRooms.Backyard = {}
    PetAI.tPathFunc.tRooms.Backyard.iRowOffset = 0
    PetAI.tPathFunc.tRooms.Backyard.iColOffset = 49
    PetAI.tPathFunc.tRooms.Backyard.tNodes = {}
    for r=1,38 do
        PetAI.tPathFunc.tRooms.Backyard.tNodes[r] = {}
        for c=1,25 do
            table.insert(PetAI.tPathFunc.tRooms.Backyard.tNodes[r],c,PetAI.tPathFunc.tPathMap[r][c+PetAI.tPathFunc.tRooms.Backyard.iColOffset])
        end
    end
    PetAI.tPathFunc.tRooms.Backyard.iNumRows = table.getn(PetAI.tPathFunc.tRooms.Backyard.tNodes)
    PetAI.tPathFunc.tRooms.Backyard.iNumCols = table.getn(PetAI.tPathFunc.tRooms.Backyard.tNodes[1])
    PetAI.tPathFunc.tRooms.Backyard.GetHomePosition = function() return(PetAI.tPathFunc.tPathMap[33][57]) end
    PetAI.tPathFunc.tRooms.Backyard.GetAttentionNode = function()
        local iRow = PetAI.tPathFunc.tRooms.Backyard.iRowOffset + PetAI.tPathFunc.tRooms.Backyard.iNumRows-2
        local iCol = PetAI.tPathFunc.tRooms.Backyard.iColOffset + math.floor(PetAI.tPathFunc.tRooms.Backyard.iNumCols/3)
        return(PetAI.tPathFunc.tPathMap[iRow][iCol])
    end
    PetAI.tPathFunc.tRooms.Backyard.GetRunAwayNode = function() return(PetAI.tPathFunc.tPathMap[3][67]) end
    PetAI.tPathFunc.tRooms.Backyard.GetPeeNode = function() return(PetAI.tPathFunc.tPathMap[33][58]) end
  PetAI.tPathFunc.tRooms.Backyard.GetLeftBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][53]) end
  PetAI.tPathFunc.tRooms.Backyard.GetRightBottomCornerNode = function() return(PetAI.tPathFunc.tPathMap[35][61]) end
end


//-----------------------------------------------------------------
// EnableTransNodes
//-----------------------------------------------------------------
function tPathFunc.EnableTransNodes()
  for i,v in ipairs(PetAI.tPathFunc.tTransNodes) do
    PetAI.tPathFunc.tPathMap[v[1]][v[2]][2] = 0
  end
end

//-----------------------------------------------------------------
// DisableTransNodes
//-----------------------------------------------------------------
function tPathFunc.DisableTransNodes()
  for i,v in ipairs(PetAI.tPathFunc.tTransNodes) do
    PetAI.tPathFunc.tPathMap[v[1]][v[2]][2] = -1
  end
end

//-----------------------------------------------------------------
// FindClosestPathNode
//-----------------------------------------------------------------
function tPathFunc.FindClosestPathNode(fX,fY,fZ,szRoom)

    local tSearchPoint = {fX,fY,fZ}

    local fDistance = 0
    local fShortestDistance = 99999

    local iRow = 0
    local iCol = 0

    local iTempRow = 0
    local iTempCol = 0

    fDistance = 0
    fShortestDistance = 99999

  if (szRoom == nil) then

      for r=1,table.getn(PetAI.tPathFunc.tPathMap) do
        for c=1,table.getn(PetAI.tPathFunc.tPathMap[1]) do
            if (PetAI.tPathFunc.tPathMap[r][c][2] != -1) then
                fDistance = PetAI.tPathFunc.DistanceBetweenTwoPoints(tSearchPoint,PetAI.tPathFunc.tPathMap[r][c])
                  if (fDistance < fShortestDistance) then
                    fShortestDistance = fDistance
                    iRow = r
                    iCol = c
                  end
            end
        end
    end
  else
      local tRoomToSearch = PetAI.tPathFunc.tRooms[szRoom]

      for r=1,table.getn(tRoomToSearch.tNodes) do
        iTempRow = r + tRoomToSearch.iRowOffset
        for c=1,table.getn(tRoomToSearch.tNodes[1]) do
            iTempCol = c + tRoomToSearch.iColOffset
            if (PetAI.tPathFunc.tPathMap[iTempRow][iTempCol][2] != -1) then
                fDistance = PetAI.tPathFunc.DistanceBetweenTwoPoints(tSearchPoint,PetAI.tPathFunc.tPathMap[iTempRow][iTempCol])
                  if (fDistance < fShortestDistance) then
                    fShortestDistance = fDistance
                    iRow = iTempRow
                    iCol = iTempCol
                  end
            end
        end
    end
  end

    return(PetAI.tPathFunc.tPathMap[iRow][iCol])
end

//-----------------------------------------------------------------
// BlockOutSection
//-----------------------------------------------------------------
function tPathFunc.BlockOutSection(tRoom,iStartRow,iEndRow,iStartCol,iEndCol)
    for i = iStartRow,iEndRow do for j = iStartCol,iEndCol do PetAI.tPathFunc.tPathMap[i+tRoom.iRowOffset][j+tRoom.iColOffset][2] = -1 end end
end

//-----------------------------------------------------------------
// RestoreSection
//-----------------------------------------------------------------
function tPathFunc.RestoreSection(tRoom,iStartRow,iEndRow,iStartCol,iEndCol)
    for i = iStartRow,iEndRow do for j = iStartCol,iEndCol do PetAI.tPathFunc.tPathMap[i+tRoom.iRowOffset][j+tRoom.iColOffset][2] = 0 end end
end

//-----------------------------------------------------------------
// tRoomNames
//-----------------------------------------------------------------
tPathFunc.tRoomNames = {"Backyard","LivingRoom","Kitchen","Bathroom"}

//-----------------------------------------------------------------
// RenderPathMap
//-----------------------------------------------------------------
function tPathFunc.RenderPathMap(tRoom)
    local fVisibilityOffset = 5

  if (tRoom == nil) then
      for r in ipairs(PetAI.tPathFunc.tPathMap) do
          table.insert(PetAI.tPathFunc.tPathObjects,{})
      if (r >= 21) then
             for c in ipairs(PetAI.tPathFunc.tPathMap[r]) do
              local tNode = PetAI.tPathFunc.tPathMap[r][c]
              local o = Room.AddObjectToScene("Objects/dbg_mapnode.lua",tNode[1],tNode[2] + fVisibilityOffset,tNode[3])
                o:SetName("Node R=" .. r .. " C=" .. c)
                if (PetAI.tPathFunc.IsBlockerNode(tNode)) then o:SetTexture("Mod_xGrayT.dds")
                elseif (PetAI.tPathFunc.tDisableSet[r][c] == 2) then o:SetTexture("Mod_xGreenT.dds") end
                table.insert(PetAI.tPathFunc.tPathObjects[r],o)
            end
          end
      end
    else
      for r in ipairs(tRoom.tNodes) do
           table.insert(PetAI.tPathFunc.tPathObjects,{})
            for c in ipairs(tRoom.tNodes[r]) do
              local tNode = tRoom.tNodes[r][c]
              local o = Room.AddObjectToScene("Objects/dbg_mapnode.lua",tNode[1],tNode[2] + fVisibilityOffset,tNode[3])
                o:SetName("Node R=" .. r .. " C=" .. c)
                if (PetAI.tPathFunc.IsBlockerNode(tNode)) then o:SetTexture("Mod_xGrayT.dds")
                elseif (PetAI.tPathFunc.tDisableSet[r][c] == 2) then o:SetTexture("Mod_xGreenT.dds") end
                table.insert(PetAI.tPathFunc.tPathObjects[r],o)
            end
      end
  end
end

//-----------------------------------------------------------------
// PrintTagPoints
//-----------------------------------------------------------------
function tPathFunc.PrintTagPoints()

  local PrintTag = function(szName,cObject,szTag)
    local x,y,z = cObject:GetBonePosition(cObject:GetBoneIndex(szTag))
    LogInfo(szName.." = {"..x..","..y..","..z.."}")
  end

    -- ****************************************************************
     -- Bathroom Tag Points
     -- ****************************************************************
  if CATZ then
    PrintTag("CatTubScratch",Room.m_cTub,"-tag cat scratch")
    PrintTag("CatLitterBoxUrinate",Room.m_cLitterBox,"-tag urinate")
    PrintTag("CatPerchOffPoint",Room.m_cCatPerch,"-Tag Pets Off Sofa")
  elseif (PetAI.tBreed.szName == "Beagle") then
    PrintTag("BeagleTubScratch",Room.m_cTub,"-tag beagle scratch")
  elseif (PetAI.tBreed.szName == "Pug") then
    PrintTag("PugTubScratch",Room.m_cTub,"-tag pug scratch")
  else
    PrintTag("DogTubScratch",Room.m_cTub,"-tag dog scratch")
  end

  -- ****************************************************************
    -- Kitchen Tag Points
    -- ****************************************************************
  if CATZ then
    PrintTag("CatCounterOffPoint",Room.m_cKitchen,"-Tag Cat Off Counter")
    PrintTag("CatCounterLandPoint",Room.m_cKitchen,"-Tag Cat Land")
    PrintTag("CatWaterBowlPoint",Room.m_cWaterBowl,"-tag cat body")
    PrintTag("CatFoodBowlPoint",Room.m_cFoodBowl,"-tag cat body")
  elseif (PetAI.tBreed.szName == "Beagle") then
    PrintTag("BeagleWaterBowlPoint",Room.m_cWaterBowl,"-tag beagle body")
    PrintTag("BeagleFoodBowlPoint",Room.m_cFoodBowl,"-tag beagle body")
  elseif (PetAI.tBreed.szName == "Pug") then
    PrintTag("PugWaterBowlPoint",Room.m_cWaterBowl,"-tag pug body")
    PrintTag("PugFoodBowlPoint",Room.m_cFoodBowl,"-tag pug body")
  else
    PrintTag("DogWaterBowlPoint",Room.m_cWaterBowl,"-tag dog body")
    PrintTag("DogFoodBowlPoint",Room.m_cFoodBowl,"-tag dog body")
  end

  -- ****************************************************************
  -- Living Room Tag Points
  -- ****************************************************************
  PrintTag("PetCoffeeTableOffPoint",Room.m_cCoffeeTable,"-tag Pets Off Sofa")
  PrintTag("PetBedSleepPoint",Room.m_cPetBed,"-Tag Sleep")
  PrintTag("PetFireplaceSniffPoint",Room.m_cLivingRoom,"-tag Investigate")

  if CATZ then
    PrintTag("PetSofaJumpToPoint",Room.m_cSofa,"-Tag Cat Off Sofa")
    PrintTag("CatSofaScratch",Room.m_cSofa,"-tag cat scratch")
    PrintTag("CatScratchingPostScratch",Room.m_cScratchingPost,"-tag cat body")
  elseif (PetAI.tBreed.szName == "Beagle") then
    PrintTag("PetSofaJumpToPoint",Room.m_cSofa,"-tag Pets Jump To Sofa")
    PrintTag("BeagleSofaScratch",Room.m_cSofa,"-tag beagle scratch")
  elseif (PetAI.tBreed.szName == "Pug") then
    PrintTag("PetSofaJumpToPoint",Room.m_cSofa,"-tag Pets Jump To Sofa")
    PrintTag("PugSofaScratch",Room.m_cSofa,"-tag pug scratch")
  else
    PrintTag("PetSofaJumpToPoint",Room.m_cSofa,"-tag Pets Jump To Sofa")
    PrintTag("DogSofaScratch",Room.m_cSofa,"-tag dog scratch")
   end

 -- ****************************************************************
 -- Backyard Tag Points
 -- ****************************************************************
  PrintTag("PetDogHouseSleep",Room.m_cDogHouse,"-Tag Sleep")
  PrintTag("PetPlantSniff",Room.m_cBackyard,"-tag investigate")
end

tPathFunc.tTagNodes = {}

//-----------------------------------------------------------------
// ConfigureTagPoints
//-----------------------------------------------------------------
function tPathFunc.ConfigureTagPoints()

 local tTagPoints = nil

  if CATZ then
    tTagPoints = {
     TubScratch = {552.23333740234,-0.057195544242859,-24.600830078125},
     LitterBoxUrinate = {586.24182128906,-0.040740821510553,-30.984745025635},
     PerchOffPoint = {460.48330688477,-0.054561518132687,44.329807281494},
     CounterOffPoint = {253.85670471191,-9.5367431640625e-007,-22.465885162354},
     CounterLandPoint = {253.85670471191,-4.7683715820313e-007,-30.962348937988},
     WaterBowlPoint = {328.2063293457,-0.061757523566484,-14.05695438385},
     FoodBowlPoint = {328.2033996582,0.069139786064625,7.1879496574402},
     CoffeeTableOffPoint = {44.11120223999,-0.042274475097656,-33.379867553711},
     BedSleepPoint = {87.516967773438,-0.042274475097656,41.679256439209},
     FireplaceSniffPoint = {-2.9723510742188,-0.042274475097656,-66.451385498047},
     SofaJumpToPoint = {-37.711353302002,-0.042274530977011,-16.587139129639},
     SofaScratch = {-48.779853820801,-0.099469766020775,-16.632329940796},
     ScratchingPostScratch = {28.58185005188,-0.099469684064388,-67.129928588867},
     DogHouseSleep = {-415.23443603516,-1.3239498138428,-380.50921630859},
     PlantSniff = {-220.77853393555,-1.3239498138428,-380.50921630859},
    }
  elseif (PetAI.tBreed.szName == "Beagle") then
    tTagPoints = {
     TubScratch = {552.23333740234,-0.057195544242859,-23.299331665039},
     WaterBowlPoint = {328.17361450195,0.058514092117548,-12.948556900024},
     FoodBowlPoint = {328.17361450195,0.058514092117548,8.3171644210815},
     CoffeeTableOffPoint = {44.11120223999,-0.042274475097656,-33.379867553711},
     BedSleepPoint = {87.516967773438,-0.042274475097656,41.679256439209},
     FireplaceSniffPoint = {-2.9723510742188,-0.042274475097656,-66.451385498047},
     SofaJumpToPoint = {-44.872051239014,-0.042274218052626,-16.587139129639},
     SofaScratch = {-47.478366851807,-0.099469766020775,-16.632331848145},
     DogHouseSleep = {-415.23443603516,-1.3239498138428,-380.50921630859},
     PlantSniff = {-220.77853393555,-1.3239498138428,-380.50921630859},
    }
  elseif (PetAI.tBreed.szName == "Pug") then
    tTagPoints = {
     TubScratch = {552.23333740234,-0.05719530582428,-30.573364257813},
     WaterBowlPoint = {334.09997558594,0.058514412492514,-14.12343120575},
     FoodBowlPoint = {334.09997558594,0.058514405041933,7.1422882080078},
     CoffeeTableOffPoint = {44.11120223999,-0.042274475097656,-33.379867553711},
     BedSleepPoint = {87.516967773438,-0.042274475097656,41.679256439209},
     FireplaceSniffPoint = {-2.9723510742188,-0.042274475097656,-66.451385498047},
     SofaJumpToPoint = {-44.872051239014,-0.042274218052626,-16.587139129639},
     SofaScratch = {-54.752391815186,-0.099469527602196,-16.632329940796},
     DogHouseSleep = {-415.23443603516,-1.3239498138428,-380.50921630859},
     PlantSniff = {-220.77853393555,-1.3239498138428,-380.50921630859},
    }
  else
    tTagPoints = {
     TubScratch = {552.23333740234,-0.057195544242859,-24.046440124512},     
     WaterBowlPoint = {325.01104736328,0.058513946831226,-11.480848312378},
     FoodBowlPoint = {325.01104736328,0.058513946831226,9.7848720550537}, 
     CoffeeTableOffPoint = {44.11120223999,-0.042274475097656,-33.379867553711},
     BedSleepPoint = {87.516967773438,-0.042274475097656,41.679256439209},
     FireplaceSniffPoint = {-2.9723510742188,-0.042274475097656,-66.451385498047},
     SofaJumpToPoint = {-44.872051239014,-0.042274218052626,-16.587139129639},
     SofaScratch = {-48.225467681885,-0.099469766020775,-16.632328033447},
     DogHouseSleep = {-415.23443603516,-1.3239498138428,-380.50921630859},
     PlantSniff = {-220.77853393555,-1.3239498138428,-380.50921630859},
    }
  end

  local tClosestNode = nil
    local tIndex = nil
    local tAlteredNodes = {}

    for k,v in pairs(tTagPoints) do
      v[2] = 0
    tClosestNode = PetAI.tPathFunc.FindClosestPathNode(v[1],v[2],v[3])
    tIndex = PetAI.tPathFunc.GetNodeIndex(tClosestNode)
      PetAI.tPathFunc.tPathMap[tIndex[1]][tIndex[2]] = v
      PetAI.tPathFunc.tPathMap[tIndex[1]][tIndex[2]][2] = -1
    table.insert(tAlteredNodes,tIndex)
    PetAI.tPathFunc.tTagNodes[k] = v
    end

    for i,v in ipairs(tAlteredNodes) do
    PetAI.tPathFunc.tPathMap[v[1]][v[2]][2] = 0
    end

end

//-----------------------------------------------------------------
// GetAdjacentNodes
//-----------------------------------------------------------------
function tPathFunc.GetAdjacentNodes(tNode)

    local tAdjNodes = {}

    local tIndex = PetAI.tPathFunc.GetNodeIndex(tNode)
    local iRow = tIndex[1]
    local iCol = tIndex[2]

    local iRowStart = iRow-1
    local iRowEnd = iRow+1

    if (iRowStart <= 0) then iRowStart = 1 end
    if (iRowEnd > PetAI.tPathFunc.GetNumRows()) then iRowEnd = PetAI.tPathFunc.GetNumRows() end

    local iColStart = iCol-1
    local iColEnd = iCol+1

    if (iColStart <= 0) then iColStart = 1 end
    if (iColEnd > PetAI.tPathFunc.GetNumCols()) then iColEnd = PetAI.tPathFunc.GetNumCols() end

    for r = iRowStart,iRowEnd do
        for c = iColStart,iColEnd do
            if (not (PetAI.tPathFunc.tPathMap[r][c] == nil)) then
                if (not (r == iRow and c == iCol)) and (not PetAI.tPathFunc.IsBlockerNode(PetAI.tPathFunc.tPathMap[r][c])) then
                    table.insert(tAdjNodes,PetAI.tPathFunc.tPathMap[r][c])
                end
            end
        end
    end

    return(tAdjNodes)
end

//-----------------------------------------------------------------
// RemoveAllPathMaps
//-----------------------------------------------------------------
function tPathFunc.RemoveAllPathMaps()
    if not (PetAI.tPathFunc.tPathObjects == nil) then
        for j in ipairs(PetAI.tPathFunc.tPathObjects) do
            for k in ipairs(PetAI.tPathFunc.tPathObjects[j]) do
                Room.RemoveObject(PetAI.tPathFunc.tPathObjects[j][k])
            end
        end
    end
end

//-----------------------------------------------------------------
// Refresh
//-----------------------------------------------------------------
function tPathFunc.Refresh()

    PetAI.tPathFunc.RemoveAllCurvePoints()

    for r in ipairs(PetAI.tPathFunc.tPathObjects) do
        for c in ipairs(PetAI.tPathFunc.tPathObjects[r]) do
            if not (PetAI.tPathFunc.IsBlockerNode(PetAI.tPathFunc.tPathMap[r][c])) then PetAI.tPathFunc.tPathObjects[r][c]:SetTexture("Mod_xPurpleT.dds") end
        end
    end
end

//-----------------------------------------------------------------
// HighlighPetAI.tPathFunc.tPathMapsNode
//-----------------------------------------------------------------
function tPathFunc.HighlightPathMapNode(iRow,iCol)
        local o = PetAI.tPathFunc.tPathObjects[iRow][iCol]
        o:SetTexture("Mod_xGreenT.dds")
        PetAI.tPathFunc.PrintPoint(PetAI.tPathFunc.tPathMap[iRow][iCol])
end

//-----------------------------------------------------------------
// GetCostToGoal
//-----------------------------------------------------------------
function tPathFunc.GetCostOfMovement(tStartNode,tEndNode)

    local fCost = 0

    fCost = PetAI.tPathFunc.DistanceBetweenTwoPoints(tStartNode,tEndNode)

    return(fCost)
end

//-----------------------------------------------------------------
// FindPath
//-----------------------------------------------------------------
function tPathFunc.FindPath(tStartLoc,tEndLoc)
    local tOpen = {}
    local tClosed = {}

    -- initialize start node
    local tStartNode = {
        tLocation = tStartLoc,
        fCostFromStart = 0,
        fCostToGoal = PetAI.tPathFunc.GetCostOfMovement(tStartLoc,tEndLoc),
        tParent = nil,
    }

    -- define node searching function for use in path finding
    local FindNode = function(tTable,tNode)
        for i in ipairs(tTable) do
            if (tTable[i].tLocation == tNode.tLocation) then return(i) end
        end
        return(-1)
    end

    -- define node checking function for use in path finding
    local CheckIfNodeDefinedAndCheaper = function(tNodeTable,tNodeToFind,fCost)
        local i = FindNode(tNodeTable,tNodeToFind)

        if (i != -1) then return(tNodeTable[i].fCostFromStart < fCost) end
        return(false)
    end

    -- calculate total cost for start node
    tStartNode.fTotalCost = tStartNode.fCostFromStart + tStartNode.fCostToGoal

    -- push start node onto Open
    table.insert(tOpen,1,tStartNode)

    -- process the list until success or failure
    while not (table.getn(tOpen) == 0) do

        -- node has lowest TotalCost
        local tNode = table.remove(tOpen,1)

        -- if at a goal, we're done
        if (tNode.tLocation == tEndLoc) then

            -- create table for completed path nodes
            local tPathNodes = {}

            -- track backwards along the path nodes, pushing them onto the table
            while not(tNode == nil) do
                table.insert(tPathNodes,1,tNode.tLocation)
                tNode = tNode.tParent
            end

            -- return the completed path
            return(tPathNodes)

        -- otherwise, if not at goal
        else

            -- retrieve a table of all nodes adjacent to the current node
            local tAdjNodes = PetAI.tPathFunc.GetAdjacentNodes(tNode.tLocation)

            -- for each adjacent node...
            for i in ipairs(tAdjNodes) do

                -- construct a node for the adjacent point
                local tNewNode = {}
                tNewNode.tLocation = tAdjNodes[i]

                -- calculate the cost of the new adjacent node
                local fNewCost = tNode.fCostFromStart + PetAI.tPathFunc.GetCostOfMovement(tNode.tLocation,tNewNode.tLocation)

                -- if the new node is not in the open or closed tables, and its cost from the start is less than the calculated new cost
                if not (CheckIfNodeDefinedAndCheaper(tOpen,tNewNode,fNewCost) or CheckIfNodeDefinedAndCheaper(tClosed,tNewNode,fNewCost)) then

                    -- store new node information
                    tNewNode.tParent = tNode
                    tNewNode.fCostFromStart = fNewCost
                    tNewNode.fCostToGoal = PetAI.tPathFunc.GetCostOfMovement(tNewNode.tLocation,tEndLoc)
                    tNewNode.fTotalCost = tNewNode.fCostFromStart + tNewNode.fCostToGoal

                    -- if the new node is in closed, remove it
                    local j = FindNode(tClosed,tNewNode)

                    if (j != -1) then table.remove(tClosed, j) end

                    -- if the new node is in open, PetAI.tThinkFunc.Update its information
                    j = FindNode(tOpen,tNewNode)
                    if (j != -1) then
                        tOpen[j] = tNewNode
                    else -- otherwise, push the new node onto open
                        table.insert(tOpen,1,tNewNode)
                    end

                    table.sort(tOpen,function(a,b) return(a.fTotalCost < b.fTotalCost) end)

                    end
            end -- finished with adjacent node
        end -- finished with node

        -- insert the processed node into the 'closed' table
        table.insert(tClosed,1,tNode)
    end

    -- no path found, return failure
    return(nil)
end

//-----------------------------------------------------------------
// WalkToPathNode
//-----------------------------------------------------------------
function tPathFunc.WalkToPathNode(tNode,szWalkAnim,szStopAnim,fSpeed,nChannel)
  if (nChannel == nil) then nChannel = ActivePetChannel end
    if (fSpeed == nil) then fSpeed = 65.0 end
    local tPathNodes = PetAI.tPathFunc.FindPath(PetAI.tPathFunc.GetClosestNodeToPet(),tNode)
    if (tPathNodes != nil) then
      if (table.getn(tPathNodes) >= 2) then
          DoEvent("Pet.WalkToPosition(" .. szWalkAnim .. "," .. szStopAnim .. ",100,"..fSpeed..PetAI.tPathFunc.GetPointListString(tPathNodes) .. ")",nChannel)
      end
  end
end

//-----------------------------------------------------------------
// WalkToRandomPathNode
//-----------------------------------------------------------------
function tPathFunc.WalkToRandomPathNode(szWalkAnim,szStopAnim)

    local tCurRoom = PetAI.tPathFunc.tRooms[PetAI.szLocation]

    local iRandRow = tCurRoom.iRowOffset + math.random(tCurRoom.iNumRows)
    local iRandCol = tCurRoom.iColOffset + math.random(tCurRoom.iNumCols)

    while (PetAI.tPathFunc.tPathMap[iRandRow][iRandCol][2] == -1) do
        iRandRow = tCurRoom.iRowOffset + math.random(tCurRoom.iNumRows)
        iRandCol = tCurRoom.iColOffset + math.random(tCurRoom.iNumCols)
    end

    PetAI.tPathFunc.WalkToPathNode(PetAI.tPathFunc.tPathMap[iRandRow][iRandCol],szWalkAnim,szStopAnim)
end

//-----------------------------------------------------------------
// IsAdjacentNode
//-----------------------------------------------------------------
function tPathFunc.IsAdjacentNode(tNodeToCheck,tOriginNode)
    local tAdjNodes = PetAI.tPathFunc.GetAdjacentNodes(tOriginNode)
    for i in ipairs(tAdjNodes) do if (PetAI.tPathFunc.PointsEqual(tAdjNodes[i],tNodeToCheck)) then return(true) end end
    return(false)
end

//-----------------------------------------------------------------
// tBlockedNodes
//-----------------------------------------------------------------

tPathFunc.tBlockedNodes = {}

//-----------------------------------------------------------------
// BlockAdjacentNodesExceptNorth
//-----------------------------------------------------------------
function tPathFunc.BlockAdjacentNodesExceptNorth(tNode)
    local tAdjNodes = PetAI.tPathFunc.GetAdjacentNodes(tNode)
    for i in ipairs(tAdjNodes) do
        if (not (i == 2)) and (not (tAdjNodes[i][2] == -1)) then
            tAdjNodes[i][2]=-1
            table.insert(PetAI.tPathFunc.tBlockedNodes,PetAI.tPathFunc.GetNodeIndex(tAdjNodes[i]))
        end
    end
end

//-----------------------------------------------------------------
// BlockAdjacentNodesExceptEast
//-----------------------------------------------------------------
function tPathFunc.BlockAdjacentNodesExceptEast(tNode)
    local tAdjNodes = PetAI.tPathFunc.GetAdjacentNodes(tNode)
    for i in ipairs(tAdjNodes) do
        if (not (i == 5)) and (not (tAdjNodes[i][2] == -1)) then
            tAdjNodes[i][2]=-1
            table.insert(PetAI.tPathFunc.tBlockedNodes,PetAI.tPathFunc.GetNodeIndex(tAdjNodes[i]))
        end
    end
end

//-----------------------------------------------------------------
// UnBlockAdjacentNodes
//-----------------------------------------------------------------
function tPathFunc.UnBlockAdjacentNodes(tNode)
    for i in ipairs(PetAI.tPathFunc.tBlockedNodes) do
        local tIndex = PetAI.tPathFunc.tBlockedNodes[i]
        PetAI.tPathFunc.tPathMap[tIndex[1]][tIndex[2]][2] = 0
    end
end

return(tPathFunc)