tThinkFunc = {}

//-----------------------------------------------------------------
// Species Checking Functions
//-----------------------------------------------------------------
function tThinkFunc.IsDog() return(PetAI.szSpecies == "Dog") end
function tThinkFunc.IsCat() return(PetAI.szSpecies == "Cat") end

//-----------------------------------------------------------------
// IsInAttnMode
//-----------------------------------------------------------------
function tThinkFunc.IsInAttnMode()
  return(PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE))
end

//-----------------------------------------------------------------
// IsTrick
//-----------------------------------------------------------------
function tThinkFunc.IsTrick(szActionName)
  local tTricks = {"Promenade","Sit","LieDown","Speak","GivePaw","Beg","RollOver","Hide","NodHead","ShakeHead","Jump","StandOnHindLegs"}
  for i = 1,table.getn(tTricks) do if (tTricks[i] == szActionName) then return(true) end end
  return(false)
end

//-----------------------------------------------------------------
// IsSleepIdle
//-----------------------------------------------------------------
function tThinkFunc.IsSleepIdle()
	if (type(PetAI.vCurAnim) == "string") then
		return(not (string.find(PetAI.vCurAnim,"SleepIdle") == nil))
	else return(false) end
end

//-----------------------------------------------------------------
// IsIdle
//-----------------------------------------------------------------
function tThinkFunc.IsIdle()
	if (type(PetAI.vCurAnim) == "string") then
		return(not (string.find(PetAI.vCurAnim,"Idle") == nil))
	elseif (type(PetAI.vCurAnim) == "table") then

	  local bBodyIdle = false
	  local bHeadIdle = false

		if (PetAI.vCurAnim.szBody == nil) then bBodyIdle = true
		else bBodyIdle = (not (string.find(PetAI.vCurAnim.szBody,"Idle") == nil)) end

	  if (PetAI.vCurAnim.szHead == nil) then bHeadIdle = true
		else bHeadIdle = (not (string.find(PetAI.vCurAnim.szHead,"Idle") == nil)) end

	  return(bBodyIdle and bHeadIdle)

		--if (PetAI.vCurAnim.szBody == nil) then return(true)
		--else return(not (string.find(PetAI.vCurAnim.szBody,"Idle") == nil)) end
	else
		return(false)
	end
end

//-----------------------------------------------------------------
// UpdateNeedLabels
//-----------------------------------------------------------------
function tThinkFunc.UpdateNeedLabels()

  local hGreen = 0xFF42FF39
  local hYellow = 0xFFFFFF39
  local hRed = 0xFFFF3939
  local hBlack = 0x0
  local tLabelObject = nil

  for k,v in pairs(PetAI.tNeeds) do
    if (v.szName != "Bladder") then
      tLabelObject = Room.stats_dialog["m_k"..v.szName.."Label"]
      if (v.iCurValue < v.iLowThreshold) then tLabelObject:SetColor(hRed)
      else tLabelObject:SetColor(hBlack) end
    end
  end
end


//-----------------------------------------------------------------
// PrintAnim
//-----------------------------------------------------------------
function tThinkFunc.PrintAnim(vAnim)

	local szAnim = ""

	if (vAnim == nil) then
		szAnim = "INVALID ANIM"
	elseif (type(vAnim) == "string") then
		szAnim = "Full Body Anim = "..vAnim
	elseif (type(vAnim) == "table") then
		szAnim = "Masked Anim = "
		local tPieceTable = {}
		if not (vAnim.szHead == nil) then table.insert(tPieceTable,vAnim.szHead) end
		if not (vAnim.szLEar == nil) then table.insert(tPieceTable,vAnim.szLEar) end
		if not (vAnim.szREar == nil) then table.insert(tPieceTable,vAnim.szREar) end
		if not (vAnim.szTail == nil) then table.insert(tPieceTable,vAnim.szTail) end
		if not (vAnim.szBody == nil) then table.insert(tPieceTable,vAnim.szBody) end
		if not (vAnim.szTongue == nil) then table.insert(tPieceTable,vAnim.szTongue) end
		if not (vAnim.szEye == nil) then table.insert(tPieceTable,vAnim.szEye) end

		for i in ipairs(tPieceTable) do
			szAnim = szAnim..tPieceTable[i]
			if not (i == table.getn(tPieceTable)) then szAnim = szAnim..", " end
		end
	end

	if (vAnim == nil) then
		print("INVALID ANIM")
	elseif (type(vAnim) == "string") then
		print("Full Body Anim = "..vAnim)
	elseif (type(vAnim) == "table") then
		if not (vAnim.szHead == nil) then print("Head = "..vAnim.szHead) end
		if not (vAnim.szLEar == nil) then print("LEar = "..vAnim.szLEar) end
		if not (vAnim.szREar == nil) then print("REar = "..vAnim.szREar) end
		if not (vAnim.szTail == nil) then print("Tail = "..vAnim.szTail) end
		if not (vAnim.szBody == nil) then print("Body = "..vAnim.szBody) end
		if not (vAnim.szTongue == nil) then print("Tongue = "..vAnim.szTongue) end
		if not (vAnim.szEye == nil) then print("Eye = "..vAnim.szEye) end
	end

	return(szAnim)
end

//-----------------------------------------------------------------
// DoAnimSwitch
//-----------------------------------------------------------------
function tThinkFunc.DoAnimSwitch(szBodyRegion,szNewAnim,fTime, bContinue)

	if (bContinue == nil) then bContinue = false end

	-- if the current anim is NOT a full-body anim...
	if (type(PetAI.vCurAnim) == "table") then

		-- grab the current animation set for particular body region
		local szCurRegionAnim = PetAI.vCurAnim["sz"..szBodyRegion]

		-- if the new and current animations match, return immediately
		if (szCurRegionAnim == szNewAnim) then return end

		-- ************************************************************
		-- Blend Out Current Anim (if any)
		-- ************************************************************
		if not (szCurRegionAnim == nil) then
			DoEvent("Pet"..".BlendOut("..szCurRegionAnim..","..fTime..")", ActivePetChannel)
		end

		-- ************************************************************
		-- Blend In New Anim (if any)
		-- ************************************************************
		if not (szNewAnim == "CLEAR") then
			-- if the new animation is NOT set to "CLEAR", blend it in
			DoEvent("Pet"..".BlendIn("..szNewAnim..","..fTime..")", ActivePetChannel)
			PetAI.vCurAnim["sz"..szBodyRegion] = szNewAnim
		else
			-- otherwise, set the current animation to nil
			PetAI.vCurAnim["sz"..szBodyRegion] = nil
		end
	end
end

//-----------------------------------------------------------------
// LoadNewAnim
//-----------------------------------------------------------------
function tThinkFunc.LoadNewAnim(vNewAnim,fTime,bContinue)
--[[
	print("CURRENT ANIM-------------------------------------------")
	PetAI.tThinkFunc.PrintAnim(PetAI.vCurAnim)
	print("-------------------------------------------------------")
	print("NEW ANIM***********************************************")
	PetAI.tThinkFunc.PrintAnim(vNewAnim)
	print("*******************************************************")
]]--

  local szPose = "Standing"
  if (PetAI.tThinkFunc.IsSitting()) then szPose = "Sitting"
  elseif (PetAI.tThinkFunc.IsLyingDown()) then szPose = "LyingDown" end

	if (bContinue == nil) then bContinue = false end

	if (type(vNewAnim) == "string") then
		-- ************************************************************
		-- Process New Full-Body Anim
		-- ************************************************************
		PetAI.tThinkFunc.BlendOutCurAnim(fTime,bContinue)

    if (vNewAnim != "CLEAR") then
  		DoEvent("Pet"..".BlendIn("..vNewAnim..","..fTime..")", ActivePetChannel, bContinue)
  		PetAI.vCurAnim = vNewAnim
  	else
  		PetAI.vCurAnim = nil
  	end

	elseif (type(vNewAnim) == "table") then
		-- ************************************************************
		-- Process New Body Region-Specific Anim
		-- ************************************************************

		-- if the current anim is full-body, dump it and reset it to a table
    if (PetAI.vCurAnim == nil) then
			PetAI.vCurAnim = PetAI.tThinkFunc.GetNewAnimTable()
      PetAI.tThinkFunc.DoAnimSwitch("Body","BodyIdle",fTime, bContinue)
		elseif (type(PetAI.vCurAnim) == "string") then
			PetAI.tThinkFunc.BlendOutCurAnim(fTime, bContinue)
			PetAI.vCurAnim = PetAI.tThinkFunc.GetNewAnimTable()
			if (vNewAnim.szBody == nil) then
        if (szPose == "Standing") then PetAI.tThinkFunc.DoAnimSwitch("Body","BodyIdle",fTime, bContinue)
        elseif (szPose == "Sitting") then PetAI.tThinkFunc.DoAnimSwitch("Body","BodySitIdle",fTime, bContinue)
        elseif (szPose == "LyingDown") then PetAI.tThinkFunc.DoAnimSwitch("Body","BodyLayIdle",fTime, bContinue) end
			end
		end

		-- process any included body region-specific anim components
		if not (vNewAnim.szHead == nil) then PetAI.tThinkFunc.DoAnimSwitch("Head",vNewAnim.szHead,fTime, bContinue) end
		if not (vNewAnim.szLEar == nil) then PetAI.tThinkFunc.DoAnimSwitch("LEar",vNewAnim.szLEar,fTime, bContinue) end
		if not (vNewAnim.szREar == nil) then PetAI.tThinkFunc.DoAnimSwitch("REar",vNewAnim.szREar,fTime, bContinue) end
		if not (vNewAnim.szTail == nil) then PetAI.tThinkFunc.DoAnimSwitch("Tail",vNewAnim.szTail,fTime, bContinue) end
		if not (vNewAnim.szBody == nil) then PetAI.tThinkFunc.DoAnimSwitch("Body",vNewAnim.szBody,fTime, bContinue) end
		if not (vNewAnim.szTongue == nil) then PetAI.tThinkFunc.DoAnimSwitch("Tongue",vNewAnim.szTongue,fTime, bContinue) end
		if not (vNewAnim.szEye == nil) then PetAI.tThinkFunc.DoAnimSwitch("Eye",vNewAnim.szEye,fTime,bContinue) end
	end

	if (PetAI.tPettingFunc.CheckPoseChange()) then PetAI.tPettingFunc.ResetTrickLadder() end
	PetAI.tPettingFunc.SetCurPose()

end

//-----------------------------------------------------------------
// GetNewAnimTable
//-----------------------------------------------------------------
function tThinkFunc.GetNewAnimTable()
	local tNewAnim = {
		szHead = nil,
		szLEar = nil,
		szREar = nil,
		szTail = nil,
		szBody = nil,
		szTongue = nil,
		szEye = nil
	}
	return(tNewAnim)
end

//-----------------------------------------------------------------
// BlendOutCurAnim
//-----------------------------------------------------------------
function tThinkFunc.BlendOutCurAnim(fTime,bContinue)
--[[
	print("BLEND OUT ANIM-----------------------------------------")
	PetAI.tThinkFunc.PrintAnim(PetAI.vCurAnim)
	print("-------------------------------------------------------")
]]--
	if (type(PetAI.vCurAnim) == "string") then
		DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim..","..fTime..")", ActivePetChannel, bContinue)
	elseif (type(PetAI.vCurAnim) == "table") then
		if not (PetAI.vCurAnim.szHead == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szHead..","..fTime..")", ActivePetChannel, bContinue) end
		if not (PetAI.vCurAnim.szLEar == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szLEar..","..fTime..")", ActivePetChannel, bContinue) end
		if not (PetAI.vCurAnim.szREar == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szREar..","..fTime..")", ActivePetChannel, bContinue) end
		if not (PetAI.vCurAnim.szTail == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szTail..","..fTime..")", ActivePetChannel, bContinue) end
		if not (PetAI.vCurAnim.szBody == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szBody..","..fTime..")", ActivePetChannel, bContinue) end
		if not (PetAI.vCurAnim.szTongue == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szTongue..","..fTime..")", ActivePetChannel, bContinue) end
		if not (PetAI.vCurAnim.szEye == nil) then DoEvent("Pet"..".BlendOut("..PetAI.vCurAnim.szEye..","..fTime..")", ActivePetChannel, bContinue) end
	end
end

//-----------------------------------------------------------------
// DoAnimTransition
//-----------------------------------------------------------------
function tThinkFunc.DoAnimTransition(vTransAnim, vGoalAnim, fTime)

    local fTimeToSleep = fTime
    --if (fTimeToSleep <= 1000) then fTimeToSleep = 1000
    --else fTimeToSleep = fTimeToSleep - 1000 end

    if (fTimeToSleep > 1000) then fTimeToSleep = fTimeToSleep - 1000 end

    PetAI.tThinkFunc.LoadNewAnim(vTransAnim,0.5)
    DoEvent("System.Sleep("..fTimeToSleep..")", ActivePetChannel)
    PetAI.tThinkFunc.LoadNewAnim(vGoalAnim,0.5)
    DoEvent("System.Sleep(500)", ActivePetChannel)
end

//-----------------------------------------------------------------
// DoSitAndSingleAnim
//-----------------------------------------------------------------
function tThinkFunc.DoSitAndSingleAnim(vAnim,szAnimSoundName,iAnimTime)
	PetAI.tThinkFunc.DoSit()
  PetAI.tThinkFunc.DoSingleAnim(vAnim,szAnimSoundName,iAnimTime)
end

//-----------------------------------------------------------------
// DoLieDownAndSingleAnim
//-----------------------------------------------------------------
function tThinkFunc.DoLieDownAndSingleAnim(vAnim,szAnimSoundName,iAnimTime)
	PetAI.tThinkFunc.DoLieDown()
  PetAI.tThinkFunc.DoSingleAnim(vAnim,szAnimSoundName,iAnimTime)
end

//-----------------------------------------------------------------
// DoStandAndSingleAnim
//-----------------------------------------------------------------
function tThinkFunc.DoStandAndSingleAnim(vAnim,szAnimSoundName,iAnimTime)
	PetAI.tThinkFunc.DoStand()
  PetAI.tThinkFunc.DoSingleAnim(vAnim,szAnimSoundName,iAnimTime)
end

//-----------------------------------------------------------------
// DoSingleAnim
//-----------------------------------------------------------------
function tThinkFunc.DoSingleAnim(vAnim,szAnimSoundName,iAnimTime)
	if not (szAnimSoundName == nil) then DoLuaEvent("PetAI.tSoundFX[\""..szAnimSoundName.."\"]:Play()",ActivePetChannel) end
	PetAI.tThinkFunc.LoadNewAnim(vAnim,0.25)
	DoEvent("System.Sleep("..math.floor(iAnimTime)..")",ActivePetChannel)
	PetAI.tThinkFunc.DoIdle()
end

//-----------------------------------------------------------------
// DoIdle
//-----------------------------------------------------------------
function tThinkFunc.DoIdle(fTime, bContinue)

	if (fTime == nil) then fTime = 0.25 end
	if (bContinue == nil) then bContinue = false end

	local tNewAnim = { szBody = nil }

	if PetAI.tThinkFunc.IsStanding() then tNewAnim.szBody = "BodyIdle"
	elseif PetAI.tThinkFunc.IsSitting() then tNewAnim.szBody = "BodySitIdle"
	elseif PetAI.tThinkFunc.IsLyingDown() then tNewAnim.szBody = "BodyLayIdle"
	end

	tNewAnim.szHead="HeadIdle"
	tNewAnim.szTail="TailIdle"
  tNewAnim.szEye="CLEAR"

	PetAI.tThinkFunc.LoadNewAnim(tNewAnim,fTime,bContinue)
end


//-----------------------------------------------------------------
// DoStand
//-----------------------------------------------------------------
function tThinkFunc.DoStand()
	if PetAI.tThinkFunc.IsStanding() then
		return
	elseif PetAI.tThinkFunc.IsSitting() then
		PetAI.tThinkFunc.DoAnimTransition("SitGetUp",{szBody="BodyIdle",szHead="HeadIdle",szTail="TailIdle"},pet:GetAnimDuration("SitGetUp")*1000)
	elseif PetAI.tThinkFunc.IsLyingDown() then
		PetAI.tThinkFunc.DoAnimTransition("LayGetUp",{szBody="BodyIdle",szHead="HeadIdle",szTail="TailIdle"},pet:GetAnimDuration("LayGetUp")*1000)
	end
end

//-----------------------------------------------------------------
// DoSit
//-----------------------------------------------------------------
function tThinkFunc.DoSit()
	if PetAI.tThinkFunc.IsSitting() then
		return
	elseif PetAI.tThinkFunc.IsStanding() then
		PetAI.tThinkFunc.DoAnimTransition("Sit",{szBody="BodySitIdle"},pet:GetAnimDuration("Sit")*1000)
	elseif PetAI.tThinkFunc.IsLyingDown() then
		PetAI.tThinkFunc.DoAnimTransition("Sit",{szBody="BodySitIdle"},pet:GetAnimDuration("Sit")*1000)
	end
end

//-----------------------------------------------------------------
// DoLieDown
//-----------------------------------------------------------------
function tThinkFunc.DoLieDown()
	if PetAI.tThinkFunc.IsLyingDown() then
		return
	elseif PetAI.tThinkFunc.IsStanding() then
		PetAI.tThinkFunc.DoAnimTransition("LayDown",{szBody="BodyLayIdle"},pet:GetAnimDuration("LayDown")*1000)
	elseif PetAI.tThinkFunc.IsSitting() then
		if (PetAI.tBreed.szName == "Pug") then PetAI.tThinkFunc.DoStand() end
		PetAI.tThinkFunc.DoAnimTransition("LayDown",{szBody="BodyLayIdle"},pet:GetAnimDuration("LayDown")*1000)
	end
end

//-----------------------------------------------------------------
// IsStanding
//-----------------------------------------------------------------
function tThinkFunc.IsStanding()
	return((not PetAI.tThinkFunc.IsSitting()) and (not PetAI.tThinkFunc.IsLyingDown()))
end

//-----------------------------------------------------------------
// IsSitting
//-----------------------------------------------------------------
function tThinkFunc.IsSitting()
	if (PetAI.vCurAnim == nil) then return false end

	local szAnimToCheck = ""
	if (type(PetAI.vCurAnim) == "string") then szAnimToCheck = PetAI.vCurAnim
	elseif (type(PetAI.vCurAnim) == "table") and (not (PetAI.vCurAnim.szBody == nil)) then szAnimToCheck = PetAI.vCurAnim.szBody
	else return(false) end

	return(not (string.find(szAnimToCheck,"Sit") == nil))
end

//-----------------------------------------------------------------
// IsLyingDown
//-----------------------------------------------------------------
function tThinkFunc.IsLyingDown()
	if (PetAI.vCurAnim == nil) then return false end

	local szAnimToCheck = ""
	if (type(PetAI.vCurAnim) == "string") then szAnimToCheck = PetAI.vCurAnim
	elseif (type(PetAI.vCurAnim) == "table") and (not (PetAI.vCurAnim.szBody == nil)) then szAnimToCheck = PetAI.vCurAnim.szBody
	else return(false) end

	return(not (string.find(szAnimToCheck,"Lay") == nil))
end

//-----------------------------------------------------------------
// DoTurnLeft
//-----------------------------------------------------------------
function tThinkFunc.DoTurnLeft()
  local fTurnTarget = 0

  if (PetAI.fCurOrientation == 0) then PetAI.fCurOrientation = 360 fTurnTarget = 300
  elseif (PetAI.fCurOrientation == 300) then fTurnTarget = 180
  elseif (PetAI.fCurOrientation == 180) then fTurnTarget = 300
  elseif (PetAI.fCurOrientation == 60) then fTurnTarget = 0 end

  local fTurnAmount = fTurnTarget - PetAI.fCurOrientation
  PetAI.fCurOrientation = fTurnTarget

  if (math.abs(PetAI.fCurOrientation) == 360.0) then PetAI.fCurOrientation = 0 end

	DoEvent("Pet.TurnInPlace(5, "..fTurnAmount..")", ActivePetChannel)
	DoLuaEvent("PetAI.tThinkFunc.DoIdle(0,TRUE)",ActivePetChannel)
end

//-----------------------------------------------------------------
// DoTurnRight
//-----------------------------------------------------------------
function tThinkFunc.DoTurnRight()
  local fTurnTarget = 0

  if (PetAI.fCurOrientation == 0) then fTurnTarget = 60
  elseif (PetAI.fCurOrientation == 60) then fTurnTarget = 180
  elseif (PetAI.fCurOrientation == 180) then fTurnTarget = 60
  elseif (PetAI.fCurOrientation == 300) then fTurnTarget = 360 end

  local fTurnAmount = fTurnTarget - PetAI.fCurOrientation
  PetAI.fCurOrientation = fTurnTarget

  if (math.abs(PetAI.fCurOrientation) == 360.0) then PetAI.fCurOrientation = 0 end

	DoEvent("Pet.TurnInPlace(5, "..fTurnAmount..")", ActivePetChannel)
	DoLuaEvent("PetAI.tThinkFunc.DoIdle(0,TRUE)",ActivePetChannel)
end

//-----------------------------------------------------------------
// TurnForPromenade
//-----------------------------------------------------------------
function tThinkFunc.TurnForPromenade()
  local fTurnAmount = 60 - PetAI.fCurOrientation
  PetAI.fCurOrientation = 60

	DoEvent("Pet.TurnInPlace(5, "..fTurnAmount..")", ActivePetChannel)
end

//-----------------------------------------------------------------
// TransitionTo
//-----------------------------------------------------------------
function tThinkFunc.TransitionTo(szRoom)
	PetAI.tThinkFunc.LoadNewAnim({szBody = "Walk1"},0.5)
	PetAI.tPathFunc.WalkToPathNode(PetAI.tPathFunc.tRooms[szRoom].GetHomePosition(),"BodyWalk","BodyIdle")
	DoLuaEvent("PetAI.szLocation = \""..szRoom.."\"",ActivePetChannel)
end

//-----------------------------------------------------------------
// AddPetAI.tThinkFunc.TurnRight
//-----------------------------------------------------------------
function tThinkFunc.AddTurnRight()
	PetAI.tThinkFunc.LoadNewAnim({szHead="TurnRight"},1)
end

//-----------------------------------------------------------------
// AddPetAI.tThinkFunc.TurnLeft
//-----------------------------------------------------------------
function tThinkFunc.AddTurnLeft()
	PetAI.tThinkFunc.LoadNewAnim({szHead="TurnLeft"},1)
end

//-----------------------------------------------------------------
// AdjustMax
//-----------------------------------------------------------------
function tThinkFunc.AdjustMax(tAttrib, fChange)

	-- adjust max value
	tAttrib.iMaxValue = tAttrib.iMaxValue + fChange

	-- clamp to max value to range 0..100
	if (tAttrib.iMaxValue < 0) then tAttrib.iMaxValue = 0
	elseif (tAttrib.iMaxValue > 100) then tAttrib.iMaxValue = 100 end
end

//-----------------------------------------------------------------
// SetValue
//-----------------------------------------------------------------
function tThinkFunc.SetValue(tAttrib,fChange)

	-- if the attribute is currently disabled, return immediately
	if not tAttrib.bEnabled then return end

	if ((fChange > 0) and not (tAttrib.tSatiateModifiers == nil)) then

		-- if change is positive and satiate modifiers exist, apply them to the change value
		fChange = PetAI.tThinkFunc.ApplyModifiers(tAttrib.tSatiateModifiers,fChange)

	elseif ((fChange < 0) and not (tAttrib.tDepleteModifiers == nil)) then

		-- if change is negative and deplete modifiers exist, apply them to the change value
		fChange = PetAI.tThinkFunc.ApplyModifiers(tAttrib.tDepleteModifiers,fChange)
	end

	-- adjust current value
	tAttrib.iCurValue = tAttrib.iCurValue + fChange

	-- clamp to bounds
	PetAI.tThinkFunc.PreserveBounds(tAttrib)

	-- check for critical value states and effects on other attributes
	PetAI.tThinkFunc.CheckDepress(tAttrib)
end

//-----------------------------------------------------------------
// CreatePercentageModifier
//-----------------------------------------------------------------
function tThinkFunc.CreatePercentageModifier(fPercentage)

	-- return a function that adjusts a given value by the specified percentage
	return(function (fValue) return(fValue + fPercentage*fValue) end)
end

//-----------------------------------------------------------------
// SetBreed
//-----------------------------------------------------------------
function tThinkFunc.SetBreed(szBreedName)

	-- store breed data table reference
	if (PetAI.szSpecies == "Dog") then
 		PetAI.tBreed = PetAI.tBreeds.tDog[szBreedName]
 	else
 		PetAI.tBreed = PetAI.tBreeds.tCat[szBreedName]
 	end

	-- set initial max/cur values for pet statistics using breed defaults
	for k,v in pairs(PetAI.tStats) do
		v.iMaxValue = PetAI.tBreed.tInitStats["iInit" .. v.szName]
		v.iCurValue = v.iMaxValue
	end

	-- for each pet need...
	for k,v in pairs(PetAI.tNeeds) do

		-- retrieve satiate modifier value
		fSatiateModifier = PetAI.tBreed.tNeedSatiateModifiers["f" .. v.szName .. "Mod"]

		-- if the satiate modifier value is non-zero use it to create a satiate modifier
		if (not ((fSatiateModifier*100.0) == 0.0)) then
			table.insert(v.tSatiateModifiers,PetAI.tThinkFunc.CreatePercentageModifier(fSatiateModifier))
		end

		-- retrieve deplete modifier value
		fDepleteModifier = PetAI.tBreed.tNeedDepleteModifiers["f" .. v.szName .. "Mod"]

		-- if the deplete modifier value is non-zero user it to create a deplete modifier
		if (not ((fDepleteModifier*100.0) == 0.0)) then
			table.insert(v.tDepleteModifiers,PetAI.tThinkFunc.CreatePercentageModifier(fDepleteModifier))
		end
	end
end

//-----------------------------------------------------------------
// ApplyModifiers
//-----------------------------------------------------------------
function tThinkFunc.ApplyModifiers(tModifierSet, fValue)

	-- multiply given value by each of the modifiers in the given set
	for i in ipairs(tModifierSet) do
		fValue = tModifierSet[i](fValue)
	end

	-- return resulting value
	return(fValue)
end

//-----------------------------------------------------------------
// PrintModifiers
//-----------------------------------------------------------------
function tThinkFunc.PrintModifiers(tAttrib)

		-- print given attribute name
		print("Attrib: " .. tAttrib.szName)

		-- if attribute has any satiate modifiers...
		if table.getn(tAttrib.tSatiateModifiers) > 0 then

			-- print title and modifier count
			print("Satiate Modifiers")
			print("NumModifiers = %d" .. table.getn(tAttrib.tSatiateModifiers))

			-- print out each satiate modifier, demonstrating its impact on a base value of 100
			for i in ipairs(tAttrib.tSatiateModifiers) do
				print("Modifier Effect: funcModifier(100) =  " .. tAttrib.tSatiateModifiers[i](100))
			end
		end

		-- if attribute has any deplete modifiers...
		if table.getn(tAttrib.tDepleteModifiers) > 0 then

			-- print title and modifier count
			print("Deplete Modifiers")
			print("NumModifiers = %d" .. table.getn(tAttrib.tDepleteModifiers))

			-- print out each deplete modifier, demonstrating its impact on a base value of 100
			for i in ipairs(tAttrib.tDepleteModifiers) do
				print("Modifier Effect: func`Modifier(100) =  " .. tAttrib.tDepleteModifiers[i](100))
			end
		end
end

//-----------------------------------------------------------------
// PreserveBounds
//-----------------------------------------------------------------
function tThinkFunc.PreserveBounds(tAttrib)

	-- clamp given attribute's value to range between zero and max
	if (tAttrib.iCurValue < 0) then tAttrib.iCurValue = 0
	elseif (tAttrib.iCurValue > tAttrib.iMaxValue) then tAttrib.iCurValue = tAttrib.iMaxValue end
end

//-----------------------------------------------------------------
// CheckDecay
//-----------------------------------------------------------------
function tThinkFunc.CheckDecay(tAttribSet,iElapsedTime)

  local bOk = true

	-- for each attribute in the given set...
	for k,v in pairs(tAttribSet) do

		-- if decay for the specific attribute is turned on...
		if v.bDecayOn then

			-- if enough time has passed since last decay event...
			if iElapsedTime - v.iLastDecayTime > v.iDecayRate then

				-- apply decay effect and record time for next iteration
				PetAI.tThinkFunc.SetValue(v,v.iDecayAmount)
				v.iLastDecayTime = iElapsedTime
			end
		end

  	-- if any attribute has fallen below threshold, set boolean return
  	if (v.iCurValue < v.iLowThreshold) then
      if (not v.bCritical) then
  		  v.bCritical = true
  		  if (v.szName != "Bladder") then
  		    Room.stats_dialog:Paint()
  		    Room.bReplayNeedsAlert = true
  		  end
  		end

  		-- for all attributes other than the bladder need, set the return flag
  		if (v.szName != "Bladder") then bOk = false end
    elseif (v.bCritical) then
  	  v.bCritical = false
    end
	end

	return(bOk)
end

//-----------------------------------------------------------------
// CheckDepress
//-----------------------------------------------------------------
function tThinkFunc.CheckDepress(tAttrib)

	-- if the given attribute's current value is less than its lower threshold
	if tAttrib.iCurValue < tAttrib.iLowThreshold then

		-- for each of the attributes depressed by the given attribute's critical state...
		for i,v in ipairs(tAttrib.tDepressAttribs) do

			-- determine the disparity between the current value and threshold, represented as a percentage
			fPercent = (tAttrib.iLowThreshold - tAttrib.iCurValue)/tAttrib.iLowThreshold

			-- set the depressed attribute's value relative to the determined percentage (unless its current value is lower)
			if (v.iCurValue > (v.iMaxValue - fPercent*v.iMaxValue)) then v.iCurValue = v.iMaxValue - fPercent*v.iMaxValue end

      -- if the depressed attribute's value has dropped to zero, disable decay to avoid value thrashing
      if (v.iCurValue <= 0) and (v.bDecayOn) then 
        v.bDecayOn = false 
        if (PetAI["AttribDecayDisabledBy"..tAttrib.szName] == nil) then PetAI["AttribDecayDisabledBy"..tAttrib.szName] = {} end
        table.insert(PetAI["AttribDecayDisabledBy"..tAttrib.szName],v)
      end

			-- clamp the depressed attribute's value to appropriate bounds
			PetAI.tThinkFunc.PreserveBounds(v)
		end
	elseif (tAttrib.iCurValue >= tAttrib.iLowThreshold) and (PetAI["AttribDecayDisabledBy"..tAttrib.szName] != nil) then
	 	
	  -- restore decay to any attributes previously depressed
	  for i,v in ipairs(PetAI["AttribDecayDisabledBy"..tAttrib.szName]) do v.bDecayOn = true end
	  
	  -- destroy the utilized table
	  PetAI["AttribDecayDisabledBy"..tAttrib.szName] = nil
	end
	
end

//-----------------------------------------------------------------
// GiveActiveVoiceCommand
//-----------------------------------------------------------------
function tThinkFunc.GiveActiveVoiceCommand(szCommand)

	-- play voice command sfx
	PetAI.tVoiceCommands[szCommand].cSoundFX:Play()

end

//-----------------------------------------------------------------
// GiveVoiceCommand
//-----------------------------------------------------------------
function tThinkFunc.GiveVoiceCommand(szCommand)

  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.SLEEPING_MODE)) then return end
  if (PetAI.tJumpDownInfo != nil) then return end

	-- do help dialog the second time the first available trick button is pressed
	if (PetAI.MiscHelpInfo.szFirstActiveTrickButton == nil) then PetAI.MiscHelpInfo.szFirstActiveTrickButton = szCommand
	elseif (PetAI.MiscHelpInfo.szFirstActiveTrickButton == szCommand) then PetAI.DoHelpDialog(29,PetAI.MiscHelpInfo.szFirstActiveTrickButton) end

  -- if in attention mode, restore the attention span with each voice command
  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE)) then PetAI.tThinkFunc.RestoreAttentionSpan() end

	-- play voice command sfx
	PetAI.tVoiceCommands[szCommand].cSoundFX:Play()

	-- do help dialog if commands are given too rapidly
	if (not (PetAI.MiscHelpInfo.iLastTrickButtonTime == 0)) and (FoxGetTickCount() - PetAI.MiscHelpInfo.iLastTrickButtonTime < 500) then
		PetAI.DoHelpDialog(39)
		PetAI.tThinkFunc.ChangeAction(PetAI.tActions.CockHeadQuizzically)
		return
	end
	PetAI.MiscHelpInfo.iLastTrickButtonTime = FoxGetTickCount()

	-- if the voice command has not been activated, return immediately
	if not (PetAI.tVoiceCommands[szCommand].bActive) then
		ShowTempRewardButton(szCommand)
		PetAI.bVoiceCommandGiven = true
		return
	elseif (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE)) then
  	PetAI.bVoiceCommandGiven = true
	end

	-- store the trick currently mapped to the given voice command
	local tChosenAction = PetAI.tActions[PetAI.tVoiceCommands[szCommand].szTrick]

  -- check if voice command has activated a 'secret' trick
  local tSecretTrick = PetAI.CheckForSecretTrick(szCommand)

  -- if not a secret trick...
  if (tSecretTrick == nil) then

    -- attempt to perform the normal trick
    PetAI.tThinkFunc.ChangeAction(PetAI.tThinkFunc.AttemptTrick(tChosenAction),true)
  else

    -- attempt to perform the secret trick (following current action execution)
    PetAI.tSecretTrickToDo = PetAI.tThinkFunc.AttemptTrick(tSecretTrick)
  end
end

//-----------------------------------------------------------------
// AttemptTrick
//-----------------------------------------------------------------
function tThinkFunc.AttemptTrick(tTrickAction)

  -- determine whether or not the trick is a 'secret' trick
  local bIsSecretTrick = PetAI.IsSecretTrick(tTrickAction.szName)

	-- perform MEMORY CHECK on chosen action
	if (not PetAI.tThinkFunc.DoMemoryCheck(tTrickAction)) and (not bIsSecretTrick) then

		-- do help dialog
		PetAI.DoHelpDialog(31,tTrickAction.szName)

    -- if in talent show, play failing sfx
    if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.TALENT_SHOW)) then PetAI.tSoundFX.wavTSFail:Play() end

		-- if unsuccessful, select and execute a random memory failure action
		return(PetAI.tActions[PetAI.tThinkFunc.PickRandomAction(tTrickAction.tMemoryFailureActions)])
	end

	-- perform OBEDIENCE CHECK on chosen action
	if not PetAI.tThinkFunc.DoObedienceCheck(tTrickAction) and (not bIsSecretTrick) then

		-- do help dialog
		if (PetAI.tNeeds.tGrooming.iCurValue < PetAI.tNeeds.tGrooming.iLowThreshold) then PetAI.DoHelpDialog(35)
		elseif (PetAI.tNeeds.tLove.iCurValue < PetAI.tNeeds.tLove.iLowThreshold) then PetAI.DoHelpDialog(34)
		elseif (PetAI.tNeeds.tAlertness.iCurValue < PetAI.tNeeds.tAlertness.iLowThreshold) then PetAI.DoHelpDialog(36)
		else PetAI.DoHelpDialog(37) end

    -- if in talent show, play failing sfx
    if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.TALENT_SHOW)) then PetAI.tSoundFX.wavTSLaugh:Play() end

		-- if unsuccessful, select and execute a random obedience failure action
		return(PetAI.tActions[PetAI.tThinkFunc.PickRandomAction(tTrickAction.tObedienceFailureActions)])
	end

	-- perform AGILITY CHECK on chosen action
	if not PetAI.tThinkFunc.DoAgilityCheck(tTrickAction) then

    
		-- do help dialog
		if (not bIsSecretTrick) then
  		if (PetAI.tNeeds.tHunger.iCurValue < PetAI.tNeeds.tHunger.iLowThreshold) then PetAI.DoHelpDialog(32)
  		elseif (PetAI.tNeeds.tThirst.iCurValue < PetAI.tNeeds.tThirst.iLowThreshold) then PetAI.DoHelpDialog(33)
  		else PetAI.DoHelpDialog(38,tTrickAction.szName) end
    end

	  return(PetAI.tActions[PetAI.tThinkFunc.PickRandomAction(tTrickAction.tAgilityFailureActions)])
	end

	return(tTrickAction)
end

//-----------------------------------------------------------------
// GetIdleAction
//-----------------------------------------------------------------
function tThinkFunc.GetIdleAction(tIdleSet)

	-- for each of the pet's needs...
	for k,v in pairs(PetAI.tNeeds) do

		-- if need found to be in critical value state
		if v.iCurValue < v.iLowThreshold then

			-- select and execute a need-based emergency action
			PetAI.tThinkFunc.ChangeAction(PetAI.tActions[PetAI.tThinkFunc.PickRandomAction(v.tEmergencyActions)]);
			return
		end
	end

	-- if all needs are ok, select and execute a normal idle action
	PetAI.tThinkFunc.ChangeAction(PetAI.tActions[PetAI.tThinkFunc.PickRandomAction(PetAI.tThinkFunc.CreateSubTableByUseCount(tIdleSet))]);
end

//-----------------------------------------------------------------
// CreateSubTableByPrecondition
//-----------------------------------------------------------------
function tThinkFunc.CreateSubTableByPrecondition(tTable)

	-- declare a local table to contain subset of given table
	local tSubTable = {}

	-- process given table, inserting all entries passing their preconditions into the subtable
	for i in ipairs(tTable) do
		if (PetAI.tActions[tTable[i]].CheckPrecondition()) then table.insert(tSubTable,tTable[i]) end
	end

	-- returned the created subtable
	return(tSubTable)
end

//-----------------------------------------------------------------
// CreateSubTableByUseCount
//-----------------------------------------------------------------
function tThinkFunc.CreateSubTableByUseCount(tTable)

	-- declare a local table to contain subset of given table
	local tSubTable = {}

	-- process given table, inserting all entries passing their preconditions into the subtable
	for i in ipairs(tTable) do
		if (PetAI.tThinkFunc.CheckUseCount(tTable[i])) then table.insert(tSubTable,tTable[i]) end
	end

	-- returned the created subtable
	return(tSubTable)
end

//-----------------------------------------------------------------
// PickRandomAction
//-----------------------------------------------------------------
function tThinkFunc.PickRandomAction(tTable)

	-- if the table is empty, return error
	if (table.getn(tTable) == 0) then return(nil) end

  -- reduce the table's actions by removing any whose preconditions fail
  local tSubTable = PetAI.tThinkFunc.CreateSubTableByPrecondition(tTable)

  local fMemMax = 0
  local tMemTable = {}

  -- for each possible action...
  for i,v in ipairs(tSubTable) do

    -- add to total mem value sum of all possible actions
    fMemMax = fMemMax + PetAI.tActions[v].fMemValue

    -- insert the action's name and mem value into the table
    tMemTable[v] = PetAI.tActions[v].fMemValue
  end

  -- calculate a random number between 1 and mem value sum
  local iRand = math.random(0,fMemMax)

  -- for each possible action in the mem table...
  for k,v in pairs(tMemTable) do

      -- if the random value is less than the action's mem value...
    if (iRand <= v) then
      -- return the action
      return(k)
    else
      -- otherwise, subtract the action's mem value from the sum and continue
      iRand = iRand - v
    end
  end
end

//-----------------------------------------------------------------
// PickRandom
//-----------------------------------------------------------------
function tThinkFunc.PickRandom(tTable)

	-- if table is empty, return error
	if (table.getn(tTable) == 0) then return(nil) end

	-- return a randomly-selected entry from the given table
	return tTable[math.random(table.getn(tTable))]
end

//-----------------------------------------------------------------
// DoMemoryCheck
//-----------------------------------------------------------------
function tThinkFunc.DoMemoryCheck(tAction)

	-- calculate random number between 0 and 100
	local iRandNum = math.random(100)

	-- retrieve current memory value for proposed action
	local iCurMemory = tAction.fMemValue

	-- if the pet is in attention mode, add any memory bonus resulting from petting sequences
	if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE)) then
		iCurMemory = iCurMemory + PetAI.fMemBonus
	end

	-- if random number is less than current memory value, return success
	if (iRandNum <= iCurMemory) then return true
	else return false end
end

//-----------------------------------------------------------------
// DoObedienceCheck
//-----------------------------------------------------------------
function tThinkFunc.DoObedienceCheck(tAction)

	-- calculate random number between 0 and 100
	local iRandNum = math.random(100)

	-- retrieve pet's current obedience value
	local iCurObedience = PetAI.tStats.tObedience.iCurValue

	-- if sum of two values exceeds the proposed action's obedience difficulty value, return success
	if (iRandNum + iCurObedience) >= tAction.iObDifficulty then return true
	else return false end
end

//-----------------------------------------------------------------
// DoAgilityCheck
//-----------------------------------------------------------------
function tThinkFunc.DoAgilityCheck(tAction)

	-- calculate random number between 0 and 200
	local iRandNum = math.random(200)

	-- retrieve pet's current agility value
	local iCurAgility = PetAI.tStats.tAgility.iCurValue

	-- if sum of two values exceeds the proposed action's agility difficulty value, return success
	if (iRandNum + iCurAgility) >= tAction.iAgDifficulty then return true
	else return false end
end

//-----------------------------------------------------------------
// TableSize
//-----------------------------------------------------------------
function tThinkFunc.TableSize(t)
	local i = 0
	for k in pairs(t) do i = i + 1 end
	return i
end

//-----------------------------------------------------------------
// ProcessEffects
//-----------------------------------------------------------------
function tThinkFunc.ProcessEffects(tEffector,bIncludeStats)

  -- set default parameters
  if (bIncludeStats == nil) then bIncludeStats = true end

	-- use local action reference
	local tAction = nil

	-- set the action reference to either the effector itself or its valid usage action (if an item)
	if (tEffector.szUsageAction == nil) then tAction = tEffector
	else tAction = PetAI.tActions[tEffector.szUsageAction] end

  -- if stats included in effect processing...
  if (bIncludeStats) then

  	-- for each of the pet's stats...
  	for k,v in pairs(PetAI.tStats) do

  		-- retrieve the effector's specified modifier value for the particular stat
  		local iMaxMod = tEffector.tAttribModifiers["i" .. v.szName .. "Mod"]

  		if not (iMaxMod == 0) then
  		  -- if the modifier value is non-zero, create a time-release additive stat effect
  			local StatEffect = PetAI.tEffects.CreateStatTimeReleaseEffect(tAction,FoxGetTickCount(),v,iMaxMod,2000)

				-- insert the newly-created effect into the current active set
				table.insert(PetAI.tActiveEffects,StatEffect)
		  end
	  end
  end

	-- for each of the pet's needs...
	for k,v in pairs(PetAI.tNeeds) do

		-- retrieve the effector's specified modifier value for the particular need
		local iMod = tEffector.tAttribModifiers["i" .. v.szName .. "Mod"]

		if not (iMod == 0) then

			-- if the modifier value is non-zero, create an additive need effect to be executed over a particular duration
			local NeedEffect = PetAI.tEffects.CreateNeedOverTimeEffect(tAction,FoxGetTickCount(),v,iMod,500)

			-- insert the newly-created effect into the current active set
			table.insert(PetAI.tActiveEffects,NeedEffect)
		end
	end

	-- if the effector includes any special effects...
	if not (tEffector.tSpecialEffects == nil) then

		-- execute each of the special effect functions stored in the effector's table
		for i in ipairs(tEffector.tSpecialEffects) do tEffector.tSpecialEffects[i]() end
	end
end

//-----------------------------------------------------------------
// GiveItem
//-----------------------------------------------------------------
function tThinkFunc.GiveItem(tItem)

  local bUsageActionSuccess = false

  -- if currently sleeping, return immediately
  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.SLEEPING_MODE)) then return end

	-- execute the item's usage action
	bUsageActionSuccess = PetAI.tThinkFunc.ChangeAction(PetAI.tActions[tItem.szUsageAction])

  -- if the usage action is not utilized...
  if (not bUsageActionSuccess) then

    -- if the pet is eating a treat, just play the eating sfx
    if (tItem.szUsageAction == "EatTreat") then
    	local nNumTreats = 0
	    for k, v in pairs(player.m_cTreats) do nNumTreats = nNumTreats + v end

      -- if the player has any treats, play the eating sfx in lieu of any animation
      if (nNumTreats > 0) then PetAI.tSoundFX.wavPetEatTreat:Play() end
    end
  end

  if (tItem.bSingleUse) then
  	for k,v in pairs(PetAI.tStats) do
  	  local iMod = tItem.tAttribModifiers["i" .. v.szName .. "Mod"]
  	  if (iMod != 0) then
   	    PetAI.tThinkFunc.AdjustMax(v,iMod)
  	    Room.EarnStat(v.szName)
  	  end
	  end
  	for k,v in pairs(PetAI.tNeeds) do
	    PetAI.tThinkFunc.SetValue(v,tItem.tAttribModifiers["i" .. v.szName .. "Mod"])
	  end
  end

	-- if the usage action is executed, process the given item's effects
	--if (PetAI.tCurAction == PetAI.tActions[tItem.szUsageAction]) then PetAI.tThinkFunc.ProcessEffects(tItem) end
end

//-----------------------------------------------------------------
// MakeAvailable
//-----------------------------------------------------------------
function tThinkFunc.MakeAvailable()
        
	-- reset various stored action parameters
	PetAI.bActionInProgress = false
	PetAI.tLastAction = PetAI.tCurAction
	PetAI.tCurAction = nil

  if not (PetAI.tSecretTrickToDo == nil) then
    PetAI.tThinkFunc.ChangeAction(PetAI.tSecretTrickToDo)
    PetAI.tSecretTrickToDo = nil
  else
    PetAI.tThinkFunc.DoIdle()
  end

	if (PetAI.tLastAction == nil) then return end

	-- if the last action is reinforceable...
	if (PetAI.tLastAction.bReinforceable) then

		-- set the reinforcement duration
		PetAI.iReinforcementTime = 3000

		if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE)) then
			for k,v in pairs(PetAI.tVoiceCommands) do
				if (v.szTrick == PetAI.tLastAction.szName) and (not v.bActive) then
					ActivateTrickButton(v.szTrick)
				end
			end
		end
	end
end

//-----------------------------------------------------------------
// Reward
//-----------------------------------------------------------------
function tThinkFunc.Reward(bTreat)

	-- play sound fx (relative to gender)
	if (PetAI.szGender == "Girl") then PetAI.tSoundFX.wavTreatGirl:Play()
	else PetAI.tSoundFX.wavTreatBoy:Play() end


  if ((PetAI.tCurAction == nil) or
     ((PetAI.tCurAction != nil) and (PetAI.tCurAction.bInterruptible)) or
     ((PetAI.tCurAction != nil) and (not PetAI.tCurAction.bInterruptible)) and ((PetAI.bCanBeCalledDown) and (PetAI.tJumpDownInfo != nil))) then

    if (PetAI.tJumpDownInfo != nil) then
      FlushChannel(ActivePetChannel)
      PetAI.tThinkFunc.DumpAllAnims()
      PetAI.tThinkFunc.MakeAvailable()
      Room.DisableNavButtons()
    end

    local iRewardBase = 30

    -- give the pet a treat as a reward
  	if (bTreat) then 
  	    	  
  	  local tTreatItem = nil
  	  
  	  if (gLastTreatTaken == "treats_03") then 
  	    tTreatItem = PetAI.tItems.tPetTreatGeneric
  	    iRewardBase = 40
  	  elseif (gLastTreatTaken == "treats_01") then
  	    tTreatItem = PetAI.tItems.tPetTreatOrganic
  	    iRewardBase = 65
  	  elseif (gLastTreatTaken == "treats_02") then
  	    tTreatItem = PetAI.tItems.tPetTreatLuxury
  	    iRewardBase = 80  	  
  	  end  	  
  	  
  	  PetAI.tThinkFunc.GiveItem(tTreatItem) 
  	end

	  -- if last action invalid or reinforcement time depleted, return immediately
	  if (PetAI.tLastAction == nil) or (PetAI.iReinforcementTime <= 0) then return end

	  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE) and (not PetAI.bVoiceCommandGiven)) then return
	  else PetAI.bVoiceCommandGiven = false end

	  -- calculate amount of reward memory bonus
	  local iNumTricksLearned = 0
	  local iRewardAmount = 0
	  local fIntBonusPercent = 0

	  -- count num tricks learned and use to determine base reward amount
	  for k,v in pairs(PetAI.tVoiceCommands) do if (v.bActive) then iNumTricksLearned = iNumTricksLearned + 1 end end
	  iRewardAmount = iRewardBase/(iNumTricksLearned+1)

	  -- determine intelligence bonus percentage
	  fIntBonusPercent = (2*PetAI.tStats.tIntelligence.iCurValue)/100.0

	  -- calculate final reward amount
	  iRewardAmount = iRewardAmount + (fIntBonusPercent*iRewardAmount)

  	-- increase the memory value for the last executed action
  	PetAI.tLastAction.fMemValue = PetAI.tLastAction.fMemValue + iRewardAmount

   	for k,v in pairs(PetAI.tVoiceCommands) do
  		if ((v.szTrick == PetAI.tLastAction.szName) and (not v.bActive)) then

	  		if (PetAI.tLastAction.fMemValue > 50.0) then
	  			SaveTrickButton(v.szTrick)
	  			v.bActive = true

	  			-- do help dialog
	  			if (PetAI.tLastAction.szName == "Promenade") then PetAI.DoHelpDialog(45,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "Sit") then PetAI.DoHelpDialog(46,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "LieDown") then PetAI.DoHelpDialog(47,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "Speak") then PetAI.DoHelpDialog(48,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "GivePaw") then PetAI.DoHelpDialog(49,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "Beg") then PetAI.DoHelpDialog(50,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "RollOver") then PetAI.DoHelpDialog(51,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "Hide") then PetAI.DoHelpDialog(52,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "NodHead") then PetAI.DoHelpDialog(53,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "ShakeHead") then PetAI.DoHelpDialog(54,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "Jump") then PetAI.DoHelpDialog(55,nil,nil,FALSE)
	  			elseif (PetAI.tLastAction.szName == "StandOnHindLegs") then PetAI.DoHelpDialog(56,nil,nil,FALSE) end

				  -- send help dialogs if a new talent show performance level has been achieved
				  local iTalentShowLevel = PetAI.TalentShow.GetHighestTalentShowLevel()
			  	if (iTalentShowLevel == 1) then PetAI.DoHelpDialog(62)
		  		elseif (iTalentShowLevel == 2) then PetAI.DoHelpDialog(63)
	  			elseif (iTalentShowLevel == 3) then PetAI.DoHelpDialog(64) end
  			else
				  if (not PetAI.tHelpDialogInfo[80]) then
					  DoOkDialog(string.format(GetLanguageString("Tip_80"), GetLanguageString("TrickName_"..PetAI.tLastAction.szName)))
					  PetAI.tHelpDialogInfo[80] = true
				  end
			  end
		  end
	  end

	  PetAI.tThinkFunc.DeActivateVCButtons()

	  -- clamp the memory value to its specified max (if necessary)
	  if (PetAI.tLastAction.fMemValue > PetAI.tLastAction.fMemMaxValue) then PetAI.tLastAction.fMemValue = PetAI.tLastAction.fMemMaxValue end

	  -- reset the last action parameter
	  PetAI.tLastAction = nil
	end
end

//-----------------------------------------------------------------
// Scold
//-----------------------------------------------------------------
function tThinkFunc.Scold()

	-- play sound fx
	PetAI.tSoundFX.wavNo:Play()

  if ((PetAI.tCurAction == nil) or
     ((PetAI.tCurAction != nil) and (PetAI.tCurAction.bInterruptible)) or
     ((PetAI.tCurAction != nil) and (not PetAI.tCurAction.bInterruptible)) and ((PetAI.bCanBeCalledDown) and (PetAI.tJumpDownInfo != nil))) then

    if (PetAI.tJumpDownInfo != nil) then
      FlushChannel(ActivePetChannel)
      PetAI.tThinkFunc.DumpAllAnims()
      PetAI.tThinkFunc.MakeAvailable()
      Room.DisableNavButtons()      
    end

    -- check for repeated scoldings
    if (FoxGetTickCount() - PetAI.iLastScoldTime < 10000) then PetAI.iScoldsInARow = PetAI.iScoldsInARow + 1
    else PetAI.iScoldsInARow = 0 end

    -- store the scolding time
    PetAI.iLastScoldTime = FoxGetTickCount()

    if (PetAI.tThinkFunc.ChangeAction(PetAI.tActions.BeSprayed)) then

  	  PetAI.tPettingFunc.ResetTrickLadder()
  	  PetAI.tThinkFunc.DeActivateVCButtons()

  	  -- if last action invalid or reinforcement time depleted, return immediately
  	  if (PetAI.tLastAction == nil) or (PetAI.iReinforcementTime <= 0) then
        local x,y,z = pet:GetPosition()
  	    return
  	  end

  	  if (PetAI.tThinkFunc.IsTrick(PetAI.tLastAction.szName)) then PetAI.DoHelpDialog(40,PetAI.tLastAction.szName) end

  	  -- decrease the memory value for the last executed action
  	  PetAI.tLastAction.fMemValue = PetAI.tLastAction.fMemValue - 5.0

  	  -- clamp the memory value to zero
  	  if (PetAI.tLastAction.fMemValue <= 0.0) then
  		  PetAI.tLastAction.fMemValue = 0.0
  		  PetAI.DoHelpDialog(41,PetAI.tLastAction.szName)
  	  end

  	  -- reset the last action parameter
  	  PetAI.tLastAction = nil
    end

  end
end

//-----------------------------------------------------------------
// DumpAllAnims
//-----------------------------------------------------------------
function tThinkFunc.DumpAllAnims()
  PetAI.vCurAnim = nil
  pet:BlendOutAll(0,-1)
end

//-----------------------------------------------------------------
// ChangeAction
//-----------------------------------------------------------------
function tThinkFunc.ChangeAction(tNewAction,bFromVoiceCommand)

    if (bFromVoiceCommand == nil) then bFromVoiceCommand = false end

    if (tNewAction != nil) then

	  	-- if the current action can be modified...
		  if (PetAI.tCurAction == nil) or (not PetAI.bActionInProgress) then

        -- stat effect inclusion flag
        local bIncludeStats = true

        if (PetAI.tThinkFunc.IsTrick(tNewAction.szName) and bFromVoiceCommand and (not PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.TALENT_SHOW))) then
          if not (((tNewAction.szName == "Sit") and PetAI.tThinkFunc.IsSitting()) or ((tNewAction.szName == "LieDown") and PetAI.tThinkFunc.IsLyingDown())) then
            PetAI.DoHelpDialog(65)
            Room.EarnMoney(5)
            bIncludeStats = true
	          PetAI.tCurTrickOfTheDay.ProcessUpdate("PerformedTrick",tNewAction.szName)
	        else
	          bIncludeStats = false
	        end
        elseif (PetAI.tThinkFunc.IsTrick(tNewAction.szName)) then
          bIncludeStats = false
        end

			  PetAI.tThinkFunc.DeActivateVCButtons()

			  -- set the new action to execute and process its effects
			  PetAI.tCurAction = tNewAction
			  PetAI.tThinkFunc.ProcessEffects(PetAI.tCurAction,bIncludeStats)

        -- increment use count
        PetAI.tCurAction.iUseCount = PetAI.tCurAction.iUseCount + 1

			  -- run the action's execution function
			  PetAI.tCurAction.ExecuteAction()

			  -- set the busy flag for the pet
			  PetAI.bActionInProgress = true

        -- update talent show with the action execution event
        if (PetAI.tCurAction != nil) then PetAI.TalentShow.UpdateRound(PetAI.tCurAction.szName) end

        -- return success
        return(true)
		  end
    else
        LogInfo("Warning: tThinkFunc.ChangeAction() - tNewAction is nil")
	end
	return(false)
end

//-----------------------------------------------------------------
// ChangeState
//-----------------------------------------------------------------
function tThinkFunc.ChangeState(NEW_STATE,bReset)

  -- default the reset flag to true
  if (bReset == nil) then bReset = true end

  if (bReset) then

    -- flush the active channel
    if (ActivePetChannel != nil) then FlushChannel(ActivePetChannel) end

    -- remove all animation
    PetAI.tThinkFunc.DumpAllAnims()

    -- remove bone tweaking
    Room.PetLookAtStop()
  end

  -- set pet in idle
  if (NEW_STATE != PetAI.PET_STATES.AI_DISABLED) then 
    if (not ((PetAI.CUR_STATE == PetAI.PET_STATES.ATTENTION_MODE) and (PetAI.tCurAction != nil))) then
      PetAI.tThinkFunc.MakeAvailable() 
    end   
  end

  -- destroy any handheld objects or toys
  if (NEW_STATE != PetAI.PET_STATES.INFORMAL_TRICK) then
    if (toy != nil) then 
      if (NEW_STATE == PetAI.PET_STATES.TALENT_SHOW) then Room.RemoveToy()
      else ToggleToy() end
    end
  end

  -- enable camera tilting
  Room.m_bTransition = false

  -- disable call ignoring
  PetAI.bIgnoringCalls = false

  -- restore glitter settings to default
  SetAllowGlitter(TRUE)
	SetActiveGlitterSprite("default")

  -- terminate current state
  if (PetAI.CUR_STATE == PetAI.PET_STATES.AI_DISABLED) then
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.IDLE_MODE) then
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.EATING_MODE) then
    PetAI.tSoundFX.wavEating:Stop()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.DRINKING_MODE) then
    PetAI.tSoundFX.wavDrinking:Stop()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.ATTENTION_MODE) then
    local bImmediate = false
    if (NEW_STATE != PetAI.PET_STATES.IDLE_MODE) then bImmediate = true end
    PetAI.tThinkFunc.TerminateAttentionMode(bImmediate)
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.BATHTUB_MODE) then
    PetAI.tThinkFunc.TerminateTubMode()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.TALENT_SHOW) then
    PetAI.tThinkFunc.TerminateTalentShow()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.INFORMAL_TRICK) then
    if (PetAI.tActiveInformalTrick != nil) then PetAI.tActiveInformalTrick = nil end
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.BRUSHING_MODE) then
    PetAI.tThinkFunc.KillHandheld()
    PetAI.tThinkFunc.TerminateBrushingMode()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.SLEEPING_MODE) then
    PetAI.tSoundFX.wavSnore:Stop()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.SHUTDOWN) then
  end

  -- open new state
  if (NEW_STATE == PetAI.PET_STATES.AI_DISABLED) then
  elseif (NEW_STATE == PetAI.PET_STATES.IDLE_MODE) then
  elseif (NEW_STATE == PetAI.PET_STATES.EATING_MODE) then
  elseif (NEW_STATE == PetAI.PET_STATES.DRINKING_MODE) then
  elseif (NEW_STATE == PetAI.PET_STATES.ATTENTION_MODE) then
    local x,y,z = pet:GetPosition()
    if (PetAI.tJumpDownInfo != nil and (y > 0)) then PetAI.tThinkFunc.JumpDownFromObject()
    elseif (PetAI.tJumpDownInfo != nil) then PetAI.tJumpDownInfo = nil end
    PetAI.tThinkFunc.InitiateAttentionMode()
  elseif (NEW_STATE == PetAI.PET_STATES.BATHTUB_MODE) then
    PetAI.tThinkFunc.InitiateTubMode()
  elseif (NEW_STATE == PetAI.PET_STATES.TALENT_SHOW) then
    PetAI.tThinkFunc.InitiateTalentShow()
  elseif (NEW_STATE == PetAI.PET_STATES.INFORMAL_TRICK) then
  elseif (NEW_STATE == PetAI.PET_STATES.BRUSHING_MODE) then
    PetAI.tThinkFunc.InitiateBrushingMode()
  elseif (NEW_STATE == PetAI.PET_STATES.SLEEPING_MODE) then
    PetAI.tThinkFunc.InitiateSleepingMode()
  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.SHUTDOWN) then
  end

  -- set current state
  PetAI.CUR_STATE = NEW_STATE
end

//-----------------------------------------------------------------
// IsInState
//-----------------------------------------------------------------
function tThinkFunc.IsInState(STATE_TO_CHECK)
  return(PetAI.CUR_STATE == STATE_TO_CHECK)
end

//-----------------------------------------------------------------
// Update
//-----------------------------------------------------------------
function tThinkFunc.Update(iElapsedTime)

  -- *TEMP*
  local bStateTrace = false

  -- determine time passed since last update
	local iTimeBetweenUpdates = iElapsedTime - PetAI.iLastUpdateTime

  -- update operation flags
  local bDoDecay = true
  local bDoEffects = true

  if (PetAI.CUR_STATE == PetAI.PET_STATES.AI_DISABLED) then
  	-- ***************************************************************************************************************************************
  	-- AI_DISABLED State
  	-- ***************************************************************************************************************************************

  	-- set the last update time for the next iteration
  	PetAI.iLastUpdateTime = iElapsedTime

    -- return immediately, performing no changes
    return

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.IDLE_MODE) then
  	-- ***************************************************************************************************************************************
  	-- IDLE_MODE State
  	-- ***************************************************************************************************************************************
    if (bStateTrace) then print("IDLE_MODE State") end

    -- if pet is currently idle...
    if (IsChannelEmpty(ActivePetChannel) and PetAI.tThinkFunc.IsIdle()) then

  		  -- update wait time between idle action selection
  		  PetAI.iWaitTime = PetAI.iWaitTime - iTimeBetweenUpdates

  		  -- pick new idle and reset when time is up
  		  if PetAI.iWaitTime <= 0 then
          if (math.random(100) > 25) then PetAI.tThinkFunc.GetIdleAction(PetAI.tIdleScripts)
          else PetAI.tThinkFunc.GetIdleAction(PetAI.tIdleActions) end
  			  PetAI.iWaitTime = math.random(2000) + 2000
  		  end
	  end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.EATING_MODE) then
  	-- ***************************************************************************************************************************************
  	-- EATING_MODE State
  	-- ***************************************************************************************************************************************

    if (PetAI.tNeeds.tHunger.iCurValue >= PetAI.tNeeds.tHunger.iMaxValue) or (PetAI.iFoodInBowl <= 0) then
      PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE)
    end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.DRINKING_MODE) then
  	-- ***************************************************************************************************************************************
  	-- DRINKING_MODE State
  	-- ***************************************************************************************************************************************

    if (PetAI.tNeeds.tThirst.iCurValue >= PetAI.tNeeds.tThirst.iMaxValue) or (PetAI.iWaterInBowl <= 0) then
      PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE)
    end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.BRUSHING_MODE) then
  	-- ***************************************************************************************************************************************
  	-- BRUSHING_MODE State
  	-- ***************************************************************************************************************************************

    if (PetAI.bIsBeingPet) then PetAI.tPettingFunc.UpdatePettingSfx(iTimeBetweenUpdates) end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.ATTENTION_MODE) then
  	-- ***************************************************************************************************************************************
  	-- ATTENTION_MODE State
  	-- ***************************************************************************************************************************************
    if (bStateTrace) then print("ATTENTION_MODE State") end

    if (IsChannelEmpty(ActivePetChannel) and (not PetAI.bIsBeingPet)) then

  		-- update attention span time
  		PetAI.iAttentionSpan = PetAI.iAttentionSpan - iTimeBetweenUpdates

 		  -- update wait time between idle action selection
 		  PetAI.iWaitTime = PetAI.iWaitTime - iTimeBetweenUpdates

      -- if attention span runs out, revert to idle state
  		if (PetAI.iAttentionSpan <= 0) then
  		  PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE,false)
  		elseif (PetAI.iWaitTime <= 0) and (PetAI.iReinforcementTime <= 0) then
    		if (PetAI.bNeedToBlink) then
        	PetAI.tThinkFunc.ChangeAction(PetAI.tActions.BlinkInAttn)
        	PetAI.bNeedToBlink = false
        else
        	PetAI.tThinkFunc.ChangeAction(PetAI.tActions[PetAI.tThinkFunc.PickRandomAction(PetAI.tAttnIdleActions)]);
  			  PetAI.iWaitTime = math.random(3000) + 2000
  			  PetAI.bNeedToBlink = true
  			end
  		end

    -- otherwise, if being pet, reset attention span to maximum
    elseif (PetAI.bIsBeingPet) then PetAI.tThinkFunc.RestoreAttentionSpan() end

    if (PetAI.bIsBeingPet) then PetAI.tPettingFunc.UpdatePettingSfx(iTimeBetweenUpdates) end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.BATHTUB_MODE) then
  	-- ***************************************************************************************************************************************
  	-- BATHTUB_MODE State
  	-- ***************************************************************************************************************************************
    if (bStateTrace) then print("BATHTUB_MODE State") end

    if (PetAI.bIsBeingPet) then
      PetAI.tThinkFunc.UpdateTubSoap(iTimeBetweenUpdates)
      PetAI.tPettingFunc.UpdatePettingSfx(iTimeBetweenUpdates)
    end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.TALENT_SHOW) then
  	-- ***************************************************************************************************************************************
  	-- TALENT_SHOW State
  	-- ***************************************************************************************************************************************
    if (bStateTrace) then print("TALENT_SHOW State") end

    -- turn off decay and effects during show
    bDoDecay = false
    bDoEffects = false

    -- update talent show, if currently active
    if (PetAI.TalentShow.bActive) then PetAI.TalentShow.Update(iElapsedTime) end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.INFORMAL_TRICK) then
  	-- ***************************************************************************************************************************************
  	-- INFORMAL_TRICK State
  	-- ***************************************************************************************************************************************
    if (bStateTrace) then print("INFORMAL_TRICK State") end

  	if (PetAI.tActiveInformalTrick != nil) then

	  	-- if current active informal trick is done...
	  	if not (PetAI.tActiveInformalTrick.Update()) then

        -- if the active trick not replaced in the interim, terminate the trick
        PetAI.tActiveInformalTrick = nil
		  end
		else
      PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE,false)
 	  end

  elseif (PetAI.CUR_STATE == PetAI.PET_STATES.SLEEPING_MODE) then
  	-- ***************************************************************************************************************************************
  	-- SLEEPING_MODE State
  	-- ***************************************************************************************************************************************
    if (bStateTrace) then print("SLEEPING_MODE State") end

    -- decrement sleep anim timer
    PetAI.iTimeUntilSleepChange = PetAI.iTimeUntilSleepChange - iTimeBetweenUpdates

	  -- if the timer has been depleted...
    if (PetAI.iTimeUntilSleepChange <= 0) then

      -- load a new sleeping idle
      PetAI.tThinkFunc.LoadNewAnim(PetAI.tThinkFunc.GetNewSleepIdleAnim(),1)

      -- determine a new timer setting
      PetAI.iTimeUntilSleepChange = math.random(10*60*1000)
    end

    if (IsChannelEmpty(ActivePetChannel) and PetAI.tThinkFunc.IsSleepIdle() and (PetAI.tSoundFX.wavSnore:IsPlaying() != 1)) then PetAI.tSoundFX.wavSnore:Play() end

    if (PetAI.tSoundFX.wavSnore:IsPlaying() == 1) and ((PetAI.iLastSnoozTime == nil) or (FoxGetTickCount() - PetAI.iLastSnoozTime > 2000)) then 
      Room.DoSnooz() 
      PetAI.iLastSnoozTime = FoxGetTickCount()
    end

    -- if alertness value is maxed out, restore idle AI operation
    if (IsChannelEmpty(ActivePetChannel) and (PetAI.tNeeds.tAlertness.iCurValue >= PetAI.tNeeds.tAlertness.iMaxValue)) then
      PetAI.tThinkFunc.WakeUp()
      DoLuaEvent("PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE,false)",ActivePetChannel)
    end

  end

	-- **************************************************************
	-- Miscellaneous
	-- **************************************************************

  -- update pet attributes
	if (bDoDecay) then PetAI.tThinkFunc.UpdateAttributes(iElapsedTime) end

  -- update active effects (if any)
  if (bDoEffects) then PetAI.tThinkFunc.UpdateActiveEffects() end

  -- check if need help dialogs invoked
  PetAI.CheckNeedHelpDialogs()

  -- update the curren trick of the day
  PetAI.tThinkFunc.UpdateTrickOfTheDay()

  -- update trick reinforcement time (if necessary)
  PetAI.tThinkFunc.UpdateTrickReinforcementTime(iTimeBetweenUpdates)

  -- send initial help dialog after five seconds of play
  if (iElapsedTime > 5000) then PetAI.DoHelpDialog(1) end

  -- send help dialog if player money drops below $25
  if (GetPlayerMoney() <= 25) then PetAI.DoHelpDialog(72) end

	-- set the last update time for the next iteration
	PetAI.iLastUpdateTime = iElapsedTime

  -- if the pet is currently watching something, update look at position
  if (PetAI.CUR_LOOK_STATE == PetAI.LOOK_STATES.CAMERA) then Room.PetLookAtPoint(GetCameraPosition())
  elseif (PetAI.CUR_LOOK_STATE == PetAI.LOOK_STATES.OBJECT) then
    if (PetAI.cFocalObjectTag == nil) then Room.PetLookAtPoint(PetAI.cFocalObjectRef:GetPosition())
    else Room.PetLookAtPoint(PetAI.cFocalObjectRef:GetBonePosition(PetAI.cFocalObjectTag)) end
  end

	if (not PetAI.m_bSkipWaterDroplets and PetAI.tSoundFX.wavShakeOffWater:IsPlaying() != 0) then
		local tRandBones = {"Bip01 TailNub","Bip01 Tail3","Bip01 Tail2","Bip01 Tail1","Bip01 Tail","Bip01 Pelvis","Bip01 Spine","Bip01 Spine1","Bip01 Spine2","Bip01 R Thigh","Bip01 R Calf","Bip01 R HorseLink","Bip01 R Foot","Bip01 R Toe0","Bip01 R Toe0Nub","Bip01 L Thigh","Bip01 L Calf","Bip01 L HorseLink","Bip01 L Foot","Bip01 L Toe0","Bip01 L Toe0Nub","Bip01 L UpperArm","Bip01 L Forearm","Bip01 L Hand","Bip01 L Finger0","Bip01 L Finger0Num","Bip01 R UpperArm","Bip01 R Forearm","Bip01 R Hand","Bip01 R Finger0","Bip01 R Finger0Num","Bip01 Neck","Bip01 Neck1","Bip01 Head","Bip01 HeadNub","NUB","Bip01 BottomJaw","Bip01 BottomJawNub","Thongue_01","Thongue_02","Bip01 Ponytail1","Bip01 Ponytail11","Bip01 Ponytail1Nub","Bip01 Ponytail2","Bip01 Ponytail21","Bip01 Ponytail2Nub","Bip01 L Clavicle","Bip01 R Clavicle","-tag Hat01","-tag Collar01","-tag Mouth01","-tag TailPom01","-tag Balance","EyeLid_Upper_R","EyeLid_Upper_L","EyeLid_Upper_NUB","EyeLid_Lower_NUB","EyeLid_Lower_R","EyeLid_Lower_L","-tag Bootie01","-tag Bootie02","-tag Bootie03","-tag Bootie04"}
		local iBone = pet:GetBoneIndex(tRandBones[math.random(table.getn(tRandBones))])

		local xx, yy, zz = pet:GetBonePosition(iBone)
		local x, y = GetScreenPosition(xx, yy, zz)

		DoGlitter("WaterDroplet", x + math.random(20) - 10, y + math.random(20) - 10, TRUE, 500, 10)
	end

  -- run flee-bitten effect if grooming is below nominal levels
  if (PetAI.tNeeds.tGrooming.iCurValue < PetAI.tNeeds.tGrooming.iLowThreshold) then
    if (PetAI.tSoundFX.wavFleeBitten:IsPlaying() != 1) then PetAI.tSoundFX.wavFleeBitten:Play() end
    PetAI.tThinkFunc.DoFleeEffect(iElapsedTime)
  elseif (PetAI.tSoundFX.wavFleeBitten:IsPlaying() == 1) then PetAI.tSoundFX.wavFleeBitten:Stop() end

	-- **************************************************************
	-- TEMP (to be moved)
	-- **************************************************************

  -- if pet is not currently in the same room as the player
  --if not (PetAI.szLocation == PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom]) then
  --  -- walk to the player's current room
  --  PetAI.tThinkFunc.ChangeAction(PetAI.tActions["WalkTo"..PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom]])
  --end


  if not (cHandheld == nil) then
    local x,y,z = MouseToToy()
    local xx, yy, zz = cHandheld:GetOffset()
    cHandheld:SetPosition(x + xx, y + yy, z + zz)
  end

  if (cPoo != nil) then
    if (PetAI.szPooLocation != PetAI.szLocation) then
      PetAI.szPooLocation = nil
      PetAI.tThinkFunc.KillPooAndPee()
    else cPoo:Update(iElapsedTime) end
  end

  if (cPeeStain != nil) then
    if (PetAI.szPeeLocation != PetAI.szLocation) then
      PetAI.tThinkFunc.KillPooAndPee()
      PetAI.szPeeLocation = nil
    else cPeeStain:Update(iElapsedTime) end
  end

end

//-----------------------------------------------------------------
// WakeUp
//-----------------------------------------------------------------
function tThinkFunc.WakeUp()
  FlushChannel(ActivePetChannel)
  PetAI.tSoundFX.wavSnore:Stop()
  PetAI.tThinkFunc.DoAnimTransition("SleepEnd",{szBody="BodyIdle",szHead="HeadIdle",szTail="TailIdle"},pet:GetAnimDuration("SleepEnd")*1000)
end

//-----------------------------------------------------------------
// CheckUseCount
//-----------------------------------------------------------------
function tThinkFunc.CheckUseCount(szActionName)

  if (PetAI.tActions[szActionName].iUseCount < 2) then
    return(true)
  else
    PetAI.tActions[szActionName].iUseCount = 0
    return(false)
  end
end

//-----------------------------------------------------------------
// RestoreAttentionSpan
//-----------------------------------------------------------------
function tThinkFunc.RestoreAttentionSpan()
  PetAI.iAttentionSpan = PetAI.iMaxAttentionSpan
end

//-----------------------------------------------------------------
// UpdateAttributes
//-----------------------------------------------------------------
function tThinkFunc.UpdateAttributes(iElapsedTime)
  if (PetAI.tThinkFunc.CheckDecay(PetAI.tNeeds,iElapsedTime)) then
    Room.FlashNeeds(false)
  else
    Room.FlashNeeds(true)
  end

  PetAI.tThinkFunc.CheckDecay(PetAI.tStats,iElapsedTime)
end

//-----------------------------------------------------------------
// UpdateTrickOfTheDay
//-----------------------------------------------------------------
function tThinkFunc.UpdateTrickOfTheDay()
  if (PetAI.tCurTrickOfTheDay != nil) then
    if (PetAI.tCurTrickOfTheDay.IsFinished()) then
      if (PetAI.tCurTrickOfTheDay.IsDelayTOD()) then
	    	if (IsChannelEmpty(Room.m_nCameraChannel) and
		      IsMouseAllowed() != FALSE and
		      (PetAI.CUR_STATE != PetAI.PET_STATES.ATTENTION_MODE) and
		      (PetAI.CUR_STATE != PetAI.PET_STATES.BATHTUB_MODE) and
		      (PetAI.CUR_STATE != PetAI.PET_STATES.BRUSHING_MODE) and
		      (PetAI.CUR_STATE != PetAI.PET_STATES.TALENT_SHOW)) then
			    PetAI.SetTrickOfTheDay()
		    end
      else
        if (PetAI.tCurTrickOfTheDay.iReward > 0) then
        	Room.EarnMoney(PetAI.tCurTrickOfTheDay.iReward)
          DoDialogBox(string.format(GetLanguageString("TOD_Winner"),PetAI.tCurTrickOfTheDay.iReward))
        end

        -- wait for 15 minutes to offer a new trick of the day
        PetAI.SetDelayTrickOfTheDay(15*60*1000)
      end
    end
  end
end

//-----------------------------------------------------------------
// DumpActiveEffects
//-----------------------------------------------------------------
function tThinkFunc.DumpActiveEffects()
	-- if any effects are currently active...
	if not (table.getn(PetAI.tActiveEffects) == 0) then

		-- run the termination method of all active effects
		for i in ipairs(PetAI.tActiveEffects) do if (PetAI.tActiveEffects[i].OnTerminate != nil) then PetAI.tActiveEffects[i].OnTerminate() end end

		-- remove all active effects from the table
		while (table.getn(PetAI.tActiveEffects) > 0) do table.remove(PetAI.tActiveEffects) end
  end
end

//-----------------------------------------------------------------
// UpdateActiveEffects
//-----------------------------------------------------------------
function tThinkFunc.UpdateActiveEffects()

	-- if any effects are currently active...
	if not (table.getn(PetAI.tActiveEffects) == 0) then

		-- update all active effects, removing any that terminate as a result
		for i in ipairs(PetAI.tActiveEffects) do
			if not (PetAI.tActiveEffects[i].Update()) then table.remove(PetAI.tActiveEffects,i) end
		end
	end
end

//-----------------------------------------------------------------
// UpdateTrickReinforcementTime
//-----------------------------------------------------------------
function tThinkFunc.UpdateTrickReinforcementTime(iTimeBetweenUpdates)

	-- if reinforcement time is greater than zero
	if PetAI.iReinforcementTime > 0 then

		-- PetAI.tThinkFunc.Update reinforcement time and clamp to zero
		PetAI.iReinforcementTime = PetAI.iReinforcementTime - iTimeBetweenUpdates
		if PetAI.iReinforcementTime <= 0 then
			PetAI.iReinforcementTime = 0
			PetAI.tThinkFunc.DeActivateVCButtons()
		end

	-- otherwise, reset last action storage variable (if necessary)
	elseif PetAI.iReinforcementTime <= 0 and not PetAI.tLastAction == nil then
		PetAI.tLastAction = nil
	end
end

//-----------------------------------------------------------------
// GetNewSleepIdleAnim
//-----------------------------------------------------------------
function tThinkFunc.GetNewSleepIdleAnim()
  local tSleepAnims = {}
  for i = 1,3 do if (PetAI.vCurAnim != "SleepIdle"..i) then table.insert(tSleepAnims,"SleepIdle"..i) end end
  return(tSleepAnims[math.random(table.getn(tSleepAnims))])
end

//-----------------------------------------------------------------
// DeActivateVCButtons
//-----------------------------------------------------------------
function tThinkFunc.DeActivateVCButtons()
	for k,v in pairs(PetAI.tVoiceCommands) do
		if (not v.bActive) then
			DeActivateTrickButton(v.szTrick)
		end
	end
end

//-----------------------------------------------------------------
// GetAttention
//-----------------------------------------------------------------
function tThinkFunc.GetAttention()

	-- if the dog is already paying attention, return immediately
  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE) or (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.AI_DISABLED))) then return end

	-- play whistle sound
	PetAI.tSoundFX.wavWhistle:Play()

	if not (PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom] == PetAI.szLocation) then return end

	Room.IconizeToy()

  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.SLEEPING_MODE) and (not PetAI.bIgnoringCalls)) then
    if (IsChannelEmpty(ActivePetChannel)) then
      PetAI.tThinkFunc.WakeUp()
      DoLuaEvent("PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.ATTENTION_MODE,false)",ActivePetChannel)
    end
  elseif (not PetAI.bIgnoringCalls) then
	  if (PetAI.tCurAction == nil) or ((PetAI.tCurAction != nil) and (PetAI.tCurAction.bInterruptible)) then
  	  PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.ATTENTION_MODE)
  	elseif ((PetAI.tCurAction != nil) and (not PetAI.tCurAction.bInterruptible)) and ((PetAI.bCanBeCalledDown) and (PetAI.tJumpDownInfo != nil)) then
  	  PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.ATTENTION_MODE)
  	end
	end
end

//-----------------------------------------------------------------
// StartUp
//-----------------------------------------------------------------
function tThinkFunc.StartUp()

  ActivePetChannel = CreateChannel()

  -- seed the random number generator with the current time
  math.randomseed(FoxGetTimeSeconds())

  PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE)

	PetAI.tThinkFunc.LoadNewAnim({szBody="BodyIdle",szHead="HeadIdle",szTail="TailIdle"},0.25)
	PetAI.tPathFunc.GeneratePathMap()
	for k,v in pairs(PetAI.tVoiceCommands) do v.InitCommand() end
	PetAI.tPettingFunc.ResetTrickLadder()

  PetAI.tCurTrickOfTheDay = PetAI.tTrickOfTheDayFunc.CreateTOD_None(60000)
  Room.stats_dialog.m_cTrickOfTheDayBody:SetString(GetTrickOfTheDay())

	-- load sounds
	PetAI.tSoundFX.wavNo:Load("VO_No.wav")
	PetAI.tSoundFX.wavWhistle:Load("Whistle.wav")
	PetAI.tSoundFX.wavTreatGirl:Load("VO_GoodGirl.wav")
	PetAI.tSoundFX.wavTreatBoy:Load("VO_GoodBoy.wav")
	PetAI.tSoundFX.wavDogHowl:Load(PetAI.szSpecies.."_Howl.wav")
	PetAI.tSoundFX.wavDogGrowl:Load(PetAI.szSpecies.."_Growl.wav")
	PetAI.tSoundFX.wavDogLowGrowl:Load(PetAI.szSpecies.."_LowGrowl.wav")
  PetAI.tSoundFX.wavConfused:Load("confused.wav")
  PetAI.tSoundFX.wavNeedToGo:Load("bladder.wav")
  PetAI.tSoundFX.wavSnore:Load("snoring.wav")

 	for i = 1,7 do PetAI.tSoundFX["wavPetSpeak"..i]:Load(PetAI.szSpecies.."_Speak"..i..".wav") end
 	for i = 1,7 do PetAI.tSoundFX["wavHeadBark"..i]:Load(PetAI.szSpecies.."_HeadBark"..i..".wav") end
  for i = 1,10 do PetAI.tSoundFX["wavPetting"..i]:Load(string.lower(PetAI.szSpecies).."pet"..i..".wav") end

  PetAI.tSoundFX.wavRefuse:Load("refuse.wav")
  PetAI.tSoundFX.wavCleanSelf:Load("cleaning-fur.wav")
  PetAI.tSoundFX.wavDrink:Load("drinking.wav")
  PetAI.tSoundFX.wavEat:Load("eating.wav")
  PetAI.tSoundFX.wavPee:Load("peeing.wav")
	PetAI.tSoundFX.wavScratchFurniture:Load("scratch-furniture.wav")
	PetAI.tSoundFX.wavScratchSelf:Load("scratch-self.wav")
	PetAI.tSoundFX.wavSniffGround:Load("sniffing.wav")
	PetAI.tSoundFX.wavStandSprayed:Load("spray-stand.wav")
	PetAI.tSoundFX.wavSitSprayed:Load("spray-sit.wav")
	PetAI.tSoundFX.wavLaySprayed:Load("spray-lie.wav")

	PetAI.tSoundFX.wavEating:Load("eating.wav")
	PetAI.tSoundFX.wavDrinking:Load("drinking.wav")

	PetAI.tSoundFX.wavEating:SetNumLoops(9999)
	PetAI.tSoundFX.wavDrinking:SetNumLoops(9999)

  if CATZ then PetAI.tSoundFX.wavJingleBall:Load("jingle.wav")
	else PetAI.tSoundFX.wavSqueakyDuck:Load("ducky.wav") end

  PetAI.tSoundFX.wavBrushing1:Load("brush1.wav")
  PetAI.tSoundFX.wavBrushing2:Load("brush2.wav")

	PetAI.tSoundFX.wavPetYawn:Load(string.lower(PetAI.szSpecies).."yawn.wav")
	PetAI.tSoundFX.wavPetEatTreat:Load("chewtreat.wav")

	-- load talent show sounds
	PetAI.tSoundFX.wavTSApplause:Load("ts_applause.wav")
	PetAI.tSoundFX.wavTSLaugh:Load("ts_laugh.wav")

	PetAI.tSoundFX.wavTSClap01:Load("ts_clap1.wav")
	PetAI.tSoundFX.wavTSClap02:Load("ts_clap2.wav")
	PetAI.tSoundFX.wavTSClap03:Load("ts_clap3.wav")

	PetAI.tSoundFX.wavTSWow:Load("ts_wow.wav")
  PetAI.tSoundFX.wavTSFail:Load("ts_fail.wav")

	PetAI.tSoundFX.wavTSIntro:Load("ts_intro.wav")
	PetAI.tSoundFX.wavTSEnding:Load("ts_ending.wav")
	PetAI.tSoundFX.wavTSRound1Begin:Load("ts_music1.wav")
	PetAI.tSoundFX.wavTSRound1End:Load("ts_round1done.wav")
	PetAI.tSoundFX.wavTSRound1Fail:Load("ts_round1fail.wav")
	PetAI.tSoundFX.wavTSRound2Begin:Load("ts_music2.wav")

	PetAI.tSoundFX.wavTSRound3Begin:Load("ts_music3.wav")

  PetAI.tSoundFX.wavShakeOffWater:Load(string.lower(PetAI.szSpecies).."shake.wav")
  PetAI.tSoundFX.wavFleeBitten:Load("dirty.wav")

	-- configure animation parameters
	DoEvent("Pet"..".SetLoopCount(\"LeanOnSurface\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"EyeBlink\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"EyeClose\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"EyeSquint\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"Promenade\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"JumpDownFromMed\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"JumpDownFromHigh\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"DefecateBegin\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"DefecateEnd\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"JumpOntoSurfaceLow\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"JumpOntoSurfaceMed\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"JumpOntoSurfaceHigh\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SleepBegin\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SleepEnd\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"Sit\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"LayDown\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"LayGetUp\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitGetUp\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"RollOver\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"ShakeOffWater\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"TurnAndBark\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"ScratchFurniture\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"ChaseTail\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"BackFlip\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"LayPettingStart\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"LayPettingEnd\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"CatchObject1\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"CatchObject2\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"CatchObject3\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"BodyBendOverBegin\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"BodyBendOverEnd\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadRetrieveBegin\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadRetrieveEnd\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"StandSprayedWithWater\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitSprayedWithWater\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"LaySprayedWithWater\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadYawn\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadPetBegin\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadPetEnd\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SniffGround\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"LayStretch\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitScratch\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitSpeak\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadLickChops\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"CleanFur\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"Confused\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadShake\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitGivePaw\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitBeg\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitWavePaw\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"RollOver\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"Hide\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadNod\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"SitBow\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"Jump\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"Jump1\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"PlayDead\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"StandOnHindLegs\", 1)", GetSystemChannel())
	DoEvent("Pet"..".SetLoopCount(\"HeadBark\", 1)", GetSystemChannel())
end

//-----------------------------------------------------------------
// Shutdown
//-----------------------------------------------------------------
function tThinkFunc.Shutdown()

    PetAI.Disable()

    if (ActivePetChannel != nil) then
        FlushChannel(ActivePetChannel)
        DestroyChannel(ActivePetChannel)
        ActivePetChannel = nil
    end

--  LogInfo("Final Path Map Print----------------------------------------------------------------")
--  local szRow = ""
--  for r in ipairs(PetAI.tPathFunc.tDisableSet) do
--    szRow = ""
--   for c in ipairs(PetAI.tPathFunc.tDisableSet[r]) do
--      szRow = szRow..PetAI.tPathFunc.tDisableSet[r][c].. ","
--    end
--    LogInfo("{"..szRow.."},")
--  end
--  LogInfo("------------------------------------------------------------------------------------")

  Room.bNeedsFlashing = false

  KillToy()
  PetAI.tThinkFunc.KillHandheld()

  PetAI.tThinkFunc.KillPooAndPee()
	for k,v in pairs(PetAI.tVoiceCommands) do v.cSoundFX:Release() end
	for k,v in pairs(PetAI.tSoundFX) do v:Stop() v:Release() end
end

//-----------------------------------------------------------------
// LookTowardDesiredRoom
//-----------------------------------------------------------------
function tThinkFunc.LookTowardDesiredRoom(nTargetRoom)

  local tCurRoom = PetAI.tPathFunc.tRooms[PetAI.szLocation]
  local tSitNode = nil
  local szHeadTurnAnim = "HeadLeft"

  if (nTargetRoom > Room.m_nCurRoom) then
		tSitNode = tCurRoom.GetLeftBottomCornerNode()
    szHeadTurnAnim = "HeadLeft"
  elseif (nTargetRoom < Room.m_nCurRoom) then
		tSitNode = tCurRoom.GetRightBottomCornerNode()
    szHeadTurnAnim = "HeadRight"
  end

  PetAI.tThinkFunc.LoadNewAnim("Walk1",0.5)
	PetAI.tPathFunc.WalkToPathNode(tSitNode,"BodyWalk","BodyIdle")

  DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
  PetAI.tThinkFunc.DoSitAndSingleAnim({szBody="BodySitIdle",szHead="HeadBark"},"wavHeadBark"..math.random(7),pet:GetAnimDuration("HeadBark")*1000)
	PetAI.tThinkFunc.LoadNewAnim({szBody="BodySitIdle",szHead=szHeadTurnAnim},0.75)
	DoEvent("System.Sleep(750)",ActivePetChannel)
  PetAI.tThinkFunc.DoSitAndSingleAnim({szHead="HeadBark"},"wavHeadBark"..math.random(7),pet:GetAnimDuration("HeadBark")*1000)
  PetAI.tThinkFunc.LoadNewAnim({szBody="BodySitIdle",szHead=szHeadTurnAnim},0.75)
	DoEvent("System.Sleep(750)",ActivePetChannel)
end

//-----------------------------------------------------------------
// InitiateAttentionMode
//-----------------------------------------------------------------
function tThinkFunc.InitiateAttentionMode()

  PetAI.iAttentionSpan = PetAI.iMaxAttentionSpan

	PetAI.bWaitingForCamera = true

  PetAI.fCurOrientation = 0

	local tCurRoom = PetAI.tPathFunc.tRooms[PetAI.szLocation]
	local tAttnNode = tCurRoom.GetAttentionNode()
	local fPetSpeed = 65.0

  local fDistance = PetAI.tPathFunc.DistanceBetweenTwoPoints({pet:GetPosition()},tAttnNode)
  local fTimeToAttnNode = math.ceil((fDistance/fPetSpeed)*1000.0)

  local tEyePos = {}
  local tLookAtPos = {}

  -- disable the left/right nav buttons
  DoLuaEvent("Room.DisableNavButtons()",ActivePetChannel)

  -- turn off mouse input until after camera move
  SetMouseAllowed(FALSE)

	DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
	PetAI.tThinkFunc.DoStandAndSingleAnim({szHead="HeadBark"},"wavHeadBark"..math.random(7),pet:GetAnimDuration("HeadBark")*1000)
	PetAI.tThinkFunc.LoadNewAnim({szHead="CLEAR"},0.25)

  local tAttnModeCam = Room.m_tDogzAttnModeCam
  if CATZ then tAttnModeCam = Room.m_tCatzAttnModeCam end
  for i=1,3 do table.insert(tEyePos,tAttnNode[i]+tAttnModeCam.EyeOffset[i]) end
  for i=1,3 do table.insert(tLookAtPos,tAttnNode[i]+tAttnModeCam.LookAtOffset[i]) end

	DoLuaEvent("PetAI.tThinkFunc.ZoomCameraForAttnMode("..fTimeToAttnNode..",{"..tEyePos[1]..", "..tEyePos[2]..", "..tEyePos[3].."},{"..tLookAtPos[1]..", "..tLookAtPos[2]..", "..tLookAtPos[3].."})",ActivePetChannel)

	PetAI.tThinkFunc.LoadNewAnim({szTail="TailExcited"},0.5)

  PetAI.tPathFunc.EnableTransNodes()

	if not (PetAI.tPathFunc.PointsEqual(PetAI.tPathFunc.GetClosestNodeToPet(),tAttnNode)) then
		PetAI.tThinkFunc.LoadNewAnim({szBody="Walk1"},0.5)
 	  PetAI.tPathFunc.WalkToPathNode(tAttnNode,"BodyWalk","BodyIdle",fPetSpeed)
	end

  DoLuaEvent("PetAI.tPathFunc.DisableTransNodes()",ActivePetChannel)
	DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
	DoLuaEvent("PetAI.tThinkFunc.MakeAvailable()",ActivePetChannel)

	PetAI.fCurAttnAngle = 270.0
end

//-----------------------------------------------------------------
// ZoomCameraForAttnMode
//-----------------------------------------------------------------
function tThinkFunc.ZoomCameraForAttnMode(fTime,tEyePos,tLookAtPos)
	DoEvent("Room.MoveCamera(200, 50, " ..tEyePos[1]..", "..tEyePos[2]..", "..tEyePos[3]..")", Room.m_nCameraChannel)
	DoEvent("Room.TrackCamera(200, 50, "..tLookAtPos[1]..", "..tLookAtPos[2]..", "..tLookAtPos[3]..")", Room.m_nCameraTrackChannel)
	DoEvent("System.WaitForChannel("..Room.m_nCameraTrackChannel..")", Room.m_nCameraChannel)
	DoLuaEvent("PetAI.tThinkFunc.StartAttnModeAfterCameraZoom()",Room.m_nCameraChannel)
end

//-----------------------------------------------------------------
// RemoveAllBlendAnims
//-----------------------------------------------------------------
function tThinkFunc.RemoveAllWalkAnims()
  pet:BlendOut("Run",0.05,-1)
  pet:BlendOut("Walk1",0.05,-1)
  pet:BlendOut("Walk2",0.05,-1)
  pet:BlendOut("Walk3",0.05,-1)
  pet:BlendOut("BodyWalk",0.05,-1)
end

//-----------------------------------------------------------------
// UpdateTubSoap
//-----------------------------------------------------------------
function tThinkFunc.UpdateTubSoap(iTimeBetweenUpdates)

    if (PetAI.szCurTubZone == nil) then return end

		local tCurZoneData = PetAI.tCurTubData[PetAI.szCurTubZone]
    local fPercent = 0

		-- update time for the zone currently being cleaned
		tCurZoneData.iCurLevelTime = tCurZoneData.iCurLevelTime + iTimeBetweenUpdates
    fPercent = iTimeBetweenUpdates/tCurZoneData.iTimePerLevel
    PetAI.tThinkFunc.SetValue(PetAI.tNeeds.tGrooming,fPercent*(PetAI.tNeeds.tGrooming.iMaxValue/8))

		local tAdjacentZone = nil
		local tNonAdjacentZone = nil

		-- update the adjacent zones by a fraction of the time update
		for i in ipairs(tCurZoneData.tAdjacent) do
			tAdjacentZone = PetAI.tCurTubData[tCurZoneData.tAdjacent[i]]
			if (not (tAdjacentZone == nil)) and (tAdjacentZone.iLevel > 1) then
				tAdjacentZone.iCurLevelTime = tAdjacentZone.iCurLevelTime + (iTimeBetweenUpdates/4)
				fPercent = (iTimeBetweenUpdates/4)/tAdjacentZone.iTimePerLevel
		    PetAI.tThinkFunc.SetValue(PetAI.tNeeds.tGrooming,fPercent*(PetAI.tNeeds.tGrooming.iMaxValue/8))
			end
		end

		-- update the non-adjacent zones by a smaller fraction of the time update
		for i in ipairs(tCurZoneData.tNonAdjacent) do
			tNonAdjacentZone = PetAI.tCurTubData[tCurZoneData.tNonAdjacent[i]]
			if (not (tNonAdjacentZone == nil)) and (tNonAdjacentZone.iLevel > 1) then
				tNonAdjacentZone.iCurLevelTime = tNonAdjacentZone.iCurLevelTime + (iTimeBetweenUpdates/8)
				fPercent = (iTimeBetweenUpdates/8)/tNonAdjacentZone.iTimePerLevel
        PetAI.tThinkFunc.SetValue(PetAI.tNeeds.tGrooming,fPercent*(PetAI.tNeeds.tGrooming.iMaxValue/8))
			end
		end

		-- update any zone cleaning level changes
		for k,v in pairs(PetAI.tCurTubData) do
			if (v.iLevel > 1) and (v.iCurLevelTime > v.iTimePerLevel) then
				v.iLevel = v.iLevel - 1
				v.iCurLevelTime = 0

				-- if the current zone's level changes, switch to the appropriate new glitter sprite
				if (k == PetAI.szCurTubZone) then SetActiveGlitterSprite("bubblesLv"..v.iLevel) end
			end
		end
end

//-----------------------------------------------------------------
// InitiateSleepingMode
//-----------------------------------------------------------------
function tThinkFunc.InitiateSleepingMode()

	  PetAI.tPathFunc.EnableTransNodes()

    -- ignore calls temporarily while lying down
    PetAI.bIgnoringCalls = true

	  -- retrieve bed position and closest node
	  local tBedPos = {Room.m_cPetBed:GetBonePosition(Room.m_cPetBed:GetBoneIndex("-Tag Sleep"))}
	  local tBedNode = PetAI.tPathFunc.FindClosestPathNode(tBedPos[1],tBedPos[2],tBedPos[3],"LivingRoom")

    -- walk to the couch
    PetAI.tThinkFunc.LoadNewAnim("Walk1",0.5)
		PetAI.tPathFunc.WalkToPathNode(tBedNode,"BodyWalk","BodyIdle")

    DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
		DoLuaEvent("PetAI.tPathFunc.DisableTransNodes()")
		PetAI.tThinkFunc.DoAnimTransition('SleepBegin','SleepIdle1',pet:GetAnimDuration('SleepBegin')*1000)
    DoEvent("System.Sleep(1000)",ActivePetChannel)
    DoLuaEvent("PetAI.bIgnoringCalls = false",ActivePetChannel)

    --DoLuaEvent("table.insert(PetAI.tActiveEffects,PetAI.tEffects.CreateNeedWhileSleepingEffect(FoxGetTickCount(),PetAI.tNeeds.tAlertness,"..(100/90)..",1000,0))",ActivePetChannel)
    DoLuaEvent("table.insert(PetAI.tActiveEffects,PetAI.tEffects.CreateNeedWhileSleepingEffect(FoxGetTickCount(),PetAI.tNeeds.tAlertness,"..(10)..",1000,0))",ActivePetChannel)

    PetAI.iTimeUntilSleepChange = 4000
   --PetAI.iTimeUntilSleepChange = math.random(10*60*1000)
end

//-----------------------------------------------------------------
// InitiateBrushingMode
//-----------------------------------------------------------------
function tThinkFunc.InitiateBrushingMode()
	local tEyePos = {450,65,65}
	local tLookAtPos = {505,20,15}
	local tPetPos = {505,0,15}

	PetAI.szLocation = "Bathroom"
  Room.SetRoom(4)

  KillToy()

  -- reset jump info (if necessary)
	if (PetAI.tJumpDownInfo != nil) then PetAI.tJumpDownInfo = nil end

	PetAI.tThinkFunc.DoIdle()

  SetCameraPosition(tEyePos[1],tEyePos[2],tEyePos[3])
  SetCameraLookAt(tLookAtPos[1], tLookAtPos[2], tLookAtPos[3])

  local tOrientMatrix = {0,0,-1,0,0,1,0,0,1,0,0,0,tPetPos[1],tPetPos[2],tPetPos[3],1}
  pet:SetOrientation(tOrientMatrix)  

  local DisableUI = function(cElement) cElement:Erase() cElement:Disable() end
  local EnableUI = function(cElement) cElement:Enable() cElement:Paint() end

  EnableUI(Room.m_cExitZoomMode)

  DisableUI(Room.m_cLeftNavButton)
  DisableUI(Room.m_cRightNavButton)
  DisableUI(Room.m_cStoreButton)
  DisableUI(Room.m_cTalentMenuButton)
  DisableUI(Room.m_cOptionsButton)
	DisableUI(Room.m_cInvButton)
	DisableUI(Room.m_cCameraButton)
  DisableUI(Room.m_cRewardButton)
  DisableUI(Room.m_cScoldButton)
  DisableUI(Room.hud_left)
  DisableUI(Room.hud_right)
  DisableUI(Room.trick_frame)
  DisableUI(Room.trick_button)

  Room.SetInvIcon(nil)

  -- force the stats dialog to appear
	Room.stats_dialog:Paint()

  -- disable camera tilting
  Room.m_bTransition = true

  RemoveAllPetAccessories()
end

//-----------------------------------------------------------------
// TerminateBrushingMode
//-----------------------------------------------------------------
function tThinkFunc.TerminateBrushingMode()

  local DisableUI = function(cElement) cElement:Erase() cElement:Disable() end
  local EnableUI = function(cElement) cElement:Enable() cElement:Paint() end

  DisableUI(Room.m_cExitZoomMode)

  EnableUI(Room.m_cStoreButton)
  EnableUI(Room.m_cTalentMenuButton)
	EnableUI(Room.m_cOptionsButton)
	EnableUI(Room.m_cInvButton)
	EnableUI(Room.m_cCameraButton)
  EnableUI(Room.m_cRewardButton)
  EnableUI(Room.m_cScoldButton)
  EnableUI(Room.hud_left)
  EnableUI(Room.hud_right)
  EnableUI(Room.trick_frame)
  EnableUI(Room.trick_button)

  Room.SetRoom(Room.BATHROOM)

	local tAttnNode = PetAI.tPathFunc.tRooms.Bathroom.GetAttentionNode()
	pet:SetPosition(tAttnNode[1],tAttnNode[2],tAttnNode[3])
	DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
	PetAI.tThinkFunc.LoadNewAnim("ShakeOffWater",0.5)
	DoLuaEvent("PetAI.tSoundFX.wavShakeOffWater:Play()",ActivePetChannel)
	DoEvent("System.Sleep("..(pet:GetAnimDuration("ShakeOffWater")*1000-500)..")",ActivePetChannel)
	PetAI.tThinkFunc.DoIdle()
	PetAI.m_bSkipWaterDroplets = true
end

//-----------------------------------------------------------------
// InitiateTubMode
//-----------------------------------------------------------------
function tThinkFunc.InitiateTubMode()
	local tEyePos = {500,75,-30}
	local tLookAtPos = {555,30,-75}
	local tPetPos = {555,10,-75}

	PetAI.szLocation = "Bathroom"
  Room.SetRoom(Room.BATHROOM)

  PetAI.m_bSkipWaterDroplets = false

  TubSoundChannel = CreateChannel()

  -- reset jump info (if necessary)
	if (PetAI.tJumpDownInfo != nil) then PetAI.tJumpDownInfo = nil end

	PetAI.tThinkFunc.DoIdle()

  SetCameraPosition(tEyePos[1],tEyePos[2],tEyePos[3])
  SetCameraLookAt(tLookAtPos[1], tLookAtPos[2], tLookAtPos[3])

  local tOrientMatrix = {0,0,-1,0,0,1,0,0,1,0,0,0,tPetPos[1],tPetPos[2],tPetPos[3],1}
  pet:SetOrientation(tOrientMatrix)  

  local DisableUI = function(cElement) cElement:Erase() cElement:Disable() end
  local EnableUI = function(cElement) cElement:Enable() cElement:Paint() end

  EnableUI(Room.m_cExitZoomMode)

  DisableUI(Room.m_cLeftNavButton)
  DisableUI(Room.m_cRightNavButton)
  DisableUI(Room.m_cStoreButton)
  DisableUI(Room.m_cTalentMenuButton)
  DisableUI(Room.m_cOptionsButton)
	DisableUI(Room.m_cInvButton)
	DisableUI(Room.m_cCameraButton)
  DisableUI(Room.m_cRewardButton)
  DisableUI(Room.m_cScoldButton)
  DisableUI(Room.hud_left)
  DisableUI(Room.hud_right)
  DisableUI(Room.trick_frame)
  DisableUI(Room.trick_button)

  Room.SetInvIcon(nil)

  -- disable camera tilting
  Room.m_bTransition = true

  -- force the stats dialog to appear
	Room.stats_dialog:Paint()

  RemoveAllPetAccessories()

	local iDirtyLevel = math.ceil((PetAI.tNeeds.tGrooming.iMaxValue - PetAI.tNeeds.tGrooming.iCurValue)/(PetAI.tNeeds.tGrooming.iMaxValue/4))
	if (iDirtyLevel == 0) then iDirtyLevel = 1 end

	SetActiveGlitterSprite("bubblesLv"..iDirtyLevel)

  -- play tub music
  StartMusic("tubloop.wav")

  local iGroomingRate = PetAI.tCurShampooInUse.tAttribModifiers.iGroomingMod
  local iCalcTimePerLevel = (25/iGroomingRate)*1000

	PetAI.tCurTubData = {
		Head = { iLevel = iDirtyLevel, iCurLevelTime = 0, iTimePerLevel = iCalcTimePerLevel, tAdjacent = {"Back"}, tNonAdjacent = {"Legs","Tail"}},
		Back = { iLevel = iDirtyLevel, iCurLevelTime = 0, iTimePerLevel = iCalcTimePerLevel, tAdjacent = {"Legs","Head","Tail"}, tNonAdjacent = {}},
		Legs = { iLevel = iDirtyLevel, iCurLevelTime = 0, iTimePerLevel = iCalcTimePerLevel, tAdjacent = {"Back"}, tNonAdjacent = {"Tail","Head"}},
		Tail = { iLevel = iDirtyLevel, iCurLevelTime = 0, iTimePerLevel = iCalcTimePerLevel, tAdjacent = {"Back"}, tNonAdjacent = {"Legs","Head"}},
	}
end


//-----------------------------------------------------------------
// TerminateTubMode
//-----------------------------------------------------------------
function tThinkFunc.TerminateTubMode()

  -- dump tub sound channel
  FlushChannel(TubSoundChannel)
  DestroyChannel(TubSoundChannel)

  -- stop any sfx that may be playing
  Room.m_tHouseSFX.wavTub:Stop()
  Room.m_tHouseSFX.wavShowerStart:Stop()
  Room.m_tHouseSFX.wavShowerLoop:Stop()
  Room.m_tHouseSFX.wavShowerStop:Stop()

  -- stop tub music
  StopMusic("tubloop.wav")

  -- setup helper functions
  local DisableUI = function(cElement) cElement:Erase() cElement:Disable() end
  local EnableUI = function(cElement) cElement:Enable() cElement:Paint() end

  -- disable tub-specific UI
  DisableUI(Room.m_cExitZoomMode)

  -- restore normal game UI
  EnableUI(Room.m_cStoreButton)
  EnableUI(Room.m_cTalentMenuButton)
	EnableUI(Room.m_cOptionsButton)
	EnableUI(Room.m_cInvButton)
	EnableUI(Room.m_cCameraButton)
  EnableUI(Room.m_cRewardButton)
  EnableUI(Room.m_cScoldButton)
  EnableUI(Room.hud_left)
  EnableUI(Room.hud_right)
  EnableUI(Room.trick_frame)
  EnableUI(Room.trick_button)

  -- restore camera settings to bathroom default
  Room.SetRoom(Room.BATHROOM)

  -- set default cursor
  PetAI.SetCursor("Default")

  -- restore normal arrow cursor
  SetCursorEx(CURSOR_ARROW)

  -- move the pet to the bathroom attention mode position
	local tAttnNode = PetAI.tPathFunc.tRooms.Bathroom.GetAttentionNode()
	pet:SetPosition(tAttnNode[1],tAttnNode[2],tAttnNode[3])

	-- play a series of pet anims in closing tub mode
	DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
	PetAI.tThinkFunc.LoadNewAnim("ShakeOffWater",0.5)
	DoLuaEvent("PetAI.tSoundFX.wavShakeOffWater:Play()",ActivePetChannel)
	DoEvent("System.Sleep("..(pet:GetAnimDuration("ShakeOffWater")*1000-500)..")",ActivePetChannel)
	PetAI.tThinkFunc.DoIdle()

  -- dump the current tub data set
	PetAI.tCurTubData = nil

  -- send tip dialog if out of shampoo
  if (PetAI.tThinkFunc.HasNoShampoo()) then PetAI.DoHelpDialog(76) end
end

//-----------------------------------------------------------------
// StartAttnModeAfterCameraZoom
//-----------------------------------------------------------------
function tThinkFunc.StartAttnModeAfterCameraZoom()

	PetAI.bWaitingForCamera = false
 	Room.m_bAttentionMode = true

	-- restore mouse input following camera move
	SetMouseAllowed(TRUE)

  -- send introductory attention mode help dialog
  PetAI.DoHelpDialog(66)

	-- do help dialogs
	if (PetAI.MiscHelpInfo.iNumAttnModeCalls < 3) then
		PetAI.MiscHelpInfo.iNumAttnModeCalls = PetAI.MiscHelpInfo.iNumAttnModeCalls+1
		if (PetAI.MiscHelpInfo.iNumAttnModeCalls == 1) then PetAI.DoHelpDialog(13)
		elseif (PetAI.MiscHelpInfo.iNumAttnModeCalls == 2 and (not PetAI.MiscHelpInfo.bHasSatDown)) then PetAI.DoHelpDialog(14) end
	end

	if (PetAI.tVoiceCommands.Sit.bActive) and (not PetAI.MiscHelpInfo.bHasLaidDown) then
		PetAI.DoHelpDialog(27)
	end


	DoLuaEvent("Room.SetNavButtonsForAttnMode()")
end

//-----------------------------------------------------------------
// TerminateAttentionMode
//-----------------------------------------------------------------
function tThinkFunc.TerminateAttentionMode(bImmediate)

  -- set immediate flag to false as default
  if (bImmediate == nil) then bImmediate = false end

  -- drop attention span to zero
	PetAI.iAttentionSpan = 0

  -- check for help dialog
	if not (PetAI.MiscHelpInfo.szFirstTrickPosition == nil) then PetAI.DoHelpDialog(28,PetAI.MiscHelpInfo.szFirstTrickPosition) end

	-- disable the left/right nav buttons
	Room.DisableNavButtons()

	if (Room.m_bScolding) then
	    Room.EndScolding()
	end

  -- disable mouse input during camera movement
  SetMouseAllowed(FALSE)

  -- reset petting trick ladder
	PetAI.tPettingFunc.ResetTrickLadder()

  -- deactive any floating voice command buttons
	PetAI.tThinkFunc.DeActivateVCButtons()

  if (bImmediate) then
    SetCameraPosition(Room.rooms[Room.m_nCurRoom].HomeEye[1], Room.rooms[Room.m_nCurRoom].HomeEye[2], Room.rooms[Room.m_nCurRoom].HomeEye[3])
    SetCameraLookAt(Room.rooms[Room.m_nCurRoom].HomeLookAt[1], Room.rooms[Room.m_nCurRoom].HomeLookAt[2], Room.rooms[Room.m_nCurRoom].HomeLookAt[3])
	  Room.SetNavButtonsForRoom(Room.m_nCurRoom)
    SetMouseAllowed(TRUE)
  	Room.m_bAttentionMode = false
  else
    SetCursor(CURSOR_ARROW)
  	DoEvent("Room.MoveCamera(200, 100, "..Room.rooms[Room.m_nCurRoom].HomeEye[1]..","..Room.rooms[Room.m_nCurRoom].HomeEye[2]..","..Room.rooms[Room.m_nCurRoom].HomeEye[3]..")", Room.m_nCameraChannel)
  	DoEvent("Room.TrackCamera(200, 100, "..Room.rooms[Room.m_nCurRoom].HomeLookAt[1]..","..Room.rooms[Room.m_nCurRoom].HomeLookAt[2]..","..Room.rooms[Room.m_nCurRoom].HomeLookAt[3]..")", Room.m_nCameraTrackChannel)
	  DoLuaEvent("Room.SetNavButtonsForRoom("..Room.m_nCurRoom..")",Room.m_nCameraChannel)
  	DoLuaEvent("Room.m_bAttentionMode = false", Room.m_nCameraChannel)
  	DoLuaEvent("SetMouseAllowed(TRUE)",Room.m_nCameraChannel)
	end
end

//-----------------------------------------------------------------
// DoPickUpFromGround
//-----------------------------------------------------------------
function tThinkFunc.DoPickUpFromGround()
		PetAI.tThinkFunc.LoadNewAnim({szBody="BodyBendOverBegin",szHead="HeadRetrieveBegin"},1)
		DoEvent("System.Sleep(1200)",ActivePetChannel)
		PetAI.tThinkFunc.LoadNewAnim({szBody="BodyBendOverDrink",szHead="HeadRetrieveMid"},1)
		DoEvent("System.Sleep(1200)",ActivePetChannel)
		PetAI.tThinkFunc.LoadNewAnim({szBody="BodyBendOverEnd",szHead="HeadRetrieveEnd"},1)
		DoEvent("System.Sleep(1200)",ActivePetChannel)
		PetAI.tThinkFunc.LoadNewAnim({szBody="BodyIdle",szHead="HeadRetrieveHoldInMouth",szTail="TailIdle"},1)
end

//-----------------------------------------------------------------
// TempPlayWithPet
//-----------------------------------------------------------------
function tThinkFunc.TempPlayWithPet()
	if not (PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom] == "Backyard") then
		if (DoYesNoDialog(GetLanguageString("Tip_57")) != FALSE) then
			Room.SetRoom(1)
		else return end
	end
end

//-----------------------------------------------------------------
// HasNoFood
//-----------------------------------------------------------------
function tThinkFunc.HasNoFood()
  for i = 1,4 do
    if ((player.m_tInventory["foodcan_0"..i] != nil) and (player.m_tInventory["foodcan_0"..i] != 0)) then return(false) end
    if ((player.m_tInventory["foodbag_0"..i] != nil) and (player.m_tInventory["foodbag_0"..i] != 0)) then return(false) end
  end
  return(true)
end

//-----------------------------------------------------------------
// HasNoWater
//-----------------------------------------------------------------
function tThinkFunc.HasNoWater()
  for i = 1,4 do if ((player.m_tInventory["waterbottle_0"..i] != nil) and (player.m_tInventory["waterbottle_0"..i] != 0)) then return(false) end end
  return(true)
end

//-----------------------------------------------------------------
// HasNoWater
//-----------------------------------------------------------------
function tThinkFunc.HasNoShampoo()
  for i = 1,3 do if ((player.m_tInventory["shampoo_0"..i] != nil) and (player.m_tInventory["shampoo_0"..i] != 0)) then return(false) end end
  return(true)
end

//-----------------------------------------------------------------
// ShampooPet
//-----------------------------------------------------------------
function tThinkFunc.ShampooPet(szInventoryName,szItemName)

	if (DoYesNoDialog(GetLanguageString("Tip_44")) != FALSE) then

	  -- remove the shampoo item from the inventory
  	RemoveItemFromInventory('*', szInventoryName)

  	local tShampooItem = PetAI.tItems[szItemName]
	  PetAI.tThinkFunc.ProcessShampooEffects(tShampooItem)
    PetAI.tCurShampooInUse = tShampooItem

		PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.BATHTUB_MODE)
	else return end
end

//-----------------------------------------------------------------
// BrushPet
//-----------------------------------------------------------------
function tThinkFunc.BrushPet(iTextureNum,tBrushItemRef)

	if (DoYesNoDialog(GetLanguageString("BrushPrompt2")) != FALSE) then

    PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.BRUSHING_MODE)

    PetAI.tThinkFunc.CreateHandheld('Objects/obj_brush.lua',iTextureNum,25)

    SetActiveGlitterSprite('brushcloud')
    PetAI.cBrushInHand = tBrushItemRef
    PetAI.tThinkFunc.ProcessBrushEffects(tBrushItemRef)
	else return end
end

//-----------------------------------------------------------------
// ProcessClickableObject
//-----------------------------------------------------------------
function tThinkFunc.ProcessClickableObject(szObjectName)

  if (PetAI.tCurAction == nil) or ((PetAI.tCurAction != nil) and (PetAI.tCurAction.bInterruptible)) then

  PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.IDLE_MODE)

  if (szObjectName == "Dog House") then
  	-- ************************************************************
  	-- Dog House
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ClimbIntoDogHouse)
  elseif ((szObjectName == "Fireplace") and Room.m_bIsFireplaceOn) then
  	-- ************************************************************
  	-- Fireplace
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptLieDownFireplace)
  elseif (szObjectName == "Pet Bed") then
  	-- ************************************************************
  	-- Pet Bed
  	-- ************************************************************

    local bSleep = false

    if (PetAI.tNeeds.tAlertness.iCurValue < PetAI.tNeeds.tAlertness.iLowThreshold) then bSleep = true
    elseif (PetAI.tNeeds.tAlertness.iCurValue > PetAI.tNeeds.tAlertness.iHighThreshold) then bSleep = false
    elseif (math.random(PetAI.tNeeds.tAlertness.iCurValue) < PetAI.tNeeds.tAlertness.iMaxValue/2) then bSleep = true end

    if (bSleep) then PetAI.tThinkFunc.ChangeAction(PetAI.tActions.GoToBed)
    else PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ClimbOntoPetBed) end

  elseif (szObjectName == "Sofa") then
  	-- ************************************************************
  	-- Sofa
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ClimbOntoSofa)
  elseif (szObjectName == "Scratching Post") then
  	-- ************************************************************
  	-- Scratching Post
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScratchScratchingPost)
  elseif (szObjectName == "BoomBox") then
    -- ************************************************************
  	-- Boom Box
  	-- ************************************************************
    if ((radio.m_nCurStation == 0) or (radio.m_nCurStation == nil)) then PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptLookWhineRadio)
    else PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptLookBarkRadio) end

  elseif (szObjectName == "Refrigerator") then
  	-- ************************************************************
  	-- Refrigerator
  	-- ************************************************************
    if (Room.m_bIsFridgeOpen) then PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptSniffOpenFridge)
    else PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptScratchBarkFridge) end

  elseif (szObjectName == "Coffee Table") then
  	-- ************************************************************
  	-- Coffee Table
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ClimbOntoTable)
  elseif (szObjectName == "Kitchen Faucet") then
  	-- ************************************************************
  	-- Kitchen Faucet
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptLookBarkKitchenSink)
  elseif (szObjectName == "Cat Perch") then
  	-- ************************************************************
  	-- Cat Perch
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ClimbOntoPerch)
  elseif (szObjectName == "Bathroom Faucet") then
  	-- ************************************************************
  	-- Bathroom Faucet
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptLookBarkBathroomSink)
  elseif (szObjectName == "Toilet") then
  	-- ************************************************************
  	-- Toilet
  	-- ************************************************************

    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptSniffToilet)

  elseif (szObjectName == "Tub") then
  	-- ************************************************************
  	-- Tub
  	-- ************************************************************
    PetAI.tThinkFunc.ChangeAction(PetAI.tActions.ScriptLookBarkTub)
  end

  end
end

//-----------------------------------------------------------------
// ProcessBrushEffects
//-----------------------------------------------------------------
function tThinkFunc.ProcessBrushEffects(tBrushItem)

  -- for each of the pet's stats...
	for k,v in pairs(PetAI.tStats) do

	  -- retrieve the effector's specified modifier value for the particular stat
		local iMaxMod = tBrushItem.tAttribModifiers["i" .. v.szName .. "Mod"]

		if not (iMaxMod == 0) then

			-- if the modifier value is non-zero, create a time-release additive stat effect
		  local StatEffect = PetAI.tEffects.CreateBrushStatTimeReleaseEffect(FoxGetTickCount(),v,iMaxMod,2000,5000)

			-- insert the newly-created effect into the current active set
			table.insert(PetAI.tActiveEffects,StatEffect)
		end
	end
end

//-----------------------------------------------------------------
// ProcessShampooEffects
//-----------------------------------------------------------------
function tThinkFunc.ProcessShampooEffects(tShampooItem)

  -- for each of the pet's stats...
	for k,v in pairs(PetAI.tStats) do

	  -- retrieve the effector's specified modifier value for the particular stat
		local iMaxMod = tShampooItem.tAttribModifiers["i" .. v.szName .. "Mod"]

		if not (iMaxMod == 0) then

			-- if the modifier value is non-zero, create a time-release additive stat effect
		  local StatEffect = PetAI.tEffects.CreateTubStatTimeReleaseEffect(FoxGetTickCount(),v,iMaxMod,2000,5000)

			-- insert the newly-created effect into the current active set
			table.insert(PetAI.tActiveEffects,StatEffect)
		end
	end
end

//-----------------------------------------------------------------
// WarpTo
//-----------------------------------------------------------------
function tThinkFunc.WarpTo(szNewRoom)
  
  local x,y,z = pet:GetPosition()
  
  PetAI.tThinkFunc.DumpAllAnims()
  PetAI.tThinkFunc.MakeAvailable()
                
  PetAI.szLocation = PetAI.tPathFunc.GetRoomPetIsIn()
	local tDoorNode = PetAI.tPathFunc.GetDoorwayNode(PetAI.szLocation,szNewRoom)
	if (PetAI.tJumpDownInfo != nil) then PetAI.tJumpDownInfo = nil end
	pet:SetPosition(tDoorNode[1],tDoorNode[2],tDoorNode[3])
	PetAI.szLocation = szNewRoom
          
  local tOrientMatrix = nil
  if (x > tDoorNode[1]) then tOrientMatrix = {0,0,-1,0,0,1,0,0,1,0,0,0,tDoorNode[1],tDoorNode[2],tDoorNode[3],1}
  else tOrientMatrix = {0,0,1,0,0,1,0,0,-1,0,0,0,tDoorNode[1],tDoorNode[2],tDoorNode[3],1} end
  pet:SetOrientation(tOrientMatrix)       
end

//-----------------------------------------------------------------
// FeedPet
//-----------------------------------------------------------------
function tThinkFunc.FeedPet(szInventoryName,szItemName)

	if not (PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom] == "Kitchen") then
		if (DoYesNoDialog(GetLanguageString("Tip_42")) != FALSE) then
			Room.SetRoom(3)
			FlushChannel(ActivePetChannel)
			PetAI.tThinkFunc.WarpTo("Kitchen")
			PetAI.tSoundFX.wavWhistle:Play()
		else return
		end
	end

  -- change state to eating mode
  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.IDLE_MODE) or (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE))) then
    PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.EATING_MODE,false)
  else PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.EATING_MODE) end

  -- remove the food item from the inventory
	RemoveItemFromInventory('*', szInventoryName)

  -- send tip dialog if out of food
  if (PetAI.tThinkFunc.HasNoFood()) then PetAI.DoHelpDialog(74) end

  -- launch a particle effect around the food bowl
  Room.HighlightFoodBowl()

  -- reset food counter
  PetAI.iFoodInBowl = 100

	-- make the pet food object visible
	Room.m_cKibble:SetVisible(1)

  -- set food item to be eaten
  PetAI.tFoodItemInBowl = PetAI.tItems[szItemName]

	-- give the food item to the pet
	DoLuaEvent("PetAI.tThinkFunc.GiveItem(PetAI.tItems."..szItemName..")",ActivePetChannel)
end

//-----------------------------------------------------------------
// WaterPet
//-----------------------------------------------------------------
function tThinkFunc.WaterPet(szInventoryName,szItemName)

	if (PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom] != "Kitchen") then
		if (DoYesNoDialog(GetLanguageString("Tip_43")) != FALSE) then
			Room.SetRoom(3)
			FlushChannel(ActivePetChannel)			
			PetAI.tThinkFunc.WarpTo("Kitchen")
			PetAI.tSoundFX.wavWhistle:Play()
		else return
		end
	end

  -- change state to eating mode
  if (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.IDLE_MODE) or (PetAI.tThinkFunc.IsInState(PetAI.PET_STATES.ATTENTION_MODE))) then
    PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.DRINKING_MODE,false)
  else PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.DRINKING_MODE) end

  -- remove the water item from the inventory
	RemoveItemFromInventory('*', szInventoryName)

  -- send tip dialog if out of water
  if (PetAI.tThinkFunc.HasNoWater()) then PetAI.DoHelpDialog(75) end

  -- launch a particle effect around the water bowl
  Room.HighlightWaterBowl()

  -- reset water counter
  PetAI.iWaterInBowl = 100

	-- make the bowl water object visible
	Room.m_cBowlWater:SetVisible(1)

  -- set water item to be drank
  PetAI.tWaterItemInBowl = PetAI.tItems[szItemName]

  -- give the water item to the pet
	DoLuaEvent("PetAI.tThinkFunc.GiveItem(PetAI.tItems."..szItemName..")",ActivePetChannel)
end


//-----------------------------------------------------------------
// TempJumpToBackyard
//-----------------------------------------------------------------
function tThinkFunc.JumpToBackyard()
	if not (PetAI.tPathFunc.tRoomNames[Room.m_nCurRoom] == "Backyard") then
		if (DoYesNoDialog(GetLanguageString("Tip_57")) != FALSE) then
			Room.SetRoom(1)
			PetAI.tThinkFunc.WarpTo("Backyard")
			PetAI.tSoundFX.wavWhistle:Play()
		else
  		return(false)
  	end
  end
	return(true)
end

//-----------------------------------------------------------------
// LoadToyFetchTrick
//-----------------------------------------------------------------
function tThinkFunc.LoadToyFetchTrick()
  PetAI.szLocation = PetAI.tPathFunc.GetRoomPetIsIn()

  -- change state to informal trick mode
  if (PetAI.szLocation == "Backyard") then
    if ((PetAI.tCurAction == nil) or ((PetAI.tCurAction != nil) and (PetAI.tCurAction.bInterruptible))) then FlushChannel(ActivePetChannel) end    
    PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.INFORMAL_TRICK,false)
  else PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.INFORMAL_TRICK) end
  
	PetAI.tActiveInformalTrick = PetAI.tInformalTricks.CreateFetchTrick(toy,25.0)
end

//-----------------------------------------------------------------
// LoadToyPlayTrick
//-----------------------------------------------------------------
function tThinkFunc.LoadToyPlayTrick()
  PetAI.szLocation = PetAI.tPathFunc.GetRoomPetIsIn()
  
  -- change state to informal trick mode
  if (PetAI.szLocation == "Backyard") then
    if ((PetAI.tCurAction == nil) or ((PetAI.tCurAction != nil) and (PetAI.tCurAction.bInterruptible))) then FlushChannel(ActivePetChannel) end    
    PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.INFORMAL_TRICK,false)
  else PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.INFORMAL_TRICK) end
  
  PetAI.tActiveInformalTrick = PetAI.tInformalTricks.CreatePlayWithToyTrick(toy,20.0)
end

//-----------------------------------------------------------------
// DoTempAction
//-----------------------------------------------------------------
function tThinkFunc.DoTempAction(tAction)
	PetAI.tThinkFunc.MakeAvailable()
	PetAI.tThinkFunc.ChangeAction(tAction)
end

//-----------------------------------------------------------------
// KillHandheld
//-----------------------------------------------------------------
function tThinkFunc.KillHandheld()
  if not (cHandheld == nil) then
    if (PetAI.cBrushInHand != nil) then
      globals.TOY_DRAG_Z_DISTANCE = fToyZDistanceOrig
      fToyZDistanceOrig = nil
      SetActiveGlitterSprite("default")
      PetAI.cBrushInHand = nil
    end
    cHandheld:UnLoad()
    cHandheld = nil
  end
end

//-----------------------------------------------------------------
// InitiateBrushMode
//-----------------------------------------------------------------
function tThinkFunc.CreateHandheld(szFilename,iTextureNum,fZDist)

    -- destroy the current toy in use (if any)
    KillToy()

    -- destroy the current handheld object (if any)
    PetAI.tThinkFunc.KillHandheld()

    if not (fZDist == nil) then
      fToyZDistanceOrig = globals.TOY_DRAG_Z_DISTANCE
      globals.TOY_DRAG_Z_DISTANCE = fZDist
    end

    cHandheld = Room.AddObjectToScene(szFilename,0,0,0)

    if (iTextureNum != nil) then cHandheld:SetTexture(cHandheld.data.available_textures[iTextureNum]) end

    return(cHandheld)
end

//-----------------------------------------------------------------
// LoadBrush
//-----------------------------------------------------------------
function tThinkFunc.LoadBrush(iTextureNum,tBrushItemRef)

    PetAI.tThinkFunc.CreateHandheld('Objects/obj_brush.lua',iTextureNum,25)

    SetActiveGlitterSprite('brushcloud')
 		PetAI.tThinkFunc.ChangeState(PetAI.PET_STATES.ATTENTION_MODE)
    PetAI.cBrushInHand = tBrushItemRef

    PetAI.tThinkFunc.ProcessBrushEffects(tBrushItemRef)
end


//-----------------------------------------------------------------
// InitiateTalentShow
//-----------------------------------------------------------------
function tThinkFunc.InitiateTalentShow()

  local szDiffLevel = TALENT_SHOW_DIFF_LEVEL
  local iDiffNum = TALENT_SHOW_DIFF_NUM

  -- destroy temp variables
  TALENT_SHOW_DIFF_LEVEL = nil
  TALENT_SHOW_DIFF_NUM = nil

  -- reset jump info (if necessary)
	if (PetAI.tJumpDownInfo != nil) then PetAI.tJumpDownInfo = nil end

  PetAI.TalentShow.DialogBox:Link(Room.this)
  PetAI.TalentShow.DialogBoxText:Link(Room.this)

  -- remove any toys or handheld objects
  PetAI.tThinkFunc.KillHandheld()
  KillToy()

  -- clear out all rising text elements
  Room.DumpAllRisingText()

  -- hide the inventory window (if open)
  HideInventory()

  -- toggle the talent timer sprite
  ToggleTalentTimer()

  -- paint the pet's learned trick voice command buttons onscreen
  Room.trick_frame:Paint()

  -- turn off glitter during the talent show
  SetAllowGlitter(FALSE)

  -- shutdown the radio
  radio.ShutDown()

  -- Stop any Camera Tilting
  FlushChannel(Room.m_nCameraTrackChannel)

  PetAI.iRoomBeforeTalentShow = Room.m_nCurRoom

  -- if a talent show is already active, return immediately
  if (PetAI.TalentShow.bActive) then return end

  -- camera eye position for talent show
  local tEyePos = {749, 86, 103}

  -- pet position variable
	local tPetPos = {747,0,-35}

	-- set the pet's position and orientation
	pet:SetPosition(tPetPos[1],tPetPos[2],tPetPos[3])
	pet:FacePoint(tPetPos[1]+200,tPetPos[2],tPetPos[3]+200)

  -- disable camera tilting
  Room.m_bTransition = true

  -- set the room and pet location
  Room.SetRoom(5)
	PetAI.szLocation = "TalentShow"

  -- make the pet stand up in preparation
	PetAI.tThinkFunc.DoStand()
  --PetAI.tThinkFunc.DoIdle()
  PetAI.tThinkFunc.MakeAvailable()

  -- move the camera into position
  FlushChannel(Room.m_nCameraTrackChannel)
	DoEvent("Room.MoveCamera(100, 20, " ..tEyePos[1]..", "..tEyePos[2]..", "..tEyePos[3]..")", Room.m_nCameraTrackChannel)

	-- disable different UI controls
	local EnableUI = function(cElement) cElement:Enable() cElement:Paint() end
  local EraseUI = function(cElement) cElement:Erase() cElement:Disable() end
  local DisableUI = function(cElement) cElement:Disable() end

  EnableUI(Room.m_cReturnToHouseButton)

  DisableUI(Room.m_cStoreButton)
  DisableUI(Room.trick_frame)
  DisableUI(Room.stats_dialog)

  EraseUI(Room.m_cTalentMenuButton)
  EraseUI(Room.m_cStatsButton)
	EraseUI(Room.m_cInvButton)
	EraseUI(Room.m_cCameraButton)
  EraseUI(Room.m_cRewardButton)
  EraseUI(Room.m_cScoldButton)
  EraseUI(Room.hud_right)
  EraseUI(Room.m_cRightNavButton)
  EraseUI(Room.m_cLeftNavButton)
  EraseUI(Room.stats_dialog)

  PetAI.TalentShow.Initialize(szDiffLevel,iDiffNum)
end

//-----------------------------------------------------------------
// TerminateTalentShow
//-----------------------------------------------------------------
function tThinkFunc.TerminateTalentShow()

  -- toggle the talent timer sprite
  ToggleTalentTimer()

  -- restore position in house
  Room.SetRoom(PetAI.iRoomBeforeTalentShow)

  -- restore pet's position to match player's
  if (PetAI.iRoomBeforeTalentShow == Room.BATHROOM) then PetAI.szLocation = "Bathroom"
  elseif (PetAI.iRoomBeforeTalentShow == Room.KITCHEN) then PetAI.szLocation = "Kitchen"
  elseif (PetAI.iRoomBeforeTalentShow == Room.LIVINGROOM) then PetAI.szLocation = "LivingRoom"
  elseif (PetAI.iRoomBeforeTalentShow == Room.BACKYARD) then PetAI.szLocation = "Backyard" end

  -- restore pet to attention mode position
	local tAttnNode = PetAI.tPathFunc.tRooms[PetAI.szLocation].GetAttentionNode()
	pet:SetPosition(tAttnNode[1],tAttnNode[2],tAttnNode[3])

  PetAI.tThinkFunc.DoIdle()

  -- flush the camera movement channel
  FlushChannel(Room.m_nCameraTrackChannel)

  -- terminate the talent show
  PetAI.TalentShow.Terminate()

  -- stop sound fx
  PetAI.tSoundFX.wavTSEnding:Stop()

  -- destroy temp talent show variables
  PetAI.iRoomBeforeTalentShow = nil

	-- enable different UI controls
  local EnableUI = function(cElement) cElement:Enable() cElement:Paint() end
  local EraseUI = function(cElement) cElement:Erase() end

  EnableUI(Room.m_cStoreButton)
  EnableUI(Room.m_cTalentMenuButton)
  EnableUI(Room.m_cStatsButton)
	EnableUI(Room.m_cInvButton)
	EnableUI(Room.m_cCameraButton)
  EnableUI(Room.m_cRewardButton)
  EnableUI(Room.m_cScoldButton)
  EnableUI(Room.hud_right)
  EnableUI(Room.trick_frame)
  EnableUI(Room.stats_dialog)
  EraseUI(Room.m_cReturnToHouseButton)
end

//-----------------------------------------------------------------
// DoPromenade
//-----------------------------------------------------------------
function tThinkFunc.DoPromenade()
	PetAI.tThinkFunc.LoadNewAnim('Promenade',0.25)

 	if (PetAI.tThinkFunc.IsDog()) then DoEvent("System.Sleep(5000)",ActivePetChannel)
  else DoEvent("System.Sleep(6000)",ActivePetChannel) end

 	DoLuaEvent("PetAI.tThinkFunc.DoIdle()",ActivePetChannel)
end

//-----------------------------------------------------------------
// DoDefecate
//-----------------------------------------------------------------
function tThinkFunc.DoDefecate()
  TempPooChannel = CreateChannel()

  if (PetAI.szDefecateToDo == "Poop") then

    -- make the pet poo
    PetAI.tThinkFunc.DoStandAndSingleAnim("Defecate","wavPee",pet:GetAnimDuration("Defecate")*1000)
    DoEvent("System.Sleep(1200)", TempPooChannel)
    DoLuaEvent("PetAI.tThinkFunc.CreateAndDropPoo()",TempPooChannel)
    DoLuaEvent("PetAI.szDefecateToDo = 'Pee'",TempPooChannel)

  elseif (PetAI.szDefecateToDo == "Pee") then

    if not (CATZ and (PetAI.szLocation == "Bathroom")) then
      -- walk to appropriate peeing location
      PetAI.tThinkFunc.LoadNewAnim({szBody="Walk1"},0.5)
  	  PetAI.tPathFunc.WalkToPathNode(PetAI.tPathFunc.tRooms[PetAI.szLocation].GetPeeNode(),"BodyWalk","BodyIdle",fPetSpeed)
      DoEvent("System.WaitForChannel("..ActivePetChannel..")",TempPooChannel)

      -- make the pet pee
      DoLuaEvent("PetAI.tThinkFunc.DoStandAndSingleAnim('Urinate','wavPee',pet:GetAnimDuration('Urinate')*1000)",TempPooChannel)
      DoEvent("System.Sleep(1200)", TempPooChannel)
      DoLuaEvent("PetAI.tThinkFunc.CreatePeeStain()",TempPooChannel)
    else
      PetAI.tThinkFunc.DoStandAndSingleAnim("Urinate","wavPee",pet:GetAnimDuration("Urinate")*1000)
      DoEvent("System.Sleep(1200)", ActivePetChannel)
    end

    DoLuaEvent("PetAI.szDefecateToDo = 'Poo'",TempPooChannel)
  end

  DoLuaEvent("PetAI.tNeeds.tBladder.bDecayOn = false",TempPooChannel)
  DoLuaEvent("PetAI.tNeeds.tBladder.iCurValue = 100",TempPooChannel)

  --DoLuaEvent("DestroyChannel("..TempPooChannel..")", TempPooChannel)
end

//-----------------------------------------------------------------
// CreateAndDropPoo
//-----------------------------------------------------------------
function tThinkFunc.CreatePeeStain()

  if not (cPeeStain == nil) then
    cPeeStain:UnLoad()
    cPeeStain = nil
  end

  local iPeeOffset = 0
  if (PetAI.szLocation == "Bathroom") then iPeeOffset = 1
  elseif (PetAI.szLocation == "Kitchen") then iPeeOffset = 0
  elseif (PetAI.szLocation == "LivingRoom") then iPeeOffset = 0 end

	cPeeStain = C3DPhysicsObject()
	cPeeStain = Load3DObject("Objects/toy_peestain.lua", cPeeStain)
	cPeeStain:SetTexture(cPeeStain.data.available_textures[1])

  cPeeStain:DisableCollisions()

  cPeeStain:SetVisible(1)
  cPeeStain:Link(Room.m_cSceneManager)

  PetAI.szPeeLocation = PetAI.szLocation

  local iPooSource = pet:GetBoneIndex("-tag Butt")
  local x,y,z = pet:GetBonePosition(iPooSource)
  cPeeStain:SetPosition(x,iPeeOffset,z)
end

//-----------------------------------------------------------------
// CreateAndDropPoo
//-----------------------------------------------------------------
function tThinkFunc.CreateAndDropPoo()

  if not (cPoo == nil) then
    cPoo:UnLoad()
    cPoo = nil
  end

	cPoo = C3DPhysicsObject()
	cPoo = Load3DObject("Objects/toy_poop.lua", cPoo)
	cPoo:SetTexture(cPoo.data.available_textures[1])

  cPoo:DisableCollisions()

  cPoo:SetVisible(1)
  cPoo:Link(Room.m_cSceneManager)

  local iPooSource = pet:GetBoneIndex("-tag Butt")
  local x,y,z = pet:GetBonePosition(iPooSource)

  PetAI.MiscHelpInfo.iNumPoopMesses = PetAI.MiscHelpInfo.iNumPoopMesses+1
  if (PetAI.MiscHelpInfo.iNumPoopMesses == 1) then PetAI.DoHelpDialog(69)
  elseif (PetAI.MiscHelpInfo.iNumPoopMesses >= 2) then PetAI.DoHelpDialog(70) end

  PetAI.szPooLocation = PetAI.szLocation

  cPoo:SetPosition(x,y,z)
  cPoo:Reset(x,y,z)
  cPoo:Throw(0,1,-1)
end

//-----------------------------------------------------------------
// JumpUpOntoObject
//-----------------------------------------------------------------
function tThinkFunc.JumpUpOntoObject(tPosition,szHeight,szOrientSide)

  -- grab coordinates
  local x = tPosition[1]
  local y = tPosition[2]
  local z = tPosition[3]

  -- stand up, if not already standing
  PetAI.tThinkFunc.DoStand()

  local tOrientPos = {x,y,z}
  if (szOrientSide == "Right") then tOrientPos[1] = tOrientPos[1] - 200
  elseif (szOrientSide == "Left") then tOrientPos[1] = tOrientPos[1] + 200
  elseif (szOrientSide == "Away") then tOrientPos[3] = tOrientPos[3] - 200
  elseif (szOrientSide == "Toward") then tOrientPos[3] = tOrientPos[3] + 200 end

  -- face the position to be climbed to
  DoEvent("Pet.FacePosition("..(5)..","..tOrientPos[1]..","..tOrientPos[2]..","..tOrientPos[3]..")",ActivePetChannel)

  -- run the jump animation, then sleep while it plays
  PetAI.tThinkFunc.LoadNewAnim("JumpOntoSurface"..szHeight,0.25,true)
  DoEvent("System.Sleep("..(pet:GetAnimDuration("JumpOntoSurface"..szHeight)*1000-(0.25*1000))..")",ActivePetChannel,true)

  -- snap the pet's position to the point atop the climbed obstacle
  DoLuaEvent("pet:SetPosition("..x..","..y..","..z..")",ActivePetChannel,true)

  -- load in idle anims immediately to wipe out jump anim's positional movement
  PetAI.tThinkFunc.LoadNewAnim({szHead="HeadIdle",szBody="BodyIdle",szTail="TailIdle"},0,true)

  -- face camera
  DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
end

//-----------------------------------------------------------------
// JumpDownFromObject
//-----------------------------------------------------------------
function tThinkFunc.JumpDownFromObject()

  local x = PetAI.tJumpDownInfo.tPosition[1]
  local y = PetAI.tJumpDownInfo.tPosition[2]
  local z = PetAI.tJumpDownInfo.tPosition[3]
  local szHeight = PetAI.tJumpDownInfo.szHeight
  local szOrientSide = PetAI.tJumpDownInfo.szOrientSide

  -- stand pet up
  PetAI.tThinkFunc.DoStand()

  -- reset call down flag
  PetAI.bCanBeCalledDown = false

  local tOrientPos = {x,y,z}
  if (szOrientSide == "Right") then tOrientPos[1] = tOrientPos[1] - 200
  elseif (szOrientSide == "Left") then tOrientPos[1] = tOrientPos[1] + 200
  elseif (szOrientSide == "Away") then tOrientPos[3] = tOrientPos[3] - 200
  elseif (szOrientSide == "Toward") then tOrientPos[3] = tOrientPos[3] + 200 end

  -- face the position to be climbed down to
  DoEvent("Pet.FacePosition("..(5)..","..tOrientPos[1]..","..tOrientPos[2]..","..tOrientPos[3]..")",ActivePetChannel)

  DoLuaEvent("PetAI.bIgnoringCalls = true",ActivePetChannel)
  DoLuaEvent("Room.DisableNavButtons()",ActivePetChannel)

  -- run the jump down animation, then sleep while it plays
  PetAI.tThinkFunc.LoadNewAnim("JumpDownFrom"..szHeight,0.25,true)
  DoEvent("System.Sleep("..(pet:GetAnimDuration("JumpDownFrom"..szHeight)*1000-(0.25*1000))..")",ActivePetChannel)

  -- snap the pet's position to the point on the floor
  DoLuaEvent("pet:SetPosition("..x..","..y..","..z..")",ActivePetChannel,true)

  -- load in idle anims immediately to wipe out jump down anim's positional movement
  PetAI.tThinkFunc.LoadNewAnim({szHead="HeadIdle",szBody="BodyIdle",szTail="TailIdle"},0,true)

  -- face camera and idle
  DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)

  DoLuaEvent("PetAI.tThinkFunc.DoIdle()",ActivePetChannel)

  DoLuaEvent("Room.SetNavButtonsForRoom("..Room.m_nCurRoom..")",ActivePetChannel)
  DoLuaEvent("PetAI.bIgnoringCalls = false",ActivePetChannel)

  -- set flags and restore pet action availability
  DoLuaEvent("PetAI.tJumpDownInfo = nil",ActivePetChannel)
  DoLuaEvent("PetAI.tThinkFunc.MakeAvailable()",ActivePetChannel)
end

//-----------------------------------------------------------------
// KillPooAndPee
//-----------------------------------------------------------------
function tThinkFunc.KillPooAndPee()
  if (cPoo != nil) then
    cPoo:UnLoad()
    cPoo = nil
  elseif (cPeeStain != nil) then
    cPeeStain:UnLoad()
    cPeeStain = nil
  end
end

//-----------------------------------------------------------------
// TempToggleAnim
//-----------------------------------------------------------------
function tThinkFunc.ToggleAnim()
	ClearAllTempOutputBox()
	PetAI.tThinkFunc.LoadNewAnim(PetAI.tAnimList[PetAI.iAnimIndex],1)
	PetAI.iAnimIndex = PetAI.iAnimIndex + 1
	if (PetAI.iAnimIndex > table.getn(PetAI.tAnimList)) then PetAI.iAnimIndex = 1 end
end

//-----------------------------------------------------------------
// DoChannelSleep
//-----------------------------------------------------------------
function tThinkFunc.DoChannelSleep(iTime)
  DoEvent("System.Sleep("..iTime..")",ActivePetChannel)
end

//-----------------------------------------------------------------
// DoPetSleep
//-----------------------------------------------------------------
function tThinkFunc.DoPetSleep(iTime)
  PetAI.tThinkFunc.DoStand()
  PetAI.tThinkFunc.DoAnimTransition("SleepBegin","SleepIdle"..math.random(3),pet:GetAnimDuration("SleepBegin")*1000)
  PetAI.tThinkFunc.DoChannelSleep(iTime)
  PetAI.tThinkFunc.DoAnimTransition("SleepEnd",{szBody="BodyIdle",szHead="HeadIdle",szTail="TailIdle"},pet:GetAnimDuration("SleepEnd")*1000)
end

//-----------------------------------------------------------------
// GetTagPrefix
//-----------------------------------------------------------------
function tThinkFunc.GetTagPrefix()
  -- determine appropriate tag point prefix based on species, breed
  local szTagPrefix = "Dogs"
  if (PetAI.tThinkFunc.IsCat()) then szTagPrefix = "Cat"
  elseif (PetAI.tBreed.szName == "Pug") then szTagPrefix = "Pug" end
  return(szTagPrefix)
end

//-----------------------------------------------------------------
// DoActionOnObject
//-----------------------------------------------------------------
function tThinkFunc.DoActionOnObject(tOnPos,tOffPos,szObjHeight,szJumpUpDir,szJumpDownDir,DoActionFunc,tLandPos)

    -- set call down flag
    PetAI.bCanBeCalledDown = false

    -- set jump down info
    PetAI.tJumpDownInfo = {tPosition = tOffPos, szHeight = szObjHeight, szOrientSide = szJumpDownDir}

    -- if a separate landing position was given, use it instead for the jump down
    if (tLandPos != nil) then PetAI.tJumpDownInfo.tPosition = tLandPos end

    local Enable = function()  DoLuaEvent("PetAI.bIgnoringCalls = true Room.DisableNavButtons()",ActivePetChannel) end
    local Disable = function() DoLuaEvent("Room.SetNavButtonsForRoom("..Room.m_nCurRoom..") PetAI.bIgnoringCalls = false",ActivePetChannel) end

    Enable()
    PetAI.tThinkFunc.DoWalkToClosestNode(tOffPos)
    PetAI.tThinkFunc.JumpUpOntoObject(tOnPos,szObjHeight,szJumpUpDir)
    Disable()
    DoLuaEvent("PetAI.bCanBeCalledDown = true",ActivePetChannel)
    DoActionFunc()
    DoLuaEvent("PetAI.bCanBeCalledDown = false",ActivePetChannel)
    Enable()
    PetAI.tThinkFunc.JumpDownFromObject()
    Disable()
end

//-----------------------------------------------------------------
// DoFaceCamera
//-----------------------------------------------------------------
function tThinkFunc.DoFaceCamera()
  DoEvent("Room.MakePetFaceCamera("..PetAI.fTurningSpeed..")",ActivePetChannel)
end

//-----------------------------------------------------------------
// DoFaceObject
//-----------------------------------------------------------------
function tThinkFunc.DoFaceObject(cObject)
  local x,y,z = cObject:GetPosition()
  DoEvent("Pet.FacePosition("..(5)..","..x..","..y..","..z..")",ActivePetChannel)
end

//-----------------------------------------------------------------
// DoWalkToClosestNode
//-----------------------------------------------------------------
function tThinkFunc.DoWalkToClosestNode(tPosition)
	PetAI.tPathFunc.EnableTransNodes()

	local tClosestNode = PetAI.tPathFunc.FindClosestPathNode(tPosition[1],tPosition[2],tPosition[3],PetAI.szLocation)

  PetAI.tThinkFunc.LoadNewAnim("Walk1",0.5)
  PetAI.tPathFunc.WalkToPathNode(tClosestNode,"BodyWalk","BodyIdle")

  DoLuaEvent("PetAI.tPathFunc.DisableTransNodes()",ActivePetChannel)
end

//-----------------------------------------------------------------
// DoFleeEffect
//-----------------------------------------------------------------
function tThinkFunc.DoFleeEffect(iElapsed)
  if (iElapsed - PetAI.iLastFleeTime > 500) then

   	local tRandBones = {"Bip01 TailNub","Bip01 Tail3","Bip01 Tail2","Bip01 Tail1","Bip01 Tail","Bip01 Pelvis","Bip01 Spine","Bip01 Spine1","Bip01 Spine2","Bip01 R Thigh","Bip01 R Calf","Bip01 R HorseLink","Bip01 R Foot","Bip01 R Toe0","Bip01 R Toe0Nub","Bip01 L Thigh","Bip01 L Calf","Bip01 L HorseLink","Bip01 L Foot","Bip01 L Toe0","Bip01 L Toe0Nub","Bip01 L UpperArm","Bip01 L Forearm","Bip01 L Hand","Bip01 L Finger0","Bip01 L Finger0Num","Bip01 R UpperArm","Bip01 R Forearm","Bip01 R Hand","Bip01 R Finger0","Bip01 R Finger0Num","Bip01 Neck","Bip01 Neck1","Bip01 Head","Bip01 HeadNub","NUB","Bip01 BottomJaw","Bip01 BottomJawNub","Thongue_01","Thongue_02","Bip01 Ponytail1","Bip01 Ponytail11","Bip01 Ponytail1Nub","Bip01 Ponytail2","Bip01 Ponytail21","Bip01 Ponytail2Nub","Bip01 L Clavicle","Bip01 R Clavicle","-tag Hat01","-tag Collar01","-tag Mouth01","-tag TailPom01","-tag Balance","EyeLid_Upper_R","EyeLid_Upper_L","EyeLid_Upper_NUB","EyeLid_Lower_NUB","EyeLid_Lower_R","EyeLid_Lower_L","-tag Bootie01","-tag Bootie02","-tag Bootie03","-tag Bootie04"}
   	local iBone = pet:GetBoneIndex(tRandBones[math.random(table.getn(tRandBones))])

   	local xx, yy, zz = pet:GetBonePosition(iBone)
   	local x, y = GetScreenPosition(xx, yy, zz)

    DoGlitter("flees", x + math.random(20) - 10, y + math.random(20) - 10, TRUE, 500,100)

    PetAI.iLastFleeTime = iElapsed
  end
end

//-----------------------------------------------------------------
// ExecuteRandom
//-----------------------------------------------------------------
function tThinkFunc.ExecuteRandomAction(tSet)
  PetAI.tActions[tSet[math.random(table.getn(tSet))]].ExecuteAction(false)
end


//-----------------------------------------------------------------
// RunMapDbgMode
//-----------------------------------------------------------------
function tThinkFunc.RunMapDbgMode()

  -- render the path map
  PetAI.tPathFunc.RenderPathMap()

  -- erase UI elements
  local EraseUI = function(cElement) cElement:Erase() cElement:Disable() end
  EraseUI(Room.m_cStoreButton)
  EraseUI(Room.trick_frame)
  EraseUI(Room.stats_dialog)
  EraseUI(Room.m_cTalentMenuButton)
  EraseUI(Room.m_cStatsButton)
	EraseUI(Room.m_cInvButton)
	EraseUI(Room.m_cCameraButton)
  EraseUI(Room.m_cRewardButton)
  EraseUI(Room.m_cScoldButton)
  EraseUI(Room.hud_left)
  EraseUI(Room.hud_right)
  EraseUI(Room.stats_dialog)
end

return(tThinkFunc)
