------------------------------------------------------------------------------
-- Globals
------------------------------------------------------------------------------

--
-- Which application is it?
--
DOGZ = (fox.app.GetAppName() == "Dogz")
CATZ = (fox.app.GetAppName() == "Catz")
assert((CATZ or DOGZ) and not (CATZ and DOGZ))

-- Since BOOL is an I32, all FOX bindings that use BOOL expect 0, or 1
-- Passing true, or false to these functions will cause a Lua script error.
-- Therefore you should use TRUE and FALSE instead.
TRUE = 1
FALSE = 0

-- Sprite positioning
--
EPOS = {
	TOPLEFT		= 0,
	TOPRIGHT	= 1,
	BOTTOMLEFT	= 2,
	BOTTOMRIGHT = 3,
	CENTERLEFT	= 4,
	CENTERRIGHT = 5,
	TOPCENTER	= 6,
	BOTTOMCENTER = 7,
   	CENTER		= 8,
}

-- Sprite Z-Order
--
ZORDER =
{
	TOPMOST	   = 0,
	FOREGROUND = 1073741823,
	MIDDLE	   = 2147483647,
	BACKGROUND = 3221225470,
	HINDMOST   = 4294967295,
}

-- Key codes for OnKeyHit
-- NOTE: Must match FoxKeys.h
--
Keys = {
    ENTER = 0x0D,
    -- TODO: Add more keys...
}

-- Blit Effects for CxSprite.SetEffects()
--
BLT_EFFECTS = {
    BLT_EFFECT_NONE = 0,
    BLT_INVISIBLE = 16,
}

-- Cursor IDs
--
CURSOR_ARROW = 0
CURSOR_BUSY = 1
CURSOR_CARET = 2
CURSOR_INVALID = 3
CURSOR_HAND = 4
CURSOR_SHAMPOO = 11
CURSOR_BRUSH = 12
CURSOR_SHOWER = 13
CURSOR_SPRAY = 14
CURSOR_ACTIVE = 15
CURSOR_LEFT = 16
CURSOR_RIGHT = 17
CURSOR_UP = 18


--------------------------------------------------------
-- Font Formatting Flags
--
TEXT_DX_LEFT =      0x00000001    -- Left justified
TEXT_DX_RIGHT =     0x00000002    -- Right justified
TEXT_DX_CENTER =    0x00000004    -- Justify center

-- Mutually Exclusive in this group
TEXT_DY_TOP =       0x00000008    -- Align top
TEXT_DY_BOTTOM =    0x00000010    -- Align bottom
TEXT_DY_CENTER =    0x00000020    -- Align center

TEXT_SHADOWED =     0x00000040    -- Text has shadow
TEXT_WORD_WRAP =    0x00000080    -- Use word wrap if text does not fit

-- Mutually Exclusive in this group
TEXT_TOP_LEFT = (TEXT_DY_TOP + TEXT_DX_LEFT)
TEXT_TOP_RIGHT = (TEXT_DY_TOP + TEXT_DX_RIGHT)
TEXT_TOP_CENTER = (TEXT_DY_TOP + TEXT_DX_CENTER)
TEXT_BOTTOM_LEFT = (TEXT_DY_BOTTOM + TEXT_DX_LEFT)
TEXT_BOTTOM_RIGHT = (TEXT_DY_BOTTOM + TEXT_DX_RIGHT)
TEXT_BOTTOM_CENTER = (TEXT_DY_BOTTOM + TEXT_DX_CENTER)
TEXT_CENTER_LEFT = (TEXT_DY_CENTER + TEXT_DX_LEFT)
TEXT_CENTER_RIGHT = (TEXT_DY_CENTER + TEXT_DX_RIGHT)
TEXT_CENTER_CENTER = (TEXT_DY_CENTER + TEXT_DX_CENTER)
TEXT_PRESERVE_ALPHA = 0x10000000

--------------------------------------------------------

-- Button Styles
--
NORMAL_BUTTON = 0
RADIO_BUTTON = 1
CHECKBOX_BUTTON = 2

-- Button states/actions
--
BUTTON_UP = 0                   -- Button is in Normal state (not pushed in)
BUTTON_DOWN = 1                   -- Button is Pushed in
BUTTON_FOCUS_UP = 2                   -- Button has mouse over it but is not puched in
BUTTON_FOCUS_DOWN = 3                   -- Button is pushed in with mouse over it
BUTTON_DISABLED = 4                   -- Button is disabled
BUTTON_FOCUS = BUTTON_FOCUS_UP     -- Mouse is over button
BUTTON_ON = BUTTON_DOWN         -- RadioButton is ON
BUTTON_OFF = BUTTON_UP           -- RadioButton is OFF
BUTTON_CHECKED = BUTTON_ON           -- CheckBox is ON
BUTTON_UNCHECKED =BUTTON_OFF         -- CheckBox is OFF


-- D3D9 Light types
--
LIGHT_POINT = 1
LIGHT_SPOT = 2
LIGHT_DIRECTIONAL = 3

globals = {
    KENNEL_PRICE = 20,

    MIN_PUSH_DISTANCE = 20.0,
    MAX_POKE_DISTANCE = 5.0,
    MIN_PETTING_SWIPE = 10.0,

    TWEAK_BLEND_QUATERNION_ANGULAR_VELOCITY = 0.80,
    TWEAK_BLEND_EPSILON = 1e-3,
    TWEAK_BLEND_MAX_SLERP = 0.4,

    TOY_DRAG_Z_DISTANCE = 65.0,

    TOOLTIP_POINTSIZE = 14,

    TREATS_PER_CAN = 20,

    camera = {
        near_clip = 5.0,
        far_clip = 2400.0,
        field_of_view = 45.0,
    },
}

if CATZ then
    globals.preview_offset = {
        x = -1,
        y = 4,
        z = -4,
    }
elseif DOGZ then
    globals.preview_offset = {
        x = 0,
        y = 0,
        z = 1.5,
    }
end

dofile("physics.lua")

-- Default Trick of the Day prize amounts

TODPrize_OneEasyTrick = 5
TODPrize_OneHardTrick = 10
TODPrize_TwoTricks = 25
TODPrize_ThreeTricks = 40
TODPrize_OneFetch = 5
TODPrize_TwoFetches = 15
TODPrize_ThreeFetches = 30
TODPrize_Balance = 10
TODPrize_Treat = 15
TODPrize_EasyTrickPhoto = 10
TODPrize_HardTrickPhoto = 20
TODPrize_BeautyPhoto = 5
TODPrize_CutePhoto = 5
TODPrize_PhotoMinimum = 5

util = util or {}

function math.udir(x)
    if x >= 0 then
        return 1
    else
        return -1
    end
end

function util.Bool2String(b)
    if (b) then
        return "true"
    else
        return "false"
    end
end

function util.String2Bool(s)
    if (s == "true") then
        return true
    else
        return false
    end
end

function util.GetGlobals()
    local t = {}
    for k, v in pairs(_G) do
        if (type(v) != "function") then
            table.insert(t, k)
        end
    end

    return(t)
end


function util.DiffTables(t1, t2)
    local t = {}
    local bFound

    for i, v in ipairs(t2) do

        bFound = false

        for ii, vv in ipairs(t1) do
            if (v == vv) then
                bFound = true
                break
            end
        end
        if (not bFound) then
            table.insert(t, v)
        end
    end

    for i, v in ipairs(t) do
        print(v)
    end
end

function util.GlobalSnapshot()
    util.globals = util.GetGlobals()
end

function util.DiffGlobals()
    util.DiffTables(util.globals, util.GetGlobals())
end

function util.DumpStack()
    local level = 1
    while true do
        local info = debug.getinfo(level, "Sn")
        if not info then break end

        local s = string.format("%d] %s ", level, info.what);

        if (info.name != nil) then
            s = s .. info.name
        end

        LogInfo(s)

        level = level + 1
    end
end

util.GlobalSnapshot()

function SetCursorEx(n)
    SetCursor(n)
    SetDefaultCursor(n)
end

function GetTrickOfTheDay()
  if not (PetAI.tCurTrickOfTheDay == nil) then
    return(PetAI.tCurTrickOfTheDay.szText)
  else
    return(GetLanguageString("TOD_None"))
  end
end


function SaveStats(tPetAI, cSaveGameFile)

    -- cSaveGameFile must be a CxOptions object

    assert(tPetAI != nil)

    cSaveGameFile:SetString("Pet", "Name", tPetAI.szName)
    cSaveGameFile:SetString("Pet", "Breed", tPetAI.tBreed.szName)
    cSaveGameFile:SetString("Pet", "DOB", tPetAI.sBirthday)
    cSaveGameFile:SetString("Pet", "Gender", PetAI.szGender)

	for k,v in pairs(tPetAI.tStats) do
	    cSaveGameFile:SetString("Stats", k,  ""..v.iCurValue..", ".. v.iMaxValue)
	end

	for k,v in pairs(tPetAI.tNeeds) do
	    cSaveGameFile:SetString("Needs", k,  ""..v.iCurValue..", ".. v.iMaxValue)
	end

	for k,v in pairs(tPetAI.tActions) do
	    if (v.bReinforceable) then
	        cSaveGameFile:SetFloat("Actions", k, v.fMemValue)
	    end
	end

	for i,v in ipairs(tPetAI.tHelpDialogInfo) do
	    cSaveGameFile:SetString("HelpDialogFlags", "Tip_"..i, util.Bool2String(v))
	end

  for k,v in pairs(tPetAI.TalentShow.tCurRanking) do
	    cSaveGameFile:SetInt("TalentShowRankings", k, v)
  end

  for k,v in pairs(tPetAI.tTalentShowCounts) do
	    cSaveGameFile:SetInt("TalentShowCounts", k, v)
  end

  for k,v in pairs(tPetAI.tTalentShowWins) do
	    cSaveGameFile:SetInt("TalentShowWins", k, v)
  end

end


function LoadStats(tPetAI, cSaveGameFile)

    -- cSaveGameFile must be a CxOptions object

    assert(tPetAI != nil)

    tPetAI.szName = cSaveGameFile:GetString("Pet", "Name", tPetAI.szName)
    tPetAI.sBirthday = cSaveGameFile:GetString("Pet", "DOB", tPetAI.sBirthday)

    PetAI.szGender = cSaveGameFile:GetString("Pet", "Gender", PetAI.szGender)
    --tPetAI.tThinkFunc.SetBreed(cSaveGameFile:GetString("Pet", "Breed", tPetAI.szBreedName))

	for k,v in pairs(tPetAI.tStats) do
	    local x, y = cSaveGameFile:GetNumericPair("Stats", k, v.iCurValue, v.iMaxValue)
	    tPetAI.tStats[k].iCurValue, tPetAI.tStats[k].iMaxValue = math.floor(x), math.floor(y)
	end

	for k,v in pairs(tPetAI.tNeeds) do
	    local x, y = cSaveGameFile:GetNumericPair("Needs", k, v.iCurValue, v.iMaxValue)

	    tPetAI.tNeeds[k].iCurValue = math.floor(x)
	    tPetAI.tNeeds[k].iMaxValue = math.floor(y)
	end

	for k,v in pairs(tPetAI.tActions) do
	    if (v.bReinforceable) then
	        tPetAI.tActions[k].fMemValue = cSaveGameFile:GetFloat("Actions", k, v.fMemValue)
	    end
	end

	for i,v in ipairs(tPetAI.tHelpDialogInfo) do
	    tPetAI.tHelpDialogInfo[i] = util.String2Bool(cSaveGameFile:GetString("HelpDialogFlags", "Tip_"..i, util.Bool2String(v)))
	end

  for k,v in pairs(tPetAI.TalentShow.tCurRanking) do
    tPetAI.TalentShow.tCurRanking[k] = cSaveGameFile:GetInt("TalentShowRankings",k,v)
  end

  for k,v in pairs(tPetAI.tTalentShowCounts) do
    tPetAI.tTalentShowCounts[k] = cSaveGameFile:GetInt("TalentShowCounts",k,v)
  end

  for k,v in pairs(tPetAI.tTalentShowWins) do
    tPetAI.tTalentShowWins[k] = cSaveGameFile:GetInt("TalentShowWins",k,v)
  end

end


SaveGame = {

    m_cUserData = CxOptions(),

    Restore = function(filename)

        LogInfo("SaveGame.Restore()")

        -- Close any current SaveFile
        SaveGame.m_cUserData:Close()

        -- Open the user's saved game file
        SaveGame.m_cUserData:Open(filename)

        LoadPet(SaveGame.m_cUserData:GetString("Pet", "File", "invalid_file.omg"))

        player = {}
        player.m_nMoney = SaveGame.m_cUserData:GetInt("Pet", "Money", 0)

        player.m_cTreats = {}

        player.m_cTreats.treats_01 = SaveGame.m_cUserData:GetInt("Pet", "treats_01", 0)
        player.m_cTreats.treats_02 = SaveGame.m_cUserData:GetInt("Pet", "treats_02", 0)
        player.m_cTreats.treats_03 = SaveGame.m_cUserData:GetInt("Pet", "treats_03", 0)

        player.m_szPetFileName = SaveGame.m_cUserData:GetString("Pet", "File", gszBreed)
        player.m_nCurRoom = SaveGame.m_cUserData:GetInt("House", "CurRoom", 1)
        player.m_pModel = nil            -- Pet Model (Cx3DObject)

        PetAI.m_bKenneled = util.String2Bool(SaveGame.m_cUserData:GetString("Pet", "IsKenneled", "false"))
        PetAI.m_nKennelTime = SaveGame.m_cUserData:GetInt("Pet", "KennelTime", 0)

        -- tricks are saved as comma seperated list (i.e. "Sit, Stay, Beg")
        local s = SaveGame.m_cUserData:GetString("Pet", "Tricks", "")

        player.m_tTricks = {}
        for p in string.gfind(s, "%w+") do
            table.insert(player.m_tTricks, p)
        end

        -- Inventory is stored as:
        --
        -- [Inventory]
        -- BookOfTricks=1
        --
        local t = SaveGame.m_cUserData:GetSection("Inventory")

        player.m_tInventory = {}
        for k, v in pairs(t) do
            player.m_tInventory[k] = tonumber(v)
        end

        assert(PetAI != nil)
        LoadStats(PetAI, SaveGame.m_cUserData)

        -- Load Pet info
        assert(pet != nil)

        SetPetFur(SaveGame.m_cUserData:GetInt("Pet", "Texture", 1))

        SetPetAccessory("collar", SaveGame.m_cUserData:GetString("Collar", "FileName", ""), "-tag collar01", SaveGame.m_cUserData:GetInt("Collar", "Texture", 1))
        SetPetAccessory("hat", SaveGame.m_cUserData:GetString("Hat", "FileName", ""), "-tag hat01", SaveGame.m_cUserData:GetInt("Hat", "Texture", 1))

        SetPetAccessory("bootie1", SaveGame.m_cUserData:GetString("Bootie1", "FileName", ""), "-tag bootie01", SaveGame.m_cUserData:GetInt("Bootie1", "Texture", 1))
        SetPetAccessory("bootie2", SaveGame.m_cUserData:GetString("Bootie2", "FileName", ""), "-tag bootie02", SaveGame.m_cUserData:GetInt("Bootie2", "Texture", 1))
        SetPetAccessory("bootie3", SaveGame.m_cUserData:GetString("Bootie3", "FileName", ""), "-tag bootie03", SaveGame.m_cUserData:GetInt("Bootie3", "Texture", 1))
        SetPetAccessory("bootie4", SaveGame.m_cUserData:GetString("Bootie4", "FileName", ""), "-tag bootie04", SaveGame.m_cUserData:GetInt("Bootie4", "Texture", 1))

        SetPetAccessory("tailpom", SaveGame.m_cUserData:GetString("tailpom", "FileName", ""), "-tag tailpom01", SaveGame.m_cUserData:GetInt("tailpom", "Texture", 1))

        -- TODO: anything else?
        --SetPetAccessory("collar", SaveGame.m_cUserData:GetString("Collar", "FileName", ""), "-tag collar01", SaveGame.m_cUserData:GtInt("Collar", "Texture", 1))

    end,

    Save = function()

        LogInfo("SaveGame.Save()")

        --assert(SaveGame.m_cUserData:IsOpen())

        SaveGame.m_cUserData:SetString("Pet", "File", player.m_szPetFileName)
        SaveGame.m_cUserData:SetInt("Pet", "Money", player.m_nMoney)
        SaveGame.m_cUserData:SetInt("Pet", "treats_01", player.m_cTreats.treats_01)
        SaveGame.m_cUserData:SetInt("Pet", "treats_02", player.m_cTreats.treats_02)
        SaveGame.m_cUserData:SetInt("Pet", "treats_03", player.m_cTreats.treats_03)
        SaveGame.m_cUserData:SetInt("House", "CurRoom", player.m_nCurRoom)

        SaveGame.m_cUserData:SetString("Pet", "IsKenneled", util.Bool2String(PetAI.m_bKenneled))
        SaveGame.m_cUserData:SetInt("Pet", "KennelTime", PetAI.m_nKennelTime)

        -- Save list of tricks as a string:
        -- [Pet]
        -- Tricks=Beg, Sit, Stay
        --
        local s = ""
        for i, v in ipairs(player.m_tTricks) do
            if (i == 1) then
                s = v
            else
                s = s..", "..v
            end

        end
        SaveGame.m_cUserData:SetString("Pet", "Tricks", s)

        SaveGame.m_cUserData:DeleteSection("Inventory")

        for k, v in pairs(player.m_tInventory) do
            SaveGame.m_cUserData:SetInt("Inventory", k, v)
        end

        assert(PetAI != nil)
        SaveStats(PetAI, SaveGame.m_cUserData)

        -- Save Pet info (Collar, texture, hat, booties, etc...)
        --
        SaveGame.m_cUserData:SetInt("Pet", "Texture", pet.texture)

        -- Remove all possible accessorie data
        --
        SaveGame.m_cUserData:DeleteSection("Collar")
        SaveGame.m_cUserData:DeleteSection("Hat")
        SaveGame.m_cUserData:DeleteSection("Bootie1")
        SaveGame.m_cUserData:DeleteSection("Bootie2")
        SaveGame.m_cUserData:DeleteSection("Bootie3")
        SaveGame.m_cUserData:DeleteSection("Bootie4")
        SaveGame.m_cUserData:DeleteSection("tailpom")

        -- and re-fill
        for k, v in pairs(pet.acc) do
            SaveGame.m_cUserData:SetString(k, "FileName", v.filename)
            SaveGame.m_cUserData:SetInt(k, "Texture", v.texture)
        end

        SaveGame.m_cUserData:Save()

        -- Record last used pet
        LogInfo("lastPet="..SaveGame.m_cUserData:GetFileName())
        fox.options:SetString("Petz", "LastPet", SaveGame.m_cUserData:GetFileName())

    end,

    Init = function(filename, name)

        LogInfo("SaveGame.Init()")

        -- Close any current SaveFile
        SaveGame.m_cUserData:Close()

        -- Create the master player info structure
        player = {}
        player.m_nMoney = 100               -- Initial money
        player.m_szPetFileName = gszBreed   -- Selected pet
        player.m_nCurRoom = 1               -- Initial house location
        player.m_pModel = nil               -- Pet Model (Cx3DObject)
        player.m_tTricks = {}               -- No Tricks unlocked
        player.m_tInventory = {             -- Initial starting inventory
            BookOfTricks = 1,
            remove_clothes = 1,
            treats_03 = globals.TREATS_PER_CAN,
            foodbag_04 = 1,
            waterbottle_01 = 1,
            ball_01 = 1,
            foodbowl_01 = 1,
            kitchenchair_01 = 1,
            kitchentable_01 = 1,
            fridge_01 = 1,
            Kitchen_01 = 1,
            petbed_01 = 1,
            sofa_01 = 1,
            coffeetable_01 = 1,
            arearug_01 = 1,
            livingroom_01 = 1,
            toilet_01 = 1,
            tub_01 = 1,
            Bathroom_01 = 1,
            doghouse_01 = 1,
            brush_01 = 1,
        }

        -- Add in the user's choices for collar, and fur (both should be in the range [1..3]
        player.m_tInventory["fur_0"..pet.texture] = 1
        player.m_tInventory["collar_0"..pet.acc["collar"].texture] = 1

        if (fox.app.GetAppName() == "Catz") then
            player.m_tInventory["scratchingpost_01"] = 1
            player.m_tInventory["perch_02"] = 1
            player.m_tInventory["litterbox_01"] = 1
        end

        player.m_cTreats = {}
        player.m_cTreats.treats_01 = player.m_tInventory["treats_01"] or 0
        player.m_cTreats.treats_02 = player.m_tInventory["treats_02"] or 0
        player.m_cTreats.treats_03 = player.m_tInventory["treats_03"] or 0

        PetAI.szName = name
        PetAI.m_bKenneled = true
        PetAI.m_nKennelTime = FoxGetTimeSeconds()

        -- Create new SaveFile
        SaveGame.m_cUserData:Create(filename)

        -- Save Init settings now
        SaveGame.Save()

    end,
}

function AddOneTreat(item)
    player.m_cTreats[item] = player.m_cTreats[item] + 1
end

function OneLessTreat(kind)

    if (kind == nil) then

        local order = { "treats_02", "treats_01", "treats_03" }

        for i, v in ipairs(order) do
            if (player.m_cTreats[v] != nil and player.m_cTreats[v] > 0) then
                kind = v
                break
            end
        end
    end

    if (kind != nil) then
        gLastTreatTaken = kind
        player.m_cTreats[kind] = player.m_cTreats[kind] - 1
        RemoveItemFromInventory("Food", kind)
    end

    return(kind)
end

function SetHouseAccessory(name, texture_index)

    local o = Room["m_c"..name]

    if (o != nil) then
        o.texture = texture_index
        local t = o.data.available_textures[texture_index]
        if (t != nil) then
            o:SetTexture(t)
        end
    end
end


function SetPetFur(texture_index)
    pet.texture = texture_index
    pet:SetTexture(pet.data.available_textures[pet.texture])
end

function SetPetAccessory(name, filename, bone, texture_index)

    MyAssert(pet != nil)

    pet.acc = pet.acc or {}

    if (pet.acc[name] != nil) then
        pet.acc[name]:UnLoad()
        pet.acc[name] = nil
    end

    if (filename != "") then

        pet.acc[name] = Load3DObject(filename)
        pet.acc[name]:SetTexture(pet.acc[name].data.available_textures[texture_index])
        pet.acc[name]:Link(pet)
        pet.acc[name]:SetAttachmentBoneIndex(pet:GetBoneIndex(bone))

        pet.acc[name].texture = texture_index
        pet.acc[name].filename = filename
    end
end

function PrivateGetPlayerMoney()
    return(player.m_nMoney)
end

function PrivateSetPlayerMoney(nTotal)
    assert(player != nil)

    player.m_nMoney = nTotal
end

-- Set the default money handlers
SetPlayerMoney = PrivateSetPlayerMoney
GetPlayerMoney = PrivateGetPlayerMoney

function HideToolTip()
	if (Room.m_pToolTip != nil) then
		Room.m_pToolTip:Kill()
	end
end

function ShowToolTip(x, y, text)

	-- Hide any current tooltip
	HideToolTip()

	if (g_bShowToolTips) then

    	text = string.gsub(text, '\n', ' ', 99)

    	local cx, cy = GetTextSize(text, g_nFontArial, globals.TOOLTIP_POINTSIZE, 200, 40)

    	-- TODO: Create a box for the text to sit in

    	local o = CxDrawText()
    	o:Create(0, 0, cx + 8 + 1, cy + 1)
    	o:SetFont(g_nFontArial)
    	o:SetPointSize(globals.TOOLTIP_POINTSIZE)
    	o:SetBkColor(0xBEF5E797)
    	o:SetString(text)
    	o:SetDefaultDepth(32)
    	o:SetFlags(TEXT_CENTER_CENTER + TEXT_PRESERVE_ALPHA + TEXT_WORD_WRAP)
    	o:SetZOrder(ZORDER.TOPMOST)
    	o:LinkRoot()
    	o:Paint(x, y, EPOS.BOTTOMCENTER)

    	-- Keep it on screen
    	x, y = o:GetPosition()

    	if (x < 0) then
    		x = 0
    	end
    	if (y < 0) then
    		y = 0
    	end
    	o:Paint(x, y)

    	x, y = o:GetPosition(EPOS.BOTTOMRIGHT)

    	if (x > Room.this:GetWidth() - 1) then
    		x = Room.this:GetWidth() - 1;
    	end
    	if (y > Room.this:GetHeight() - 1) then
    		y = Room.this:GetHeight() - 1;
    	end

    	o:Paint(x, y, EPOS.BOTTOMRIGHT)

    	Room.m_pToolTip = o
    end
end

function DoToolTipForButton(button, name)
	if (button != nil) then
		local x, y = button:GetGlobalPosition(EPOS.TOPCENTER)

		if (name == nil) then
			name = button:GetName()
		end

		ShowToolTip(x, y, GetLanguageString("ToolTip_"..name))
	end
end

function NewDrawText(x, y, cx, cy, text, font, color, bkcolor)

    if (font == nil) then
        font = g_nFontArial
    end

    if (color == nil) then
        color = 0
    end

    if (bkcolor == nil) then
        bkcolor = 0
    end

    local o = CxDrawText()
	o:Create(x, y, cx, cy)
	o:SetFont(font)
	o:SetBkColor(bkcolor)
	o:SetColor(color)
	o:SetString(text)
	o:SetDefaultDepth(32)
	o:SetFlags(TEXT_CENTER_CENTER + TEXT_PRESERVE_ALPHA)

	return o
end


function Text(t)

    local o = CxDrawText()

    if (t.size != nil) then
        o:Create(0, 0, t.size[1], t.size[2])
    end

    if (t.rect != nil) then
        o:Create(t.area[1], t.area[2], t.area[3] - t.area[1] + 1, t.area[4] - t.area[2] + 1)
    end

    if (t.position != nil) then
        if (t.epos != nil) then
            o:Paint(t.position[1], t.position[2], t.epos)
        else
            o:Paint(t.position[1], t.position[2])
        end
    end

    if (t.zorder != nil) then
        o:SetZOrder(t.zorder)
    end

    if (t.parent != nil) then
        o:Link(t.parent)
    end

    if (t.font == nil) then
        t.font = g_nFontArial
    end

    if (t.color == nil) then
        t.color = 0
    end

    if (t.bkcolor == nil) then
        t.bkcolor = 0
    end

    if (t.flags == nil) then
        t.flags = TEXT_TOP_LEFT
    end

    if (t.pointsize == nil) then
        t.pointsize = 12
    end

	o:SetFont(t.font)
	o:SetPointSize(t.pointsize)
	o:SetBkColor(t.bkcolor)
	o:SetColor(t.color)
	o:SetString(t.text)
	o:SetDefaultDepth(32)
	o:SetFlags(t.flags + TEXT_PRESERVE_ALPHA)

	o:Paint()

	return o
end

function Sprite(t)

    local o = CxSprite()

    if (t.filename != nil) then
      if (t.numframes != nil) then
        o:LoadSurface(t.filename,t.numframes)
      else
        o:LoadSurface(t.filename)
      end
    end

    if (t.position != nil) then
        if (t.epos != nil) then
            o:Paint(t.position[1], t.position[2], t.epos)
        else
            o:Paint(t.position[1], t.position[2])
        end
    end

    if (t.parent != nil) then
        o:Link(t.parent)
    end

	return o
end

function MyAssert(b)
    if (not b) then
        util.DumpStack()
        assert(false)
    end
end

function Load3DObject(filename, o)

    o = o or Cx3DObject()

    local t = dofile(filename)

    o.data = t

    -- Create a handy function for switching textures by index (from available_textures)
    o.SetTextureIndex = function(self, index) self:SetTexture(self.data.available_textures[index]) end

    -- Create a function for caching all available textures
    o.PreLoadAllTextures = function(self) for i, v in ipairs(self.data.available_textures) do PreLoadTexture(v) end end

    o:LoadModel(t.base_model)

    if (t.anims != nil) then
        for i, v in ipairs(t.anims) do
            o:LoadAnim(v.name, v.file, v.bone or "")
        end
    end

    o:SetAttachOrientation(TRUE)
    if (t.attach_orientation != nil and t.attach_orientation == false) then
        o:SetAttachOrientation(FALSE)
    end

    if (t.attach_offset != nil) then
        o:SetOffset(t.attach_offset[1], t.attach_offset[2], t.attach_offset[3])
    end

    if (t.texture != nil) then
        o:SetTexture(t.texture)
    end

    if t.material then
        if t.material.ambient then
            o:SetMaterialAmbient(t.material.ambient[1],
                                 t.material.ambient[2],
                                 t.material.ambient[3],
                                 t.material.ambient[4])
        end
        if t.material.diffuse then
            o:SetMaterialDiffuse(t.material.diffuse[1],
                                 t.material.diffuse[2],
                                 t.material.diffuse[3],
                                 t.material.diffuse[4])
        end
        if t.material.specular then
            o:SetMaterialSpecular(t.material.specular[1],
                                  t.material.specular[2],
                                  t.material.specular[3],
                                  t.material.specular[4])
        end
        if t.material.emissive then
            o:SetMaterialEmissive(t.material.emissive[1],
                                  t.material.emissive[2],
                                  t.material.emissive[3],
                                  t.material.emissive[4])
        end
        if t.material.power then
            o:SetMaterialPower(t.material.power)
        end
    end

    if t.has_alpha != nil then
        if t.has_alpha == true then
            o:SetHasAlpha(TRUE)
        end
    end

    if t.two_sided != nil then
        if t.two_sided == true then
            o:SetTwoSided(TRUE)
        end
    end

    if t.foreground != nil then
        if t.foreground == true then
            o:SetForeground(TRUE)
        end
    end

    o:SetName(t.name or filename)

    -- If there was a scale set for this model, then use it
    o:SetScale(t.scale or 1.0)

    return(o)
end


function UnLoadPet()

    if (pet != nil) then
        UnRegisterPet()
        pet:UnLoad()
        pet = nil

    end

    if (PetAI != nil) then
        if (PetAI.TalentShow != nil) then
            if (PetAI.TalentShow.DialogBox != nil) then
                PetAI.TalentShow.DialogBox:Kill()
                PetAI.TalentShow.DialogBox = nil
            end
            PetAI.TalentShow = nil
        end
    end

    -- Get rid of all the Pet globals
    --
    PetAI = nil
    TalentShow = nil
    tPathFunc = nil
    tThinkFunc = nil
    fSatiateModifier = nil
    fDepleteModifier = nil
end

function LoadPet(filename)
    UnLoadPet()

    pet = Load3DObject(filename, CPet())
    pet:RegisterEventName("Pet")
    pet:SetOrientation({-1, 0, 0, 0, 0, 1, 0, 0, 0, 0, -1, 0, 0, 0, 0, 1})
    -- Load the Shadow for under the pet
    local o = Load3DObject("objects/shadow.lua")

    o:Link(pet)
    --o:SetAttachmentBoneIndex(pet:GetBoneIndex("-tag Shadow"))
    o:SetAttachmentBoneIndex(pet:GetBoneIndex("Bip01 SPine1"))
    o:SetFlags(2)
    o:SetFixedPos(0, 0.4, 0)
    pet.shadow = o

    if CATZ then
        pet.whiskers = Load3DObject("objects/whiskers.lua")
        pet.whiskers:Link(pet)
        pet.whiskers:SetVisible(TRUE)
        pet.whiskers:SetAttachmentBoneIndex(pet:GetBoneIndex("-tag whiskers"))
        pet.whiskers:SetLocalOrientation({ 0,-1, 0, 0,
                                           0, 0,-1, 0,
                                           1, 0, 0, 0,
                                           0, 0, 0, 1})
    end

    RegisterPet(pet)
end


function NewLuaButton(t, o)
    o = o or CLuaButton()

    local nNumFrames = 5

    if (t.name != nil) then o:SetName(t.name) end
    if (t.num_frames != nil) then nNumFrames = t.num_frames end
    if (t.file != nil) then o:LoadSurface(t.file, nNumFrames) end

    if (t.position != nil) then o:Paint(t.position[1], t.position[2]) end

    if (t.parent != nil) then o:Link(t.parent) end
    if (t.OnClick != nil) then o:SetOnClicked(t.OnClick) end
    if (t.OnChecked != nil) then o:SetOnChecked(t.OnChecked) end

    -- TODO: Add more settings

    return o
end


-- Manages Event Channels
--
EventChannelManager = {

    CreateChannel = function()
        local n = CreateChannel()

        EventChannelManager.m_cChannelList = EventChannelManager.m_cChannelList or {}

        table.insert(EventChannelManager.m_cChannelList, n)

        return(n)
    end,

    DestroyChannel = function(nChannel)

        if (EventChannelManager.m_cChannelList != nil) then

            for i, v in ipairs(EventChannelManager.m_cChannelList) do
                if (v == nChannel) then
                    table.remove(EventChannelManager.m_cChannelList, i)
                    DestroyChannel(nChannel)
                    break
                end
            end
        end
    end,

    DestroyAll = function()
        if (EventChannelManager.m_cChannelList != nil) then

            for i, v in ipairs(EventChannelManager.m_cChannelList) do
                DestroyChannel(v)
            end
            EventChannelManager.m_cChannelList = nil
        end
    end,

    Pause = function()
        if (EventChannelManager.m_cChannelList != nil) then

            for i, v in ipairs(EventChannelManager.m_cChannelList) do
                PauseChannel(v)
            end
        end
    end,

    Resume = function()
        if (EventChannelManager.m_cChannelList != nil) then

            for i, v in ipairs(EventChannelManager.m_cChannelList) do
                ResumeChannel(v)
            end
        end
    end,
}


----------------------------------
-- Creates CxWave objects
--
function CreateSound(t)

    local o = CxWave()

    o:Load(t.filename)

    if (t.loops != nil) then
        o:SetNumLoops(t.loops)
    end

    if (t.volume != nil) then
        o:SetVol(t.volume)
    end

    if (t.flags != nil) then
        o:SetVol(t.flags)
    end

    if (t.position != nil) then
        o.position = t.position
    end

    if (t.falloff_distance == nil) then
        t.falloff_distance = 400
    end

    o.SetDistanceVol = function(self, x, y, z)
        local dist = math.sqrt((x - self.position[1]) * (x - self.position[1]) + (y - self.position[2]) * (y - self.position[2]) + (z - self.position[3]) * (z - self.position[3]))

        local nVol = 127 * ((t.falloff_distance - dist) / t.falloff_distance)

        if (nVol < 0) then
            nVol = 0
        end

        if (nVol > 127) then
            nVol = 127
        end

        self:SetVol(math.floor(nVol))
    end

    return(o)
end
