#pragma once
#include "../acl_cpp_define.hpp"
#include <vector>
#include <utility>
#include "../stdlib/string.hpp"
#include "../stdlib/noncopyable.hpp"

#if !defined(ACL_CLIENT_ONLY) && !defined(ACL_REDIS_DISABLE)

namespace acl
{

/**
 * Ҫ redis_cluster ȡйؼȺ redis Ϣ
 * this class is mainly used for redis_cluster command class to
 * get some information about the nodes in redis cluster
 */
class ACL_CPP_API redis_node : public noncopyable
{
public:
	/**
	 * ʹô˹캯ʵʱҪ set_id  set_addr 
	 *  redis Ψһʶַͬʱɵ set_xxx ÷
	 */
	redis_node(void);
	~redis_node(void);

	/**
	 * ڹ캯еĲдý ID ʶ⣬ͨ˺
	 * set the node's  ID
	 * @param id {const char*} Ⱥ redis Ψһʶ
	 *  the unique ID for one redis node in the reids cluster
	 * @return {redis_node&}
	 */
	redis_node& set_id(const char* id);

	/**
	 * ڹ캯еĲдýĵַ⣬ͨ˺
	 * set the node's listening addr
	 * @param addr {const char*} Ⱥ redis ķַʽip:port
	 *  the listening addr of one redis node in the reids cluster
	 * @return {redis_node&}
	 */
	redis_node& set_addr(const char* addr);

	/**
	 * õǰ
	 * set the current node's type
	 * @param type {const char*}
	 * @return {redis_node&}
	 */
	redis_node& set_type(const char* type);

	/**
	 * õǰǷΪǰӶ
	 * set if the current node is belonging to the current connection
	 * @param yesno {bool}
	 * @return {redis_node&}
	 */
	redis_node& set_myself(bool yesno);

	/**
	 * Ϊӽʱõǰ
	 * setting current slave node's master node
	 * @param master {const redis_node*} 
	 *  the redis master node of the current slave in cluster
	 * @return {redis_node&}
	 */
	redis_node& set_master(const redis_node* master);

	/**
	 * õǰֽ׶
	 * set the current node being in handshaking status
	 * @param yesno {bool}
	 * @return {redis_node&}
	 */
	redis_node& set_handshaking(bool yesno);

	/**
	 * õǰ㴦״̬
	 * set the node been connected in the cluster
	 * @param yesno {bool}
	 * @return {redis_node&}
	 */
	redis_node& set_connected(bool yesno);

	/**
	 * Ϊӽʱõǰʶ
	 * setting current node's master node when the node is slave node
	 * @param id {const char*} Ψһʶ
	 *  the unique ID of the master node
	 * @return {redis_node&}
	 */
	redis_node& set_master_id(const char* id);

	/**
	 * Ϊʱһӽ
	 * add one slave node to the current node if it's one master node
	 * @return {bool} ǷɹӽѾڵǰʱ򷵻 false
	 *  false will be returned when the slave to be added is already
	 *  existing in the current master node
	 */
	bool add_slave(redis_node* slave);

	/**
	 * ΪʱݽΨһʶɾһӽ
	 * when the current node is a master node, this function will
	 * remove one slave node by the unique ID
	 * @param id {const char*} redis Ψһʶ
	 *  the unique ID of the redis node
	 * @return {const redis_node*} رɾĴӽ㣬򷵻 NULL
	 *  the slave node according to the ID will be returned, and if
	 *  not exists NULL will be returned
	 */
	redis_node* remove_slave(const char* id);

	/**
	 * Ϊʱձдӽ
	 * clear all the slave nodes in the current master node
	 * @param free_all {bool} ǷҪͬʱͷЩӽ
	 *  if freeing the all slave nodes memory meanwhile
	 */
	void clear_slaves(bool free_all = false);

	/**
	 * Ϊʱӹϣ۷Χ
	 * add hash-slots range to the master node
	 * @param min {size_t} ϣ۷Χʼֵ
	 *  the begin hash-slot of the slots range
	 * @param max {size_t} ϣ۷ΧĽֵ
	 *  the end hash-slot of the slots range
	 */
	void add_slot_range(size_t min, size_t max);

	/**
	 * ΪʱĹϣ۷ΧΪӽʱӦ
	 * Ĺϣ۷Χ
	 * @return {const std::vector<std::pair<size_t, size_t> >&}
	 */
	const std::vector<std::pair<size_t, size_t> >& get_slots() const;

	/**
	 * õǰ
	 * get the node's type
	 * @return {const char*}
	 */
	const char* get_type() const
	{
		return type_.c_str();
	}

	/**
	 * жϵǰǷΪǰӶ
	 * check if the node belongs to the current connection
	 * @return {bool}
	 */
	bool is_myself() const
	{
		return myself_;
	}

	/**
	 * жϵǰǷֽ׶
	 * check if the node is in handshaking status
	 * @return {bool}
	 */
	bool is_handshaking() const
	{
		return handshaking_;
	}

	/**
	 * жϵǰǷѾ״̬
	 * check if the node is connected in the cluster
	 * @return {bool}
	 */
	bool is_connected() const
	{
		return connected_;
	}

	/**
	 * Ϊӽʱøôӽ
	 * get the current slave's master node
	 * @return {const redis_node*}
	 */
	const redis_node* get_master() const
	{
		return master_;
	}

	/**
	 * ΪӽʱøôӽӦ ID ʶ
	 * when the current node is slave, getting its master's ID
	 * @return {const char*}
	 */
	const char* get_master_id() const
	{
		return master_id_.c_str();
	}

	/**
	 * Ϊʱøдӽ
	 * getting all the slaves of the master
	 * @return {const std::vector<redis_node*>*}
	 */
	const std::vector<redis_node*>* get_slaves() const
	{
		return &slaves_;
	}

	/**
	 * жϵǰǷΪȺеһ
	 * check if the current node is a master in the redis cluster
	 * @return {bool}
	 */
	bool is_master() const
	{
		return master_ == this;
	}

	/**
	 * õǰ ID ʶ
	 * get the unique ID of the current node, set in constructor
	 * @return {const char*}
	 */
	const char* get_id() const
	{
		return id_.c_str();
	}

	/**
	 * õǰļַ
	 * get the listening addr of the current node, set in constructor
	 * @reutrn {const char*}
	 */
	const char* get_addr() const
	{
		return addr_.c_str();
	}

	/**
	 * result of CLUSTER NODES for redis.4.x.x:
	 * d52ea3cb4cdf7294ac1fb61c696ae6483377bcfc 127.0.0.1:16385@116385 master - 0 1428410625374 73 connected 5461-10922
	 * @return return 127.0.0.1:16385@116385 for redis.4.x.x
	 */
	const char* get_addr_info() const
	{
		return addr_info_.c_str();
	}

private:
	string id_;
	string addr_;
	string addr_info_;
	string type_;
	bool myself_;
	bool handshaking_;
	bool connected_;
	const redis_node* master_;
	string master_id_;
	std::vector<redis_node*> slaves_;
	std::vector<std::pair<size_t, size_t> > slots_;
};

} // namespace acl

#endif // !defined(ACL_CLIENT_ONLY) && !defined(ACL_REDIS_DISABLE)
