var _ = require('../src/index.js');
var expect = require('chai').expect;
const path = require('path');
const fs = require('fs');
const EventEmitter = require('events');
const cmlFileContent = fs.readFileSync(path.resolve(__dirname, './testlib/index.cml'), {
  encoding: 'utf-8'
});
const interfaceFileContent = fs.readFileSync(path.resolve(__dirname, './testlib/index.interface'), {
  encoding: 'utf-8'
});

let cmlplatforms = ['wx', 'weex', 'baidu', 'alipay', 'web'];
let miniplatform = [{
  key: 'wx',
  name: 'wxml'
},
{
  key: 'alipay',
  name: 'axml'
},
{
  key: 'baidu',
  name: 'swan'
}
]

describe('index.js', function () {
  it('is Object', function () {
    let result = _.is({}, 'Object')
    expect(result).to.be.equal(true);
  })
  it('is Array', function () {
    let result = _.is([], 'Array')
    expect(result).to.be.equal(true);
  })
  it('is String', function () {
    let result = _.is('', 'String')
    expect(result).to.be.equal(true);
  })
  it('is Cli', function () {
    let originCml = global.cml;
    delete global.cml;
    let result = _.isCli();
    expect(result).to.be.equal(false);
    global.cml = {};
    _.setCli(true);
    result = _.isCli();
    expect(result).to.be.equal(true);
    global.cml = originCml;
  })

  it('map', function () {
    var obj = {
      a: 1,
      b: {
        c: 2
      }
    };
    _.map(obj, function (key, value) {
      switch (key) {
        case 'a':
          expect(value).to.equal(1);
          break;
        case 'b':
          expect(value).to.deep.equal({
            c: 2
          });
          break;
        default:
          expect(true).to.equal(false);
      }
    });
  });
  it('merge', function () {
    var obj = {
      a: 1,
      b: {
        c: 2
      }
    };
    var ret = {
      b: {
        d: 3
      },
      e: 'abc'
    };
    _.map(obj, ret, true);
    expect(ret).to.deep.equal({
      a: 1,
      b: {
        c: 2
      },
      e: 'abc'
    });
  });

  it('setBuiltinNpmName', function () {
    let result = _.setBuiltinNpmName('cml-ui')
    expect(result).to.be.equal('cml-ui');
  })
  it('isDir', function () {
    let result = _.isDir(__dirname)
    expect(result).to.be.equal(true);
  })

  it('escapeShellArg', function () {
    let result = _.escapeShellArg('name')
    expect(result).to.be.equal('"name"');
  })

  it('isWin', function () {
    let result = _.isWin()
    expect(result).to.be.equal(process.platform.indexOf('win') === 0);
  })

  it('isFile true', function () {
    let result = _.isFile(path.join(__dirname, "index.test.js"))
    expect(result).to.be.equal(true);
  })

  it('isFile false', function () {
    let result = _.isFile(path.join(__dirname))
    expect(result).to.be.equal(false);
  })

  it('isDirectory false', function () {
    let result = _.isDirectory(path.join(__dirname, "index.test.js"))
    expect(result).to.be.equal(false);
  })

  it('isDirectory true', function () {
    let result = _.isDirectory(path.join(__dirname))
    expect(result).to.be.equal(true);
  })


  cmlplatforms.forEach(cmlType => {
    it(`getJsonFileContent cml文件 ${cmlType}`, function () {
      global.cml = {};
      _.setCli(false);
      let result = _.getJsonFileContent(path.join(__dirname, `./testlib/index.cml`), cmlType);
      expect(result).to.have.property('usingComponents');
      expect(result).to.have.property(`attr${cmlType}`);
    })
  })


  miniplatform.forEach(item => {
    it(`getJsonFileContent ${item.key}  ${item.name}`, function () {
      global.cml = {};
      _.setCli(false);
      let result = _.getJsonFileContent(path.join(__dirname, `./testlib/${item.key}/${item.key}.${item.name}`), item.key);
      expect(result).to.have.property('name');
    })
  })

  it('splitParts', function () {
    let result = _.splitParts({
      content: cmlFileContent
    });
    expect(result.template.length).to.equal(1);
    expect(result.script.length).to.equal(2);
    expect(result.style.length).to.equal(1);
  })

  cmlplatforms.forEach(cmlType => {
    it(`getScriptPart ${cmlType}`, function () {
      let result = _.getScriptPart({
        content: interfaceFileContent,
        cmlType
      }).content;
      result = result.indexOf(cmlType);
      expect(result).to.not.equal(-1);
    })
  })

  cmlplatforms.forEach(cmlType => {
    it(`getScriptContent ${cmlType}`, function () {
      let result = _.getScriptContent({
        content: interfaceFileContent,
        cmlType
      });
      result = result.indexOf(cmlType);
      expect(result).to.not.equal(-1);
    })
  })

  cmlplatforms.forEach(cmlType => {
    it(`deleteScript ${cmlType}`, function () {
      let result = _.deleteScript({
        content: interfaceFileContent,
        cmlType
      });
      result = result.indexOf(cmlType);
      expect(result).to.equal(-1);
    })
  })

  it(`getNpmComponents not cli`, function () {
    global.cml = {};
    _.setCli(false);
    let result = _.getNpmComponents();
    expect(_.is(result, 'Array')).to.equal(true);

  })

  it(`getNpmComponents cli`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })
    // let result = _.getNpmComponents('wx', cml.projectRoot);
    // expect(result.length).to.equal(1);
  })

  it(`getBuildinComponents`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');

    // let result = _.getBuildinComponents('wx', cml.projectRoot);
    // expect(result.components.length).to.equal(1);
    // expect(Object.keys(result.compileTagMap).length).to.equal(1);
  })

  it(`getTargetInsertComponents`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })
    // let result = _.getTargetInsertComponents(__dirname, 'wx', cml.projectRoot);
    // expect(result.length).to.equal(2);
  })

  it(`isBuildIn  isBuildInProject: true`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      isBuildInProject: true
    })

    let result = _.isBuildIn();
    expect(result).to.equal(true);
  })

  it(`isBuildIn   buildinpath`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      isBuildInProject: false
    })
    let result = _.isBuildIn('/Users/didi/work/chameleon-open/chameleon/packages/chameleon-tool-utils/test/testlib/demo-project/node_modules/cml-ui/components/picker/picker.cml', 'wx', cml.projectRoot);
    expect(result).to.equal(false);
  })

  it(`addNpmComponents`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })
    let obj = {};
    // _.addNpmComponents(obj, __dirname, 'wx', cml.projectRoot);


    // expect(Object.keys(obj.usingComponents).length).to.equal(2);
  })

  it(`getOnePackageComponents`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })
    // let result = _.getOnePackageComponents('cml-ui', '/Users/didi/Documents/cml/chameleon-cli-utils/test/testlib/demo-project/node_modules/cml-ui/package.json', 'wx', cml.projectRoot);
    // expect(result.length).to.equal(1);
  })

  it(`handleComponentUrl`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })

    miniplatform.forEach(item => {
      let params = [
        cml.projectRoot,
        path.join(cml.projectRoot, 'index.cml'),
        `../${item.key}/${item.key}`,
        item.key
      ]

      let result = _.handleComponentUrl(...params);
      expect(result.filePath).to.equal(path.join(__dirname, `./testlib/${item.key}/${item.key}.${item.name}`));
    })
  })

  it('lintHandleComponentUrl', function() {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    global.cml.event = new EventEmitter();

    cml.config.merge({
      cmlComponents: ['cml-ui']
    })

    miniplatform.forEach(item => {
      let params = [
        cml.projectRoot,
        path.join(cml.projectRoot, 'index.cml'),
        `../${item.key}/${item.key}`
      ]
      let result = _.lintHandleComponentUrl(...params);
      expect(result.filePath).to.equal(path.join(__dirname, `./testlib/${item.key}/${item.key}.${item.name}`));
      expect(result.isCml).to.equal(undefined);
    })
  })


  it('lintHandleComponentUrl isCml true', function() {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    global.cml.event = new EventEmitter();

    cml.config.merge({
      cmlComponents: ['cml-ui']
    })
    var cmlFilePath = path.join(__dirname, 'testlib/demo-project/index.cml');
    var comrefPath = './comb/comb'


    let result = _.lintHandleComponentUrl(cml.projectRoot, cmlFilePath, comrefPath);
    expect(result.filePath).to.equal(path.join(__dirname, `testlib/demo-project/comb/comb.cml`));
    expect(result.isCml).to.equal(true);
  })

  it('handleComponentUrl isCml true wxml', function() {
    global.cml = {};
    _.setCli(true);
    cml.event = new EventEmitter();
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
    })
    var cmlFilePath = path.join(__dirname, 'testlib/demo-project/src/pages/pagea/pagea.cml');
    var comrefPath = 'vant-weapp/test'

    let result = _.handleComponentUrl(cml.projectRoot, cmlFilePath, comrefPath, 'wx');
    expect(result.refUrl).to.equal('./../../npm/vant-weapp/test');
  })

  it('findInterfaceFile', function() {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })

    var cmlFilePath = path.join(__dirname, 'testlib/demo-project/index.cml');
    var comrefPath = './coma/coma'
    let result = _.findInterfaceFile(cml.projectRoot, cmlFilePath, comrefPath);
    expect(result.filePath).to.equal(path.join(__dirname, `testlib/demo-project/coma/coma.interface`));
  })


  it(`findComponent true`, function () {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })

    miniplatform.forEach(item => {
      let result = _.findComponent(path.join(__dirname, `./testlib/${item.key}/${item.key}`), item.key);
      expect(result).to.equal(path.join(__dirname, `./testlib/${item.key}/${item.key}.${item.name}`));
    })

  })

  it(`findComponent false`, function () {
    global.cml = {};
    _.setCli(true);
    cml.event = new EventEmitter();
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    cml.config.merge({
      cmlComponents: ['cml-ui']
    })

    let result = _.findComponent(__dirname, 'wx');
    expect(result).to.equal(false);

  })

  it('getDevServerPath', function() {
    process.env.HOME = __dirname;
    let devpath = cml.utils.getDevServerPath();
    expect(devpath).to.be.equal(path.join(__dirname, '.chameleon/www'))
  })

  it('handleRelativePath same level', function() {
    let sourcePath = 'C:/src/compoents/a/a.cml';
    let targetPath = 'C:/src/compoents/a/b.cml';
    let relativePath = _.handleRelativePath(sourcePath, targetPath);
    expect(relativePath).to.be.equal('./b.cml')
  })

  it('handleRelativePath diff level1', function() {
    let sourcePath = 'C:/src/compoents/a/b/a.cml';
    let targetPath = 'C:/src/compoents/a/b.cml';
    let relativePath = _.handleRelativePath(sourcePath, targetPath);
    expect(relativePath).to.be.equal('./../b.cml')
  })

  it('handleRelativePath diff level1', function() {
    let sourcePath = 'C:/src/compoents/a/c.cml';
    let targetPath = 'C:/src/compoents/a/b/a.cml';
    let relativePath = _.handleRelativePath(sourcePath, targetPath);
    expect(relativePath).to.be.equal('./b/a.cml')
  })


  it('getCmlFileType', function() {
    global.cml = {};
    _.setCli(true);
    cml.config = require('./testlib/cli/config.js');
    cml.utils = require('../src/index.js');
    cml.projectRoot = path.join(__dirname, 'testlib/demo-project');
    let pagea = path.join(__dirname, './testlib/demo-project/src/pages/pagea/pagea.cml')
    let coma = path.join(__dirname, './testlib/demo-project/src/components/coma/coma.cml')
    let app = path.join(__dirname, './testlib/demo-project/src/app/app.cml')
    let result1 = cml.utils.getCmlFileType(pagea, cml.projectRoot, 'wx')
    let result2 = cml.utils.getCmlFileType(coma, cml.projectRoot, 'wx')
    let result3 = cml.utils.getCmlFileType(app, cml.projectRoot, 'wx')
    expect(result1).to.be.equal('page')
    expect(result2).to.be.equal('component')
    expect(result3).to.be.equal('app')

  })

  it('npmRefPathToRelative', function() {
    let context = path.join(__dirname, './testlib/demo-project');
    let npmRef = '/npm/cml-ui/button/button';
    let notNpmRef = './npm/cml-ui';

    let file1 = path.join(context, 'src/pages/pagea/pagea.cml') // ./../../npm
    let file2 = path.join(context, 'src/pages/pagea.cml') // ./../npm
    let file3 = path.join(context, 'src/pages.cml') // ./npm

    let result1 = _.npmRefPathToRelative(npmRef, file1, context);
    let result2 = _.npmRefPathToRelative(npmRef, file2, context);
    let result3 = _.npmRefPathToRelative(npmRef, file3, context);
    let result4 = _.npmRefPathToRelative(notNpmRef, file3, context);
    expect(result1).to.be.equal('./../../npm/cml-ui/button/button');
    expect(result2).to.be.equal('./../npm/cml-ui/button/button');
    expect(result3).to.be.equal('./npm/cml-ui/button/button');
    expect(result4).to.be.equal(notNpmRef);

  })

  it('handleSpecialChar', function() {
    let result = _.handleSpecialChar('npm/@didi/name');
    expect(result).to.be.equal('npm/_didi/name')
  })

  it(`getJsonFileContent cli app`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {}
      }
    }
    _.setCli(true);
    let result = _.getJsonFileContent(path.join(__dirname, './testlib/demo-project/src/app/app.cml'), 'wx');
    expect(typeof result).to.equal('object');
  })

  it(`getJsonFileContent cli page`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {}
      }
    }
    _.setCli(true);
    let result = _.getJsonFileContent(path.join(__dirname, './testlib/demo-project/src/pages/pagea/pagea.cml'), 'wx');
    expect(typeof result).to.equal('object');
  })

  it(`getJsonFileContent cli component`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {}
      }
    }
    _.setCli(true);
    let result = _.getJsonFileContent(path.join(__dirname, './testlib/demo-project/src/components/coma/coma.cml'), 'wx');
    expect(typeof result).to.equal('object');
  })

  it(`mkdir`, function () {
    let pathdir = path.join(__dirname, 'temp')
    _.mkdir(pathdir)
    expect(_.isDir(pathdir)).to.be.equal(true)
  })

  it(`copyNpm and subProject`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {
          subProject: [{npmName: 'cml-pages'}],
          copyNpm: {
            wx: ['copy-npm']
          }
        }
      }
    }
    _.setCli(true);
    let result = _.getJsonFileContent(path.join(__dirname, './testlib/demo-project/src/app/app.cml'), 'wx');
    expect(typeof result).to.equal('object');

  })

  it(`getNpmComponents`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {
          cmlComponents: ['npm-components']
        }
      }
    }
    _.setCli(true);
    cml.utils = _;
    let result = _.getNpmComponents('wx', cml.projectRoot);
    expect(typeof result).to.equal('object');

  })

  it(`getBuildinComponents`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {
          cmlComponents: ['npm-components']
        }
      }
    }
    _.setCli(true);
    cml.utils = _;
    let result = _.getBuildinComponents('wx', cml.projectRoot);
    expect(typeof result).to.equal('object');

  })

  it(`getTargetInsertComponents`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {
          cmlComponents: ['npm-components']
        }
      }
    }
    _.setCli(true);
    cml.utils = _;
    let result = _.getTargetInsertComponents(path.join(path.join(__dirname, './testlib/demo-project/src/pages/pagea/pagea.cml')), 'wx', cml.projectRoot);
    expect(typeof result).to.equal('object');

  })

  it(`addNpmComponents`, function () {
    global.cml = {};
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {
          cmlComponents: ['npm-components']
        }
      }
    }
    cml.event = new EventEmitter();
    _.setCli(true);
    cml.utils = _;
    var obj = {};
    _.addNpmComponents(obj, path.join(path.join(__dirname, './testlib/demo-project/src/pages/pagea/pagea.cml')), 'wx', cml.projectRoot);
    console.log(obj)
    expect(obj).to.has.property('usingComponents');

  })

  it(`getSubProjectRouter`, function () {
    global.cml = {};
    _.setCli(true);
    cml.projectRoot = path.join(__dirname, './testlib/demo-project');
    cml.config = {
      get() {
        return {
          subProject: ['cml-subproject']
        }
      }
    }

    let result = _.getSubProjectRouter();

    expect(Object.keys(result).length).to.be.equal(1);

  })

  it(`deleteExt`, function () {
    global.cml = {};
    _.setCli(true);
    global.cml.event = new EventEmitter();
    let path1 = '/src/pages/name.web.cml';
    let path2 = '/src/pages/name.cml';
    let path3 = 'name.cml';
    let result1 = _.deleteExt(path1);
    let result2 = _.deleteExt(path2);
    let result3 = _.deleteExt(path3);
    expect(result1).to.be.equal('/src/pages/name');
    expect(result2).to.be.equal('/src/pages/name');
    expect(result3).to.be.equal('name');

  })

  it(`getExportEntry`, function () {
    global.cml = {};
    _.setCli(true);
    global.cml.event = new EventEmitter();
    global.cml.utils = _;
    global.projectRoot = path.join(__dirname, 'testlib/demo-project');
    let result = _.getExportEntry('web', global.projectRoot, [
      'src/components',
      'src/notfinr.cml',
      'src/components/com2/com2.interface'
    ]);
    console.log(result)
    expect(result.length).to.be.equal(2);
  })

  it(`creatMD5`, function () {
    let source = '12345678';
    let result = _.createMd5(source);
    expect(result).to.be.equal('25d55ad283aa400af464c76d713c07ad');
  })

  it(`delQueryPath`, function () {
    let filePath = '/user/cml/name.png?__inline';
    let result = _.delQueryPath(filePath);
    expect(result).to.be.equal('/user/cml/name.png');
  })


  it(`splitFileName`, function () {
    let filePath = '/user/cml/name.web.cml?__inline';
    let result = _.splitFileName(filePath);
    expect(!!~result.indexOf('name')).to.be.equal(true);
    expect(!!~result.indexOf('web')).to.be.equal(true);
    expect(!!~result.indexOf('cml')).to.be.equal(true);
  })

  it(`isInline`, function () {
    let filePath = '/user/cml/name.web.cml?__inline';
    let result = _.isInline(filePath);
    let filePath2 = '/user/cml/name.web.cml';
    let result2 = _.isInline(filePath2);
    expect(result).to.be.equal(true);
    expect(result2).to.be.equal(false);

  })

  it(`addHashName has ext`, function () {
    let filePath = '/user/cml/name.web.cml';
    let result = _.addHashName(filePath, '3123123123123sd');
    expect(result).to.be.equal('/user/cml/name.web_3123123123123sd.cml');

  })

  it(`addHashName no ext`, function () {
    let filePath = '/user/cml/name';
    let result = _.addHashName(filePath, '3123123123123sd');
    expect(result).to.be.equal('/user/cml/name_3123123123123sd');

  })
  it(`addHashName multi .`, function () {
    let filePath = '/user/cml/name.web.cml';
    let result = _.addHashName(filePath, '3123123123123sd');
    expect(result).to.be.equal('/user/cml/name.web_3123123123123sd.cml');

  })

  it(`resolveSync relativePath`, function () {
    let filePath = path.join(__dirname, 'index.test.js');
    let relativePath = './testlib/index.cml';
    let result = _.resolveSync(filePath, relativePath);

    expect(result).to.be.equal(path.join(__dirname, './testlib/index.cml'));

  })

  it(`resolveSync npmPath`, function () {
    let filePath = path.join(__dirname, 'index.test.js');
    let relativePath = 'glob';
    let result = _.resolveSync(filePath, relativePath);
    console.log(result)
    expect(!!~result.indexOf('glob')).to.be.equal(true);

  })

  it(`resolveInterfaceRequire npmPath`, function () {
    let oldFilePath = path.join(__dirname, 'index.test.js');
    let relativePath = 'glob';
    let result = _.resolveInterfaceRequire(oldFilePath, '', relativePath);
    expect(result).to.be.equal(relativePath);
  })

  it(`resolveInterfaceRequire relativePath1`, function () {
    let oldFilePath = path.join(__dirname, 'index.test.js');
    let newFilePath = path.join(__dirname, '../index.test.js');
    let relativePath = './glob';
    let result = _.resolveInterfaceRequire(oldFilePath, newFilePath, relativePath);
    expect(result).to.be.equal('./test/glob');
  })

  it(`resolveInterfaceRequire relativePath2`, function () {
    let oldFilePath = path.join(__dirname, 'index.test.js');
    let newFilePath = path.join(__dirname, 'temp/index.test.js');
    let relativePath = './glob';
    let result = _.resolveInterfaceRequire(oldFilePath, newFilePath, relativePath);
    expect(result).to.be.equal('../glob');
  })


})
