import packageJson from '../package.json'
import HypetriggerManager from '../src/lib/HypetriggerManager'
import OBSManager from '../src/lib/OBSManager'
import BadgeTrigger from '../src/triggers/badge'
import KillNotifTrigger from '../src/triggers/kill-notif'
import KillTotalTrigger from '../src/triggers/kill-total'
import Trigger from '../src/triggers/trigger'
import WarzoneVictoryTrigger from '../src/triggers/warzone-victory'
import YellowTextTrigger from '../src/triggers/yellow-text'

// Globals
const manager = new HypetriggerManager()
const obs = new OBSManager()
const triggers: Trigger[] = [
  new KillTotalTrigger({ title: 'Kill Total', manager }),
  new YellowTextTrigger({ title: 'Yellow Text', manager }),
  new BadgeTrigger({ title: 'Badges', manager }),
  new KillNotifTrigger({ title: 'Kill Notification', manager }),
  new WarzoneVictoryTrigger({ title: 'Warzone Victory', manager }),
]

async function main() {
  console.log(`Clip It v${packageJson.version}`)
  console.log(`===============================`)

  // createWindow()
  await initOBS()
  await startTesseract()

  console.log(`Game is hardcoded to COD: Warzone`)
  console.log(`Make sure OBS is recording the correct display/game capture`)
  console.log(`Clips will appear in your OBS output directory: ${obs.settings.ClipDirectory}`)
  console.log(`===============================`)
  console.log(`Ready to clip automatically...`)
}

async function runTrigger(trigger: Trigger) {
  try {
    if (obs.screenshot) await trigger.run(obs.screenshot)
  } catch (err) {
    console.error(err)
  } finally {
    if (trigger.runForever) runTrigger(trigger)
  }
}

async function initOBS() {
  await obs.init()
  obs.getScreenshot()
}

async function startTesseract() {
  manager.onEvent((event) => {
    if(!['kill', 'warzone-victory', 'badge'].includes(event.type)) return
  
    const recentClip = manager.eventLog.find(event =>
      event.type === 'clip'
      && new Date().getTime() - event.timestamp.getTime() < obs.settings.ClipDuration! * 1000
    )
  
    if (recentClip) {
      //console.log('Skipping due to recent clip')
      return
    }

    manager.logEvent({
      type: 'clip',
      msg: 'CLIP IT!',
      timestamp: new Date(),
    })

    setTimeout(obsClip, obs.settings.ClipDelay ? obs.settings.ClipDelay * 1000 : 0)
  })

  await manager.initTesseract()
  triggers.forEach(runTrigger)
}

function obsClip() {
  if (!obs) {
    console.error('No obs instance')
    return
  }

  obs.ws.send('SaveReplayBuffer')
    .then(() => console.log(`Saved clip to ${obs.settings.ClipDirectory}`))
    .catch(console.error)
}

main()