import { watch } from 'chokidar'
import { app, BrowserWindow, ipcMain, shell } from 'electron'
import { IpcMainInvokeEvent } from 'electron/main'
import type { Stats } from 'fs'
import { mkdirSync, readdirSync, readFileSync, unlinkSync } from 'fs'
import { OBSStats } from 'obs-websocket-js'
import { existsSync } from 'original-fs'
import { extname, join, resolve } from 'path'
import packageJson from '../package.json'
import GameAutoDetect from '../src/lib/GameAutoDetect'
import HypetriggerManager from '../src/lib/HypetriggerManager'
import OBSManager, { OBSError } from '../src/lib/OBSManager'
import triggerFromJson from '../src/lib/triggerFromJson'
import Trigger from '../src/triggers/trigger'
import { ConfigData } from '../src/types'

// Globals
const manager = new HypetriggerManager()
const obs = new OBSManager()
const gameAutoDetect = new GameAutoDetect()
let mainWindow: BrowserWindow
let triggers: Trigger[] = []
let selectedConfig: ConfigData = null
let obsError: OBSError | null = null
let configs: ConfigData[] = []

// Hard-coded startup
app.on('ready', async () => {
  try {
    console.log(`Clip It v${packageJson.version}`)
    console.log(`===============================`)

    initDebug()
    createWindow()
    await initOBS()
    await initTesseract()

    console.log(`Make sure OBS is recording the correct display/game capture`)
    console.log(`Clips will appear in your OBS output directory: ${obs.settings.ClipDirectory}`)
    console.log(`===============================`)
    console.log(`Ready to clip automatically...`)
  } catch (e) {
    console.error(e)
  }
})

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1920,
    height: 1080,
    frame: false,
    show: false,
    webPreferences: {
      nodeIntegration: false, // is default value after Electron v5
      contextIsolation: true, // protect against prototype pollution
      enableRemoteModule: false, // turn off remote
      preload: join(__dirname, 'preload.js') // use a preload script
    },
    icon: `${__dirname}/../public/clipit-logo-vertical-light.png`,
  })
  const startURL = `file://${join(__dirname, '../public/index.html')}`
  mainWindow.setMenuBarVisibility(false)
  mainWindow.loadURL(startURL)
  mainWindow.once('ready-to-show', () => mainWindow.show())
  mainWindow.webContents.setWindowOpenHandler(({ url }) => {
    console.log(`Opening new window in a browser: ${url}`)
    shell.openExternal(url)
    return { action: 'deny' }
  })
  // mainWindow.on('closed', () => { mainWindow = null })
}

async function initOBS() {
  obs.onError = (err: OBSError) => {
    //err.msg.split('\n').forEach(console.error)
    obsError = err
    mainWindow.webContents.send('obsError', err)
  }

  try {
    await obs.init()
    await obs.getScreenshot((stats: OBSStats) => {
      (stats as any).pollingInterval = obs.screenshotPollingInterval
      mainWindow.webContents.send('obsStats', stats)
    })
  } catch (e) {
    throw e // rethrow
  }
}

function initDebug() {
  const DEBUG_DIR = 'debug'
  if (existsSync(DEBUG_DIR)) return
  mkdirSync(DEBUG_DIR)
  console.log(`Created directory for debug output at ${DEBUG_DIR}`)
}

function initChokidar() {
  if (obsError) {
    obs.onError(obsError) // Re-send the error message if it was too soon the first time
    return
  }

  const path = obs.settings.ClipDirectory!
  console.log(`Watching clip directory for changes: ${path}`)
  const watcher = watch(path)
  watcher
    .on('add', (path: any, stats: Stats) => { console.log(`Clip added: ${path}`); mainWindow.webContents.send('clipAdd', path, stats) })
    // .on('change', (path: any, stats: Stats) => { console.log(`Clip changed: ${path}`); mainWindow.webContents.send('clipChange', path, stats) })
    .on('unlink', (path: any, stats: Stats) => { console.log(`Clip removed: ${path}`); mainWindow.webContents.send('clipRemove', path, stats) })
}

async function initTesseract(event?: IpcMainInvokeEvent) {
  manager.eventListeners = []
  manager.onEvent((event) => {
    if (!selectedConfig) return
    const presetEvent = selectedConfig.triggers.find(trigger => trigger.id === event.type)
    if (!presetEvent || !presetEvent.enabled) return
  
    // Don't let clips overlay (give the replay buffer time to fill completely)
    const recentClip = manager.eventLog.find(event =>
      event.type === 'clip'
      && new Date().getTime() - event.timestamp.getTime() < obs.settings.ClipDuration! * 1000
    )
    if (recentClip) return

    manager.logEvent({
      type: 'clip',
      msg: 'CLIP IT!',
      timestamp: new Date(),
    })

    setTimeout(obsClip, obs.settings.ClipDelay ? obs.settings.ClipDelay * 1000 : 0)
  })

  await manager.initTesseract()

  const prophylaxis = async () => {
    console.log('Preemptively restarting Tesserract to flush RAM')
    await manager.restartTesseract()
  }
  setInterval(prophylaxis, 1000 * 60 * 15) // every 15 minutes

  triggers.forEach(runTrigger)
}

async function runTrigger(trigger: Trigger) {
  if (!trigger.enabled) return

  try {
    if (obs.screenshot && manager.tesseractInitialized) {
      trigger.lastScreenshotNumber = obs.screenshotNumber
      await trigger.run(obs.screenshot)
    }
  } catch (err) {
    console.error(err)
    // console.warn('============================================')
    // console.warn('THIS COULD BE A MEMORY LEAK')
    // console.warn('IF IT HAPPENS AGAIN, RESTART THE PROGRAM')
    // console.warn('============================================')
  } finally {
    if (trigger.runForever) {
      let delay = 100
      if (obs.screenshotNumber === trigger.lastScreenshotNumber)
        delay = obs.nextScreenshotTime - Date.now()
      if (delay < 100) delay = 100 // minimum delay for a bit of breathing room
      //console.log(`${trigger.title} delay=${delay}ms`)
      setTimeout(() => runTrigger(trigger), delay)
    }
  }
}

function obsClip() {
  if (!obs) {
    console.error('No obs instance')
    return
  }

  obs.ws.send('SaveReplayBuffer')
    .then(() => console.log(`Saved clip to ${obs.settings.ClipDirectory}`))
    .catch(console.error)
}

function deleteClip(event: IpcMainInvokeEvent, path: string) {
  try {
    unlinkSync(path)
    console.log(`Deleted ${path}`)
  } catch(err) {
    console.log(`Error deleting ${path}`)
    console.error(err)
  }
}

/**
 * Note a minor distinction between "raw" triggers (the actual triggers array in main.ts)
 * versus the "preset" triggers which are shown to the user in the UI.
 * These may overlap but are not guaranteed to be the same, despite the similar naming!
 */
function setTriggerEnabled(event: IpcMainInvokeEvent, triggerEvent: string, enabled: boolean) {
  if (!selectedConfig) return
  const triggerConfig = selectedConfig.triggers.find(trigger => trigger.id === triggerEvent)
  const trigger = triggers.find(trigger => trigger.id === triggerEvent)
  if (!triggerConfig || !trigger) {
    console.warn(`Could not find specified trigger event ${triggerEvent}`)
    return
  }
  trigger.enabled = enabled
  triggerConfig.enabled = enabled
  if (trigger.enabled) runTrigger(trigger)
  console.log(`Set trigger ${triggerEvent} to ${enabled}`)
}

function restartTesseract(event: IpcMainInvokeEvent) {
  manager.restartTesseract()
}

function quit(event: IpcMainInvokeEvent) {
  console.log('Render thread requested quit')
  app.quit()
}

function close(event: IpcMainInvokeEvent) {
  mainWindow.close()
}

function minimize(event: IpcMainInvokeEvent) {
  mainWindow.minimize()
}

function maximize(event: IpcMainInvokeEvent) {
  if (!mainWindow.isMaximized())
    mainWindow.maximize()
  else
    mainWindow.unmaximize()
}

function getAllConfigs(event: IpcMainInvokeEvent) {
  const path = resolve(__dirname, '../public/clipit-configs/configs')
  configs = []
  readdirSync(path).forEach(file => {
    try {
      if (extname(file) !== '.json') return
      const config = JSON.parse(readFileSync(`${path}/${file}`, 'utf8'))
      configs.push(config)
      console.log(`Found config ${file}`)
    } catch (e) {
      console.error(`Failed to get config file ${path}/${file}`)
      console.error(e)
    }
  })

  // Start game auto detection
  gameAutoDetect.games = configs.map(config => config.exe).filter(exe => !!exe)
  gameAutoDetect.callback = (game: string) => mainWindow.webContents.send('gameRunning', game)
  gameAutoDetect.period = 1000
  gameAutoDetect.start()

  return configs
}

function setSelectedConfig(event: IpcMainInvokeEvent, config: ConfigData) {
  if (selectedConfig && selectedConfig.id === config.id) return
  selectedConfig = config
  triggers.forEach(trigger => trigger.enabled = false)
  triggers = selectedConfig.triggers.map(triggerJson => triggerFromJson(triggerJson, manager))
  console.log(`Selected config ${selectedConfig.title}`)
  if (manager.tesseractInitialized) triggers.forEach(runTrigger)
}

// IPC comms
ipcMain.handle('deleteClip', deleteClip)
ipcMain.handle('setTriggerEnabled', setTriggerEnabled)
ipcMain.handle('clip', obsClip)
ipcMain.handle('initChokidar', initChokidar)
ipcMain.handle('restartTesseract', restartTesseract)
ipcMain.handle('quit', quit)
ipcMain.handle('close', close)
ipcMain.handle('minimize', minimize)
ipcMain.handle('maximize', maximize)
ipcMain.handle('getAllConfigs', getAllConfigs)
ipcMain.handle('setSelectedConfig', setSelectedConfig)