import type { Stats } from 'fs'
import type { ClipData, TriggerData, ConfigData, VideoCaptureStats, ClipSection } from './types'

//////// TYPES
declare global {
  interface Window {
    electron: {
      ipcRenderer: {
        send: (channel: string, data?: any) => any
        on: (channel: string, func: (event: any, ...args: any[]) => any) => any
        once: (channel: string, func: (event: any, ...args: any[]) => any) => any
        invoke: (channel: string, ...args: any[]) => Promise<any>
      }
    }
  }
}

//////// GLOBALS
const dom = {
  minimize: document.querySelector('.chrome-buttons .minimize') as HTMLButtonElement,
  maximize: document.querySelector('.chrome-buttons .maximize') as HTMLButtonElement,
  maximizeIcon: document.querySelector('.chrome-buttons .maximize img') as HTMLImageElement,
  close: document.querySelector('.chrome-buttons .close') as HTMLButtonElement,
  middleContent: document.querySelector('.middle-content') as HTMLDivElement,
  clips: document.getElementById('clips-wrapper') as HTMLDivElement,
  clipPreview: document.getElementById('clip-preview') as HTMLDivElement,
  clipPreviewVideo: document.querySelector('#clip-preview video') as HTMLVideoElement,
  clipPreviewSource: document.querySelector('#clip-preview video source') as HTMLSourceElement,
  obsStatusText: document.getElementById('obs-status-text') as HTMLDivElement,
  obsStatusInfo: document.getElementById('obs-status-info') as HTMLDivElement,
  obsCpuPercent: document.getElementById('obs-cpu-percent') as HTMLDivElement,
  obsFps: document.getElementById('obs-fps') as HTMLDivElement,
  obsPollingInterval: document.getElementById('obs-polling-interval') as HTMLDivElement,
  triggersList: document.getElementById('triggers-list') as HTMLDivElement,
  clipItBtn: document.getElementById('clip-it-btn') as HTMLButtonElement,
  overlay: document.getElementById('overlay') as HTMLDivElement,
  modalContent: document.querySelector('#modal .modal-content') as HTMLDivElement,
  modalTitle: document.querySelector('#modal .modal-title') as HTMLDivElement,
  quitBtn: document.getElementById('quit-btn') as HTMLButtonElement,
  fab: document.querySelector('.fab') as HTMLButtonElement,
  feedback: document.getElementById('feedback') as HTMLDivElement,
  playNew: document.getElementById('play-new') as HTMLAnchorElement,
  soundEffectDemo: document.getElementById('sound-effect-demo') as HTMLAnchorElement,
  gameSelect: document.getElementById('game-select') as HTMLSelectElement,
  gameWaitingOption: document.getElementById('game-waiting-option') as HTMLOptionElement,
  gameAutoDetect: document.getElementById('game-auto-detect') as HTMLInputElement,
}

const sections: ClipSection[] = [
  {
    date: new Date(),
    elem: document.getElementById('this-session-clips')! as HTMLDivElement,
  }
]
const clips: ClipData[] = []
const configs: ConfigData[] = []
const startupTime = new Date()
const playlist: ClipData[] = []
const soundEffects = {
  newClip: new Audio('sound/camera.wav'),
  deleteClip: new Audio('sound/recycle.wav'),
}
let activeClip: ClipData | null = null
let autoSelectGame = true

//////// METHODS
function unsetActiveClip() {
  if (!activeClip) return
  activeClip.elem.classList.remove('active')
  activeClip = null
  dom.clipPreviewVideo.pause()
  dom.clipPreviewSource.src = ''
}

function setActiveClip(clip: ClipData) {
  unsetActiveClip()
  clip.new = false
  clip.elem.classList.add('active')
  clip.elem.classList.remove('new')
  checkNewClips()
  activeClip = clip
  const wrapper = dom.clipPreview
  const video = dom.clipPreviewVideo
  const source = dom.clipPreviewSource
  wrapper.classList.add('visible')
  source.src = (clip.elem.querySelector('source') as HTMLSourceElement).src
  video.load()
  video.play()
}

function checkNewClips() {
  const newClips = clips.filter(clip => clip.new)
  if (newClips.length === 0)
    dom.playNew.classList.remove('visible')
}

function checkEmptyState() {
  if (clips.length === 0)
    dom.middleContent.classList.add('empty')
  else
    dom.middleContent.classList.remove('empty')
}

function handleClips() {
  function clipOnClick(clip: ClipData) {
    console.log(`Clicked clip: ${clip.path}`)
    if (activeClip === clip) {
      unsetActiveClip()
      dom.clipPreview.classList.remove('visible')
    } else
      setActiveClip(clip)
  }
  
  function deleteClip(clip: ClipData, event: MouseEvent) {
    console.log(`Delete clip: ${clip.path}`)
  
    if (activeClip === clip) {
      unsetActiveClip()
      dom.clipPreview.classList.remove('visible')
    }
    window.electron.ipcRenderer.invoke('deleteClip', clip.path)
  
    event.stopPropagation()
    event.preventDefault()
    return false
  }

  const getClipSection = (date: Date): ClipSection => {
    for (let i = 0; i <= sections.length; i++) {
      const section = sections[i]
      if (section.date.getDate() === date.getDate()
          && section.date.getMonth() === date.getMonth()
          && section.date.getFullYear() === date.getFullYear()) {
        return section
      }

      if (date < section.date || i === sections.length) {
        // Container
        const elem = document.createElement('div') as HTMLDivElement
        elem.classList.add('clip-section')
        dom.clips.insertBefore(elem, section.elem.nextSibling)
        //dom.clips.appendChild(elem)

        // Heading
        const h3 = document.createElement('h3') as HTMLDivElement
        h3.innerText = `${date.toDateString()}`
        elem.appendChild(h3)

        // Publish
        const newSection = { date, elem }
        sections.splice(i, 0, newSection)
        return newSection
      }
    }
    throw new Error(`Could not find or create a section for clip with date ${date}`)
  }

  const onClipAdd = (path: string, stat: Stats) => {
    console.log(`Clip added: ${path}`)

    // Ignore non-video files
    const split = path.split('.')
    const videoExtensions = ['flv', 'mp4', 'mov', 'mkv', 'ts', 'm3u8']
    if (split.length < 2 || !videoExtensions.includes(split[1].toLowerCase())) {
      console.warn(`Ignoring file with extension ${split[1]}`)
      return
    }

    // Construct ClipData
    const date = new Date(stat.birthtimeMs)
    const elem = document.createElement('div')
    const isNew = date > startupTime
    const clipData: ClipData = { path, date, new: isNew, elem }
    const section = getClipSection(date)

    // Prepare DOM element
    elem.onclick = () => clipOnClick(clipData)
    elem.classList.add('clip')
    if (isNew) {
      elem.classList.add('new')
      dom.playNew.classList.add('visible')
      soundEffects.newClip.currentTime = 0
      soundEffects.newClip.volume = 0.5
      soundEffects.newClip.play()
    }
    elem.tabIndex = 0
    let innerHTML = `<video><source src="${path}"></source></video>`
    innerHTML    += `<img src="trash.svg" alt="Delete" class="trash" title="Delete clip" tabindex="0">`
    elem.innerHTML = innerHTML
    
    // Handle trash button click
    const img = elem.querySelector('.trash')! as HTMLImageElement
    img.onclick = event => {
      const sure = confirm(`Are you sure you want to delete ${path}?`)
      if (!sure) return
      deleteClip(clipData, event)
      soundEffects.deleteClip.currentTime = 0
      soundEffects.deleteClip.volume = 0.5
      soundEffects.deleteClip.play()
      event.preventDefault()
      event.stopPropagation()
      return false
    }
  
    // Handle video loading problems
    const video = elem.querySelector('video') as HTMLVideoElement
    const source = video.querySelector('source') as HTMLSourceElement
    (video as any).retries = 0
    const retryDelay = 500
    const MAX_RETRIES = 5
    const checkLoaded = () => {
      setTimeout(() => {
        if (video.networkState !== HTMLMediaElement.NETWORK_NO_SOURCE)
          return

        console.warn(`Could not load video; Trying again in ${retryDelay}ms`)
        if ((video as any).retries++ >= MAX_RETRIES) {
          console.error(`Cannot load ${path}; tried ${MAX_RETRIES} times`)
          return
        }

        const sourceBase: string = source.src.split('?')[0]
        source.src = `${sourceBase}?retry=${(video as any).retries}`
        video.load()
        checkLoaded()
      }, retryDelay)
    }
    checkLoaded()

    // Publish
    section.elem.appendChild(elem)
    clips.push(clipData)
    checkEmptyState()
  }

  const onClipRemove = (path: string, stat: Stats) => {
    console.log(`Clip removed: ${path}`)
    const clipData = clips.find(clip => clip.path === path)
    if (!clipData) return
    clipData.elem.parentNode!.removeChild(clipData.elem)
    clips.splice(clips.indexOf(clipData), 1)
    checkNewClips()
    checkEmptyState()
  }

  window.electron.ipcRenderer.on('clipAdd', onClipAdd)
  window.electron.ipcRenderer.on('clipRemove', onClipRemove)
}

function handleObsStats() {
  const onObsStats = (stats: VideoCaptureStats) => {
    if (!dom.obsStatusText.textContent?.includes('Connected')) {
      dom.obsStatusText.innerHTML = '✔️ Connected to OBS'
      dom.obsStatusInfo.classList.add('visible')
    }
    dom.obsCpuPercent.innerHTML = `${Math.round(stats['cpu-usage'])}%`
    dom.obsFps.innerHTML = `${Math.round(stats['fps'])}`
    dom.obsPollingInterval.innerHTML = `${Math.round(stats.pollingInterval)}ms`
  }

  window.electron.ipcRenderer.on('obsStats', onObsStats)
}

function handleTriggers(config: ConfigData | undefined) {
  if (!config) return

  const onTriggerClick = (trigger: TriggerData) => {
    console.log(`Clicked ${trigger.title}`)
    trigger.enabled = !trigger.enabled
    window.electron.ipcRenderer.invoke('setTriggerEnabled', trigger.id, trigger.enabled)
  }

  const toggleSwitch = (elem: HTMLElement) => {
    if (elem.classList.contains('off')) {
      elem.classList.remove('off')
      elem.classList.add('on')
    } else {
      elem.classList.remove('on')
      elem.classList.add('off')
    }
  }

  dom.triggersList.innerHTML = ''
  const section = dom.triggersList.parentNode as HTMLElement
  section.style.display = config.triggers.length === 0 ? 'none' : 'block'
  config.triggers.forEach(trigger => {
    const elem = document.createElement('div')
    elem.classList.add('trigger')
    if (!trigger.enabled) elem.classList.add('off')
    elem.tabIndex = 0
    let innerHTML = trigger.title
    innerHTML += `<div class="toggle ${trigger.enabled ? 'on' : 'off'}"></div>`
    elem.innerHTML = innerHTML
    elem.onclick = () => {
      toggleSwitch(elem.querySelector('.toggle')!)
      toggleSwitch(elem)
      onTriggerClick(trigger)
    }
    dom.triggersList.appendChild(elem)
  })
}

function getSelectedConfig(): ConfigData | undefined {
  if (dom.gameSelect.value === dom.gameWaitingOption.value)
    return { id: dom.gameWaitingOption.value, title: dom.gameWaitingOption.innerText, triggers: [] }

  return configs.find(config => config.id === dom.gameSelect.value)
}

function onGameSelect() {
  const selectedConfig = getSelectedConfig()
  if (selectedConfig) {
    handleTriggers(selectedConfig)
    window.electron.ipcRenderer.invoke('setSelectedConfig', selectedConfig)
  }
}

async function handleTriggerConfigs() {
  const configsReceived = await window.electron.ipcRenderer.invoke('getAllConfigs')
  configs.push(...configsReceived)

  configs.forEach(config => {
    const option = document.createElement('option')
    option.innerText = config.title
    option.value = config.id
    dom.gameSelect.appendChild(option)
  })

  onGameSelect()
  dom.gameSelect.onchange = () => {
    if (autoSelectGame) dom.gameAutoDetect.click()
    onGameSelect()
  }
}

function handleManualClip() {
  dom.clipItBtn.onclick = () => window.electron.ipcRenderer.invoke('clip')
}

function handleErrors() {
  dom.quitBtn.onclick = () => window.electron.ipcRenderer.invoke('quit')

  window.electron.ipcRenderer.on('obsError', (error) => {
    console.error(error)
    dom.overlay.classList.add('visible')
    dom.modalTitle.innerHTML = `💀 ${error.title}`
    dom.modalContent.innerHTML = error.msg//.replaceAll('\n', '<br>')
  })
}

function handleFeedback() {
  dom.fab.onclick = () => dom.feedback.classList.add('visible')
  dom.feedback.onmouseleave = () => dom.feedback.classList.remove('visible')
}

function handlePlayNew() {
  const onEnded = () => {
    if (playlist.length === 0)
      return
    setActiveClip(playlist.shift() as ClipData)
  }

  const onPlayNew = () => {
    console.log('Playing all new clips')
    playlist.push(...clips.filter(clip => clip.new))
    onEnded()
  }

  dom.playNew.onclick = onPlayNew
  dom.clipPreviewVideo.onended = onEnded
}

function handleSounds() {
  dom.soundEffectDemo.onclick = () => {
    soundEffects.newClip.currentTime = 0
    soundEffects.newClip.volume = 0.5
    soundEffects.newClip.play()
  }
}

function handleChrome() {
  dom.minimize.onclick = () => window.electron.ipcRenderer.invoke('minimize')
  dom.maximize.onclick = () => {
    if (dom.maximizeIcon.src.includes('maximize'))
      dom.maximizeIcon.src = 'codicon-chrome-restore.svg'
    else
      dom.maximizeIcon.src = 'codicon-chrome-maximize.svg'
    window.electron.ipcRenderer.invoke('maximize')
  }
  dom.close.onclick = () => window.electron.ipcRenderer.invoke('close')
}

function handleGameAutoDetect() {
  const onGameRunning = (game: string) => {
    if (!autoSelectGame) return

    // No game running
    if (game === '') {
      if (dom.gameSelect.value !== dom.gameWaitingOption.value) {
        console.log(`Switched to waiting for game`)
        dom.gameSelect.value = dom.gameWaitingOption.value
        onGameSelect()
      }
      return
    }

    // Game running
    const config = configs.find(config => config.exe && config.exe.toLowerCase() === game.toLowerCase())
    if (!config)
      return console.error(`Could not find config for ${game}`)
    if (dom.gameSelect.value === config.id) return
    console.log(`Switched to game ${config.id}`)
    dom.gameSelect.value = config.id
    onGameSelect()
  }

  const onCheckboxClick = () => {
    autoSelectGame = dom.gameAutoDetect.checked
    const label = dom.gameAutoDetect.parentNode as HTMLElement
    label.classList.toggle('off')
    dom.gameSelect.title = autoSelectGame ? 'Disable auto selection to manually change the game' : ''
    dom.gameWaitingOption.innerText = autoSelectGame ? 'Waiting for game...' : 'Select game...'
  }

  dom.gameAutoDetect.onclick = onCheckboxClick
  window.electron.ipcRenderer.on('gameRunning', onGameRunning)
}

//////// MAIN
function main() {
  console.log('Clipit started')
  window.electron.ipcRenderer.invoke('initChokidar')
  handleClips()
  handleObsStats()
  handleTriggerConfigs()
  handleManualClip()
  handleErrors()
  handleFeedback()
  handlePlayNew()
  handleSounds()
  handleChrome()
  handleGameAutoDetect()
}

main()