import { Region, Sharp } from 'sharp'

export type CropUnit = 'px' | '%'
export type CropYAnchor = 'top' | 'bottom'
export type CropXAnchor = 'left' | 'right'
export type CropRegion = { y: number, x: number, width: number, height: number }

export default class CropFunction {
  yAnchor: CropYAnchor = 'top'
  xAnchor: CropXAnchor = 'left'
  x: number = 0
  y: number = 0
  xUnit: CropUnit = 'px'
  yUnit: CropUnit = 'px'
  width: number = 100
  height: number = 100
  widthUnit: CropUnit = '%'
  heightUnit: CropUnit = '%'

  constructor(config: any = {}) {
    this.yAnchor = config.yAnchor || this.yAnchor
    this.xAnchor = config.xAnchor || this.xAnchor
    this.x = config.x || this.x
    this.y = config.y || this.y
    this.xUnit = config.xUnit || this.xUnit
    this.yUnit = config.yUnit || this.yUnit
    this.width = config.width || this.width
    this.height = config.height || this.height
    this.widthUnit = config.widthUnit || this.widthUnit
    this.heightUnit = config.heightUnit || this.heightUnit
  }

  /**
   * Renders a cropped section of @param video onto a given @param canvas,
   * according to the various parameters set as member fields of the CropFunction object
   */
   public async cropImage(sharp: Sharp): Promise<Sharp> {
    const region: Region = {
      top: this.y,
      left: this.x,
      width: this.width,
      height: this.height,
    }

    const metadata = await sharp.metadata()
    if (this.yUnit === '%') region.top *= metadata.height! / 100
    if (this.yAnchor === 'bottom') region.top = metadata.height! - region.top
    if (this.heightUnit === '%') region.height *= metadata.height! / 100

    if (this.xUnit === '%') region.left *= metadata.width! / 100
    if (this.xAnchor === 'right') region.left = metadata.width! - region.left
    if (this.widthUnit === '%') region.width *= metadata.width! / 100

    region.top = Math.round(region.top)
    region.left = Math.round(region.left)
    region.width = Math.round(region.width)
    region.height = Math.round(region.height)

    const extract = await sharp.extract(region)
    return Promise.resolve(extract)
  }
}