import type { Scheduler } from 'tesseract.js'
import { createScheduler, createWorker } from 'tesseract.js'

export const NUM_TESSERACT_WORKERS: number = 3

export default class HypetriggerManager {
  /** Array of game events (kills, streaks, other info) */
  eventLog: any[] = []

  /** Collection of all parsed text for comparison & combination */
  parseHistory: any[] = []
  
  /** Tesseract scheduler (pool of workers) */
  scheduler: Scheduler

  /** Flag to ensure that tesseract only gets initialized once */
  tesseractInitialized = false

  /** Observer pattern for event log */
  eventListeners: ((event: any) => any)[] = []

  async initTesseract() {
    if (this.tesseractInitialized) return
    this.scheduler = createScheduler()

    this.logEvent({
      type: 'debug',
      timestamp: new Date(),
      msg: 'Initializing Tesseract...'
    })

    for (let i = 0; i < NUM_TESSERACT_WORKERS; i++) {
      const worker = createWorker({
        errorHandler: (error: any) => {
          console.error('Encountered an error inside a Tesseract worker')
          console.error(error)
          this.restartTesseract()
        }
      })
      await worker.load()
      await worker.loadLanguage('eng')
      await worker.initialize('eng')
      this.scheduler.addWorker(worker)
    }

    this.logEvent({
      type: 'debug',
      timestamp: new Date(),
      msg: `${NUM_TESSERACT_WORKERS} Tesseract worker${NUM_TESSERACT_WORKERS !== 1 ? 's' : ''} I N I T I A L I Z E D`
    })

    this.tesseractInitialized = true
  }

  async restartTesseract(): Promise<any> {
    console.log('Restarting Tesseract...')
    console.time('Tesseract restart')
    this.tesseractInitialized = false
    await this.scheduler.terminate()
    this.scheduler = null
    const promise = await this.initTesseract()
    console.timeEnd('Tesseract restart')
    return promise
  }

  logEvent(event: any){
    this.eventLog.push(event)
    this.eventListeners.forEach(callback => callback(event))
    console.log(event.msg)

    // if (!this.eventLogRef || !this.eventLogRef.current) return
    // const eventLogElem = this.eventLogRef.current
    // eventLogElem.innerHTML += event.msg + '\n'
    // eventLogElem.scrollTo({
    //   top: eventLogElem.scrollHeight,
    //   behavior: 'smooth'
    // })
  }

  onEvent(callback: (event: any) => any) {
    this.eventListeners.push(callback)
  }
}
