import { execFile, execSync } from 'child_process'
import { existsSync, readFileSync, writeFileSync } from 'fs'
import { parse, stringify } from 'ini'
import ObsWebSocket, { OBSStats } from 'obs-websocket-js'
import { basename, resolve } from 'path'
import dataUriToBuffer from './dataUriToBuffer'
import isRunning from './isRunning'

const OBS_POSSIBLE_PATHS = [
  'C:\\Program Files\\obs-studio\\bin\\64bit\\obs64.exe',
  'C:\\Program Files (x86)\\obs-studio\\bin\\64bit\\obs64.exe',
  'C:\\Program Files (x86)\\obs-studio\\bin\\32bit\\obs32.exe',
  'C:\\Program Files\\obs-studio\\bin\\32bit\\obs32.exe',
]
const SETTINGS_FILENAME = `settings.json`
const OBS_LAUNCH_WAIT = 1000
const OBS_LAUNCH_MAX_TRIES = 10
const OBS_MIN_SCREENSHOT_DELAY = 500
const OBS_MIN_VERSION = 27

export interface Settings {
  ObsPath?: string
  ObsWebsocketPort?: number
  ObsWebsocketPassword?: string
  ClipDuration?: number
  ClipDirectory?: string
  ClipDelay?: number
}

export type OBSError = {
  title: string
  msg: string
}

export default class OBSManager {
  connected = false
  ws: ObsWebSocket = new ObsWebSocket()
  screenshot?: Buffer
  stats?: OBSStats
  screenshotPollingInterval = OBS_MIN_SCREENSHOT_DELAY
  screenshotTimeout: NodeJS.Timeout | undefined
  settings: Settings = {}
  nextScreenshotTime: number
  screenshotNumber = 0

  onError: (error: OBSError) => any = (error) => {
    error.msg.split('\n').forEach(console.error)
    process.exit(1)
  }

  async init() {
    try {
      await this.loadSettings()
      await this.findOBS()
      await this.ensureWebsocketPluginInstalled()
      await this.checkOBSSettings()
      await this.ensureOBSRunning()
      await this.connectToWebsocket()
      await this.checkVersion()
      await this.ensureReplayBufferActive()
    } catch (e) {
      throw e // rethrow
    }
  }

  getSettingsPath() {
    return `${process.cwd()}\\${SETTINGS_FILENAME}`
  }

  getOBSExe() {
    return this.settings.ObsPath!
  }

  getOBSRoot() {
    return resolve(this.settings.ObsPath!, '../../../')
  }

  getOBSBasename() {
    return basename(this.settings.ObsPath!)
  }

  getOBSWebsocketDLL() {
    return resolve(this.settings.ObsPath!, '../../../obs-plugins/64bit/obs-websocket.dll')
  }

  getDependenciesRedist() {
    const path_debug = `${process.cwd()}\\dependencies\\obs-websocket-4.9.1-Windows`
    const path_prod = `${process.cwd()}\\resources\\app\\dependencies\\obs-websocket-4.9.1-Windows`
    if (existsSync(path_debug))
      return path_debug
    else if (existsSync(path_prod))
      return path_prod
    else
      throw new Error(`OBS websocket plugin redistributable not found at ${path_debug} or ${path_prod}`)
  }

  getOBSUserDataPath() {
    return `${process.env[(process.platform == 'win32') ? 'USERPROFILE' : 'HOME']}\\AppData\\Roaming\\obs-studio`
  }

  debugPaths() {
    console.group('Debugging file paths')
    console.log(`getSettingsPath() => ${this.getSettingsPath()}`)
    console.log(`getOBSExe() => ${this.getOBSExe()}`)
    console.log(`getOBSRoot() => ${this.getOBSRoot()}`)
    console.log(`getOBSBasename() => ${this.getOBSBasename()}`)
    console.log(`getOBSWebsocketDLL() => ${this.getOBSWebsocketDLL()}`)
    console.log(`getDependenciesRedist() => ${this.getDependenciesRedist()}`)
    console.groupEnd()
  }

  async loadSettings() {
    const path = this.getSettingsPath()
    try {
      const content = readFileSync(path, 'utf8')
      this.settings = JSON.parse(content)
      console.log(`Loaded app settings from ${path}`)
    } catch (e) {
      console.log(`No saved settings found at ${path}`)
    }
  }

  async saveSettings() {
    const path = this.getSettingsPath()
    const content = JSON.stringify(this.settings, null, 2)
    writeFileSync(path, content)
    console.log(`Saved settings to ${path}`)
  }

  async findOBS() {
    const paths: string[] = []
    if (this.settings.ObsPath)
      paths.push(this.settings.ObsPath)
    paths.push(...OBS_POSSIBLE_PATHS)

    for (const path of paths) {
      if (existsSync(path)) {
        console.log(`Found OBS executable at ${path}`)
        if (this.settings.ObsPath !== path) {
          this.settings.ObsPath = path
          await this.saveSettings()
        }
        return
      }
    }

    await this.saveSettings()
    const title = `Couldn't find OBS`
    this.onError({
      title,
      msg: `<p>Clip It works side-by-side with OBS on your computer and uses it to record clips. `
        + `No valid OBS installation could be found after checking ${paths.length} typical default locations.</p>` 
        + `<p>If OBS is already installed in a non-standard location (such as on another drive), please specify it in the settings file: <code>${this.getSettingsPath()}</code>.</p>`
        + `<p>Otherwise, please install OBS from the official website: <a href="https://obsproject.com/" target="blank">https://obsproject.com/</a>.</p>`
        + `<p>Afterwards, run this program again to continue.</p>`
    })
    throw new Error(title)
  }

  async checkWebsocketPluginInstalled() {
    const path = this.getOBSWebsocketDLL()
    const exists = existsSync(path)
    if (!exists)
      console.error(`OBS websocket plugin not found at ${path}`)
    return exists
  }

  async closeOBS() {
    // Should this prompt the user before exiting?
    execSync(`taskkill /im ${this.getOBSBasename()} /t /f /fi "STATUS eq RUNNING"`)
    console.log('OBS closed')
  }

  async ensureOBSClosed() {
    if (await this.OBSRunning())
      await this.closeOBS()
  }

  async installWebsocketPlugin() {
    console.log(`Installing OBS websocket plugin`)

    await this.ensureOBSClosed()
    const from = `"${this.getDependenciesRedist()}"`
    const to = `"${this.getOBSRoot()}"`
    console.log(`Copying ${from} to ${to}`)
    execSync(
      `start-process cmd -Wait -verb runas -argumentlist '/c Xcopy /E /I /Y ${from} ${to}'`,
      { shell: 'powershell.exe' },
    )
    console.log(`Successfully installed OBS websocket plugin`)
  }

  async ensureWebsocketPluginInstalled() {
    let installed = await this.checkWebsocketPluginInstalled()
    if (!installed) await this.installWebsocketPlugin()
    installed = await this.checkWebsocketPluginInstalled()

    if (!installed) {
      const title = `Couldn't install OBS plugin`
      this.onError({
        title,
        msg: `<p>Clip It works side-by-side with OBS on your computer and uses it to record clips. `
          + `To do this, Clip It installs a plugin which allows it to communicate directly with OBS. `
          + `The plugin will start the OBS replay buffer, save clips at the right moment, and receive video frames for AI processing.</p>` 
          + `<p>An automatic installation of the OBS plugin was attempted, but it didn't work. `
          + `The installation typically requires admin permissions — try re-running <code>clipit.exe</code> as administrator.</p>`
          + `<p>If plugin installation fails again, or if you're more comfortable installing the plugin manually, you can do so from the plugin's official Github: <a href="https://github.com/Palakis/obs-websocket" target="blank">https://github.com/Palakis/obs-websocket</a>.</p>`
          + `<p>Afterwards, run this program again to continue.</p>`
      })
      throw new Error(title)
    }
  }

  async checkOBSSettings() {
    const settingsRoot = this.getOBSUserDataPath()

    // Read global settings
    const globalSettingsPath = `${settingsRoot}\\global.ini`
    const globalSettings = parse(readFileSync(globalSettingsPath, 'utf8'))
    const profileDir = globalSettings.Basic.ProfileDir
    console.log(`Loaded global OBS settings from ${globalSettingsPath}`)

    // Global Websocket settings
    if (!globalSettings.WebsocketAPI) {
      globalSettings.WebsocketAPI = {
        AuthSetupPrompted: true // skip the websocket setup prompt if we're installing the plugin automatically
      }

      // Write global settings
      if (await this.OBSRunning()) await this.closeOBS()
      writeFileSync(globalSettingsPath, stringify(globalSettings))
      console.log(`Wrote global OBS settings to ${globalSettingsPath}`)
    }

    // Read basic settings
    const basicSettingsPath = `${settingsRoot}\\basic\\profiles\\${profileDir}\\basic.ini`
    const basicSettings = parse(readFileSync(basicSettingsPath, 'utf8'))
    console.log(`Loaded OBS profile basic settings from ${basicSettingsPath}`)

    // Automatically configure websocket plugin
    let basicSettingsChanged = false
    if (!basicSettings.WebsocketAPI) {
      basicSettings.WebsocketAPI = {
        ServerEnabled: true,
        ServerPort: 4444,
        LockToIPv4: false,
        DebugEnabled: false,
        AlertsEnabled: false,
        AuthRequired: false,
        AuthSecret: "",
        AuthSalt: "",
      }
      basicSettingsChanged = true
    }

    // "Save Replay" hotkey must be set (even if it is only used programmatically)
    if (!basicSettings.Hotkeys || !basicSettings.Hotkeys.ReplayBuffer) {
      if (!basicSettings.Hotkeys)
        basicSettings.Hotkeys = {}
      basicSettings.Hotkeys.ReplayBuffer = `{"ReplayBuffer.Save":[{"key":"OBS_KEY_DEAD_GRAVE"}]}`
      basicSettingsChanged = true
    }

    // Enable replay buffer
    if (!basicSettings.SimpleOutput.RecRB || basicSettings.SimpleOutput.RecRB === 'false') {
      basicSettings.SimpleOutput.RecRB = 'true'
      basicSettings.SimpleOutput.RecRBTime = 15
      basicSettings.SimpleOutput.RecRBSize = 1024
      basicSettingsChanged = true
    }

    // Set duration of replay buffer
    if (this.settings.ClipDuration && parseInt(basicSettings.SimpleOutput.RecRBTime) !== this.settings.ClipDuration) {
      basicSettings.SimpleOutput.RecRBTime = this.settings.ClipDuration
      basicSettingsChanged = true
    } else if (!this.settings.ClipDuration && basicSettings.SimpleOutput.RecRBTime) {
      this.settings.ClipDuration = parseInt(basicSettings.SimpleOutput.RecRBTime)
    }

    // Write basic settings
    if (basicSettingsChanged) {
      if (await this.OBSRunning()) await this.closeOBS()
      writeFileSync(basicSettingsPath, stringify(basicSettings))
      console.log(`Wrote basic OBS settings to ${basicSettingsPath}`)
    }

    // If needed, copy some OBS settings to this app's settings
    let settingsChanged = false
    if (this.settings.ObsWebsocketPort !== basicSettings.WebsocketAPI.ServerPort){
      this.settings.ObsWebsocketPort = basicSettings.WebsocketAPI.ServerPort
      settingsChanged = true
    }

    if (this.settings.ClipDirectory !== basicSettings.SimpleOutput.FilePath){
      this.settings.ClipDirectory = basicSettings.SimpleOutput.FilePath
      settingsChanged = true
    }

    if (settingsChanged)
      await this.saveSettings()
  }

  async OBSRunning() {
    const fileName = this.getOBSBasename()
    const fileNameUnix = fileName.split('.')[0]
    return await isRunning(fileName, fileNameUnix, fileNameUnix)
  }

  async ensureOBSRunning() { 
    const running = await this.OBSRunning()

    if (!running) {
      this.startOBS()
      return new Promise(resolve => setTimeout(resolve, OBS_LAUNCH_WAIT))
    }
  }

  startOBS() {
    const path = this.getOBSExe()
    console.log(`Launching OBS (${path})`)

    try {
      const lastSlash = path.lastIndexOf('\\')
      const cwd = path.substr(0, lastSlash)
      execFile(path, { cwd })
    } catch (e) {
      const title = `Couldn't start OBS`
      this.onError({
        title,
        msg: `<p>Clip It works side-by-side with OBS on your computer and uses it to record clips.</p>`
          + `<p>A valid OBS installation was found at <code>${path}</code>, but it failed to start.</p>`
          + `<p>You can resolve this by confirming that the OBS executable at the specified path is valid, or you can start OBS manually yourself before opening Clip It.</p>`
          + `<p>Afterwards, run this program again to continue.</p>`
      })
      throw new Error(title)
    }
  }

  async connectToWebsocket(tryNum: number = 0) {
    const address = `localhost:${this.settings.ObsWebsocketPort}`
    const password = this.settings.ObsWebsocketPassword
    try {
      await this.ws.connect({ address, password })
      this.connected = true
      console.log(`Connected to OBS via websocket on try #${tryNum} (address: ${address})`)
    } catch (e) {
      // If it was too early and OBS is still starting up, wait a bit and try again
      if (tryNum < OBS_LAUNCH_MAX_TRIES) {
        await new Promise(resolve => setTimeout(resolve, OBS_LAUNCH_WAIT))
        console.log(`Retrying OBS websocket connection... (#${tryNum + 1})`)
        await this.connectToWebsocket(tryNum + 1)
        return
      }

      const title = `Failed to connect to OBS via websocket`
      this.onError({
        title,
        msg: `<p>Clip It works side-by-side with OBS on your computer and uses it to record clips.</p>`
          + `<p>It does this by connecting to OBS via websocket. Connection failed when attempting to use address <code>${address}</code>.`
          + (password ? ` A custom password was specified in the config file and used at time of connection.</p>` : '</p>')
          + `<p>Compare the websocket settings in the Clip It settings file (<code>${this.getSettingsPath()}</code>) to the websocket settings inside OBS (by going to <code>Tools > WebSockets Server Settings</code>).</p>`
          + `<p>Afterwards, run this program again to continue.</p>`
      })
      throw new Error(title)
    }
  }

  async checkVersion() {
    const response = await this.ws.send('GetVersion')
    const obsVersion = response['obs-studio-version'];
    const majorVersion = parseInt(obsVersion.split('.')[0])
    console.log(`OBS version ${obsVersion}`)

    if (majorVersion < OBS_MIN_VERSION) {
      const title = `OBS out of date`
      this.onError({
        title,
        msg: `<p>Clip It works side-by-side with OBS on your computer and uses it to record clips.</p>`
          + `<p>Your current version of OBS (${obsVersion}) is out of date (requires >= ${OBS_MIN_VERSION}.x).</p>`
          + `<p>Please update OBS to the latest version via the standard updater inside of OBS, or via the official website: <a href="https://obsproject.com/" target="blank">https://obsproject.com/</a>.</p>`
          + `<p>Afterwards, run this program again to continue.</p>`
      })
      throw new Error(title)
    }
  }

  async ensureReplayBufferActive() {
    const isActive = (await this.ws.send('GetReplayBufferStatus' as any)).isReplayBufferActive
    console.log(`Replay buffer is currently ${isActive ? 'on' : 'off'}`)

    if (!isActive) {
      try {
        await this.ws.send('StartReplayBuffer')
        console.log(`Started replay buffer recording`)
      } catch (e) {
        const title = `Could not start OBS replay buffer`
        this.onError({
          title,
          msg: `<p>Clip It works side-by-side with OBS on your computer and uses it to record clips.</p>`
            + `<p>The OBS replay buffer was unable to start. Double-check in the OBS settings that Replay Buffer is enabled (under <code>Settings > Output > Recording > Enable Replay Buffer</code>).</p>`
            + `<p>Afterwards, run this program again to continue.</p>`
        })
        throw new Error(title)
      }
    }
  }

  async getScreenshot(cb?: Function) {
    const start = Date.now()

    if (!this.connected) {
      console.error(`getScreenshot called, but websocket is not connected`)
      return
    }

    // Get latest stats
    try {
      this.stats = (await this.ws.send('GetStats')).stats
      if (cb) cb(this.stats)
    } catch (e) {
      console.error(e)
      throw e
    }

    // Fetch screenshot
    const screenshot = await this.ws.send('TakeSourceScreenshot', { embedPictureFormat: 'png' } as any)
    this.screenshot = dataUriToBuffer(screenshot.img)

    // Measure elapsed time and throttle CPU usage from screenshots if needed (prevent OBS from freezing)
    const elapsed = Date.now() - start
    if (elapsed > this.screenshotPollingInterval)
      this.screenshotPollingInterval = elapsed + 100; // Always give at least a little breathing room so OBS doesn't freeze
    else {
      const cpuUsage = this.stats['cpu-usage']
      if (cpuUsage > 90) {
        this.screenshotPollingInterval += 100
        console.warn(`CPU usage is ${Math.round(cpuUsage)}% -- throttling polling interval up to ${this.screenshotPollingInterval}ms`)
      } else if (cpuUsage < 50 && this.screenshotPollingInterval > OBS_MIN_SCREENSHOT_DELAY) {
        this.screenshotPollingInterval -= 100
        this.screenshotPollingInterval = Math.max(this.screenshotPollingInterval, OBS_MIN_SCREENSHOT_DELAY)
        console.log(`CPU usage is ${Math.round(cpuUsage)}% -- decreasing polling interval down to ${this.screenshotPollingInterval}ms`)
      }
    }

    // Go again
    const delay = this.screenshotPollingInterval - elapsed
    this.nextScreenshotTime = Date.now() + delay
    this.screenshotNumber++
    this.screenshotTimeout = setTimeout(() => this.getScreenshot(cb), delay)
  }
}