import FilterFunction from '../filters/FilterFunction';
import ThresholdFilter from '../filters/ThresholdFilter';
import Trigger from '../triggers/trigger';
import type { FilterData, ParserData, TriggerData } from '../types';
import CropFunction from './CropFunction';
import HypetriggerManager from './HypetriggerManager';

export function filterFromJson(json: FilterData): FilterFunction {
  if (json.type === 'threshold')
    return new ThresholdFilter(json)
  throw new Error(`Unsupported filter type: ${json.type}`);
}

export function parserFromJson(trigger: TriggerData, parser: ParserData, manager: HypetriggerManager): Function {
  if (parser.type === 'regex') {
    return (text: string) => {
      if (!text.match(parser.regex)) return false
      manager.logEvent({
        type: trigger.id,
        timestamp: new Date(),
        rawText: text,
        msg: `[${trigger.id}] EVENT => ${trigger.title}`,
      })
      return true
    }
  }
  throw new Error(`Unsupported parser type: ${parser.type}`);
}

export default function triggerFromJson(json: TriggerData, manager: HypetriggerManager): Trigger {
  return new Trigger({
    manager,
    title: json.title,
    id: json.id,
    enabled: json.enabled,
    cropFunction: new CropFunction(json.crop),
    filters: json.filters.map(filter => filterFromJson(filter)),
    parseFunction: parserFromJson(json, json.parser, manager),
    debug: json.debug,
  })
}