import FuzzySet from 'fuzzyset'
import CropFunction from '../lib/CropFunction'
import ThresholdFilter from '../filters/ThresholdFilter'
import Trigger from './trigger'

const kills_fuzzyset = FuzzySet(['KILL'])

/**
 * Targets the red box which contains the phrase "Nth KILL"
 * in the bottom middle of the screen. It displays for
 * approximately 2 seconds.
 * 
 * The "N" in "Nth KILL" refers to number of kills this life,
 * (without dying), not the total number of kills (which shows
 * in the top-right in Warzone)
 */
export default class KillNotifTrigger extends Trigger {
  /**
   * This crop is optimized for single-digit killcounts,
   * however it usually has *just* enough space to capture
   * 2-digit killcounts as well
   */
  cropFunction = new CropFunction({
    y: (245/1440) * 100,
    yUnit: '%',
    yAnchor: 'bottom',
    x: 47.5,
    xUnit: '%',
    xAnchor: 'left',
    width: (98/1920) * 100,
    widthUnit: '%',
    height: (52/1440) * 100,
    heightUnit: '%',
  })

  filters = [new ThresholdFilter({
    title: 'Threshold',
    r: 255,
    g: 0,
    b: 0,
    threshold: 42,
  })]

  /**
   * Primarily looks for fuzzy text match for the word 'KILL'
   * The guaranteed-contrast red text against a transparent background
   * is pretty consistent to parse.
   * 
   * Duplication is prevented by (naively) limiting kill events
   * to only ONE per two second period (the animation duration of the kill notif)
   */
  parseFunction = (text: string) => {
    const parts = text.split(' ')
    if (parts.length !== 2) return false
    const fuzzy = kills_fuzzyset.get(parts[1])
    if (fuzzy && fuzzy[0] && fuzzy[0][0] && fuzzy[0][0] < 0.5) return false

    this.manager.parseHistory.push({
      type: 'kill-notif',
      rawText: text,
      timestamp: new Date(),
    })

    const killEvent = this.manager.eventLog.find(event =>
      event.type === 'kill'
      && new Date().getTime() - event.timestamp.getTime() < 2000
    )

    if (killEvent) return false

    this.manager.logEvent({
      type: 'kill',
      timestamp: new Date(),
      rawText: text,
      msg: '[KillNotifTrigger] Player killed',
    })

    return true
  }

  effects = ['notification']
}