import CropFunction from '../lib/CropFunction';
import ThresholdFilter from '../filters/ThresholdFilter';
import { mode } from '../lib/util';
import Trigger from './trigger';

/**
 * Targets the total number of kills displayed in
 * white text in the top right of the screen at almost
 * all times.
 */
export default class KillTotalTrigger extends Trigger {
  /**
   * This crop region unfortunately coincides with the
   * "number of players remaining in the airplane" number
   * during intial plane flight.
   * 
   * That drawback is mitigated in the parsing function.
   */
  cropFunction = new CropFunction({
    y: (60/1440) * 100,
    yUnit: '%',
    yAnchor: 'top',
    x: (2426/2560) * 100,
    xUnit: '%',
    xAnchor: 'left',
    width: (70/2560) * 100,
    widthUnit: '%',
    height: (40/1440) * 100,
    heightUnit: '%',
  })

  filters = [new ThresholdFilter({
    title: 'Threshold',
    r: 255,
    g: 255,
    b: 255,
    threshold: 42,
  })]

 /**
   * The contrast of this text is not great, so multiple frames
   * are compared in order to extract the correct number of total kills
   * with maximal confidence.
   */
  parseFunction = (text: string) => {
    // Strict format
    const match = text.match(/^[\n\s]*\d{1,2}[\n\s]*$/)
    if (!match) return false
  
    // Parse number
    const parsed = parseInt(text)
    if (!parsed) return false
  
    this.manager.parseHistory.push({
      type: 'kill-total',
      rawText: text,
      parsed,
      timestamp: new Date(),
    })

    // Generate Event
    let killTotalResults = this.manager.parseHistory.filter(result => result.type === 'kill-total')
    killTotalResults.reverse()
    killTotalResults = killTotalResults.slice(0, 5)
    if (killTotalResults.length < 2) return false // Double check AT LEAST once the first time
    let numbers = killTotalResults.map(result => result.parsed)
    // numbers is now an array of raw numbers (from the 5 most recent successful parses)
    let modes = mode(numbers) // find the consensus (the result gotten most often)
    if (modes.length !== 1) return false // If there is a tie, it's not good enough
    const newKillTotal = modes[0]
    
    const killTotalEvents = this.manager.eventLog.filter(event => event.type === 'kill-total')
    const lastKillTotal = killTotalEvents[killTotalEvents.length - 1]
    if (lastKillTotal && lastKillTotal.value === newKillTotal) return false // We already announced this kill total
  
    const event = {
      timestamp: new Date(),
      type: 'kill-total',
      value: newKillTotal,
      msg: `[KillTotalTrigger] ${newKillTotal}`,
    }
    this.manager.logEvent(event)
    this.handle10XBomb(event)
    return true
  }

  /**
   * This trigger function handles "10X-BOMBS",
   * aka 10-bombs, 20-bombs, 30-bombs, etc.
   * 
   * To help reduce unusual behavior, it also checks
   * that there has been a recent kill before dispatching the event.
   * 
   * @TODO BUGFIX this will only work one time per session!
   */
  handle10XBomb = (event: any) => {
    const _10bombEvent = this.manager.eventLog.filter(event => event.type === '10-bomb')[0]
    const _20bombEvent = this.manager.eventLog.filter(event => event.type === '20-bomb')[0]
    const _30bombEvent = this.manager.eventLog.filter(event => event.type === '30-bomb')[0]
    const _40bombEvent = this.manager.eventLog.filter(event => event.type === '40-bomb')[0]

    const recentKills = this.manager.eventLog.filter(event =>
      event.type === 'kill'
      && new Date().getTime() - event.timestamp <= 5000
    )
    if (recentKills.length === 0) return

    if (!_40bombEvent && event.value >= 40) {
      this.manager.logEvent({
        timestamp: new Date(),
        type: '40-bomb',
        msg: '[KillTotalTrigger] 40-BOMB'
      })
      // this.manager.socket.emit('triggerEvent', {
      //   type: 'notification',
      //   msg: '40-BOMB'
      // })
    } else if (!_30bombEvent && event.value >= 30) {
      this.manager.logEvent({
        timestamp: new Date(),
        type: '30-bomb',
        msg: '[KillTotalTrigger] 30-BOMB'
      })
      // this.manager.socket.emit('triggerEvent', {
      //   type: 'notification',
      //   msg: '30-BOMB'
      // })
    } else if (!_20bombEvent && event.value >= 20) {
      this.manager.logEvent({
        timestamp: new Date(),
        type: '20-bomb',
        msg: '[KillTotalTrigger] 20-BOMB'
      })
      // this.manager.socket.emit('triggerEvent', {
      //   type: 'notification',
      //   msg: '20-BOMB'
      // })
    } else if (!_10bombEvent && event.value >= 10) {
      this.manager.logEvent({
        timestamp: new Date(),
        type: '10-bomb',
        msg: '[KillTotalTrigger] 10-BOMB'
      })
      // this.manager.socket.emit('triggerEvent', {
      //   type: 'notification',
      //   msg: '10-BOMB'
      // })
    }
  }

  effects = ['notification', 'sound', 'something else']
}