import sharp, { Sharp } from 'sharp'
import { RecognizeResult, Scheduler } from 'tesseract.js'
import FilterFunction from '../filters/FilterFunction'
import CropFunction from '../lib/CropFunction'
import HypetriggerManager from '../lib/HypetriggerManager'

export default class Trigger {
  /** Title of the trigger which shows in the sidebar */
  title: string

  /** Unique ID of the trigger */
  id: string

  /** Reference to parent Hypetrigger manager object */
  manager: HypetriggerManager

  /** An instance of CropFunction defining the rectangle subimage boundaries */
  cropFunction: CropFunction = new CropFunction()

  /** An array of filters to apply to the cropped section */
  filters: FilterFunction[] = []

  /** An array of attached effects */
  effects: any[] = []

  /** Whether to generate debug output for this trigger **/
  debug = false

  /** Whether this instance is active and listening for trigger events */
  enabled = true

  /** Recursively re-runs the recognize step as fast as possible */
  runForever = true

  /** Remember the index of the last frame that was processed */
  lastScreenshotNumber: number

  /**
   * A parse function to execute on recognized text
   * @param {string} text recognized text (possibly an empty string)
   * @returns {any} context-dependent return value, or no return
   */
  parseFunction: (text: string) => boolean
    = () => false

  constructor(config: any) {
    this.title = config.title!
    this.id = config.id!
    this.manager = config.manager!
    this.cropFunction = config.cropFunction || this.cropFunction
    this.parseFunction = config.parseFunction || this.parseFunction
    this.filters = config.filters || this.filters
    this.effects = config.effects || this.effects
    this.enabled = (config.enabled === true  || config.enabled === false) ? config.enabled : this.enabled
    this.runForever = (config.runForever === true  || config.runForever === false) ? config.runForever : this.runForever
    this.debug = (config.debug === true  || config.debug === false) ? config.debug : this.debug
  }

  /**
   * Takes a snapshot of pixels from the <video> element and draws it to the <canvas>,
   * cropping according to the region defined by #this.cropFunction(this.hypetrigger.videoElem)
   */
  public async filter(input: Sharp) {
    const { data, info } = await input.clone().raw().toBuffer({ resolveWithObject: true })
    let pixelArray = new Uint8ClampedArray(data.buffer)

    for (let i = 0; i < this.filters.length; i++) {
      pixelArray = this.filters[i].function(pixelArray)
    }

    const { width, height, channels } = info
    const output = await sharp(pixelArray as any, { raw: { width, height, channels: channels as any }})
    return Promise.resolve(output)
  }

  /**
   * Recognizes text in the cropped & filtered image 
   * @param {Scheduler} scheduler Tesseract Scheduler used for this job
   */
  public recognize(scheduler: Scheduler, buffer: Buffer): Promise<RecognizeResult> {
    return scheduler.addJob('recognize', buffer) as Promise<RecognizeResult>
  }

  public async run(inputBuffer: Buffer) {
    if (this.debug)
      console.time(this.title)
    
    // Load
    const img = await sharp(inputBuffer)

    // Crop
    const cropped = await this.cropFunction.cropImage(img)
    if (this.debug)
      cropped.clone().withMetadata({ density: 70 }).png()
        .toFile(`debug/${this.id}-cropped.png`)
        .then(() => console.log(`[${this.id}] CROP => Wrote debug/${this.id}-cropped.png`))

    // Filter
    const filtered = await this.filter(cropped.clone())
    if (this.debug)
      filtered.clone().withMetadata({ density: 70 }).png()
        .toFile(`debug/${this.id}-filtered.png`)
        .then(() => console.log(`[${this.id}] FILTER => Wrote debug/${this.id}-filtered.png`))

    // Recognize
    const buffer = await filtered.withMetadata({ density: 70 }).png().toBuffer()
    const result = await this.recognize(this.manager.scheduler, buffer)

    if (this.debug)
      console.log(`[${this.id}] OCR => "${result.data.text.trim()}"`)

    // Parse
    const parse = this.parseFunction(result.data.text)
    if (this.debug)
      console.log(`[${this.id}] PARSE => ${parse}`)
  
    if (this.debug)
      console.timeEnd(this.title)
  }
}