/*
 * Copyright (C) 2013  Rolf Timmermans
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#define INITGUID

#include <d7z.h>
#include <7zip/Common/StreamObjects.h>
#include <7zip/Compress/DeflateEncoder.h>
#include <7zip/Compress/ZlibEncoder.h>

#include "deflate7z.h"
#include "dynstream.h"

struct state {
    NCompress::NDeflate::NEncoder::CEncProps props;

    /* Wrap streams to prevent encoder from releasing them. */
    CMyComPtr<CDynStream> in;
    CMyComPtr<CDynStream> out;
};

static inline void stream_consume(d7zs* stream) {
    if (!stream->avail_in) return;
    unsigned int length;
    stream->state->in->Write(stream->next_in, stream->avail_in, &length);
    stream->next_in += length;
    stream->avail_in -= length;
    stream->total_in += length;
}

static inline void stream_produce(d7zs* stream) {
    if (!stream->avail_out) return;
    unsigned int length;
    stream->state->out->Read(stream->next_out, stream->avail_out, &length);
    stream->next_out += length;
    stream->avail_out -= length;
    stream->total_out += length;
}

static inline void stream_compress_buffered(d7zs* stream) {
    NCompress::NZlib::CEncoder zlib;
    zlib.Create();
    zlib.DeflateEncoderSpec->SetProps(&stream->state->props);
    zlib.Code(stream->state->in, stream->state->out, 0, 0, 0);
}

static inline bool stream_buffering(d7zs* stream) {
    return stream->state->out->GetSize() == 0;
}

static inline bool stream_ended(d7zs* stream) {
    return stream->state->in->Eof() && stream->state->out->Eof();
}

static void reset_stream(d7zs* stream) {
    stream->total_in = 0;
    stream->total_out = 0;
    stream->adler = 1;
    stream->data_type = Z_UNKNOWN;
    stream->msg = NULL;

    /* We can safely reassign because CMyComPtr implements refcounting. */
    stream->state->in = new CDynStream;
    stream->state->out = new CDynStream;
}

int d7z_deflateInit(d7zs* stream, int level) {
    if (!stream) return Z_STREAM_ERROR;

    stream->state = new state;
    stream->state->props.Level = level;

    reset_stream(stream);

    return Z_OK;
}

int d7z_deflate(d7zs* stream, int flush) {
    if (!stream || !stream->state) return Z_STREAM_ERROR;

    /* Refuse to be called without output space. */
    if (stream->avail_out == 0) return Z_BUF_ERROR;

    /* Calling deflate without Z_NU_FLUSH after Z_FINISH is an error. */
    if (!stream_buffering(stream) && flush != Z_FINISH) return Z_STREAM_ERROR;

    /* Consume any input and compress stream if called with Z_FINISH. */
    stream_consume(stream);
    if (stream_buffering(stream) && flush == Z_FINISH) stream_compress_buffered(stream);

    /* Flush any output we have and return Z_STREAM_END if done. */
    stream_produce(stream);
    if (stream_ended(stream) && flush == Z_FINISH) return Z_STREAM_END;

    /* Waiting for client to process input/output buffers. */
    return Z_OK;
}

int d7z_deflateEnd(d7zs* stream) {
    if (!stream || !stream->state) return Z_STREAM_ERROR;

    /* Check if client completely processed the input/output buffers. */
    bool ended = stream_ended(stream);

    delete stream->state;
    stream->state = NULL;

    return ended ? Z_OK : Z_DATA_ERROR;
}

int d7z_deflateReset(d7zs* stream) {
    if (!stream || !stream->state) return Z_STREAM_ERROR;

    reset_stream(stream);

    return Z_OK;
}
