"use strict";
const events_1 = require("events");
const trtc = require('../build/Release/trtc_electron_edu_sdk.node');
const Renderer_1 = require("./Renderer");
const YUVBuffer = require('yuv-buffer');
const YCbCr = require('yuv-canvas/src/YCbCr');
const pkg = require('../package.json');
const {
    TRTCParams,
    TRTCVideoEncParam,
    TRTCNetworkQosParam,
    TRTCTranscodingConfig,
    TRTCPublishCDNParam,
    TRTCVideoStreamType,
    TRTCVideoFillMode,
    TRTCVideoFrame,
    TRTCVideoPixelFormat,
    TRTCVideoBufferType
} = require('./trtc_define');


/**
 * 腾讯云视频通话功能的主要接口类
 * 
 * Nouns[1]: 主流 - TRTC 里称摄像头这一路的画面叫做“主流”（或主路）画面。
 * Nouns[2]: 辅流(substream) - TRTC 里称屏幕分享或者播片这一路的画面叫做“辅流”（或辅路）画面。
 * Nouns[3]: 播片(vodplay) - TRTC 的 Windows 版本支持将本地的一个视频文件分享出去，这个功能成为“播片”。
 * 
 * @example
 * // 创建/使用/销毁 TRTCCloud 对象的示例代码：
 * const TRTCCloud = require('trtc-electron-sdk');
 * this.rtcCloud = new TRTCCloud();
 * // 获取 SDK 版本号
 * let version = this.rtcCloud.getSDKVersion();
 */
class TRTCCloud extends events_1.EventEmitter {

    constructor() {
        super();
        this.rtcCloud = new trtc.NodeRTCCloud(pkg.version);
        this.streams = new Map();
        this.firstVideoFrameList = [];
        this.playStreamType = TRTCVideoStreamType.TRTCVideoStreamTypeBig;
        this.renderMode = this._isSupportWebGL() ? 1 : 2;
        this.initEventHandler();
        console.log("constructor render:" + this.renderMode + "|sdk version:" + pkg.version +
            "|native sdk version:" + this.rtcCloud.getSDKVersion());

        //视频数据回调
        this.imageData = null;

        //VideoCallback 本地和远程
        this.localVideoCallback = null;
        this.remoteVideoCallback = new Map();
    }

    /**
     * 创建 TRTCCloud 对象单例
     */
    static getTRTCShareInstance() {
        if (!this.instance) {
            this.instance = new TRTCCloud();
        }
        return this.instance;
    }

    /**
     * 释放 TRTCCloud 对象并清理资源
     */
    static destroyTRTCShareInstance() {
        if (!this.instance) {
            this.instance.destroy();
            this.instance = null;
        }
    }

    /**
     * 清理资源
     */
    destroy() {
        this.firstVideoFrameList = [];
        this.rtcCloud.destroy();
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                       设置 TRTCCallback 回调
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 设置 TRTCCloud 回调
     * 
     * @example
     * // 创建/使用/销毁 TRTCCloud 对象的示例代码：
     * const TRTCCloud = require('trtc-electron-sdk');
     * this.rtcCloud = new TRTCCloud();
     * 
     * // 注册回调的方法，事件关键字详见下方"通用事件回调"
     * subscribeEvents = (rtcCloud) => {
     *  rtcCloud.on('onError', (errcode, errmsg) => {
     *      console.info('trtc_demo: onError :' + errcode + " msg" + errmsg);
     *  });
     * };
     * 
     * @namespace TRTCCallback
     */
    initEventHandler() {
        const self = this;
        const fire = (event, ...args) => {
            setImmediate(() => {
                this.emit(event, ...args);
            });
        };
        this.rtcCloud.onEvent('apierror', (funcName) => {
            console.error(`api ${funcName} failed. this is an error
              thrown by c++ addon layer. it often means sth is
              going wrong with this function call and it refused
              to do what is asked. kindly check your parameter types
              to see if it matches properly.`);
        });
        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （一）通用事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 1.1 错误回调：SDK 不可恢复的错误，一定要监听，并分情况给用户适当的界面提示。
         * 
         * @event TRTCCallback#onError
         * @param {Number} errCode - 错误码
         * @param {String} errMsg  - 错误信息
         */
        this.rtcCloud.onEvent('onError', function (errCode, errMsg) {
            fire('onError', errCode, errMsg);
        });

        /**
         * 1.2 警告回调：用于告知您一些非严重性问题，例如出现了卡顿或者可恢复的解码失败。
         * 
         * @event TRTCCallback#onWarning
         * @param {Number} warningCode - 警告码
         * @param {String} warningMsg  - 警告信息
         */
        this.rtcCloud.onEvent('onWarning', function (warningCode, warningMsg) {
            fire('onWarning', warningCode, warningMsg);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （二）房间事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 2.1 已加入房间的回调
         * 
         * 调用 TRTCCloud 中的 enterRoom() 接口执行进房操作后，会收到来自 SDK 的 onEnterRoom(result) 回调：
         * 如果加入成功，result 会是一个正数（result > 0），代表加入房间的时间消耗，单位是毫秒（ms）。
         * 如果加入失败，result 会是一个负数（result < 0），代表进房失败的错误码。
         * 进房失败的错误码含义请查阅[错误码表](https://cloud.tencent.com/document/product/647/32257)。
         * 
         * 注意:
         * - 在 Ver6.6 之前的版本，只有进房成功会抛出 onEnterRoom(result) 回调，进房失败由 onError() 回调抛出。
         * - 在 Ver6.6 及之后改为：进房成功返回正的 result，进房失败返回负的 result，同时进房失败也会有 onError() 回调抛出。
         *
         * @event TRTCCallback#onEnterRoom
         * @param {Number} result - result > 0 时为进房耗时（ms），result < 0 时为进房错误码。
         */
        this.rtcCloud.onEvent('onEnterRoom', function (result) {
            fire('onEnterRoom', result);
        });

        /**
         * 2.2 离开房间的事件回调
         *
         * 调用 TRTCCloud 中的 exitRoom() 接口会执行退出房间的相关逻辑，例如释放音视频设备资源和编解码器资源等。
         * 待资源释放完毕，SDK 会通过 onExitRoom() 回调通知到您。
         *
         * 如果您要再次调用 enterRoom() 或者切换到其他的音视频 SDK，请等待 onExitRoom() 回调到来后再执行相关操作。
         * 否则可能会遇到例如摄像头、麦克风设备被强占等各种异常问题。
         *
         * @event TRTCCallback#onExitRoom
         * @param {Number} reason - 离开房间原因，0：主动调用 exitRoom 退房；1：被服务器踢出当前房间；2：当前房间整个被解散。
         */
        this.rtcCloud.onEvent('onExitRoom', function (reason) {
            self.firstVideoFrameList = [];
            fire('onExitRoom', reason);
        });

        /**
         * 2.3 切换角色的事件回调
         *
         * 调用 TRTCCloud 中的 switchRole() 接口会切换主播和观众的角色，该操作会伴随一个线路切换的过程，
         * 待 SDK 切换完成后，会抛出 onSwitchRole() 事件回调。
         *
         * @event TRTCCallback#onSwitchRole
         * @param {Number} errCode - 错误码，ERR_NULL 代表切换成功，其他请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param {String} errMsg  - 错误信息。
         */
        this.rtcCloud.onEvent('onSwitchRole', function (errCode, errMsg) {
            fire('onSwitchRole', errCode, errMsg);
        });

        /**
         * 2.4 请求跨房通话（主播 PK）的结果回调
         *
         * 调用 TRTCCloud 中的 connectOtherRoom() 接口会将两个不同房间中的主播拉通视频通话，也就是所谓的“主播PK”功能。
         * 调用者会收到 onConnectOtherRoom() 回调来获知跨房通话是否成功，
         * 如果成功，两个房间中的所有用户都会收到 PK 主播的 onUserVideoAvailable() 回调。
         *
         * @event TRTCCallback#onConnectOtherRoom
         * @param {String} userId  - 要 PK 的目标主播 userId。
         * @param {Number} errCode - 错误码，ERR_NULL 代表切换成功，其他请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param {String} errMsg  - 错误信息。
         */
        this.rtcCloud.onEvent('onConnectOtherRoom', function (userId, errCode, errMsg) {
            fire('onConnectOtherRoom', userId, errCode, errMsg);
        });

        /**
         * 2.5 结束跨房通话（主播 PK）的结果回调
         * 
         * @event TRTCCallback#onDisconnectOtherRoom
         * @param {Number} errCode - 错误码，ERR_NULL 代表切换成功，其他请查阅[错误码表](https://cloud.tencent.com/document/product/647/32257)。
         * @param {String} errMsg  - 错误信息。
         */
        this.rtcCloud.onEvent('onDisconnectOtherRoom', function (errCode, errMsg) {
            fire('onDisconnectOtherRoom', errCode, errMsg);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （三）成员事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 3.1 有用户加入当前房间
         *
         * 出于性能方面的考虑，在两种不同的应用场景下，该通知的行为会有差别：
         * - 视频通话场景（TRTCAppSceneVideoCall）：该场景下用户没有角色的区别，任何用户进入房间都会触发该通知。
         * - 在线直播场景（TRTCAppSceneLIVE）：在线直播场景不限制观众的数量，如果任何用户进出都抛出回调会引起很大的性能损耗，所以该场景下只有主播进入房间时才会触发该通知，观众进入房间不会触发该通知。
         *
         * 注意: onRemoteUserEnterRoom 和 onRemoteUserLeaveRoom 只适用于维护当前房间里的“成员列表”，如果需要显示远程画面，建议使用监听 onUserVideoAvailable() 事件回调。
         *
         * @event TRTCCallback#onRemoteUserEnterRoom
         * @param {String} userId - 用户标识
         */
        this.rtcCloud.onEvent('onRemoteUserEnterRoom', function (userId) {
            fire('onRemoteUserEnterRoom', userId);
        });

        /**
         * 3.2 有用户离开当前房间
         *
         * 与 onRemoteUserEnterRoom 相对应，在两种不同的应用场景下，该通知的行为会有差别：
         * - 视频通话场景（TRTCAppSceneVideoCall）：该场景下用户没有角色的区别，任何用户的离开都会触发该通知。
         * - 在线直播场景（TRTCAppSceneLIVE）：只有主播离开房间时才会触发该通知，观众离开房间不会触发该通知。
         *
         * @event TRTCCallback#onRemoteUserLeaveRoom
         * @param {String} userId - 用户标识
         * @param {Number} reason - 离开原因，0表示用户主动退出房间，1表示用户超时退出，2表示被踢出房间。
         */
        this.rtcCloud.onEvent('onRemoteUserLeaveRoom', function (userId, reason) {
            fire('onRemoteUserLeaveRoom', userId, reason);
        });

        /**
         * 3.3 用户是否开启摄像头视频
         *
         * 当您收到 onUserVideoAvailable(userId, YES) 通知时，代表该路画面已经有可用的视频数据帧到达。
         * 之后，您需要调用 startRemoteView(userId) 接口加载该用户的远程画面。
         * 再之后，您还会收到名为 onFirstVideoFrame(userId) 的首帧画面渲染回调。
         *
         * 当您收到了 onUserVideoAvailable(userId, NO) 通知时，代表该路远程画面已经被关闭，这可能是
         * 由于该用户调用了 muteLocalVideo() 或 stopLocalPreview() 所致。
         *
         * @event TRTCCallback#onUserVideoAvailable
         * @param {String}  userId    - 用户标识
         * @param {Boolean} available - 画面是否开启
         */
        this.rtcCloud.onEvent('onUserVideoAvailable', function (userId, available) {
            if (process.platform == 'darwin') {
                if (available == 0) {
                    let index = self.firstVideoFrameList.findIndex(function (item) {
                        return userId === item.uid && item.type === self.playStreamType;
                    });
                    if (index !== -1) {
                        self.firstVideoFrameList.splice(index, 1);
                    }
                }
            }
            fire('onUserVideoAvailable', userId, available);
        });

        /**
         * 3.4 用户是否开启屏幕分享
         *
         * 注意: 显示辅路画面使用的函数是 startRemoteSubStreamView() 而非 startRemoteView()。
         * 
         * @event TRTCCallback#onUserSubStreamAvailable
         * @param {String}  userId    - 用户标识
         * @param {Boolean} available - 屏幕分享是否开启
         */
        this.rtcCloud.onEvent('onUserSubStreamAvailable', function (userId, available) {
            if (process.platform == 'darwin') {
                if (available == 0) {
                    let index = self.firstVideoFrameList.findIndex(function (item) {
                        return userId === item.uid && item.type === TRTCVideoStreamType.TRTCVideoStreamTypeSub;
                    });
                    if (index !== -1) {
                        self.firstVideoFrameList.splice(index, 1);
                    }
                }
            }
            fire('onUserSubStreamAvailable', userId, available);
        });

        /**
         * 3.5 用户是否开启音频上行
         *
         * @event TRTCCallback#onUserAudioAvailable
         * @param {String}  userId    - 用户标识
         * @param {Boolean} available - 声音是否开启
         */
        this.rtcCloud.onEvent('onUserAudioAvailable', function (userId, available) {
            fire('onUserAudioAvailable', userId, available);
        });

        /**
         * 3.6 开始渲染本地或远程用户的首帧画面
         *
         * 如果 userId 为 null，表示开始渲染本地采集的摄像头画面，需要您先调用 startLocalPreview 触发。
         * 如果 userId 不为 null，表示开始渲染远程用户的首帧画面，需要您先调用 startRemoteView 触发。
         *
         * 注意: 只有当您调用 startLocalPreview()、startRemoteView() 或 startRemoteSubStreamView() 之后，才会触发该回调。
         *
         * @event TRTCCallback#onFirstVideoFrame
         * @param {String} userId     - 本地或远程用户 ID，如果 userId == null 代表本地，userId != null 代表远程。
         * @param {Number} streamType - 视频流类型：摄像头或屏幕分享。
         * @param {Number} width      - 画面宽度
         * @param {Number} height     - 画面高度
         */
        this.rtcCloud.onEvent('onFirstVideoFrame', function (userId, streamType, width, height) {
            fire('onFirstVideoFrame', userId, streamType, width, height);
        });

        /**
         * 3.7 开始播放远程用户的首帧音频（本地声音暂不通知）
         *
         * @event TRTCCallback#onFirstAudioFrame
         * @param {String} userId - 远程用户 ID。
         */
        this.rtcCloud.onEvent('onFirstAudioFrame', function (userId) {
            fire('onFirstAudioFrame', userId);
        });

        /**
         * 3.8 首帧本地视频数据已经被送出
         *
         * SDK 会在 enterRoom() 并 startLocalPreview() 成功后开始摄像头采集，并将采集到的画面进行编码。
         * 当 SDK 成功向云端送出第一帧视频数据后，会抛出这个回调事件。
         *
         * @event TRTCCallback#onSendFirstLocalVideoFrame
         * @param {Number} streamType - 视频流类型，主画面、小画面或辅流画面（屏幕分享）
         */
        this.rtcCloud.onEvent('onSendFirstLocalVideoFrame', function (streamType) {
            fire('onSendFirstLocalVideoFrame', streamType);
        });

        /**
         * 3.9 首帧本地音频数据已经被送出
         *
         * SDK 会在 enterRoom() 并 startLocalAudio() 成功后开始麦克风采集，并将采集到的声音进行编码。
         * 当 SDK 成功向云端送出第一帧音频数据后，会抛出这个回调事件。
         * 
         * @event TRTCCallback#onSendFirstLocalAudioFrame
         */
        this.rtcCloud.onEvent('onSendFirstLocalAudioFrame', function () {
            fire('onSendFirstLocalAudioFrame');
        });

        /**
         * 3.10 废弃接口：有主播加入当前房间
         *
         * 该回调接口可以被看作是 onRemoteUserEnterRoom 的废弃版本，不推荐使用。请使用 onUserVideoAvailable 或 onRemoteUserEnterRoom 进行替代。
         *
         * @deprecated 从 TRTCSDK6.8 后该接口已被废弃，不推荐使用
         * @event TRTCCallback#onUserEnter
         * @param {String} userId - 用户标识
         */
        this.rtcCloud.onEvent('onUserEnter', function (userId) {
            fire('onUserEnter', userId);
        });

        /**
         * 3.11 废弃接口：有主播离开当前房间
         *
         * 该回调接口可以被看作是 onRemoteUserLeaveRoom 的废弃版本，不推荐使用。请使用 onUserVideoAvailable 或 onRemoteUserEnterRoom 进行替代。
         *
         * @deprecated 从 TRTCSDK6.8 后该接口已被废弃，不推荐使用
         * @event TRTCCallback#onUserExit
         * @param {String} userId - 用户标识
         * @param {Number} reason - 离开原因代码，区分用户是正常离开，还是由于网络断线等原因离开。
         */
        this.rtcCloud.onEvent('onUserExit', function (userId, reason) {
            fire('onUserExit', userId, reason);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （四）统计和质量回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 4.1 网络质量：该回调每2秒触发一次，统计当前网络的上行和下行质量
         *
         * 注意: userId == null 代表自己当前的视频质量
         *
         * @event TRTCCallback#onNetworkQuality
         * @param {TRTCLocalStatistics} localQuality  - 上行网络质量
         * @param {TRTCRemoteStatistics[]} remoteQuality - 下行网络质量
         */
        this.rtcCloud.onEvent('onNetworkQuality', function (localQuality, remoteQuality) {
            fire('onNetworkQuality', localQuality, remoteQuality);
        });

        /**
         * 4.2 技术指标统计回调
         *
         * 如果您是熟悉音视频领域相关术语，可以通过这个回调获取 SDK 的所有技术指标。
         * 如果您是首次开发音视频相关项目，可以只关注 onNetworkQuality 回调。
         *
         * 注意: 每2秒回调一次
         * 
         * @event TRTCCallback#onStatistics
         * @param {TRTCStatistics} statis - 统计数据，包括本地和远程的
         */
        this.rtcCloud.onEvent('onStatistics', function (statis) {
            fire('onStatistics', statis);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （五）服务器事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 5.1 SDK 跟服务器的连接断开
         * 
         * @event TRTCCallback#onConnectionLost
         */
        this.rtcCloud.onEvent('onConnectionLost', function () {
            fire('onConnectionLost');
        });

        /**
         * 5.2 SDK 尝试重新连接到服务器
         * 
         * @event TRTCCallback#onTryToReconnect
         */
        this.rtcCloud.onEvent('onTryToReconnect', function () {
            fire('onTryToReconnect');
        });

        /**
         * 5.3 SDK 跟服务器的连接恢复
         * 
         * @event TRTCCallback#onConnectionRecovery
         */
        this.rtcCloud.onEvent('onConnectionRecovery', function () {
            fire('onConnectionRecovery');
        });

        /**
         * 5.4 服务器测速的回调，SDK 对多个服务器 IP 做测速，每个 IP 的测速结果通过这个回调通知
         *
         * @event TRTCCallback#onSpeedTest
         * @param {TRTCSpeedTestResult} currentResult - 当前完成的测速结果
         * @param {Number} finishedCount - 已完成测速的服务器数量
         * @param {Number} totalCount    - 需要测速的服务器总数量
         */
        this.rtcCloud.onEvent('onSpeedTest', function (currentResult, finishedCount, totalCount) {
            fire('onSpeedTest', currentResult, finishedCount, totalCount);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （六）硬件设备事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 6.1 摄像头准备就绪
         * 
         * @event TRTCCallback#onCameraDidReady
         */
        this.rtcCloud.onEvent('onCameraDidReady', function () {
            fire('onCameraDidReady');
        });

        /**
         * 6.2 麦克风准备就绪
         * 
         * @event TRTCCallback#onMicDidReady
         */
        this.rtcCloud.onEvent('onMicDidReady', function () {
            fire('onMicDidReady');
        });

        /**
         * 6.3 userId 对应的成员语音音量
         * 
         * 您可以通过调用 TRTCCloud 中的 enableAudioVolumeEvaluation 接口来开关这个回调或者设置它的触发间隔。
         * 需要注意的是，调用 enableAudioVolumeEvaluation 开启音量回调后，无论频道内是否有人说话，都会按设置的时间间隔调用这个回调，
         * 如果没有人说话，则 userVolumes 为空，totalVolume 为0。
         * 
         * @event TRTCCallback#onUserVoiceVolume
         * @param {TRTCVolumeInfo} userVolumes      - 每位发言者的语音音量，取值范围0 - 100
         * @param {Number} userVolumesCount - 发言者的人数，即 userVolumes 数组的大小
         * @param {Number} totalVolume      - 总的语音音量, 取值范围0 - 100
         */
        this.rtcCloud.onEvent('onUserVoiceVolume', function (userVolumes, userVolumesCount, totalVolume) {
            fire('onUserVoiceVolume', userVolumes, userVolumesCount, totalVolume);
        });

        /**
         * 6.4 设备事件的回调
         *
         * @event TRTCCallback#onDeviceChange
         * @param {String} deviceId - 设备 ID
         * @param {Number} type     - 设备类型
         * @param {Number} state    - 事件类型
         */
        this.rtcCloud.onEvent('onDeviceChange', function (deviceId, type, state) {
            fire('onDeviceChange', deviceId, type, state);
        });

        /**
         * 6.5 麦克风测试音量回调
         * 
         * 麦克风测试接口 startMicDeviceTest 会触发这个回调
         *
         * @event TRTCCallback#onTestMicVolume
         * @param {Number} volume - 音量值，取值范围0 - 100
         */
        this.rtcCloud.onEvent('onTestMicVolume', function (volume) {
            fire('onTestMicVolume', volume);
        });

        /**
         * 6.6 扬声器测试音量回调
         * 
         * 扬声器测试接口 startSpeakerDeviceTest 会触发这个回调
         *
         * @event TRTCCallback#onTestSpeakerVolume
         * @param {Number} volume - 音量值，取值范围0 - 100
         */
        this.rtcCloud.onEvent('onTestSpeakerVolume', function (volume) {
            fire('onTestSpeakerVolume', volume);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （七）自定义消息的接收回调
        //
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 7.1 收到自定义消息回调
         * 
         * 当房间中的某个用户使用 sendCustomCmdMsg 发送自定义消息时，房间中的其它用户可以通过 onRecvCustomCmdMsg 接口接收消息
         *
         * @event TRTCCallback#onRecvCustomCmdMsg
         * @param {String} userId - 用户标识
         * @param {Number} cmdId  - 命令 ID
         * @param {Number} seq    - 消息序号
         * @param {String} msg    - 消息数据
         */
        this.rtcCloud.onEvent('onRecvCustomCmdMsg', function (userId, cmdId, seq, msg) {
            fire('onRecvCustomCmdMsg', userId, cmdId, seq, msg);
        });

        /**
         * 7.2 自定义消息丢失回调
         * 
         * 实时音视频使用 UDP 通道，即使设置了可靠传输（reliable）也无法确保100@%不丢失，只是丢消息概率极低，能满足常规可靠性要求。
         * 在发送端设置了可靠传输（reliable）后，SDK 都会通过此回调通知过去时间段内（通常为5s）传输途中丢失的自定义消息数量统计信息。
         *
         * 注意: 只有在发送端设置了可靠传输(reliable)，接收方才能收到消息的丢失回调
         * 
         * @event TRTCCallback#onMissCustomCmdMsg
         * @param {String} userId  - 用户标识
         * @param {Number} cmdId   - 命令 ID
         * @param {Number} errCode - 错误码
         * @param {Number} missed  - 丢失的消息数量
         */
        this.rtcCloud.onEvent('onMissCustomCmdMsg', function (userId, cmdId, errCode, missed) {
            fire('onMissCustomCmdMsg', userId, cmdId, errCode, missed);
        });

        /**
         * 7.3 收到 SEI 消息的回调
         * 
         * 当房间中的某个用户使用 sendSEIMsg 发送数据时，房间中的其它用户可以通过 onRecvSEIMsg 接口接收数据。
         * 
         * @event TRTCCallback#onRecvSEIMsg
         * @param {String} userId  - 用户标识
         * @param {String} message - 数据
         */
        this.rtcCloud.onEvent('onRecvSEIMsg', function (userId, message) {
            fire('onRecvSEIMsg', userId, message);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （八）CDN 旁路转推回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 8.1 旁路推流到 CDN 的回调
         * 
         * 对应于 TRTCCloud 的 startPublishCDNStream() 接口
         *
         * 注意: Start 回调如果成功，只能说明转推请求已经成功告知给腾讯云，如果目标 CDN 有异常，还是有可能会转推失败。
         *
         * @event TRTCCallback#onStartPublishCDNStream
         * @param {Number} errCode 错误码
         * @param {String} errMsg  错误详细信息
         */
        this.rtcCloud.onEvent('onStartPublishCDNStream', function (errCode, errMsg) {
            fire('onStartPublishCDNStream', errCode, errMsg);
        });

        /**
         * 8.2 停止旁路推流到 CDN 的回调
         * 
         * 对应于 TRTCCloud 中的 stopPublishCDNStream() 接口
         * 
         * @event TRTCCallback#onStopPublishCDNStream
         * @param {Number} errCode - 错误码
         * @param {String} errMsg  - 错误详细信息
         */
        this.rtcCloud.onEvent('onStopPublishCDNStream', function (errCode, errMsg) {
            fire('onStopPublishCDNStream', errCode, errMsg);
        });

        /**
         * 8.3 混流接口的状态回调
         *
         * 对应于 TRTCCloud 中的 setMixTranscodingConfig() 接口
         *
         * @event TRTCCallback#onSetMixTranscodingConfig
         * @param {Number} errCode - 错误码
         * @param {String} errMsg  - 错误详细信息
         */
        this.rtcCloud.onEvent('onSetMixTranscodingConfig', function (errCode, errMsg) {
            fire('onSetMixTranscodingConfig', errCode, errMsg);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （九）音效回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        //TODO:暂时未完成
        // | [onAudioEffectFinished] | 播放音效结束回调 |

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （十）屏幕分享回调
        //
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 10.1 当屏幕分享窗口被遮挡无法正常捕获时，SDK 会通过此回调通知，可在此回调里通知用户移开遮挡窗口
         * 
         * @event TRTCCallback#onScreenCaptureCovered
         */
        this.rtcCloud.onEvent('onScreenCaptureCovered', function () {
            fire('onScreenCaptureCovered');
        });

        /**
         * 10.2 当屏幕分享开始时，SDK 会通过此回调通知
         * 
         * @event TRTCCallback#onScreenCaptureStarted
         */
        this.rtcCloud.onEvent('onScreenCaptureStarted', function () {
            fire('onScreenCaptureStarted');
        });

        /**
         * 10.3 当屏幕分享暂停时，SDK 会通过此回调通知
         *
         * @event TRTCCallback#onScreenCapturePaused
         * @param {Number} reason - 停止原因，0：表示用户主动暂停；1：表示设置屏幕分享参数导致的暂停；2：表示屏幕分享窗口被最小化导致的暂停；3：表示屏幕分享窗口被隐藏导致的暂停
         */
        this.rtcCloud.onEvent('onScreenCapturePaused', function (reason) {
            fire('onScreenCapturePaused', reason);
        });

        /**
         * 10.4 当屏幕分享恢复时，SDK 会通过此回调通知
         *
         * @event TRTCCallback#onScreenCaptureResumed
         * @param {Number} reason - 停止原因，0：表示用户主动恢复，1：表示屏幕分享参数设置完毕后自动恢复；2：表示屏幕分享窗口从最小化被恢复；3：表示屏幕分享窗口从隐藏被恢复
         */
        this.rtcCloud.onEvent('onScreenCaptureResumed', function (reason) {
            fire('onScreenCaptureResumed', reason);
        });

        /**
         * 10.5 当屏幕分享停止时，SDK 会通过此回调通知
         *
         * @event TRTCCallback#onScreenCaptureStoped
         * @param {Number} reason - 停止原因，0：表示用户主动停止；1：表示屏幕分享窗口被关闭
         */
        this.rtcCloud.onEvent('onScreenCaptureStoped', function (reason) {
            fire('onScreenCaptureStoped', reason);
        });

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                      （十一）背景混音事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////
        /**
         * 11.1 开始播放背景音乐
         *
         * @event TRTCCallback#onPlayBGMBegin
         * @param {Number} errCode - 错误码
         */
        this.rtcCloud.onEvent('onPlayBGMBegin', function (errCode) {
            fire('onPlayBGMBegin', errCode);
        });

        /**
         * 11.2 播放背景音乐的进度
         *
         * @event TRTCCallback#onPlayBGMProgress
         * @param {Number} progressMS - 已播放时间
         * @param {Number} durationMS - 总时间
         */
        this.rtcCloud.onEvent('onPlayBGMProgress', function (progressMS, durationMS) {
            fire('onPlayBGMProgress', progressMS, durationMS);
        });

        /**
         * 11.3 播放背景音乐结束
         *
         * @event TRTCCallback#onPlayBGMComplete
         * @param {Number} errCode - 错误码
         */
        this.rtcCloud.onEvent('onPlayBGMComplete', function (errCode) {
            fire('onPlayBGMComplete', errCode);
        });
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （一）房间相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 1.1 进入房间
     * 
     * 如果加入成功，您会收到 onEnterRoom() 回调；如果失败，您会收到 onEnterRoom(result) 回调。
     * 跟进房失败相关的错误码，请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
     * 
     * 注意: 不管进房是否成功，enterRoom 都必须与 exitRoom 配对使用，在调用 exitRoom 前再次调用 enterRoom 函数会导致不可预期的错误问题。
     * 
     * @param {TRTCParams} params - 进房参数，具体参考{@link TRTCParams}
     * @param {Number} params.sdkAppId      - 应用标识（必填）
     * @param {String} params.userId        - 用户标识（必填）
     * @param {String} params.userSig       - 用户签名（必填）
     * @param {Number} params.roomId        - 房间号码（必填）
     * @param {TRTCRoleType} params.role    - 直播场景下的角色，默认值：主播
     * - TRTCRoleAnchor: 主播，可以上行视频和音频，一个房间里的主播人数不能超过 50 人。
     * - TRTCRoleAudience: 观众，只能观看，不能上行视频和音频，一个房间里的观众人数没有上限。
     * @param {String} params.privateMapKey - 房间签名（非必填）
     * @param {String} params.businessInfo  - 业务数据（非必填）
     * @param {TRTCAppScene} scene - 应用场景，目前支持视频通话（VideoCall）和在线直播（Live）两种场景
     * - TRTCAppSceneVideoCall: 视频通话场景，内部编码器和网络协议优化侧重流畅性，降低通话延迟和卡顿率。
     * - TRTCAppSceneLIVE: 在线直播场景，内部编码器和网络协议优化侧重性能和兼容性，性能和清晰度表现更佳。
     */
    enterRoom(params, scene) {
        if (params instanceof TRTCParams) {
            this.rtcCloud.enterRoom(params.sdkAppId, params.roomId, params.userId, params.userSig, params.privateMapKey, params.businessInfo, params.role, scene);
        } else {
            console.error("enterRoom, params is not instanceof TRTCParams!");
        }
    }

    /**
     * 1.2 离开房间
     * 
     * 调用 exitRoom() 接口会执行退出房间的相关逻辑，例如释放音视频设备资源和编解码器资源等。
     * 待资源释放完毕，SDK 会通过 TRTCCloudCallback 中的 onExitRoom() 回调通知您。
     * 
     * 如果您要再次调用 enterRoom() 或者切换到其他的音视频 SDK，请等待 onExitRoom() 回调到来后再执行相关操作。
     * 否则可能会遇到如摄像头、麦克风设备被强占等各种异常问题。
     */
    exitRoom() {
        let self = this;
        this.streams.forEach(function (value, key, map) {
            self._destroyRender(key, null);
        });
        this.rtcCloud.exitRoom();
    }

    /**
     * 1.3 切换角色，仅适用于直播场景（TRTCAppSceneLIVE）
     * 
     * 在直播场景下，一个用户可能需要在“观众”和“主播”之间来回切换。
     * 您可以在进房前通过 TRTCParams 中的 role 字段确定角色，也可以通过 switchRole 在进房后切换角色。
     *
     * @param {TRTCRoleType} role - 目标角色，默认为主播
     * - TRTCRoleAnchor: 主播，可以上行视频和音频，一个房间里的主播人数不能超过 50 人。
     * - TRTCRoleAudience: 观众，只能观看，不能上行视频和音频，一个房间里的观众人数没有上限。
     */
    switchRole(role) {
        this.rtcCloud.switchRole(role);
    }

    /**
     * 1.4 请求跨房通话（主播 PK）
     * 
     * TRTC 中两个不同音视频房间中的主播，可以通过“跨房通话”功能拉通连麦通话功能。使用此功能时，
     * 两个主播无需退出各自原来的直播间即可进行“连麦 PK”。
     * 
     * 例如：当房间“001”中的主播 A 通过 connectOtherRoom() 跟房间“002”中的主播 B 拉通跨房通话后，
     * 房间“001”中的用户都会收到主播 B 的 onUserEnter(B) 回调和 onUserVideoAvailable(B,true) 回调。
     * 房间“002”中的用户都会收到主播 A 的 onUserEnter(A) 回调和 onUserVideoAvailable(A,true) 回调。
     *
     * 简言之，跨房通话的本质，就是把两个不同房间中的主播相互分享，让每个房间里的观众都能看到两个主播。
     *
     * <pre>
     *                 房间 001                     房间 002
     *               -------------               ------------
     *  跨房通话前： | 主播 A      |             | 主播 B     |
     *              | 观众 U V W  |             | 观众 X Y Z |
     *               -------------               ------------
     *
     *                 房间 001                     房间 002
     *               -------------               ------------
     *  跨房通话后： | 主播 A B    |             | 主播 B A   |
     *              | 观众 U V W  |             | 观众 X Y Z |
     *               -------------               ------------
     * </pre>
     * 
     * 跨房通话的参数考虑到后续扩展字段的兼容性问题，暂时采用了 JSON 格式的参数，要求至少包含两个字段：
     * - roomId：房间“001”中的主播 A 要跟房间“002”中的主播 B 连麦，主播 A 调用 connectOtherRoom() 时 roomId 应指定为“002”。
     * - userId：房间“001”中的主播 A 要跟房间“002”中的主播 B 连麦，主播 A 调用 connectOtherRoom() 时 userId 应指定为 B 的 userId。
     *
     * 跨房通话的请求结果会通过 TRTCCloudCallback 中的 onConnectOtherRoom() 回调通知给您。
     *
     * @example
     * let json = JSON.stringify.({roomId: 2, userId: "userB"});
     * rtcCloud.connectOtherRoom(json);
     *
     * @param {String} params - JSON 字符串连麦参数，roomId 代表目标房间号，userId 代表目标用户 ID。
     *
     */
    connectOtherRoom(params) {
        this.rtcCloud.connectOtherRoom(params);
    }

    /**
     * 1.5 关闭跨房连麦
     * 
     * 跨房通话的退出结果会通过 TRTCCloudCallback 中的 onDisconnectOtherRoom() 回调通知给您。
     */
    disconnectOtherRoom() {
        this.rtcCloud.disconnectOtherRoom();
    }
    
    //TODO:暂时未完成
    // | [setDefaultStreamRecvMode] | 设置音视频数据接收模式（需要在进房前设置才能生效） |

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （二）视频相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 2.1 启动本地摄像头采集和预览
     * 
     * 这个接口会启动默认的摄像头，可以通过 setCurrentCameraDevice 接口选用其他摄像头
     * 当开始渲染首帧摄像头画面时，您会收到 TRTCCloudCallback 中的 onFirstVideoFrame(null) 回调。
     *
     * @param {HTMLElement} view - 承载预览画面的 DOM
     */
    startLocalPreview(view) {
        if (view !== undefined) {
            if (view !== null) {
                let key = this.getKey("local_video", TRTCVideoStreamType.TRTCVideoStreamTypeBig);
                this._initRender(key, view);
            }
            this.addLocalVideoRenderCallback();
            this.rtcCloud.startLocalPreview();
        } else {
            console.error("startLocalPreview, view is undefined!");
        }
    }

    /**
     * 2.2 停止本地视频采集及预览
     */
    stopLocalPreview() {
        let key = this.getKey("local_video", TRTCVideoStreamType.TRTCVideoStreamTypeBig);
        this._destroyRender(key, null);
        this.rtcCloud.stopLocalPreview();
    }

    /**
     * 2.3 是否屏蔽自己的视频画面
     * 
     * 当屏蔽本地视频后，房间里的其它成员将会收到 onUserVideoAvailable 回调通知
     * @param {Boolean} mute - true：屏蔽；false：开启，默认值：false
     */
    muteLocalVideo(mute) {
        this.rtcCloud.muteLocalVideo(mute);
    }

    /**
     * 2.4 开始显示远端视频画面
     * 
     * 在收到 SDK 的 onUserVideoAvailable(userId, true) 通知时，可以获知该远程用户开启了视频，
     * 此后调用 startRemoteView(userId) 接口加载该用户的远程画面时，可以用 loading 动画优化加载过程中的等待体验。
     * 待该用户的首帧画面开始显示时，您会收到 onFirstVideoFrame(userId) 事件回调。
     * 
     * @param {String}      userId - 对方的用户标识
     * @param {HTMLElement} view   - 承载预览画面的 DOM
     */
    startRemoteView(userId, view) {
        if (view !== undefined) {
            if (view !== null) {
                var key = this.getKey(userId, TRTCVideoStreamType.TRTCVideoStreamTypeBig);
                this._initRender(key, view);
            }
            this.addRemoteVideoRenderCallback(userId);
            this.rtcCloud.startRemoteView(userId);
        } else {
            console.error("startRemoteView, view is undefined!");
        }
    }

    /**
     * 2.5 停止显示远端视频画面
     * 
     * 调用此接口后，SDK 会停止接收该用户的远程视频流，同时会清理相关的视频显示资源。
     *
     * @param {String} userId - 对方的用户标识
     */
    stopRemoteView(userId) {
        var key = this.getKey(userId, TRTCVideoStreamType.TRTCVideoStreamTypeBig);
        this._destroyRender(key, null);
        this.rtcCloud.stopRemoteView(userId);
    }

    /**
     * 2.6 停止显示所有远端视频画面
     * 
     * 注意: 如果有屏幕分享的画面在显示，则屏幕分享的画面也会一并被关闭。
     * 
     */
    stopAllRemoteView() {
        let self = this;
        this.streams.forEach(function (value, key, map) {
            if (key.indexOf(String('local')) != 0) {
                self._destroyRender(key, null);
            }
        });
        this.rtcCloud.stopAllRemoteView();
    }

    /**
     * 2.7 暂停接收指定的远端视频流
     * 
     * 该接口仅停止接收远程用户的视频流，但并不释放显示资源，所以视频画面会冻屏在 mute 前的最后一帧。
     *
     * @param {String}  userId - 对方的用户标识
     * @param {Boolean} mute   - 是否停止接收
     */
    muteRemoteVideoStream(userId, mute) {
        this.rtcCloud.muteRemoteVideoStream(userId, mute);
    }

    /**
     * 2.8 停止接收所有远端视频流
     *
     * @param {Boolean} mute - 是否停止接收
     */
    muteAllRemoteVideoStreams(mute) {
        this.rtcCloud.muteAllRemoteVideoStreams(mute);
    }

    /**
     * 2.9 设置视频编码器相关参数
     * 
     * 该设置决定了远端用户看到的画面质量（同时也是云端录制出的视频文件的画面质量）
     *
     * @param {TRTCVideoEncParam} params - 视频编码参数，具体参考{@link TRTCVideoEncParam}
     * @param {TRTCVideoResolution} params.videoResolution - 视频分辨率
     * @param {TRTCVideoResolutionMode} params.resMode - 分辨率模式（横屏分辨率 - 竖屏分辨率）
     * - TRTCVideoResolutionModeLandscape: 横屏分辨率
     * - TRTCVideoResolutionModePortrait : 竖屏分辨率
     * @param {Number} params.videoFps     - 视频采集帧率
     * @param {Number} params.videoBitrate - 视频上行码率
     */
    setVideoEncoderParam(params) {
        if (params instanceof TRTCVideoEncParam) {
            this.rtcCloud.setVideoEncoderParam(params.videoResolution, params.resMode, params.videoFps, params.videoBitrate);
        } else {
            console.error("setVideoEncoderParam, params is not instanceof TRTCVideoEncParam!");
        }
    }

    /**
     * 2.10 设置网络流控相关参数
     * 
     * 该设置决定了 SDK 在各种网络环境下的调控策略（例如弱网下是“保清晰”还是“保流畅”）
     * 
     * @param {TRTCNetworkQosParam} params - 网络流控参数，具体参考{@link TRTCNetworkQosParam}
     * @param {TRTCVideoQosPreference} params.preference - 弱网下是“保清晰”还是“保流畅”
     * - TRTCVideoQosPreferenceSmooth: 弱网下保流畅，在遭遇弱网环境时首先确保声音的流畅和优先发送，画面会变得模糊且会有较多马赛克，但可以保持流畅不卡顿。
     * - TRTCVideoQosPreferenceClear : 弱网下保清晰，在遭遇弱网环境时，画面会尽可能保持清晰，但可能会更容易出现卡顿。
     * @param {TRTCQosControlMode} params.controlMode - 视频分辨率（云端控制 - 客户端控制）
     * - TRTCQosControlModeClient: 客户端控制（用于 SDK 开发内部调试，客户请勿使用）
     * - TRTCQosControlModeServer: 云端控制 （默认）
     */
    setNetworkQosParam(params) {
        if (params instanceof TRTCNetworkQosParam) {
            this.rtcCloud.setNetworkQosParam(params.preference, params.controlMode);
        } else {
            console.error("setNetworkQosParam, params is not instanceof TRTCNetworkQosParam!");
        }
    }

    /**
     * 2.11 设置本地图像的渲染模式
     *
     * @param {TRTCVideoFillMode} mode - 填充（画面可能会被拉伸裁剪）或适应（画面可能会有黑边），默认值：TRTCVideoFillMode_Fit
     * - TRTCVideoFillMode_Fill: 图像铺满屏幕，超出显示视窗的视频部分将被截掉，所以画面显示可能不完整。
     * - TRTCVideoFillMode_Fit: 图像长边填满屏幕，短边区域会被填充黑色，但画面的内容肯定是完整的。
     */
    setLocalViewFillMode(mode) {
        let key = this.getKey("local_video", TRTCVideoStreamType.TRTCVideoStreamTypeBig);
        if (this.streams.has(key)) {
            const renderer = this.streams.get(key);
            renderer.setContentMode(mode);
        } else {
            console.error('api setLocalViewFillMode failed. the local stream is not exsit.');
        }
    }

    /**
     * 2.12 设置远端图像的渲染模式
     *
     * @param {String} userId - 用户 ID
     * @param {TRTCVideoFillMode} mode - 填充（画面可能会被拉伸裁剪）或适应（画面可能会有黑边），默认值：TRTCVideoFillMode_Fit
     * - TRTCVideoFillMode_Fill: 图像铺满屏幕，超出显示视窗的视频部分将被截掉，所以画面显示可能不完整。
     * - TRTCVideoFillMode_Fit: 图像长边填满屏幕，短边区域会被填充黑色，但画面的内容肯定是完整的。
     */
    setRemoteViewFillMode(userId, mode) {
        let key = this.getKey(userId, TRTCVideoStreamType.TRTCVideoStreamTypeBig);
        if (this.streams.has(key)) {
            const renderer = this.streams.get(key);
            renderer.setContentMode(mode);
        } else {
            console.error('api setRemoteViewFillMode failed. the remote stream is not exsit.');
        }
    }

    /**
     * 2.13 设置本地图像的顺时针旋转角度
     *
     * @param {TRTCVideoRotation} rotation - 支持 TRTCVideoRotation90 、 TRTCVideoRotation180 、 TRTCVideoRotation270 旋转角度，默认值：TRTCVideoRotation0
     * - TRTCVideoRotation0  : 顺时针旋转0度
     * - TRTCVideoRotation90 : 顺时针旋转90度
     * - TRTCVideoRotation180: 顺时针旋转180度
     * - TRTCVideoRotation270: 顺时针旋转270度
     */
    setLocalViewRotation(rotation) {
        this.rtcCloud.setLocalViewRotation(rotation);
    }

    /**
     * 2.14 设置远端图像的顺时针旋转角度
     *
     * @param {String} userId - 用户 ID
     * @param {TRTCVideoRotation} rotation - 支持 TRTCVideoRotation90 、 TRTCVideoRotation180 、 TRTCVideoRotation270 旋转角度，默认值：TRTCVideoRotation0
     * - TRTCVideoRotation0  : 顺时针旋转0度
     * - TRTCVideoRotation90 : 顺时针旋转90度
     * - TRTCVideoRotation180: 顺时针旋转180度
     * - TRTCVideoRotation270: 顺时针旋转270度
     */
    setRemoteViewRotation(userId, rotation) {
        this.rtcCloud.setRemoteViewRotation(userId, rotation);
    }

    /**
     * 2.15 设置视频编码输出的（也就是远端用户观看到的，以及服务器录制下来的）画面方向
     *
     * @param {TRTCVideoRotation} rotation - 目前支持 TRTCVideoRotation0 和 TRTCVideoRotation180 两个旋转角度，默认值：TRTCVideoRotation0
     * - TRTCVideoRotation0  : 顺时针旋转0度
     * - TRTCVideoRotation180: 顺时针旋转180度
     */
    setVideoEncoderRotation(rotation) {
        this.rtcCloud.setVideoEncoderRotation(rotation);
    }

    /**
     * 2.16 设置本地摄像头预览画面的镜像模式
     *
     * @param {Boolean} mirror - 镜像模式, windows 默认值: false(非镜像模式), mac 默认值: true(镜像模式)
     */
    setLocalViewMirror(mirror) {
        this.rtcCloud.setLocalViewMirror(mirror);
    }

    /**
     * 2.17 设置编码器输出的画面镜像模式
     * 
     * 该接口不改变本地摄像头的预览画面，但会改变另一端用户看到的（以及服务器录制的）画面效果。
     *
     * @param {Boolean} mirror - 是否开启远端镜像, true：远端画面镜像；false：远端画面非镜像。默认值：false
     */
    setVideoEncoderMirror(mirror) {
        this.rtcCloud.setVideoEncoderMirror(mirror);
    }

    /**
     * 2.18 开启大小画面双路编码模式
     * 
     * 如果当前用户是房间中的主要角色（例如主播、老师、主持人等），并且使用 PC 或者 Mac 环境，可以开启该模式。
     * 开启该模式后，当前用户会同时输出【高清】和【低清】两路视频流（但只有一路音频流）。
     * 对于开启该模式的当前用户，会占用更多的网络带宽，并且会更加消耗 CPU 计算资源。
     * 
     * 对于同一房间的远程观众而言：
     * - 如果用户的下行网络很好，可以选择观看【高清】画面
     * - 如果用户的下行网络较差，可以选择观看【低清】画面
     * 
     * @param {Boolean} enable - 是否开启小画面编码，默认值：false
     * @param {TRTCVideoEncParam} params - 小流的视频参数，具体参考{@link TRTCVideoEncParam}
     * @param {TRTCVideoResolution} params.videoResolution - 视频分辨率
     * @param {TRTCVideoResolutionMode} params.resMode - 分辨率模式（横屏分辨率 - 竖屏分辨率）
     * - TRTCVideoResolutionModeLandscape: 横屏分辨率
     * - TRTCVideoResolutionModePortrait : 竖屏分辨率
     * @param {Number} params.videoFps     - 视频采集帧率
     * @param {Number} params.videoBitrate - 视频上行码率
     */
    enableSmallVideoStream(enable, params) {
        if (params instanceof TRTCVideoEncParam) {
            this.rtcCloud.enableSmallVideoStream(enable, params.videoResolution, params.resMode, params.videoFps, params.videoBitrate);
        } else {
            console.error("enableSmallVideoStream, params is not instanceof TRTCVideoEncParam!");
        }
    }

    /**
     * 2.19 选定观看指定 userId 的大画面还是小画面
     * 
     * 此功能需要该 userId 通过 enableEncSmallVideoStream 提前开启双路编码模式。
     * 如果该 userId 没有开启双路编码模式，则此操作无效。
     * 
     * @param {String} userId - 用户 ID
     * @param {TRTCVideoStreamType} type - 视频流类型，即选择看大画面还是小画面，默认为 TRTCVideoStreamTypeBig
     * - TRTCVideoStreamTypeBig  : 大画面视频流
     * - TRTCVideoStreamTypeSmall: 小画面视频流
     */
    setRemoteVideoStreamType(userId, type) {
        this.rtcCloud.setRemoteVideoStreamType(userId, type);
    }

    /**
     * 2.20 设定观看方优先选择的视频质量
     * 
     * 低端设备推荐优先选择低清晰度的小画面。
     * 如果对方没有开启双路视频模式，则此操作无效。
     * 
     * @param {TRTCVideoStreamType} type - 默认观看大画面还是小画面，默认为 TRTCVideoStreamTypeBig
     * - TRTCVideoStreamTypeBig  : 大画面视频流
     * - TRTCVideoStreamTypeSmall: 小画面视频流
     */
    setPriorRemoteVideoStreamType(type) {
        this.playStreamType = type;
        this.rtcCloud.setPriorRemoteVideoStreamType(type);
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （三）音频相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 3.1 开启本地音频的采集和上行
     * 
     * 该函数会启动麦克风采集，并将音频数据传输给房间里的其他用户。
     * SDK 并不会默认开启本地的音频上行，也就说，如果您不调用这个函数，房间里的其他用户就听不到您的声音。
     * 
     * 注意: TRTC SDK 并不会默认打开本地的麦克风采集。
     */
    startLocalAudio() {
        this.rtcCloud.startLocalAudio();
    }

    /**
     * 3.2 关闭本地音频的采集和上行
     * 
     * 当关闭本地音频的采集和上行，房间里的其它成员会收到 onUserAudioAvailable(false) 回调通知。
     */
    stopLocalAudio() {
        this.rtcCloud.stopLocalAudio();
    }

    /**
     * 3.3 静音本地的音频
     * 
     * 当静音本地音频后，房间里的其它成员会收到 onUserAudioAvailable(false) 回调通知。
     * 与 stopLocalAudio 不同之处在于，muteLocalAudio 并不会停止发送音视频数据，而是会继续发送码率极低的静音包。
     * 在对录制质量要求很高的场景中，选择 muteLocalAudio 是更好的选择，能录制出兼容性更好的 MP4 文件。
     * 这是由于 MP4 等视频文件格式，对于音频的连续性是要求很高的，简单粗暴地 stopLocalAudio 会导致录制出的 MP4 不易播放。
     *
     * @param {Boolean} mute - true：屏蔽；false：开启，默认值：false
     */
    muteLocalAudio(mute) {
        this.rtcCloud.muteLocalAudio(mute);
    }

    /**
     * 3.4 静音掉某一个用户的声音
     *
     * @param {String}  userId - 用户 ID
     * @param {Boolean} mute   - true：静音；false：非静音
     */
    muteRemoteAudio(userId, mute) {
        this.rtcCloud.muteRemoteAudio(userId, mute);
    }

    /**
     * 3.5 静音掉所有用户的声音
     *
     * @param {Boolean} mute - true：静音；false：非静音
     */
    muteAllRemoteAudio(mute) {
        this.rtcCloud.muteAllRemoteAudio(mute);
    }

    /**
     * 3.6 启用或关闭音量大小提示
     * 
     * 开启此功能后，SDK 会在 onUserVoiceVolume() 中反馈对每一路声音音量大小值的评估。
     * 我们在 Demo 中有一个音量大小的提示条，就是基于这个接口实现的。
     * 如希望打开此功能，请在 startLocalAudio() 之前调用。
     *
     * @param {Number} interval - 设置 onUserVoiceVolume 回调的触发间隔，单位为ms，最小间隔为100ms，如果小于等于0则会关闭回调，建议设置为300ms
     */
    enableAudioVolumeEvaluation(interval) {
        this.rtcCloud.enableAudioVolumeEvaluation(interval);
    }

    /**
     * 3.7 开始录音
     * 
     * 该方法调用后， SDK 会将通话过程中的所有音频(包括本地音频，远端音频，BGM等)录制到一个文件里。
     * 无论是否进房，调用该接口都生效。
     * 如果调用 exitRoom 时还在录音，录音会自动停止。
     * 
     * 注意: 录音文件路径需精确到文件名及格式后缀，格式后缀决定录制文件的格式。
     * 例如：指定路径为 path/to/audio.aac，则会生成一个 AAC 格式的文件。目前支持的格式有 PCM, WAV, AAC
     * 
     * @param {String} path - 录音文件路径（必填），录音文件的保存路径，该路径需要用户自行指定，请确保路径存在且可写。
     * @return {Number} 0：成功；-1：录音已开始；-2：文件或目录创建失败；-3：后缀指定的音频格式不支持
     */
    startAudioRecording(path) {
        this.rtcCloud.startAudioRecording(path);
    }

    /**
     * 3.8 停止录音
     * 
     * 如果调用 exitRoom 时还在录音，录音会自动停止。
     */
    stopAudioRecording() {
        this.rtcCloud.stopAudioRecording();
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （四）摄像头相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 4.1 获取摄像头设备列表
     *
     * @example
     * var cameralist = this.rtcCloud.getCameraDevicesList();
     * for (i=0;i<cameralist.length;i++) {
     *    var camera = cameralist[i];
     *    console.info("camera deviceName: " + camera.deviceName + " deviceId:" + camera.deviceId);
     * }
     * @return {TRTCDeviceInfo[]} 摄像头管理器列表，具体参考{@link TRTCDeviceInfo}
     */
    getCameraDevicesList() {
        return this.rtcCloud.getCameraDevicesList();
    }

    /**
     * 4.2 设置要使用的摄像头
     *
     * @param {String} deviceId - 从 getCameraDevicesList 中得到的设备 ID
     */
    setCurrentCameraDevice(deviceId) {
        this.rtcCloud.setCurrentCameraDevice(deviceId);
    }

    /**
     * 4.3 获取当前使用的摄像头
     *
     * @return {TRTCDeviceInfo} 设备信息，能获取设备 ID 和设备名称，具体参考{@link TRTCDeviceInfo}
     */
    getCurrentCameraDevice() {
        return this.rtcCloud.getCurrentCameraDevice();
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （五）音频设备相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 5.1 获取麦克风设备列表
     * 
     * @example
     *   var miclist = this.rtcCloud.getMicDevicesList();
     *   for (i=0;i<miclist.length;i++) {
     *     var mic = miclist[i];
     *     console.info("mic deviceName: " + mic.deviceName + " deviceId:" + mic.deviceId);
     *   }
     * @return {TRTCDeviceInfo[]} 麦克风管理器列表，具体参考{@link TRTCDeviceInfo}
     */
    getMicDevicesList() {
        return this.rtcCloud.getMicDevicesList();
    }

    /**
     * 5.2 设置要使用的麦克风
     * 
     * 选择指定的麦克风作为录音设备，不调用该接口时，默认选择索引为0的麦克风
     *
     * @param {String} micId - 从 getMicDevicesList 中得到的设备 ID
     */
    setCurrentMicDevice(micId) {
        this.rtcCloud.setCurrentMicDevice(micId);
    }

    /**
     * 5.3 获取当前选择的麦克风
     *
     * @return {TRTCDeviceInfo} 设备信息，能获取设备 ID 和设备名称，具体参考{@link TRTCDeviceInfo}
     */
    getCurrentMicDevice() {
        return this.rtcCloud.getCurrentMicDevice();
    }

    /**
     * 5.4 获取系统当前麦克风设备音量
     *
     * @return {Number} 音量值，范围是0 - 100
     */
    getCurrentMicDeviceVolume() {
        return this.rtcCloud.getCurrentMicDeviceVolume();
    }

    /**
     * 5.5 设置系统当前麦克风设备的音量
     *
     * @param {Number} volume - 麦克风音量值，范围0 - 100
     */
    setCurrentMicDeviceVolume(volume) {
        this.rtcCloud.setCurrentMicDeviceVolume(volume);
    }

    /**
     * 5.6 获取扬声器设备列表
     *
     * @example
     *   var speakerlist = this.rtcCloud.getSpeakerDevicesList();
     *   for (i=0;i<speakerlist.length;i++) {
     *     var speaker = speakerlist[i];
     *     console.info("mic deviceName: " + speaker.deviceName + " deviceId:" + speaker.deviceId);
     *   }
     * @return {TRTCDeviceInfo[]} 扬声器管理器列表，具体参考{@link TRTCDeviceInfo}
     */
    getSpeakerDevicesList() {
        return this.rtcCloud.getSpeakerDevicesList();
    }

    /**
     * 5.7 设置要使用的扬声器
     *
     * @param {String} speakerId - 从 getSpeakerDevicesList 中得到的设备 ID
     */
    setCurrentSpeakerDevice(speakerId) {
        this.rtcCloud.setCurrentSpeakerDevice(speakerId);
    }

    /**
     * 5.8 获取当前的扬声器设备
     *
     * @return {TRTCDeviceInfo} 设备信息，能获取设备 ID 和设备名称，具体参考{@link TRTCDeviceInfo}
     */
    getCurrentSpeakerDevice() {
        return this.rtcCloud.getCurrentSpeakerDevice();
    }

    /**
     * 5.9 当前扬声器设备音量
     * 
     * 注意:
     * - SDK6.7 及以下版本，windows 查询的不是系统扬声器的音量大小，mac 是查询系统扬声器音量大小
     * - SDK6.8 及以上版本，windows 和 mac 查询的都是系统扬声器的音量大小
     * 
     * @return {Number} 扬声器音量，范围0 - 100
     */
    getCurrentSpeakerVolume() {
        return this.rtcCloud.getCurrentSpeakerVolume();
    }

    /**
     * 5.10 设置当前扬声器音量
     * 
     * 注意:
     * - SDK6.7 及以下版本，windows 设置的不是系统扬声器的音量大小，mac 是设置系统扬声器音量大小
     * - SDK6.8 及以上版本，windows 和 mac 设置的都是系统扬声器的音量大小
     * 
     * @param {Number} volume - 设置的扬声器音量，范围0 - 100
     */
    setCurrentSpeakerVolume(volume) {
        this.rtcCloud.setCurrentSpeakerVolume(volume);
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （六）图像前处理相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 6.1 设置美颜、美白、红润效果级别
     * 
     * SDK 内部集成了两套风格不同的磨皮算法，一套我们取名叫“光滑”，适用于美女秀场，效果比较明显。
     * 另一套我们取名“自然”，磨皮算法更多地保留了面部细节，主观感受上会更加自然。
     * 
     * @param {TRTCBeautyStyle} style - 美颜风格，光滑或者自然，光滑风格磨皮更加明显，适合娱乐场景。
     * - TRTCBeautyStyleSmooth: 光滑，适用于美女秀场，效果比较明显。
     * - TRTCBeautyStyleNature: 自然，磨皮算法更多地保留了面部细节，主观感受上会更加自然。
     * @param {Number} beauty    - 美颜级别，取值范围0 - 9，0表示关闭，1 - 9值越大，效果越明显
     * @param {Number} white     - 美白级别，取值范围0 - 9，0表示关闭，1 - 9值越大，效果越明显
     * @param {Number} ruddiness - 红润级别，取值范围0 - 9，0表示关闭，1 - 9值越大，效果越明显，该参数 windows 平台暂未生效
     */
    setBeautyStyle(style, beauty, white, ruddiness) {
        this.rtcCloud.setBeautyStyle(style, beauty, white, ruddiness);
    }

    /**
     * 6.2 设置水印
     * 
     * 水印的位置是通过 xOffset, yOffset, fWidthRatio 来指定的。
     * - xOffset：水印的坐标，取值范围为0 - 1的浮点数。
     * - yOffset：水印的坐标，取值范围为0 - 1的浮点数。
     * - fWidthRatio：水印的大小比例，取值范围为0 - 1的浮点数。
     * 
     * 注意: 大小流暂未支持，mac 平台必须使用透明底的 png 格式
     * 
     * @param {TRTCVideoStreamType} streamType - 要设置水印的流类型(TRTCVideoStreamTypeBig、TRTCVideoStreamTypeSub)
     * @param {ArrayBuffer} srcData - 水印图片源数据（传 null 表示去掉水印）
     * @param {TRTCWaterMarkSrcType} srcType - 水印图片源数据类型
     * - TRTCWaterMarkSrcTypeFile  : 图片文件路径，支持 BMP、GIF、JPEG、PNG、TIFF、Exif、WMF 和 EMF 文件格式
     * - TRTCWaterMarkSrcTypeBGRA32: BGRA32格式内存块
     * - TRTCWaterMarkSrcTypeRGBA32: RGBA32格式内存块
     * @param {Number} nWidth      - 水印图片像素宽度（源数据为文件路径时忽略该参数）
     * @param {Number} nHeight     - 水印图片像素高度（源数据为文件路径时忽略该参数）
     * @param {Number} xOffset     - 水印显示的左上角 x 轴偏移
     * @param {Number} yOffset     - 水印显示的左上角 y 轴偏移
     * @param {Number} fWidthRatio - 水印显示的宽度占画面宽度比例（水印按该参数等比例缩放显示）
     */
    setWaterMark(streamType, srcData, srcType, nWidth, nHeight, xOffset, yOffset, fWidthRatio) {
        if (srcData === null || srcData === undefined || srcType === null || srcType === undefined) {
            srcType = -1; //关闭水印
        }
        this.rtcCloud.setWaterMark(streamType, srcData, srcType, nWidth, nHeight, xOffset, yOffset, fWidthRatio);
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （七）辅流相关接口函数（屏幕共享，播片等）
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 7.1 开始渲染远端用户辅流画面
     * 
     * 对应于 startRemoteView() 用于显示主画面，该接口只能用于显示辅路（屏幕分享、远程播片）画面。
     * 
     * 注意: 请在 onUserSubStreamAvailable 回调后再调用这个接口。
     * 
     * @param {String}      userId - 对方的用户标识
     * @param {HTMLElement} view   - 承载预览画面的 DOM
     */
    startRemoteSubStreamView(userId, view) {
        if (view !== undefined) {
            if (view !== null) {
                var key = this.getKey(userId, TRTCVideoStreamType.TRTCVideoStreamTypeSub);
                this._initRender(key, view);
            }
            this.addRemoteVideoRenderCallback(userId);
            this.rtcCloud.startRemoteSubStreamView(userId);
        } else {
            console.error("startRemoteSubStreamView, view is undefined!");
        }
    }

    /**
     * 7.2 停止显示远端用户的屏幕分享画面。
     * 
     * @param {String} userId - 对方的用户标识
     */
    stopRemoteSubStreamView(userId) {
        if (userId !== undefined && userId !== null) {
            var key = this.getKey(userId, TRTCVideoStreamType.TRTCVideoStreamTypeSub);
            this._destroyRender(key, null);
            this.rtcCloud.stopRemoteSubStreamView(userId);
        } else {
            console.error("stopRemoteSubStreamView, userId is error!");
        }
    }

    /**
     * 7.3 设置辅流画面的渲染模式
     * 
     * 对应于 setRemoteViewFillMode() 于设置远端的主路画面，该接口用于设置远端的辅路（屏幕分享、远程播片）画面。
     *
     * @param {String} userId - 用户的 ID
     * @param {TRTCVideoFillMode} mode - 填充（画面可能会被拉伸裁剪）或适应（画面可能会有黑边），默认值：TRTCVideoFillMode_Fit
     * - TRTCVideoFillMode_Fill: 图像铺满屏幕，超出显示视窗的视频部分将被截掉，所以画面显示可能不完整。
     * - TRTCVideoFillMode_Fit: 图像长边填满屏幕，短边区域会被填充黑色，但画面的内容肯定是完整的。
     */
    setRemoteSubStreamViewFillMode(userId, mode) {
        let key = this.getKey(userId, TRTCVideoStreamType.TRTCVideoStreamTypeSub);
        if (this.streams.has(key)) {
            const renderer = this.streams.get(key);
            renderer.setContentMode(mode);
        } else {
            console.error('api setRemoteSubStreamViewFillMode failed. the remote sub stream is not exsit.');
        }
    }

    /**
     * 7.4 枚举可共享的窗口列表，
     * 
     * 如果您要给您的 App 增加屏幕分享功能，一般需要先显示一个窗口选择界面，这样用户可以选择希望分享的窗口。
     * 通过如下函数，您可以获得可分享窗口的 ID、类型、窗口名称以及缩略图。
     * 拿到这些信息后，您就可以实现一个窗口选择界面，当然，您也可以使用我们在 Demo 源码中已经实现好的一个界面。
     * 
     * 注意: 返回的列表中包括屏幕和应用窗口，屏幕会在列表的前面几个元素中。
     * 
     * @param {Number} thumbWidth  - 缩略图宽度，指定要获取的窗口缩略图大小，缩略图可用于绘制在窗口选择界面上
     * @param {Number} thumbHeight - 缩略图高度，指定要获取的窗口缩略图大小，缩略图可用于绘制在窗口选择界面上
     * @param {Number} iconWidth   - 图标宽度，指定要获取的窗口图标大小
     * @param {Number} iconHeight  - 图标高度，指定要获取的窗口图标大小
     *
     * @return {TRTCScreenCaptureSourceInfo[]} 窗口列表包括屏幕，具体参考{@link TRTCScreenCaptureSourceInfo}
     */
    getScreenCaptureSources(thumbWidth, thumbHeight, iconWidth, iconHeight) {
        return this.rtcCloud.getScreenCaptureSources(thumbWidth, thumbHeight, iconWidth, iconHeight);
    }

    /**
     * 7.5 设置屏幕共享参数，该方法在屏幕共享过程中也可以调用
     * 
     * 如果您期望在屏幕分享的过程中，切换想要分享的窗口，可以再次调用这个函数而不需要重新开启屏幕分享。
     * 
     * 支持如下四种情况：
     * - 共享整个屏幕：sourceInfoList 中 type 为 Screen 的 source，captureRect 设为 { 0, 0, 0, 0 }
     * - 共享指定区域：sourceInfoList 中 type 为 Screen 的 source，captureRect 设为非 NULL，比如 { 100, 100, 300, 300 }
     * - 共享整个窗口：sourceInfoList 中 type 为 Window 的 source，captureRect 设为 { 0, 0, 0, 0 }
     * - 共享窗口区域：sourceInfoList 中 type 为 Window 的 source，captureRect 设为非 NULL，比如 { 100, 100, 300, 300 }
     * 
     * @param {TRTCScreenCaptureSourceType} type - 采集源类型
     * - TRTCScreenCaptureSourceTypeWindow: 该分享目标是某一个Windows窗口
     * - TRTCScreenCaptureSourceTypeScreen: 该分享目标是整个Windows桌面
     * - TRTCScreenCaptureSourceTypeCustom: 该分享目标是自定义窗口区域
     * @param {String} sourceId    - 采集源ID，对于窗口，该字段指示窗口句柄；对于屏幕，该字段指示屏幕ID
     * @param {String} sourcename  - 采集源名称，UTF8编码
     * @param {Rect} captureRect - 指定捕获的区域
     * @param {Number} captureRect.left   - 指定捕获的区域的左坐标
     * @param {Number} captureRect.top    - 指定捕获的区域的上坐标
     * @param {Number} captureRect.right  - 指定捕获的区域的右坐标
     * @param {Number} captureRect.bottom - 指定捕获的区域的下坐标
     * @param {Boolean} captureMouse    - 指定是否捕获鼠标指针
     * @param {Boolean} highlightWindow - 指定是否高亮正在共享的窗口，以及当捕获图像被遮挡时高亮遮挡窗口提示用户移走遮挡
     */
    selectScreenCaptureTarget(type, sourceId, sourcename, captureRect, captureMouse, highlightWindow) {
        if (sourceId !== null && sourceId !== undefined && sourcename !== null && sourcename !== undefined) {
            return this.rtcCloud.selectScreenCaptureTarget(type, sourceId, sourcename, captureRect.left, captureRect.top,
                captureRect.right, captureRect.bottom, captureMouse, highlightWindow);
        } else {
            console.error("selectScreenCaptureTarget, sourceId or sourcename is undefined!");
        }
    }

    /**
     * 7.6 启动屏幕分享（暂不支持 mac 平台预览界面）
     * 
     * @param {HTMLElement} view - 承载预览画面的 DOM
     */
    startScreenCapture(view = null) {
        if (process.platform == 'darwin') {
            this.rtcCloud.startScreenCapture();
        } else {
            if (view !== undefined) {
                if (view !== null) {
                    var key = this.getKey("local_video", TRTCVideoStreamType.TRTCVideoStreamTypeSub);
                    this._initRender(key, view);
                }
                this.addLocalVideoRenderCallback();
                this.rtcCloud.startScreenCapture();
            } else {
                this.rtcCloud.startScreenCapture();
            }
        }
    }

    /**
     * 7.7 暂停屏幕分享
     */
    pauseScreenCapture() {
        this.rtcCloud.pauseScreenCapture();
    }

    /**
     * 7.8 恢复屏幕分享
     */
    resumeScreenCapture() {
        this.rtcCloud.resumeScreenCapture();
    }

    /**
     * 7.9 停止屏幕采集
     */
    stopScreenCapture() {
        var key = this.getKey("local_video", TRTCVideoStreamType.TRTCVideoStreamTypeSub);
        this._destroyRender(key, null);
        this.rtcCloud.stopScreenCapture();
    }

    /**
     * 7.10 设置屏幕分享的编码器参数
     * 
     * 对应于 setVideoEncoderParam() 设置主路画面的编码质量，该函数仅用于设置辅路（屏幕分享、远程播片）的编码参数。
     * 该设置决定了远端用户看到的画面质量，同时也是云端录制出的视频文件的画面质量。
     * 
     * @param {TRTCVideoEncParam} params - 辅流编码参数
     * @param {TRTCVideoResolution} params.videoResolution - 视频分辨率
     * @param {TRTCVideoResolutionMode} params.resMode - 分辨率模式（横屏分辨率 - 竖屏分辨率）
     * - TRTCVideoResolutionModeLandscape: 横屏分辨率
     * - TRTCVideoResolutionModePortrait : 竖屏分辨率
     * @param {Number} params.videoFps     - 视频采集帧率
     * @param {Number} params.videoBitrate - 视频上行码率
     */
    setSubStreamEncoderParam(params) {
        if (params instanceof TRTCVideoEncParam) {
            this.rtcCloud.setSubStreamEncoderParam(params.videoResolution, params.resMode, params.videoFps, params.videoBitrate);
        } else {
            console.error("setSubStreamEncoderParam, params is not instanceof TRTCVideoEncParam!");
        }
    }

    /**
     * 7.11 设置辅流的混音音量大小
     * 
     * 这个数值越高，辅路音量的占比就约高，麦克风音量占比就越小，所以不推荐设置得太大，否则麦克风的声音就被压制了。
     *
     * @param {Number} volume - 设置的混音音量大小，范围0 - 100
     */
    setSubStreamMixVolume(volume) {
        this.rtcCloud.setSubStreamMixVolume(volume);
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （八）自定义采集和渲染
    //
    /////////////////////////////////////////////////////////////////////////////////
    //TODO:暂时未完成
    // | [enableCustomVideoCapture] | 启用视频自定义采集模式 |
    // | [sendCustomVideoData] | 向 SDK 投送自己采集的视频数据 |
    // | [enableCustomAudioCapture] | 向 SDK 投送自己采集的音频数据 |
    // | [setAudioFrameCallback] | 设置音频数据回调 |

    /**
    * 8.5 设置本地视频自定义渲染
    *
    * 注意: 设置此方法，SDK 内部会把采集到的数据回调出来，SDK 跳过 HWND 渲染逻辑
            调用 setLocalVideoRenderCallback(TRTCVideoPixelFormat_Unknown, TRTCVideoBufferType_Unknown, nullptr) 停止回调
    * @param {TRTCVideoPixelFormat} pixelFormat 指定回调的像素格式
    * @param {TRTCVideoBufferType}  bufferType  指定视频数据结构类型
    * @param {Fuction} callback    自定义渲染回调
    * @return {Boolean} 0：成功；<0：错误
    */
    setLocalVideoRenderCallback(pixelFormat, bufferType, callback) {
        if (pixelFormat !== undefined && bufferType !== undefined && callback !== undefined) {
            if (callback !== null) {
                this.localVideoCallback = new VideoRenderCallback();
                this.localVideoCallback.callback = callback;
                this.localVideoCallback.pixelFormat = pixelFormat;
            } else {
                this.localVideoCallback = null;
            }
        } else {
            console.error("setLocalVideoRenderCallback, param is error!");
        }
    }

    /**
    * 8.6 设置远端视频自定义渲染
    * 
    * 此方法同 setLocalVideoRenderDelegate，区别在于一个是本地画面的渲染回调， 一个是远程画面的渲染回调。
    * 
    * 注意: 设置此方法，SDK 内部会把远端的数据解码后回调出来，SDK 跳过 HWND 渲染逻辑
            调用 setRemoteVideoRenderCallback(userId, TRTCVideoPixelFormat_Unknown, TRTCVideoBufferType_Unknown, null) 停止回调。
    * @param {String}               userId      用户标识
    * @param {TRTCVideoPixelFormat} pixelFormat 指定回调的像素格式
    * @param {TRTCVideoBufferType}  bufferType  指定视频数据结构类型
    * @param {Fuction} callback    自定义渲染回调
    * @return {Boolean} 0：成功；<0：错误
    */
    setRemoteVideoRenderCallback(userId, pixelFormat, bufferType, callback) {
        if (userId !== undefined && userId !== null
            && pixelFormat !== undefined && bufferType !== undefined && callback !== undefined) {
            if (callback !== null) {
                let remotecallback = new VideoRenderCallback();
                remotecallback.callback = callback;
                remotecallback.pixelFormat = pixelFormat;
                this.remoteVideoCallback.set(String(userId), remotecallback);
            } else {
                this.remoteVideoCallback.delete(String(userId));
            }
        } else {
            console.error("setRemoteVideoRenderCallback, userId is error!");
        }
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （九）自定义消息发送
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 9.1 发送自定义消息给房间内所有用户
     * 
     * 该接口可以借助音视频数据通道向当前房间里的其他用户广播您自定义的数据，但因为复用了音视频数据通道，
     * 请务必严格控制自定义消息的发送频率和消息体的大小，否则会影响音视频数据的质量控制逻辑，造成不确定性的问题。
     * 
     * 注意: 本接口有以下限制：
     *       - 发送消息到房间内所有用户，每秒最多能发送30条消息。
     *       - 每个包最大为1KB，超过则很有可能会被中间路由器或者服务器丢弃。
     *       - 每个客户端每秒最多能发送总计8KB数据。
     *       - 将 reliable 和 ordered 同时设置为 true 或 false，暂不支持交叉设置。
     *       - 强烈建议不同类型的消息使用不同的 cmdID，这样可以在要求有序的情况下减小消息时延。
     * 
     * @param {Number}  cmdId    - 消息 ID，取值范围为1 - 10
     * @param {String}  msg      - 待发送的消息，最大支持1KB（1000字节）的数据大小
     * @param {Boolean} reliable - 是否可靠发送，可靠发送的代价是会引入一定的延时，因为接收端要暂存一段时间的数据来等待重传
     * @param {Boolean} ordered  - 是否要求有序，即是否要求接收端接收的数据顺序和发送端发送的顺序一致，这会带来一定的接收延时，因为在接收端需要暂存并排序这些消息
     * @return {Boolean} true：消息已经发出；false：消息发送失败
     */
    sendCustomCmdMsg(cmdId, msg, reliable, ordered) {
        this.rtcCloud.sendCustomCmdMsg(cmdId, msg, reliable, ordered);
    }

    /**
     * 9.2 将小数据量的自定义数据嵌入视频帧中
     * 
     * 跟 sendCustomCmdMsg 的原理不同，sendSEIMsg 是将数据直接塞入视频数据头中。因此，即使视频帧被旁路到了直播 CDN 上，
     * 这些数据也会一直存在。但是由于要把数据嵌入视频帧中，所以数据本身不能太大，推荐几个字节就好。
     * 
     * 最常见的用法是把自定义的时间戳（timstamp）用 sendSEIMsg 嵌入视频帧中，这种方案的最大好处就是可以实现消息和画面的完美对齐。
     * 
     * 注意: 本接口有以下限制：
     *       - 数据在接口调用完后不会被即时发送出去，而是从下一帧视频帧开始带在视频帧中发送。
     *       - 发送消息到房间内所有用户，每秒最多能发送30条消息（与 sendCustomCmdMsg 共享限制）。
     *       - 每个包最大为1KB，若发送大量数据，会导致视频码率增大，可能导致视频画质下降甚至卡顿（与 sendCustomCmdMsg 共享限制）。
     *       - 每个客户端每秒最多能发送总计8KB数据（与 sendCustomCmdMsg 共享限制）。
     *       - 若指定多次发送（repeatCount>1），则数据会被带在后续的连续 repeatCount 个视频帧中发送出去，同样会导致视频码率增大。
     *       - 如果 repeatCount>1，多次发送，接收消息 onRecvSEIMsg 回调也可能会收到多次相同的消息，需要去重。
     * 
     * @param {String} msg         - 待发送的数据，最大支持1kb（1000字节）的数据大小
     * @param {Number} repeatCount - 发送数据次数
     * @return {Boolean} true：消息已通过限制，等待后续视频帧发送；false:消息被限制发送
     */
    sendSEIMsg(msg, repeatCount) {
        this.rtcCloud.sendSEIMsg(msg, repeatCount);
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （十）背景混音相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 10.1 启动播放背景音乐
     *
     * @param {String} path - 音乐文件路径
     */
    playBGM(path) {
        this.rtcCloud.playBGM(path);
    }

    /**
     * 10.2 停止播放背景音乐
     */
    stopBGM() {
        this.rtcCloud.stopBGM();
    }

    /**
     * 10.3 暂停播放背景音乐
     */
    pauseBGM() {
        this.rtcCloud.pauseBGM();
    }

    /**
     * 10.4 继续播放背景音乐
     */
    resumeBGM() {
        this.rtcCloud.resumeBGM();
    }

    /**
     * 10.5 获取音乐文件总时长，单位毫秒
     *
     * @param {String} path - 音乐文件路径，如果 path 为空，那么返回当前正在播放的 music 时长
     * @return {Number} 成功返回时长，失败返回-1
     */
    getBGMDuration(path) {
        return this.rtcCloud.getBGMDuration(path);
    }

    /**
     * 10.6 设置 BGM 播放进度
     *
     * @param {Number} pos - 单位毫秒
     */
    setBGMPosition(pos) {
        this.rtcCloud.setBGMPosition(pos);
    }

    /**
     * 10.7 设置麦克风的音量大小，播放背景音乐混音时使用，用来控制麦克风音量大小
     *
     * @param {Number} volume - 音量大小，100为正常音量，取值范围为0 - 200。
     */
    setMicVolumeOnMixing(volume) {
        this.rtcCloud.setMicVolumeOnMixing(volume);
    }

    /**
     * 10.8 设置背景音乐的音量大小，播放背景音乐混音时使用，用来控制背景音音量大小
     *
     * @param {Number} volume - 音量大小，100为正常音量，取值范围为0 - 200。
     */
    setBGMVolume(volume) {
        this.rtcCloud.setBGMVolume(volume);
    }
    //TODO:暂时未完成，mac 端暂时不支持系统混音
    // | [startSystemAudioLoopback] | 打开系统声音采集 |
    // | [stopSystemAudioLoopback] | 关闭系统声音采集 |
    // | [setSystemAudioLoopbackVolume] | 设置系统声音采集的音量 |

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （十一）音效相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    //TODO:暂时未完成
    // | [playAudioEffect] | 播放音效 |
    // | [setAudioEffectVolume] | 设置单个音效音量 |
    // | [stopAudioEffect] | 停止音效 |
    // | [stopAllAudioEffects] | 停止所有音效 |
    // | [setAllAudioEffectsVolume] | 设置所有音效音量 |

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （十二）设备和网络测试
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 12.1 开始进行网络测速（视频通话期间请勿测试，以免影响通话质量）
     * 
     * 测速结果将会用于优化 SDK 接下来的服务器选择策略，因此推荐您在用户首次通话前先进行一次测速，这将有助于我们选择最佳的服务器。
     * 同时，如果测试结果非常不理想，您可以通过醒目的 UI 提示用户选择更好的网络。
     * 
     * 注意: 测速本身会消耗一定的流量，所以也会产生少量额外的流量费用。
     * 
     * @param {Number} sdkAppId - 应用标识
     * @param {String} userId   - 用户标识
     * @param {String} userSig  - 用户签名
     */
    startSpeedTest(sdkAppId, userId, userSig) {
        this.rtcCloud.startSpeedTest(sdkAppId, userId, userSig);
    }

    /**
     * 12.2 停止网络测速
     */
    stopSpeedTest() {
        this.rtcCloud.stopSpeedTest();
    }

    /**
     * 12.3 开始进行摄像头测试
     * 
     * 会触发 onFirstVideoFrame 回调接口
     * 
     * 注意: 在测试过程中可以使用 setCurrentCameraDevice 接口切换摄像头。
     * 
     * @param {HTMLElement} view - 承载预览画面的 DOM
     */
    startCameraDeviceTest(view) {
        if (view !== undefined) {
            if (view !== null) {
                let key = this.getKey("camera_device_test_view", TRTCVideoStreamType.TRTCVideoStreamTypeBig);
                this._initRender(key, view);
                const renderer = this.streams.get(key);
                renderer.setContentMode(TRTCVideoFillMode.TRTCVideoFillMode_Fill);
            }
            this.addLocalVideoRenderCallback();
            this.rtcCloud.startCameraDeviceTest();
        } else {
            console.error("startCameraDeviceTest, view is undefined!");
        }
    }

    /**
     * 12.4 停止摄像头测试
     */
    stopCameraDeviceTest() {
        this.rtcCloud.stopCameraDeviceTest();
    }

    /**
     * 12.5 开启麦克风测试
     * 
     * 回调接口 onTestMicVolume 获取测试数据
     * 
     * 该方法测试麦克风是否能正常工作，volume 的取值范围为0 - 100。
     *
     * @param {Number} interval - 反馈音量提示的时间间隔（ms），建议设置到大于 200 毫秒
     */
    startMicDeviceTest(interval) {
        this.rtcCloud.startMicDeviceTest(interval);
    }

    /**
     * 12.6 停止麦克风测试
     */
    stopMicDeviceTest() {
        this.rtcCloud.stopMicDeviceTest();
    }

    /**
     * 12.7 开启扬声器测试
     * 
     * 回调接口 onTestSpeakerVolume 获取测试数据
     * 
     * 该方法播放指定的音频文件测试播放设备是否能正常工作。如果能听到声音，说明播放设备能正常工作。
     * 
     * @param {String} testAudioFilePath - 音频文件的绝对路径，路径字符串使用 UTF-8 编码格式，支持文件格式：WAV、MP3
     */
    startSpeakerDeviceTest(testAudioFilePath) {
        this.rtcCloud.startSpeakerDeviceTest(testAudioFilePath);
    }

    /**
     * 12.8 停止扬声器测试
     */
    stopSpeakerDeviceTest() {
        this.rtcCloud.stopSpeakerDeviceTest();
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （十二）混流转码并发布到 CDN
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 13.1 启动(更新)云端的混流转码：通过腾讯云的转码服务，将房间里的多路画面叠加到一路画面上
     * 
     * 该接口会向腾讯云的转码服务器发送一条指令，目的是将房间里的多路画面叠加到一路画面上。
     * 
     * 如果您在实时音视频 [控制台](https://console.cloud.tencent.com/rav/) 中的功能配置页开启了“启动自动旁路直播”功能，
     * 房间里的每一路画面都会有一个对应的直播 [CDN 地址](https://cloud.tencent.com/document/product/647/16826)，
     * 此时您可以通过云端混流，将多路直播地址的画面混合成一路，这样直播 CDN 上就可以看到混合后的画面。
     * 
     * 您可以通过转码参数来调整每一路画面的位置以及最终输出的画面质量。
     * 
     * 参考文档：[云端混流转码](https://cloud.tencent.com/document/product/647/16827)。
     * 示例代码：我们在 Demo 中增加了该功能的体验入口，您可以在“更多功能”面板中看到“云端画面混合”和“分享播放地址”体验到该功能。
     *
     * <pre>
     * 【画面1】=> 解码 => =>
     *                         \
     * 【画面2】=> 解码 =>  画面混合 => 编码 => 【混合后的画面】
     *                         /
     * 【画面3】=> 解码 => =>
     * </pre>
     * 
     * 注意: 关于云端混流的注意事项：
     *       - 云端转码会引入一定的 CDN 观看延时，大概会增加1 - 2秒。
     *       - 调用该函数的用户，会将多路画面混合到自己这一路的 [CDN 地址](https://cloud.tencent.com/document/product/647/16826) 上。
     * 
     * @param {TRTCTranscodingConfig} config - 如果传入 null 取消云端混流转码，具体参考{@link TRTCTranscodingConfig}
     * @param {TRTCTranscodingConfigMode} config.mode - 转码 config 模式
     * @param {Number} config.appId - 腾讯云直播 AppID
     * @param {Number} config.bizId - 腾讯云直播 bizid
     * @param {Number} config.videoWidth   - 最终转码后的视频分辨率的宽度（px）
     * @param {Number} config.videoHeight  - 最终转码后的视频分辨率的高度（px）
     * @param {Number} config.videoBitrate - 最终转码后的视频分辨率的码率（kbps）
     * @param {Number} config.videoFramerate  - 最终转码后的视频分辨率的帧率（FPS）
     * @param {Number} config.videoGOP        - 最终转码后的视频分辨率的关键帧间隔（也被称为 GOP），单位秒
     * @param {Number} config.audioSampleRate - 最终转码后的音频采样率
     * @param {Number} config.audioBitrate    - 最终转码后的音频码率，单位：kbps
     * @param {Number} config.audioChannels   - 最终转码后的音频声道数
     * @param {TRTCMixUser[]} config.mixUsersArray - 每一路子画面的位置信息，具体参考{@link TRTCMixUser}
     * @param {String}  config.mixUsersArray[].userId      - 参与混流的 userId
     * @param {String}  config.mixUsersArray[].roomId      - 参与混流的 roomId，跨房流传入的实际 roomId，当前房间流传入 roomId = null
     * @param {Rect}  config.mixUsersArray[].rect        - 图层位置坐标以及大小，左上角为坐标原点(0,0) （绝对像素值）
     * @param {Number}  config.mixUsersArray[].rect.left   - 图层位置的左坐标
     * @param {Number}  config.mixUsersArray[].rect.top    - 图层位置的上坐标
     * @param {Number}  config.mixUsersArray[].rect.right  - 图层位置的右坐标
     * @param {Number}  config.mixUsersArray[].rect.bottom - 图层位置的下坐标
     * @param {Number}  config.mixUsersArray[].zOrder      - 图层层次（1 - 15）不可重复
     * @param {Boolean} config.mixUsersArray[].pureAudio   - 是否纯音频
     * @param {TRTCVideoStreamType} config.mixUsersArray[].streamType  - 参与混合的是主路画面（TRTCVideoStreamTypeBig）或屏幕分享（TRTCVideoStreamTypeSub）画面
     * @param {Number} config.mixUsersArraySize    - 数组 mixUsersArray 的大小
     */
    setMixTranscodingConfig(config) {
        if (config instanceof TRTCTranscodingConfig) {
            this.rtcCloud.setMixTranscodingConfig(config.mode, config.appId, config.bizId, config.videoWidth, config.videoHeight, config.videoBitrate, config.videoFramerate, config.videoGOP, config.audioSampleRate, config.audioBitrate, config.audioChannels, config.mixUsersArray);
        } else if (config === null) {
            this.rtcCloud.setMixTranscodingConfig(-1);
        } else {
            console.error("setMixTranscodingConfig, config is not instanceof TRTCTranscodingConfig!");
        }
    }

    /**
     * 13.2 旁路转推到指定的推流地址
     * 
     * 该接口会向腾讯云的转推服务器发送一条指令，腾讯云会将当前一路的音视频画面转推到您指定的 rtmp 推流地址上。
     * 
     * 在实时音视频 [控制台](https://console.cloud.tencent.com/rav/) 中的功能配置页开启了“启动自动旁路直播”功能后，
     * 房间里的每一路画面都有一路默认的腾讯云 CDN 地址，所以该功能并不常用，仅在您需要适配多家 CDN 服务商时才需要关注该功能。
     * 
     * 由于仅转推单独的一路画面到直播 CDN 并没有什么太大的意义，所以该方案通常是跟云端转码混合使用的。
     * 也就是先通过 setMixTranscodingConfig 将房间里的多路画面混合到一路上，再转推出去。
     * 
     * 注意: 关于旁路转推的注意事项：
     *       - 默认只支持转推到腾讯云的 rtmp [推流地址](https://cloud.tencent.com/document/product/267/32720) 上，转推其他云的需求请通过工单联系我们。
     *       - 调用该函数的用户，只会转推自己这一路画面到指定的 rtmp 推流地址上，因此一般需要配合 setMixTranscodingConfig 一起使用。
     *       - TRTC 房间里的每一路画面都有一路默认的腾讯云 CDN 地址（需要开启），所以该功能并不常用，仅在您需要适配多家 CDN 服务商时才需要关注该功能。
     * 
     * @param {TRTCPublishCDNParam} param - 转推参数，具体参考{@link TRTCPublishCDNParam}
     * @param {Number} param.appId - 腾讯云 AppID
     * @param {Number} param.bizId - 腾讯云直播 bizId
     * @param {String} param.url   - 旁路转推的 URL
     */
    startPublishCDNStream(param) {
        if (param instanceof TRTCPublishCDNParam) {
            this.rtcCloud.startPublishCDNStream(param.appId, param.bizId, param.url);
        } else {
            console.error("startPublishCDNStream, param is not instanceof TRTCPublishCDNParam!");
        }
    }

    /**
     * 13.3 停止旁路推流
     */
    stopPublishCDNStream() {
        this.rtcCloud.stopPublishCDNStream();
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                      （十四）LOG 相关接口函数
    //
    /////////////////////////////////////////////////////////////////////////////////
    /**
     * 14.1 获取 SDK 版本信息
     *
     * @return {String} UTF-8 编码的版本号。
     */
    getSDKVersion() {
        return this.rtcCloud.getSDKVersion();
    }

    /**
     * 14.2 设置 Log 输出级别
     *
     * @param {TRTCLogLevel} level - Log 输出等级，默认值：TRTCLogLevelNone
     * - TRTCLogLevelNone   : 不输出任何 SDK Log
     * - TRTCLogLevelVerbose: 输出所有级别的 Log
     * - TRTCLogLevelDebug  : 输出 DEBUG，INFO，WARNING，ERROR 和 FATAL 级别的 Log
     * - TRTCLogLevelInfo   : 输出 INFO，WARNNING，ERROR 和 FATAL 级别的 Log
     * - TRTCLogLevelWarn   : 只输出WARNNING，ERROR 和 FATAL 级别的 Log
     * - TRTCLogLevelError  : 只输出ERROR 和 FATAL 级别的 Log
     * - TRTCLogLevelFatal  : 只输出 FATAL 级别的 Log
     */
    setLogLevel(level) {
        this.rtcCloud.setLogLevel(level);
    }

    /**
     * 14.3 启用或禁用控制台日志打印
     *
     * @param {Boolean} enabled - 指定是否启用，默认为禁止状态
     */
    setConsoleEnabled(enabled) {
        this.rtcCloud.setConsoleEnabled(enabled);
    }

    /**
     * 14.4 启用或禁用 Log 的本地压缩
     * 
     * 开启压缩后，Log 存储体积明显减小，但需要腾讯云提供的 Python 脚本解压后才能阅读。
     * 禁用压缩后，Log 采用明文存储，可以直接用记事本打开阅读，但占用空间较大。
     *
     * @param {Boolean} enabled - 指定是否启用，默认为禁止状态
     */
    setLogCompressEnabled(enabled) {
        this.rtcCloud.setLogCompressEnabled(enabled);
    }

    /**
     * 14.5 设置日志保存路径
     * 
     * 注意: 
     * - windows 日志文件默认保存在 C:/Users/[系统用户名]/AppData/Roaming/Tencent/liteav/log，即 %appdata%/Tencent/liteav/log 下，如需修改，必须在所有方法前调用。
     * - mac 日志文件默认保存在 sandbox Documents/log 下，如需修改，必须在所有方法前调用。
     * 
     * @param {String} path - 存储日志的文件夹，例如 "D:\\Log"，UTF-8 编码
     */
    setLogDirPath(path) {
        this.rtcCloud.setLogDirPath(path);
    }

    //TODO:暂时未完成
    // | [setLogCallback] | 设置日志回调 |
    // | [showDebugView] | 显示仪表盘 |

    /**
     * 14.8 调用实验性 API 接口
     * 
     * 注意: 该接口用于调用一些实验性功能
     * 
     * @param {String} jsonStr - 接口及参数描述的 JSON 字符串
     */
    callExperimentalAPI(jsonStr) {
        this.rtcCloud.callExperimentalAPI(jsonStr);
    }

    ///------------------------内部方法-----------------------------

    addLocalVideoRenderCallback() {
        let self = this;
        this.rtcCloud.addLocalVideoRenderCallback(function (id, type, width, height, timestamp, rotation, data) {
            self._onRenderFrame(id, type, width, height, timestamp, rotation, data);
        });
    }

    addRemoteVideoRenderCallback(userid) {
        let self = this;
        this.rtcCloud.addRemoteVideoRenderCallback(userid, function (id, type, width, height, timestamp, rotation, data) {
            self._onRenderFrame(id, type, width, height, timestamp, rotation, data);
        });
    }

    //----------------------------------//
    // Render 
    //----------------------------------//
    /**
     * 选择绘制的模式 (webgl/yuvcanvs/)
     * @param mode:
     * - 1  webgl
     * - 2  yuvcanvs
     */
    setRenderMode(mode = 1) {
        if (this.renderMode !== mode) {
        this.renderMode = mode;
            // 遍历原有的renderer，改为新的渲染模式
            this.streams.forEach(function (value, key, map) {
                let view = value.self.getView();
                _initRender(key, view)
            });
    }

    }

    getKey(uid, type) {
        return String(uid) + "_" + String(type);
    }

    /**
     * 判断是否支持webGL
     * @returns {boolean}
     * @private
     */
    _isSupportWebGL() {
        const canvas = document.createElement('canvas');
        let gl;
        canvas.width = 1;
        canvas.height = 1;
        const options = {
            alpha: false,
            depth: false,
            stencil: false,
            antialias: false,
            preferLowPowerToHighPerformance: true
        };
        try {
            gl = canvas.getContext('webgl', options) || canvas.getContext('experimental-webgl', options);
        }
        catch (e) {
            return false;
        }
        if (gl) {
            return true;
        }
        else {
            return false;
        }
    }

    _getRenderer(uid, type) {
        let key = this.getKey(uid, type);
        return this.streams.get(key);
    }

    _onRenderFrame(uid, type, width, height, timestamp, rotation, data) {
        // 由于 mac 监听自定义渲染回调后不会回调 onFirstVideoFrame ，所以在这里自己实现回调 onFirstVideoFrame 
        let self = this;
        if (process.platform == "darwin") {
            setImmediate(() => {
                let index = self.firstVideoFrameList.findIndex(function (item) {
                    return uid === item.uid && type === item.type;
                });
                if (index === -1) {
                    self.emit('onFirstVideoFrame', uid, type, width, height);
                    self.firstVideoFrameList.push({ uid: uid, type: type });
                }
            });
        }
        if (uid.length == 0) {
            this._onRealRender('local_video', type, width, height, timestamp, rotation, data); //本地社视频返回的是空
            this._onRealRender('camera_device_test_view', type, width, height, timestamp, rotation, data);  //摄像头测试

            //回调local view 给用户数据，本地数据
            if (type == TRTCVideoStreamType.TRTCVideoStreamTypeBig) {
                if (this.localVideoCallback !== null && this.localVideoCallback.callback !== null) {
                    this._onUserRenderCallback(this.localVideoCallback.callback, this.localVideoCallback.pixelFormat, uid, type, width, height, timestamp, rotation, data);
                }
            }
        } else {
            // 画面不区分大小流，只区分主流和辅流，这里统一使用主流当做 key
            let streamType = type;
            if (streamType === TRTCVideoStreamType.TRTCVideoStreamTypeSmall) {
                streamType = TRTCVideoStreamType.TRTCVideoStreamTypeBig;
            }
            this._onRealRender(uid, streamType, width, height, timestamp, rotation, data);

            //回调远程给用户数据
            let remoteCallback = this.remoteVideoCallback.get(uid);
            if (remoteCallback !== undefined && remoteCallback !== null && remoteCallback.callback !== undefined && remoteCallback.callback !== null) {
                this._onUserRenderCallback(remoteCallback.callback, remoteCallback.pixelFormat, uid, type, width, height, timestamp, rotation, data);
            }
        }
        data = null;
    }

    _onRealRender(uid, type, width, height, timestamp, rotation, data) {
        const renderer = this._getRenderer(uid, type);

        if (!renderer) {
            return;
        }
        if (this.renderMode === 3) {
            var realRatation = rotation * 90; //ratation是0, 1, 2, 3. 绘制内部使用0, 90, 180, 270
            renderer.drawFrame({
                width: width,
                height: height,
                rotation: realRatation,
                timestamp: timestamp,
                data: data,
            });
        }else {
        let ylen = width * height;
        let ulen = width * height / 4;
        let vlen = ulen;
        let ydata = new Uint8Array(data, 0, ylen);
        let udata = new Uint8Array(data, ylen, ulen);
        let vdata = new Uint8Array(data, ylen + ulen, data.le);

        var realRatation = rotation * 90; //ratation是0, 1, 2, 3. 绘制内部使用0, 90, 180, 270
        renderer.drawFrame({
            width: width,
            height: height,
            rotation: realRatation,
            timestamp: timestamp,
            yUint8Array: ydata,
            uUint8Array: udata,
            vUint8Array: vdata,
        });
        ydata = null;
        udata = null;
        vdata = null;
    }
    }

    _initRender(key, view) {
        if (this.streams.has(String(key))) {
            this._destroyRender(key);
        }
        let renderer;
        if (this.renderMode === 1) {
            renderer = new Renderer_1.GlRenderer();
        }
        else if (this.renderMode === 2) {
            renderer = new Renderer_1.SoftwareRenderer();
        }
        else if (this.renderMode === 3) {
            renderer = new Renderer_1.RGBRenderer();
        }
        else {
            console.warn('Unknown render mode, fallback to 1');
            renderer = new Renderer_1.GlRenderer();
        }

        renderer.bind(view);
        this.streams.set(String(key), renderer);
    }

    _destroyRender(key, onFailure) {
        if (!this.streams.has(String(key))) {
            return;
        }
        const renderer = this.streams.get(String(key));
        try {
            renderer.unbind();
            this.streams.delete(String(key));
        }
        catch (err) {
            onFailure && onFailure(err);
        }
    }

    /** 
     * 写文件
     *   path, path
     *   data, Uint8Array
     *   let testdata = new Uint8Array(data, 0, data.byteLength);
     *   this._writefile('./testyuv', testdata);
     * @private
     */
    _writefile(path, data) {
        let fs = require('fs');
        fs.writeFile(path, data, (err) => {
            if (err) {
                throw err
            }
            else {
                console.log(" write file ok, path:" + path)
            }
        })
    }

    _onUserRenderCallback(callback, format, userid, streamType, width, height, timestamp, rotation, data) {
        if (callback !== null) {

            let videoframe = new TRTCVideoFrame();
            videoframe.videoFormat = format;
            videoframe.bufferType = TRTCVideoBufferType.TRTCVideoBufferType_Buffer;
            videoframe.textureId = 0;
            videoframe.width = width;
            videoframe.height = height;
            videoframe.timestamp = timestamp;
            videoframe.rotation = rotation;

            if (format == TRTCVideoPixelFormat.TRTCVideoPixelFormat_I420 && this.renderMode !== 3) {
                videoframe.length = data.byteLength;
                videoframe.data = data;
                callback(userid, streamType, videoframe);
            }
            else if (format == TRTCVideoPixelFormat.TRTCVideoPixelFormat_BGRA32) {
                if (this.renderMode !== 3) {
                videoframe.data = this._YUV2RGBA(width, height, data);
                videoframe.length = width * height * 4;
                callback(userid, streamType, videoframe)
                }else {
                    videoframe.data = data;
                    videoframe.length = width * height * 4;
                    callback(userid, streamType, videoframe)
                }

            }
        }
    }

    _initImageData(width, height) {
        var pixelCount = width * height * 4;
        var data = new Uint8ClampedArray(pixelCount);
        this.imageData = new ImageData(data, width, height);
        for (var i = 0; i < pixelCount; i += 4) {
            data[i + 3] = 255;
        }
    }

    _YUV2RGBA(width, height, buffer) {
        if (this.imageData === null ||
            this.imageData.width != width ||
            this.imageData.height != height) {
            this._initImageData(width, height);
        }

        let format = YUVBuffer.format({
            width,
            height,
            chromaWidth: width / 2,
            chromaHeight: height / 2
        });

        let ylen = width * height;
        let ulen = width * height / 4;
        let vlen = ulen;

        let ydata = new Uint8Array(buffer, 0, ylen);
        let udata = new Uint8Array(buffer, ylen, ulen);
        let vdata = new Uint8Array(buffer, ylen + ulen, vlen);

        let y = YUVBuffer.lumaPlane(format, ydata);
        let u = YUVBuffer.chromaPlane(format, udata);
        let v = YUVBuffer.chromaPlane(format, vdata);
        let frame = YUVBuffer.frame(format, y, u, v);

        // YUV -> RGB over the entire encoded frame
        YCbCr.convertYCbCr(frame, this.imageData.data);

        ydata = null;
        udata = null;
        vdata = null;

        return this.imageData.data;
    }
}

class VideoRenderCallback {
    constructor() {
        this.callback = null;
        this.pixelFormat = TRTCVideoPixelFormat.TRTCVideoPixelFormat_BGRA32;
    }
};

module.exports = TRTCCloud;