/**
 * TRTC 关键类型定义
 * 
 * @description 分辨率、质量等级等枚举和常量值的定义
 * 
 */

/////////////////////////////////////////////////////////////////////////////////
//
//                    【（一）视频相关枚举值定义】
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * 1.1 视频分辨率
 * 
 * 此处仅定义了横屏分辨率，如果要使用360 × 640这样的竖屏分辨率，需要同时指定 TRTCVideoResolutionMode 为 Portrait。
 * 
 * @enum {Number}
 */
const TRTCVideoResolution = exports.TRTCVideoResolution = {
    /// 宽高比1:1
    /** [C] 建议码率80kbps */
    TRTCVideoResolution_120_120: 1,
    /** [C] 建议码率100kbps */
    TRTCVideoResolution_160_160: 3,
    /** [C] 建议码率200kbps */
    TRTCVideoResolution_270_270: 5,
    /** [C] 建议码率350kbps */
    TRTCVideoResolution_480_480: 7,
    
    /// 宽高比4:3
    /** [C] 建议码率100kbps */
    TRTCVideoResolution_160_120: 50,
    /** [C] 建议码率150kbps */
    TRTCVideoResolution_240_180: 52,
    /** [C] 建议码率200kbps */
    TRTCVideoResolution_280_210: 54,
    /** [C] 建议码率250kbps */
    TRTCVideoResolution_320_240: 56,
    /** [C] 建议码率300kbps */
    TRTCVideoResolution_400_300: 58,
    /** [C] 建议码率400kbps */
    TRTCVideoResolution_480_360: 60,
    /** [C] 建议码率600kbps */
    TRTCVideoResolution_640_480: 62,
    /** [C] 建议码率1000kbps */
    TRTCVideoResolution_960_720: 64,
    
    /// 宽高比16:9
    /** [C] 建议码率150kbps */
    TRTCVideoResolution_160_90: 100,
    /** [C] 建议码率200kbps */
    TRTCVideoResolution_256_144: 102,
    /** [C] 建议码率250kbps */
    TRTCVideoResolution_320_180: 104,
    /** [C] 建议码率350kbps */
    TRTCVideoResolution_480_270: 106,
    /** [C] 建议码率550kbps */
    TRTCVideoResolution_640_360: 108,
    /** [C] 建议码率850kbps */
    TRTCVideoResolution_960_540: 110,
    /** 
     * [C] 摄像头采集 - 建议码率1200kbps
     * [S] 屏幕分享   - 建议码率 低清：400kbps 高清：600kbps 
     */
    TRTCVideoResolution_1280_720: 112,
    /** [S] 屏幕分享   - 建议码率800kbps */
    TRTCVideoResolution_1920_1080: 114,
};

/**
 * 1.2 视频分辨率模式
 * 
 * - 横屏分辨率：TRTCVideoResolution_640_360 + TRTCVideoResolutionModeLandscape = 640 × 360 
 * - 竖屏分辨率：TRTCVideoResolution_640_360 + TRTCVideoResolutionModePortrait  = 360 × 640 
 * 
 * @enum {Number}
 */
const TRTCVideoResolutionMode = exports.TRTCVideoResolutionMode = {
    /** 横屏分辨率 */
    TRTCVideoResolutionModeLandscape: 0,
    /** 竖屏分辨率 */
    TRTCVideoResolutionModePortrait: 1,
};

/**
 * 1.3 视频流类型
 * 
 * TRTC 内部有三种不同的音视频流，分别是：
 * - 主画面：最常用的一条线路，一般用来传输摄像头的视频数据。
 * - 小画面：跟主画面的内容相同，但是分辨率和码率更低。
 * - 辅流画面：一般用于屏幕分享，以及远程播片（比如老师放一段视频给学生）。
 * 
 * 注意:
 * - 如果主播的上行网络和性能比较好，则可以同时送出大小两路画面。
 * - SDK 不支持单独开启小画面，小画面必须依附于主画面而存在。
 * 
 * @enum {Number}
 */
const TRTCVideoStreamType = exports.TRTCVideoStreamType = {
    /** 大画面视频流 */
    TRTCVideoStreamTypeBig: 0,
    /** 小画面视频流 */
    TRTCVideoStreamTypeSmall: 1,
    /** 辅流（屏幕分享） */
    TRTCVideoStreamTypeSub: 2,
};

/**
 * 1.4 画质级别
 * 
 * TRTC SDK 对画质定义了六种不同的级别，Excellent 代表最好，Down 代表不可用。
 * 
 * @enum {Number}
 */
const TRTCQuality = exports.TRTCQuality = {
    /** 未定义 */
    TRTCQuality_Unknown: 0,
    /** 最好 */
    TRTCQuality_Excellent: 1,
    /** 好 */
    TRTCQuality_Good: 2,
    /** 一般 */
    TRTCQuality_Poor: 3,
    /** 差 */
    TRTCQuality_Bad: 4, 
    /** 很差 */
    TRTCQuality_Vbad: 5,
    /** 不可用 */
    TRTCQuality_Down: 6,
};

/**
 * 1.5 视频画面填充模式
 * 
 * 如果画面的显示分辨率不等于画面的原始分辨率，就需要您设置画面的填充模式:
 * - TRTCVideoFillMode_Fill，图像铺满屏幕，超出显示视窗的视频部分将被截掉，所以画面显示可能不完整。
 * - TRTCVideoFillMode_Fit，图像长边填满屏幕，短边区域会被填充黑色，但画面的内容肯定是完整的。
 * 
 * @enum {Number}
 */
const TRTCVideoFillMode = exports.TRTCVideoFillMode = {
    /** 图像铺满屏幕，超出显示视窗的视频部分将被截掉 */
    TRTCVideoFillMode_Fill: 0,
    /** 图像长边填满屏幕，短边区域会被填充黑色 */
    TRTCVideoFillMode_Fit: 1,
};

/**
 * 1.6 视频画面旋转方向
 * 
 * TRTC SDK 提供了对本地和远程画面的旋转角度设置 API，如下的旋转角度都是指顺时针方向的。
 * 
 * @enum {Number}
 */
const TRTCVideoRotation = exports.TRTCVideoRotation = {
    /** 顺时针旋转0度 */
    TRTCVideoRotation0: 0,
    /** 顺时针旋转90度 */
    TRTCVideoRotation90: 1,
    /** 顺时针旋转180度 */
    TRTCVideoRotation180: 2,
    /** 顺时针旋转270度 */
    TRTCVideoRotation270: 3,
};

/**
 * 1.7 美颜（磨皮）算法
 * 
 * TRTC SDK 内置了多种不同的磨皮算法，您可以选择最适合您产品定位的方案。
 * 
 * @enum {Number}
 * 
 */
const TRTCBeautyStyle = exports.TRTCBeautyStyle = {
    /** 光滑，适用于美女秀场，效果比较明显。 */
    TRTCBeautyStyleSmooth: 0,
    /** 自然，磨皮算法更多地保留了面部细节，主观感受上会更加自然。 */
    TRTCBeautyStyleNature: 1,
};

/**
 * 1.8 视频像素格式
 * 
 * TRTC SDK 提供针对视频的自定义采集和自定义渲染功能，在自定义采集功能中，您可以用如下枚举值描述您采集的视频像素格式。
 * 在自定义渲染功能中，您可以指定您期望 SDK 回调的视频像素格式。
 * 
 * @enum {Number}
 * 
 */
const TRTCVideoPixelFormat = exports.TRTCVideoPixelFormat = {
    /** 未定义 */
    TRTCVideoPixelFormat_Unknown: 0,
    /** I420 */
    TRTCVideoPixelFormat_I420: 1,
    /** OpenGL 2D 纹理 */
    // TRTCVideoPixelFormat_Texture_2D: 2,
    /** BGRA32 */
    TRTCVideoPixelFormat_BGRA32: 3,
};

/**
 * 1.9 视频数据包装格式
 * 
 * @enum {Number}
 * 
 */
const TRTCVideoBufferType = exports.TRTCVideoBufferType = {
    /** 未知类型 */
    TRTCVideoBufferType_Unknown: 0,
    /** 二进制Buffer类型 */
    TRTCVideoBufferType_Buffer: 1,
    /** 纹理类型 */
    // TRTCVideoBufferType_Texture: 3,
};

/////////////////////////////////////////////////////////////////////////////////
//
//                    【（二）网络相关枚举值定义】
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * 2.1 应用场景
 * 
 * TRTC 可用于视频会议和在线直播等多种应用场景，针对不同的应用场景，TRTC SDK 的内部会进行不同的优化配置：
 * - VideoCall：视频通话场景，即绝大多数时间都是两人或两人以上视频通话的场景，比如1v1的在线课程辅导，1vN (N < 8) 的视频会议或者小班课堂。
 * - LIVE：在线直播场景，即绝大多数时间都是一人直播，偶尔有多人视频互动的场景，比如美女秀场连麦等场景。
 * 
 * @enum {Number}
 * 
 */
const TRTCAppScene = exports.TRTCAppScene = {
    /** 视频通话场景，内部编码器和网络协议优化侧重流畅性，降低通话延迟和卡顿率。 */
    TRTCAppSceneVideoCall: 0,
    /** 在线直播场景，内部编码器和网络协议优化侧重性能和兼容性，性能和清晰度表现更佳。 */
    TRTCAppSceneLIVE: 1,
};

/**
 * 2.2 角色，仅适用于直播场景（TRTCAppSceneLIVE）
 * 
 * 在直播场景中，多数用户只是观众，只有个别用户是主播，这种角色区分可以有利于 TRTC 进行更好的定向优化。
 *
 * - Anchor：主播，可以上行视频和音频，一个房间里的主播人数不能超过 50 人。
 * - Audience：观众，只能观看，不能上行视频和音频，一个房间里的观众人数没有上限。
 * 
 * @enum {Number}
 * 
 */
const TRTCRoleType = exports.TRTCRoleType = {
    /** 主播 */
    TRTCRoleAnchor: 20,
    /** 观众 */
    TRTCRoleAudience: 21,
};

/**
 * 2.3 流控模式
 * 
 * TRTC SDK 内部需要时刻根据网络情况调整内部的编解码器和网络模块，以便能够对网络的变化做出反应。
 * 为了支持快速算法升级，SDK 内部设置了两种不同的流控模式：
 * - ModeClient： 本地控制，用于 SDK 开发内部调试，客户请勿使用。
 * - ModeServer： 云端控制，推荐模式，也是默认默认。
 * 
 * >推荐您使用云端控制，这样每当我们升级 Qos 算法时，您无需升级 SDK 即可体验更好的效果。
 * 
 * @enum {Number}
 * 
 */
const TRTCQosControlMode = exports.TRTCQosControlMode = {
    /** 客户端控制（用于 SDK 开发内部调试，客户请勿使用） */
    TRTCQosControlModeClient: 0,
    /** 云端控制（默认） */
    TRTCQosControlModeServer: 1,
};

/**
 * 2.4 画质偏好
 * 
 * 指当 TRTC SDK 在遇到弱网络环境时，您是希望“保清晰”还是“保流畅”：
 *
 * - Smooth：弱网下保流畅，在遭遇弱网环境时首先确保声音的流畅和优先发送，画面会变得模糊且会有较多马赛克，但可以保持流畅不卡顿。
 * - Clear：弱网下保清晰，在遭遇弱网环境时，画面会尽可能保持清晰，但可能会更容易出现卡顿。
 * 
 * @enum {Number}
 * 
 */
const TRTCVideoQosPreference = exports.TRTCVideoQosPreference = {
    /** 弱网下保流畅 */
    TRTCVideoQosPreferenceSmooth: 1,
    /** 弱网下保清晰 */
    TRTCVideoQosPreferenceClear: 2,
};

/////////////////////////////////////////////////////////////////////////////////
//
//                    【（三）继承 TXLiteAVBase 的定义】
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * 3.1 音频帧的格式
 * 
 * @enum {Number}
 * 
 */
const TRTCAudioFrameFormat = exports.TRTCAudioFrameFormat = {
    /** 未指定 */
    TRTCAudioFrameFormatNone: 0,
    /** PCM，每个采样点占16bit数据量。 */
    TRTCAudioFrameFormatPCM: 1,
};

/**
 * 3.2 屏幕分享目标信息
 * 
 * @enum {Number}
 * 
 */
const TRTCScreenCaptureSourceType = exports.TRTCScreenCaptureSourceType = {
    /** 未知类型 */
    TRTCScreenCaptureSourceTypeUnknown: -1,
    /** 该分享目标是某一个Windows窗口 */
    TRTCScreenCaptureSourceTypeWindow: 0,
    /** 该分享目标是整个Windows桌面 */
    TRTCScreenCaptureSourceTypeScreen: 1,
    /** 该分享目标是自定义窗口 */
    TRTCScreenCaptureSourceTypeCustom: 2,
};

/**
 * 3.3 图缓存
 * 
 * @param {ArrayBuffer} buffer - 图内容
 * @param {Number}      length - 图缓存大小
 * @param {Number}      width  - 图宽
 * @param {Number}      heigth - 图高
 * 
 */
exports.TRTCImageBuffer = class TRTCImageBuffer {
    constructor() {
        this.buffer = null;
        this.length = 0;
        this.width = 0;
        this.heigth = 0;
    }
};

/**
 * 3.4 屏幕采集源信息
 * 
 * @param {TRTCScreenCaptureSourceType} type       - 采集源类型
 * @param {String}                      sourceId   - 采集源ID；对于窗口，该字段指示窗口句柄；对于屏幕，该字段指示屏幕ID
 * @param {String}                      sourceName - 采集源名称，UTF8编码
 * @param {TRTCImageBuffer}             thumbBGRA  - 缩略图内容
 * @param {TRTCImageBuffer}             iconBGRA   - 图标内容
 * 
 */
exports.TRTCScreenCaptureSourceInfo = class TRTCScreenCaptureSourceInfo {
    constructor() {
        this.type = exportsTRTCScreenCaptureSourceType.TRTCScreenCaptureSourceTypeUnknown;
        this.sourceId = null;
        this.sourceName = null;
        this.thumbBGRA = null;
        this.iconBGRA = null;
    }
};

/**
 * 3.5 设备信息
 * 
 * @param {String} deviceId   - 设备PID，字符编码格式是UTF-8
 * @param {String} deviceName - 设备名称，字符编码格式是UTF-8
 * 
 */
exports.TRTCDeviceInfo = class TRTCDeviceInfo {
    constructor() {
        this.deviceId = null;
        this.deviceName = null;
    }
};

/**
 * 3.6 视频帧数据
 * 
 * @param {String} videoFormat   - 视频帧的格式
 * @param {String} bufferType - 视频数据结构类型
 * @param {String} data - 视频数据，字段bufferType是LiteAVVideoBufferType_Buffer时生效
 * @param {String} textureId - 视频纹理ID，字段bufferType是LiteAVVideoBufferType_Texture时生效
 * @param {String} length - 视频数据的长度，单位是字节，对于i420而言， length = width * height * 3 / 2，对于BGRA32而言， length = width * height * 4
 * @param {String} width - 画面的宽度
 * @param {String} height - 画面的高度
 * @param {String} timestamp - 时间戳，单位ms
 * @param {String} rotation - 画面旋转角度
 * 
 */
exports.TRTCVideoFrame = class TRTCVideoFrame {
    constructor() {
        this.videoFormat = TRTCVideoPixelFormat.TRTCVideoPixelFormat_Unknown;
        this.bufferType = TRTCVideoBufferType.TRTCVideoBufferType_Unknown;
        this.data = null;
        this.textureId = 0;
        this.length = 0;
        this.width = 0;
        this.height = 0;
        this.timestamp = 0;
        this.rotation = 0;
    }
};


/////////////////////////////////////////////////////////////////////////////////
//
//                    【（四）更多枚举值定义】
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * 4.1 Log 级别
 * 
 * @enum {Number}
 * 
 */
const TRTCLogLevel = exports.TRTCLogLevel = {
    /** 不输出任何 SDK Log */
    TRTCLogLevelNone: 0,
    /** 输出所有级别的 Log */
    TRTCLogLevelVerbose: 1,
    /** 输出 DEBUG，INFO，WARNING，ERROR 和 FATAL 级别的 Log */
    TRTCLogLevelDebug: 2,
    /** 输出 INFO，WARNNING，ERROR 和 FATAL 级别的 Log */
    TRTCLogLevelInfo: 3,
    /** 只输出WARNNING，ERROR 和 FATAL 级别的 Log */
    TRTCLogLevelWarn: 4,
    /** 只输出ERROR 和 FATAL 级别的 Log */
    TRTCLogLevelError: 5,
    /** 只输出 FATAL 级别的 Log */
    TRTCLogLevelFatal: 6,
};

/**
 * 4.2 设备操作
 * 
 * @enum {Number}
 * 
 */
const TRTCDeviceState = exports.TRTCDeviceState = {
    /** 添加设备 */
    TRTCDeviceStateAdd: 0,
    /** 移除设备 */
    TRTCDeviceStateRemove: 1,
    /** 设备已启用 */
    TRTCDeviceStateActive: 2,
};

/**
 * 4.3 设备类型
 * 
 * @enum {Number}
 * 
 */
const TRTCDeviceType = exports.TRTCDeviceType = {
    /** 未知类型 */
    TRTCDeviceTypeUnknow: -1,
    /** 麦克风 */
    TRTCDeviceTypeMic: 0,
    /** 扬声器 */
    TRTCDeviceTypeSpeaker: 1,
    /** 摄像头 */
    TRTCDeviceTypeCamera: 2,
};

/**
 * 4.4 水印图片的源类型
 * 
 * @enum {Number}
 * 
 */
const TRTCWaterMarkSrcType = exports.TRTCWaterMarkSrcType = {
    /** 图片文件路径，支持 BMP、GIF、JPEG、PNG、TIFF、Exif、WMF 和 EMF 文件格式 */
    TRTCWaterMarkSrcTypeFile: 0,
    /** BGRA32格式内存块 */
    TRTCWaterMarkSrcTypeBGRA32: 1,
    /** RGBA32格式内存块 */
    TRTCWaterMarkSrcTypeRGBA32: 2,
};

/////////////////////////////////////////////////////////////////////////////////
//
//                      【（五）TRTC 核心类型定义】
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * 5.1 进房相关参数
 * 
 * 只有该参数填写正确，才能顺利调用 enterRoom 进入 roomId 所指定的音视频房间。
 * 
 * @param {Number}       sdkAppId      - 【字段含义】应用标识（必填），腾讯视频云基于 sdkAppId 完成计费统计。
 *                                       【推荐取值】在腾讯云 [TRTC 控制台](https://console.cloud.tencent.com/rav/) 中创建应用，之后可以在账号信息页面中得到该 ID
 * @param {String}       userId        - 【字段含义】用户标识（必填）。当前用户的 userId，相当于用户名，UTF-8编码。
 *                                       【推荐取值】如果一个用户在您的账号系统中的 ID 为“abc”，则 userId 即可设置为“abc”。
 * @param {String}       userSig       - 【字段含义】用户签名（必填），当前 userId 对应的验证签名，相当于登录密码。
 *                                       【推荐取值】请参考 [如何计算UserSig](https://cloud.tencent.com/document/product/647/17275)。
 * @param {Number}       roomId        - 【字段含义】房间号码（必填），指定房间号，在同一个房间里的用户（userId）可以彼此看到对方并进行视频通话。
 *                                       【推荐取值】您可以随意指定，但请不要重复，如果您的用户账号 ID 是数字类型的，可以直接用创建者的用户 ID 作为 roomId。
 * @param {TRTCRoleType} role          - 【字段含义】直播场景下的角色，仅适用于直播场景（TRTCAppSceneLIVE），视频通话场景下指定无效。
 *                                       【推荐取值】默认值：主播（TRTCRoleAnchor）
 * @param {String}       privateMapKey - 【字段含义】房间签名（非必填），如果您希望某个房间只能让特定的某些 userId 进入，就需要使用 privateMapKey 进行权限保护。
 *                                       【推荐取值】仅建议有高级别安全需求的客户使用，参考文档：[进房权限保护](https://cloud.tencent.com/document/product/647/32240)
 * @param {String}       businessInfo  - 【字段含义】业务数据（非必填），某些非常用的高级特性才需要用到此字段。
 *                                       【推荐取值】不建议使用
 */
exports.TRTCParams = class TRTCParams {
    constructor() {
        this.sdkAppId = 0;
        this.userId = null;
        this.userSig = null;
        this.roomId = 0;
        this.role = TRTCRoleType.TRTCRoleAnchor;
        this.privateMapKey = null;
        this.businessInfo = null;
    }
};

/**
 * 5.2 视频编码参数
 * 
 * 该设置决定了远端用户看到的画面质量（同时也是云端录制出的视频文件的画面质量）。
 * 
 * @param {TRTCVideoResolution}     videoResolution - 【字段含义】 视频分辨率
 *                                                    【推荐取值】 
 *                                                     - 视频通话建议选择360 × 640及以下分辨率，resMode 选择 Portrait。
 *                                                     - 手机直播建议选择 540 × 960，resMode 选择 Portrait。
 *                                                     - Window 和 iMac 建议选择 640 × 360 及以上分辨率，resMode 选择 Landscape。
 *                                                    【特别说明】 您在 TRTCVideoResolution 只能找到横屏模式的分辨率，例如：640 × 360 这样的分辨率。
 *                                                                如果想要使用竖屏分辨率，请指定 resMode 为 Portrait，例如：640 × 360 + Portrait = 360 × 640。
 * @param {TRTCVideoResolutionMode} resMode         - 【字段含义】分辨率模式（横屏分辨率 - 竖屏分辨率）
 *                                                    【推荐取值】手机直播建议选择 Portrait，Window 和 Mac 建议选择 Landscape。
 *                                                    【特别说明】如果 videoResolution 指定分辨率 640 × 360，resMode 指定模式为 Portrait，则最终编码出的分辨率为360 × 640。
 * @param {Number}                  videoFps        - 【字段含义】视频采集帧率
 *                                                    【推荐取值】15fps 或 20fps，10fps 以下会有轻微卡顿感，5fps 以下卡顿感明显，20fps 以上的帧率则过于浪费（电影的帧率也只有 24fps）。
 *                                                    【特别说明】很多 Android 手机的前置摄像头并不支持15fps以上的采集帧率，部分过于突出美颜功能的 Android 手机前置摄像头的采集帧率可能低于10fps。
 * @param {Number}                  videoBitrate    - 【字段含义】视频上行码率
 *                                                    【推荐取值】推荐设置请参考本文件前半部分 TRTCVideoResolution 定义处的注释说明
 *                                                    【特别说明】码率太低会导致视频中有很多的马赛克
 * 
 */
exports.TRTCVideoEncParam = class TRTCVideoEncParam {
    constructor() {
        this.videoResolution = TRTCVideoResolution.TRTCVideoResolution_640_360;
        this.resMode = TRTCVideoResolutionMode.TRTCVideoResolutionModeLandscape;
        this.videoFps = 15;
        this.videoBitrate = 550;
    }
};

/**
 * 5.3 网络流控相关参数
 * 
 * 网络流控相关参数，该设置决定了SDK在各种网络环境下的调控方向（比如弱网下是“保清晰”还是“保流畅”）
 * 
 * @param {TRTCVideoQosPreference} preference  - 【字段含义】弱网下是“保清晰”还是“保流畅”
 *                                               【特别说明】
 *                                                - 弱网下保流畅：在遭遇弱网环境时，画面会变得模糊，且会有较多马赛克，但可以保持流畅不卡顿
 *                                                - 弱网下保清晰：在遭遇弱网环境时，画面会尽可能保持清晰，但可能会更容易出现卡顿
 * @param {TRTCQosControlMode}     controlMode - 【字段含义】视频分辨率（云端控制 - 客户端控制）
 *                                               【特别说明】
 *                                                  - Client 模式：客户端控制模式，用于 SDK 开发内部调试，客户请勿使用
 *                                                  - Server 模式（默认）：云端控制模式，若没有特殊原因，请直接使用该模式
 * 
 */
exports.TRTCNetworkQosParam = class TRTCNetworkQosParam {
    constructor() {
        this.preference = TRTCVideoQosPreference.TRTCVideoQosPreferenceClear;
        this.controlMode = TRTCQosControlMode.TRTCQosControlModeServer;
    }
};

/**
 * 5.4 视频质量
 * 
 * 表示视频质量的好坏，通过这个数值，您可以在 UI 界面上用图标表征 userId 的通话线路质量
 * 
 * @param {String}      userId  - 用户标识
 * @param {TRTCQuality} quality - 视频质量
 * 
 */
exports.TRTCQualityInfo = class TRTCQualityInfo {
    constructor() {
        this.userId = null;
        this.quality = TRTCQuality.TRTCQuality_Unknown;
    }
};

/**
 * 5.5 音量大小
 * 
 * 表示语音音量的评估大小，通过这个数值，您可以在 UI 界面上用图标表征 userId 是否有在说话。
 * 
 * @param {String} userId - 说话者的 userId，字符编码格式是 UTF-8
 * @param {Number} volume - 说话者的音量， 取值范围0 - 100
 * 
 */
exports.TRTCVolumeInfo = class TRTCVolumeInfo {
    constructor() {
        this.userId = null;
        this.volume = 0;
    }
};

/**
 * 5.8 网络测速结果 
 * 
 * 您可以在用户进入房间前通过 TRTCCloud 的 startSpeedTest 接口进行测速 （注意：请不要在通话中调用），
 * 测速结果会每2 - 3秒钟返回一次，每次返回一个 IP 地址的测试结果。
 * 
 * 注意:
 * - quality 是内部通过评估算法测算出的网络质量，loss 越低，rtt 越小，得分也就越高。
 * - upLostRate 是指上行丢包率，例如0.3代表每向服务器发送10个数据包，可能有3个会在中途丢失。
 * - downLostRate 是指下行丢包率，例如0.2代表从服务器每收取10个数据包，可能有2个会在中途丢失。
 * - rtt 是指当前设备到腾讯云服务器的一次网络往返时间，正常数值在10ms - 100ms之间。
 * 
 * @param {String}      ip           - 服务器 IP 地址
 * @param {TRTCQuality} quality      - 网络质量，内部通过评估算法测算出的网络质量，loss 越低，rtt 越小，得分也就越高
 * @param {Number}      upLostRate   - 上行丢包率，范围是[0 - 1.0]，例如0.3代表每向服务器发送10个数据包，可能有3个会在中途丢失。
 * @param {Number}      downLostRate - 下行丢包率，范围是[0 - 1.0]，例如0.2代表从服务器每收取10个数据包，可能有2个会在中途丢失。
 * @param {Number}      rtt          - 延迟（毫秒），代表 SDK 跟服务器一来一回之间所消耗的时间，这个值越小越好，正常数值在10ms - 100ms之间。
 * 
 */
exports.TRTCSpeedTestResult = class TRTCSpeedTestResult {
    constructor() {
        this.ip = null;
        this.quality = TRTCQuality.TRTCQuality_Unknown;
        this.upLostRate = 0.0;
        this.downLostRate = 0.0;
        this.rtt = 0;
    }
};

/**
 * 记录矩形的四个点坐标
 * 
 * @param {Number} left   - 左坐标
 * @param {Number} top    - 上坐标
 * @param {Number} right  - 右坐标
 * @param {Number} bottom - 下坐标
 * 
 */
exports.Rect = class Rect {
    constructor(left = 0, top = 0, right = 0, bottom = 0) {
        this.left = left;
        this.top = top;
        this.right = right;
        this.bottom = bottom;
    }
};

/**
 * 5.9 云端混流中每一路子画面的位置信息
 * 
 * TRTCMixUser 用于指定每一路（即每一个 userId）视频画面的具体摆放位置
 * 
 * @param {String}              userId      - 参与混流的 userId
 * @param {String}              roomId      - 参与混流的 roomId，跨房流传入的实际 roomId，当前房间流传入 roomId = null
 * @param {Rect}                rect        - 图层位置坐标以及大小，左上角为坐标原点(0,0) （绝对像素值）
 * @param {Number}              rect.left   - 图层位置的左坐标
 * @param {Number}              rect.top    - 图层位置的上坐标
 * @param {Number}              rect.right  - 图层位置的右坐标
 * @param {Number}              rect.bottom - 图层位置的下坐标
 * @param {Number}              zOrder      - 图层层次（1 - 15）不可重复
 * @param {Boolean}             pureAudio   - 是否纯音频
 * @param {TRTCVideoStreamType} streamType  - 参与混合的是主路画面（TRTCVideoStreamTypeBig）或屏幕分享（TRTCVideoStreamTypeSub）画面
 * 
 */
exports.TRTCMixUser = class TRTCMixUser {
    constructor() {
        this.userId = null;
        this.roomId = null;
        this.rect = null;
        this.zOrder = 0;
        this.pureAudio = false;
        this.streamType = TRTCVideoStreamType.TRTCVideoStreamTypeBig;
    }
};

/**
 * 5.10 混流参数配置模式
 * 
 * 目前暂仅支持手动配置这一种模式，即需要指定 TRTCTranscodingConfig 的全部参数。
 * 
 * @enum {Number}
 * 
 */
const TRTCTranscodingConfigMode = exports.TRTCTranscodingConfigMode = {
    /** 未定义 */
    TRTCTranscodingConfigMode_Unknown: 0,

    /** 手动配置混流参数 */
    TRTCTranscodingConfigMode_Manual: 1,
};

/**
 * 5.11 云端混流（转码）配置
 * 
 * 包括最终编码质量和各路画面的摆放位置
 * 
 * @param {TRTCTranscodingConfigMode} mode - 【字段含义】转码 config 模式
 * @param {Number} appId - 【字段含义】腾讯云直播 AppID
 *                         【推荐取值】请在 [实时音视频控制台](https://console.cloud.tencent.com/rav) 选择已经创建的应用，单击【帐号信息】后，在“直播信息”中获取
 * @param {Number} bizId - 【字段含义】腾讯云直播 bizid
 *                         【推荐取值】请在 [实时音视频控制台](https://console.cloud.tencent.com/rav) 选择已经创建的应用，单击【帐号信息】后，在“直播信息”中获取
 * @param {Number} videoWidth   - 【字段含义】最终转码后的视频分辨率的宽度（px）
 * @param {Number} videoHeight  - 【字段含义】最终转码后的视频分辨率的高度（px）
 * @param {Number} videoBitrate - 【字段含义】最终转码后的视频分辨率的码率（kbps）
 * @param {Number} videoFramerate  - 【字段含义】最终转码后的视频分辨率的帧率（FPS）
 *                                   【推荐取值】15
 * @param {Number} videoGOP        - 【字段含义】最终转码后的视频分辨率的关键帧间隔（也被称为 GOP），单位秒
 *                                   【推荐取值】3
 * @param {Number} audioSampleRate - 【字段含义】最终转码后的音频采样率
 *                                   【推荐取值】48000
 * @param {Number} audioBitrate    - 【字段含义】最终转码后的音频码率，单位：kbps
 *                                   【推荐取值】64
 * @param {Number} audioChannels   - 【字段含义】最终转码后的音频声道数
 *                                   【推荐取值】2
 * @param {TRTCMixUser[]} mixUsersArray - 【字段含义】每一路子画面的位置信息
 * @param {Number} mixUsersArraySize    - 【字段含义】 数组 mixUsersArray 的大小
 * 
 */
exports.TRTCTranscodingConfig = class TRTCTranscodingConfig {
    constructor() {
        this.mode = TRTCTranscodingConfigMode.TRTCTranscodingConfigMode_Unknown;
        this.appId = 0;
        this.bizId = 0;
        this.videoWidth = 0;
        this.videoHeight = 0;
        this.videoBitrate = 0;
        this.videoFramerate = 0;
        this.videoGOP = 0;
        this.audioSampleRate = 0;
        this.audioBitrate = 0;
        this.audioChannels = 0;
        this.mixUsersArray = null;
        this.mixUsersArraySize = 0;
    }
};

/**
 * 5.12 CDN 旁路推流参数 
 * 
 * @param {Number} appId - 腾讯云 AppID，请在 [实时音视频控制台](https://console.cloud.tencent.com/rav) 选择已经创建的应用，单击【帐号信息】后，在“直播信息”中获取
 * @param {Number} bizId - 腾讯云直播 bizId，请在 [实时音视频控制台](https://console.cloud.tencent.com/rav) 选择已经创建的应用，单击【帐号信息】后，在“直播信息”中获取
 * @param {String} url - 旁路转推的 URL
 * 
 */
exports.TRTCPublishCDNParam = class TRTCPublishCDNParam {
    constructor() {
        this.appId = 0;
        this.bizId = 0;
        this.url = null;
    }
};

/**
 * 5.13 录音参数
 * 
 * 请正确填写参数，确保录音文件顺利生成。
 * 
 * @param {String} filePath - 【字段含义】文件路径（必填），录音文件的保存路径。该路径需要用户自行指定，请确保路径存在且可写。
 *                            【特别说明】该路径需精确到文件名及格式后缀，格式后缀决定录制文件的格式，例如：指定路径为 path/to/audio.aac，则会生成一个 AAC 格式的文件。目前支持的格式有 PCM, WAV, AAC
 * 
 */
exports.TRTCAudioRecordingParams = class TRTCAudioRecordingParams {
    constructor() {
        this.filePath = null;
    }
};

/**
 * 本地的音视频统计信息
 * 
 * @param {Number} width           - 视频宽度
 * @param {Number} height          - 视频高度
 * @param {Number} frameRate       - 帧率（fps）
 * @param {Number} videoBitrate    - 视频发送码率（Kbps）
 * @param {Number} audioSampleRate - 音频采样率（Hz）
 * @param {Number} audioBitrate    - 音频发送码率（Kbps）
 * @param {TRTCVideoStreamType} streamType - 流类型（大画面 | 小画面 | 辅路画面）
 * 
 */
exports.TRTCLocalStatistics = class TRTCLocalStatistics {
    constructor() {
        this.width = 0;
        this.height = 0;
        this.frameRate = 0;
        this.videoBitrate = 0;
        this.audioSampleRate = 0;
        this.audioBitrate = 0;
        this.streamType = TRTCVideoStreamType.TRTCVideoStreamTypeBig;
    }
};

/**
 * 远端成员的音视频统计信息
 * 
 * @param {String} userId          - 用户 ID，指定是哪个用户的视频流
 * @param {Number} finalLoss       - 该线路的总丢包率（％）
 *                                   这个值越小越好，比如：0%的丢包率代表网络很好。
 *                                   这个丢包率是该线路的 userId 从上行到服务器再到下行的总丢包率。
 *                                   如果 downLoss 为 0%, 但是 finalLoss 不为0%，说明该 userId 在上行就出现了无法恢复的丢包。
 * @param {Number} width           - 视频宽度
 * @param {Number} height          - 视频高度
 * @param {Number} frameRate       - 接收帧率（fps）
 * @param {Number} videoBitrate    - 视频码率（Kbps）
 * @param {Number} audioSampleRate - 音频采样率（Hz）
 * @param {Number} audioBitrate    - 音频码率（Kbps）
 * @param {TRTCVideoStreamType} streamType - 流类型（大画面 | 小画面 | 辅路画面）
 * 
 */
exports.TRTCRemoteStatistics = class TRTCRemoteStatistics {
    constructor() {
        this.userId = null;
        this.finalLoss = 0;
        this.width = 0;
        this.height = 0;
        this.frameRate = 0;
        this.videoBitrate = 0;
        this.audioSampleRate = 0;
        this.audioBitrate = 0;
        this.streamType = TRTCVideoStreamType.TRTCVideoStreamTypeBig;
    }
};

/**
 * 统计数据
 * 
 * @param {Number} upLoss    - C -> S 上行丢包率（％），这个值越小越好，例如，0%的丢包率代表网络很好，
 *                             而 30% 的丢包率则意味着 SDK 向服务器发送的每10个数据包中就会有3个会在上行传输中丢失。
 * @param {Number} downLoss  - S -> C 下行丢包率（％），这个值越小越好，例如，0%的丢包率代表网络很好，
 *                             而 30% 的丢包率则意味着服务器向 SDK 发送的每10个数据包中就会有3个会在下行传输中丢失。
 * @param {Number} appCpu    - 当前 App 的 CPU 使用率（％）
 * @param {Number} systemCpu - 当前系统的 CPU 使用率（％）
 * @param {Number} rtt       - 延迟（毫秒），代表 SDK 跟服务器一来一回之间所消耗的时间，这个值越小越好。
 *                             一般低于50ms的 rtt 是比较理想的情况，而高于100ms的 rtt 会引入较大的通话延时。
 *                             由于数据上下行共享一条网络连接，所以 local 和 remote 的 rtt 相同。
 * @param {Number} receivedBytes - 总接收字节数（包含信令和音视频）
 * @param {Number} sentBytes     - 总发送字节总数（包含信令和音视频）
 * @param {TRTCLocalStatistics} localStatisticsArray - 自己本地的音视频统计信息，由于可能有大画面、小画面以及辅路画面等多路的情况，所以是一个数组
 * @param {Number} localStatisticsArraySize - 数组 localStatisticsArray 的大小
 * @param {TRTCRemoteStatistics} remoteStatisticsArray - 远端成员的音视频统计信息，由于可能有大画面、小画面以及辅路画面等多路的情况，所以是一个数组
 * @param {Number} remoteStatisticsArraySize - 数组 remoteStatisticsArray 的大小
 * 
 */
exports.TRTCStatistics = class TRTCStatistics {
    constructor() {
        this.upLoss = 0;
        this.downLoss = 0;
        this.appCpu = 0;
        this.systemCpu = 0;
        this.rtt = 0;
        this.receivedBytes = 0;
        this.sentBytes = 0;
        this.localStatisticsArray = null;
        this.localStatisticsArraySize = 0;
        this.remoteStatisticsArray = null;
        this.remoteStatisticsArraySize = 0;
    }
};

