
local tdOption = tdCore(...)
local GUI = tdCore('GUI')
local L = tdCore:GetLocale('tdCore')

local Frame = tdOption:NewModule('Frame', GUI('MainFrame'):New(UIParent))
Frame.options = {}

local function iter(t, i)
    i = i + 1
    if i <= #t then
        return i, t[i].value
    end
end

function Frame:IterateOptions()
    return iter, self.options, 0
end

function Frame:GetAddonList()
    return self.options
end

function Frame:AddOption(option, title)
    local frame = option:GetFrame()
    frame:Hide()
    frame:SetParent(self)
    frame:ClearAllPoints()
    if frame:IsWidgetType('Widget') then
        frame:SetPoint('BOTTOMRIGHT', -20, 50)
        frame:SetPoint('TOPLEFT', self.addonList, 'TOPRIGHT', 10, 0)
    elseif frame:IsWidgetType('TabWidget') then
        frame:SetPoint('BOTTOMRIGHT', -19, 49)
        frame:SetPoint('TOPLEFT', self.addonList, 'TOPRIGHT', 9, 22)
    else
        error('error frame type ' .. frame:GetWidgetType())
    end
    frame:HookScript('OnShow', frame.Update)
    
    tinsert(self.options, {text = title or option:GetTitle(), value = option})
end

function Frame:GetOption()
    return self.option
end

function Frame:SetOption(option)
    self.profileWidget:Hide()
    
    for i, obj in self:IterateOptions() do
        if obj == option then
            obj:Show()
            self.option = obj
            self.addonList:SetSelected(obj)
            if obj:GetDB() then
                obj:GetDB():BackupCurrentProfile()
                self.profileButton:Enable()
            else
                self.profileButton:Disable()
            end
        else
            obj:Hide()
        end
    end
end

function Frame:IsProfileUnSave()
    for _, option in self:IterateOptions() do
        local db = option:GetDB()
        if db and db:IsProfileChanged() then
            return true
        end
    end
    return false
end

function Frame:OnAccept()
    for _, option in self:IterateOptions() do
        local db = option:GetDB()
        if db then
            db:RemoveBackupProfile()
        end
    end
end

function Frame:OnCancel()
    if not self:IsProfileUnSave() then
        return self:OnAccept()
    end
    
    self:ShowDialog(
        'Dialog',
        L['You change the configuration of some addons, you want to save ?'],
        GUI.DialogIcon.Warning,
        function()
            self:OnAccept()
        end,
        function()
            for _, option in self:IterateOptions() do
                local db = option:GetDB()
                if db and db:IsProfileChanged() then
                    db:RestoreCurrentProfile()
                    option:GetAddon():UpdateProfile()
                end
            end
        end
    )
end

function Frame:OnProfileDefault()
    local option = self:GetOption()
    if not option or not option:GetDB() then return end
    
    self:ShowDialog(
        'Dialog',
        L['Are you sure to reset the addon |cffff0000[%s]|r configuration file?']:format(option:GetTitle()) .. (option:GetAddon().__reloaduiWhileReset and L[' And will reload addon'] or ''),
        GUI.DialogIcon.Warning,
        function()
            option:GetDB():ResetProfile()
            if option:GetAddon().__reloaduiWhileReset then
                ReloadUI()
            else
                option:GetDB():BackupCurrentProfile()
                option:GetAddon():UpdateProfile()
            end
            option:RunHandle('OnProfileDefault')
        end
    )
end

function Frame:OnProfileCopy(key)
    local option = self:GetOption()
    if not option or not option:GetDB() then return end
    
    self:ShowDialog(
        'Dialog',
        L['Are you sure overwrites the current configuration file to |cffffffff[%s]|r?']:format(key),
        GUI.DialogIcon.Question,
        function()
            option:GetDB():CopyProfile(key)
            option:GetDB():BackupCurrentProfile()
            option:GetAddon():UpdateProfile()
            self.profileWidget:Update()
        end
    )
end

function Frame:OnProfileRemove(key)
    local option = self:GetOption()
    if not option or not option:GetDB() then return end
    
    self:ShowDialog(
        'Dialog',
        L['Are you sure to delete configuration file |cffffffff[%s]|r?']:format(key),
        GUI.DialogIcon.Warning,
        function()
            option:GetDB():DeleteProfile(key)
            self.profileWidget:Update()
        end
    )
end

do
    Frame:Hide()
    Frame:SetSize(800, 600)
    Frame:SetChildOrientation('HORIZONTAL')
    Frame:SetAllowEscape(true)
    Frame:SetPadding(20, -20, -50, 50)
    Frame:SetLabelText(L['Taiduo\'s Addons'])
    Frame:HookScript('OnHide', function(self)
        if self.__accept then
            self:OnAccept()
        else
            self:OnCancel()
        end
        self.__accept = nil
    end)
    
    local addonList = GUI('ListWidget'):New(Frame)
    addonList:SetWidth(180)
    addonList:SetLabelText(ADDONS)
    addonList:SetItemList(Frame:GetAddonList())
    addonList:SetSelectMode('RADIO')
    addonList:SetHorizontalArgs(180, 0, 0, 0)
    addonList:SetHandle('OnItemClick', function(self, index)
        Frame:SetOption(self:GetItemValue(index))
    end)
    addonList:Into()
    
    local profileButton = GUI('Button'):New(Frame)
    profileButton:SetWidth(130)
    profileButton:SetText(L['Profile manager'])
    profileButton:SetPoint('BOTTOMLEFT', 20, 20)
    profileButton:SetScript('OnClick', function()
        Frame:GetOption():Hide()
        Frame.profileWidget:Show()
    end)
    
    local cancelButton = GUI('Button'):New(Frame)
    cancelButton:SetText(CANCEL)
    cancelButton:SetPoint('BOTTOMRIGHT', -20, 20)
    cancelButton:SetScript('OnClick', function()
        Frame:Hide()
    end)
    
    local acceptButton = GUI('Button'):New(Frame)
    acceptButton:SetText(OKAY)
    acceptButton:SetPoint('RIGHT', cancelButton, 'LEFT', -5, 0)
    acceptButton:SetScript('OnClick', function()
        Frame.__accept = true
        Frame:Hide()
    end)
    
    local profileWidget = GUI('Widget'):New(Frame)
    profileWidget:SetPoint('BOTTOMRIGHT', -20, 50)
    profileWidget:SetPoint('TOPLEFT', addonList, 'TOPRIGHT', 10, 0)
    
    function profileWidget:Update()
        self:SetLabelText(Frame:GetOption():GetTitle() .. ' - ' .. L['Profile manager'])
        
        local list = Frame:GetOption():GetDB():GetProfileList()
        
        Frame.copyComboBox:SetItemList(list)
        Frame.deleteComboBox:SetItemList(list)
    end
    
    profileWidget:SetScript('OnShow', profileWidget.Update)
    
    local copyComboBox = GUI('ComboBox'):New(profileWidget)
    copyComboBox:SetLabelText(L['Copy Profile'])
    copyComboBox:SetValueText(L['Please choose profile ...'])
    copyComboBox:GetValueFontString():SetPoint('LEFT', 10, 0)
    copyComboBox:SetHandle('OnValueChanged', function(self, value)
        Frame:OnProfileCopy(value)
    end)
    copyComboBox:Into()
    
    local deleteComboBox = GUI('ComboBox'):New(profileWidget)
    deleteComboBox:SetLabelText(L['Remove Profile'])
    deleteComboBox:SetValueText(L['Please choose profile ...'])
    deleteComboBox:GetValueFontString():SetPoint('LEFT', 10, 0)
    deleteComboBox:SetHandle('OnValueChanged', function(self, value)
        Frame:OnProfileRemove(value)
    end)
    deleteComboBox:Into()
    
    local defaultButton = GUI('Button'):New(profileWidget)
    defaultButton:SetWidth(150)
    defaultButton:SetText(DEFAULTS)
    defaultButton:SetScript('OnClick', function()
        Frame:OnProfileDefault()
    end)
    defaultButton:Into()
    
    local returnButton = GUI('Button'):New(profileWidget)
    returnButton:SetWidth(150)
    returnButton:SetText(L['Return addon option'])
    returnButton:SetScript('OnClick', function()
        Frame.profileWidget:Hide()
        Frame:GetOption():Show()
    end)
    returnButton:Into()
    
    Frame.addonList = addonList
    Frame.profileButton = profileButton
    Frame.copyComboBox = copyComboBox
    Frame.deleteComboBox = deleteComboBox
    Frame.profileWidget = profileWidget
end
