# Command

The default command component of Hyperf provided by [hyperf/command](https://github.com/hyperf/command) component，And this component is a abstraction of [symfony/console](https://github.com/symfony/console).

# Installation

This component usually exists by default, but if you want to use it for non-Hyperf projects, you can also rely on the [hyperf/command](https://github.com/hyperf/command) component with the following command:

```bash
composer require hyperf/command
```

# Command List

Run `php bin/hyperf.php` without any arguments directly is to display the command list.

# Custom Command

## Generate a Command

If you have the [hyperf/devtool](https://github.com/hyperf/devtool) component installed, you can generate a custom command with the `gen:command` command:

```bash
php bin/hyperf.php gen:command FooCommand
```
After executing the above command, a configured `FooCommand` class will be generated in the `app/Command` folder.

### Definition of Command

There are two forms of commands that define the command class. One is defined by the `$name` property, and the other is defined by the constructor argument. We demonstrate this through code examples, assuming we want to define the command. The class command is `foo:hello`:

#### Define the command by `$name` property：

```php
<?php

declare(strict_types=1);

namespace App\Command;

use Hyperf\Command\Command as HyperfCommand;
use Hyperf\Command\Annotation\Command;

/**
 * @Command
 */
class FooCommand extends HyperfCommand
{
    /**
     * The command
     *
     * @var string
     */
    protected $name = 'foo:hello';
}
```

#### Define the command by constructor：

```php
<?php

declare(strict_types=1);

namespace App\Command;

use Hyperf\Command\Command as HyperfCommand;
use Hyperf\Command\Annotation\Command;

/**
 * @Command
 */
class FooCommand extends HyperfCommand
{
    public function __construct()
    {
        parent::__construct('foo:hello');    
    }
}
```

### Define the logic of the command

The logic that the command class actually runs depends on the `handle` method inside the code, which means that the `handle` method is the entry point to the command.

```php
<?php

declare(strict_types=1);

namespace App\Command;

use Hyperf\Command\Command as HyperfCommand;
use Hyperf\Command\Annotation\Command;

/**
 * @Command
 */
class FooCommand extends HyperfCommand
{
    /**
     * The command
     *
     * @var string
     */
    protected $name = 'foo:hello';
    
    public function handle()
    {
        // Output Hello Hyperf. in the Console via the built-in method line()
        $this->line('Hello Hyperf.', 'info');
    }
}
```

### Define the arguments of the command

When writing a command, the user's input is usually collected by `parameter` and `option`, and the `parameter` or `option` must be defined before collecting a user input.

#### Parameter

Suppose we want to define a `name` parameter, and then pass the arbitrary string such as `Hyperf` to the command and execute `php bin/hyperf.php foo:hello Hyperf` to output `Hello Hyperf`. Let's demonstrate it by code:

```php
<?php

declare(strict_types=1);

namespace App\Command;

use Hyperf\Command\Annotation\Command;
use Hyperf\Command\Command as HyperfCommand;
use Symfony\Component\Console\Input\InputArgument;

/**
 * @Command
 */
class FooCommand extends HyperfCommand
{
    /**
     * The command
     *
     * @var string
     */
    protected $name = 'foo:hello';

    public function handle()
    {
        // Get the name argument from $input
        $argument = $this->input->getArgument('name') ?? 'World';
        $this->line('Hello ' . $argument, 'info');
    }
    
    protected function getArguments()
    {
        return [
            ['name', InputArgument::OPTIONAL, 'Here is an explanation of this parameter']
        ];
    }
}
``` 

Execute `php bin/hyperf.php foo:hello Hyperf` command and we can see `Hello Hyperf` display on Console.

