///////////////////////////////////////////////////////////////////////////////
// Name:        wx/listctrl.h
// Purpose:     wxODListCtrl class
// Author:      Vadim Zeitlin
// Modified by:
// Created:     04.12.99
// RCS-ID:      $Id: listbase.h,v 1.17 2005/12/02 19:06:53 avivahl Exp $
// Copyright:   (c) wxWindows team
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

#ifndef _LISTBASE_H_
#define _LISTBASE_H_

#include <wx/colour.h>                      // wxColour
#include <wx/font.h>                        // wxFont
#include <wx/event.h>                       // wxNotifyEvent

// ----------------------------------------------------------------------------
// types
// ----------------------------------------------------------------------------

// type of compare function for wxODListCtrl sort operation
typedef int (wxCALLBACK *wxODListCtrlCompare)(long item1, long item2, long sortData);

// ----------------------------------------------------------------------------
// wxODListCtrl constants
// ----------------------------------------------------------------------------

// style flags
#define wxODLC_VRULES          0x0001
#define wxODLC_HRULES          0x0002

#define wxODLC_ICON            0x0004
#define wxODLC_SMALL_ICON      0x0008
#define wxODLC_LIST            0x0010
#define wxODLC_REPORT          0x0020

#define wxODLC_ALIGN_TOP       0x0040
#define wxODLC_ALIGN_LEFT      0x0080
#define wxODLC_AUTOARRANGE     0x0100
#define wxODLC_VIRTUAL         0x0200
#define wxODLC_EDIT_LABELS     0x0400
#define wxODLC_NO_HEADER       0x0800
#define wxODLC_NO_SORT_HEADER  0x1000
#define wxODLC_SINGLE_SEL      0x2000
#define wxODLC_SORT_ASCENDING  0x4000
#define wxODLC_SORT_DESCENDING 0x8000

#define wxODLC_MASK_TYPE       (wxODLC_ICON | wxODLC_SMALL_ICON | wxODLC_LIST | wxODLC_REPORT)
#define wxODLC_MASK_ALIGN      (wxODLC_ALIGN_TOP | wxODLC_ALIGN_LEFT)
#define wxODLC_MASK_SORT       (wxODLC_SORT_ASCENDING | wxODLC_SORT_DESCENDING)

// for compatibility only
#define wxODLC_USER_TEXT       wxODLC_VIRTUAL

// Omitted because
//  (a) too much detail
//  (b) not enough style flags
//  (c) not implemented anyhow in the generic version
//
// #define wxODLC_NO_SCROLL
// #define wxODLC_NO_LABEL_WRAP
// #define wxODLC_OWNERDRAW_FIXED
// #define wxODLC_SHOW_SEL_ALWAYS

// Mask flags to tell app/GUI what fields of wxODListItem are valid
#define wxODLIST_MASK_STATE           0x0001
#define wxODLIST_MASK_TEXT            0x0002
#define wxODLIST_MASK_IMAGE           0x0004
#define wxODLIST_MASK_DATA            0x0008
#define wxODLIST_SET_ITEM             0x0010
#define wxODLIST_MASK_WIDTH           0x0020
#define wxODLIST_MASK_FORMAT          0x0040

// State flags for indicating the state of an item
#define wxODLIST_STATE_DONTCARE       0x0000
#define wxODLIST_STATE_DROPHILITED    0x0001      // MSW only
#define wxODLIST_STATE_FOCUSED        0x0002
#define wxODLIST_STATE_SELECTED       0x0004
#define wxODLIST_STATE_CUT            0x0008      // MSW only
#define wxODLIST_STATE_DISABLED       0x0010      // OS2 only
#define wxODLIST_STATE_FILTERED       0x0020      // OS2 only
#define wxODLIST_STATE_INUSE          0x0040      // OS2 only
#define wxODLIST_STATE_PICKED         0x0080      // OS2 only
#define wxODLIST_STATE_SOURCE         0x0100      // OS2 only

// Hit test flags, used in HitTest
#define wxODLIST_HITTEST_ABOVE            0x0001  // Above the client area.
#define wxODLIST_HITTEST_BELOW            0x0002  // Below the client area.
#define wxODLIST_HITTEST_NOWHERE          0x0004  // In the client area but below the last item.
#define wxODLIST_HITTEST_ONITEMICON       0x0020  // On the bitmap associated with an item.
#define wxODLIST_HITTEST_ONITEMLABEL      0x0080  // On the label (string) associated with an item.
#define wxODLIST_HITTEST_ONITEMRIGHT      0x0100  // In the area to the right of an item.
#define wxODLIST_HITTEST_ONITEMSTATEICON  0x0200  // On the state icon for a tree view item that is in a user-defined state.
#define wxODLIST_HITTEST_TOLEFT           0x0400  // To the left of the client area.
#define wxODLIST_HITTEST_TORIGHT          0x0800  // To the right of the client area.

#define wxODLIST_HITTEST_ONITEM (wxODLIST_HITTEST_ONITEMICON | wxODLIST_HITTEST_ONITEMLABEL | wxODLIST_HITTEST_ONITEMSTATEICON)

// Flags for GetNextItem (MSW only except wxODLIST_NEXT_ALL)
enum
{
    wxODLIST_NEXT_ABOVE,          // Searches for an item above the specified item
    wxODLIST_NEXT_ALL,            // Searches for subsequent item by index
    wxODLIST_NEXT_BELOW,          // Searches for an item below the specified item
    wxODLIST_NEXT_LEFT,           // Searches for an item to the left of the specified item
    wxODLIST_NEXT_RIGHT           // Searches for an item to the right of the specified item
};

// Alignment flags for Arrange (MSW only except wxODLIST_ALIGN_LEFT)
enum
{
    wxODLIST_ALIGN_DEFAULT,
    wxODLIST_ALIGN_LEFT,
    wxODLIST_ALIGN_TOP,
    wxODLIST_ALIGN_SNAP_TO_GRID
};

// Column format (MSW only except wxODLIST_FORMAT_LEFT)
enum wxODListColumnFormat
{
    wxODLIST_FORMAT_LEFT,
    wxODLIST_FORMAT_RIGHT,
    wxODLIST_FORMAT_CENTRE,
    wxODLIST_FORMAT_CENTER = wxODLIST_FORMAT_CENTRE
};

// Autosize values for SetColumnWidth
enum
{
    wxODLIST_AUTOSIZE = -1,
    wxODLIST_AUTOSIZE_USEHEADER = -2      // partly supported by generic version
};

// Flag values for GetItemRect
enum
{
    wxODLIST_RECT_BOUNDS,
    wxODLIST_RECT_ICON,
    wxODLIST_RECT_LABEL
};

// Flag values for FindItem (MSW only)
enum
{
    wxODLIST_FIND_UP,
    wxODLIST_FIND_DOWN,
    wxODLIST_FIND_LEFT,
    wxODLIST_FIND_RIGHT
};

// ----------------------------------------------------------------------------
// wxODListItemAttr: a structure containing the visual attributes of an item
// ----------------------------------------------------------------------------

class wxODListItemAttr
{
public:
    // ctors
    wxODListItemAttr() { }
    wxODListItemAttr(const wxColour& colText,
                   const wxColour& colBack,
                   const wxFont& font)
        : m_colText(colText), m_colBack(colBack), m_font(font) { }

    // setters
    void SetTextColour(const wxColour& colText) { m_colText = colText; }
    void SetBackgroundColour(const wxColour& colBack) { m_colBack = colBack; }
    void SetFont(const wxFont& font) { m_font = font; }

    // accessors
    bool HasTextColour() const { return m_colText.Ok(); }
    bool HasBackgroundColour() const { return m_colBack.Ok(); }
    bool HasFont() const { return m_font.Ok(); }

    const wxColour& GetTextColour() const { return m_colText; }
    const wxColour& GetBackgroundColour() const { return m_colBack; }
    const wxFont& GetFont() const { return m_font; }

private:
    wxColour m_colText,
             m_colBack;
    wxFont   m_font;
};

// ----------------------------------------------------------------------------
// wxODListItem: the item or column info, used to exchange data with wxODListCtrl
// ----------------------------------------------------------------------------

class wxODListItem : public wxObject
{
public:
    wxODListItem() { Init(); m_attr = NULL; }
    wxODListItem(const wxODListItem& item)
        : wxObject(),
          m_mask(item.m_mask),
          m_itemId(item.m_itemId),
          m_col(item.m_col),
          m_state(item.m_state),
          m_stateMask(item.m_stateMask),
          m_text(item.m_text),
          m_image(item.m_image),
          m_data(item.m_data),
          m_format(item.m_format),
          m_width(item.m_width),
          m_attr(NULL)
    {
        // copy list item attributes
        if( item.HasAttributes() )
            m_attr = new wxODListItemAttr(*item.GetAttributes());
    }
    virtual ~wxODListItem() { delete m_attr; }

    // resetting
    void Clear() { Init(); m_text.clear(); ClearAttributes(); }
    void ClearAttributes() { if ( m_attr ) { delete m_attr; m_attr = NULL; } }

    // setters
    void SetMask(long mask)
        { m_mask = mask; }
    void SetId(long id)
        { m_itemId = id; }
    void SetColumn(int col)
        { m_col = col; }
    void SetState(long state)
        { m_mask |= wxODLIST_MASK_STATE; m_state = state; m_stateMask |= state; }
    void SetStateMask(long stateMask)
        { m_stateMask = stateMask; }
    void SetText(const wxString& text)
        { m_mask |= wxODLIST_MASK_TEXT; m_text = text; }
    void SetImage(int image)
        { m_mask |= wxODLIST_MASK_IMAGE; m_image = image; }
    void SetData(long data)
        { m_mask |= wxODLIST_MASK_DATA; m_data = data; }
    void SetData(void *data)
        { m_mask |= wxODLIST_MASK_DATA; m_data = (long)data; }

    void SetWidth(int width)
        { m_mask |= wxODLIST_MASK_WIDTH; m_width = width; }
    void SetAlign(wxODListColumnFormat align)
        { m_mask |= wxODLIST_MASK_FORMAT; m_format = align; }

    void SetTextColour(const wxColour& colText)
        { Attributes().SetTextColour(colText); }
    void SetBackgroundColour(const wxColour& colBack)
        { Attributes().SetBackgroundColour(colBack); }
    void SetFont(const wxFont& font)
        { Attributes().SetFont(font); }

    // accessors
    long GetMask() const { return m_mask; }
    long GetId() const { return m_itemId; }
    int GetColumn() const { return m_col; }
    long GetState() const { return m_state & m_stateMask; }
    const wxString& GetText() const { return m_text; }
    int GetImage() const { return m_image; }
    long GetData() const { return m_data; }

    int GetWidth() const { return m_width; }
    wxODListColumnFormat GetAlign() const { return (wxODListColumnFormat)m_format; }

    wxODListItemAttr *GetAttributes() const { return m_attr; }
    bool HasAttributes() const { return m_attr != NULL; }

    wxColour GetTextColour() const
        { return HasAttributes() ? m_attr->GetTextColour() : wxNullColour; }
    wxColour GetBackgroundColour() const
        { return HasAttributes() ? m_attr->GetBackgroundColour()
                                 : wxNullColour; }
    wxFont GetFont() const
        { return HasAttributes() ? m_attr->GetFont() : wxNullFont; }

    // this conversion is necessary to make old code using GetItem() to
    // compile
    operator long() const { return m_itemId; }

    // these members are public for compatibility

    long            m_mask;     // Indicates what fields are valid
    long            m_itemId;   // The zero-based item position
    int             m_col;      // Zero-based column, if in report mode
    long            m_state;    // The state of the item
    long            m_stateMask;// Which flags of m_state are valid (uses same flags)
    wxString        m_text;     // The label/header text
    int             m_image;    // The zero-based index into an image list
    long            m_data;     // App-defined data

    // For columns only
    int             m_format;   // left, right, centre
    int             m_width;    // width of column

#ifdef __WXPM__
    int             m_miniImage; // handle to the mini image for OS/2
#endif

protected:
    // creates m_attr if we don't have it yet
    wxODListItemAttr& Attributes()
    {
        if ( !m_attr )
            m_attr = new wxODListItemAttr;

        return *m_attr;
    }

    void Init()
    {
        m_mask = 0;
        m_itemId = 0;
        m_col = 0;
        m_state = 0;
        m_stateMask = 0;
        m_image = 0;
        m_data = 0;

        m_format = wxODLIST_FORMAT_CENTRE;
        m_width = 0;
    }

    wxODListItemAttr *m_attr;     // optional pointer to the items style

private:
    // VZ: this is strange, we have a copy ctor but not operator=(), why?
    wxODListItem& operator=(const wxODListItem& item);

    DECLARE_DYNAMIC_CLASS(wxODListItem)
};

// ----------------------------------------------------------------------------
// wxODListEvent - the event class for the wxODListCtrl notifications
// ----------------------------------------------------------------------------

class wxODListEvent : public wxNotifyEvent
{
public:
    wxODListEvent(wxEventType commandType = wxEVT_NULL, int id = 0)
        : wxNotifyEvent(commandType, id)
        , m_code(0)
        , m_oldItemIndex(0)
        , m_itemIndex(0)
        , m_col(0)
        , m_pointDrag()
        , m_item()
        { }

    wxODListEvent(const wxODListEvent& event)
        : wxNotifyEvent(event)
        , m_code(event.m_code)
        , m_oldItemIndex(event.m_oldItemIndex)
        , m_itemIndex(event.m_itemIndex)
        , m_col(event.m_col)
        , m_pointDrag(event.m_pointDrag)
        , m_item(event.m_item)
        { }

    int GetKeyCode() const { return m_code; }
    long GetIndex() const { return m_itemIndex; }
    int GetColumn() const { return m_col; }
    wxPoint GetPoint() const { return m_pointDrag; }
    const wxString& GetLabel() const { return m_item.m_text; }
    const wxString& GetText() const { return m_item.m_text; }
    int GetImage() const { return m_item.m_image; }
    long GetData() const { return m_item.m_data; }
    long GetMask() const { return m_item.m_mask; }
    const wxODListItem& GetItem() const { return m_item; }

    // for wxEVT_COMMAND_ODLIST_CACHE_HINT only
    long GetCacheFrom() const { return m_oldItemIndex; }
    long GetCacheTo() const { return m_itemIndex; }

    virtual wxEvent *Clone() const { return new wxODListEvent(*this); }

//protected: -- not for backwards compatibility
    int           m_code;
    long          m_oldItemIndex; // only for wxEVT_COMMAND_ODLIST_CACHE_HINT
    long          m_itemIndex;
    int           m_col;
    wxPoint       m_pointDrag;

    wxODListItem    m_item;

private:
    DECLARE_DYNAMIC_CLASS(wxODListEvent)
};

// ----------------------------------------------------------------------------
// wxODListCtrl event macros
// ----------------------------------------------------------------------------

BEGIN_DECLARE_EVENT_TYPES()
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_BEGIN_DRAG, 700)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_BEGIN_RDRAG, 701)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_BEGIN_LABEL_EDIT, 702)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_END_LABEL_EDIT, 703)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_DELETE_ITEM, 704)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_DELETE_ALL_ITEMS, 705)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_GET_INFO, 706)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_SET_INFO, 707)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_ITEM_SELECTED, 708)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_ITEM_DESELECTED, 709)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_KEY_DOWN, 710)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_INSERT_ITEM, 711)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_COL_CLICK, 712)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_ITEM_RIGHT_CLICK, 713)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_ITEM_MIDDLE_CLICK, 714)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_ITEM_ACTIVATED, 715)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_CACHE_HINT, 716)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_COL_RIGHT_CLICK, 717)
	DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_COL_MIDDLE_CLICK, 718)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_COL_BEGIN_DRAG, 719)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_COL_DRAGGING, 720)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_COL_END_DRAG, 721)
    DECLARE_EVENT_TYPE(wxEVT_COMMAND_ODLIST_ITEM_FOCUSED, 722)
END_DECLARE_EVENT_TYPES()

typedef void (wxEvtHandler::*wxODListEventFunction)(wxODListEvent&);

#define EVT_ODLIST_BEGIN_DRAG(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_BEGIN_DRAG, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_BEGIN_RDRAG(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_BEGIN_RDRAG, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_BEGIN_LABEL_EDIT(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_BEGIN_LABEL_EDIT, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_END_LABEL_EDIT(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_END_LABEL_EDIT, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_DELETE_ITEM(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_DELETE_ITEM, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_DELETE_ALL_ITEMS(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_DELETE_ALL_ITEMS, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_GET_INFO(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_GET_INFO, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_SET_INFO(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_SET_INFO, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_KEY_DOWN(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_KEY_DOWN, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_INSERT_ITEM(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_INSERT_ITEM, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),

#define EVT_ODLIST_COL_CLICK(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_COL_CLICK, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_COL_RIGHT_CLICK(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_COL_RIGHT_CLICK, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_COL_MIDDLE_CLICK(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_COL_MIDDLE_CLICK, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_COL_BEGIN_DRAG(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_COL_BEGIN_DRAG, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_COL_DRAGGING(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_COL_DRAGGING, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_COL_END_DRAG(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_COL_END_DRAG, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),

#define EVT_ODLIST_ITEM_SELECTED(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_ITEM_SELECTED, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_ITEM_DESELECTED(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_ITEM_DESELECTED, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, NULL ),
#define EVT_ODLIST_ITEM_RIGHT_CLICK(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_ITEM_RIGHT_CLICK, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, (wxObject *) NULL ),
#define EVT_ODLIST_ITEM_MIDDLE_CLICK(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_ITEM_MIDDLE_CLICK, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, (wxObject *) NULL ),
#define EVT_ODLIST_ITEM_ACTIVATED(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_ITEM_ACTIVATED, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, (wxObject *) NULL ),
#define EVT_ODLIST_ITEM_FOCUSED(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_ITEM_FOCUSED, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, (wxObject *) NULL ),

#define EVT_ODLIST_CACHE_HINT(id, fn) DECLARE_EVENT_TABLE_ENTRY( wxEVT_COMMAND_ODLIST_CACHE_HINT, id, -1, (wxObjectEventFunction) (wxEventFunction) (wxODListEventFunction) & fn, (wxObject *) NULL ),

#endif // _LISTBASE_H_
