﻿// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview New tab page
 * This is the main code for the new tab page used by touch-enabled Chrome
 * browsers.  For now this is still a prototype.
 */

// Use an anonymous function to enable strict mode just for this file (which
// will be concatenated with other files when embedded in Chrome
cr.define('ntp', function () {
  'use strict';

  /**
  * The CardSlider object to use for changing app pages.
  * @type {CardSlider|undefined}
  */
  var cardSlider;

  /**
  * The 'page-list' element.
  * @type {!Element|undefined}
  */
  var pageList;

  /**
  * A list of all 'tile-page' elements.
  * @type {!NodeList|undefined}
  */
  var tilePages;

  /**
  * The Most Visited page.
  * @type {!Element|undefined}
  */
  var mostVisitedPage;

  /**
  * A list of all 'apps-page' elements.
  * @type {!NodeList|undefined}
  */
  var appsPages;

  var navPage;

  /**
  * The 'dots-list' element.
  * @type {!Element|undefined}
  */
  var dotList;

  /**
  * A list of all 'dots' elements.
  * @type {!NodeList|undefined}
  */
  var dots;

  /**
  * The 'trash' element.  Note that technically this is unnecessary,
  * JavaScript creates the object for us based on the id.  But I don't want
  * to rely on the ID being the same, and JSCompiler doesn't know about it.
  * @type {!Element|undefined}
  */
  var trash;

  /**
  * The time in milliseconds for most transitions.  This should match what's
  * in new_tab.css.  Unfortunately there's no better way to try to time
  * something to occur until after a transition has completed.
  * @type {number}
  * @const
  */
  var DEFAULT_TRANSITION_TIME = 500;

  /**
  * The left and right paging buttons.
  * @type {!Element|undefined}
  */
  var pageSwitcherStart;
  var pageSwitcherEnd;

  /**
  * The type of page that is currently shown. The value is a numerical ID.
  * @type {number}
  */
  var shownPage = 0;

  /**
  * The index of the page that is currently shown, within the page type.
  * For example if the third Apps page is showing, this will be 2.
  * @type {number}
  */
  var shownPageIndex = 0;
  /**
  * If non-null, this is the ID of the app to highlight to the user the next
  * time getAppsCallback runs. "Highlight" in this case means to switch to
  * the page and run the new tile animation.
  * @type {String}
  */
  var highlightAppId = null;

  var curDotList = [];
  /**
  * Invoked at startup once the DOM is available to initialize the app.
  */
  function initialize() {
    // Load the current theme colors.
    localStorage.thumbMarginValue = localStorage.thumbMarginValue || ((screen.availWidth <= 1366 && screen.availHeight <= 800)? 8:13);
    themeChanged();
    appendSearchFormAndHotWords();

    HotKeyword.init($$('#search-kw'), loadTimeData.getBoolean('enableHotSuggestion'));

    var hideSearchSgList = function (e) {
      if (HotKeyword.opening) {
        return;
      }
      var targetNode = e.target;
      var searchSg = $("search-sg");
      if (targetNode.id != "search-sg" && (targetNode == window || !searchSg.contains(targetNode))) {
          searchSg.style.display = "none";
          $$('#search-hotword').classList.remove('open');
      } else {
        HotKeyword.opening = true;
      }
    }

    document.addEventListener('mouseup', function(){
      HotKeyword.opening = false;
    });
    document.addEventListener('mousedown', hideSearchSgList);
    $$('#search-kw').addEventListener('blur', hideSearchSgList);
    window.addEventListener('blur', hideSearchSgList);

    $('search-sg').addEventListener('mouseover', function(e){
      var current = e.target;
      if (current.tagName == 'A') {
        var list = this.querySelectorAll('a');
        for (var i=0; i<list.length; i++) {
          list[i].className = '';
        }
        current.classList.add('search-sg-select');
      }
    });

    dotList = getRequiredElement('dot-list');
    pageList = getRequiredElement('page-list');
    trash = getRequiredElement('trash');
    trash.hidden = true;
    shownPage = loadTimeData.getInteger('shown_page_type');
    if (shownPage == 4096 || shownPage == 5120) {
      shownPage = 15360;
    }
    if ((shownPage == loadTimeData.getInteger('nav_page_id') && !loadTimeData.getBoolean('hao360'))) {
      shownPage = 1024;
    }

    shownPageIndex = loadTimeData.getInteger('shown_page_index');

    // Request data on the apps so we can fill them in.
    // Note that this is kicked off asynchronously.  'getAppsCallback' will be
    // invoked at some point after this function returns.
    chrome.send('getApps');


    // Prevent touch events from triggering any sort of native scrolling
    document.addEventListener('touchmove', function (e) {
      e.preventDefault();
    }, true);

    dots = dotList.getElementsByClassName('dot');
    tilePages = pageList.getElementsByClassName('tile-page');
    appsPages = pageList.getElementsByClassName('apps-page');

    pageSwitcherStart = getRequiredElement('page-switcher-start');
    ntp.initializePageSwitcher(pageSwitcherStart);
    pageSwitcherEnd = getRequiredElement('page-switcher-end');
    ntp.initializePageSwitcher(pageSwitcherEnd);

    // Initialize the cardSlider without any cards at the moment
    var sliderFrame = getRequiredElement('card-slider-frame');
    cardSlider = new cr.ui.CardSlider(sliderFrame, pageList, sliderFrame.offsetWidth);
    cardSlider.initialize();

    var setContainerXY = function (setId) {
      if ($(setId).style.display == "block") {
        var con = $(setId);
        con.style.left = Math.max((document.body.offsetWidth - con.clientWidth) / 2, 0) + "px";
        con.style.top = Math.max((document.body.offsetHeight - con.clientHeight) / 2  - (setId == 'add-dot-container'? 40 : 0), 0) + "px";
      }
    };

    // Ensure the slider is resized appropriately with the window
    window.addEventListener('resize', function () {
      cardSlider.resize(sliderFrame.offsetWidth);
      /* add by 360 */
      setTimeout(updatePageSwitchers, 10);
      resetTilePagePaddingTop();
      setTimeout(updateGridShadow, 300);
      updateSearchMenu();
      updateHotItemAndSuggesst();
      setTimeout(function(){
        updateSearchMenu();
        updateHotItemAndSuggesst();
      }, 310);
      setTimeout(function(){
          var titles = document.querySelectorAll('.title');
          var firstSite = document.querySelector(".most-visited");
          if (firstSite) {
            for (var i = 0; i < titles.length; i++) {
              titles[i].style.width = firstSite.clientWidth - 75 + "px";
            }
          }
      });
      setContainerXY("user-add-container");
      setPopPos("add-dot-container");
    });

    function savePageSelected(page) {
      if (!document.documentElement.classList.contains('starting-up')) {
        if (page.classList.contains('apps-page')) {
          shownPage = loadTimeData.getInteger('apps_page_id');
          shownPageIndex = getAppsPageIndex(page);
        } else if (page.classList.contains('most-visited-page')) {
          shownPage = loadTimeData.getInteger('most_visited_page_id');
          shownPageIndex = 0;
        } else if (page.classList.contains('bookmarks-page')) {
          shownPage = loadTimeData.getInteger('bookmarks_page_id');
          shownPageIndex = 0;
        } else if (page.classList.contains('nav-page')) {
          shownPage = loadTimeData.getInteger('nav_page_id');
          shownPageIndex = 0;
        } else {
          console.error('unknown page selected');
        }
        var curDot = dotList.querySelector('.selected');
        if (curDot && curDot.querySelector(".channel_new")) {
          var newInfo = curDot.querySelectorAll(".channel_new");
          for(var i=0;i<newInfo.length;i++)
            curDot.removeChild(newInfo[i]);
        }
        chrome.send('pageSelected', [shownPage, shownPageIndex]);
        loadTimeData.data_.shown_page_type = shownPage;
      }
    }
    // Handle the page being changed
    pageList.addEventListener(
        'cardSlider:card_changed',
        function (e) {
          var page = e.cardSlider.currentCardValue;
          updateSearchForm(page);

          // Don't change shownPage until startup is done (and page changes actually
          // reflect user actions).
          savePageSelected(page);

          var curDot = dotList.querySelector('.selected'); //edit by lichao3@360.cn(clean jquery)

          if (curDot) {
            curDot.classList.remove('selected');
          }

          var newPageIndex = e.cardSlider.currentCard;
          setCurDotList();
          if (curDotList[newPageIndex]) curDotList[newPageIndex].classList.add('selected');

          updatePageSwitchers();
          updateGridShadow();
          // If an app was being dragged, move it to the end of the new page
          //if (draggingAppContainer)
          //  appsPages[newPageIndex].appendChild(draggingAppContainer);
          /* add by 360 from cr15*/
          if (document.documentElement.classList.contains('starting-up'))
              document.documentElement.classList.remove('starting-up');
        }
    );

    // Add a drag handler to the body (for drags that don't land on an existing
    // app)
    document.addEventListener(Grabber.EventType.DRAG_ENTER, appDragEnter);

    // Handle dropping an app anywhere other than on the trash
    document.addEventListener(Grabber.EventType.DROP, appDrop);

    // Add handles to manage the transition into/out-of rearrange mode
    // Note that we assume here that we only use a Grabber for moving apps,
    // so ANY GRAB event means we're enterring rearrange mode.
    sliderFrame.addEventListener(Grabber.EventType.GRAB, enterRearrangeMode);
    sliderFrame.addEventListener(Grabber.EventType.RELEASE, leaveRearrangeMode);

    // Add handlers for the tash can
    trash.addEventListener(Grabber.EventType.DRAG_ENTER, function (e) {
      trash.classList.add('hover');
      e.grabbedElement.classList.add('trashing');
      e.stopPropagation();
    });
    trash.addEventListener(Grabber.EventType.DRAG_LEAVE, function (e) {
      e.grabbedElement.classList.remove('trashing');
      trash.classList.remove('hover');
    });
    trash.addEventListener(Grabber.EventType.DROP, appTrash);

    cr.ui.decorate($('recently-closed-menu-button'), ntp.RecentMenuButton);
    // hide recently close button
    // chrome.send('getRecentlyClosedTabs');

    //mostVisitedPage = new ntp.MostVisitedPage('Most Visited');
    mostVisitedPage = new ntp.MostVisitedPage(loadTimeData.getString('mostvisited'));
    appendTilePage(mostVisitedPage);
    chrome.send('getMostVisited');
    var make_channel_dot = function () {
      if(!loadTimeData.data_.use360international){
        if (!dotList.querySelector('li[title="' + loadTimeData.getString('Hao360') + '"]')) {
          navPage = new ntp.NavPage(loadTimeData.getString('Hao360'));
          appendTilePage(navPage);
        }
        dot_display(loadTimeData.getString('Hao360'), loadTimeData.getBoolean("hao360"));
      }
      var appsPage = getAppsPage();
      appendTilePage(appsPage);
      dot_display(loadTimeData.getString('appDefaultPageName'), true, true);
      // Tell the slider about the pages
      updateSliderCards();
    };
    make_channel_dot();

    // add by huangximing 2013-05-25
    if(!loadTimeData.data_.use360international){
    var visited_count_div = document.getElementById('visited_count_div'),
    visited_count_i = visited_count_div.querySelectorAll('i'),
    v_len = visited_count_i.length;

    for (var i = 0; i < v_len; i++) {
      visited_count_i[i].addEventListener('click', function () {
        if (this.classList.contains('selected')) return;
        var cur_i = visited_count_div.querySelector('.selected');
        if (cur_i) cur_i.classList.remove('selected');
        this.classList.add('selected');
        var num = parseInt(this.getAttribute('data-num'));
        $('thumbnailCountSelectText').innerText = num;
        loadTimeData.data_.mostVisitedCount = num;
        chrome.send('saveThumbnailCount', [num]);
        chrome.send('getMostVisited');
        setTimeout(updateSearchForm, 300);
      });
    }
    var setSearchRange = function (_this) {
      if (_this.value == 0) {
        _this.setAttribute('class', 'rangeSwitch range-on');
        loadTimeData.data_.showSearchEngineBar = true;
        $('search-form').style.display = 'block';
      } else {
        _this.setAttribute('class', 'rangeSwitch range-off');
        loadTimeData.data_.showSearchEngineBar = false;
        $('search-form').style.display = 'none';
      }
      updateSearchForm();
      setTimeout(updateSearchForm, 300);
    },
    show_search_mousedown_val = null;
    $('showSearchEngineRange').addEventListener('change', function () {
      setSearchRange(this);
    });
    $("showSearchEngineRange").addEventListener("mouseup", function () {
      if (show_search_mousedown_val == this.value) this.value = show_search_mousedown_val == 1 ? 0 : 1;
      setSearchRange(this);
      chrome.send('saveShowSearchEngine', [this.value]);
    });
    $("showSearchEngineRange").addEventListener("mousedown", function () {
      show_search_mousedown_val = this.value;
  });

  var showHao360 = function (el_show) {
        var is_show = (el_show.value == 1) ? false : true;
        el_show.setAttribute('class', is_show ? 'rangeSwitch range-on' :  'rangeSwitch range-off');
        chrome.send('saveChannel', [is_show, 'hao360']);
        loadTimeData.data_['hao360'] = is_show;
        dot_display(loadTimeData.getString('Hao360'), is_show);
        updateSliderCards();
        if (is_show) {
          ntp.getCardSlider().selectCard(1, true);
        }
  };
  var show_hao360 = null;
  $("showHao360Range").addEventListener("change",function () {
     showHao360(this);
  });
  $("showHao360Range").addEventListener("mouseup",function () {
     if (show_hao360 == this.value) this.value = show_hao360 == 1 ? 0 : 1;
     showHao360(this);
  });
  $("showHao360Range").addEventListener("mousedown",function () {
     show_hao360 = this.value;
  });

    var showFavoriteChange = function(_this) {
      if (_this.value == 0) {
        _this.setAttribute('class', 'rangeSwitch range-on');
      } else {
        _this.setAttribute('class', 'rangeSwitch range-off');
      }
      chrome.send('setPref', ['se.browser.new_tab.show_bookmark_bar', _this.value == 0]);
    };
    var show_favorite_mousedown_val = null;
    $("showFavorite").addEventListener("change",function () {
      showFavoriteChange(this);
    });
    $("showFavorite").addEventListener("mousedown", function () {
        show_favorite_mousedown_val = this.value;
    });
    $("showFavorite").addEventListener("mouseup", function () {
      if (show_favorite_mousedown_val == this.value) this.value = show_favorite_mousedown_val == 1 ? 0 : 1;
      showFavoriteChange(this);
    });


    chrome.send('getPref', ['se.browser.new_tab.show_bookmark_bar']);

      var dot_add = $('setting');
      dot_add.addEventListener('click', function (e) {
        e.stopPropagation();
        $("add-dot-container").style.display = $("add-dot-container").style.display == 'block' ? "none" : "block";

        var cachedMarginValue = parseInt(localStorage.thumbMarginValue) || 0;
        var thumbnailMarginRange = $("thumbnailMarginRange");
        thumbnailMarginRange.value = cachedMarginValue;
        $("thumbnailMarginRangeText").innerText = cachedMarginValue;
        $("thumbnailMarginbar").style.width = (cachedMarginValue / 16) * (thumbnailMarginRange.clientWidth - 17) + "px";

        var mostcount = loadTimeData.getInteger('mostVisitedCount');
        var count_i = document.querySelector('#visited_count_div i[data-num="' + mostcount + '"]');
        if (count_i) count_i.classList.add('selected');
        else document.querySelector('#visited_count_div i[data-num="12"]').classList.add('selected');
        $('thumbnailCountSelectText').innerText = mostcount;

        var showSearch = loadTimeData.getBoolean('showSearchEngineBar');
        $('showSearchEngineRange').value = showSearch ? 0 : 1;
        if (!showSearch) $('showSearchEngineRange').classList.add('range-off');

        var showHao360 = loadTimeData.getBoolean('hao360');
        $("showHao360Range").value = showHao360 ? 0 : 1;
        if (!showHao360) $('showHao360Range').classList.add('range-off');


        setPopPos("add-dot-container");
      });

    }
    // add by huangximing end
    document.querySelector("#add-dot-container .c_body").addEventListener("click", function (e) {
        chrome.send("clearCacheNTP",["add click"]);
    });

    /* add by 360 from cr15*/
    if (document.documentElement.classList.contains('starting-up'))
      document.documentElement.classList.remove('starting-up');

    if (loadTimeData.valueExists('serverpromo')) {
      var serverpromo = loadTimeData.getBoolean('serverpromo');
      showNotification(parseHtmlSubset(serverpromo), [], function () {
        chrome.send('closePromo');
      }, 60000);
    }
  }

  function onPrefChanged(list) {
    switch(list[0]) {
      case 'se.browser.new_tab.show_bookmark_bar':
        var showFavorite = list[1];
        $('showFavorite').value = showFavorite ? 0 : 1;
        if (!showFavorite) $('showFavorite').classList.add('range-off');
        break;
    }
  }

   function setPopPos(elId) {
        if ($(elId).style.display == "block") {
        var con = $(elId);
         var offset = $("setting").getBoundingClientRect();
        con.style.left = (offset.left - 222)+ "px";
        con.style.top = (offset.top + 20) + "px";
      }
   }

  function updateSearchForm(page) {
    if(loadTimeData.data_.use360international) return;
    ntp.TilePage.prototype.updateTopMargin_(page);
  }

  function appendSearchFormAndHotWords() {
    var searchForm_ = document.createElement('form');
    searchForm_.id = "search-form";
    searchForm_.setAttribute("type", "get");
    if (localStorage['searchEngine'] || !localStorage['searchEngine2']) {
      if (localStorage['searchEngine']) {
        var old_setting = JSON.parse(localStorage['searchEngine']);
        var default_webpage = old_setting.type;
      } else { var default_webpage = searchTypeMaps.webpage[0]; }
      var default_str = { active: "webpage", catlist: { webpage: default_webpage, news: searchTypeMaps.news[0], video: searchTypeMaps.video[0], image: searchTypeMaps.image[0], music: searchTypeMaps.music[0], map: searchTypeMaps.map[0], wenda: searchTypeMaps.wenda[0]} };
      localStorage['searchEngine2'] = JSON.stringify(default_str);
      localStorage.removeItem("searchEngine");
    }

    var searchEngine2 = JSON.parse(localStorage['searchEngine2']);
    searchEngine2.active = "webpage";
    localStorage["searchEngine2"] = JSON.stringify(searchEngine2);
    var active_Type = searchEngine2.active; //webapp

    var searchEngine = searchEngines[searchEngine2.catlist[active_Type]]; //
    searchForm_.setAttribute('action', searchEngine.url);

    var search_cat = document.createElement("div");
    search_cat.className = "search-cat";
    var search_types = searchTypeNames;
    for (var p in search_types) {
      //            console.log(p);
      var search_item = document.createElement("a");
      search_item.setAttribute("catname", p);
      search_item.innerText = searchTypeNames[p];
      if (p == active_Type) search_item.className = "on";
      search_cat.appendChild(search_item);
    }
    search_cat.addEventListener("click", function (e) {
      var targetNode = e.target || e.srcElement;
      if (targetNode.nodeName.toLowerCase() === 'a') {
        var alist = document.querySelectorAll(".search-cat a");
        for (var j = 0; j < alist.length; j++) {
          alist[j].classList.remove("on");
        }
        targetNode.classList.add("on");
        var current_cat = targetNode.getAttribute("catname");
        var currentSetting = JSON.parse(localStorage["searchEngine2"]);
        var current_search = currentSetting.catlist[current_cat];
        document.querySelector("#search-switch a").className = current_search;
        var searchItem = searchEngines[current_search];
        $('search-form').setAttribute('action', searchItem.url);
        $('search-kw').setAttribute('name', searchItem.key);
        //$('search-kw').setAttribute('placeholder', searchItem.desc);
        var from_input = $("from_input");
        if (searchItem.type.indexOf("so") > -1) {
          from_input.name = "src";
          from_input.value = "360chrome_newtab_search";
        } else if (searchItem.type.indexOf("google") > -1) {
          from_input.name = "client";
          from_input.value = "aff-cs-360chromium";
        } else {
          from_input.name = "";
          from_input.value = "";
        }
        currentSetting.active = current_cat;
        localStorage["searchEngine2"] = JSON.stringify(currentSetting);
      }
    });

    var search_switch_div = document.createElement('div');
    search_switch_div.id = 'search-switch';
    search_switch_div.className = "search-engine";
    search_switch_div.innerHTML = '<a class="' + searchEngine.type + '"></a>'
        + '<div class="search-arr"><span></span></div>';
    search_switch_div.addEventListener("click", toggleSearchMenu);

    var search_kw_input = document.createElement('input');
    search_kw_input.id = "search-kw";
    search_kw_input.name = searchEngine.key;
    search_kw_input.type = "text"; //can be search
    search_kw_input.placeholder = "这里也可以搜索";//searchEngine.desc;
    search_kw_input.autocomplete = "off";

    search_kw_input.addEventListener("keydown", searchSuggention);
    search_kw_input.addEventListener("keyup", searchSuggention);

    var toggle_hotword = document.createElement('div');
    toggle_hotword.id = 'search-hotword';

    var ipt_div = document.createElement('div');
    ipt_div.className = 'ipt';
    ipt_div.appendChild(search_switch_div);
    ipt_div.appendChild(search_kw_input);
    ipt_div.appendChild(toggle_hotword);

    var search_btn = document.createElement('div');
    search_btn.className = "btn";
    search_btn.id = "s_btn";
    search_btn.onclick = function (e) {
      var words = encodeURIComponent(search_kw_input.value.trim().xssFilter());
      var actionUrl = searchForm_.getAttribute('action');
      if (actionUrl == searchEngines.news_so.url && !words)
        var searchUrl = "http://sh.qihoo.com";
      else
        var searchUrl = actionUrl + "&" + search_kw_input.name + "=" + words;
      chrome.send("openURL", [searchUrl]);
    };

    var from_input = document.createElement("input");
    from_input.type = "hidden";
    from_input.id = "from_input";
    if (searchEngine.type.indexOf("so") > -1) {
      from_input.name = "src";
      from_input.value = "360chrome_newtab_search";
    } else if (searchEngine.type.indexOf("google") > -1) {
      from_input.name = "client";
      from_input.value = "aff-cs-360chromium";
    } else {
      from_input.name = "";
      from_input.value = "";
    }

    var search_div = document.createElement('div');
    search_div.className = 'search';

    search_div.appendChild(ipt_div);
    search_div.appendChild(search_btn);
    search_div.appendChild(from_input);

    searchForm_.appendChild(search_cat);
    searchForm_.appendChild(search_div);

    $("search_test").appendChild(searchForm_);
    searchForm_.style.display = loadTimeData.getBoolean("showSearchEngineBar") ? "block" : "none";
    searchForm_.style.opacity = "0";

    if (searchEngine.type == 'so') {
      $('s_btn').classList.remove('btn-normal');
    } else {
      $('s_btn').classList.add('btn-normal');
    }

  }

  function toggleSearchMenu() {
    event.stopPropagation();
    var searchMenu = $('search-menu');
    searchMenu.innerHTML = "";
    var searchEngine2 = JSON.parse(localStorage['searchEngine2']);
    var active_Type = searchEngine2.active; //webapp
    var type_list = searchTypeMaps[active_Type];
    for (var i = 0; i < type_list.length; i++) {
      searchMenu.innerHTML += '<li><a href="#" class="' + type_list[i] + '">' + searchEngines[type_list[i]].desc + '</a></li>'
    }
    updateSearchMenu();
    searchMenu.classList.toggle("toggle-display");
    document.querySelector(".search-arr >span").classList.toggle("toggle-triangle");
  }

  function updateSearchMenu() {
    var searchMenu = $('search-menu');
    var searchSwitch = $("search-switch");
    var offset = searchSwitch.getBoundingClientRect();
    searchMenu.style.left = offset.left - 8 + "px";
    searchMenu.style.top = offset.top + 24 + "px";    
  }

  function getKeyword(ele) {
      if (ele.querySelector('em')) {
          return ele.querySelector('span').innerText;
      }
      return ele.innerText;
  }

  function searchSuggention() {
    var searchInput = $("search-kw");
    var keyCode = event.keyCode;
    var keywords = searchInput.value.trim();
    var filterPos = keywords.indexOf("?");
    if (filterPos > -1)
      keywords = keywords.substr(0, filterPos);

    var searchSg = $("search-sg");
    if (!keywords && searchSg.style.display == 'none') {
        searchSg.style.display = "none";
        searchSg.innerHTML = "";
        return;
    }

    if (keyCode == 13) {
      if (event.type == "keyup") return;
      event.preventDefault();
      event.stopPropagation();
      var searchBtn = document.querySelector(".search .btn");
      searchBtn.click();
    }
    if (keyCode == 38 || keyCode == 40) {
      if (event.type == "keyup") return;
      event.preventDefault();
      var sgList = document.querySelectorAll("#search-sg a");
      var sgLen = sgList.length;
      if (!sgList || !sgLen) {
        return;
      }
      $("search-sg").style.display = "block";
      var current = document.querySelector("#search-sg .search-sg-select");
      if (keyCode == "38") {//up
        if (!current) {
          current = sgList[sgLen - 1];
          current.classList.add("search-sg-select");
          searchInput.value = getKeyword(current);
          return;
        }
        current.classList.remove("search-sg-select");
        var nextSpan = current.previousSibling;
        if (!nextSpan || nextSpan.tagName.toLowerCase() != "a") {
          nextSpan = sgList[sgLen - 1];
        }
        nextSpan.classList.add("search-sg-select");
        searchInput.value = getKeyword(nextSpan);
      } else if (keyCode == "40") {//down
        if (!current) {
          current = sgList[0];
          current.classList.add("search-sg-select");
          searchInput.value = getKeyword(current);
          return;
        }
        current.classList.remove("search-sg-select");
        var nextSpan = current.nextSibling;
        if (!nextSpan || nextSpan.tagName.toLowerCase() != "a") {
          nextSpan = sgList[0];
        }
        nextSpan.classList.add("search-sg-select");
        searchInput.value = getKeyword(nextSpan);
      }
    } else {
      if (!(keyCode >= 16 && keyCode <= 18)) {
        jsonp({
          url: "http://sug.so.360.cn/suggest/word?callback=suggest_so&encodein=utf-8&encodeout=utf-8&word=" + keywords,
          callbackName: 'suggest_so',
          onsuccess: function (data) {
            suggest_so(data);
          }
        });
      }
    }
    if (keyCode == 37 || keyCode == 39) {
      event.stopPropagation();
    }
  }

  var HotKeyword = function() {
    var keywordData;
    var input, toggle, layer, _enabled;
    function toggleSug(e) {
      if (!_enabled) {
        return;
      }
      if (e && e.button !=0) {
        return;
      }
      clearTimeout(window.sug_hide_timeout);
      if (!toggle.classList.contains('open')) {
        input.focus();
        if(render()){
          var offset = input.getBoundingClientRect();
          var iptDiv = $$("#search-form .ipt").getBoundingClientRect();
          layer.style.left = iptDiv.left - 8 + "px";
          layer.style.top = offset.top + input.clientHeight - 5 + "px";
          layer.classList.add('hot-keyword');
          layer.style.display = 'block';
        }
        HotKeyword.opening = true;
      } else {
        toggle.classList.remove('open');
        layer.style.display = 'none';
      }
      e && e.stopPropagation();
    }
    function render() {
      var cat = 'webpage';
      var list = keywordData && keywordData[cat];
      if (!list) {
        return false;
      }
      var searchUrl = $("search-form").getAttribute('action') + "&" + input.name + "=";
      searchUrl = searchUrl.replace(/src=.*?(?=&|$)/, 'src=360chrome_newtab_hotkeyword');
      var ul = layer;
      var sb = [];
      list.forEach(function(item, i) {
        var link = item.link || searchUrl + encodeURIComponent(item.keyword || item.text);
        sb.push('<a' + (link ? ' href="' + link + '"': '') + '><em class="hot">' + (i + 1) + '</em><span class="' + (item.new == '1' ? 'new': '') + '">' + item.text + '</span></a>');
      });
      ul.innerHTML = sb.join('');
      toggle.classList.add('open');
      return true;
    }
    return {
      init: function(ele, enabled) {
        input = ele;
        toggle = $$('#search-hotword');
        layer = $$('#search-sg');
        HotKeyword.enable(enabled);
        input.addEventListener('mousedown', function(e) {
          if (!_enabled) {
            return;
          }
          if (e.button != 0) {
            return;
          }
          if (this.value == '') {
            toggleSug();
          }
          e.stopPropagation();
        });
        toggle.addEventListener('mousedown', toggleSug);
        toggle.addEventListener('mouseup', function(){
          HotKeyword.opening = false;
        });
        input.addEventListener('mouseup', function(){
          HotKeyword.opening = false;
        });
      },
      enable: function(enabled) {
        _enabled = enabled;

        if (_enabled) {
          DC.get('http://site.browser.360.cn/sword.php?callback=?', {
            rn: Date.now(),
            v: '360chrome8'
          },
          function(ret) {
            keywordData = ret && ret.data;
            toggle.style.display = 'block';
          });
        } else {
          toggle.style.display = 'none';
          layer.style.display = 'none';
        }
      }
    };
  } ();

  var DC = function () {
      var _key = '__ajax_cache';
      var expires = 1000 * 60 * 60;
      var storage = {
          get: function (key) {
              try {
                  return JSON.parse(localStorage[key] || '{}');
              } catch (e) {
                  return {};
              }
          },
          set: function (key, data) {
              localStorage[key] = JSON.stringify(data);
          },
          getCache: function (key, expire) {
              var cache = storage.get(_key)[key];
              expire = expire || expires;
              if (cache && (Date.now() - cache['expires'] < expire)) {
                  return cache['data'];
              }
              return null;
          },
          setCache: function (key, val) {
              var data = storage.get(_key);
              data[key] = {
                  data: val,
                  expires: Date.now()
              };
              storage.set(_key, data);
          }
      };

      return {
          storage: storage,
          get: function (url, data, success, expire, cacheCallback) {
              var cache;
              if (cache = storage.getCache(url, expire)) {
                  if (cacheCallback !== false) {
                      success(cache);
                  }
              } else {
                  jsonp({
                      url: url.replace('callback=?', 'callback=DC_callback'),
                      callbackName: 'DC_callback',
                      onsuccess: function (ret) {
                          storage.setCache(url, ret);
                          success(ret);
                      }
                  });
                  // $.getJSON(url, data, function(ret) {
                  //   storage.setCache(url, ret);
                  //   success(ret);
                  // });
              }
          }
      };
  }();

  function updateCurrentWallpaperInfo() {
    var currentWrapper = localStorage["currentWrapper"] ? JSON.parse(localStorage["currentWrapper"]) : {};
    currentWrapper.type = "l";
    currentWrapper.url = "";
    localStorage["currentWrapper"] = JSON.stringify(currentWrapper);
  }

  var firstAppData;
  function getAppsPage() {
    var appsPage = new ntp.AppsPage(loadTimeData.getString('appDefaultPageName'));
    appsPage.addEventListener('removed', function (e) {
      if (window._myAppsCount_ < 1) {
        var pageNo = Math.max(0, Math.min(cardSlider.currentCard, tilePages.length - 1));
        removePage(pageNo);
      }
    });
    return appsPage;
  }
  /**
  * Callback invoked by chrome with the apps available.
  *
  * Note that calls to this function can occur at any time, not just in
  * response to a getApps request. For example, when a user installs/uninstalls
  * an app on another synchronized devices.
  * @param {Object} data An object with all the data on available
  *        applications.
  */
  function getAppsCallback(data) {
    //console.log('getAppsCallback: ', data);
    // Clear any existing apps pages and dots.
    // TODO(rbyers): It might be nice to preserve animation of dots after an
    // uninstall. Could we re-use the existing page and dot elements?  It seems
    // unfortunate to have Chrome send us the entire apps list after an
    // uninstall.
    window._myAppsCount_ = 0;
    for(var i=0;i<appsPages.length;i++){
        var page = appsPages[i];
        page.querySelector(".tile-grid").innerHTML = '';
    }
    /*
    while (appsPages.length > 0) {
    var page = appsPages[0];
    var dot = page.navigationDot;

    page.tearDown();
    page.parentNode.removeChild(page);
    dot.parentNode.removeChild(dot);
    }
    */
    // Get the array of apps and add any special synthesized entries
    var apps = data.apps;
    for (var i = 0; i < apps.length; i++) {
      if (apps[i].is_webstore) {
        firstAppData = apps[i];
        break;
      }
    }
    if (apps.length <= 1) {
      if(appsPages.length<=0){
        var appsPage = getAppsPage();
        appendTilePage(appsPage);
      }
      appsPages[0].appendApp(firstAppData);
      if(!loadTimeData.data_.use360international) firstAppData = null;
      dot_display(loadTimeData.getString('appDefaultPageName'), false);
      updateSliderCards();
      return;
    }
    window._myAppsCount_ = apps.length - 1;
    function getChar(page_index) {
      var code = (page_index || 0) + 97;
      code = Math.max(code, 97);
      code = Math.min(code, 122);
      return String.fromCharCode(code);
    }
    // Sort by launch index
    apps.sort(function (a, b) {
      var a_ord = getChar(a.page_index) + a.app_launch_ordinal;
      var b_ord = getChar(b.page_index) + b.app_launch_ordinal;
      return a_ord > b_ord ? 1 :
          a_ord < b_ord ? -1 : 0;
    });

    // An app to animate (in case it was just installed).
    var highlightApp;

    // Add the apps, creating pages as necessary
    for (var i = 0; i < apps.length; i++) {
      var app = apps[i];
      // all in first page
      app.page_index = 0;
      var pageIndex = (app.page_index || 0);
      while (pageIndex >= appsPages.length) {
        var origPageCount = appsPages.length;
        //appendTilePage(new ntp.AppsPage('Apps'));
        var appsPage = getAppsPage();
        appendTilePage(appsPage);

        // Confirm that appsPages is a live object, updated when a new page is
        // added (otherwise we'd have an infinite loop)
        assert(appsPages.length == origPageCount + 1, 'expected new page');
      }

      if (app.id == highlightAppId)
        highlightApp = app;
      else
        appsPages[pageIndex].appendApp(app);
    }
    dot_display(loadTimeData.getString('appDefaultPageName'), true);
    // Tell the slider about the pages
    updateSliderCards();

    if (highlightApp)
      appAdded(highlightApp, true);

    // Mark the current page
    setCurDotList();
    if (curDotList[cardSlider.currentCard]) curDotList[cardSlider.currentCard].classList.add('selected');

    var winHeight = window.innerHeight;
    document.querySelectorAll('.apps-page')[0].lastChild.style.paddingTop = winHeight <= 615 ? '40px' : (winHeight <= 747 ? '60px' : '100px');
  }


  /**
  * Called by chrome when a new app has been added to chrome or has been
  * enabled if previously disabled.
  * @param {Object} appData A data structure full of relevant information for
  *     the app.
  */
  function appAdded(appData, opt_highlight) {
    if (appData.id == highlightAppId) {
      opt_highlight = true;
      highlightAppId = null;
    }

    appData.page_index = 0;
    var pageIndex = appData.page_index || 0;

    if (pageIndex >= appsPages.length) {
      while (pageIndex >= appsPages.length) {
        var appsPage = getAppsPage();
        appendTilePage(appsPage);
      }
      if (firstAppData) {
        appsPages[pageIndex].appendApp(firstAppData);
      }
    }

    conClose('add-dot-container');
    if ($('dot_add')) dotList.removeChild($('dot_add'));
    window._myAppsCount_++;
    dot_display(loadTimeData.getString('appDefaultPageName'), true);
    updateSliderCards();
    var page = appsPages[pageIndex];
    var app = $(appData.id);
    if (app)
      app.replaceAppData(appData);
    else
      page.appendApp(appData, opt_highlight);
  }

  /**
  * Sets that an app should be highlighted if it is added. Called right before
  * appAdded for new installs.
  */
  function setAppToBeHighlighted(appId) {
    highlightAppId = appId;
  }

  /**
  * Called by chrome when an existing app has been disabled or
  * removed/uninstalled from chrome.
  * @param {Object} appData A data structure full of relevant information for
  *     the app.
  * @param {boolean} isUninstall True if the app is being uninstalled;
  *     false if the app is being disabled.
  */
  function appRemoved(appData, isUninstall) {
    var app = $(appData.id);
    assert(app, 'trying to remove an app that doesn\'t exist');

    if (!isUninstall)
      app.replaceAppData(appData);
    else {
      var appsPage = getAppsPage();
      if (window._myAppsCount_ <= 1) {
        dot_display(loadTimeData.getString('appDefaultPageName'), false);
        updateSliderCards();
        app.remove();
      } else {
        app.remove();
      }
    }
    conClose('add-dot-container');
    if ($('dot_add')) dotList.removeChild($('dot_add'));
    window._myAppsCount_--;
  }

  /**
  * Given a theme resource name, construct a URL for it.
  * @param {string} resourceName The name of the resource.
  * @return {string} A url which can be used to load the resource.
  */
  function getThemeUrl(resourceName) {
    return 'chrome://theme/' + resourceName;
  }

  /**
  * Callback invoked by chrome whenever an app preference changes.
  * @param {Object} data An object with all the data on available
  *     applications.
  */
  function appsPrefChangeCallback(data) {
    for (var i = 0; i < data.apps.length; ++i) {
      var app = $(data.apps[i].id);
      if (app) {
        app.appData = data.apps[i];
      }
    }
  }

  /**
  * Listener for offline status change events. Updates apps that are
  * not offline-enabled to be grayscale if the browser is offline.
  */
  function updateOfflineEnabledApps() {
    var apps = document.querySelectorAll('.app');
    for (var i = 0; i < apps.length; ++i) {
      if (apps[i].appData.enabled && !apps[i].appData.offline_enabled) {
        apps[i].setIcon();
        apps[i].loadIcon();
      }
    }
  }

  /**
  * Make a synthesized app object representing the chrome web store.  It seems
  * like this could just as easily come from the back-end, and then would
  * support being rearranged, etc.
  * @return {Object} The app object as would be sent from the webui back-end.
  */
  function makeWebstoreApp() {
    return {
      id: '',   // Empty ID signifies this is a special synthesized app
      page_index: 0,
      app_launch_index: -1,   // always first
      name: loadTimeData.getString('web_store_title'),
      launch_url: loadTimeData.getString('web_store_url'),
      icon_big: getThemeUrl('IDR_WEBSTORE_ICON')
    };
  }


  function getCardSlider() {
    return cardSlider;
  }
  function setCurDotList() {
    curDotList = [];
    var doListE = dotList.querySelectorAll('.dot:not(.dot-add)');
    if(doListE){
    doListE.foreach(function (e, k) {
      if (e.style.display != 'none') {
        curDotList.push(e);
      }
    });
    }
  }

  function dot_display(name, is_show, width_0) {
    var doDot = dotList.querySelector('.dot[title="' + name + '"]');
    if (doDot) {
      doDot.style.display = is_show ? 'inline-block' : 'none';
      if (width_0) doDot.setAttribute('hidden', 'true');
      else doDot.removeAttribute('hidden');
      setPopPos("add-dot-container");
    }
  }
  /**
  * Invoked whenever the pages in apps-page-list have changed so that
  * the Slider knows about the new elements.
  */
  function updateSliderCards() {
    var tmpTilePages = [], _currentDotIndex = 0, i = 0;
    var dotListE = dotList.querySelectorAll('.dot:not(.dot-add)');
    if(dotListE){
    dotListE.foreach(function (e, k) {
      if (e.style.display != 'none') {
        if (tilePages[k]) tilePages[k].removeAttribute('hidden');
        else {
          dotList.removeChild(e);
          return;
        }
        if (e.classList.contains('selected')) _currentDotIndex = i;
        tmpTilePages.push(tilePages[k]);
        i++;
    } else {
        if (tilePages[k]){
            tilePages[k].setAttribute('hidden', true);
            tilePages[k].classList.remove('selected-card');
        }
        if (e.classList.contains('selected') && i > 0) _currentDotIndex = i - 1;
      }
    });
    }
    cardSlider.currentCard = _currentDotIndex;
    var pageNo = Math.max(0, Math.min(cardSlider.currentCard,
                                      tmpTilePages.length - 1));

    cardSlider.setCards(Array.prototype.slice.call(tmpTilePages), pageNo);

    switch (shownPage) {
      /*
      case templateData['bookmarks_page_id']:
      */
      case loadTimeData.getInteger('apps_page_id'):
        cardSlider.selectCardByValue(appsPages[Math.min(shownPageIndex, appsPages.length - 1)]);
        break;
      /*
      cardSlider.selectCardByValue(bookmarksPage);
      break;
      */
      case loadTimeData.getInteger('most_visited_page_id'):
        cardSlider.selectCardByValue(mostVisitedPage);
        break;

      case loadTimeData.getInteger('nav_page_id'):
        cardSlider.selectCardByValue(navPage);
        break;
    }
    setCurDotList();
    if (curDotList[cardSlider.currentCard]) curDotList[cardSlider.currentCard].classList.add('selected');

    updateSearchForm();
  }

  function updateGridShadow() {
    updateSearchForm();
  }

  function updateHotItemAndSuggesst() {
    var hotItems = $('hot-items');
    var searchInput = document.querySelector("#search-kw");
    var offset = searchInput.getBoundingClientRect();
    if (!hotItems.classList.contains("hidden")) {
      hotItems.style.left = offset.left + "px";
      hotItems.style.top = offset.top + searchInput.clientHeight + 5 + "px";
    }
    var searchSg = $("search-sg");
    var iptDiv = document.querySelector("#search-form .ipt").getBoundingClientRect();
    if (searchSg.style.display == "block") {
      searchSg.style.left = iptDiv.left - 8 + "px";
      searchSg.style.top = offset.top + searchInput.clientHeight - 5 + "px";
    }
  }
  /**
  * Appends a tile page (for apps or most visited).
  *
  * @param {TilePage} page The page element.
  * @param {boolean=} opt_animate If true, add the class 'new' to the created
  *        dot.
  */
  function appendTilePage(page, opt_animate) {
    pageList.appendChild(page);
    /* add by 360 */
    // redraw scrollbars
    page.queueUpdateScrollbars_();

    // Make a deep copy of the dot template to add a new one.
    var newDot = new ntp.NavDot(page, page.pageName);
    if (opt_animate)
      newDot.classList.add('new');

    dotList.appendChild(newDot);
    page.navigationDot = newDot;
  }


  function insertAfter(newElement, targetElement) {
    var parent = targetElement.parentNode;
    if (parent.lastChild == targetElement) {
      parent.appendChild(newElement);
    } else {
      parent.insertBefore(newElement, targetElement.nextSibling);
    }
  }
  /**
  * Search an elements ancestor chain for the nearest element that is a member
  * of the specified class.
  * @param {!Element} element The element to start searching from.
  * @param {string} className The name of the class to locate.
  * @return {Element} The first ancestor of the specified class or null.
  */
  function getParentByClassName(element, className) {
    for (var e = element; e; e = e.parentElement) {
      if (e.classList.contains(className))
        return e;
    }
    return null;
  }

  /**
  * The container where the app currently being dragged came from.
  * @type {!Element|undefined}
  */
  var draggingAppContainer;

  /**
  * The apps-page that the app currently being dragged camed from.
  * @type {!Element|undefined}
  */
  var draggingAppOriginalPage;

  /**
  * The element that was originally after the app currently being dragged (or
  * null if it was the last on the page).
  * @type {!Element|undefined}
  */
  var draggingAppOriginalPosition;

  /**
  * Invoked when app dragging begins.
  * @param {Grabber.Event} e The event from the Grabber indicating the drag.
  */
  function appDragStart(e) {
    // Pull the element out to the sliderFrame using fixed positioning. This
    // ensures that the app is not affected (remains under the finger) if the
    // slider changes cards and is translated.  An alternate approach would be
    // to use fixed positioning for the slider (so that changes to its position
    // don't affect children that aren't positioned relative to it), but we
    // don't yet have GPU acceleration for this.
    var element = e.grabbedElement;

    var pos = element.getBoundingClientRect();
    element.style.webkitTransform = '';

    element.style.position = 'fixed';
    // Don't want to zoom around the middle since the left/top co-ordinates
    // are post-transform values.
    element.style.webkitTransformOrigin = 'left top';
    element.style.left = pos.left + 'px';
    element.style.top = pos.top + 'px';

    // Keep track of what app is being dragged and where it came from
    assert(!draggingAppContainer, 'got DRAG_START without DRAG_END');
    draggingAppContainer = element.parentNode;
    assert(draggingAppContainer.classList.contains('app-container'));
    draggingAppOriginalPosition = draggingAppContainer.nextSibling;
    draggingAppOriginalPage = draggingAppContainer.parentNode;

    // Move the app out of the container
    // Note that appendChild also removes the element from its current parent.
    sliderFrame.appendChild(element);
  }

  /**
  * Invoked when app dragging terminates (either successfully or not)
  * @param {Grabber.Event} e The event from the Grabber.
  */
  function appDragEnd(e) {
    // Stop floating the app
    var appBeingDragged = e.grabbedElement;
    assert(appBeingDragged.classList.contains('app'));
    appBeingDragged.style.position = '';
    appBeingDragged.style.webkitTransformOrigin = '';
    appBeingDragged.style.left = '';
    appBeingDragged.style.top = '';

    // Ensure the trash can is not active (we won't necessarily get a DRAG_LEAVE
    // for it - eg. if we drop on it, or the drag is cancelled)
    trash.classList.remove('hover');
    appBeingDragged.classList.remove('trashing');

    // If we have an active drag (i.e. it wasn't aborted by an app update)
    if (draggingAppContainer) {
      // Put the app back into it's container
      if (appBeingDragged.parentNode != draggingAppContainer)
        draggingAppContainer.appendChild(appBeingDragged);

      // If we care about the container's original position
      if (draggingAppOriginalPage) {
        // Then put the container back where it came from
        if (draggingAppOriginalPosition) {
          draggingAppOriginalPage.insertBefore(draggingAppContainer,
                                               draggingAppOriginalPosition);
        } else {
          draggingAppOriginalPage.appendChild(draggingAppContainer);
        }
      }
    }

    draggingAppContainer = undefined;
    draggingAppOriginalPage = undefined;
    draggingAppOriginalPosition = undefined;
  }

  /**
  * Invoked when an app is dragged over another app.  Updates the DOM to affect
  * the rearrangement (but doesn't commit the change until the app is dropped).
  * @param {Grabber.Event} e The event from the Grabber indicating the drag.
  */
  function appDragEnter(e) {
    assert(draggingAppContainer, 'expected stored container');
    var sourceContainer = draggingAppContainer;

    // Ensure enter events delivered to an app-container don't also get
    // delivered to the document.
    e.stopPropagation();

    var curPage = appsPages[cardSlider.currentCard];
    var followingContainer = null;

    // If we dragged over a specific app, determine which one to insert before
    if (e.currentTarget != document) {

      // Start by assuming we'll insert the app before the one dragged over
      followingContainer = e.currentTarget;
      assert(followingContainer.classList.contains('app-container'),
             'expected drag over container');
      assert(followingContainer.parentNode == curPage);
      if (followingContainer == draggingAppContainer)
        return;

      // But if it's after the current container position then we'll need to
      // move ahead by one to account for the container being removed.
      if (curPage == draggingAppContainer.parentNode) {
        for (var c = draggingAppContainer; c; c = c.nextElementSibling) {
          if (c == followingContainer) {
            followingContainer = followingContainer.nextElementSibling;
            break;
          }
        }
      }
    }

    // Move the container to the appropriate place on the page
    curPage.insertBefore(draggingAppContainer, followingContainer);
  }

  /**
  * Invoked when an app is dropped on the trash
  * @param {Grabber.Event} e The event from the Grabber indicating the drop.
  */
  function appTrash(e) {
    var appElement = e.grabbedElement;
    assert(appElement.classList.contains('app'));
    var appId = appElement.getAttribute('app-id');
    assert(appId);

    // Mark this drop as handled so that the catch-all drop handler
    // on the document doesn't see this event.
    e.stopPropagation();

    // Tell chrome to uninstall the app (prompting the user)
    chrome.send('uninstallApp', [appId]);
  }

  /**
  * Called when an app is dropped anywhere other than the trash can.  Commits
  * any movement that has occurred.
  * @param {Grabber.Event} e The event from the Grabber indicating the drop.
  */
  function appDrop(e) {
    if (!draggingAppContainer)
    // Drag was aborted (eg. due to an app update) - do nothing
      return;

    // If the app is dropped back into it's original position then do nothing
    assert(draggingAppOriginalPage);
    if (draggingAppContainer.parentNode == draggingAppOriginalPage &&
        draggingAppContainer.nextSibling == draggingAppOriginalPosition)
      return;

    // Determine which app was being dragged
    var appElement = e.grabbedElement;
    assert(appElement.classList.contains('app'));
    var appId = appElement.getAttribute('app-id');
    assert(appId);

    // Update the page index for the app if it's changed.  This doesn't trigger
    // a call to getAppsCallback so we want to do it before reorderApps
    var pageIndex = cardSlider.currentCard;
    assert(pageIndex >= 0 && pageIndex < appsPages.length,
           'page number out of range');
    if (appsPages[pageIndex] != draggingAppOriginalPage)
      chrome.send('setPageIndex', [appId, pageIndex]);

    // Put the app being dragged back into it's container
    draggingAppContainer.appendChild(appElement);

    // Create a list of all appIds in the order now present in the DOM
    var appIds = [];
    for (var page = 0; page < appsPages.length; page++) {
      var appsOnPage = appsPages[page].getElementsByClassName('app');
      for (var i = 0; i < appsOnPage.length; i++) {
        var id = appsOnPage[i].getAttribute('app-id');
        if (id)
          appIds.push(id);
      }
    }

    // We are going to commit this repositioning - clear the original position
    draggingAppOriginalPage = undefined;
    draggingAppOriginalPosition = undefined;

    // Tell chrome to update its database to persist this new order of apps This
    // will cause getAppsCallback to be invoked and the apps to be redrawn.
    chrome.send('reorderApps', [appId, appIds]);
    appMoved = true;
  }

  /**
  * Set to true if we're currently in rearrange mode and an app has
  * been successfully dropped to a new location.  This indicates that
  * a getAppsCallback call is pending and we can rely on the DOM being
  * updated by that.
  * @type {boolean}
  */
  var appMoved = false;

  /**
  * Invoked whenever some app is grabbed
  * @param {Grabber.Event} e The Grabber Grab event.
  */
  function enterRearrangeMode(e) {
    // Stop the slider from sliding for this touch
    cardSlider.cancelTouch();

    // Add an extra blank page in case the user wants to create a new page
    appendTilePage(new ntp.AppsPage(''), true);
    var pageAdded = appsPages.length - 1;
    window.setTimeout(function () {
      dots[pageAdded].classList.remove('new');
    }, 0);

    updateSliderCards();

    // Cause the dot-list to grow
    getRequiredElement('footer').classList.add('rearrange-mode');

    assert(!appMoved, 'appMoved should not be set yet');
  }

  /**
  * Invoked whenever some app is released
  * @param {Grabber.Event} e The Grabber RELEASE event.
  */
  function leaveRearrangeMode(e) {
    // Return the dot-list to normal
    getRequiredElement('footer').classList.remove('rearrange-mode');

    // If we didn't successfully re-arrange an app, then we won't be
    // refreshing the app view in getAppCallback and need to explicitly remove
    // the extra empty page we added.  We don't want to do this in the normal
    // case because if we did actually drop an app there, we want to retain that
    // page as our current page number.
    if (!appMoved) {
      assert(appsPages[appsPages.length - 1].
             getElementsByClassName('app-container').length == 0,
             'Last app page should be empty');
      removePage(appsPages.length - 1);
    }
    appMoved = false;
  }

  /**
  * Remove the page with the specified index and update the slider.
  * @param {number} pageNo The index of the page to remove.
  */
  function removePage(pageNo) {
    pageList.removeChild(tilePages[pageNo]);

    // Remove the corresponding dot
    // Need to give it a chance to animate though
    var dot = dots[pageNo];
    dot.classList.add('new');
    window.setTimeout(function () {
      // If we've re-created the apps (eg. because an app was uninstalled) then
      // we will have removed the old dots from the document already, so skip.
      if (dot.parentNode)
        dot.parentNode.removeChild(dot);
    }, DEFAULT_TRANSITION_TIME);

    updateSliderCards();
  }

  // TODO(estade): rename newtab.css to new_tab_theme.css
  function themeChanged(hasAttribution) {
    $('themecss').href = 'chrome://theme/css/new_tab_theme.css?' + Date.now();
    if (typeof hasAttribution != 'undefined')
      document.documentElement.setAttribute('hasattribution', hasAttribution);
    updateAttribution();
    chrome.send('getWallpaper');
  }

  /**
  * Attributes the attribution image at the bottom left.
  */
  function updateAttribution() {
    var attribution = $('attribution');
    if (document.documentElement.getAttribute('hasattribution') == 'true') {
      $('attribution-img').src =
          'chrome://theme/IDR_THEME_NTP_ATTRIBUTION?' + Date.now();
      attribution.hidden = false;
    } else {
      attribution.hidden = true;
    }

  }

  function setRecentlyClosedTabs(dataItems) {
    $('recently-closed-menu-button').dataItems = dataItems;
  }

  function setMostVisitedPages(data, hasBlacklistedUrls) {
    var newData = [];
    data.forEach(function(item){
        if (item.url) {
            newData.push(item);
        }
    });
    mostVisitedPage.data = newData;
    setTimeout(updatePageSwitchers, 0);
  }

  function updateSiteLogo(url, uselogo, defaultlogo) {
    if (url) {
      if (!mostVisitedPage || !mostVisitedPage.data) return;
      var curIndex = -1;
      for (var i = 0; i < mostVisitedPage.data.length; i++) {
        if (mostVisitedPage.data[i].url == url) {
          curIndex = i;
          break;
        }
      }
      if (curIndex < 0) return;
      var tileList = document.getElementsByClassName("tile real");
      var currentTile = tileList[curIndex].querySelector('.thumbnail');
      currentTile.classList.remove("add-loading");
      mostVisitedPage.flushSelf(curIndex, uselogo, defaultlogo);
    }
  }

  function siteLogoReady(url, ready, uselogo, defaultlogo) {
    console.log(url, ready, mostVisitedPage, mostVisitedPage.data);
    if (mostVisitedPage && mostVisitedPage.data) {
      mostVisitedPage.data[currentTileIndex].pinned = true;
      mostVisitedPage.data[currentTileIndex].filler = false;
      mostVisitedPage.data[currentTileIndex].url = url;
      if (currentTitle)
        mostVisitedPage.data[currentTileIndex].title = currentTitle;
      else
        mostVisitedPage.data[currentTileIndex].title = url;
      if (ready) {
          mostVisitedPage.flushSelf(currentTileIndex, uselogo, defaultlogo);
        var tileList = document.getElementsByClassName("tile real");
        var currentTile = tileList[currentTileIndex].querySelector('.thumbnail');
        currentTile.classList.remove("add-loading");
      } else mostVisitedPage.flushSelf(currentTileIndex, uselogo, defaultlogo);
    }
  }
  function loadWallpaper(args) {
    document.querySelector("html").style.backgroundImage = "url(chrome://theme/IDR_THEME_NTP_BACKGROUND?" + Date.now() + ")";
    // change backgroundImage can't take effect immediately unless add this statement, by zhangle
    document.body.focus();
    if(args && args.length == 5){
      var installed = args[0];
      var x = args[1];
      var y = args[2];
      var width = args[3];
      var height = args[4];
    if (installed == true) {
      updateWallpaperOffset(x, y);
      var searchTypes = document.querySelectorAll(".search-cat a");
      searchTypes.foreach(function (e, i, a) {
        e.classList.add("haswrapper");
      });
      $("setting").classList.add("setting-glass");
      $("recently-closed-menu-button").classList.add("haswrapper");
    } else {
      var searchTypes = document.querySelectorAll(".search-cat a");
      searchTypes.foreach(function (e, i, a) {
        e.classList.remove("haswrapper");
      });
      $("setting").classList.remove("setting-glass");
      $("recently-closed-menu-button").classList.remove("haswrapper");
      }
      if(width != 0 && height != 0){
        document.querySelector("html").style.backgroundSize = width + 'px ' + height + 'px';
      }
    }
  }
  function updateWallpaperOffset(x, y) {
    //        console.log("updateWallpaperOffset" + (-x) + "px " + (-y) + "px");
    document.querySelector("html").style.backgroundPosition = (-x) + "px " + (-y) + "px";
}

  function applySuccess() {
    var starDetails = document.querySelectorAll(".star-detail,.star-special-item");
    starDetails.foreach(function (e, i, a) {
      var loading = e.querySelector(".star-bg-loading");
      if (loading) e.removeChild(loading);
    });
    clearTimeout(wrapperTimeOut);
    var searchTypes = document.querySelectorAll(".search-cat a");
    searchTypes.foreach(function (e, i, a) {
      e.classList.add("haswrapper");
    });
    $("setting").classList.add("setting-glass");
    $("recently-closed-menu-button").classList.add("haswrapper");
    cacheNTP()
  }

  function updateUrlList(data, container) {
    container.innerHTML = "";
    var dataLen = data.length;
    var aTpl = document.createElement("a");
    for (var i = 0; i < dataLen; i++) {
      if (data[i].url.indexOf("chrome://") > -1)
        continue;
      var aEl = aTpl.cloneNode(false);
      var faviconUrl = data[i].faviconUrl || 'chrome://favicon/' + data[i].url;
      aEl.href = data[i].url;
      aEl.style.backgroundImage = url(faviconUrl);
      aEl.innerText = data[i].title;
      aEl.onclick = (function (item) {
        return function () {
          $("new_title").value = item.title;
          $("new_url").value = item.url;
          var allElements = container.querySelectorAll("a");
          for (var j = 0; j < allElements.length; j++) {
            allElements[j].classList.remove("url-selected");
          }
          this.classList.add("url-selected");
          return false;
        };
      })(data[i]);
      aEl.ondblclick = function () {
        $("u_ok").click();
      };
      container.appendChild(aEl);
    }
  }

  function setMostVisitedForSelect(data) {
    var latest_url = $("latest_url");
    updateUrlList(data, latest_url);
  }

  function setOpenedURLs(data) {
    var opened_url = $("opened_url");
    updateUrlList(data, opened_url);
  }
  /**
  * Callback for the 'click' event on a page switcher.
  * @param {Event} e The event.
  */
  function onPageSwitcherClicked(e) {
    cardSlider.selectCard(cardSlider.currentCard +
        (e.currentTarget == pageSwitcherStart ? -1 : 1), true);
  }

  /**
  * Handler for the mousewheel event on a pager. We pass through the scroll
  * to the page.
  * @param {Event} e The mousewheel event.
  */
  function onPageSwitcherScrolled(e) {
    cardSlider.currentCardValue.scrollBy(-e.wheelDeltaY);
  };

  /**
  * Callback for the 'pagelayout' event.
  * @param {Event} e The event.
  */
  function onPageLayout(e) {
    if (Array.prototype.indexOf.call(tilePages, e.currentTarget) !=
        cardSlider.currentCard) {
      return;
    }

    updatePageSwitchers();
  }

  /**
  * Adjusts the size and position of the page switchers according to the
  * layout of the current card.
  */
  function updatePageSwitchers() {
    var page = cardSlider.currentCardValue;

    pageSwitcherStart.hidden = !page || (cardSlider.currentCard == 0);
    pageSwitcherEnd.hidden = !page || (cardSlider.currentCard == cardSlider.cardCount - 1);

    if (!page) return;

    var pageSwitcherLeft = isRTL() ? pageSwitcherEnd : pageSwitcherStart;
    var pageSwitcherRight = isRTL() ? pageSwitcherStart : pageSwitcherEnd;
    var scrollbarWidth = parseInt(page.scrollbarWidth) | 0;

    pageSwitcherLeft.style.width = (page.sideMargin + 13) + 'px';
    pageSwitcherLeft.style.left = '0';
    pageSwitcherRight.style.width = (page.sideMargin - scrollbarWidth + 13) + 'px';
    pageSwitcherRight.style.right = scrollbarWidth + 'px';

    //var offsetTop = page.querySelector('.tile-page-content').offsetTop + 'px';
    var offsetTop = page.offsetTop + 'px';
    pageSwitcherLeft.style.top = offsetTop;
    pageSwitcherRight.style.top = offsetTop;
    pageSwitcherLeft.style.paddingBottom = offsetTop;
    pageSwitcherRight.style.paddingBottom = offsetTop;
  }

  /**
  * Returns the index of the given page.
  * @param {AppsPage} page The AppsPage for we wish to find.
  * @return {number} The index of |page|, or -1 if it is not here.
  */
  function getAppsPageIndex(page) {
    return Array.prototype.indexOf.call(appsPages, page);
  }

  /**
  * Check the directionality of the page.
  * @return {boolean} True if Chrome is running an RTL UI.
  */
  function isRTL() {
    return document.documentElement.dir == 'rtl';
  }

  /*
  * Save the name of an app page.
  * Store the app page name into the preferences store.
  * @param {AppsPage} appPage The app page for which we wish to save.
  * @param {string} name The name of the page.
  */
  function saveAppPageName(appPage, name) {
    var index = getAppsPageIndex(appPage);
    assert(index != -1);
    chrome.send('saveAppPageName', [name, index]);
  }

  // Return an object with all the exports
  return {
    appAdded: appAdded,
    appRemoved: appRemoved,
    appsPrefChangeCallback: appsPrefChangeCallback,
    getAppsCallback: getAppsCallback,
    getAppsPageIndex: getAppsPageIndex,
    getCardSlider: getCardSlider,
    initialize: initialize,
    themeChanged: themeChanged,
    setRecentlyClosedTabs: setRecentlyClosedTabs,
    setMostVisitedPages: setMostVisitedPages,
    updateWallpaperOffset: updateWallpaperOffset,
    loadWallpaper: loadWallpaper,
    applySuccess: applySuccess,
    isRTL: isRTL,
    leaveRearrangeMode: leaveRearrangeMode,
    saveAppPageName: saveAppPageName,
    setAppToBeHighlighted: setAppToBeHighlighted,
    updateOfflineEnabledApps: updateOfflineEnabledApps,
    setMostVisitedForSelect: setMostVisitedForSelect,
    setOpenedURLs: setOpenedURLs,
    updateSiteLogo: updateSiteLogo,
    siteLogoReady: siteLogoReady,
    updateCurrentWallpaperInfo: updateCurrentWallpaperInfo,
    onGetPref: onPrefChanged,
    onPrefChanged: onPrefChanged,
    HotKeyword: HotKeyword,
  };
});

document.addEventListener('DOMContentLoaded', ntp.initialize);
