// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

///////////////////////////////////////////////////////////////////////////////
// Globals:
var RESULTS_PER_PAGE = 99999;//150;
var MAX_SEARCH_DEPTH_MONTHS = 18;

// Amount of time between pageviews that we consider a 'break' in browsing,
// measured in milliseconds.
var BROWSING_GAP_TIME = 15 * 60 * 1000;
var is_cn = false;
var en_month_arr = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

function $(o) {return document.getElementById(o);}

function createElementWithClassName(type, className) {
  var elm = document.createElement(type);
  elm.className = className;
  return elm;
}

// Escapes a URI as appropriate for CSS.
function encodeURIForCSS(uri) {
  // CSS URIs need to have '(' and ')' escaped.
  return uri.replace(/\(/g, "\\(").replace(/\)/g, "\\)");
}

function findAncestorWithClass(node, className) {
  while ((node = node.parentNode)) {
    if (node.classList.contains(className)) return node;
  }
  return null;
}

function myDelegate(pid, doEl, eventType, func){
  var delegateFunc = function(){
      var e = event, target = e.target;
      var isDoElChild = function(el){
        if(typeof(el.parentNode.classList) == 'undefined') return false;
        if(el.parentNode && 
            ((doEl.indexOf('.') == 0 && el.parentNode.classList && el.parentNode.classList.contains(doEl.replace('.',''))) || el.parentNode.tagName.toLowerCase() == doEl.toLowerCase())
        ){
          target = el.parentNode;
          return true;
        }else if(el.parentNode.parentNode){
          return isDoElChild(el.parentNode);
        }
        return false;
      };
      if((doEl.indexOf('.') == 0 && target.classList.contains(doEl.replace('.',''))) || target.tagName.toLowerCase() == doEl.toLowerCase() || isDoElChild(e.target)){
        func && func(target);
      }
    };
  document.getElementById(pid).addEventListener(eventType, delegateFunc, true);
}
var trim = function (str){ 
  return str.replace(/^(\s|\u00A0)+/,'').replace(/(\s|\u00A0)+$/,''); 
}

var myClickType = 1;
function doHistoryNavClick(n,event,callback){
  myClickType = n; 
  var clickTimer = setTimeout(function(){historyNavClickCallBack(event,callback)},300); 
  if(myClickType==2){
    clearTimeout(clickTimer);
  }
}
function historyNavClickCallBack(event,callback){ 
  event.stopPropagation && event.stopPropagation();
  if(myClickType==2){
    return false;
  }
  callback&&callback();
}

// TODO(glen): Get rid of these global references, replace with a controller
//     or just make the classes own more of the page.
var historyModel;
var historyView;
var localStrings;
var pageState;
var deleteQueue = [];
var selectionAnchor = -1;
var activeVisit = null;

const MenuButton = cr.ui.MenuButton;
const Command = cr.ui.Command;
const Menu = cr.ui.Menu;

/*function createDropDownBgImage(canvasName, colorSpec) {
  var ctx = document.getCSSCanvasContext('2d', canvasName, 6, 4);
  ctx.fillStyle = ctx.strokeStyle = colorSpec;
  ctx.beginPath();
  ctx.moveTo(0, 0);
  ctx.lineTo(6, 0);
  ctx.lineTo(3, 3);
  ctx.closePath();
  ctx.fill();
  ctx.stroke();
  return ctx;
}*/

// Create the canvases to be used as the drop down button background images.
/*var arrow = createDropDownBgImage('drop-down-arrow', 'hsl(214, 91%, 85%)');
var hoverArrow = createDropDownBgImage('drop-down-arrow-hover', '#6A86DE');
var activeArrow = createDropDownBgImage('drop-down-arrow-active', 'white');*/

///////////////////////////////////////////////////////////////////////////////
// Visit:
/**
 * Class to hold all the information about an entry in our model.
 * @param {Object} result An object containing the visit's data.
 * @param {boolean} continued Whether this visit is on the same day as the
 *     visit before it.
 */
function Visit(result, continued, model, id) {
  this.model_ = model;
  this.title_ = result.title;
  this.url_ = result.url;
  this.starred_ = result.starred;
  this.snippet_ = result.snippet || "";
  this.id_ = id;
  this.domain_ = trim(this.getDomainFromURL_(result.url));
  this.removed_ = result.removed || false;

  this.changed = false;

  this.isRendered = false;

  // All the date information is public so that owners can compare properties of
  // two items easily.

  // We get the time in seconds, but we want it in milliseconds.
  this.time = new Date(result.time * 1000);

  // See comment in BrowsingHistoryHandler::QueryComplete - we won't always
  // get all of these.
  this.dateRelativeDay = result.dateRelativeDay || "";
  this.dateTimeOfDay = result.dateTimeOfDay || "";
  this.dateShort = result.dateShort || "";

  // Whether this is the continuation of a previous day.
  this.continued = true;//continued;
}

// Visit, public: -------------------------------------------------------------
/**
 * Returns a dom structure for a browse page result or a search page result.
 * @param {boolean} Flag to indicate if result is a search result.
 * @return {Element} The dom structure.
 */
Visit.prototype.getResultDOM = function(searchResultFlag, group_hd) {
  var node = createElementWithClassName('li', 'entry');
  var domain = createElementWithClassName('div', 'domain fr');
  var del = createElementWithClassName('div', 'del bg-img hidden');
  if(this.removed_) node.classList.add('fade-out');
  del.title = localStrings.getString('IDS_HISTORY_DEL_BTN_TIPS');
  if(group_hd){
    del.classList.add('group-hd-del');
    domain.appendChild(del);
    var span = document.createElement('span');
    var img = document.createElement('span');
    img.className = 'icon';
    img.style.backgroundImage =
      'url(chrome://favicon/' + encodeURIForCSS(this.url_) + ')';
    span.textContent = this.domain_;
    domain.appendChild(img);
    domain.appendChild(span);
    node.appendChild(domain);
    node.classList.add('group-hd');
  }else{
    var time = createElementWithClassName('div', 'time fr');
  var entryBox = createElementWithClassName('label', 'entry-box');
    var search = createElementWithClassName('div', 'search bg-img fr');
    search.title = localStrings.getString('IDS_HISTORY_SEARCH_ABOUT_TIPS');

    /*var dropDown = createElementWithClassName('button', 'drop-down');
  dropDown.value = 'Open action menu';
  dropDown.title = localStrings.getString('actionMenuDescription');
  dropDown.setAttribute('menu', '#action-menu');
    cr.ui.decorate(dropDown, MenuButton);*/

  // Checkbox is always created, but only visible on hover & when checked.
  var checkbox = document.createElement('input');
  checkbox.type = 'checkbox';
  checkbox.id = 'checkbox-' + this.id_;
  checkbox.time = this.time.getTime();
  checkbox.addEventListener('click', checkboxClicked);
  time.appendChild(checkbox);

  // Keep track of the drop down that triggered the menu, so we know
  // which element to apply the command to.
  // TODO(dubroy): Ideally we'd use 'activate', but MenuButton swallows it.
  var self = this;
  var setActiveVisit = function(e) {
    activeVisit = self;
  };
    /*dropDown.addEventListener('mousedown', setActiveVisit);
    dropDown.addEventListener('focus', setActiveVisit);*/

    domain.textContent = this.domain_;

  // Clicking anywhere in the entryBox will check/uncheck the checkbox.
  entryBox.setAttribute('for', checkbox.id);
  entryBox.addEventListener('mousedown', entryBoxMousedown, false);

  // Prevent clicks on the drop down from affecting the checkbox.
    //dropDown.addEventListener('click', function(e) { e.preventDefault(); });

  // We use a wrapper div so that the entry contents will be shinkwrapped.
    entryBox.appendChild(del);
  entryBox.appendChild(this.getTitleDOM_());
    //entryBox.appendChild(dropDown);

  // Let the entryBox be styled appropriately when it contains keyboard focus.
    /*entryBox.addEventListener('focus', function() {
    this.classList.add('contains-focus');
  }, true);
  entryBox.addEventListener('blur', function() {
    this.classList.remove('contains-focus');
    }, true);*/

    node.appendChild(time);
    node.appendChild(search);
    node.appendChild(domain);
  node.appendChild(entryBox);

  if (searchResultFlag) {
      time.appendChild(document.createTextNode(this.dateShort + ' ' + this.dateTimeOfDay));
      /*var snippet = createElementWithClassName('div', 'snippet');
    this.addHighlightedText_(snippet,
                             this.snippet_,
                             this.model_.getSearchText());
      node.appendChild(snippet);*/
  } else {
    time.appendChild(document.createTextNode(this.dateTimeOfDay));
  }
  }

  this.domNode_ = node;

  return node;
};

// Visit, private: ------------------------------------------------------------
/**
 * Extracts and returns the domain (and subdomains) from a URL.
 * @param {string} The url
 * @return (string) The domain. An empty string is returned if no domain can
 *     be found.
 */
Visit.prototype.getDomainFromURL_ = function(url) {
  if (/^file:\/\/\//.test(url)) {
    return url.replace(/^file:\/\/\//,'');
  }
  var domain = url.replace(/^.+?:\/\//, '').match(/[^/]+/);
  return domain ? domain[0] : '';
};

/**
 * Add child text nodes to a node such that occurrences of the specified text is
 * highlighted.
 * @param {Node} node The node under which new text nodes will be made as
 *     children.
 * @param {string} content Text to be added beneath |node| as one or more
 *     text nodes.
 * @param {string} highlightText Occurences of this text inside |content| will
 *     be highlighted.
 */
Visit.prototype.addHighlightedText_ = function(node, content, highlightText) {
  var i = 0;
  if (highlightText) {
    var re = new RegExp(Visit.pregQuote_(highlightText), 'gim');
    var match;
    while (match = re.exec(content)) {
      if (match.index > i)
        node.appendChild(document.createTextNode(content.slice(i,
                                                               match.index)));
      i = re.lastIndex;
      // Mark the highlighted text in bold.
      var b = document.createElement('b');
      b.textContent = content.substring(match.index, i);
      node.appendChild(b);
    }
  }
  if (i < content.length)
    node.appendChild(document.createTextNode(content.slice(i)));
};

/**
 * @return {DOMObject} DOM representation for the title block.
 */
Visit.prototype.getTitleDOM_ = function() {
  var node = createElementWithClassName('div', 'title');
  node.style.backgroundImage =
      'url(chrome://favicon/' + encodeURIForCSS(this.url_) + ')';

  var link = document.createElement('a');
  link.href = this.url_;
  link.id = "id-" + this.id_;
  link.setAttribute('target','_blank');

  // Add a tooltip, since it might be ellipsized.
  // TODO(dubroy): Find a way to show the tooltip only when necessary.
  link.title = this.title_;

  this.addHighlightedText_(link, this.title_, this.model_.getSearchText());
  node.appendChild(link);

  if (this.starred_)
    node.appendChild(createElementWithClassName('div', 'starred'));

  return node;
};

/**
 * Launch a search for more history entries from the same domain.
 */
Visit.prototype.showMoreFromSite_ = function() {
  setSearch(this.getDomainFromURL_(this.url_));
};

/**
 * Remove a single entry from the history.
 */
Visit.prototype.removeFromHistory_ = function() {
  var self = this;
  var onSuccessCallback = function() {
    setVisitRemovedById(self.id_);
    removeEntryFromView(self.domNode_);
  };
  queueURLsForDeletion(this.time, [this.url_], onSuccessCallback);
  deleteNextInQueue();
};


// Visit, private, static: ----------------------------------------------------

/**
 * Quote a string so it can be used in a regular expression.
 * @param {string} str The source string
 * @return {string} The escaped string
 */
Visit.pregQuote_ = function(str) {
  return str.replace(/([\\\.\+\*\?\[\^\]\$\(\)\{\}\=\!\<\>\|\:])/g, "\\$1");
};

///////////////////////////////////////////////////////////////////////////////
// HistoryModel:
/**
 * Global container for history data. Future optimizations might include
 * allowing the creation of a HistoryModel for each search string, allowing
 * quick flips back and forth between results.
 *
 * The history model is based around pages, and only fetching the data to
 * fill the currently requested page. This is somewhat dependent on the view,
 * and so future work may wish to change history model to operate on
 * timeframe (day or week) based containers.
 */
function HistoryModel() {
  this.clearModel_();
}

// HistoryModel, Public: ------------------------------------------------------
/**
 * Sets our current view that is called when the history model changes.
 * @param {HistoryView} view The view to set our current view to.
 */
HistoryModel.prototype.setView = function(view) {
  this.view_ = view;
};

/**
 * Start a new search - this will clear out our model.
 * @param {String} searchText The text to search for
 * @param {Number} opt_page The page to view - this is mostly used when setting
 *     up an initial view, use #requestPage otherwise.
 */
HistoryModel.prototype.setSearchText = function(searchText, opt_page) {
  /*if(this.view_.resultDiv_.getAttribute('result-index') && searchText == ''){
    var cDiv_ = this.view_.resultDiv_.cloneNode();
    cDiv_.innerHTML = this.view_.resultDiv_.innerHTML;
    cDiv_.removeAttribute('id');
    
    this.view_.model_.Animobj={};
    this.view_.model_.animObj.oldUl = cDiv_;
    this.view_.model_.animObj.newUl = this.view_.resultDiv_;
    this.view_.model_.animObj.oldIndex = cDiv_.getAttribute('result-index');
    
    $('main').appendChild(cDiv_);
    this.view_.resultDiv_.style.top = $('main').clientHeight+'px';
  }*/
  this.clearModel_();
  this.searchText_ = searchText;
  this.requestedPage_ = opt_page ? opt_page : 0;
  this.getSearchResults_();
};

/**
 * Reload our model with the current parameters.
 */
HistoryModel.prototype.reload = function() {
  var search = this.searchText_;
  var page = this.requestedPage_;
  this.clearModel_();
  this.searchText_ = search;
  this.requestedPage_ = page;
  this.getSearchResults_();
};

/**
 * @return {String} The current search text.
 */
HistoryModel.prototype.getSearchText = function() {
  return this.searchText_;
};

/**
 * Tell the model that the view will want to see the current page. When
 * the data becomes available, the model will call the view back.
 * @page {Number} page The page we want to view.
 */
HistoryModel.prototype.requestPage = function(page) {
  this.requestedPage_ = page;
  this.changed = true;
  this.updateSearch_(false);
};
HistoryModel.prototype.anim = function(callback, durtime){
  this.animing = true;
  var that = this,
      oldEl = that.animObj.oldUl,
      newEl = that.animObj.newUl,
      oi = that.animObj.oldIndex,
      ni = that.animObj.newIndex,
      t = 0,
      durtime = durtime || 360,
      intervalTime = 10,
      timer,
      tween = function(t, b, c, d){
        return c * t / d + b;
      },
      start = function(){
        clearInterval(timer);
        timer = setInterval(function(){
          var h = oldEl.clientHeight;
          if (t < durtime / intervalTime) {
            t++;
            var b1 = 0,
            c1 = (oi>ni?h:-h) - b1,
            b2 = oi>ni?-h:h,
            c2 = 0 - b2,
            d = durtime / intervalTime;
            oldEl.style.top = tween(t, b1, c1, d) + "px";
            newEl.style.top = tween(t, b2, c2, d) + "px";
          }else{
            end();
          }
        }, intervalTime);
      },
      end = function(){
        var h = oldEl.clientHeight;
        oldEl.style.top = (oi>ni?h:-h) + "px";
        newEl.style.top = 0 + "px";
        if(timer){
          clearInterval(timer);
          timer = undefined;
        }
        oldEl.remove();
        that.animing = false;
        callback && callback();
      };
  if(durtime > 0) start();
  else end();
};

HistoryModel.prototype.makeLeftNav = function(info, results){
  if(results.length<=0){
    this.view_.summaryTd_.innerText='';
    return;
  }
  var this_,
  df = document.createDocumentFragment('div'),
  cur_date_ = new Date(),
  cur_y = cur_date_.getFullYear(),
  cur_m = cur_date_.getMonth() + 1,
  month_index = 0,
  j = 0,
  makeNavHtml = function(){
    for(var i=0;i<results.length;i++){
      var date_ = new Date(results[i].dateShort),
      y = date_.getFullYear(),
      m = date_.getMonth() + 1,
      d = date_.getDate()
      ;
      if(y<=cur_y){
        if(month_index != m){
          month_index = m;
          var y_text = '';
          if(cur_y != y){
            y_text = '<span>' + (is_cn? y + '\u5E74' : ', ' + y) + '</span>';
          }
          var hd = document.createElement('h2');
          hd.innerHTML = is_cn?(y_text + m + '\u6708'):(en_month_arr[m-1] + y_text);
          hd.className = 'day' + (j == 0? ' d':'');
          hd.id = 'h2_' + month_index;
          df.appendChild(hd);
          var ol = document.createElement('ol');
          ol.id = 'ol_' + y + '_' + month_index;
          ol.style.height = '0px';
          df.appendChild(ol);
        }
        var ol = df.querySelector('#ol_' + y + '_' + month_index);
        if(ol){
          var li = document.createElement('li');
          li.innerText = d;
          if(j == 0) li.className = 'cur';
          li.setAttribute('date', y + '-' + m + '-' + d);
          li.setAttribute('nav-index', i);
          ol.appendChild(li);
        }
        j++;
      }
    }
    var nav_div = document.createElement('div');
    nav_div.appendChild(df);
    nav_div.className = 'navDiv';
    $('left_nav').appendChild(nav_div);
    setNavOlHeight($('left_nav').querySelector('ol'));
  },
  makeNavEvent = function(){
    myDelegate('left_nav', 'li', 'click', function(e){
      var callback = function(){
        if(!$('left_nav').querySelector('li.cur')){
          historyView.summaryTd_.innerHTML = '<span class="loading">' + localStrings.getString('loading') + '</span>';
          historyView.summaryTd_.style.display = 'block';
        }
        if(!e.classList.contains('cur')){
          var curli = $('left_nav').querySelector('li.cur');
          if(curli) curli.classList.remove('cur');
          e.classList.add('cur');
          window.last_date_link = e;
          historyView.setSearch('');
        }
      };
      callback();
      //doHistoryNavClick(1, e, callback);
    });
    myDelegate('left_nav', 'li', 'dblclick', function(e){
      return false;
    });
    myDelegate('left_nav', 'h2', 'click', function(e){
      var olArr = $('left_nav').querySelectorAll('ol'),
      h2Arr = $('left_nav').querySelectorAll('h2'),
      e_show = e.classList.contains('d');
      for(var i=0; i<h2Arr.length; i++){
        h2Arr[i].classList.remove('d');
      }
      for(var i=0; i<olArr.length; i++){
        if(olArr[i].clientHeight > 0 ){
          setNavOlHeight(olArr[i], 0);
        }
      }
      if(e_show){
        e.classList.remove('d');
      }else {
        setNavOlHeight(e.nextSibling);
        e.classList.add('d');
      }
      setTimeout(this_.view_.model_.navScroll.reset_.bind(this_.view_.model_.navScroll),360);
    });
  },
  setNavOlHeight = function(el, h){
    if(!el) return;
    var li_height = 34,
    li_size = el.querySelectorAll('li');
    if(typeof(h) == 'undefined' && li_size.length>0) h = parseInt(li_size.length * li_height);
    el.style.height = (h || 0) + 'px';
  }
  ;
  
  makeNavHtml();
  
  var styles = document.defaultView.getComputedStyle($('main'));
  var height_ = (document.body.clientHeight - (parseInt(styles.top)||0)) + 'px';
  var navObj = $('left_nav').querySelector('.navDiv');
  navObj.style.height = height_;
  this.navScroll = new myScroll({obj:navObj, over_show: true});
  this_ = this;
  
  makeNavEvent();
  
  $('left_nav').onselectstart = function(){return false};
  
  // Create default view.
  var hashData = pageState.getHashData();
  historyView.setSearch(hashData.q, hashData.p);
};

/**
 * Receiver for history query.
 * @param {String} term The search term that the results are for.
 * @param {Array} results A list of results
 */
HistoryModel.prototype.addResults = function(info, results) {
  this.inFlight_ = false;
  if (info.term != this.searchText_) {
    // If our results aren't for our current search term, they're rubbish.
    return;
  }

  // Currently we assume we're getting things in date order. This needs to
  // be updated if that ever changes.
  if (results) {
    var lastURL, lastDay;
    var oldLength = this.visits_.length;
    if (oldLength) {
      var oldVisit = this.visits_[oldLength - 1];
      lastURL = oldVisit.url;
      lastDay = oldVisit.dateRelativeDay;
    }

    for (var i = 0, thisResult; thisResult = results[i]; i++) {
      var thisURL = thisResult.url;
      var thisDay = thisResult.dateRelativeDay;

      // Remove adjacent duplicates.
      if (!lastURL || lastURL != thisURL) {
        // Figure out if this visit is in the same day as the previous visit.
        // This is used to determine how day headers should be drawn.
        this.visits_.push(
            new Visit(thisResult, thisDay == lastDay, this, this.last_id_++));
        lastDay = thisDay;
        lastURL = thisURL;
      }
    }
    if (results.length)
      this.changed = true;
  }

  this.updateSearch_(info.finished);
};

/**
 * @return {Number} The number of visits in the model.
 */
HistoryModel.prototype.getSize = function() {
  return this.visits_.length;
};

/**
 * Get a list of visits between specified index positions.
 * @param {Number} start The start index
 * @param {Number} end The end index
 * @return {Array.<Visit>} A list of visits
 */
HistoryModel.prototype.getNumberedRange = function(start, end) {
  if (start >= this.getSize())
    return [];

  var end = end > this.getSize() ? this.getSize() : end;
  return this.visits_.slice(start, end);
};

// HistoryModel, Private: -----------------------------------------------------
HistoryModel.prototype.clearModel_ = function() {
  this.inFlight_ = false; // Whether a query is inflight.
  this.searchText_ = '';
  this.searchDepth_ = 0;
  this.visits_ = []; // Date-sorted list of visits.
  this.last_id_ = 0;
  selectionAnchor = -1;

  // The page that the view wants to see - we only fetch slightly past this
  // point. If the view requests a page that we don't have data for, we try
  // to fetch it and call back when we're done.
  this.requestedPage_ = 0;

  if (this.view_)
    this.view_.clear_();
};

/**
 * Figure out if we need to do more searches to fill the currently requested
 * page. If we think we can fill the page, call the view and let it know
 * we're ready to show something.
 */
HistoryModel.prototype.updateSearch_ = function(finished) {
  if ((this.searchText_ && this.searchDepth_ >= MAX_SEARCH_DEPTH_MONTHS) ||
      finished) {
    // We have maxed out. There will be no more data.
    this.complete_ = true;
    this.view_.onModelReady();
    this.changed = false;
  } else {
    // If we can't fill the requested page, ask for more data unless a request
    // is still in-flight.
    if (!this.canFillPage_(this.requestedPage_) && !this.inFlight_) {
      this.getSearchResults_(this.searchDepth_ + 1);
    }

    // If we have any data for the requested page, show it.
    if (this.changed && this.haveDataForPage_(this.requestedPage_)) {
      this.view_.onModelReady();
      this.changed = false;
    }
  }
};

/**
 * Get search results for a selected depth. Our history system is optimized
 * for queries that don't cross month boundaries, but an entire month's
 * worth of data is huge. When we're in browse mode (searchText is empty)
 * we request the data a day at a time. When we're searching, a month is
 * used.
 *
 * TODO: Fix this for when the user's clock goes across month boundaries.
 * @param {number} opt_day How many days back to do the search.
 */
HistoryModel.prototype.getSearchResults_ = function(depth) {
  this.searchDepth_ = depth || 0;

  var navCur = $('left_nav').querySelector('li.cur');
  if (this.searchText_) {
    if(navCur) navCur.classList.remove('cur');
    chrome.send('searchHistory',
        [this.searchText_, String(this.searchDepth_)]);
  } else {
    /*chrome.send('getHistory',
        [String(this.searchDepth_)]);*/
    if(navCur){
      var day = new Date(navCur.getAttribute('date') + ' GMT').getTime();
      chrome.send('queryHistory',
        ['', 0, 3, 0, 0, day]);
    }
  }

  this.inFlight_ = true;
};

/**
 * Check to see if we have data for a given page.
 * @param {number} page The page number
 * @return {boolean} Whether we have any data for the given page.
 */
HistoryModel.prototype.haveDataForPage_ = function(page) {
  return (page * RESULTS_PER_PAGE < this.getSize());
};

/**
 * Check to see if we have data to fill a page.
 * @param {number} page The page number.
 * @return {boolean} Whether we have data to fill the page.
 */
HistoryModel.prototype.canFillPage_ = function(page) {
  return ((page + 1) * RESULTS_PER_PAGE <= this.getSize());
};

///////////////////////////////////////////////////////////////////////////////
// HistoryView:
/**
 * Functions and state for populating the page with HTML. This should one-day
 * contain the view and use event handlers, rather than pushing HTML out and
 * getting called externally.
 * @param {HistoryModel} model The model backing this view.
 */
function HistoryView(model) {
  this.summaryTd_ = $('results-summary');
  this.summaryTd_.innerHTML = '<span class="loading">' + localStrings.getString('loading') + '</span>';
  this.editButtonTd_ = $('edit-button');
  this.editingControlsDiv_ = $('editing-controls');
  this.resultDiv_ = $('results-display');
  this.pageDiv_ = $('results-pagination');
  this.model_ = model
  this.pageIndex_ = 0;
  this.lastDisplayed_ = [];

  this.model_.setView(this);

  this.currentVisits_ = [];

  var self = this;
  
  window.onresize = function() {
    self.updateEntryAnchorWidth_();
  };

  $('clear-browsing-data').addEventListener('click', openClearBrowsingData);
  $('remove-selected').addEventListener('click', removeItems);
}

// HistoryView, public: -------------------------------------------------------
/**
 * Do a search and optionally view a certain page.
 * @param {string} term The string to search for.
 * @param {number} opt_page The page we wish to view, only use this for
 *     setting up initial views, as this triggers a search.
 */
HistoryView.prototype.setSearch = function(term, opt_page) {
  var display_val = 'none';
  if(term!='') display_val = 'block';
  $('delTerm').style.display = display_val;
  this.pageIndex_ = parseInt(opt_page || 0, 10);
  window.scrollTo(0, 0);
  this.model_.setSearchText(term, this.pageIndex_);
  pageState.setUIState(term, this.pageIndex_);
};

/**
 * Reload the current view.
 */
HistoryView.prototype.reload = function() {
  this.model_.reload();
  this.updateRemoveButton();
};

/**
 * Switch to a specified page.
 * @param {number} page The page we wish to view.
 */
HistoryView.prototype.setPage = function(page) {
  this.clear_();
  this.pageIndex_ = parseInt(page, 10);
  window.scrollTo(0, 0);
  this.model_.requestPage(page);
  pageState.setUIState(this.model_.getSearchText(), this.pageIndex_);
};

/**
 * @return {number} The page number being viewed.
 */
HistoryView.prototype.getPage = function() {
  return this.pageIndex_;
};

/**
 * Callback for the history model to let it know that it has data ready for us
 * to view.
 */
HistoryView.prototype.onModelReady = function() {
  this.displayResults_();
  this.selectAllCheckbox();
};

HistoryView.prototype.selectAllCheckbox = function(){
  var cbArr = $('results-display').querySelectorAll('.time input');
  for(var i=0; i<cbArr.length; i++){
    if(!cbArr[i].checked) cbArr[i].click();
  }
};

/**
 * Enables or disables the 'Remove selected items' button as appropriate.
 */
HistoryView.prototype.updateRemoveButton = function() {
  var anyChecked = document.querySelector('.entry input:checked') != null;
  $('remove-selected').disabled = !anyChecked;
  //update sortby radio
  var sortbyArr = $('editing-controls').querySelectorAll('.sortby');
  for(var i=0; i<sortbyArr.length; i++){
    sortbyArr[i].disabled = !anyChecked;
  }
}

// HistoryView, private: ------------------------------------------------------
/**
 * Clear the results in the view.  Since we add results piecemeal, we need
 * to clear them out when we switch to a new page or reload.
 */
HistoryView.prototype.clear_ = function() {
  this.resultDiv_.textContent = '';

  this.currentVisits_.forEach(function(visit) {
    visit.isRendered = false;
  });
  this.currentVisits_ = [];
};

HistoryView.prototype.setVisitRendered_ = function(visit) {
  visit.isRendered = true;
  this.currentVisits_.push(visit);
};

HistoryView.prototype.sortBySite_ = function(results) {
  var len = results.length,
  canMove = function(d1,d2){
    var len = d1.length > d2.length ? d2.length : d1.length;
    for(var i=0; i<len; i++){
      if(d1.charCodeAt(i) == d2.charCodeAt(i)) continue;
      else if(d1.charCodeAt(i) > d2.charCodeAt(i)) return true;
      else return false
    }
    return false;
  },
  mySort_ = function(){
    if(len>0){
      for(var i=0; i<len; i++){
        for(var j=i; j<len; j++){
          if(canMove(results[i].domain_, results[j].domain_)){
            var temp = results[i];
            results[i] = results[j];
            results[j] = temp; 
          } 
        } 
      }
      for(var i=0;i<len;i++){
        if(i+1 < len && results[i].domain_ != results[i+1].domain_){
          var ci = 0;
          for(var j=i+1; j<len; j++){
            if(results[i].domain_ != results[j].domain_){
              if(ci == 0) ci = j;
            }else if(ci != 0){
              var d1 = results.splice(j,1);
              var d2 = results.splice(ci,1);
              results.splice(ci,0,d1[0],d2[0]);
              break;
            }
          }
        }
      }
      for(var i=0; i<len; i++){
        for(var j=i; j<len; j++){
          if(results[i].domain_ == results[j].domain_ && new Date(results[i].time).getTime() < new Date(results[j].time).getTime()){
            var temp = results[i];
            results[i] = results[j];
            results[j] = temp; 
          } 
        } 
      }
    }
  };
  mySort_();
  return results;
}

/**
 * Update the page with results.
 */
HistoryView.prototype.displayResults_ = function() {
  var results = this.model_.getNumberedRange(
      this.pageIndex_ * RESULTS_PER_PAGE,
      this.pageIndex_ * RESULTS_PER_PAGE + RESULTS_PER_PAGE);

  var history_sort_by = sessionStorage.history_sort_by,
  group;
  if(history_sort_by == 'site'){
    results = this.sortBySite_(results);
  }
  this.displaySummaryBar_();
  if (this.model_.getSearchText()) {
    var ol_sr = document.querySelector('ol.search-results');
    if(ol_sr) ol_sr.remove();
    var searchResults = createElementWithClassName('ol', 'search-results');
    var results_summary = $('results-summary'),
    empty_text_id = 'empty_text_for_search_results',
    makeFirstLetter = function(str, u){   
      return (u?str.substring(0,1).toUpperCase():str.substring(0,1).toLowerCase()) + str.substring(1);    
    };
    if($('clone_results_summary')) $('clone_results_summary').remove();
    if (results.length == 0) {
      results_summary.style.display = 'block';
      var noResults = document.createElement('div');
      noResults.textContent = localStrings.getString('noResults');
      
      searchResults.classList.add('empty-results');
      results_summary.classList.add('empty-results-text');
      results_summary.innerText = makeFirstLetter(results_summary.innerText, false);
      var empty_text = document.createElement('span');
      empty_text.id=empty_text_id;
      empty_text.innerText = localStrings.getString('IDS_HISTORY_EMPTY_TEXT') + (!is_cn?' ':'');
      var child = results_summary.childNodes[0];
      results_summary.insertBefore(empty_text, child);
      
      searchResults.appendChild(noResults);
    } else {
      searchResults.classList.remove('empty-results');
      results_summary.classList.remove('empty-results-text');
      if($(empty_text_id)){
        $(empty_text_id).remove();
        results_summary.innerText = makeFirstLetter(results_summary.innerText, true);
      }
      results_summary.style.display = 'none';
      var clone_results_summary = results_summary.cloneNode();
      clone_results_summary.innerHTML = results_summary.innerHTML;
      clone_results_summary.style.display = 'block';
      clone_results_summary.id = 'clone_results_summary';
      this.resultDiv_.appendChild(clone_results_summary);
      for (var i = 0, visit; visit = results[i]; i++) {
        if (!visit.isRendered) {
          if(history_sort_by == 'site'){
            if(!group || (group && group != visit.domain_)){
              searchResults.appendChild(visit.getResultDOM(false, true));
            }
            // ûɾģhdɾȥ
            if(group == visit.domain_ && !visit.removed_){
              var gArr = searchResults.querySelectorAll('li.group-hd');
              gArr.length > 0 && gArr[gArr.length-1].classList.remove('fade-out');
            }
            group = visit.domain_;
          }
          searchResults.appendChild(visit.getResultDOM(true));
          this.setVisitRendered_(visit);
        }
      }
    }
    this.resultDiv_.removeAttribute('result-index');
    this.resultDiv_.appendChild(searchResults);
  } else {
    var resultsFragment = document.createDocumentFragment();
    var lastTime = Math.infinity;
    var dayResults;
    for (var i = 0, visit; visit = results[i]; i++) {
      if (visit.isRendered) {
        continue;
      }
      // Break across day boundaries and insert gaps for browsing pauses.
      // Create a dayResults element to contain results for each day
      var thisTime = visit.time.getTime();

      if ((i == 0 && visit.continued) || !visit.continued) {
        var day = createElementWithClassName('h2', 'day');
        day.time = thisTime;
        day.appendChild(document.createTextNode(visit.dateRelativeDay));
        /*if (i == 0 && visit.continued) {
          day.appendChild(document.createTextNode(' ' +
              localStrings.getString('cont')));
        }*/

        // If there is an existing dayResults element, append it.
        if (dayResults) {
          resultsFragment.appendChild(dayResults);
        }
        resultsFragment.appendChild(day);
        dayResults = createElementWithClassName('ol', 'day-results');
      } else if (lastTime - thisTime > BROWSING_GAP_TIME) {
        /*if (dayResults) {
          dayResults.appendChild(createElementWithClassName('li', 'gap'));
        }*/
      }
      lastTime = thisTime;
      // Add entry.
      if (dayResults) {
        if(history_sort_by == 'site'){
          if(!group || (group && group != visit.domain_)){
            dayResults.appendChild(visit.getResultDOM(false, true));
          }
          // ûɾģhdɾȥ
          if(group == visit.domain_ && !visit.removed_){
            var gArr = dayResults.querySelectorAll('li.group-hd');
            gArr.length > 0 && gArr[gArr.length-1].classList.remove('fade-out');
          }
          group = visit.domain_;
        }
        dayResults.appendChild(visit.getResultDOM(false));
        this.setVisitRendered_(visit);
      }
    }
    // Add final dayResults element.
    if (dayResults) {
      resultsFragment.appendChild(dayResults);
    }
    var nav_index = $('left_nav').querySelector('li.cur').getAttribute('nav-index');
    this.resultDiv_.setAttribute('result-index', nav_index);
    if(!!this.model_.visits_info.sortby) this.resultDiv_.textContent = '';
    
    this.resultDiv_.appendChild(resultsFragment);
    
    /*if(this.model_.animObj && !!!this.model_.visits_info.sortby){
      this.model_.animObj.newIndex = nav_index;
      if(nav_index<this.model_.animObj.oldIndex) this.model_.animObj.newUl.style.top = (-parseInt($('main').clientHeight))+'px';
      var this_ = this;
      this.model_.anim(function(){
        var moveDiv_ = $('main').querySelector('.results-display:not([id="results-display"])');
        if(moveDiv_)
          moveDiv_.remove();
        if(this_.model_){
          this_.model_.animObj = null;
          this_.listScroll.reset_();
        }
      },160);
    }*/
  }

  this.displayNavBar_();
  this.updateEntryAnchorWidth_();
/*
  if(this.listScroll){
    this.listScroll.reset_();
  }else{
    this.listScroll = new myScroll({obj:this.resultDiv_, over_show: true});
  }
*/
};

/**
 * Update the summary bar with descriptive text.
 */
HistoryView.prototype.displaySummaryBar_ = function() {
  var searchText = this.model_.getSearchText();
  if (searchText != '') {
    this.summaryTd_.textContent = localStrings.getStringF('searchResultsFor',
        searchText);
  } else {
    this.summaryTd_.textContent = "";//localStrings.getString('history');
  }
};

/**
 * Update the pagination tools.
 */
HistoryView.prototype.displayNavBar_ = function() {
  this.pageDiv_.textContent = '';

  if (this.pageIndex_ > 0) {
    this.pageDiv_.appendChild(
        this.createPageNav_(0, localStrings.getString('newest')));
    this.pageDiv_.appendChild(
        this.createPageNav_(this.pageIndex_ - 1,
                            localStrings.getString('newer')));
  }

  // TODO(feldstein): this causes the navbar to not show up when your first
  // page has the exact amount of results as RESULTS_PER_PAGE.
  if (this.model_.getSize() > (this.pageIndex_ + 1) * RESULTS_PER_PAGE) {
    this.pageDiv_.appendChild(
        this.createPageNav_(this.pageIndex_ + 1,
                            localStrings.getString('older')));
  }
};

/**
 * Make a DOM object representation of a page navigation link.
 * @param {number} page The page index the navigation element should link to
 * @param {string} name The text content of the link
 * @return {HTMLAnchorElement} the pagination link
 */
HistoryView.prototype.createPageNav_ = function(page, name) {
  anchor = document.createElement('a');
  anchor.className = 'page-navigation';
  anchor.textContent = name;
  var hashString = PageState.getHashString(this.model_.getSearchText(), page);
  var link = 'chrome://history/' + (hashString ? '#' + hashString : '');
  anchor.href = link;
  anchor.onclick = function() {
    setPage(page);
    return false;
  };
  return anchor;
};

/**
 * Updates the CSS rule for the entry anchor.
 * @private
 */
HistoryView.prototype.updateEntryAnchorWidth_ = function() {
  // We need to have at least on .title div to be able to calculate the
  // desired width of the anchor.
  var titleElement = document.querySelector('.entry .title');
  
  var styles = document.defaultView.getComputedStyle($('main')),
  navDiv = $('left_nav').querySelector('.navDiv'),
  height_ = (document.body.clientHeight - (parseInt(styles.top)||0)) + 'px';
  this.resultDiv_.style.height = height_;
  if(navDiv) navDiv.style.height = height_;
  
  if (!titleElement)
    return;

  // Create new CSS rules and add them last to the last stylesheet.
  // TODO(jochen): The following code does not work due to WebKit bug #32309
  // if (!this.entryAnchorRule_) {
  //   var styleSheets = document.styleSheets;
  //   var styleSheet = styleSheets[styleSheets.length - 1];
  //   var rules = styleSheet.cssRules;
  //   var createRule = function(selector) {
  //     styleSheet.insertRule(selector + '{}', rules.length);
  //     return rules[rules.length - 1];
  //   };
  //   this.entryAnchorRule_ = createRule('.entry .title > a');
  //   // The following rule needs to be more specific to have higher priority.
  //   this.entryAnchorStarredRule_ = createRule('.entry .title.starred > a');
  // }
  //
  // var anchorMaxWith = titleElement.offsetWidth;
  // this.entryAnchorRule_.style.maxWidth = anchorMaxWith + 'px';
  // // Adjust by the width of star plus its margin.
  // this.entryAnchorStarredRule_.style.maxWidth = anchorMaxWith - 23 + 'px';
};

///////////////////////////////////////////////////////////////////////////////
// State object:
/**
 * An 'AJAX-history' implementation.
 * @param {HistoryModel} model The model we're representing
 * @param {HistoryView} view The view we're representing
 */
function PageState(model, view) {
  // Enforce a singleton.
  if (PageState.instance) {
    return PageState.instance;
  }

  this.model = model;
  this.view = view;
/*
  if (typeof this.checker_ != 'undefined' && this.checker_) {
    clearInterval(this.checker_);
  }

  // TODO(glen): Replace this with a bound method so we don't need
  //     public model and view.
  this.checker_ = setInterval((function(state_obj) {
    var hashData = state_obj.getHashData();

    if (hashData.q != state_obj.model.getSearchText(term)) {
      state_obj.view.setSearch(hashData.q, parseInt(hashData.p, 10));
    } else if (parseInt(hashData.p, 10) != state_obj.view.getPage()) {
      state_obj.view.setPage(hashData.p);
    }
  }), 50, this);*/
}

PageState.instance = null;

/**
 * @return {Object} An object containing parameters from our window hash.
 */
PageState.prototype.getHashData = function() {
  var result = {
    e: 0,
    q: '',
    p: 0
  };

  if (!window.location.hash) {
    return result;
  }

  var hashSplit = window.location.hash.substr(1).split('&');
  for (var i = 0; i < hashSplit.length; i++) {
    var pair = hashSplit[i].split('=');
    if (pair.length > 1) {
      result[pair[0]] = decodeURIComponent(pair[1].replace(/\+/g, ' '));
    }
  }
  // ûзҳpΪ0
  result['p'] = 0;
  return result;
};

/**
 * Set the hash to a specified state, this will create an entry in the
 * session history so the back button cycles through hash states, which
 * are then picked up by our listener.
 * @param {string} term The current search string.
 * @param {string} page The page currently being viewed.
 */
PageState.prototype.setUIState = function(term, page) {
  // Make sure the form looks pretty.
  document.forms[0].term.value = term;
  var currentHash = this.getHashData();
  if (currentHash.q != term || currentHash.p != page) {
    window.hashchanged = true;
    window.location.hash = PageState.getHashString(term, page);
  }
};

/**
 * Static method to get the hash string for a specified state
 * @param {string} term The current search string.
 * @param {string} page The page currently being viewed.
 * @return {string} The string to be used in a hash.
 */
PageState.getHashString = function(term, page) {
  var newHash = [];
  if (term) {
    newHash.push('q=' + encodeURIComponent(term));
  }
  if (page != undefined) {
    newHash.push('p=' + page);
  }

  return newHash.join('&');
};

///////////////////////////////////////////////////////////////////////////////
// Document Functions:
function sortbyInit(){
  if(!sessionStorage.history_sort_by){
    $('editing-controls').querySelector('input[value="time"]').checked = true;
  }else{
    var sortby = $('editing-controls').querySelectorAll('input[name="sortby"]');
    for(var i=0; i<sortby.length; i++){
      if(sortby[i].getAttribute('value') == sessionStorage.history_sort_by){
        sortby[i].checked = true;
      }else{
        sortby[i].checked = false;
      }
    }; 
  }
  myDelegate('editing-controls', '.sortby', 'change', function(e){
    sessionStorage.history_sort_by = e.value;
    historyView.model_.visits_info.sortby = true;
    historyResult(historyView.model_.visits_info, getResultsFromModelVisits());
  });
}

function getResultsFromModelVisits(){
  var visits_ = [];
  for(var i = 0; i<historyView.model_.visits_.length; i++){
    historyView.model_.visits_[i].model = historyView.model_.visits_[i].model_;
    historyView.model_.visits_[i].title = historyView.model_.visits_[i].title_;
    historyView.model_.visits_[i].url = historyView.model_.visits_[i].url_;
    historyView.model_.visits_[i].starred = historyView.model_.visits_[i].starred_;
    historyView.model_.visits_[i].snippet = historyView.model_.visits_[i].snippet_;
    historyView.model_.visits_[i].removed = historyView.model_.visits_[i].removed_;
    historyView.model_.visits_[i].time = historyView.model_.visits_[i].time.getTime()/1000;
    visits_[i] = historyView.model_.visits_[i];
  }
  historyModel.last_id_ = 0;
  return visits_;
}

/**
 * Window onload handler, sets up the page.
 */
function load() {
  //$('term').focus();
  is_cn = escape($('history-section').innerText).indexOf("%u")>=0;
  sortbyInit();

  localStrings = new LocalStrings();
  historyModel = new HistoryModel();
  historyView = new HistoryView(historyModel);
  pageState = new PageState(historyModel, historyView);

  chrome.send('queryHistory',
        ['', 0, 0, 0, 0, 0, true]);

  $('term').setAttribute('placeholder', localStrings.getString('IDS_HISTORY_SEARCH_INPUT_TIPS'));

  // Setup click handlers.
  $('history-section').onclick = function () {
    var first_li = $('left_nav').querySelector('li');
    if(first_li && !first_li.classList.contains('cur')){
      first_li.click();
    }
    return false;
  };
  $('search-form').onsubmit = function() {
    if(this.term.value!='')
    setSearch(this.term.value);
    else {
      // $('history-section').click();
      if (window.last_date_link) {
        last_date_link.click();
      }
    }
    return false;
  };
  $('term').addEventListener('keyup',function(e){
    if(this.value!=''){
      $('delTerm').style.display = 'block';
    }else{
      $('delTerm').style.display = 'none';
    }
  });
  $('delTerm').addEventListener('click',function(e){
    if(historyModel.searchText_!=''){
      historyModel.view_.summaryTd_.innerText = '';
      var first_li = window.last_date_link || $('left_nav').querySelector('li'),
      cur_li = $('left_nav').querySelector('li.cur');
      if(!cur_li && first_li) first_li.classList.add('cur');
      setSearch('');
    }
    $('term').value = '';
  });
  
  myDelegateInit();

  window.onhashchange = function(){
    if (!window.hashchanged) {
        var hashData = pageState.getHashData();
        historyView.setSearch(hashData.q, hashData.p);
        if (!hashData.q) {
          var first_li = window.last_date_link || $('left_nav').querySelector('li');
          first_li.click();
        }
    }
    window.hashchanged = false;
  };
}

function myDelegateInit(){
  var getActiveVisit = function(e){
    var entry = findAncestorByClass(e, 'entry');
    return historyModel.visits_[entry.querySelector('a').getAttribute('id').replace('id-','')];
  },
  pel = 'results-display';
  myDelegate(pel, 'li', 'mouseover', function(e){
    e.classList.add('liHover');
    e.querySelector('.del') && e.querySelector('.del').classList.remove('hidden');
  });
  myDelegate(pel, 'li', 'mouseout', function(e){
    e.classList.remove('liHover');
    e.querySelector('.del') && e.querySelector('.del').classList.add('hidden');
  });
  myDelegate(pel, '.del', 'click', function(e){
    if(e.classList.contains('group-hd-del')){
      var li = e.parentNode.parentNode,
      checked_del = [],
      get_group_checkbox = function(el){
        if(el && !el.classList.contains('group-hd')){
          var cb = el.querySelector('input[type=checkbox]:checked:not([disabled])');
          if(cb) checked_del.push(cb);
          get_group_checkbox(el.nextSibling);
        }
      }
      ;
      get_group_checkbox(li.nextSibling);
      li.classList.add('fade-out');
      removeItems(checked_del);
    }else{
      activeVisit = getActiveVisit(e);
    activeVisit.removeFromHistory_();
    activeVisit = null;
    }
  });
  myDelegate(pel, '.search', 'click', function(e){
    activeVisit = getActiveVisit(e);
    activeVisit.showMoreFromSite_();
    activeVisit = null;
  });
}

/**
 * TODO(glen): Get rid of this function.
 * Set the history view to a specified page.
 * @param {String} term The string to search for
 */
function setSearch(term) {
  if (historyView) {
    historyView.setSearch(term);
  }
}

/**
 * TODO(glen): Get rid of this function.
 * Set the history view to a specified page.
 * @param {number} page The page to set the view to.
 */
function setPage(page) {
  if (historyView) {
    historyView.setPage(page);
  }
}

/**
 * Delete the next item in our deletion queue.
 */
function deleteNextInQueue() {
  if (deleteQueue.length > 0) {
    // Call the native function to remove history entries.
    // First arg is a time in seconds (passed as String) identifying the day.
    // Remaining args are URLs of history entries from that day to delete.
    var timeInSeconds = Math.floor(deleteQueue[0].date.getTime() / 1000);
    chrome.send('removeURLsOnOneDay',
                [String(timeInSeconds)].concat(deleteQueue[0].urls));
  }
}

/**
 * Open the clear browsing data dialog.
 */
function openClearBrowsingData() {
  chrome.send('clearBrowsingData', []);
  return false;
}

/**
 * Queue a set of URLs from the same day for deletion.
 * @param {Date} date A date indicating the day the URLs were visited.
 * @param {Array} urls Array of URLs from the same day to be deleted.
 * @param {Function} opt_callback An optional callback to be executed when
 *        the deletion is complete.
 */
function queueURLsForDeletion(date, urls, opt_callback) {
  deleteQueue.push({ 'date': date, 'urls': urls, 'callback': opt_callback });
}

function reloadHistory() {
  historyView.reload();
}

function setVisitRemovedById(id){
  for(var i=0; i<historyModel.visits_.length; i++){
    if(historyModel.visits_[i].id_ == id){
      historyModel.visits_[i].removed_ = true;
      break;
    }
  }
}

/**
 * Collect IDs from checked checkboxes and send to Chrome for deletion.
 */
function removeItems(checked_del) {
  if(!(checked_del instanceof Array)) checked_del = null;
  var checked = checked_del || document.querySelectorAll(
      'input[type=checkbox]:checked:not([disabled])');
  var urls = [];
  var disabledItems = [];
  var queue = [];
  var date = new Date();
  var delVisits = [],
      pushDelVisit = function(time,url){
        for(var i=0; i<historyModel.visits_.length; i++){
          if(historyModel.visits_[i].time.getTime()==time && trim(historyModel.visits_[i].url_)== url){
            delVisits.push(historyModel.visits_[i]);
          }
        };
      },
      onSuccessCallback = function() {
        for(var i = 0; i<delVisits.length; i++){
          setVisitRemovedById(delVisits[i].id_);
          if(delVisits[i].domNode_) removeEntryFromView(delVisits[i].domNode_);
        }
        if(!checked_del){
          var group_hd_arr = $('results-display').querySelectorAll('li.group-hd');
          for(var i=0; i<group_hd_arr.length; i++){
              group_hd_arr[i].classList.add('fade-out');
          }
        }
      },
      onSuccessCallbackLast = function() {
        location.reload();
      }
  ;
  var rd_fc = $('results-display').firstChild,
  date_results = rd_fc.nodeName == 'H2';
  if(date_results) date = new Date(rd_fc.time);
  for (var i = 0; i < checked.length; i++) {
    var checkbox = checked[i];
    if(!date_results){
    var cbDate = new Date(checkbox.time);
    if (date.getFullYear() != cbDate.getFullYear() ||
        date.getMonth() != cbDate.getMonth() ||
        date.getDate() != cbDate.getDate()) {
      if (urls.length > 0) {
        queue.push([date, urls]);
      }
      urls = [];
      date = cbDate;
    }
    }
    var link = findAncestorWithClass(checkbox, 'entry').querySelector('a');
    checkbox.disabled = true;
    link.classList.add('to-be-removed');
    disabledItems.push(checkbox);
    urls.push(link.href);
    pushDelVisit(checkbox.time, link.href);
  }
  if (urls.length > 0) {
    queue.push([date, urls]);
  }
  var confirm_text = localStrings.getString('deletewarning');
  if(is_cn){
    if(historyModel.getSearchText()){
      confirm_text = localStrings.getString('IDS_HISTORY_DEL_SEARCH_CONFIRM');
    }else{
      var date_arr = $('left_nav').querySelector('li.cur').getAttribute('date').split('-'),
      date_text = date_arr[0] + '\u5E74' + date_arr[1] + '\u6708' + date_arr[2] + '\u65E5';
      confirm_text = '\u786E\u5B9A\u8981\u5220\u9664' + date_text + '\u5168\u5929\u7684\u5386\u53F2\u8BB0\u5F55\u5417\uFF1F';
    }
  }
  if (checked.length > 0 && (checked_del || confirm(confirm_text))) {
    var q_len = queue.length;
    for (var i = 0; i < q_len; i++) {
      // Reload the page when the final entry has been deleted.
      //var callback = i == 0 ? reloadHistory : null;
      var callback = (i == 0 && q_len > 1 || checked_del) ? onSuccessCallback : ((i == q_len - 1)? onSuccessCallbackLast : null);
      queueURLsForDeletion(queue[i][0], queue[i][1], callback);
    }
    deleteNextInQueue();
  } else {
    // If the remove is cancelled, return the checkboxes to their
    // enabled, non-line-through state.
    for (var i = 0; i < disabledItems.length; i++) {
      var checkbox = disabledItems[i];
      var link = findAncestorWithClass(
          checkbox, 'entry').querySelector('a');
      checkbox.disabled = false;
      link.classList.remove('to-be-removed');
    }
  }
  return false;
}

/**
 * Toggle state of checkbox and handle Shift modifier.
 */
function checkboxClicked(event) {
  var id = Number(this.id.slice("checkbox-".length));
  if (event.shiftKey && (selectionAnchor != -1)) {
    var checked = this.checked;
    // Set all checkboxes from the anchor up to the clicked checkbox to the
    // state of the clicked one.
    var begin = Math.min(id, selectionAnchor);
    var end = Math.max(id, selectionAnchor);
    for (var i = begin; i <= end; i++) {
      var checkbox = document.querySelector('#checkbox-' + i);
      if (checkbox)
        checkbox.checked = checked;
    }
  }
  selectionAnchor = id;

  historyView.updateRemoveButton();
}

function entryBoxMousedown(event) {
  // Prevent text selection when shift-clicking to select multiple entries.
  if (event.shiftKey) {
    event.preventDefault();
  }
}

function removeNode(node) {
  node.classList.add('fade-out'); // Trigger CSS fade out animation.
  node.querySelector('.time input').checked = false;
  historyModel.view_.updateRemoveButton();

  // Delete the node when the animation is complete.
  node.addEventListener('webkitTransitionEnd', function() {
    //node.parentNode.removeChild(node);
  });
}

/**
 * Removes a single entry from the view. Also removes gaps before and after
 * entry if necessary.
 */
function removeEntryFromView(entry) {
  var nextEntry = entry.nextSibling;
  var previousEntry = entry.previousSibling;

  removeNode(entry);

  // if there is no previous entry, and the next entry is a gap, remove it
  if (!previousEntry && nextEntry && nextEntry.className == 'gap') {
    removeNode(nextEntry);
  }

  // if there is no next entry, and the previous entry is a gap, remove it
  if (!nextEntry && previousEntry && previousEntry.className == 'gap') {
    removeNode(previousEntry);
  }

  // if both the next and previous entries are gaps, remove one
  if (nextEntry && nextEntry.className == 'gap' &&
      previousEntry && previousEntry.className == 'gap') {
    removeNode(nextEntry);
  }
}

///////////////////////////////////////////////////////////////////////////////
// Chrome callbacks:
/**
 * Our history system calls this function with results from searches.
 */
function historyResult(info, results) {
  info.finished = true;
  historyModel.visits_ = [];
  historyModel.visits_info = info;
  if(info.has_history_days) historyModel.makeLeftNav(info, results);
  else historyModel.addResults(info, results);
}

/**
 * Our history system calls this function when a deletion has finished.
 */
function deleteComplete() {
  if (deleteQueue.length > 0) {
    // Remove the successfully deleted entry from the queue.
    if (deleteQueue[0].callback)
      deleteQueue[0].callback.apply();
    deleteQueue.splice(0, 1);
    deleteNextInQueue();
  } else {
    console.error('Received deleteComplete but queue is empty.');
  }
}

/**
 * Our history system calls this function if a delete is not ready (e.g.
 * another delete is in-progress).
 */
function deleteFailed() {
  window.console.log('Delete failed');

  // The deletion failed - try again later.
  // TODO(dubroy): We should probably give up at some point.
  setTimeout(deleteNextInQueue, 500);
}

/**
 * Called when the history is deleted by someone else.
 */
function historyDeleted() {
  window.console.log('History deleted');
  if(deleteQueue.length == 0) location.reload();
  //var anyChecked = document.querySelector('.entry input:checked') != null;
  // Reload the page, unless the user has any items checked.
  // TODO(dubroy): We should just reload the page & restore the checked items.
  //if (!anyChecked)
  //  historyView.reload();
}

// Add handlers to HTML elements.
document.addEventListener('DOMContentLoaded', load);

// This event lets us enable and disable menu items before the menu is shown.
document.addEventListener('canExecute', function(e) {
  e.canExecute = true;
});

// scroll
(function(){
  var myScroll = function(opt){
    this.init && this.init(opt);
  };
  myScroll.prototype = {
    init: function(opt){
      opt.width = opt.width || 10;
      this.defaults = opt;
      this.forObj = (typeof(this.defaults.obj) == 'string')?document.getElementById(this.defaults.obj):this.defaults.obj;

      this.forObj.style.overflow = 'hidden';
      this.bar = document.createElement('div');
      this.bar.style.position = 'absolute';
      this.bar.style.zIndex = '99999';
      document.body.appendChild(this.bar);
      this.barIndex = document.createElement('div');
      this.barIndex.style.position = 'absolute';
      this.bar.appendChild(this.barIndex);

      this.barWidth = this.defaults.width;
      if(!this.defaults.class){
        this.bar.style.backgroundColor = 'rgba(230,230,230,.9)';
        this.barIndex.style.backgroundColor = 'rgba(0,0,0,.3)';
      }
      this.bar.style.cursor = 'default';
      this.barIndex.style.cursor = 'default';
      this.bar.style.borderRadius = '7px';
      this.barIndex.style.borderRadius = '7px';

      this.setSize_();


      this.forObj.addEventListener('mousewheel', this.scrollmove_.bind(this), true);
      this.bar.addEventListener('mousewheel', this.scrollmove_.bind(this), true);
      this.barIndex.addEventListener('mousewheel', this.scrollmove_.bind(this), true);
      this.barIndex.addEventListener('mousedown', this.mousedown_.bind(this), true);
      document.addEventListener('mouseup', this.mouseup_.bind(this), true);
      document.addEventListener('mousemove', this.mousemove_.bind(this), true);
      window.addEventListener('resize', this.resize_.bind(this), true);

      if(this.defaults.over_show){
        this.bar.style.display = 'none';
        this.forObj.addEventListener('mouseover', this.mouseover_.bind(this), true);
        this.forObj.addEventListener('mouseout', this.mouseout_.bind(this), true);
        this.bar.addEventListener('mouseover', this.mouseover_.bind(this), true);
        this.bar.addEventListener('mouseout', this.mouseout_.bind(this), true);
        this.barIndex.addEventListener('mouseover', this.mouseover_.bind(this), true);
        this.barIndex.addEventListener('mouseout', this.mouseout_.bind(this), true);
      }else{
        this.show_bar();
      }
    },
    can_select_: function(can){
      document.onselectstart = function(){return can};
      document.ondragstart = function(){return can};
      document.onselect = function(){return can};
    },
    resize_: function(){
      this.setSize_();
      if(this.defaults.over_show) this.bar.style.display = 'none';
    },
    mouseover_: function(e){
      if(this.bar.style.display != 'none') return;
      this.show_bar();
    },
    mouseout_: function(e){
      if(!!!this.barIndex.downing && this.bar.style.display != 'none'){
        this.can_hide_bar = true;
        this.hide_bar();
      }
    },
    scrollmove_: function(e){
      if(this.forObj.scrollHeight <= this.forObj.clientHeight) return;
      e = e || event;
      var step = 20;
      if(e.wheelDelta < 0) {
        if(this.forObj.scrollTop >= (this.forObj.scrollHeight - this.forObj.clientHeight)) return;
        this.forObj.scrollTop += step;
      } else {
        if(this.forObj.scrollTop == 0) return;
        this.forObj.scrollTop -= step;
      }
      this.setBarIndexPosition_();
    },
    mousedown_: function(e){
      e = e || event;
      this.barIndex.downing = true;
      this.forObjScrollY = this.forObj.scrollTop;
      this.pageScrollY = e.clientY;
      this.can_select_(false);
      if(!this.defaults.class){
        this.barIndex.style.backgroundColor = 'rgba(0,0,0,.4)';
      }
      this.show_bar();
    },
    mouseup_: function(e){
      e = e || event;
      this.barIndex.downing = false;
      this.can_select_(true);
      if(!this.defaults.class){
        this.barIndex.style.backgroundColor = 'rgba(0,0,0,.3)';
      }
      if(this.defaults.over_show){
        if((this.forObj.offsetLeft < e.clientX && this.forObj.offsetLeft + this.forObj.clientWidth > e.clientX)
           && (this.forObj.offsetTop < e.clientY && this.forObj.offsetTop + this.forObj.clientHeight > e.clientY)
        ){
          this.show_bar();
        }else{
          this.can_hide_bar = true;
          this.hide_bar();
        }
      }
    },
    mousemove_: function(e){
      if(!this.barIndex.downing) return;
      var per = (this.forObj.scrollHeight - this.forObj.clientHeight) / (this.forObj.clientHeight - this.barHeight);
      this.forObj.scrollTop = this.forObjScrollY - (this.pageScrollY - e.clientY) * per;
      this.setBarIndexPosition_();
    },
    hide_bar: function(){
      var this_ = this;
      // this.hide_bar_timer = setTimeout(function(){
        this_.bar.style.display = 'none';
      // }, 800);
     
    },
    show_bar: function(){
      if(this.forObj.scrollHeight > this.forObj.clientHeight){
        this.bar.style.display = 'block';
        clearTimeout(this.hide_bar_timer);
      }
    },
    reset_: function(){
      this.setSize_(true);
    },
    setSize_: function(is_reset){
      if(this.forObj.scrollHeight <= this.forObj.clientHeight) {
        this.forObj.scrollTop = 0;
        this.bar.style.display = 'none';
      } else {
        if(is_reset) this.forObj.scrollTop = 0;
        this.bar.style.display = 'block';
      }
      var x=0, y=0;
      var p = this.forObj;
      while(p) {
        x += p.offsetLeft;
        y += p.offsetTop;
        p = p.offsetParent;
      }
      this.bar.style.width = this.barWidth + 'px';
      this.bar.style.height = (this.forObj.offsetHeight) + 'px';
      this.bar.style.top = y + 'px';
      this.bar.style.left = x + this.forObj.offsetWidth - this.barWidth + 'px';
      this.barIndex.style.width = this.barWidth + 'px';
      var h = this.forObj.clientHeight - (this.forObj.scrollHeight - this.forObj.clientHeight);
      if(h < 30) {
        h = 30;
      }
      this.barHeight = h;
      this.barIndex.style.height = h + 'px';
      this.barIndex.style.left = '0px';
      this.setBarIndexPosition_();
    },
    setBarIndexPosition_: function(){
      this.barIndex.style.top = (this.forObj.offsetHeight - this.barHeight) * this.forObj.scrollTop / (this.forObj.scrollHeight - this.forObj.clientHeight) + 'px';
    }
  };
  window.myScroll = myScroll;
})();
// scroll end
