﻿// CCDecode: A library for decoding digital public safety standards.
// 
//  Author: Gabriel Graves
// Website: http://www.ccdecode.com/
// 
// This software has been released with no license and is public domain.
// You are free to do anything with the code for any purpose with out
// permission.

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace CCDecodeP25
{
    /// <summary>
    /// Container for a P25 control message.
    /// </summary>
    public class P25Message
    {
        /// <summary>
        /// Octet 0 (hex form)
        /// </summary>
        public String oct0 { get; set; }

        /// <summary>
        /// Octet 1 (hex form)
        /// </summary>
        public String oct1 { get; set; }

        /// <summary>
        /// Octet 2 (hex form)
        /// </summary>
        public String oct2 { get; set; }

        /// <summary>
        /// Octet 3 (hex form)
        /// </summary>
        public String oct3 { get; set; }

        /// <summary>
        /// Octet 4 (hex form)
        /// </summary>
        public String oct4 { get; set; }

        /// <summary>
        /// Octet 5 (hex form)
        /// </summary>
        public String oct5 { get; set; }

        /// <summary>
        /// Octet 6 (hex form)
        /// </summary>
        public String oct6 { get; set; }

        /// <summary>
        /// Octet 7 (hex form)
        /// </summary>
        public String oct7 { get; set; }

        /// <summary>
        /// Octet 8 (hex form)
        /// </summary>
        public String oct8 { get; set; }

        /// <summary>
        /// Octet 9 (hex form)
        /// </summary>
        public String oct9 { get; set; }

        /// <summary>
        /// Octet 10 (hex form)
        /// </summary>
        public String oct10 { get; set; }

        /// <summary>
        /// Octet 11 (hex form)
        /// </summary>
        public String oct11 { get; set; }

        /// <summary>
        /// Octet 0 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN0
        {
            get
            {
                if (String.IsNullOrEmpty(oct0) || String.IsNullOrWhiteSpace(oct0))
                {
                    throw new P25OctetIsNull("Octet 0 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct0, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct0!");
                }
            }
        }

        /// <summary>
        /// Octet 1 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN1
        {
            get
            {
                if (String.IsNullOrEmpty(oct1) || String.IsNullOrWhiteSpace(oct1))
                {
                    throw new P25OctetIsNull("Octet 1 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct1, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct1!");
                }
            }
        }

        /// <summary>
        /// Octet 2 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN2
        {
            get
            {
                if (String.IsNullOrEmpty(oct2) || String.IsNullOrWhiteSpace(oct2))
                {
                    throw new P25OctetIsNull("Octet 2 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct2, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct2!");
                }
            }
        }

        /// <summary>
        /// Octet 3 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN3
        {
            get
            {
                if (String.IsNullOrEmpty(oct3) || String.IsNullOrWhiteSpace(oct3))
                {
                    throw new P25OctetIsNull("Octet 3 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct3, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct3!");
                }
            }
        }

        /// <summary>
        /// Octet 4 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN4
        {
            get
            {
                if (String.IsNullOrEmpty(oct4) || String.IsNullOrWhiteSpace(oct4))
                {
                    throw new P25OctetIsNull("Octet 4 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct4, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct4!");
                }
            }
        }

        /// <summary>
        /// Octet 5 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN5
        {
            get
            {
                if (String.IsNullOrEmpty(oct5) || String.IsNullOrWhiteSpace(oct5))
                {
                    throw new P25OctetIsNull("Octet 5 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct5, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct5!");
                }
            }
        }

        /// <summary>
        /// Octet 6 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN6
        {
            get
            {
                if (String.IsNullOrEmpty(oct6) || String.IsNullOrWhiteSpace(oct6))
                {
                    throw new P25OctetIsNull("Octet 6 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct6, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct6!");
                }
            }
        }

        /// <summary>
        /// Octet 7 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN7
        {
            get
            {
                if (String.IsNullOrEmpty(oct7) || String.IsNullOrWhiteSpace(oct7))
                {
                    throw new P25OctetIsNull("Octet 7 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct7, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct7!");
                }
            }
        }

        /// <summary>
        /// Octet 8 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN8
        {
            get
            {
                if (String.IsNullOrEmpty(oct8) || String.IsNullOrWhiteSpace(oct8))
                {
                    throw new P25OctetIsNull("Octet 8 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct8, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct8!");
                }
            }
        }

        /// <summary>
        /// Octet 9 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN9
        {
            get
            {
                if (String.IsNullOrEmpty(oct9) || String.IsNullOrWhiteSpace(oct9))
                {
                    throw new P25OctetIsNull("Octet 9 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct9, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct9!");
                }
            }
        }

        /// <summary>
        /// Octet 10 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN10
        {
            get
            {
                if (String.IsNullOrEmpty(oct10) || String.IsNullOrWhiteSpace(oct10))
                {
                    throw new P25OctetIsNull("Octet 10 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct10, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct10!");
                }
            }
        }

        /// <summary>
        /// Octet 11 (binary form)
        /// </summary>
        /// <exception cref="CCDecodeP25.P25OctetIsNull">Thrown when the octet is null.</exception>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when the octet is invalid (thus the message is invalid)</exception>
        public String octBIN11
        {
            get
            {
                if (String.IsNullOrEmpty(oct11) || String.IsNullOrWhiteSpace(oct11))
                {
                    throw new P25OctetIsNull("Octet 11 is null!");
                }

                try
                {
                    return Convert.ToString(Convert.ToInt32(oct11, 16), 2).PadLeft(8, '0');
                }
                catch
                {
                    throw new P25InvalidMessage("Invalid oct11!");
                }
            }
        }

        /// <summary>
        /// Indicates if this message is the last one.
        /// </summary>
        public bool LastBlock
        {
            get
            {
                return (octBIN0.Substring(0, 1).Equals("1"));
            }
        }

        /// <summary>
        /// Indicates if this message is protected (encrypted). If this value is true, further decoding may not be possible.
        /// </summary>
        public bool ProtectedTrunking
        {
            get
            {
                return (octBIN0.Substring(1, 1).Equals("1"));
            }
        }

        /// <summary>
        /// Binary representation of this message.
        /// </summary>
        public String opCode
        {
            get
            {
                return octBIN0.Substring(2);
            }
        }

        /// <summary>
        /// Manufacturer ID of this message
        /// </summary>
        public int ManufacturerID
        {
            get
            {
                return Convert.ToInt32(oct1, 16);
            }
        }

        /// <summary>
        /// Converts the P25 message into a string without separating the octets.
        /// </summary>
        /// <returns>Returns the P25 message without separating the octets.</returns>
        public String ToString()
        {
            return ToString(false);
        }

        /// <summary>
        /// Converts the P25 message into a string.
        /// </summary>
        /// <param name="SeparateOctets">If true, returned string will separate octets with a space.</param>
        /// <returns>Returns the P25 message.</returns>
        public String ToString(bool SeparateOctets)
        {
            return ToString(SeparateOctets, " ");
        }

        /// <summary>
        /// Converts the P25 message into a string.
        /// </summary>
        /// <param name="SeparateOctets">If true, returned string will separate octets with the value from the parameter 'Separator'.</param>
        /// <param name="Separator">If the parameter 'SeparateOctets' is true, this will be used as the separator.</param>
        /// <returns>Returns the P25 message.</returns>
        public String ToString(bool SeparateOctets, String Separator)
        {
            if (SeparateOctets)
            {
                return oct0 + Separator + oct1 + Separator + oct2 + Separator + oct3 + Separator + oct4 + Separator + oct5 + Separator + oct6 + Separator + oct7 + Separator + oct8 + Separator + oct9 + Separator + oct10 + Separator + oct11;
            }
            else
            {
                return oct0 + oct1 + oct2 + oct3 + oct4 + oct5 + oct6 + oct7 + oct8 + oct9 + oct10 + oct11;
            }
        }

        /// <summary>
        /// Creates a new blank P25 message.
        /// </summary>
        public P25Message() { }

        /// <summary>
        /// Creates a new P25 message.
        /// </summary>
        /// <param name="oct0">Octet 0 (hex form)</param>
        /// <param name="oct1">Octet 1 (hex form)</param>
        /// <param name="oct2">Octet 2 (hex form)</param>
        /// <param name="oct3">Octet 3 (hex form)</param>
        /// <param name="oct4">Octet 4 (hex form)</param>
        /// <param name="oct5">Octet 5 (hex form)</param>
        /// <param name="oct6">Octet 6 (hex form)</param>
        /// <param name="oct7">Octet 7 (hex form)</param>
        /// <param name="oct8">Octet 8 (hex form)</param>
        /// <param name="oct9">Octet 9 (hex form)</param>
        /// <param name="oct10">Octet 10 (hex form)</param>
        /// <param name="oct11">Octet 11 (hex form)</param>
        public P25Message(string oct0, string oct1, string oct2, string oct3, string oct4, string oct5, string oct6, string oct7, string oct8, string oct9, string oct10, string oct11)
        {
            this.oct0 = oct0;
            this.oct1 = oct1;
            this.oct2 = oct2;
            this.oct3 = oct3;
            this.oct4 = oct4;
            this.oct5 = oct5;
            this.oct6 = oct6;
            this.oct7 = oct7;
            this.oct8 = oct8;
            this.oct9 = oct9;
            this.oct10 = oct10;
            this.oct11 = oct11;
        }
    }
}
