﻿// CCDecode: A library for decoding digital public safety standards.
// 
//  Author: Gabriel Graves
// Website: http://www.ccdecode.com/
// 
// This software has been released with no license and is public domain.
// You are free to do anything with the code for any purpose with out
// permission.

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace CCDecodeP25
{
    /// <summary>
    /// P25 Decoder
    /// </summary>
    public partial class P25Decode
    {
        /// <summary>
        /// Container for Frequency Identifiers received.
        /// </summary>
        public P25FrequencyIdentifiers ChannelIdentifiers { get; set; }

        /// <summary>
        /// Current System ID
        /// </summary>
        public int SystemID { get; set; }

        /// <summary>
        /// Current System WACN
        /// </summary>
        public String SystemWACN { get; set; }

        /// <summary>
        /// Last Group Voice Channel Grant (single block) received used to prevent duplicate message.
        /// </summary>
        private P25GroupVoiceChannelGrantSingleBlock _LastGroupVoiceChannelGrantSingleBlock { get; set; }

        /// <summary>
        /// Timestamp of last Group Voice Channel Grant (single block) received used to prevent duplicate message.
        /// </summary>
        private DateTime _LastGroupVoiceChannelGrantSingleBlockTimestamp { get; set; }

        /// <summary>
        /// Last Group Affiliation Response received used to prevent duplicate message.
        /// </summary>
        private P25GroupAffiliationResponse _LastGroupAffiliation { get; set; }

        /// <summary>
        /// Timestamp of last Group Affiliation Response received used to prevent duplicate message.
        /// </summary>
        private DateTime _LastGroupAffiliationTimestamp { get; set; }

        /// <summary>
        /// Last De-Registration Acknowledge received used to prevent duplicate message.
        /// </summary>
        private P25DeregistrationAcknowledge _LastDeregistration { get; set; }

        /// <summary>
        /// Timestamp of last De-Registration Acknowledge received used to prevent duplicate message.
        /// </summary>
        private DateTime _LastDeregistrationTimestamp { get; set; }

        /// <summary>
        /// Creates a new P25 decoder.
        /// </summary>
        public P25Decode()
        {
            ChannelIdentifiers = new P25FrequencyIdentifiers();
        }

        /// <summary>
        /// Decodes the P25 Message.
        /// </summary>
        /// <param name="rawmessage">Raw P25 Message</param>
        /// <param name="StopDuplicates">Throws an exception when duplicates are recognized. This does not apply to all duplicates such as those intended to be received more than once.</param>
        /// <param name="WarnOnProtection">Throws an exception if the message is protected (encrypted).</param>
        /// <returns>Returns data decoded from the message. If this method returns null as the MessageType then the message is not recognized.</returns>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when an invalid message is passed.</exception>
        /// <exception cref="CCDecodeP25.P25ProtectedMessage">Thrown if WarnOnProtection is true and the message is protected.</exception>
        public P25Return DecodeMessage(String rawmessage, bool StopDuplicates = true, bool WarnOnProtection = false)
        {
            String message = Utilities.getP25Message(rawmessage);
            String oct0;
            String oct1;
            String oct2;
            String oct3;
            String oct4;
            String oct5;
            String oct6;
            String oct7;
            String oct8;
            String oct9;
            String oct10;
            String oct11;

            oct0 = message.Substring(0, 2);
            oct1 = message.Substring(2, 2);
            oct2 = message.Substring(4, 2);
            oct3 = message.Substring(6, 2);
            oct4 = message.Substring(8, 2);
            oct5 = message.Substring(10, 2);
            oct6 = message.Substring(12, 2);
            oct7 = message.Substring(14, 2);
            oct8 = message.Substring(16, 2);
            oct9 = message.Substring(18, 2);
            oct10 = message.Substring(20, 2);
            oct11 = message.Substring(22, 2);

            return DecodeMessage(oct0, oct1, oct2, oct3, oct4, oct5, oct6, oct7, oct8, oct9, oct10, oct11, StopDuplicates, WarnOnProtection);
        }

        /// <summary>
        /// Decodes the P25 Message.
        /// </summary>
        /// <param name="oct0">Octet 0 (hex form)</param>
        /// <param name="oct1">Octet 1 (hex form)</param>
        /// <param name="oct2">Octet 2 (hex form)</param>
        /// <param name="oct3">Octet 3 (hex form)</param>
        /// <param name="oct4">Octet 4 (hex form)</param>
        /// <param name="oct5">Octet 5 (hex form)</param>
        /// <param name="oct6">Octet 6 (hex form)</param>
        /// <param name="oct7">Octet 7 (hex form)</param>
        /// <param name="oct8">Octet 8 (hex form)</param>
        /// <param name="oct9">Octet 9 (hex form)</param>
        /// <param name="oct10">Octet 10 (hex form)</param>
        /// <param name="oct11">Octet 11 (hex form)</param>
        /// <param name="StopDuplicates">Throws an exception when duplicates are recognized. This does not apply to all duplicates such as those intended to be received more than once.</param>
        /// <param name="WarnOnProtection">Throws an exception if the message is protected (encrypted).</param>
        /// <returns>Returns data decoded from the message. If this method returns null as the MessageType then the message is not recognized.</returns>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when an invalid message is passed.</exception>
        /// <exception cref="CCDecodeP25.P25ProtectedMessage">Thrown if WarnOnProtection is true and the message is protected.</exception>
        public P25Return DecodeMessage(string oct0, string oct1, string oct2, string oct3, string oct4, string oct5, string oct6, string oct7, string oct8, string oct9, string oct10, string oct11, bool StopDuplicates = true, bool WarnOnProtection = false)
        {
            return DecodeMessage(new P25Message(oct0, oct1, oct2, oct3, oct4, oct5, oct6, oct7, oct8, oct9, oct10, oct11), StopDuplicates, WarnOnProtection);
        }

        /// <summary>
        /// Decodes the P25 Message.
        /// </summary>
        /// <param name="message">P25 Message</param>
        /// <param name="StopDuplicates">Throws an exception when duplicates are recognized. This does not apply to all duplicates such as those intended to be received more than once.</param>
        /// <param name="WarnOnProtection">Throws an exception if the message is protected (encrypted).</param>
        /// <returns>Returns data decoded from the message. If this method returns null as the MessageType then the message is not recognized.</returns>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown when an invalid message is passed.</exception>
        /// <exception cref="CCDecodeP25.P25ProtectedMessage">Thrown if WarnOnProtection is true and the message is protected.</exception>
        public P25Return DecodeMessage(P25Message message, bool StopDuplicates = true, bool WarnOnProtection = false)
        {
            P25Return MethodReturn = new P25Return();

            MethodReturn.OriginalMessage = message;
            MethodReturn.TimeStamp = DateTime.Now;
            MethodReturn.ManufacturerID = message.ManufacturerID;
            MethodReturn.opCode = message.opCode;
            MethodReturn.ProtectedTrunking = message.ProtectedTrunking;
            MethodReturn.MessageType = P25ReturnType.Unknown;

            if (message.ProtectedTrunking)
            {
                if (WarnOnProtection)
                {
                    throw new P25ProtectedMessage();
                }
            }
            else
            {
                switch (message.opCode)
                {
                    case "000000":
                        if (message.ManufacturerID == 0x00)
                        {
                            MethodReturn.MessageType = P25ReturnType.GroupVoiceChannelGrant;
                            MethodReturn.dataGroupVoiceChannelGrant = HandleGroupVoiceChannelGrantSingleBlock(message);

                            if (StopDuplicates && _LastGroupVoiceChannelGrantSingleBlock != null)
                            {
                                P25GroupVoiceChannelGrantSingleBlock CurrentVoiceGrant = MethodReturn.dataGroupVoiceChannelGrant;

                                bool DuplicateFound = false;
                                try
                                {
                                    DuplicateFound = (_LastGroupVoiceChannelGrantSingleBlock != null && _LastGroupVoiceChannelGrantSingleBlockTimestamp != null && MethodReturn.TimeStamp >= _LastGroupVoiceChannelGrantSingleBlockTimestamp.AddSeconds(-1) && CurrentVoiceGrant.TalkgroupID == _LastGroupVoiceChannelGrantSingleBlock.TalkgroupID && CurrentVoiceGrant.RadioID == _LastGroupVoiceChannelGrantSingleBlock.RadioID);
                                }
                                catch { }
                                if (DuplicateFound)
                                {
                                    throw new P25DuplicateDetected("Duplate Group Voice Channel Grant (Single Block)");
                                }
                            }

                            _LastGroupVoiceChannelGrantSingleBlock = MethodReturn.dataGroupVoiceChannelGrant;
                            _LastGroupVoiceChannelGrantSingleBlockTimestamp = MethodReturn.TimeStamp;
                        }
                        else if (message.ManufacturerID == 0x90)
                        {
                            MethodReturn.MessageType = P25ReturnType.MotorolaPatch;
                            MethodReturn.dataMotorolaPatch = HandleMotorolaPatch(message);
                        }
                        break;
                    case "000010":
                        if (message.ManufacturerID == 0x00)
                        {
                            MethodReturn.MessageType = P25ReturnType.GroupVoiceChannelGrantUpdate;
                            MethodReturn.dataGroupVoiceChannelGrantUpdate = HandleGroupVoiceChannelGrantUpdate(message);
                        }
                        break;
                    case "101000":
                        if (message.ManufacturerID == 0x00)
                        {
                            MethodReturn.MessageType = P25ReturnType.GroupAffiliationResponse;
                            MethodReturn.dataGroupAffiliationResponse = HandleGroupAffiliationResponse(message);

                            if (StopDuplicates && _LastGroupAffiliation != null)
                            {
                                P25GroupAffiliationResponse CurrentAffiliationResponse = MethodReturn.dataGroupAffiliationResponse;

                                bool DupliateFound = false;
                                try
                                {
                                    DupliateFound = (_LastGroupAffiliation != null && _LastGroupAffiliationTimestamp != null && MethodReturn.TimeStamp >= _LastGroupAffiliationTimestamp.AddSeconds(-1) && CurrentAffiliationResponse.AnnouncementTalkgroupID == _LastGroupAffiliation.AnnouncementTalkgroupID && CurrentAffiliationResponse.GroupAffiliationValue == _LastGroupAffiliation.GroupAffiliationValue && CurrentAffiliationResponse.LocalOrGlobal == _LastGroupAffiliation.LocalOrGlobal && CurrentAffiliationResponse.RadioID == _LastGroupAffiliation.RadioID && CurrentAffiliationResponse.TalkgroupID == _LastGroupAffiliation.TalkgroupID);
                                }
                                catch { }
                                if (DupliateFound)
                                {
                                    throw new P25DuplicateDetected("Duplate Group Affiliation Response");
                                }
                            }

                            _LastGroupAffiliation = MethodReturn.dataGroupAffiliationResponse;
                            _LastGroupAffiliationTimestamp = MethodReturn.TimeStamp;
                        }
                        break;
                    case "101100":
                        if (message.ManufacturerID == 0x00)
                        {
                            MethodReturn.MessageType = P25ReturnType.UnitRegistrationResponse;
                            MethodReturn.dataUnitRegistrationResponse = HandleUnitRegistrationResponse(message);
                        }
                        break;
                    case "101111":
                        if (message.ManufacturerID == 0x00)
                        {
                            MethodReturn.MessageType = P25ReturnType.DeregistrationAcknowledge;
                            MethodReturn.dataDeregistrationAcknowledge = HandleDeregistrationAcknowledge(message);

                            if (StopDuplicates && _LastDeregistration != null)
                            {
                                P25DeregistrationAcknowledge CurrentDeregistration = MethodReturn.dataDeregistrationAcknowledge;

                                bool DuplicateFound = false;
                                try
                                {
                                    DuplicateFound = (_LastDeregistration != null && _LastDeregistrationTimestamp != null && MethodReturn.TimeStamp >= _LastDeregistrationTimestamp.AddSeconds(-1) && CurrentDeregistration.RadioID == _LastDeregistration.RadioID && CurrentDeregistration.SystemID == _LastDeregistration.SystemID && CurrentDeregistration.WACNID == _LastDeregistration.WACNID);
                                }
                                catch { }
                                if (DuplicateFound)
                                {
                                    throw new P25DuplicateDetected("Duplate De-Registration Acknowledge");
                                }
                            }

                            _LastDeregistration = MethodReturn.dataDeregistrationAcknowledge;
                            _LastDeregistrationTimestamp = MethodReturn.TimeStamp;
                        }
                        break;
                    case "111101":
                        if (message.ManufacturerID == 0x00)
                        {
                            P25IdentifierUpdate idUpdate = HandleIdentifierUpdate(message);
                            MethodReturn.MessageType = P25ReturnType.IdentifierUpdate;
                            MethodReturn.dataIdentifierUpdate = idUpdate;

                            if (ChannelIdentifiers.Keys.Contains(MethodReturn.dataIdentifierUpdate.IdentifierID))
                            {
                                ChannelIdentifiers[MethodReturn.dataIdentifierUpdate.IdentifierID] = new P25FrequencyIdentifier(idUpdate.BaseFrequency, idUpdate.ChannelSpacing, idUpdate.TransmitOffset);
                            }
                            else
                            {
                                ChannelIdentifiers.Add(MethodReturn.dataIdentifierUpdate.IdentifierID, (new P25FrequencyIdentifier(idUpdate.BaseFrequency, idUpdate.ChannelSpacing, idUpdate.TransmitOffset)));
                            }
                        }
                        break;
                    default:
                        break;
                }
            }

            return MethodReturn;
        }

        /// <summary>
        /// Calculates the final frequency using the stored frequency identifiers
        /// </summary>
        /// <param name="IdentifierID">Frequency Identifier #</param>
        /// <param name="ChannelID">Channel ID</param>
        /// <returns>Returns the calculated frequency.</returns>
        /// <exception cref="CCDecodeP25.P25NoChannelIdentifiers">Thrown when no frequency identifiers are available or the suppied identifier ID is not available.</exception>
        public double GetFrequency(int IdentifierID, int ChannelID)
        {
            if (ChannelIdentifiers == null || ChannelIdentifiers.Count == 0 || !ChannelIdentifiers.Keys.Contains(IdentifierID))
            {
                throw new P25NoChannelIdentifiers();
            }

            return (double)(ChannelIdentifiers.First(f => f.Key == IdentifierID).Value.BaseFrequency + (ChannelID) * ChannelIdentifiers.First(f => f.Key == IdentifierID).Value.ChannelSpacing);
        }

        /// <summary>
        /// Calculates the final frequency using the stored frequency identifiers and returns it in a string format.
        /// </summary>
        /// <param name="IdentifierID">Frequency Identifier #</param>
        /// <param name="ChannelID">Channel ID</param>
        /// <returns>Returns the calculated frequency in a string format.</returns>
        /// <exception cref="CCDecodeP25.P25NoChannelIdentifiers">Thrown when no frequency identifiers are available or the suppied identifier ID is not available.</exception>
        public String GetFrequencyString(int IdentifierID, int ChannelID)
        {
            double dblFrequency = GetFrequency(IdentifierID, ChannelID);
            return Utilities.getFrequencyString(dblFrequency.ToString());
        }
    }
}