﻿// CCDecode: A library for decoding digital public safety standards.
// 
//  Author: Gabriel Graves
// Website: http://www.ccdecode.com/
// 
// This software has been released with no license and is public domain.
// You are free to do anything with the code for any purpose with out
// permission.

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace CCDecodeP25
{
    public static class Utilities
    {
        /// <summary>
        /// Determines if all characters in the string are valid hex characters.
        /// </summary>
        /// <param name="hex">Hex String</param>
        /// <returns>Returns true if all characters in the string are valid hex characters.</returns>
        public static bool isAllHex(String hex)
        {
            return isAllHex(hex.ToCharArray());
        }

        /// <summary>
        /// Determines if all characters in the character array are valid hex characters.
        /// </summary>
        /// <param name="chars">Hex Character Array</param>
        /// <returns>Returns true if all characters in the character array are valid hex characters.</returns>
        public static bool isAllHex(char[] chars)
        {
            foreach (char SingleChar in chars)
            {
                bool is_hex_char = (SingleChar >= '0' && SingleChar <= '9') || (SingleChar >= 'a' && SingleChar <= 'f') || (SingleChar >= 'A' && SingleChar <= 'F');

                if (!is_hex_char)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Determines if the P25 message string is a valid P25 message.
        /// </summary>
        /// <param name="Message">P25 Message</param>
        /// <returns>Returns true if the supplied message is a valid P25 message.</returns>
        public static bool isValidP25Message(String Message)
        {
            if (Message.Contains(':'))
            {
                String[] p25Split = Message.Trim().Replace(" ", "").Split(new char[] { ':' }, StringSplitOptions.RemoveEmptyEntries);

                foreach (String splitString in p25Split)
                {
                    if (splitString.Length == 24)
                    {
                        return isAllHex(splitString);
                    }
                }
            }
            else
            {
                if (Message.Trim().Length == 24)
                {
                    return isAllHex(Message.Trim().Replace(" ", ""));
                }
            }

            return false;
        }

        /// <summary>
        /// Returns the recognized P25 message.
        /// </summary>
        /// <param name="Message">P25 Message</param>
        /// <returns>Returns the recognized P25 message and strips all other text.</returns>
        /// <exception cref="CCDecodeP25.P25InvalidMessage">Thrown if the supplied string does not contain a valid P25 message.</exception>
        public static String getP25Message(String Message)
        {
            if (isValidP25Message(Message))
            {
                if (Message.Contains(':'))
                {
                    String[] p25Split = Message.Trim().Replace(" ", "").Split(new char[] { ':' }, StringSplitOptions.RemoveEmptyEntries);

                    foreach (String splitString in p25Split)
                    {
                        if (splitString.Length == 24)
                        {
                            if (isAllHex(splitString))
                            {
                                return splitString;
                            }
                        }
                    }
                }
                else
                {
                    if (Message.Length == 24)
                    {
                        if (isAllHex(Message.Trim().Replace(" ", "")))
                        {
                            return Message.Trim().Replace(" ", "");
                        }
                    }
                }
            }

            throw new P25InvalidMessage();
        }

        /// <summary>
        /// Converts the double frequency to a string in a valid format (EX: xxx.xxxxx)
        /// </summary>
        /// <param name="Frequency">Frequency</param>
        /// <returns>Returns the frequency as a string.</returns>
        public static String getFrequencyString(double Frequency)
        {
            return getFrequencyString(Frequency.ToString());
        }

        /// <summary>
        /// Converts the double frequency to a string in a valid format (EX: xxx.xxxxx)
        /// </summary>
        /// <param name="Frequency">Frequency</param>
        /// <returns>Returns the frequency as a string.</returns>
        public static String getFrequencyString(String Frequency)
        {
            if (Frequency.Contains("."))
            {
                String[] FreqSplit = Frequency.Split(new String[] { "." }, StringSplitOptions.None);
                return FreqSplit[0] + "." + FreqSplit[1].PadRight(5, '0');
            }
            else
            {
                return Frequency + ".00000";
            }
        }
    }
}