﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO.Ports;
using CCDecodeP25;
using System.Threading;

namespace CCDecodeP25Example
{
    public partial class frmRawView : Form
    {
        public P25Decode decoder;
        public delegate void AddText(String Text);
        public AddText delegateAddText;

        public frmRawView()
        {
            InitializeComponent();
            decoder = new P25Decode();
            delegateAddText = new AddText(doAddText);
        }

        private void frmRawView_Load(object sender, EventArgs e)
        {
            ReloadPorts();
        }

        public void ReloadPorts()
        {
            // Get list of used COM ports

            String[] COMPorts = SerialPort.GetPortNames();

            // Clear list and add the COM ports to the screen

            comboCOMPorts.Items.Clear();

            foreach (String comport in COMPorts)
            {
                comboCOMPorts.Items.Add(comport);
            }

            // Disable COM port selection if no ports are available
            comboCOMPorts.Enabled = (COMPorts.Count() > 0);

            // Select First COM port
            if (COMPorts.Count() > 0)
            {
                comboCOMPorts.SelectedIndex = 0;
            }
        }

        private void btnStart_Click(object sender, EventArgs e)
        {
            try
            {
                int iBaudRate;

                if (!int.TryParse(txtBaudRate.Text, out iBaudRate))
                {
                    MessageBox.Show("Error: Invalid Baud Rate", "CCDecode Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    txtBaudRate.Focus();
                    return;
                }

                if (!serialPortScanner.IsOpen || serialPortScanner.PortName != (comboCOMPorts.SelectedItem as String))
                {
                    serialPortScanner.PortName = (comboCOMPorts.SelectedItem as String);
                    serialPortScanner.BaudRate = iBaudRate;
                    serialPortScanner.Open();
                }

                comboCOMPorts.Enabled = false;
                txtBaudRate.Enabled = false;
                btnStart.Enabled = false;
                btnStop.Enabled = true;
                btnReloadPorts.Enabled = false;
            }
            catch (Exception ex)
            {
                MessageBox.Show("Error: " + ex.Message, "CCDecode Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void btnStop_Click(object sender, EventArgs e)
        {
            try
            {
                if (serialPortScanner.IsOpen)
                {
                    serialPortScanner.Close();
                }

                comboCOMPorts.Enabled = true;
                txtBaudRate.Enabled = true;
                btnStart.Enabled = true;
                btnStop.Enabled = false;
                btnReloadPorts.Enabled = true;
            }
            catch (Exception ex)
            {
                MessageBox.Show("Error: " + ex.Message, "CCDecode Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void btnReloadPorts_Click(object sender, EventArgs e)
        {
            if (!serialPortScanner.IsOpen)
            {
                ReloadPorts();
            }
        }

        private void serialPortScanner_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            try
            {
                String message = serialPortScanner.ReadLine();

                if (!String.IsNullOrEmpty(message) && !String.IsNullOrWhiteSpace(message))
                {
                    if (message.StartsWith("P25"))
                    {
                        P25Return decodedmessage = null;

                        try
                        {
                            decodedmessage = decoder.DecodeMessage(message, true);
                        }
                        catch (P25InvalidMessage) { }
                        catch (P25DuplicateDetected) { }
                        catch (P25ProtectedMessage)
                        {
                            AddTextToListBox("** Unable to Decode Protected Message **");
                        }

                        if (decodedmessage != null)
                        {
                            HandleMessage(decodedmessage);
                        }
                    }
                }
            }
            catch{ }
        }

        private void HandleMessage(P25Return message)
        {
            switch (message.MessageType)
            {
                case P25ReturnType.GroupAffiliationResponse:
                    if (chkboxShowGroupAffiliationResponse.Checked)
                    {
                        P25GroupAffiliationResponse dataGAR = message.dataGroupAffiliationResponse;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: Group Affiliation Response: Local/Global: " + dataGAR.LocalOrGlobal.ToString() + " - Group Affiliation Value: " + dataGAR.GroupAffiliationValue.ToString() + " - Announcement Talkgroup ID: " + dataGAR.AnnouncementTalkgroupID.ToString() + " - Talkgroup ID: " + dataGAR.TalkgroupID.ToString() + " - Radio ID: " + dataGAR.RadioID.ToString());
                    }
                    break;
                case P25ReturnType.GroupVoiceChannelGrant:
                    if (chkboxShowGroupVoiceChannelGrants.Checked)
                    {
                        P25GroupVoiceChannelGrantSingleBlock dataGVCG = message.dataGroupVoiceChannelGrant;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: Group Voice Channel Grant: Frequency: " + decoder.GetFrequencyString(dataGVCG.Identifier, dataGVCG.Channel) + " - Talkgroup ID: " + dataGVCG.TalkgroupID.ToString() + " - Radio ID: " + dataGVCG.RadioID.ToString());
                    }
                    break;
                case P25ReturnType.GroupVoiceChannelGrantUpdate:
                    if (chkboxShowGroupVoiceChannelGrantUpdates.Checked)
                    {
                        P25GroupVoiceChannelGrantUpdate dataGVCGU = message.dataGroupVoiceChannelGrantUpdate;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: Group Voice Channel Grant Update: (#1) Frequency: " + decoder.GetFrequencyString(dataGVCGU.Identifier1, dataGVCGU.Channel1) + " - Talkgroup ID: " + dataGVCGU.TalkgroupID1.ToString() + " - (#2) Frequency: " + decoder.GetFrequencyString(dataGVCGU.Identifier2, dataGVCGU.Channel2) + " - Talkgroup ID: " + dataGVCGU.TalkgroupID2.ToString());
                    }
                    break;
                case P25ReturnType.IdentifierUpdate:
                    if (chkboxShowIdentifierUpdates.Checked)
                    {
                        P25IdentifierUpdate dataIdenUP = message.dataIdentifierUpdate;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: Identifier Update: Identifier ID: #" + dataIdenUP.IdentifierID.ToString() + " - Base Frequency: " + Utilities.getFrequencyString(dataIdenUP.BaseFrequency) + " - Bandwidth: " + dataIdenUP.Bandwidth.ToString() + " - Channel Spacing: " + dataIdenUP.ChannelSpacing.ToString() + " - Transmit Offset: " + dataIdenUP.TransmitOffset.ToString());
                    }
                    break;
                case P25ReturnType.DeregistrationAcknowledge:
                    if (chkboxShowDeregistrationAcknowledge.Checked)
                    {
                        P25DeregistrationAcknowledge dataDA = message.dataDeregistrationAcknowledge;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: De-Registration Acknowledge: WACN ID: " + dataDA.WACNID + " - System ID: " + dataDA.SystemID.ToString() + " - Radio ID: " + dataDA.RadioID.ToString());
                    }
                    break;
                case P25ReturnType.UnitRegistrationResponse:
                    if (chkboxShowUnitRegistrationResponse.Checked)
                    {
                        P25UnitRegistrationResponse dataURR = message.dataUnitRegistrationResponse;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: Unit Registration Response: Registration Value: " + dataURR.RegistrationValue.ToString() + " - System ID: " + dataURR.SystemID.ToString() + " - Radio ID: " + dataURR.RadioID.ToString() + " - Source Address: " + dataURR.SourceAddress.ToString());
                    }
                    break;
                case P25ReturnType.MotorolaPatch:
                    if (chkboxShowMotorolaPatches.Checked)
                    {
                        P25MotorolaPatch dataMotoPatch = message.dataMotorolaPatch;
                        AddTextToListBox(message.TimeStamp.ToString() + " [" + message.OriginalMessage.ToString(true) + "]: Motorola Patch: Source Talkgroup ID: " + dataMotoPatch.SupergroupID.ToString() + " - Talkgroups: " + dataMotoPatch.TalkgroupID1.ToString() + ", " + dataMotoPatch.TalkgroupID2.ToString() + ", " + dataMotoPatch.TalkgroupID3.ToString());
                    }
                    break;
            }
        }

        private void AddTextToListBox(String Text)
        {
            listBoxRawData.Invoke(delegateAddText, new Object[] { Text });
        }
        
        private void doAddText(String Text)
        {
            if (chkboxAddNewToTop.Checked)
            {
                listBoxRawData.Items.Insert(0, Text);
            }
            else
            {
                listBoxRawData.Items.Add(Text);

                if (chkboxAutoScroll.Checked)
                {
                    listBoxRawData.TopIndex = (listBoxRawData.Items.Count - 1);
                }
            }
        }

        private bool chkboxAddNewToTopSkipWarning = false;
        private void chkboxAddNewToTop_CheckedChanged(object sender, EventArgs e)
        {
            if (!chkboxAddNewToTopSkipWarning)
            {
                if (MessageBox.Show("Changing this option will clear the contents. Are you sure?", "CCDecode: Clear Contents", MessageBoxButtons.YesNo, MessageBoxIcon.Question) != System.Windows.Forms.DialogResult.Yes)
                {
                    chkboxAddNewToTopSkipWarning = true;
                    chkboxAddNewToTop.Checked = !chkboxAddNewToTop.Checked;
                }
                else
                {
                    listBoxRawData.Items.Clear();
                    chkboxAutoScroll.Enabled = !chkboxAutoScroll.Enabled;
                }
            }
            else
            {
                chkboxAddNewToTopSkipWarning = false;
            }
        }

        private void btnClear_Click(object sender, EventArgs e)
        {
            listBoxRawData.Items.Clear();
        }

        private void frmRawView_FormClosing(object sender, FormClosingEventArgs e)
        {
            // We must close the serial port in a new thread when closing a form to prevent the form from hanging upon closing it.

            if (serialPortScanner.IsOpen)
            {
                e.Cancel = true;
                Thread tClosePort = new Thread(new ThreadStart(CloseSerialAndExit));
                tClosePort.Start();
            }
        }

        private void CloseSerialAndExit()
        {
            try
            {
                serialPortScanner.Close();
                this.Invoke(new EventHandler(CloseForm));
            }
            catch { }
        }

        private void CloseForm(object sender, EventArgs e)
        {
            this.Close();
        }
    }
}