/*
 * Decompiled with CFR 0.152.
 */
package org.jackhuang.hmcl.util.platform;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jackhuang.hmcl.task.Schedulers;
import org.jackhuang.hmcl.util.Lang;
import org.jackhuang.hmcl.util.Logging;
import org.jackhuang.hmcl.util.StringUtils;
import org.jackhuang.hmcl.util.io.FileUtils;
import org.jackhuang.hmcl.util.platform.Architecture;
import org.jackhuang.hmcl.util.platform.Bits;
import org.jackhuang.hmcl.util.platform.OperatingSystem;
import org.jackhuang.hmcl.util.platform.Platform;
import org.jackhuang.hmcl.util.versioning.VersionNumber;

public final class JavaVersion {
    private final Path binary;
    private final String longVersion;
    private final Platform platform;
    private final int version;
    private final VersionNumber versionNumber;
    private static final Pattern REGEX = Pattern.compile("version \"(?<version>(.*?))\"");
    private static final Pattern VERSION = Pattern.compile("^(?<version>[0-9]+)");
    private static final Pattern OS_ARCH = Pattern.compile("os\\.arch = (?<arch>.*)");
    private static final Pattern JAVA_VERSION = Pattern.compile("java\\.version = (?<version>.*)");
    public static final int UNKNOWN = -1;
    public static final int JAVA_6 = 6;
    public static final int JAVA_7 = 7;
    public static final int JAVA_8 = 8;
    public static final int JAVA_9 = 9;
    public static final int JAVA_16 = 16;
    public static final int JAVA_17 = 17;
    private static final Map<Path, JavaVersion> fromExecutableCache = new ConcurrentHashMap<Path, JavaVersion>();
    public static final JavaVersion CURRENT_JAVA;
    private static Collection<JavaVersion> JAVAS;
    private static final CountDownLatch LATCH;

    public JavaVersion(Path binary, String longVersion, Platform platform) {
        this.binary = binary;
        this.longVersion = longVersion;
        this.platform = platform;
        if (longVersion != null) {
            this.version = JavaVersion.parseVersion(longVersion);
            this.versionNumber = VersionNumber.asVersion(longVersion);
        } else {
            this.version = -1;
            this.versionNumber = null;
        }
    }

    public Path getBinary() {
        return this.binary;
    }

    public String getVersion() {
        return this.longVersion;
    }

    public Platform getPlatform() {
        return this.platform;
    }

    public Architecture getArchitecture() {
        return this.platform.getArchitecture();
    }

    public Bits getBits() {
        return this.platform.getBits();
    }

    public VersionNumber getVersionNumber() {
        return this.versionNumber;
    }

    public int getParsedVersion() {
        return this.version;
    }

    private static int parseVersion(String version) {
        int head;
        Matcher matcher = VERSION.matcher(version);
        if (matcher.find() && (head = Lang.parseInt(matcher.group(), -1)) > 1) {
            return head;
        }
        if (version.contains("1.8")) {
            return 8;
        }
        if (version.contains("1.7")) {
            return 7;
        }
        if (version.contains("1.6")) {
            return 6;
        }
        return -1;
    }

    public static JavaVersion fromExecutable(Path executable) throws IOException {
        JavaVersion javaVersion;
        JavaVersion cachedJavaVersion = fromExecutableCache.get(executable = executable.toRealPath(new LinkOption[0]));
        if (cachedJavaVersion != null) {
            return cachedJavaVersion;
        }
        String osArch = null;
        String version = null;
        Platform platform = null;
        Process process = new ProcessBuilder(executable.toString(), "-XshowSettings:properties", "-version").start();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream(), OperatingSystem.NATIVE_CHARSET));){
            String line;
            while ((line = reader.readLine()) != null) {
                Matcher m = OS_ARCH.matcher(line);
                if (m.find()) {
                    osArch = m.group("arch");
                    if (version == null) continue;
                } else {
                    m = JAVA_VERSION.matcher(line);
                    if (!m.find()) continue;
                    version = m.group("version");
                    if (osArch == null) continue;
                }
                break;
            }
        }
        if (osArch != null) {
            platform = Platform.getPlatform(OperatingSystem.CURRENT_OS, Architecture.parseArchName(osArch));
        }
        if (version == null) {
            boolean is64Bit = false;
            process = new ProcessBuilder(executable.toString(), "-version").start();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream(), OperatingSystem.NATIVE_CHARSET));){
                String line;
                while ((line = reader.readLine()) != null) {
                    Matcher m = REGEX.matcher(line);
                    if (m.find()) {
                        version = m.group("version");
                    }
                    if (!line.contains("64-Bit")) continue;
                    is64Bit = true;
                }
            }
            if (platform == null) {
                platform = Platform.getPlatform(OperatingSystem.CURRENT_OS, is64Bit ? Architecture.X86_64 : Architecture.X86);
            }
        }
        if ((javaVersion = new JavaVersion(executable, version, platform)).getParsedVersion() == -1) {
            throw new IOException("Unrecognized Java version " + version);
        }
        fromExecutableCache.put(executable, javaVersion);
        return javaVersion;
    }

    public static Path getExecutable(Path javaHome) {
        if (OperatingSystem.CURRENT_OS == OperatingSystem.WINDOWS) {
            return javaHome.resolve("bin").resolve("java.exe");
        }
        return javaHome.resolve("bin").resolve("java");
    }

    public static JavaVersion fromCurrentEnvironment() {
        return CURRENT_JAVA;
    }

    public static Collection<JavaVersion> getJavas() throws InterruptedException {
        if (JAVAS != null) {
            return JAVAS;
        }
        LATCH.await();
        return JAVAS;
    }

    public static synchronized void initialize() {
        List<JavaVersion> javaVersions;
        if (JAVAS != null) {
            throw new IllegalStateException("JavaVersions have already been initialized.");
        }
        try (Stream<Path> stream = JavaVersion.searchPotentialJavaExecutables();){
            javaVersions = JavaVersion.lookupJavas(stream);
        }
        catch (IOException e) {
            Logging.LOG.log(Level.WARNING, "Failed to search Java homes", e);
            javaVersions = new ArrayList<JavaVersion>();
        }
        if (!javaVersions.contains(CURRENT_JAVA)) {
            javaVersions.add(CURRENT_JAVA);
        }
        JAVAS = Collections.newSetFromMap(new ConcurrentHashMap());
        JAVAS.addAll(javaVersions);
        Logging.LOG.log(Level.FINE, "Finished Java installation lookup, found " + JAVAS.size());
        LATCH.countDown();
    }

    private static List<JavaVersion> lookupJavas(Stream<Path> javaExecutables) {
        return javaExecutables.filter(Files::isExecutable).flatMap(executable -> {
            try {
                return Stream.of(executable.toRealPath(new LinkOption[0]));
            }
            catch (IOException e) {
                Logging.LOG.log(Level.WARNING, "Failed to lookup Java executable at " + executable, e);
                return Stream.empty();
            }
        }).distinct().flatMap(executable -> {
            if (executable.equals(CURRENT_JAVA.getBinary())) {
                return Stream.of(CURRENT_JAVA);
            }
            try {
                Logging.LOG.log(Level.FINER, "Looking for Java:" + executable);
                Future<JavaVersion> future = Schedulers.io().submit(() -> JavaVersion.fromExecutable(executable));
                JavaVersion javaVersion = future.get(5L, TimeUnit.SECONDS);
                Logging.LOG.log(Level.FINE, "Found Java (" + javaVersion.getVersion() + ") " + javaVersion.getBinary().toString());
                return Stream.of(javaVersion);
            }
            catch (InterruptedException | ExecutionException | TimeoutException e) {
                Logging.LOG.log(Level.WARNING, "Failed to determine Java at " + executable, e);
                return Stream.empty();
            }
        }).collect(Collectors.toList());
    }

    private static Stream<Path> searchPotentialJavaExecutables() throws IOException {
        ArrayList<Stream<Path>> javaExecutables = new ArrayList<Stream<Path>>();
        switch (OperatingSystem.CURRENT_OS) {
            case WINDOWS: {
                javaExecutables.add(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\Java Runtime Environment\\").stream().map(JavaVersion::getExecutable));
                javaExecutables.add(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\Java Development Kit\\").stream().map(JavaVersion::getExecutable));
                javaExecutables.add(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\JRE\\").stream().map(JavaVersion::getExecutable));
                javaExecutables.add(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\JDK\\").stream().map(JavaVersion::getExecutable));
                for (Optional programFiles : Arrays.asList(FileUtils.tryGetPath(Optional.ofNullable(System.getenv("ProgramFiles")).orElse("C:\\Program Files"), new String[0]), FileUtils.tryGetPath(Optional.ofNullable(System.getenv("ProgramFiles(x86)")).orElse("C:\\Program Files (x86)"), new String[0]), FileUtils.tryGetPath(Optional.ofNullable(System.getenv("ProgramFiles(ARM)")).orElse("C:\\Program Files (ARM)"), new String[0]))) {
                    if (!programFiles.isPresent()) continue;
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("Java")).map(JavaVersion::getExecutable));
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("BellSoft")).map(JavaVersion::getExecutable));
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("AdoptOpenJDK")).map(JavaVersion::getExecutable));
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("Zulu")).map(JavaVersion::getExecutable));
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("Microsoft")).map(JavaVersion::getExecutable));
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("Eclipse Foundation")).map(JavaVersion::getExecutable));
                    javaExecutables.add(JavaVersion.listDirectory(((Path)programFiles.get()).resolve("Semeru")).map(JavaVersion::getExecutable));
                }
                Optional<Path> programFilesX86 = FileUtils.tryGetPath(Optional.ofNullable(System.getenv("ProgramFiles(x86)")).orElse("C:\\Program Files (x86)"), new String[0]);
                if (programFilesX86.isPresent()) {
                    Path runtimeDir = programFilesX86.get().resolve("Minecraft Launcher").resolve("runtime");
                    javaExecutables.add(Stream.of(runtimeDir.resolve("jre-legacy").resolve("windows-x64").resolve("jre-legacy"), runtimeDir.resolve("jre-legacy").resolve("windows-x86").resolve("jre-legacy"), runtimeDir.resolve("java-runtime-alpha").resolve("windows-x64").resolve("java-runtime-alpha"), runtimeDir.resolve("java-runtime-alpha").resolve("windows-x86").resolve("java-runtime-alpha")).map(JavaVersion::getExecutable));
                }
                if (System.getenv("PATH") != null) {
                    javaExecutables.add(Arrays.stream(System.getenv("PATH").split(";")).flatMap(path -> Lang.toStream(FileUtils.tryGetPath(path, "java.exe"))));
                }
                if (System.getenv("HMCL_JRES") == null) break;
                javaExecutables.add(Arrays.stream(System.getenv("HMCL_JRES").split(";")).flatMap(path -> Lang.toStream(FileUtils.tryGetPath(path, "bin", "java.exe"))));
                break;
            }
            case LINUX: {
                Optional<Path> home;
                javaExecutables.add(JavaVersion.listDirectory(Paths.get("/usr/java", new String[0])).map(JavaVersion::getExecutable));
                javaExecutables.add(JavaVersion.listDirectory(Paths.get("/usr/lib/jvm", new String[0])).map(JavaVersion::getExecutable));
                javaExecutables.add(JavaVersion.listDirectory(Paths.get("/usr/lib32/jvm", new String[0])).map(JavaVersion::getExecutable));
                if (System.getenv("PATH") != null) {
                    javaExecutables.add(Arrays.stream(System.getenv("PATH").split(":")).flatMap(path -> Lang.toStream(FileUtils.tryGetPath(path, "java"))));
                }
                if (System.getenv("HMCL_JRES") != null) {
                    javaExecutables.add(Arrays.stream(System.getenv("HMCL_JRES").split(":")).flatMap(path -> Lang.toStream(FileUtils.tryGetPath(path, "bin", "java"))));
                }
                if (!(home = FileUtils.tryGetPath(System.getProperty("user.home", ""), new String[0])).isPresent()) break;
                Path runtimeDir = home.get().resolve(".minecraft").resolve("runtime");
                javaExecutables.add(Stream.of(runtimeDir.resolve("jre-legacy").resolve("linux").resolve("jre-legacy"), runtimeDir.resolve("java-runtime-alpha").resolve("linux").resolve("java-runtime-alpha")).map(JavaVersion::getExecutable));
                break;
            }
            case OSX: {
                javaExecutables.add(JavaVersion.listDirectory(Paths.get("/Library/Java/JavaVirtualMachines", new String[0])).flatMap(dir -> Stream.of(dir.resolve("Contents/Home"), dir.resolve("Contents/Home/jre"))).map(JavaVersion::getExecutable));
                javaExecutables.add(JavaVersion.listDirectory(Paths.get("/System/Library/Java/JavaVirtualMachines", new String[0])).map(dir -> dir.resolve("Contents/Home")).map(JavaVersion::getExecutable));
                javaExecutables.add(Stream.of(Paths.get("/Library/Internet Plug-Ins/JavaAppletPlugin.plugin/Contents/Home/bin/java", new String[0])));
                javaExecutables.add(Stream.of(Paths.get("/Applications/Xcode.app/Contents/Applications/Application Loader.app/Contents/MacOS/itms/java/bin/java", new String[0])));
                javaExecutables.add(Stream.of(Paths.get("/Library/Application Support/minecraft/runtime/jre-x64/jre.bundle/Contents/Home/bin/java", new String[0])));
                if (System.getenv("PATH") != null) {
                    javaExecutables.add(Arrays.stream(System.getenv("PATH").split(":")).flatMap(path -> Lang.toStream(FileUtils.tryGetPath(path, "java"))));
                }
                if (System.getenv("HMCL_JRES") == null) break;
                javaExecutables.add(Arrays.stream(System.getenv("HMCL_JRES").split(":")).flatMap(path -> Lang.toStream(FileUtils.tryGetPath(path, "bin", "java"))));
                break;
            }
        }
        return javaExecutables.parallelStream().flatMap(stream -> stream);
    }

    private static Stream<Path> listDirectory(Path directory) throws IOException {
        if (Files.isDirectory(directory, new LinkOption[0])) {
            try (DirectoryStream<Path> subDirs = Files.newDirectoryStream(directory);){
                ArrayList<Path> paths = new ArrayList<Path>();
                for (Path subDir : subDirs) {
                    paths.add(subDir);
                }
                Stream stream = paths.stream();
                return stream;
            }
        }
        return Stream.empty();
    }

    private static List<Path> queryJavaHomesInRegistryKey(String location) throws IOException {
        ArrayList<Path> homes = new ArrayList<Path>();
        for (String java : JavaVersion.querySubFolders(location)) {
            String home;
            if (!JavaVersion.querySubFolders(java).contains(java + "\\MSI") || (home = JavaVersion.queryRegisterValue(java, "JavaHome")) == null) continue;
            try {
                homes.add(Paths.get(home, new String[0]));
            }
            catch (InvalidPathException e) {
                Logging.LOG.log(Level.WARNING, "Invalid Java path in system registry: " + home);
            }
        }
        return homes;
    }

    private static List<String> querySubFolders(String location) throws IOException {
        ArrayList<String> res = new ArrayList<String>();
        Process process = Runtime.getRuntime().exec(new String[]{"cmd", "/c", "reg", "query", location});
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream(), OperatingSystem.NATIVE_CHARSET));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (!line.startsWith(location) || line.equals(location)) continue;
                res.add(line);
            }
        }
        return res;
    }

    private static String queryRegisterValue(String location, String name) throws IOException {
        boolean last = false;
        Process process = Runtime.getRuntime().exec(new String[]{"cmd", "/c", "reg", "query", location, "/v", name});
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream(), OperatingSystem.NATIVE_CHARSET));){
            String line;
            while ((line = reader.readLine()) != null) {
                String s2;
                int begins;
                if (!StringUtils.isNotBlank(line)) continue;
                if (last && line.trim().startsWith(name) && (begins = line.indexOf(name)) > 0 && (begins = (s2 = line.substring(begins + name.length())).indexOf("REG_SZ")) > 0) {
                    String string = s2.substring(begins + "REG_SZ".length()).trim();
                    return string;
                }
                if (!location.equals(line.trim())) continue;
                last = true;
            }
        }
        return null;
    }

    static {
        Path currentExecutable = JavaVersion.getExecutable(Paths.get(System.getProperty("java.home"), new String[0]));
        try {
            currentExecutable = currentExecutable.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            Logging.LOG.log(Level.WARNING, "Failed to resolve current Java path: " + currentExecutable, e);
        }
        CURRENT_JAVA = new JavaVersion(currentExecutable, System.getProperty("java.version"), Platform.CURRENT_PLATFORM);
        LATCH = new CountDownLatch(1);
    }
}

