/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.load.java.structure.impl

import com.intellij.openapi.diagnostic.Logger
import com.intellij.psi.PsiClass
import com.intellij.psi.PsiTypeParameter
import org.jetbrains.kotlin.asJava.KtLightClassMarker
import org.jetbrains.kotlin.descriptors.Visibility
import org.jetbrains.kotlin.load.java.structure.*
import org.jetbrains.kotlin.name.FqName
import org.jetbrains.kotlin.name.Name
import org.jetbrains.kotlin.name.SpecialNames

class JavaClassImpl(psiClass: PsiClass) : JavaClassifierImpl<PsiClass>(psiClass), JavaClass, JavaAnnotationOwnerImpl, JavaModifierListOwnerImpl {
    init {
        assert(psiClass !is PsiTypeParameter) { "PsiTypeParameter should be wrapped in JavaTypeParameter, not JavaClass: use JavaClassifier.create()" }
    }

    override val innerClasses: Collection<JavaClass>
        get() = classes(psi.innerClasses)

    override val fqName: FqName?
        get() {
            val qualifiedName = psi.qualifiedName
            return if (qualifiedName == null) null else FqName(qualifiedName)
        }

    override val name: Name
        get() = SpecialNames.safeIdentifier(psi.name)

    override val isInterface: Boolean
        get() = psi.isInterface

    override val isAnnotationType: Boolean
        get() = psi.isAnnotationType

    override val isEnum: Boolean
        get() = psi.isEnum

    override val outerClass: JavaClassImpl?
        get() {
            val outer = psi.containingClass
            return if (outer == null) null else JavaClassImpl(outer)
        }

    override val typeParameters: List<JavaTypeParameter>
        get() = typeParameters(psi.typeParameters)

    override val supertypes: Collection<JavaClassifierType>
        get() = classifierTypes(psi.superTypes)

    override val methods: Collection<JavaMethod>
        get() {
            assertNotLightClass()
            // We apply distinct here because PsiClass#getMethods() can return duplicate PSI methods, for example in Lombok (see KT-11778)
            // Return type seems to be null for example for the 'clone' Groovy method generated by @AutoClone (see EA-73795)
            return methods(psi.methods.filter { method -> !method.isConstructor && method.returnType != null }).distinct()
        }

    override val fields: Collection<JavaField>
        get() {
            assertNotLightClass()
            return fields(psi.fields.filter { field ->
                val name = field.name
                // ex. Android plugin generates LightFields for resources started from '.' (.DS_Store file etc)
                name != null && Name.isValidIdentifier(name)
            })
        }

    override val constructors: Collection<JavaConstructor>
        get() {
            assertNotLightClass()
            // See for example org.jetbrains.plugins.scala.lang.psi.light.ScFunctionWrapper,
            // which is present in getConstructors(), but its isConstructor() returns false
            return constructors(psi.constructors.filter { method -> method.isConstructor })
        }

    override val isAbstract: Boolean
        get() = JavaElementUtil.isAbstract(this)

    override val isStatic: Boolean
        get() = JavaElementUtil.isStatic(this)

    override val isFinal: Boolean
        get() = JavaElementUtil.isFinal(this)

    override val visibility: Visibility
        get() = JavaElementUtil.getVisibility(this)

    override val lightClassOriginKind: LightClassOriginKind?
        get() = (psi as? KtLightClassMarker)?.originKind

    override fun getAnnotationOwnerPsi() = psi.modifierList

    private fun assertNotLightClass() {
        val psiClass = psi
        if (psiClass !is KtLightClassMarker) return

        val message = "Querying members of JavaClass created for $psiClass of type ${psiClass.javaClass} defined in file ${psiClass.containingFile?.virtualFile?.canonicalPath}"
        LOGGER.error(message)
    }

    companion object {
        private val LOGGER = Logger.getInstance(JavaClassImpl::class.java)
    }
}
