/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.idea.filters

import com.intellij.execution.filters.ExceptionFilter
import com.intellij.execution.filters.Filter
import com.intellij.execution.filters.HyperlinkInfo
import com.intellij.execution.filters.OpenFileHyperlinkInfo
import com.intellij.openapi.project.Project
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.psi.search.GlobalSearchScope
import org.jetbrains.kotlin.idea.debugger.*
import org.jetbrains.kotlin.resolve.jvm.JvmClassName
import java.util.regex.Pattern

class KotlinExceptionFilter(private val searchScope: GlobalSearchScope) : Filter {
    private val exceptionFilter = ExceptionFilter(searchScope)

    override fun applyFilter(line: String, entireLength: Int): Filter.Result? {
        val result = exceptionFilter.applyFilter(line, entireLength)
        return if (result == null) null else patchResult(result, line)
    }

    private fun patchResult(result: Filter.Result, line: String): Filter.Result {
        val newHyperlinkInfo = createHyperlinkInfo(line) ?: return result

        return Filter.Result(result.resultItems.map {
            Filter.ResultItem(it.getHighlightStartOffset(), it.getHighlightEndOffset(), newHyperlinkInfo, it.getHighlightAttributes())
        })
    }

    private fun createHyperlinkInfo(line: String): HyperlinkInfo? {
        val project = searchScope.project ?: return null

        val element = parseStackTraceLine(line) ?: return null

        // All true classes should be handled correctly in the default ExceptionFilter. Special cases:
        // - static facades;
        // - package facades / package parts (generated by pre-M13 compiled);
        // - local classes (and closures) in top-level function and property declarations.
        val fileName = element.fileName
        // fullyQualifiedName is of format "package.Class$Inner"
        val fullyQualifiedName = element.className
        val lineNumber = element.lineNumber - 1

        val internalName = fullyQualifiedName.replace('.', '/')
        val jvmClassName = JvmClassName.byInternalName(internalName)

        val file = DebuggerUtils.findSourceFileForClassIncludeLibrarySources(project, searchScope, jvmClassName, fileName) ?: return null

        val virtualFile = file.virtualFile ?: return null

        val hyperlinkInfoForInline = createHyperlinks(jvmClassName, virtualFile, lineNumber + 1, project)
        if (hyperlinkInfoForInline != null) {
            return hyperlinkInfoForInline
        }

        return OpenFileHyperlinkInfo(project, virtualFile, lineNumber)
    }

    private fun createHyperlinks(jvmName: JvmClassName, file: VirtualFile, line: Int, project: Project): InlineFunctionHyperLinkInfo? {
        if (!isInlineFunctionLineNumber(file, line, project)) return null

        val debugInfo = readBytecodeInfo(project, jvmName, file) ?: return null
        val smapData = debugInfo.smapData ?: return null

        val inlineInfos = arrayListOf<InlineFunctionHyperLinkInfo.InlineInfo>()

        val (inlineFunctionBodyFile, inlineFunctionBodyLine) =
                mapStacktraceLineToSource(smapData, line, project, SourceLineKind.EXECUTED_LINE, searchScope) ?: return null

        inlineInfos.add(InlineFunctionHyperLinkInfo.InlineInfo.InlineFunctionBodyInfo(
                inlineFunctionBodyFile.virtualFile,
                inlineFunctionBodyLine))

        val inlineFunCallInfo = mapStacktraceLineToSource(smapData, line, project, SourceLineKind.CALL_LINE, searchScope)
        if (inlineFunCallInfo != null) {
            val (callSiteFile, callSiteLine) = inlineFunCallInfo
            inlineInfos.add(InlineFunctionHyperLinkInfo.InlineInfo.CallSiteInfo(callSiteFile.virtualFile, callSiteLine))
        }

        return InlineFunctionHyperLinkInfo(project, inlineInfos)
    }

    companion object {
        // Matches strings like "\tat test.TestPackage$foo$f$1.invoke(a.kt:3)\n"
        //                   or "\tBreakpoint reached at test.TestPackage$foo$f$1.invoke(a.kt:3)\n"
        private val STACK_TRACE_ELEMENT_PATTERN = Pattern.compile("^[\\w|\\s]*at\\s+(.+)\\.(.+)\\((.+):(\\d+)\\)\\s*$")

        private fun parseStackTraceLine(line: String): StackTraceElement? {
            val matcher = STACK_TRACE_ELEMENT_PATTERN.matcher(line)
            if (matcher.matches()) {
                val declaringClass = matcher.group(1)
                val methodName = matcher.group(2)
                val fileName = matcher.group(3)
                val lineNumber = matcher.group(4)
                //noinspection ConstantConditions
                return StackTraceElement(declaringClass, methodName, fileName, Integer.parseInt(lineNumber))
            }
            return null
        }
    }
}
