/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.js.test.semantics;

import com.google.common.collect.Lists;
import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.kotlin.cli.common.ExitCode;
import org.jetbrains.kotlin.cli.js.K2JSCompiler;
import org.jetbrains.kotlin.js.config.EcmaVersion;
import org.jetbrains.kotlin.js.test.SingleFileTranslationTest;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Lets test compiling all the JS code thats used by the library-js-library from the maven build
 *
 * This test case isn't run by default but can only be ran after the maven build
 * has completed so that there are various kotlin files to be compiled
 */
public class CompileMavenGeneratedJSLibrary extends SingleFileTranslationTest {
    protected final String generatedJsDir = "libraries/tools/kotlin-js-library/target/";
    protected String generatedJsDefinitionsDir = generatedJsDir + "generated-js-definitions";
    protected File generatedJsLibraryDir = new File( generatedJsDir + "generated-js-library");

    public CompileMavenGeneratedJSLibrary() {
        super("kotlin-js-library/");
    }

    public static void main(String[] args) throws Exception {
        CompileMavenGeneratedJSLibrary test = new CompileMavenGeneratedJSLibrary();
        test.DISABLED_testGenerateTestCase();
    }

    public void testDummy() {
    }

    public void DISABLED_testGenerateTestCase() throws Exception {
        if (generatedJsLibraryDir.exists() && generatedJsLibraryDir.isDirectory()) {
            generateJavaScriptFiles(DEFAULT_ECMA_VERSIONS,
                                    "libraries/stdlib/test",
                                    "collections/ArraysTest.kt",
                                    "dom/DomTest.kt",
                                    "js/MapTest.kt",
                                    "js/JsDomTest.kt",
                                    "collections/FunctionIteratorTest.kt",
                                    "collections/IteratorsTest.kt",
                                    "GetOrElseTest.kt",
                                    "collections/ListTest.kt",
                                    "collections/SetTest.kt",
                                    "text/StringTest.kt");
        }
        else {
            System.out.println("Warning " + generatedJsLibraryDir + " does not exist - I guess you've not run the maven build in library/ yet?");
        }
    }

    private void generateJavaScriptFiles(
            @NotNull Iterable<EcmaVersion> ecmaVersions,
            @NotNull String sourceDir,
            @NotNull String... stdLibFiles
    ) throws Exception {
        List<String> files = Lists.newArrayList();

        // now lets add all the files from the definitions and library
        //addAllSourceFiles(files, generatedJsDefinitionsDir);
        addAllSourceFiles(files, generatedJsLibraryDir);

        File stdlibDir = new File(sourceDir);
        assertTrue("Cannot find stdlib test source: " + stdlibDir, stdlibDir.exists());
        for (String file : stdLibFiles) {
            files.add(new File(stdlibDir, file).getPath());
        }

        // now lets try invoke the compiler
        for (EcmaVersion version : ecmaVersions) {
            String outputFile = getOutputFilePath(getTestName(false) + ".compiler.kt", version);
            System.out.println("Compiling with version: " + version + " to: " + outputFile);

            List<String> args = new ArrayList<String>(Arrays.asList(
                    "-output", outputFile,
                    "-library-files", generatedJsDefinitionsDir,
                    "-verbose"
            ));
            args.addAll(files);
            ExitCode answer = new K2JSCompiler().exec(System.out, ArrayUtil.toStringArray(args));

            assertEquals("Compile failed", ExitCode.OK, answer);
        }
    }

    private static void addAllSourceFiles(List<String> files, File dir) {
        File[] children = dir.listFiles();
        if (children != null && children.length > 0) {
            for (File child : children) {
                if (child.isDirectory()) {
                    addAllSourceFiles(files, child);
                }
                else {
                    String name = child.getName();
                    if (name.toLowerCase().endsWith(".kt")) {
                        files.add(child.getPath());
                    }
                }
            }
        }
    }
}
