/*------------------------------------------------------------------------------*
 * File Name: Wks_Utils.h														*
 * Creation: GJL 5/22/2002														*
 * Purpose: Header file for Wks_Utils.c											*
 * Copyright (c) OriginLab Corp.	2002-2007									*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *	LY	11/4/02 QA70-3256 v7.0428 BINARY_IMPORT									*
 *------------------------------------------------------------------------------*/

#ifndef _WKS_UTILS_H
#define _WKS_UTILS_H

//////////////////////////////////////////////////////////////////////////////////
// Function synonyms for backward compatibility...no longer using "wu" function
// name prefix 
//////////////////////////////////////////////////////////////////////////////////
//
#define wuGetNonContiguousWksSelection	GetNonContiguousWksSelection
#define wuGetContiguousWksSelection		GetContiguousWksSelection
#define wuParseWksSelection				ParseWksSelection
#define wuVectorHasUniformSpacing		VectorHasUniformSpacing
#define wuRenameToFileName				RenameToFileName

//////////////////////////////////////////////////////////////////////////////////
// Non-Localzed constants and literals used in Wks_Utils.c
//////////////////////////////////////////////////////////////////////////////////
//
#define WKS_UTILS_EXCEL_CONTIG_SEL			"[%s]%s!$%s$%d:$%s$%d"
#define WKS_UTILS_WKS_CONTIG_SEL			"%s_%s[%d]:%s[%d]"

#define WKS_UTILS_NO_ERROR					0
#define WKS_UTILS_NO_SEL_WARNING			1
#define WKS_UTILS_BAD_SEL_ERROR				100
#define WKS_UTILS_INVALID_TREENODE_ERROR	101
#define WKS_UTILS_NO_SINGLE_COL_ERROR		102

//////////////////////////////////////////////////////////////////////////////////
// Function Prototypes in Wks_Utils.c
//////////////////////////////////////////////////////////////////////////////////
//

/** 
		Gets the current selection from an Origin worksheet or Excel workbook in native
		Origin or Excel format. The selected range may be either contiguous or not. A comma
		separated selection string with one entry for each selected or partially selected
		column is returned along with the number of entries.
	Example:
		string strSelectedRanges;
		int iRet, iNumRanges;
		iRet = GetNonContiguousWksSelection( strSelectedRanges, iNumRanges );
	Parameters:
		strSelectedRanges=Output selected ranges
		iNumRanges=Number of selected ranges (one for each selected or partially selected column)
		bSelectAll=Returns entire worksheet/workbook sheet as selection range, default is FALSE
	Return:
		If successful returns WKS_UTILS_NO_ERROR and a formatted string containing the
		current Origin worksheet or Excel workbook selection. The selection string is a comma
		separated string with one entry for each selected or partially selected column. The number
		of entries is also returned. If there is a problem getting the selection then
		WKS_UTILS_BAD_SEL_ERROR, an empty string, and 0 are returned. If there is no selection
		then WKS_UTILS_NO_SEL_WARNING, an empty string, and 0 are returned.
*/
int GetNonContiguousWksSelection( string& strSelectedRanges, int& iNumRanges, BOOL bSelectAll = FALSE );

/**
		Gets the current selection from an Origin worksheet or Excel workbook in native
		Origin or Excel format. The selected range must be contiguous.
	Example:
		string strSelectedRange;
		int iRet;
		iRet = GetContiguousWksSelection( strSelectedRange );
	Parameters:
		strSelectedRange=Output selected range
		bSelectAll=Returns entire worksheet/workbook sheet as selection range, default is FALSE
	Return:
		If successful returns WKS_UTILS_NO_ERROR and a formatted string containing the
		current Origin worksheet or Excel workbook selection. If bSelectAll is TRUE then
		the entire worksheet is returned as the selection string. If there is a problem
		getting the selection or if the selected range is not contiguous then
		WKS_UTILS_BAD_SEL_ERROR and an empty string are returned. If there is no 
		selection then WKS_UTILS_NO_SEL_WARNING and an empty string are returned.
*/
int GetContiguousWksSelection( string& strSelectedRange, BOOL bSelectAll = FALSE );

/**
		Parse a worksheet/workbook selection string and return the worksheet/workbook name, workbook sheet
		name (if Excel workbook), and column and row selection indices (as integers).
	Example:
		
	Parameters:
		strWksSelection=Input worksheet/workbook selection string
		strWksName=Output worksheet/workbook name
		strSheetName=If Excel workbook output workbook sheet name else NULL
		iC1=Output index of first selected column (0 based)
		iC2=Output index of last selected column (0 based)
		iR1=Output index of first selected row (0 based)
		iR2=Output index of last selected row (0 based)
	Return:
		Returns TRUE, worksheet/workbook name, workbook sheet name (if Excel), and column and row selection
		indices (as integers) on success and FALSE on failure.
*/
BOOL ParseWksSelection( string strWksSelection, string& strWksName, string& strSheetName, int& iC1, int& iC2, int& iR1, int& iR2 );

/**
		Determine whether or not the specified elements of a vectorbase derived object are uniformly
		spaced.
	Example:
		See function omConvertWksToMatrixDirect in OMat.c for sample call.
	Parameters:
		dDelta=Output difference between elements if uniform spacing else 0.0
		vIn=Input vectorbase derived object
		dTol=Input relative tolerance between 0 and 1 (default is 0.05)
		i1=Input beginning 0 based index of vector to test (default is 0)
		i2=Input ending 0 based index of vector to test (default -1 tests to last element)
	Return:
		Returns TRUE and NANUM if only one element is tested, returns TRUE and the difference between the first
		two elements tested if the spacing is uniform, returns FALSE and the difference between the first two
		elements tested if the spacing is not uniform, or returns FALSE and NANUM if there is an error.
*/
BOOL VectorHasUniformSpacing( double& dDelta, vectorbase& vIn, double dTol = 0.05, int i1 = 0, int i2 = -1 );

/**
		rename the worksheet name to the name of binary data file
	Parameters:
		wks		= The worksheet object
		lpcszFileName	= The File name
	Return:
		Returns 0 to indicate successful. Non-zero value means error
*/
BOOL RenameToFileName(Worksheet& wks, LPCSTR lpcszFileName);

/**
		Add the current worksheet/workbook sheet selection ranges to subnodes in the specified
		TreeNode.
	Example:
		See function call in StatisticsOn.c.
	Parameters:
		tr=Input TreeNode to which selection range subnodes will added
		bAppend=Input option to append new selection range subnodes (true) or to reset tree and
			add starting in first subnode (default false)
		bSelectAll=Input option to select entire worksheet/workbook sheet if there is no selection (true)
			or to return a no selection warning code if there is no selection (default false)
		bColWise=Input option to add row wise selections (false) or column wise selections (default true)
	Return:
		Returns WKS_UTILS_NO_ERROR on success and a WKS_UTILS warning or error code on failure.
*/
int AddWksSelectionsToTreeNode(TreeNode& tr, bool bAppend = false, bool bSelectAll = false, bool bColWise = true);

/**
		Get the current worksheet/workbook sheet selection range and set the specified TreeNode
		with it. The selection range must be contained within a single worksheet/workbook column
		and the TreeNode must be passed by reference. Row ranges can optionally be shown or not
		be shown.
	Example:
		See function call in StatisticsOn.c.
	Parameters:
		tr=Input reference to TreeNode whose strVal is set
		bShowRows=Input option to show or not show row ranges, default is true (show row ranges)
	Return:
		Returns WKS_UTILS_NO_ERROR on success and a WKS_UTILS warning or error code on failure.
*/
int SetTreeNodeWithSingleColumnWksSelection(TreeNode& tr, bool bShowRows = true);

/** 
		Gets the current row wise selection from an Origin worksheet or Excel workbook in
		native Origin or Excel format. The selected range must be contiguous. A comma
		separated selection string with one entry for each selected or partially selected
		row is returned along with the number of entries.
	Example:
		string strSelectedRanges;
		int iRet, iNumRowRanges;
		iRet = GetRowWiseWksSelection(strSelectedRanges, iNumRowRanges);
	Parameters:
		strSelectedRanges=Output selected row ranges
		iNumRowRanges=Number of selected row ranges (one for each selected or partially selected row)
		bSelectAll=Returns entire worksheet/workbook sheet as selection range, default is FALSE
	Return:
		If successful returns WKS_UTILS_NO_ERROR and a formatted string containing the
		current Origin worksheet or Excel workbook selection. The selection string is a comma
		separated string with one entry for each selected or partially selected row. The number
		of entries is also returned. If there is a problem getting the selection then
		WKS_UTILS_BAD_SEL_ERROR, an empty string, and 0 are returned. If there is no selection
		then WKS_UTILS_NO_SEL_WARNING, an empty string, and 0 are returned.
*/
int GetRowWiseWksSelection( string& strSelectedRanges, int& iNumRowRanges, BOOL bSelectAll = FALSE);

/** 
		Find an empty column in the specified worksheet starting with the specified column.
	Example:
		int WksFirstEmptyColumn(LPCSTR lpcszWks)
		{
			if( NULL == lpcszWks )
				return -1;
			Worksheet wks(lpcszWks);
			if( !wks )
				return -1;
			int nEmptyCol = FindEmptyColumn(wks);
			if( -1 == nEmptyCol )
				printf("No empty columns in %s\n", wks.GetName());
			else
				printf("First empty column in %s is column %d\n", wks.GetName(), nEmptyCol);
			return nEmptyCol;
		}
	Parameters:
		wks=The worksheet to search.
		nStartCol=The index of the column to start the search
	Return:
		If an empty column is found return the index else return -1.
*/
int FindEmptyColumn(Worksheet &wks, int nStartCol=0);

/**
		Parse a worksheet/workbook selection string and return the data contained in the
		worksheet selection range.
	Parameters:
		strWksSelection=Input worksheet/workbook selection string having form "[Book1]Sheet1!$A$1:$B$7"
			or "Worksheet_A[1]:B[7]" 
		mData=Output data
	Return:
		Returns WKS_UTILS_NO_ERROR and a matrix containing the data contained in the worksheet
		selection range on success and returns WKS_UTILS_BAD_SEL_ERROR on failure.
*/
int GetDataInWksSelectionRange(string strWksSelection, matrix& mData);

/**
		Get a single column worksheet/workbook range without rows.
	Parameters:
		strRange=Input worksheet/workbook range with row ranges
	Return:
		Returns a single column worksheet/workbook range without rows.
*/
string GetSingleColumnRangeWithoutRows(const string& strRange);

/**
		Parse a column selection string and return the worksheet/workbook name, workbook sheet
		name (if Excel workbook), and column index (0 based offset, as an integer).
	Parameters:
		strColSelection=Input column selection string
		strWksName=Output worksheet/workbook name
		strSheetName=Output workbook sheet name (if Excel workbook, else NULL)
		iC1=Output index of selected column (0 based)
	Return:
		Returns TRUE, worksheet/workbook name, workbook sheet name (if Excel), and column index
		(0 based offset, as an integer) on success and FALSE on failure.
*/
BOOL ParseColSelection( string strColSelection, string& strWksName, string& strSheetName, int& iC1 );

/**
		Add the current selected columns to subnodes in the specified TreeNode.
	Example:
		See function call in StatisticsOn.c.
	Parameters:
		tr=Input TreeNode to which selected columns will added
		bAppend=Input option to append new columns (true) or to reset tree and
			add starting in first subnode (default false)
		bSelectAll=Input option to select all columns in worksheet/workbook sheet if there are
			none selected (true) or to return a no selection warning code if there is no selection
			(default false)
	Return:
		Returns WKS_UTILS_NO_ERROR on success and a WKS_UTILS warning or error code on failure.
*/
int AddColSelectionsToTreeNode(TreeNode& tr, bool bAppend = false, bool bSelectAll = false);


/**
*/
BOOL  	wks_report(Worksheet& wksOutput, TreeNode &trRep, BOOL bBatchProcessing);


/**
*/
#define STR_SEPARATOR_CELL "---"

typedef enum tagFITTERWKSREPORT
{
	FWR_PARA_NAME, 
	FWR_PARA_VALUE, 
	FWR_PARA_ERROR, 
	FWR_PARA_VARY,
	FWR_PARA_DEPEND,
	FWR_PARA_TVALUE,
	FWR_PARA_PROB,
	FWR_PARA_CONF_INT,
	FWR_PARA_LOWER_CONF,
	FWR_PARA_UPPER_CONF,
	FWR_PARA_ASYM_LOWER_CONF,
	FWR_PARA_ASYM_UPPER_CONF,
	FWR_CURVE_FITX,
	FWR_CURVE_FITY,
	FWR_CURVE_LOWER_CONF,
	FWR_CURVE_UPPER_CONF,
	FWR_CURVE_LOWER_PRED,
	FWR_CURVE_UPPER_PRED,
	FWR_CURVE_RESIDUES,
}	FITTERWKSREPORT;

int wks_report_parameters(Worksheet& wksOutput, TreeNode& trParams, TreeNode& trCols);

/**
	find column by name and return index
	Return:
		column index (0 offset) or -1 if given column name does not exist	
*/
int wks_get_col_index(Worksheet& wks, LPCSTR lpcszColName);

void wks_fill_separator_line(Worksheet& wks, int nRow, int nCol1, int nCol2);

int wks_report_table(Worksheet& wks, int nRow, int nCol, LPCSTR lpcszTitle, vector<string>& vsColTitles, vector<string>& vsRowTitles, matrix<double> matValues);

#endif //_WKS_UTILS_H