/*------------------------------------------------------------------------------*
 * File Name: Control.h															*
 * Creation: TD 11/12/2002														*
 * Purpose: Origin C support for Windows controls								*
 * Copyright (c) OriginLab Corp.2002											*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
#ifndef _CONTROL_H
#define _CONTROL_H

//#include <Common.h>
//#include <String.h>
//#include <Page.h>
#include <MSWin.h>


/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class Control : public Window
{
public:

	/**
		Example:
			Control cntl;
			cntl = MyDlg.GetItem(IDC_BUTTON1);
	*/
	Control();
	
	/**
	*/
	Control(Control& ctrl);

	/**
		Example:
			Control cntl = MyDlg.GetItem(IDC_BUTTON_OPEN);
			BOOL bRet = cntl.SetToolTip("Open File");
	*/
	BOOL SetToolTip(LPCSTR lpcszTxt);
	
	/**
		Example:
			Control cntl = MyDlg.GetItem(IDC_CHECK1);	
			cntl.Value = 1;
			ASSERT( 1 == cntl.Value );
	*/	
	int Value;
	
};

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class Edit : public Control
{
public:
	
	/**
	*/
	Edit();
	
	/**
	*/
	Edit(Control& ctrl);
	

	/**
		Call this function to retrieve the bounds of the current selection.
	Example:
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		long nStartChar, nEndChar;
		ed.GetSel(nStartChar, nEndChar);
	Parameters:
		nStartChar
			Zero-based index of the first character in the current selection.
		
		nEndChar
			Zero-based index of the last character in the current selection.
	Return:
		void
	SeeAlso:
		SetSel();
	*/
	void GetSel(int& nStartChar, int& nEndChar);

	/**
		Call this function to set the selection in the Edit control
	Example:
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		ed.SetSel(32000, 32000); // move to end of all text
		
	Parameters:
		nStartChar = Zero-based index of the first character in the current selection.
		nEndChar = Zero-based index of the last character in the current selection.
		bNoScroll =	Indicates whether the caret should be scrolled into view. 
		If FALSE, the caret is scrolled into view. If TRUE, the caret is not scrolled into view.
	Return:
		void
	
	SeeAlso:
		GetSel();
	*/
	void SetSel(int nStartChar, int nEndChar, BOOL bNoScroll = FALSE);
		
	/**
		Replaces the current selection in this Edit control with specified text.
	Example:
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		ed.SetSel(32000, 32000); // move to end of all text
		string str = "Sample"
		ed.ReplaceSel(str);

	Parameters:
		lpcszNewText = Pointer to a null-terminated string containing the replacement text.
		
		bCanUndo = To specify that this function can be undone, set the value of this parameter to TRUE. The default value is FALSE.
	Return:
		void
	
	SeeAlso:
		SetSel();
	*/
	void ReplaceSel(LPCTSTR lpcszNewText, BOOL bCanUndo = FALSE);

	/**
		Get index of first character in a line
	Example:
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		int nIndex = ed.LineIndex();
	Parameters:
		nLine = Contains the index value for the desired line in the text of the edit control, or contains -1. 
		If nLine is -1, it specifies the current line, that is, the line that contains the caret.

	Return:
		The character index of the line specified in nLine or -1 if the specified line number is greater then the number of lines in the edit control.

	Remarks:
		Call this function to retrieve the character index of a line within a multiple-line edit control. 
		The character index is the number of characters from the beginning of the edit control to the specified line. 

	*/
	int LineIndex( int nLine = -1 ) const;

	/**
	Example:	
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		int nLine = ed.LineFromChar();
	*/
	int LineFromChar( int nIndex = -1 ) const;

	/**
	Example:
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		int nLength = ed.LineLength();		
	*/
	int LineLength( int nLine = -1 ) const;
	
	/**
	Example:
		Edit ed = MyDlg.GetItem(IDC_EDIT1);
		int nCount = ed.GetLineCount();	
	*/
	int GetLineCount( ) const;
};	


//--------------------------------------------------------------------------
// RichEdit control
//--------------------------------------------------------------------------
typedef struct _charformat { 
  UINT     cbSize; 
  DWORD    dwMask; 
  DWORD    dwEffects; 
  LONG     yHeight; 
  LONG     yOffset; 
  COLORREF crTextColor; 
  BYTE     bCharSet; 
  BYTE     bPitchAndFamily; 
  char    szFaceName[LF_FACESIZE]; 
} CHARFORMAT;

/* CHARFORMAT masks */
#define CFM_BOLD		0x00000001
#define CFM_ITALIC		0x00000002
#define CFM_UNDERLINE	0x00000004
#define CFM_STRIKEOUT	0x00000008
#define CFM_PROTECTED	0x00000010
#define CFM_LINK		0x00000020		/* Exchange hyperlink extension */
#define CFM_SIZE		0x80000000
#define CFM_COLOR		0x40000000
#define CFM_FACE		0x20000000
#define CFM_OFFSET		0x10000000
#define CFM_CHARSET		0x08000000

/* CHARFORMAT effects */
#define CFE_BOLD		0x0001
#define CFE_ITALIC		0x0002
#define CFE_UNDERLINE	0x0004
#define CFE_STRIKEOUT	0x0008
#define CFE_PROTECTED	0x0010
#define CFE_LINK		0x0020
#define CFE_AUTOCOLOR	0x40000000		/* NOTE: this corresponds to */
										/* CFM_COLOR, which controls it */

/* Event notification masks */
#define ENM_NONE				0x00000000
#define ENM_CHANGE				0x00000001
#define ENM_UPDATE				0x00000002
#define ENM_SCROLL				0x00000004
#define ENM_KEYEVENTS			0x00010000
#define ENM_MOUSEEVENTS			0x00020000
#define ENM_REQUESTRESIZE		0x00040000
#define ENM_SELCHANGE			0x00080000
#define ENM_DROPFILES			0x00100000
#define ENM_PROTECTED			0x00200000
#define ENM_CORRECTTEXT			0x00400000		/* PenWin specific */
#define ENM_SCROLLEVENTS		0x00000008
#define ENM_DRAGDROPDONE		0x00000010

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class RichEdit : public Control
{
public:
	
	/**
	*/
	RichEdit();
	
	/**
	*/
	RichEdit(Control& ctrl);

	/**
			Sets the character formatting attributes in the current selection in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			CHARFORMAT cf; 
			cf.dwEffects = CFE_BOLD; 
			cf.dwMask = CFM_STRIKEOUT|CFM_BOLD;
			richEdit.SetSelectionCharFormat(cf);
			CHARFORMAT cfResult; 
			richEdit.GetSelectionCharFormat(cfResult);
			ASSERT((cfResult.dwMask&(CFM_STRIKEOUT|CFM_BOLD)) == (CFM_STRIKEOUT|CFM_BOLD));
			ASSERT((cfResult.dwEffects & (CFE_STRIKEOUT|CFE_BOLD) ) == CFE_BOLD);
		Parameters:
			cf = CHARFORMAT structure containing the new character
			formatting attributes for the current selection.
		Return:
			Nonzero if successful; otherwise, 0.
		Remarks:
			Call this function to set the character formatting attributes
			for the text in the current selection in this RichEdit control.
			Only the attributes specified by the dwMask member of cf are
			changed by this function.
		SeeAlso:
			RichEdit::GetSelectionCharFormat
	*/
	BOOL SetSelectionCharFormat(CHARFORMAT& cf);

	/**
			Gets the starting and ending positions of the current 
			selection in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			long nStartChar, nEndChar;
			richEdit.GetSel(nStartChar, nEndChar);	
			string str;
			str.Format("Start:%d, End:%d\n", nStartChar, nEndChar);
			MessageBox(NULL, str, "Rich Edit Sample");
		Parameters:
			nStartChar = Zero-based index of the first character in
			the current selection.
			nEndChar = Zero-based index of the last character in
			the current selection.
		Return:
			void
		Remarks:
			Call this function to retrieve the bounds of 
			the current selection in this RichEdit control.
		SeeAlso:
			RichEdit::SetSel
	*/
	void GetSel(long& nStartChar, long& nEndChar);

	/**
			Sets the selection in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			// Select the character between 0 and 10
			richEdit.SetSel(0, 10);
			long nStartChar, nEndChar;
			richEdit.GetSel(nStartChar, nEndChar);	
			ASSERT(0 == nStartChar && 10 == nEndChar);
		Parameters:
			nStartChar = Zero-based index of the first character
			in the current selection.
			nEndChar = Zero-based index of the last character
			in the current selection.
		Return:
			void
		Remarks:
			Call this function to set the selection within this RichEdit control. 
			To select all the text in this RichEdit control, call SetSel 
			with a start index of 0 and an end index of -1.
		SeeAlso:
			RichEdit::GetSel
	*/
	void SetSel(long nStartChar, long nEndChar);
		
	/**
			Replaces the current selection in this RichEdit control with specified text.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			richEdit.SetSel(0, -1);
			string str = "Sample";
			richEdit.ReplaceSel(str, FALSE);
			ASSERT( richEdit.Text == str );
		Parameters:
			lpszNewText = Pointer to a null-terminated string 
			containing the replacement text.
			bCanUndo = To specify that this function can be undone, 
			set the value of this parameter to TRUE. The default value is FALSE.
		Return:
			void		
		Remarks:
			Call this function to replace the current selection in this
			RichEdit control with the specified text. To replace all the
			text in this RichEdit control, use the function Text.
			If there is no current selection, the replacement text is inserted
			at the insertion point, that is, the current caret location.	
		SeeAlso:
			RichEdit::SetSel
	*/
	void ReplaceSel(LPCTSTR lpszNewText, BOOL bCanUndo = FALSE);
			
	/**
			Gets the text of the current selection in this RichEdit control
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			richEdit.SetSel(0, -1);
			string str = richEdit.GetSelText();
			ASSERT( richEdit.Text == str );
		Parameters:
			nStartChar = Zero-based index of the first character
			in the current selection.
			nEndChar = Zero-based index of the last character
			in the current selection.
		Return:
			The string containing the current selection. 
		Remarks:
			Call this function to retrieve the text from
			the current selection in this RichEdit control.
		SeeAlso:
			none.
	*/
	string GetSelText();		
	
	/**
			Sets the read-only option for this RichEdit control
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			richEdit.SetReadOnly(FALSE);	
		Parameters:
			bReadOnly = Indicates if this RichEdit control should be read only.
		Return:
			Nonzero if successful; otherwise, 0.
		Remarks:
			Call this member function to change the read-only
			option for this RichEdit control. 
		SeeAlso:
			none.
	*/
	BOOL SetReadOnly(BOOL bReadOnly = TRUE);

	/**
			Determines the topmost visible line in this RichEdit control. 
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int nFirstVisible = richEdit.GetFirstVisibleLine();
			
			// Scroll the rich edit control so that the first visible line is the first line of text.
			if (nFirstVisible > 0)
			   richEdit.LineScroll(-nFirstVisible, 0);
		Return:
			Zero-based index of the uppermost visible line in this RichEdit control.
		Remarks		
			Call this function to determine the topmost 
			visible line in this RichEdit control.
		SeeAlso:
			RichEdit::LineScroll.
	*/
	int GetFirstVisibleLine();

	/**
			Determines if the contents of this RichEdit control have changed since the last save.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			BOOL bModify = richEdit.GetModify();			
		Parameters:
			nStartChar = Zero-based index of the first character in the current selection.			
			nEndChar = Zero-based index of the last character in the current selection.
		Return:
			Nonzero if the text in this RichEdit control has been modified; otherwise 0.		
		Remarks
			Call this function to determine if the contents of this RichEdit
			control have been modified. Window maintains an internal flag 
			indicating whether the contents of the rich edit control
			have been changed. This flag is cleared when the edit control 
			is first created and can also be cleared by calling the 
			SetModify member function.
		SeeAlso:
			none.
	*/
	BOOL GetModify();

	/**
			Sets or clears the modification flag for this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			richEdit.SetModify();
		Parameters:
			bModified = A value of TRUE indicates that the text has been modified, 
			and a value of FALSE indicates it is unmodified. By default, the modified flag is set.
		Return:
			void
		Remarks		
			Call this function to set or clear the modified flag for an edit control. 
			The modified flag indicates whether or not the text within the edit control
			has been modified. It is automatically set whenever the user changes the text. 
			Its value can be retrieved with the GetModify member function.
		SeeAlso:
			none.
	*/
	void SetModify(BOOL bModified = TRUE);
	
	/**
			Sets the current default character formatting attributes in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			CHARFORMAT cf; 
			cf.dwEffects = CFE_BOLD; 
			cf.dwMask = CFM_STRIKEOUT|CFM_BOLD;
			richEdit.SetDefaultCharFormat(cf);
			CHARFORMAT cfResult; 
			richEdit.GetDefaultCharFormat(cfResult);
			ASSERT((cfResult.dwMask&(CFM_STRIKEOUT|CFM_BOLD)) == (CFM_STRIKEOUT|CFM_BOLD));
			ASSERT((cfResult.dwEffects & (CFE_STRIKEOUT|CFE_BOLD) ) == CFE_BOLD);
		Parameters:
			cf = CHARFORMAT structure containing the new default character formatting attributes.
		Return:
			Nonzero if successful; otherwise, 0.	
		Remarks:
			Call this function to set the character formatting attributes for new text
			in this RichEdit control. Only the attributes specified by the dwMask member
			of cf are changed by this function.

		SeeAlso:
			RichEdit::GetDefaultCharFormat
	*/
	BOOL SetDefaultCharFormat(CHARFORMAT& cf);
	
	/**
			 Retrieves the character formatting attributes in the 
			 current selection in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			CHARFORMAT cf; 
			cf.dwEffects = CFE_BOLD; 
			cf.dwMask = CFM_STRIKEOUT|CFM_BOLD;
			richEdit.SetSelectionCharFormat(cf);
			CHARFORMAT cfResult; 
			richEdit.GetSelectionCharFormat(cfResult);
			ASSERT((cfResult.dwMask&(CFM_STRIKEOUT|CFM_BOLD)) == (CFM_STRIKEOUT|CFM_BOLD));
			ASSERT((cfResult.dwEffects & (CFE_STRIKEOUT|CFE_BOLD) ) == CFE_BOLD);
		Parameters:
			cf = Pointer to a CHARFORMAT structure to receive the character 
			formatting attributes of the current selection
		Return:
			The dwMask data member of cf. It specifies the character formatting attributes 
			that are consistent throughout the current selection.
		Remarks		
			Call this function to get the character formatting attributes
			of the current selection. The cf parameter receives the
			attributes of the first character in the current selection.
			The return value specifies which attributes are consistent
			throughout the selection.
		SeeAlso:
			RichEdit::SetSelectionCharFormat
	*/
	DWORD GetSelectionCharFormat(CHARFORMAT& cf);
	
	/**
			Retrieves the current default character formatting attributes in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			CHARFORMAT cf; 
			cf.dwEffects = CFE_BOLD; 
			cf.dwMask = CFM_STRIKEOUT|CFM_BOLD;
			richEdit.SetDefaultCharFormat(cf);
			CHARFORMAT cfResult; 
			richEdit.GetDefaultCharFormat(cfResult);
			ASSERT((cfResult.dwMask&(CFM_STRIKEOUT|CFM_BOLD)) == (CFM_STRIKEOUT|CFM_BOLD));
			ASSERT((cfResult.dwEffects & (CFE_STRIKEOUT|CFE_BOLD) ) == CFE_BOLD);
		Parameters:
			cf = Pointer to a CHARFORMAT structure which will hold the 
			default character formatting attributes.
		Return:
			The dwMask data member of cf. It specified the default
			character formatting attributes.
	
		SeeAlso:
			RichEdit::SetDefaultCharFormat;
	*/
	DWORD GetDefaultCharFormat(CHARFORMAT& cf);

	/**
			Gets the starting and ending positions of the current selection in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			COLORREF cr = RGB(255, 0, 0);
			richEdit.SetTextColor(0, 100, cr);
		Parameters:
			nStartChar = Zero-based index of the first character in the current selection.
			nEndChar = Zero-based index of the last character in the current selection.
			cr = Specifies the color to to be used for seleted text.
		Return:
			Nonzero if successful; otherwise, 0.
		Remarks:
			Call this function to get the default character formatting
			attributes of this RichEdit control.
		SeeAlso:
			none.
	*/
	BOOL SetTextColor(long nStartChar, long nEndChar, COLORREF cr);

	/**
			Get index of first character in a line
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int nBegin, nEnd;
			// Replace the second line, if it exists, of the rich edit control
			// with the text "Origin C".
			if ((nBegin = richEdit.LineIndex(1)) != -1)
			{
			   nEnd = nBegin + richEdit.LineLength(1); 
			   richEdit.SetSel(nBegin, nEnd);
			   richEdit.ReplaceSel("Origin C");
			}
		Parameters:
			nLine = Contains the index value for the desired line in 
			the text of the richedit control, or contains -1. 
			If nLine is -1, it specifies the current line, that is, the line that contains the caret.
		Return:
			The character index of the line specified in nLine or -1
			if the specified line number is greater then the number
			of lines in the edit control.	
		Remarks:
			Call this function to retrieve the character index of a 
			line within a multiple-line edit control. The character index
			is the number of characters from the beginning of the edit
			control to the specified line. 
	*/
	int LineIndex( int nLine = -1 ) const;

	/**
			Get line number from character index
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			// The index of the char to get information on.
			int nIndex = 10;
			long nLine = richEdit.LineFromChar(nIndex);
		Parameters:
			nIndex = the zero-based index value for the desired character
			in the text of the edit control, or contains -1. 
			If nIndex is -1, it specifies the current line, that is,
			the line that contains the caret.
		Return:
			The zero-based line number of the line containing the character
			index specified by nIndex. If nIndex is -1, the number of the 
			line that contains the first character of the selection is returned. 
			If there is no selection, the current line number is returned.	
		Remarks:
			Call this function to retrieve the line number of the line that contains
			the specified character index. A character index is the number of characters
			from the beginning of the rich edit control.
	*/
	long LineFromChar( long nIndex = -1) const;

	/**
			Get the line of text at specified line index
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int ii, nLineLength;
			int nLineCount = richEdit.GetLineCount();
			string strText;
			for (ii = 0; ii < nLineCount; ii++)
			{
				nLineLength = richEdit.LineLength(ii);
				strText = richEdit.GetLine(ii);
			}
		Parameters:
			nIndex = the zero-based index value for the desired character
			in the text of the edit control, or contains -1. If nIndex is -1,
			it specifies the current line, that is, the line that contains the caret.
		Return:
			the text at given line or an empty string is line specified is not a valid line number
		Remarks:
			The implementation of this function is different from the MFC version.
			The MFC version is more difficult to use.
	*/
	string GetLine( int nIndex = -1) const;


	/**
			Retrieves the length of a given line in this RichEdit control.
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int ii, nLineLength;
			int nLineCount = richEdit.GetLineCount();
			string strText;
			for (ii = 0; ii < nLineCount; ii++)
			{
				nLineLength = richEdit.LineLength(ii);
				strText = richEdit.GetLine(ii);
			}
		Parameters
			nLine = Specifies the character index of a character in the line whose
			length is to be retrieved. If this parameter is -1, the length of the
			current line (the line that contains the caret) is returned, 
			not including the length of any selected text within the line.
			When LineLength is called for a single-line edit control, 
			this parameter is ignored.	
		Return:
			When LineLength is called for a multiple-line edit control,
			the return value is the length (in bytes) of the line specified by nLine.
			When LineLength is called for a single-line edit control, 
			the return value is the length (in bytes) of the text in the edit control.
		Remarks
			Call this function to retrieve the length of a line in a RichEdit control. 
		
	*/
	int LineLength( int nLine = -1 ) const;

	/**
			Retrieves the number of lines in this RichEdit control.	
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int ii, nLineLength;
			int nLineCount = richEdit.GetLineCount();
			string strText;
			for (ii = 0; ii < nLineCount; ii++)
			{
				nLineLength = richEdit.LineLength(ii);
				strText = richEdit.GetLine(ii);
			}
		Parameters:
			None.
		Return:
			The number of lines in this richedit control.
		Remarks
			Call this function to retrieve the number of lines in the RichEdit control.
	*/
	int GetLineCount( ) const;

	
	/**
			 Scrolls the text in this RichEdit control. 	
		Example:
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int nFirstVisible = richEdit.GetFirstVisibleLine();
			
			// Scroll the rich edit control so that the first visible line is the first line of text.
			if (nFirstVisible > 0)
			   richEdit.LineScroll(-nFirstVisible, 0);
		Parameters:
			nLines = Specifies the number of lines to scroll vertically.
			nChars = Specifies the number of character positions to scroll horizontally. 
			This value is ignored if the rich edit control has either the ES_RIGHT or ES_CENTER style.
		Return:
			Zero-based index of the uppermost visible line in this RichEdit control.

		Remarks
			Call this function to scroll the text of a multiple-line edit control. 
			The edit control does not scroll vertically past the last line of text
			in the edit control. If the current line plus the number of lines specified
			by nLines exceeds the total number of lines in the edit control, 
			the value is adjusted so that the last line of the edit control
			is scrolled to the top of the edit-control window. 
			LineScroll can be used to scroll horizontally past the last character of any line.
			
	*/
	void LineScroll( int nLines, int nChars = 0 );

	/**
			Limits the amount of text a user can enter into the RichEdit control.
		Example:
			#define TEXT_LIMITS 4096
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int nChars = TEXT_LIMITS;
			richEdit.LimitText(nChars);
			int nLimits = richEdit.GetLimitText();
			ASSERT(TEXT_LIMITS == nLimits);
		Return:
			None.
		Parameters:
			nChars = Specifies the length (in bytes) of the text that the user can enter. 
			If this parameter is 0, the text length is set to UINT_MAX bytes. This is the default behavior.
		Remarks
			Call this function to limit the length of the text that the user can enter into an edit control. 
			Changing the text limit restricts only the text the user can enter. It has no effect on any 
			text already in the edit control, nor does it affect the length of the text copied to the edit
			control by the Text member function in Window. If an application uses the Text
			function to place more text into an edit control than is specified in the call to LimitText, 
			the user can delete any of the text within the edit control. However, the text limit will 
			prevent the user from replacing the existing text with new text, unless deleting the 
			current selection causes the text to fall below the text limit.

	*/
	void LimitText( long nChars = 0 );

	/**
			Gets the limit on the amount of text a user can enter into this RichEdit control.
		Example:
			#define TEXT_LIMITS 4096
			RichEdit richEdit = EditBoxes.GetItem(IDC_RICHEDIT1);
			int nChars = TEXT_LIMITS;
			richEdit.LimitText(nChars);
			int nLimits = richEdit.GetLimitText();
			ASSERT(TEXT_LIMITS == nLimits);
		Return:
			The current text limit, in bytes, for this RichEdit control.
		Parameters:
			None.
		Remarks
			Call this member function to get the text limit for this RichEdit control.
			The text limit is the maximum amount of text, in bytes, the rich edit control can accept.

	*/
	long GetLimitText( ) const;

};	

//--------------------------------------------------------------------------
//--------------------------------------------------------------------------

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class Slider : public Control
{
public:
	
	/**
		Example:
			Slider slider = MyDlg.GetItem(IDC_SLIDER1);
			Slider m_slider(slider);
	*/	
	Slider();
	
	/**
	*/
	Slider(Control& ctrl);
	
	/**
		Example:
			Slider m_slider = MyDlg.GetItem(IDC_SLIDER1);
			m_slider.Position = m_slider.RangeMax;
			int nPos = m_slider.Position;	
	*/
	int 	Position;
	
	/**
		Example:
			Slider m_slider = MyDlg.GetItem(IDC_SLIDER1);	
			int nMax = m_slider.RangeMax;
	*/
	int 	RangeMax;
	
	/**
		Example:
			Slider m_slider = MyDlg.GetItem(IDC_SLIDER1);	
			int nMin = m_slider.RangeMin;	
	*/
	int 	RangeMin;
	
	/**
		Example:
			Slider m_slider = MyDlg.GetItem(IDC_SLIDER1);	
			int nFreq = 3;
			m_slider.SetTicFreq(nFreq);	
	*/
	void 	SetTicFreq(int freq);

};	

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class GraphControl : public Control
{
public:
	
	/**
	*/
	GraphControl();
	
	/**
	*/
	GraphControl(Control& ctrl);
	
	/**
		Example:
	*/
	GraphPage 	GetPage();

	/**
		Create the Origin Graph inside the control window
		Parameters:
			dwNoClickBits = options to diable clicking on various components on a graph
			lpcszTemplateName = template name. If no path is specified, will assume the Origin path. NULL will use Origin.otp.

		Return:
			TRUE for success.
		Example:
			static BOOL OnInitDialog()
			{
				OriginGraph og1 = GraphDlg.GetItem(IDC_TWOGRAPHS_GRAPH1);
				// must create Origin graph
				og1.Create(NOCLICK_AXES | NOCLICK_TICKLABEL | NOCLICK_LAYERICON);
				return TRUE;
			}
			
		Remarks:
			You must create an Origin graph. Typically this is done in the OnInitDialog function.
	*/
	BOOL		Create(DWORD dwNoClickBits = NOCLICK_USE_DEFAULT, LPCSTR lpcszTemplateName=NULL);	  
	

};	

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class WorksheetControl : public Control
{
public:
	
	/**
	*/
	WorksheetControl();
	
	/**
	*/
	WorksheetControl(Control& ctrl);
	
	/**
		Attach a worksheet to this control
		Parameters:
			lpcszWorksheetName = worksheet name.
		Return:
			TRUE for success.
		Example:
			WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
			wc.Attach("Data1");
			wc.Update();
		Remarks:
	*/
	BOOL Attach(LPCSTR lpcszWorksheetName);
	
	/**
		Update the control
		Parameters:
			None.
		Return:
			None.
		Example:
			WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
			wc.Attach("Data1");
			wc.Update();
		Remarks:
	*/
	void Update();
	
	/**
		Update the control
		Parameters:
			nCol = column index,
			
			int nType = the column type. it can be following constants defined in OC_const.h
			#define	OKCOLTYPE_NUMERIC			0
			#define	OKCOLTYPE_TEXT				1
			#define	OKCOLTYPE_TIME				2
			#define	OKCOLTYPE_DATE				3
			#define	OKCOLTYPE_MONTH				4
			#define	OKCOLTYPE_WEEKDAY			5
			#define	OKCOLTYPE_COLUMN			6
			#define	OKCOLTYPE_DATASET			7
			#define	OKCOLTYPE_DATASET_X			8
			#define	OKCOLTYPE_TEXT_NUMERIC		9
			#define	OKCOLTYPE_CATEGORICAL		0x000A

			int nDesignation = the column designation, it can be following values defined in OC_const.h
			enum
			{
				OKDATAOBJ_DESIGNATION_Y = 0,
				OKDATAOBJ_DESIGNATION_NONE,	
				OKDATAOBJ_DESIGNATION_ERROR,
				OKDATAOBJ_DESIGNATION_X,
				OKDATAOBJ_DESIGNATION_L,
				OKDATAOBJ_DESIGNATION_Z,
				OKDATAOBJ_DESIGNATION_X_ERROR,
			};
		Return:
			TRUE for success.
		Example:
			WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
			wc.Attach("Data1");
			int nCol = 1;
			int nType;
			int nDesignation;
			ASSERT(wc.GetColInfo(nCol, nType, nDesignation));
			
		Remarks:
	*/	
	BOOL GetColInfo( int nCol, int& nType, int& nDesignation );	

	/**
	Remark: 
		Used to apply designation pattern to entire worksheet control
	Parameters:	
		lpcszDesignations - designation pattern 
			Possible values for designations:
			'X' - X column
			'Y' - Y column
			'Z' - Z column
			'E' - Y error column
			'L' - label column
			'M' - X error column
			'N' - ignore column
		
		bRepeat	- repeat pattern. Defines how the pattern is applied 
		if number of columns in worksheet control is larger then length of pattern.
		Possible values:
			TRUE - repeat entire pattern
			FALSE - set designations for the rest of columns to be the last one deifined by pattern
		
	Example: 
		// the following code will set column designations in worksheet control
		// 1st column - X
		// 2nd column - Y
		// 3rd column - label
		// 4th column - Y
		// 5rd column - label
		// pattern will repeat if worksheet control contains more than 5 columns
		WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
		wc.Attach("Data1");
		wc.SetColDesignations("XYLYL");
	
	Return:
		TRUE for success. Otherwise FALSE;
	*/
	BOOL	SetColDesignations(LPCSTR lpcszDesignations, BOOL bRepeat = TRUE);
	
	/**
	Remark: 
		Gets column designations for entire worksheet control in form of string
		Possible values for designations:
			'X' - X column
			'Y' - Y column
			'Z' - Z column
			'E' - Y error column
			'L' - label column
			'M' - X error column
			'N' - ignore column
			
	Example:
		WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
		wc.Attach("Data1");
		string strDesignations = wc.GetColDesignations();
	*/
	string	GetColDesignations();
	
	/**
	Remark: 
		Gets column designations for entire worksheet in form of string
		Possible values for designations:
			'X' - X column
			'Y' - Y column
			'Z' - Z column
			'E' - Y error column
			'L' - label column
			'M' - X error column
			'N' - ignore column
			
	Example:
		WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
		wc.Attach("Data1");
		string strDesignations;
		wc.GetColDesignations(strDesignations);
	*/
	void	GetColDesignations(string& strDesignations);
	
	/**
	*/
	BOOL	SetColFormats(LPCSTR lpcszFormats, BOOL bRepeat = TRUE);
	
	/**
	*/
	string	GetColFormats();
	
	/**
	*/
	void	GetColFormats(string& strFormats);	

	/**
		Remove all items
		Parameters:
			None.
		Return:
			None.
		Example:
			WorksheetControl wc = dlg.GetItem(IDC_WORKSHEET);
			wc.Attach("Data1");
			wc.RemoveAllItems();
		Remarks:
	*/
	void RemoveAllItems();
};

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class Button : public Control
{
public:
	
	/**
	*/
	Button();
	
	/**
	*/
	Button(Control& ctrl);
	
	/**
		Example:
			Button m_CheckBtn = MyDlg.GetItem(IDC_CHECK1);
			if(m_CheckBtn.Check)
			{
				out_str("Selected!");
			}
	*/
	int	Check;
};


/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class BitmapRadioButton : public Button
{
public:

	/**
	*/
	BitmapRadioButton();
	
	/**
	*/
	BitmapRadioButton(Control& ctrl);

	/**
	*/
	BOOL Init(UINT nStates, UINT nBmpResID, UINT nBmpSliceWidth, vector<string> & vstrTips, UINT nBmpOffset = 0);
};


/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class ComboBox : public Control
{
public:
	
	/**
	*/
	ComboBox();
	
	/**
	*/
	ComboBox(Control& ctrl);
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			int nSel = 3;
			m_cmbBox.SetCurSel( nSel );
			ASSERT( 3 == m_cmbBox.GetCurSel() );
	*/
	int 	GetCurSel() const;
	
	/**
	*/
	void 	SetCurSel(int nSelect);
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			int nCount = m_cmbBox.GetCount();
	*/
	int		GetCount() const;
	
	/**
		Example:
			string strText;
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			m_cmbBox.GetLBText(m_cmbBox.GetCurSel() ,strText);
			out_str(strText);	
	*/
	void	GetLBText(int nIndex, string& str)	const;
	
	/**
		Example:
			string strText("Red");
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			int nCount = m_cmbBox.GetCount();
			
			int nRet = m_cmbBox.AddString(strText);
			ASSERT(nCount+1 == m_cmbBox.GetCount());
	*/
	int 	AddString(LPCSTR lpcsz);
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			string str = "try";
			int nInsert = m_cmbBox.InsertString(0, str);
	*/
	int 	InsertString(int nIndex, LPCSTR lpcsz);
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			string str = "try";
			int nRet = m_cmbBox.DeleteString(str);	
	*/
	int 	DeleteString(int nIndex);
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			DWORD dData = 100;
			int nSetData = m_cmbBox.SetItemData(0,dData);
	*/
	int		SetItemData(int nIndex, DWORD dwData);
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			DWORD dData = m_cmbBox.GetItemData(0);			
	*/
	DWORD	GetItemData(int nIndex) 	const;
	
	/**
	*/
	void	ResetContent();
	
	/**
		Example:
			ComboBox  m_cmbBox = MyDlg.GetItem(IDC_COMBO1);
			int nSel = m_cmbBox.SelectString(0,"try");
	*/
	int		SelectString(int nStartAfter, LPCSTR lpcsz); 
};

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class ListBox : public Control
{
public:
	
	/**
	*/
	ListBox();
	
	/**
	*/
	ListBox(Control& ctrl);
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			int nSel = m_List.GetCurSel();
	*/
	int 	GetCurSel() const;	
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			m_List.SetCurSel(1);	
	*/
	void 	SetCurSel(int nSelect);
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			int nCount = m_List.GetCount();
	*/
	int		GetCount() const;
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			string str;
			m_List.GetText(0,str);
			out_str(str);
	*/	
	void	GetText(int nIndex, string& str);
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			int nAdd = m_List.AddString("Test");
	*/
	int 	AddString(LPCSTR lpcsz);
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			int nRet = m_List.InsertString(0, "Good");	
	*/
	int 	InsertString(int nIndex, LPCSTR lpcsz);
	
	/**
		Example:
			ListBox  m_List = MyDlg.GetItem(IDC_LIST1);	
			int nRet = m_List.DeleteString(0);
	*/
	int 	DeleteString(int nIndex);
	
	/**
	*/
	void	ResetContent();

};



#define MF_STRING           0x00000000L
#define MF_ENABLED          0x00000000L
#define MF_GRAYED           0x00000001L
#define MF_DISABLED         0x00000002L
#define MF_UNCHECKED        0x00000000L
#define MF_CHECKED          0x00000008L

/** >User Interface Controls
		This class is only available to OriginPro users.
	Example: 
*/
class Menu
{
public:
	
	/**
	*/
	Menu();
	
	/**
	*/
	Menu(Menu& menu);
	
	/**
	*/
	BOOL	Add(LPCSTR lpcszText, Function fnHandler, UINT nFlags = MF_STRING);

	/**
	*/
	BOOL	Create();

	/**
	*/
	BOOL	TrackPopupMenu(UINT nFlags,  int x, int y, HWND hwnd);

	/**
	*/
	Window	GetOwnerWindow();

	/**
	*/
	int		GetMenuItemID(uint nPos);
	
};

#endif //_CONTROL_H
