/*------------------------------------------------------------------------------*
 * File Name: OCN_g08.h															*
 * Creation: SDB 5/22/2001														*
 * Purpose: Origin C Header for NAG functions									*
 * Copyright (c) OriginLab Corp.	2001										*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/

#ifndef _O_NAG_G08_H
#define _O_NAG_G08_H

//#importdll "ONAG" // NAG DLL prepared by OriginLab
#pragma dll(ONAG)	
#include <NAG\nag_types.h>


/* begin proto */

/**	g08aac
		Performs the Sign test on two related samples of size n.

Example:

void test_nag_sign_test()
{
	double p; 
	double x[] = {4, 4, 5, 5, 3, 2, 5, 3, 1, 5, 5, 5, 4, 5, 5, 5, 5};
	int i, s, n, non_tied;
	n= 17;
	double y[] ={2,3,3, 3,3,3,3,3,2,3,2,2,5,2,5,3,1};
	
	printf("\nSign Test\n\n");
	printf("Data values\n\n");
	for(i =0; i < n; i++)
		printf("%3.0f",x[i]);
	printf("\n");
	for(i =0; i < n; i++)
		printf("%3.0f",y[i]);
	printf("\n");
	nag_sign_test(n, x, y, &s, &p, &non_tied);
	printf("Test statistic    %5ld\n",s);
	printf("Observations      %5ld\n",non_tied);
	printf("Lower tail prob   %5.3f\n",p);	
}

The output is following:
	
Sign Test

Data values

  4  4  5  5  3  2  5  3  1  5  5  5  4  5  5  5  5
  2  3  3  3  3  3  3  3  2  3  2  2  5  2  5  3  1
Test statistic        3
Observations         14
Lower tail prob   0.029

	
Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
	successfully call of the nag_sign_test function.
	
*/

int	nag_sign_test(
	int n, // Size of each sample, n>=1.
	const double x[], // Sample1 dataset, x[i-1] and y[i-1] must be set to ith pair of data values. 
	const double y[], // Sample2 dataset, x[i-1] and y[i-1] must be set to ith pair of data values.
	int *s, // The returned Sign test statistic, S.
	double *p, // The returned lower tail probability, p, corresponding to S.
	int *non_tied // The returned number of non-tied pairs.
); // Sign test on two paired samples.



/**	g08acc
		Performs the Median test on two independent samples of possibly unequal size.

Example:

void test_nag_median_test()
{
	double p;
	int i, above, below, n1, n2;
	n1 = 16;
	n2 = 23;
	double x[] = {13, 6, 12, 7, 12, 7, 10, 7, 10, 7, 10, 7, 10, 8, 9, 8}
	double y[] = {17, 6, 16, 8, 15, 8, 15, 10, 15, 10, 14, 10, 14, 11, 14, 11,
					13, 12, 13, 12, 13, 12, 12}
	printf("\nMedian test\n\n");
	printf("Data values\n\n");
	printf("   Group1   ");
	for(i = 0; i < n1; i++)
	{
		printf("%4.0f",x[i]);
		if((i+1) % 8 == 0)
			printf("\n            ");		
	}
	
	printf("\n   Group2   ");
	for(i = 0; i < n2; i++)
	{
		printf("%4.0f",y[i]);
		if((i+1) % 8 == 0)
			printf("\n            ");		
	}
	nag_median_test(n1, x, n2, y, &above, &below, &p);
	printf("\n");
	printf("%6ld  scores below madian in group 1\n",above);
	printf("%6ld  scores below madian in group 2\n",below);
	printf("\n		Significance   %8.5f\n", p);
}	

The output is following:

Median test

Data values

   Group1     13   6  12   7  12   7  10   7
              10   7  10   7  10   8   9   8
            
   Group2     17   6  16   8  15   8  15  10
              15  10  14  10  14  11  14  11
              13  12  13  12  13  12  12
    13  scores below madian in group 1
     6  scores below madian in group 2

		Significance    0.00088
	
Parameter:

Return:
	The function returns NAG error code, 0 if no error.
*/
int	nag_median_test(
	int n1, // The size of the first sample, n1>=1.
	const double x[], // The data values of the first sample.
	int n2, // The size of the second sample, n2>=1.
	const double y[], // The data values of the second sample.
	int *above, // The returned number of scores in the first sample which lie above the pooled median.
	int *below, // The returned number of scores in the first sample which lie below the pooled median.
	double *p // The returned tail probability, p, corresponding to the observed dichotomy of the two samples.
); // Median test on two samples of unequal size.

/**	g08aec
		Performs the Friedman two-way analysis of variance by ranks on k related samples of size n.

Example:

void test_nag_friedman_test()
{
	double fr, sig;
	int i, ix, j, k, n;
	n = 18;
	k =3;
	ix =k;
	double x[] = {1, 2, 1, 1, 3, 2, 3, 1, 3, 3, 2, 2, 3, 2, 2.5, 3, 3, 2,
				  3, 3, 3, 2, 1, 3, 2, 3, 1, 1, 3, 3, 2, 3, 2.5, 2, 2, 3,
				  2, 1, 2, 3, 2, 1, 1, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1};
	printf("\nData values\n");
	printf("\n	Group   Group    Group\n");
	printf("\n 	1      2       3\n");
	for(j = 0; j < 18; j++)
	{
		for( i =0; i < 3; i++)
			printf("%7.1f", x[i * 18 + j]);
		printf("\n");	 
	}
	
	nag_friedman_test(k, n, x, n, &fr, &sig);
	printf("\n");
	printf("Test statistics       %6.3f\n", fr);
	printf("Degrees of freedom    %6ld\n", k-1);
	printf("Significance          %6.3f\n ", sig);	 
}

The output is following:
	
	Data values

	Group   Group    Group

 	1      2       3
    1.0    3.0    2.0
    2.0    3.0    1.0
    1.0    3.0    2.0
    1.0    2.0    3.0
    3.0    1.0    2.0
    2.0    3.0    1.0
    3.0    2.0    1.0
    1.0    3.0    2.0
    3.0    1.0    2.0
    3.0    1.0    2.0
    2.0    3.0    1.0
    2.0    3.0    1.0
    3.0    2.0    1.0
    2.0    3.0    1.0
    2.5    2.5    1.0
    3.0    2.0    1.0
    3.0    2.0    1.0
    2.0    3.0    1.0

Test statistics        8.583
Degrees of freedom         2
Significance           0.014
	
Parameter:
		
Return:
	NAG error code, 0 if no error.
*/

int	nag_friedman_test(
	int k, // The number of samples, k>1.
	int n, // The size of each sample, n>=1.
	const double x[], // Two dimension array (x[k,tdx]) of all data points where x[i-1][j-1] is the jth observation in sample i.
	int tdx, // The first dimension of x as declared in the function from which nag_friedman_test is called. 
	double *fr, // The returned value of the Friedman test statistic, FR.
	double *p // The returned approximate significance, p, of the Friedman test statistic.
); // Friedman two-way analysis of variance on k matched samples.



/**	g08afc
		Performs the Kruskal-Wallis one-way analysis of variance by ranks on k independent samples
		of possibly unequal sizes.
Example:
	
void test_nag_kruskal_wallis_test()
{
	double h, p;
	int count, i, ii, k, lx, nhi, ni, nlo;
	k =5;
	int l[] = { 5, 8, 6, 8, 8};
	printf("\n");
	printf("Kruskal_wallis test");
	printf("\n");
	printf("Data values \n\n");
	printf("  Group    Observations\n");
	
	lx = 0;
	for(i =0; i < 5; i++)
		lx +=l[i];
	double x[] = {23, 27, 26, 19, 30, 29, 25, 33, 36, 32,
				  28, 30, 31, 38, 31, 28, 35, 33, 36, 30,
				  27, 28, 22, 33, 34, 34, 32, 31, 33, 31,
				  28, 30, 24, 29, 30};
	nlo = 1;
	for(i = 0; i < k; i++)
	{
		ni = l[i];
		nhi = nlo + ni - 1;
		printf("%5ld     ", i);
		count = 1;
		for( ii = nlo; ii <= nhi; ii++)
		{
			printf("%4.0f",x[ii -1]);
			if(count % 10 == 0)
				printf("\n");			
		}
		nlo +=ni;
		printf("\n");	
	}	
	
	nag_kruskal_wallis_test(k, l, x, lx, &h, &p);
	printf("\n");
	printf("\n");
	printf("Test statistics       %6.3f\n",h);
	printf("Degrees of freedom    %6ld\n", k-1);
	printf("Significance          %6.3f\n ",p); 
}

The output is following:
Data values 

  Group    Observations
    0       23  27  26  19  30
    1       29  25  33  36  32  28  30  31
    2       38  31  28  35  33  36
    3       30  27  28  22  33  34  34  32
    4       31  33  31  28  30  24  29  30


Test statistics       10.537
Degrees of freedom         4
Significance           0.032
	
Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
	successfully call of the nag_kruskal_wallis_test function.
*/

int	nag_kruskal_wallis_test(
	int k, // The number of samples, k>=2.
	const int l[], // l[i-1] contains the number of observations in sample i.   
	const double x[], // Contains the observations in k groups each having l[i-1] scores (e.g. The first l[0] elements contain the scores in the first group, etc.)
	int lx, // The total number of observations, N.
	double *h, // The returned value of the Kruskal-Wallis statistic, H.
	double *p // The returned approximate significance, p, of the Kruskal-Wallis test statistic.
); // Kruskal-Wallis one-way analysis of variance on k samples of unequal size. 


/**	g08agc
		Performs the Wilcoxon one-sample (matched pairs) signed rank test.

Example:


void test_nag_wilcoxon_test()
{
	double median, p, w, z;
	int i, n, non_zero;
	double data[8];
	n = 8;
	double x[] = {82, 69, 73, 43, 58, 56, 76, 65};
	double y[] = {63, 42, 74, 37, 51, 43, 80, 62};
	printf("\n\n");
	printf("Wilcoxon one sample signed ranks test");
	printf("\n");
	printf("Data Values \n");
	for(i = 0; i < n; i++)
		printf("%5.1f", x[i]);
	printf("\n");
	for(i = 0; i < n; i++)
		printf("%5.1f", y[i]);
	printf("\n");
	for(i = 0; i < n; i++)
		data[i] = x[i] -y[i];
	median = 0.0;
	nag_wilcoxon_test(n, data, median, Nag_TwoTail, Nag_IncSignZerosN, &w, &z, &p, &non_zero);
	printf("\n");
	printf("Test statistics              =%8.4f\n",w);
	printf("Normalized  test statistics  =%8.4f\n", z);
	printf("Degree of freedom            =%8ld\n",non_zero);
	printf("Two tail probability         =%8.4f\n", p); 			 
}

The output is following:

Wilcoxon one sample signed ranks test
Data Values 
 82.0 69.0 73.0 43.0 58.0 56.0 76.0 65.0
 63.0 42.0 74.0 37.0 51.0 43.0 80.0 62.0

Test statistics              = 32.0000
Normalized  test statistics  =  1.8904
Degree of freedom            =       8
Two tail probability         =  0.0547	

Parameters:

Return:
	This function returns NAG error code, 0 if no error.	
*/

int	nag_wilcoxon_test(
	int n, // The size of the sample, n>=1.
	const double x[], // The sample observations.
	double median, // The median test value.
	Nag_TailProbability tail, // Indicates choice of tail probability and hence the alternative hypothesis.
	Nag_IncSignZeros zeros, // Indicates whether or not to include cases where di=0.0 when ranking di's.
	double *rs, // The returned Wilcoxon rank sum statistic, W, being the sum of the positive ranks.
	double *rsnor, // The returned approximate Normal test statistic, z.
	double *p, // The returned tail probability, p, as specified by the parameter tail.
	int *nz1 // The returned number of non-zero di's.
); // Performs the Wilcoxon one-sample (matched pairs) signed rank test.



/**	g08amc
		Performs the Mann-Whitney U test on two independent samples of possibly unequal size
		and calculates the exact probability for the Mann-Whitney rank sum test statistic for
		the case where there are either ties or no ties in the samples pooled together.

Example:

void test_nag_mann_whitney()
{
	double p, u, z;
	int i , n1, n2;
	n1 = 16;
	n2 = 23;
	printf("Sample size of group 1 = %5ld\n", n1);
	printf("Sample size of group 2 = %5ld\n", n2);
	double x[] ={13.0, 6.0, 12.0, 7.0, 12.0, 7.0, 10.0, 7.0,
				10.0, 7.0, 16.0, 7.0, 10.0, 8.0, 9.0, 8.0};
	printf("Mann-Whitney U test\n");
	printf("Data values\n");
	printf("	Group 1	");
	for(i = 0; i < n1; i++)
	{
		printf("%5.1f",x[i]);
		if((i + 1) % 8 == 0)
			printf("\n");
	}
	
	double y[] ={17.0, 6.0, 10.0, 8.0, 15.0, 8.0, 15.0, 10.0, 15.0, 10.0, 14.0, 10.0,
				14.0, 11.0, 14.0, 11.0, 13.0, 12.0, 13.0, 12.0,13.0,12.0, 12.0};
	printf("\n	Group 2	");
	for(i = 0; i < n2; i++)
	{
		printf("%5.1f",y[i]);
		if((i + 1) % 8 == 0)
			printf("\n");
	}
	nag_mann_whitney(n1, x, n2, y, Nag_LowerTail, Nag_CompProbApprox, &u, &z, &p);
	printf("\n\n");
	printf("Test statistics              =%8.4f\n", u);
	printf("Normal statistics 		 =%8.4f\n", z);
	printf("Approximate tail probability =%8.4f\n", p);
	nag_mann_whitney(n1, x, n2, y, Nag_LowerTail, Nag_CompProbExact, &u, &z, &p);
	printf("Exact tail probability       =%8.4f\n",p);	
}

The output is following:
	
Sample size of group 1 =    16
Sample size of group 2 =    23
Mann-Whitney U test
Data values
	Group 1	 13.0  6.0 12.0  7.0 12.0  7.0 10.0  7.0
 10.0  7.0 16.0  7.0 10.0  8.0  9.0  8.0

	Group 2	 17.0  6.0 10.0  8.0 15.0  8.0 15.0 10.0
 15.0 10.0 14.0 10.0 14.0 11.0 14.0 11.0
 13.0 12.0 13.0 12.0 13.0 12.0 12.0

Test statistics              = 86.0000
Normal statistics 		 = -2.8039
Approximate tail probability =  0.0025
Exact tail probability       =  0.0020


Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int	nag_mann_whitney(
	int n1, // The number of non-tied pairs, n1>=1. 
	const double x[], // The first vector of observations.
	int n2, // The size of the second sample, n2>=1. 
	const double y[], // The second vector of observations.
	Nag_TailProbability tail, // Indicates choice of tail probability and hence the alternative hypothesis.
	Nag_CompProb exact, // Indicates if exact probability is to be computed
	double *u, // The returned Mann-Whitney rank sum statistic, U. 
	double *z, // The returned approximate Normal test statistic, z.
	double *p // The returned exact tail probability, p, as specified by the parameter tail.
); // Performs the Mann-Whitney U test on two independent samples.

/**	g08cbc
		Performs the one-sample Kolmogorov-Smirnov test for standard distributions.
	
	
Example:


void test_nag_1_sample_ks_test()
{
	double d, p, z;
	int i, n, np, ntype;
	Nag_TestStatistics ntype_enum;
	n = 30;
	
	double x[] ={0.01, 0.30, 0.20, 0.90, 1.20, 0.09, 1.30, 0.18, 0.90, 0.48,
				 1.98, 0.03, 0.50, 0.70, 0.07, 0.60, 0.95, 1.00, 0.31, 1.45,
				 1.04, 1.25, 0.15, 0.75, 0.85, 0.22, 1.56, 0.81, 0.57, 0.55};
				 
	np =2;
	double par[] ={0.0, 2.0};
	ntype = 1;
	ntype_enum = Nag_TestStatisticsDAbs;
	
	nag_1_sample_ks_test(n, x, Nag_Uniform, par, Nag_ParaSupplied, ntype_enum, &d, &z, &p);
	printf("Test against uniform distribution on(0, 2)\n");
	printf("Test statistics  D    	= %8.4f\n", d);
	printf("Z statistic    			= %8.4f\n", z);
	printf("Tail probability		= %8.4f\n", p);
	
	
	np = 2;
	par[0] = 0.0;
	par[1] = 1.0;
	ntype = 1;
	ntype_enum = Nag_TestStatisticsDAbs;
	nag_1_sample_ks_test(n, x, Nag_Normal, par, Nag_ParaEstimated, ntype_enum, &d, &z, &p);
	printf("Test  against Normal distribution with parameters estimated from the data\n");
	printf("\n");
	printf("Mean = %6.4f	and varaince = %6.4f\n",par[0],par[1]);
	printf("Test statistics  D    	= %8.4f\n", d);
	printf("Z statistic    			= %8.4f\n", z);
	printf("Tail probability		= %8.4f\n", p); 	
	
}

The output is following:
	
Test against uniform distribution on(0, 2)
Test statistics  D    	=   0.2800
Z statistic    			=   1.5336
Tail probability		=   0.0143
Test  against Normal distribution with parameters estimated from the data

Mean = 0.6967	and varaince = 0.2564
Test statistics  D    	=   0.1108
Z statistic    			=   0.6068
Tail probability		=   0.8925


Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int nag_1_sample_ks_test(
	int n, // The number of observations in the sample, n >= 3.
	const double x[], // The sample observations x1, x2,...,xN.
	Nag_Distributions dist, // The NAG distribution.
	double par[], // Contains the known values of the parameter(s) of the null distribution.
	Nag_ParaEstimates estima, // Specifies whether the values of the parameter(s) are known or estimated from data.
	Nag_TestStatistics dtype, // The test statistic to be calculated...i.e. the choice of alternative hypothesis.
	double *d, // The Kolmogorov-Smirnov test statistic.
	double *z, // A standardized value, Z, of the test statistic, D, without correction for continuity.
	double *p // The probability, P, associated with the observerd value of D. 
); // Performs the one-sample Kolmogorov-Smirnov test for standard distributions.



/**	g08cdc
		Performs the two-sample Kolmogorov-Smirnov distribution test.
Example:


void test_nag_2_sample_ks_test()
{
	double d, enda, endb, p, z;
	int init, i, m, n, ntype;
	Nag_TestStatistics ntype_enum;
	double x[100], y[50];
	n = 100;
	m = 50;
	init = 0;
	enda = 0.0;
	endb = 2.0;
	for(i = 0; i < n; i++)
		x[i] = enda +(endb -enda) * ran(init);
	enda = 0.25;
	endb = 2.25;
	for(i = 0; i < m; i++)
		y[i] = enda + (endb - enda) * ran(init);
		
	ntype = 1;
	ntype_enum = Nag_TestStatisticsDAbs;
	nag_2_sample_ks_test(n, x, m, y, ntype_enum, &d, &z, &p);
	printf("Test statistics  D    	= %8.4f\n", d);
	printf("Z statistic    			= %8.4f\n", z);
	printf("Tail probability		= %8.4f\n", p); 
}

Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int	nag_2_sample_ks_test(
	int n1, // The number of observations in the first sample, n1>=1. 
	const double x[], // The observations from the first sample. 
	int n2, // The number of observations in the second sample, n2>=1.  
	const double y[], // The observations from the second sample.
	Nag_TestStatistics ntype, // The statistic to be computed or the choice of alternative hypothesis. 
	double *d, // The  returned Kolmogorov-Smirnov test statistic according to the value of ntype.
	double *z, // The returned standardized value, Z, of the test statistic, D, without correction for continuity.
	double *p // The returned tail probability, P, associated with the observerd value of D.
); // Performs the two-sample Kolmogorov-Smirnov distribution test.


/**	g08cgc
		Performs the chi^2 goodness of fit test, for standard continuous distributions and for data
		with a chosen number of class intervals.
Example:

void test_nag_chi_sq_goodness_of_fit_test()
{
	double chisq, p, xmax, xmin;
	int i, iclass, init, n, nclass, ndf, npest;
	double x[100], cint[4], par[2] = {0.0, 1.0};
	int ifreq[5];
	double chisqi[5], eval[5], prob[5];
	Nag_Distributions cdist_enum;
	Nag_ClassBoundary class_enum;
	 
	n = 100;
	nclass = 5;
	cdist_enum = Nag_Uniform;
	
	npest = 0;
	init = 0;
	for( i =0; i < n; i++)
		x[i] = 	ran(init);		//generate random variable from 0 to 1.
	iclass = 1;
	cint[0] = par[0] + (par[1] -par[0])/nclass;
	for(i = 1; i < nclass -1; i++)
		cint[i] = cint[i - 1] +(par[1] - par[0])/nclass;
		
	class_enum = Nag_ClassBoundaryUser;
	nag_frequency_table(n, x, nclass, class_enum, cint, ifreq, &xmin, &xmax);
	nag_chi_sq_goodness_of_fit_test(nclass, ifreq, cint, cdist_enum, par, npest, prob, &chisq, &p, &ndf, eval, chisqi);
	printf("\n");
	printf("Chi-squared test statistic			=%10.4f\n",chisq);
	printf("Degrees of freedom.					=%5ld\n", ndf);
	printf("Significance level					=%10.4f\n", p);
	printf("\n");
	printf("The contributions to the test statistic are :");
	for(i = 0; i < nclass; i++)
		printf("%10.4f\n", chisqi[i]);

}

Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int	nag_chi_sq_goodness_of_fit_test(
	int k2, // The number of clsses, k2, into which the data is divided.
	const int ifreq[], // ifreq[i-1] specifies the frequency of the ith class.
	const double cint[], // cint[i-1] specifies the upper boundary value for the ith class, cint[i]<cint[i+1].  For exponential, gamma, and chi^2 cint[0]>=0..
	Nag_Distributions dist, // The NAG distribution.
	const double par[], // par[] specifies the parameters of the distribution being tested. 
	int iparam, // The number of estimated parameters of the distribution.
	const double prob[], // If the user is supplying the probability distribution then prob[i-1] contains the probability that X lies in the ith class.
	double *chisq, // The returned test statistic, chi^2, for the chi^2 goodness of fit test.
	double *p, // The returned upper tail probability from the chi^2 distribution.
	int *ndf, // Returns the degrees of freedom associated with test.
	double eval[], // eval[i-1] contains the returned expected frequency for the ith class.  
	double chisqi[] // chisqi[i-1] contains the returned contribution from the ith class to the test statistic.
); // Performs the chi^2 goodness of fit test, for standard continuous distributions.


/**	g08eac
		Performs the runs up or runs down test on a sequence of observations.

Example:

void test_nag_runs_test()
{
	double chi, df, p;
	int i, init, max_run, n, nruns;
	n = 10000;
	double x[10000];
	init =0;
	
	for(i = 0; i < n; i++)
		x[i] = 	ran(init);		//generate random variable from 0 to 1.
	max_run = 6;
	nag_runs_test(n, x, max_run, &nruns, &chi, &df, &p);
	printf("\n");
	printf("Total number of runs found =%10ld\n" , nruns);
	printf("chisq = %10.4f\n",chi);
	printf("df    = %10,4f\n",df);
	printf("Prob  = %10.4f\n", p); 	
}

Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int	nag_runs_test(
	int n, // The length of the current sequence of observations, n>=3. 
	const double x[], // The sequence of observations. 
	int maxr, // The length of the longest run for which tabulation is desired, r. That is, all runs with length greater than or equal to r are counted together.
	int *nruns, // The returned number of runs actually found.  
	double *chi, // The returned approximate chi^2 test statistic.
	double *df, // The returned number of degrees of freedom of the chi^2 statistic.
	double *prob // The returned upper tail probability corresponding to the chi^2 test statistic (i.e. significance level).
); // Performs the runs up or runs down test for randomness. 



/**	g08ebc
		Performs the pairs (serial) test for randomness on a sequence of observations in the interval [0,1].


Example:

void test_nag_pairs_test()
{
	double chi, df, p;
	int i, max_count, n, init, lag;
	init = 0;
	n = 10000;
	double x[10000];
	for(i = 0; i < n; i++)
		x[i] = 	ran(init);		//generate random variable from 0 to 1.
	max_count = 10;
	lag =1;
	nag_pairs_test(n, x, max_count, lag, &chi, &df, &p);
	printf("\n\n");	
	printf("CHISQ	= %10.4f\n",chi);
	printf("DF		= %8.2f\n", df);
	printf("Probabitlity =%10.4f",p);	
}


Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/

int	nag_pairs_test(
	int n, // The number of observations, n>=2.
	const double x[], // The sequence of observations in the interval [0,1].
	int msize, // The size of the matrix of counts, m>=2.
	int lag, // The lag, l, to be used in choosing pairs.
	double *chi, // The returned chi^2 test statistic for testing the null hypothesis of randomness.
	double *df, // The returned degrees of freedom for the chi^2 test statistic. 
	double *p // The returned upper tail probability associated with the chi^2 test statistic (i.e. significance level).
); // Performs the pairs (serial) test for randomness.



/**	g08ecc
		Performs the triplets test for randomness on a sequence of observations from the interval [0,1]..

Example:

void test_nag_triplets_test()
{
	double chi, df, p;
	int i, init, max_count, n;
	n = 10000;
	double x[10000];
	for(i = 0; i < n; i++)
		x[i] = 	ran(init);		//generate random variable from 0 to 1.
	max_count = 5;
	nag_triplets_test(n, x, max_count, &chi, &df, &p);
	printf("\n");
	printf("CHISQ	= %10.4f\n",chi);
	printf("DF		= %8.2f\n", df);
	printf("Prob	=%10.4f",p);
}


Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int	nag_triplets_test(
	int n, // The number of observations, n>=3.
	const double x[], // The sequence of observations in the interval [0,1].
	int msize, // The size of the matrix of counts, m>=2.
	double *chi, // The returned chi^2 test statistic for testing the null hypothesis of randomness.
	double *df, // The returned degrees of freedom for the chi^2 test statistic. 
	double *p // The returned upper tail probability associated with the chi^2 test statistic (i.e. significance level).
); // Performs the triplets test for randomness.


/**	g08edc
		Performs the gaps test for randomness


Example:

void test_nag_gaps_test()
{
	int i, init, max_gap, n, num_gaps;
	double chi, df, lower, length, p, upper;
	init = 0;
	n = 5000;
	double x[5000];
	for(i = 0; i < n; i++)
		x[i] = 	ran(init);		//generate random variable from 0 to 1.
	num_gaps = 0;
	max_gap = 10;
	length = 1.0;
	lower = 0.4;
	upper = 0.6;
	nag_gaps_test(n, x, num_gaps, max_gap, lower, upper, length, &chi, &df, &p);
	printf("\n");
	printf("CHISQ	= %10.4f\n",chi);
	printf("DF		= %8.2f\n", df);
	printf("Prob	=%10.4f",p);
}


Parameters:

Return:
	This function returns NAG error code, 0 if no error.
	
*/
int	nag_gaps_test(
	int n, // The length of the current sequence of observations, n>=1.
	const double x[], // The sequence of observations.
	int m, // The maximum number of gaps to be sought. If m<=0 then there is no limit placed on the number of gaps that are found.
	int k, // The length of the longest gap for which tabulation is desired. 
	double rl, // The lower limit of the interval to be used to define the gaps (rl<ru and ru-rl<til).  
	double ru, // The upper limit of the interval to be used to define the gaps (ru>rl and ru-rl<til).
	double til, // The total lengeth of the interval which contains all possible numbers that may arise in the sequence (til>0 and ru-rl<til).
	double *chi, // The returned chi^2 test statistic for testing the null hypothesis of randomness.
	double *df, // The returned degrees of freedom for the chi^2 test statistic. 
	double *prob // The returned upper tail probability associated with the chi^2 test statistic (i.e. significance level). 
); // Performs the gaps test for randomness.

/* end proto */
#endif //!_O_NAG_G08_H