/*------------------------------------------------------------------------------*
 * File Name: OrgObj.h															*
 * Creation: CPY 1/7/2002													    *
 * Purpose: OriginObject, this the base class for all Origin internal objects	*
 * Copyright (c) OriginLab Corp.2001, 2002										*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
#ifndef _ORGOBJ_H
#define _ORGOBJ_H

/** >Internal Origin Objects
		The OriginObject class is the base class for all internal Origin objects and
		provides the methods and properties common to all Origin objects.
	Example:
		// If a worksheet is the active window in Origin than wks will be valid
		Worksheet wks;
		wks = (Worksheet) Project.ActiveLayer();
		if(wks.IsValid())
			out_str("Worksheet is Valid!");
		else
			out_str("Worksheet is not Valid!");
*/
class OriginObject
{
public:
	OriginObject();

	/**
			Checks Validity of the Origin object.
		Return:
			FALSE if an Origin Object is not valid, non-zero otherwise.
		Example:
			void test(Worksheet& aa)
			{
				if(aa.IsValid())
					out_str("Worksheet is Valid!");
				else
					out_str("Worksheet not Valid!");
			}
	*/
	BOOL		IsValid();

	/**
	DestroyObject
	Remark: 
		If the Origin object is a layer and it is the only one in the window, then also destroy the window (page).
	Example:
		Worksheet wks("Data1");
		ASSERT(wks.IsValid());
		wks.Destroy();
		ASSERT(!wks.IsValid());

	Return:
		TRUE for success
	*/
	BOOL		Destroy();

	/**
	Show or hide the Origin object
	Parameters: 
		bShow = optional argument (TRUE to show the object, FALSE - to hide)
	Return:
		TRUE for success, otherwise FALSE
	Example:
		Page page("Data1");
		page.Show(FALSE);	// hide worksheet page
	*/
	BOOL		Show(BOOL bShow = TRUE);

	/**
	Retrieve parent of origin object. 
	Parameters:
		obj = reference to object to which parent will be attached.

	Example:
		Column col("Data1", 0");
		Worksheet wks;
		col.GetParent(wks); // wks is worksheet to which column belongs.
	*/	
	void		GetParent(OriginObject& obj);		

#ifdef _OPERATION_H
	/**#
	*/	
	void		GetOperation(OperationBase& op);
#endif	//_OPERATION_H
	
#if _OC_VER > 0x0703
	/**
		Example:
			void Run_GetStorage()
			{
				Tree	tr;
				storage st;
				vector<string> vsName;	
				Page pg = Project.Pages();
				
				pg.GetStorageNames(vsName);
				
				st = pg.GetStorage("system");
				tr = st;
				out_tree(tr);
			}
	*/	
	storage		GetStorage(LPCSTR lpcszName, bool bAdd = FALSE);
	
	/**
		SeeAlso:
			GetStorage			
	*/	
	BOOL 		GetStorageNames(vector<string> &vsNames);

	
	/**
	Copy object format into Clipboard. Format from clipboard maybe saved to theme file. 
	Parameters:
		dwPropertiesFilter	= filter for properties. See FILTERPROPERTYBITS enumeration in OC_const.h for bits description
		dwObjFilter			= filter for objects. See FILTEROBJECTBITS enumeration in OC_const.h for bits description
	SeeAlso:
		CopyThemeFromClipboard

	Example:
		void Run_CopyFormat()
		{
			Page pg = Project.Pages();
			pg.CopyFormat();
		}
	*/	
	BOOL		CopyFormat(DWORD dwPropertiesFilter = FPB_ALL, DWORD dwObjFilter = FOB_ALL);

	/**
	Apply format stored in theme file to object
	Parameters:
		lpcszXMLPath = path of XML theme file
		bRepaint	 = if TRUE object will be repainted when format is applied
	
	Example:
		void Run_ApplyFormat()
		{
			Page pg = Project.Pages();
			pg.ApplyFormat("themes\\Ticks All In.OTH");
		}
	*/
	BOOL		ApplyFormat(LPCSTR lpcszXMLPath, BOOL bRepaint = TRUE);
	
	/**
	Get object format into Tree
	Parameters:
		dwPropertiesFilter	= filter for properties. See FILTERPROPERTYBITS enumeration in OC_const.h for bits description
		dwObjFilter			= filter for objects. See FILTEROBJECTBITS enumeration in OC_const.h for bits description
	Example:
	void Run_GetApplyFormat()
	{
		Page pg1 = Project.Pages(1);
		Page pg2 = Project.Pages(2);
		Tree tr;
		tr = pg1.GetFormat();
		pg2.ApplyFormat(tr);
	}
	*/
	Tree		GetFormat(DWORD dwPropertiesFilter = FPB_ALL, DWORD dwObjFilter = FOB_ALL);
	
	/**
	Apply format stored in tree to object
	Parameters:
		trNode		 = contains format information
		bRepaint	 = if TRUE object will be repainted when format is applied
	Example:
	void Run_GetApplyFormat()
	{
		Page pg1 = Project.Pages(1);
		Page pg2 = Project.Pages(2);
		Tree tr;
		tr = pg1.GetFormat();
		pg2.ApplyFormat(tr);
	}
	*/
	BOOL		ApplyFormat(const TreeNode& trNode, BOOL bRepaint = TRUE);

	/**
		get named binary storage into vector of bytes
	Paramteres
		lpcszName = storage name, must be valid C identifier
		vb = vector of bytes to be retrived
	Example:
		test_read()
		{
			Page pg = Project.Pages();
			if(!pg)
				return;
			
			vector<byte> vb;
			if(pg.GetMemory("Test", vb))
			{
				string strContents;
				if(strContents.SetBytes(vb))
					strContents.Write(WRITE_COMPILER_OUTPUT);
			}
		}
	Return:
		TRUE if success FALSE if name storage does not exist
	*/
	BOOL GetMemory(LPCSTR lpcszName, vector<byte>& vb);
	/**
		set/create named binary storage
	Paramteres
		lpcszName = storage name, must be valid C identifier
		vb = vector of bytes to be saved
	Example:
		string get_file(LPCSTR lpcszFilename)
		{
			string str;
			stdioFile ff;
			bool bRet = ff.Open(lpcszFilename, file::modeRead);
			if(!bRet)
			{
				out_str("file not found!");
				return str;
			}
			
			string strTemp;
			while(ff.ReadString(strTemp))
				str += strTemp + "\r\n";
			
			ff.Close();
			
			return str;
		}
		test_save()
		{
			Page pg = Project.Pages();
			if(!pg)
				return;
			
			string strFilename = GetAppPath() + "origin.ini";
			string strFileContent = get_file(strFilename);
			
			vector<byte> vb;
			if(strFileContent.GetBytes(vb) && pg.SetMemory("Test", vb))
				out_str("Saving origin.ini into page.info done");
		}
	Return:
		TRUE if success
	*/
	BOOL SetMemory(LPCSTR lpcszName, vector<byte>& vb);

	/**
		Show/Hide property of any Origin object
	Example:
		Worksheet wks("data1");
		wks.Columns(1).Show = 0;//hide column 2
		if(wks.Columns(1).Show == 0)
			out_str("Col(B) is hidden successfully");
	*/
	BOOL		Show;

	/**#
	Provides the name of the Origin object
	Parameters: 
	Return:
		Name of the Origin object.
	Example:
		Page page("Data1");
		string strWksName;
		strWksName = page.GetName();
	*/	
	string		GetName();
	
	/**
	Provides internal unique identification number of Origin object
	Paramteres:
		bCreate = create ID if it was not created yet
	Return:
		0 if ID for object was not created. Object unique ID otherwise
	Example:
		void Run_GetUID()
		{
			Page pg = Project.Pages();
			UINT uID = pg.GetUID(TRUE);
			printf("%u",uID);	
		}
	*/
	UINT		GetUID(BOOL bCreate = FALSE);
#endif //#if _OC_VER > 0x0703


};

#endif // _ORGOBJ_H



