/*------------------------------------------------------------------------------*
 *	File Name: 	MSMath.h	     												*
 *	Purpose:	Math function prototypes for calling standard C functions		*
 *				in msvcrt.dll and etc.											*
 *  Creation:	April 12, 2001 by CPY											* 
 *  	Copyright (c) OriginLab Corp. 2001, 2002, 2003							*
 *		All Rights Reserved														*
 *	Modification log        													*
 *------------------------------------------------------------------------------*/  
#ifndef _MSMATH_H
#define _MSMATH_H


// Origin C specific command to include an external DLL
// All function declarations will be for the importdll
// until another importdll is specified or when the end
// of the header file is reached
#pragma dll(msvcrt, system)

/** >Mathematical
		Absolute value (for integers). Note that this built-in C library abs function returns an
		int while the LabTalk abs function returns a double. The standard C library function fabs
		should typically used for doubles. To making it easier for Origin users, especially in defining
		fitting functions, an Origin C version of abs(double) has been added into internal.c, which
		is compiled automatically on Origin startup.
	Example:
		// This program computes and displays the absolute values of several numbers.
		#include <origin.h>
		int test_abs()
		{
			int nn;
			nn = abs( 0 );
			out_int("abs(0)=", nn);		//output should be abs(0)=0
			ASSERT( nn == 0 );
			nn = abs( -9 );
			out_int("abs(-9)=", nn);	//output should be abs(-9)=9
			ASSERT( nn == 9 );
			nn = abs( 200 );
			out_int("abs(200)=", nn);	//output should be abs(200)=200
			ASSERT( nn == 200 );
			nn = abs( -3.5 );
			out_int("abs(-3.5)=", nn);	//output should be abs(-3.5)=3
			ASSERT( nn == 3 );
			return 1;
		}
	Parameters:
		n=Integer value whose absolute value is returned
	Return:
		Returns the absolute value of an integer.
	SeeAlso:
		fabs
*/
int		abs(int n); // Returns the absolute value of an integer.

/** >Mathematical
		Calculates the arctangent of x.
	Example:
		This program computes and displays the arctangent of x.

		#include <origin.h>
		
		int test_atan()
		{
			double vv;
			vv = atan(1);
			out_double("atan(1)=", vv);		//output should be atan(1)=0.7854
			ASSERT( is_equal(vv, PI/4) );
			vv = atan(0);
			out_double("atan(0)=", vv);		//output should be atan(0)=0
			ASSERT( is_equal(vv, 0) );
			vv = atan(-1);
			out_double("atan(-1)=", vv);	//output should be atan(-1)=-0.7854
			ASSERT( is_equal(vv, -PI/4) );
			
			return 1;
		}
		
	Parameters:
		Any numbers.

	Return:
		The atan function returns the arctangent of x. 
*/
double	atan(double x);

/** >Mathematical
	Remarks:
		Calculates the arctangent of y/x.
		
	Example:
		This program computes and displays the arctangent of y/x.

		#include <origin.h>
		
		int test_atan2()
		{
			double vv;
			vv = atan2(2.5,2.5);
			out_double("atan2(2.5,2.5)=", vv);		//output should be atan2(2.5,2.5)=0.7854
			ASSERT( is_equal(vv, PI/4) );
			vv = atan2(0,0);
			out_double("atan2(0,0)=", vv);			//output should be atan2(0,0)=0
			ASSERT( is_equal(vv, 0) );				
			vv = atan2(0,180000);
			out_double("atan2(0,180000)=", vv);		//output should be atan2(0,180000)=0
			ASSERT( is_equal(vv, 0) );
			vv = atan2(0.0000023, 0);
			out_double("atan2(0.0000023, 0)=", vv);	//output should be atan2(0.0000023, 0)=1.5708
			ASSERT( is_equal(vv, PI/2) );
			vv = atan2(-1234567,-2345677);
			out_double("atan2(-1234567,-2345677)=", vv);	//output should be atan2(-1234567,-2345677)=-2.65711
			ASSERT( is_equal(round(vv, 5), -2.65711) );
			
			return 1;
		}
		
	Parameters:
		Any numbers.

	Return:
		The atan2 function returns the arctangent of y/x.  
*/
double	atan2(double y, double x);

/** >Mathematical
	Remarks:
		Calculate the cosine.
		
	Example:
		This program computes and displays the cosine of x.

		#include <origin.h>
		
		int test_cos()
		{
			double vv;
			vv = cos(-3*PI);
			out_double("cos(-3*PI)=", vv);	//output should be cos(-3*PI)=-1
			ASSERT( is_equal(vv, -1) );
			vv = cos(0);
			out_double("cos(0)=", vv);		//output should be cos(0)=1
			ASSERT( is_equal(vv, 1) );
			vv = cos(15.5);
			out_double("cos(15.5)=", vv);	//cos(15.5)=-0.97845
			ASSERT( is_equal(round(vv,5), -0.97845) );
			
			return 1;		
		}
		
	Parameters:
		Angle in radians.

	Return:
		The cos function returns the cosine of x.  
*/
double	cos(double x);

/** >Mathematical
	Remarks:
		Calculate sines.
		
	Example:
		This program computes and displays the sines of x.

		#include <origin.h>
		
		int test_sin()
		{
			double vv;
			vv = sin( 0 );
			out_double("sin(0)=", vv);		//output should be sin(0)=0
			ASSERT( is_equal(vv, 0) );
			vv = sin( PI/6);
			out_double("sin(PI/6)=", vv);	//output should be sin(PI/6)=0.5
			ASSERT(is_equal(round(vv,5), 0.5) );
			vv = sin(5.20);
			out_double("sin(5.20)=", vv);	//output should be sin(5.20)=-0.88345
			ASSERT(is_equal(round(vv,5), -0.88345) );
			
			return 1;
		}
		
	Parameters:
		Angle in radians.

	Return:
		The sin function returns the sines of x.  
*/
double	sin(double x);

/** >Mathematical
	Remarks:
		Calculate the hyperbolic tangent.
		
	Example:
		This program computes and displays the sines of x.

		#include <origin.h>
		
		int test_tanh()
		{
			double vv;
			vv = tanh(0);		//output should be tanh(0) =0
			out_double("tanh(0) =", vv);
			ASSERT( is_equal(vv, 0) );
			vv = tanh(4.567898);
			out_double("tanh(4.567898) =", vv);	//output should be tanh(4.567898) =0.99978
			ASSERT( is_equal(round(vv,5), 0.99978) );
			vv = tanh(-0.98765);
			out_double("tanh(-0.98765) =", vv);	//output should be tanh(-0.98765) =-0.75636
			ASSERT( is_equal(round(vv,5), -0.75636) );
			
			return 1;
		}
		
	Parameters:
		Angle in radians.

	Return:
		The tanh returns the hyperbolic tangent of x.   
*/
double	tanh(double x);

/** >Mathematical
	Remarks:
		Compute the Bessel function. 
		The _j0, _j1, and _jn routines return Bessel functions of the first kind: orders 0, 1, and n, respectively. 
		
	Example:
		This program illustrates Bessel functions.

		#include <origin.h>
		
		int test_J0()
		{
			double vv;
			
			vv = _j0( 0 );
			out_double( "_j0(0)=", vv );				//output should be _j0(0)=1
			ASSERT( is_equal(vv, 1) );
			
			vv = _j0(1.3);
			out_double( "_j0(1.3)=", vv );
			ASSERT( is_equal(round(vv,5), 0.62009) );	//output should be _j0(1.3)=0.62009
			ASSERT( is_equal(_j0(-2), _j0(2)) );
			
			return 1;
		}
		
	Parameters:
		x	Floating-point value 
		
	Return:
		The _j0 returns a Bessel function of x.    
*/
double	_j0(double x);			  

/** >Mathematical
	Remarks:
		Compute the Bessel function. 
		The _j0, _j1, and _jn routines return Bessel functions of the first kind: orders 0, 1, and n, respectively. 
		
	Example:
		This program illustrates Bessel functions.

		#include <origin.h>
		
		int test_J1()
		{
			double vv;
			
			vv = _j1( 0 );			
			out_double( "_j1(0)=", vv );			//output should be _j1(0)=0
			ASSERT( is_equal(vv, 0) );
			
			vv = _j1(1.3);			
			out_double( "_j1(1.3)=", vv );		    //output should be _j1(1.3)=0.52202
			ASSERT( is_equal(round(vv,5), 0.52202) );
			
			out_double( "_j1(2)=", _j1(2) );		//output should be _j1(2)=0.57672
			
			out_double( "_j1(-2)=", _j1(-2) );	    //output should be _j1(-2)=-0.57672	
			ASSERT( is_equal(_j1(-2), -_j1(2)) );
			
			return 1;
		}
		
	Parameters:
		x	Floating-point value 
	
	Return:
		The _j1 returns a Bessel function of x.    
*/
double	_j1(double x);

/** >Mathematical
	Remarks:
		Compute the Bessel function. 
		The _j0, _j1, and _jn routines return Bessel functions of the first kind: orders 0, 1, and n, respectively. 
		
	Example:
		This program illustrates Bessel functions.

		#include <origin.h>
		
		int test_Jn()
		{
			double vv;
			
			vv = _jn( 1,1 );		
			out_double("_jn(1,1)=", vv);			//output should be _jn(1,1)=0.44005
			ASSERT( is_equal( round(vv, 5), 0.44005) );
			
			vv = _jn( 0,0 );		
			out_double("_jn(0,0)=", vv);			//output should be _jn(0,0)=1
			ASSERT( is_equal(vv, 1) );
			
			vv = _jn(2,3.5);	
			out_double("_jn(2,3.5)=", vv);		    //output should be _jn(2,3.5)=0.45863
			ASSERT( is_equal(round(vv,5), 0.45863) );
			
			return 1;
		}
		
	Parameters:
		x	Floating-point value 
		n	Integer order of Bessel function 
		
	Return:
		The _jn returns a Bessel function of x.    
*/
double	_jn(int n, double x);
										 
/** >Mathematical
	Remarks:
		Compute the Bessel function. 
		The _y0, _y1, and _yn routines return Bessel functions of the second kind: orders 0, 1, and n, respectively.
		
	Example:
		This program illustrates Bessel functions.

		#include <origin.h>
		
		int test_Y0()
		{
			double vv;
			
			vv = _y0(0);
			out_double( "_y0(0)=", vv ); 		//output should be _y0(0)=--
			ASSERT( is_equal(vv, NANUM) );
			
			vv= _y0(11);
			out_double( "_y0(11)=", vv );		//output should be _y0(11)=-0.16885
			ASSERT( is_equal(round(vv, 5), -0.16885) );
			
			return 1;
		}
		
	Parameters:
		x	Floating-point value 
		
	Return:
		The _y0 returns a Bessel function of x.    
*/
double	_y0(double x);

/** >Mathematical
	Remarks:
		Compute the Bessel function. 
		The _y0, _y1, and _yn routines return Bessel functions of the second kind: orders 0, 1, and n, respectively.
		
	Example:
		This program illustrates Bessel functions.

		#include <origin.h>
		
		int test_Y1()
		{
			double vv;
			
			vv = _y1(0);
			out_double( "_y1(0)=", vv );	    //output should be _y1(0)=-- 
			ASSERT( is_equal(vv, NANUM) );
			
			vv= _y1(11);
			out_double( "_y1(11)=", vv );		//output should be _y1(11)=0.16371
			ASSERT( is_equal(round(vv, 5), 0.16371) );
			
			return 1;
		}
		
	Parameters:
		x	Floating-point value 
		
	Return:
		The _y1 returns a Bessel function of x.    
*/
double	_y1(double x);

/** >Mathematical
	Remarks:
		Compute the Bessel function. 
		The _y0, _y1, and _yn routines return Bessel functions of the second kind: orders 0, 1, and n, respectively.
		
	Example:
		This program illustrates Bessel functions.

		#include <origin.h>
		
		int test_Yn()
		{
			double vv;
			
			vv = _yn( 0,0 );
			out_double("_yn(0,0)=", vv);			//output should be _yn(0,0)=--
			ASSERT( is_equal(vv, NANUM) );
			
			vv = _yn( 1,1 );
			out_double("_yn(1,1)=", vv);			//output should be _yn(1,1)=-0.78121
			ASSERT( is_equal(round(vv, 5), -0.78121) );
			
			return 1;
		}
				
	Parameters:
		x	Floating-point value 
		n	Integer order of Bessel function 
		
	Return:
		The _yn returns a Bessel function of x.    
*/
double	_yn(int n, double x);
//------------------

// Maximum value that can be returned by the rand function.
#define RAND_MAX 0x7fff

/** >Mathematical
	Remarks:
		The rand function returns a pseudorandom integer in the range 0 to RAND_MAX. 
		
	Example:
		This program seeds the random-number generator.

		#include <origin.h>
		
		int test_rand()
		{
			int nn1, nn2;
			nn1 = rand();
			out_int("rand()=", nn1);
			nn2 = rand();
			out_int("rand()=", nn2);
			
			return 1;
		}
				
	Parameters:
		
	Return:
		The rand returns a pseudorandom number.    
*/
int     rand();

/** >Mathematical
	Remarks:
		The srand function sets the starting point for generating a series of pseudorandom integers.  
		
	Example:
		#include <origin.h>
		
		int test_srand()
		{
			int nn;
			srand(1);
			nn = rand();
			out_int("rand()=", nn);
			
			srand(1000);
			nn = rand();
			out_int("rand()=", nn);
			return 1;
		}
				
	Parameters:
		n 	a random starting point for random-number generation
		
	Return:
		None.  
*/
void    srand(unsigned int n);

#endif  //_MSMATH_H

