/*------------------------------------------------------------------------------*
 * File Name: AutomationExample.c												*
 * Creation: ER, 08/21/03														*
 * Purpose: OriginC Source C associated with AutomationExample Sample Project	*
 * Copyright (c) OriginLab Corp. 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010	*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
 
////////////////////////////////////////////////////////////////////////////////////
#include <Origin.h>
////////////////////////////////////////////////////////////////////////////////////

// The code in this file is associated with the sample project:
// 		Samples\Programming\Automation\AutomationExample.OPJ
// For an overview of what this example is all about, please see the Notes window
// in the above OPJ.
//
// The following function refered to by script code, which is contained in the import
// filter. This function is therefore called at the end of each file import. 
// This function performs the following post-processing of the imported data:
//			Retreives values from header variables created during import.
//			Creates new Project Explorer subfolder for each file that is opened.
//			Places imported data worksheet into PE subfolder.
//			Uses those values to fill first column of worksheet with wavelength value.
//			Creates a graph of amplitude versus wavelength data.
//			Performs NLSF curve fitting to graphed data.
//			Places fit results in labels in the graph window.
//			Exports graph to disk file, in BMP format.
//
void AutomationExample()
{
	// Get active page and declare worksheet page object
	WorksheetPage wpg = Project.Pages();
	
	// Delcare worksheet object and datasets from 1st two columns of the worksheet
	Worksheet wks(wpg.GetName());
	Dataset dsX(wks, 0);
	Dataset dsY(wks, 1);

	// Import places amplitude data in 1st column - copy this to 2nd column
	dsY = dsX;
	
	// Get info var values from header variables stored to page.info during import
	double dStartWavelength, dStepWavelength;
	page_get_info_var_value(wpg, "STARTWAVELENGTH", dStartWavelength);
	page_get_info_var_value(wpg, "STEPWAVELENGTH", dStepWavelength);
	string strSampleID;
	page_get_info_var_value(wpg, "SAMPLEID", strSampleID);
		
	// Fill column 1 with wavelength values
	for(int ii = 0; ii < dsX.GetSize(); ii++)
		dsX[ii] = dStartWavelength + ii * dStepWavelength;
	
	// Create a new PE subfolder and move this data worksheet to the new subfolder	
	// Declare root folder of PE for the current project
	Folder fldRootFolder = Project.RootFolder;
	// Create new subfolder with same name as Sample ID
	Folder fldSubFolder = fldRootFolder.AddSubfolder(strSampleID);
	// Make this subfolder active
	fldSubFolder.Activate();
	// Get path of this subfolder
	string strSubFolderPath = fldSubFolder.GetPath();
	// Move worksheet with imported data, into this subfolder
	fldRootFolder.Move(wpg.GetName(), strSubFolderPath);
	

	// Set up name of custom template to be used for creating a graph
	string strGraphTemplateName = LabTalk.System.Path.Program$ 
			+ "Samples\\Programming\\Automation\\AutomationExample.OTP";

	// Create a graph using custom template
	GraphPage grphData;
	int nOptionG = CREATE_VISIBLE_SAME;
	bool bRetG = grphData.Create(strGraphTemplateName, nOptionG);
	
	// Declare active layer in current graph page
	GraphLayer grphLayer = grphData.Layers();
	
	// Declare a curve object using x,y columns of worksheet
	Curve crvData(wks, 0, 1);

	// Get name of curve
	string strYDataName;						
	crvData.GetName(strYDataName);

	// Plot data curve to active layer
	int	nPlot = grphLayer.AddPlot(crvData, IDM_PLOT_SCATTER);
	grphLayer.Rescale();
	

	// Perform nonlinear fit to plotted data
	using NLSF = LabTalk.NLSF;						// access NLSF object properties directly
	NLSF.Init();									// initialize fitter
	NLSF.PasteToPlot = 0;							// turn off pasting results to graph
	NLSF.Func$ = "gaussamp";						// assign fitting function
	NLSF.FitData$ = strYDataName;					// assign fitting dataset
	NLSF.Execute("parainit");						// perform auto parameter initialization
	NLSF.Fit(100);									// perform fit
		
	// Write fit parameters to string variables
	string strCentroid, strHeight, strWidth;
	strCentroid.Format("%5.2f",NLSF.P2);			// Par 2, 3, 4 hold xc, w, A from fit
	strHeight.Format("%5.2f",NLSF.P4);
	strWidth.Format("%5.2f",NLSF.P3);
	
	// Declare graphic objects for text labels on graph layer
	GraphObject	grGTxtCentroid, grGTxtHeight, grGTxtWidth;	
	grGTxtCentroid = grphLayer.GraphObjects("txtCentroid");	
	grGTxtHeight = grphLayer.GraphObjects("txtHeight");
	grGTxtWidth = grphLayer.GraphObjects("txtWidth");

	// Write previously created string variables, with fit values, to these text labels	
	grGTxtCentroid.Text = "\u(Centroid:) " + strCentroid;
	grGTxtHeight.Text = "\u(Height:) " + strHeight;
	grGTxtWidth.Text = "\u(Width:) " + strWidth;		

	// Export the graph page to a bmp file
	// First build name for bmp file
	string strImageFileName = LabTalk.System.Path.Program$ 
			+ "Samples\\Programming\\Automation\\" + strSampleID + ".bmp";
	// Use LabTalk Image object
	using Image = LabTalk.Image;						// access Image object	
	Image.FileName$ = strImageFileName;					// set export file name
	Image.ShowOptions = 0;								// turn off options dialog
	Image.Export.PagePixel( "BMP", 640, 480, 24, 0 );	// perform export

	// Set active folder back to root folder, to be ready for next file 
	fldRootFolder.Activate();
}
// end of file