########################################################
#
#    3d Shapes Tool - Designed  May 2008 by Macs 
#
########################################################

# Copyright 2004-2005, @Last Software, Inc.
# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name        :  3d shapes tool 1.0
# Description :   A tool to create varius 3d shapes with the mouse.
# Menu Item   :   Draw->3D Shapes->...
# Context Menu:   NONE
# Date        :   8/05/2008
# Type        :   Tool
#Based on  : Rotated Rectangle Tool 1.0
#-----------------------------------------------------------------------------

require 'sketchup.rb'

#=============================================================================

class PyramidTool

@@ltcursor = 0

def initialize
    @ip = Sketchup::InputPoint.new
    @ip1 = Sketchup::InputPoint.new
	@planex = [Geom::Point3d.new(0,0,0), Geom::Vector3d.new(1,0,0)]
	@planey	= [Geom::Point3d.new(0,0,0), Geom::Vector3d.new(0,1,0)]
	@planez = [Geom::Point3d.new(0,0,0), Geom::Vector3d.new(0,0,1)]
    if @@ltcursor == 0
		plugins = Sketchup.find_support_file("Plugins")	
		imgdir = File.join(plugins, "3DShapesToolbar")
		c = File.join(imgdir, "3dshapes_cursor.png")
		@@ltcursor = UI::create_cursor(c, 1, 30)
    end
    reset
end

def reset
    @moved = false
    @pts = []
	@pir_pts 
    @state = 0
    @ip1.clear
    @drawn = false
    Sketchup::set_status_text "", SB_VCB_LABEL
    Sketchup::set_status_text "", SB_VCB_VALUE
    Sketchup::set_status_text "Click for start point"
    @shift_down_time = Time.now
end

def activate
    self.reset
end

def deactivate(view)
    view.invalidate if @drawn
end

#def resume(view)
#	view.invalidate
#end

def onSetCursor
	UI::set_cursor(@@ltcursor)
end


def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        @pts[4] = @pts[0]
        need_draw = @ip.display? || @drawn
    when 1
        @pts[1] = @ip.position
        @width = @pts[0].distance @pts[1]
        Sketchup::set_status_text @width.to_s, SB_VCB_VALUE
    when 2	    
        pt1 = @ip.position
        pt2 = pt1.project_to_line @pts
        vec = pt1 - pt2
        @deep = vec.length
        if( @deep > 0 )
            # test for a square
            square_point = pt2.offset(vec, @width)
            if( view.pick_helper.test_point(square_point, x, y) )
                @deep = @width
                @pts[2] = @pts[1].offset(vec, @deep)
                @pts[3] = @pts[0].offset(vec, @deep)
                view.tooltip = "Square"
            else
                @pts[2] = @pts[1].offset(vec)
                @pts[3] = @pts[0].offset(vec)
            end
			self.temp_apex(@pts, view)
        else
            @pts[2] = @pts[1]
            @pts[3] = @pts[0]
        end
        Sketchup::set_status_text @deep.to_s, SB_VCB_VALUE
	when 3
		@pir_pts = self.get_apex(view,@ip,@vect_perp,@center)
		@height = @pir_pts.distance @center
		Sketchup::set_status_text @height.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end


def onMouseMove(flags, x, y, view)
    @moved = true
    self.set_current_point(x, y, view)
end

def temp_apex( pts, view )
	@center = calculate_center(@pts)
	@height = view.pixels_to_model 50.cm, @ip.position
	vec1 = @pts[0] - @center
	vec2 = @center- @pts[1]
	@vect_perp = vec1.cross vec2		
	@pir_pts = @center.offset(@vect_perp, @height)
end

def calculate_center( pts )
	middlepoint = Geom::Point3d.linear_combination 0.5, pts[0], 0.5, pts[1]
	vec = pts[2] - pts[1]
	vec.length = vec.length / 2
	return middlepoint.offset(vec)
end

#get apex point
def get_apex(view, ip, vec, origin)
	if (ip.degrees_of_freedom != 3 )
		return ip.position.project_to_line([origin, vec])
	else
		@c_orig = view.screen_coords origin
		@c_ip = view.screen_coords ip.position
		@c_on_normal = view.screen_coords origin.offset( vec )
		@c_apex = @c_ip.project_to_line [@c_orig, @c_on_normal]
		a = Geom.closest_points [origin, vec], view.pickray(@c_apex.x, @c_apex.y)
		return a[0]
	end	
end

def create_geometry
    # check for zero deep
    if( @pts[0] != @pts[3] )
		model = Sketchup.active_model
		model.start_operation $exStrings.GetString("Pyramid")
        model.active_entities.add_face @pts
		for i in 0..3
			Sketchup.active_model.active_entities.add_face @pir_pts,@pts[i], @pts[i+1]
		end
		model.commit_operation
    end
    self.reset
end

def increment_state
	@moved = false
	@state += 1				
    case @state
    when 1
		@ip1.copy! @ip
		Sketchup::set_status_text "Click for second point"
		Sketchup::set_status_text "Length", SB_VCB_LABEL
		Sketchup::set_status_text "Move mouse", SB_VCB_VALUE
		@pts[1] = @pts[0]
    when 2
        @ip1.clear
        Sketchup::set_status_text "Click for third point"
        Sketchup::set_status_text "Width", SB_VCB_LABEL
        Sketchup::set_status_text "Move mouse", SB_VCB_VALUE
		@pts[2] = @pts[0]
		@pts[3] = @pts[0]
	when 3
        Sketchup::set_status_text "Click for fourth point"
        Sketchup::set_status_text "Height", SB_VCB_LABEL
        Sketchup::set_status_text "Move mouse", SB_VCB_VALUE	
    when 4
        self.create_geometry
    end

end

def onLButtonDown(flags, x, y, view)
    if @moved 
	    self.set_current_point(x, y, view)
	    self.increment_state
	    view.lock_inference
	end
end

def onCancel(flag, view)
    view.invalidate if @drawn
    self.reset
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
    
    case @state
    when 1
        # update the width
        vec = @pts[1] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[1] = @pts[0].offset(vec)
            view.invalidate
            self.increment_state
        end
    when 2
        # update the deep
        vec = @pts[3] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[2] = @pts[1].offset(vec)
            @pts[3] = @pts[0].offset(vec)
			temp_apex(@pts, view)
            self.increment_state
        end
	when 3
		@pir_pts = @center.offset(@vect_perp, value)
		self.increment_state
    end
end

def getExtents
    bb = Sketchup.active_model.bounds
    case @state
    when 0
        # We are getting the first point
        if( @ip.valid? )
            bb.add @ip.position
        end
    when 1
        bb.add @pts[0]
        bb.add @pts[1]
    when 2
        bb.add @pts
		bb.add @pir_pts
	when 3
		bb.add @pir_pts	
    end
    bb
end

def draw(view)
    @drawn = false
    
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end

    # show the rectangle
    if( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
        inference_locked = view.inference_locked?
        view.line_width = 3 if inference_locked
        view.draw(GL_LINE_STRIP, @pts[0], @pts[1])
        view.line_width = 1 if inference_locked
        @drawn = true
    elsif( @state > 1 )
        # draw the curve
        view.drawing_color = "black"
        view.draw(GL_LINE_STRIP, @pts)
		view.draw_lines @pir_pts, @pts[0], @pir_pts, @pts[1], @pir_pts, @pts[2], @pir_pts, @pts[3]
		view.drawing_color = "gray"		
		view.line_stipple="_"
		view.draw_lines(@pir_pts, @ip.position)		
        @drawn = true
    end
end

def onKeyDown(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY && rpt == 1 )
        @shift_down_time = Time.now
        
        # if we already have an inference lock, then unlock it
        if( view.inference_locked? )
            view.lock_inference
        elsif( @state == 0 )
            view.lock_inference @ip
        elsif( @state == 1 )
            view.lock_inference @ip, @ip1
        end
		
    end
end

def onKeyUp(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY &&
        view.inference_locked? &&
        (Time.now - @shift_down_time) > 0.5 )
        view.lock_inference
    end
end



end # of class PyramidTool

class ConeTool < PyramidTool

def reset
    @moved = false
    @pts = []
	@pir_pts 
    @state = 0
    @ip1.clear
    @drawn = false
	@sides = 24
    Sketchup::set_status_text "Sides > 2", SB_VCB_LABEL
    Sketchup::set_status_text @sides, SB_VCB_VALUE
    Sketchup::set_status_text "Click for center of cone"
    @shift_down_time = Time.now
end

def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        need_draw = @ip.valid? || @drawn
    when 1
        @pts[1] = @ip.position
        @radius = @pts[0].distance @pts[1]
        Sketchup::set_status_text @radius.to_s, SB_VCB_VALUE
    when 2
		@pts[2] = self.get_apex(view, @ip, @normal, @pts[0])
		@height = @pts[0].distance @pts[2]
        Sketchup::set_status_text @height.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end

def draw(view)

    @drawn = false   
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end

	return  if (!@ip.valid?) #this will eliminate the first draw on selecting the tool.

	
    # show the circle
	if( @state == 0 )
	    point = @ip.position
		@radius = view.pixels_to_model 50.cm, @ip.position
		@height = @radius*2 	
		if point.on_plane? @planey
			@normal = Geom::Vector3d.new(0,1,0)
			view.drawing_color = "green"
		elsif point.on_plane? @planex
			@normal = Geom::Vector3d.new(1,0,0)
			view.drawing_color = "red"			
		else 
			@normal = Geom::Vector3d.new(0,0,1)
			view.drawing_color = "blue"
		end
		face = @ip.face		
		if ( face )
		    @normal = face.normal
		end	
		pts_circle = self.points_on_circle( @ip.position,@normal, @radius, @sides)
		apex = point.offset(@normal, @height)
		view.draw_lines(point, apex)
		for i in 0..@sides
			view.draw_lines(apex, pts_circle[i])
		end
		view.draw(GL_LINE_STRIP, pts_circle)
		@drawn = true
    elsif( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
        #inference_locked = view.inference_locked?
        #view.line_width = 3 if inference_locked		
		#view.lock_inference @ip1
		vector = nil
		ptr = @ip.position		
		v1 = ptr - @pts[0]
		if (@normal == nil)
			ptr2 = ptr.clone
			ptr2.x = ptr.y
			ptr2.y = ptr.x			
			v2 = ptr2 - @pts[0]
			@normal = v1.cross v2
		else
			ptr = ptr.project_to_plane [@pts[0], @normal]
			v1 = ptr - @pts[0]
		end		
		pts_circle = self.points_on_circle( @pts[0], @normal, v1.length, @sides)	
		view.draw(GL_LINE_STRIP, @pts[0], ptr)
		apex = @pts[0].offset(@normal, @height)
		view.drawing_color = "blue"
        view.draw(GL_LINE_STRIP, pts_circle)
		for i in 0..@sides
			view.draw_lines(apex, pts_circle[i])
		end
        @drawn = true
    elsif( @state > 1 )
        # draw the curve
		view.drawing_color = "blue"		
        pts_circle = self.points_on_circle( @pts[0], @normal, @radius, @sides)		
		for i in 0..@sides
			view.draw_lines(@pts[2], pts_circle[i])
		end		
		view.draw(GL_LINE_STRIP, pts_circle)
		view.drawing_color = "gray"		
		view.line_stipple="_"
		view.draw_lines(@pts[2], @ip.position)
        @drawn = true
		#view.line_stipple=""
		#view.drawing_color = "red"		
		#view.draw2d GL_LINES, @c_orig, @c_ip
		#view.drawing_color = "green"
		#view.draw2d GL_LINES, @c_ip, @c_on_normal
		#view.drawing_color = "blue"
		#view.draw2d GL_LINES, @c_on_normal, @c_apex
		#view.drawing_color = "yellow"
		#view.draw2d GL_LINES, @c_apex, @c_orig
		
    end
end

def points_on_circle(center, normal, radius, numseg)
    # Get the x and y axes
    axes = Geom::Vector3d.new(normal).axes
    #center = Geom::Point3d.new(center)
    xaxis = axes[0]
    yaxis = axes[1]
    
    xaxis.length = radius
    yaxis.length = radius

    # compute the points
    da = (Math::PI * 2) / numseg
    pts = []
    for i in 0..numseg do
        angle = i * da
        cosa = Math.cos(angle)
        sina = Math.sin(angle)
        vec = Geom::Vector3d.linear_combination(cosa, xaxis, sina, yaxis)
        pts.push(center + vec)
    end
    
    # close the circle
    #pts.push(pts[0].clone)

    pts
end

def increment_state
	@moved = false
	@state += 1				
    case @state
    when 1
		@ip1.copy! @ip
		Sketchup::set_status_text "Click to set the radius"
		Sketchup::set_status_text "Radius", SB_VCB_LABEL
		Sketchup::set_status_text "Move mouse", SB_VCB_VALUE
    when 2
        @ip1.clear
        Sketchup::set_status_text "Click to set the height"
        Sketchup::set_status_text "Height", SB_VCB_LABEL
        Sketchup::set_status_text "Move mouse", SB_VCB_VALUE
    when 3
        self.create_geometry
    end

end

def getExtents
    bb = Sketchup.active_model.bounds
    case @state
    when 0
        # We are getting the first point
        if( @ip.valid? )
            bb.add @ip.position
        end
    when 1
        bb.add @pts[0]
		if (@pts[1] != nil)
			bb.add @pts[1]
		end
    when 2
        bb.add @pts
    end
    bb
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
	
    
    case @state
    when 0
		return if ( text.to_i < 3 )
		@sides = text.to_i
		view.invalidate
    when 1
	    # update the radius
		@radius = value
		view.invalidate		
        self.increment_state
    when 2
        # update the height
		@height = value
		@pts[2]= @pts[0].offset(@normal, @height)
		self.increment_state
    end
end

def create_geometry

	model = Sketchup.active_model
	if @sides > 8
		model.start_operation $exStrings.GetString("Cone")
	else 
		model.start_operation $exStrings.GetString("Pyramid")
	end
	

    # Create the base
	ent = model.active_entities 
    circle = ent.add_circle @pts[0], @normal, @radius, @sides
	numf = circle[0].find_faces #this actually creates a face when needed
	if ( numf == 1 && @normal != [0,0,1] )
		circle[0].faces[0].reverse!
	end
    e2 = []
	for i in 0...@sides
		e2[i] = ent.add_line(circle[i].start.position, @pts[2])
		if @sides > 8
			e2[i].soft = true
			e2[i].smooth = true
		end
	end
	for i in 0...@sides-1
        ent.add_face circle[i], e2[i], e2[i+1]
	end
	ent.add_face circle[@sides-1], e2[@sides-1], e2[0]
	model.commit_operation	
    self.reset
end





end # end ConeTool

class SphereTool < ConeTool


def reset
    @moved = false
    @pts = []
    @state = 0
    @ip1.clear
    @drawn = false
	@sides = 24
    Sketchup::set_status_text "Sides > 3", SB_VCB_LABEL
    Sketchup::set_status_text @sides, SB_VCB_VALUE
    Sketchup::set_status_text "Click for sphere's center"
    @shift_down_time = Time.now
end

def increment_state
	@moved = false
	@state += 1				
    case @state
    when 1
		@ip1.copy! @ip
		Sketchup::set_status_text "Click to set the radius"
		Sketchup::set_status_text "Radius", SB_VCB_LABEL
		Sketchup::set_status_text "Move mouse", SB_VCB_VALUE
    when 2
        self.create_geometry
    end

end

def draw(view)
    @drawn = false   
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end
	
	return if (!@ip.valid?)
		
    # show the circle
	if( @state == 0 )
	    point = @ip.position
		@radius = view.pixels_to_model 100.cm, @ip.position
		@height = @radius*2 	
		pts_circle = self.points_on_circle( point,Geom::Vector3d.new(0,0,1), @radius, @sides)
		view.draw(GL_LINE_STRIP, pts_circle)

		base = []
		top = []	
		for i in 0..@sides
			va = pts_circle[i] - point
			vdot = va.axes[0].reverse!
			pts_circle2 = self.points_on_semic( point,vdot, @radius, @sides)
			view.draw(GL_LINE_STRIP, pts_circle2)
			base.push(pts_circle2[0])
			top.push(pts_circle2.pop)

		end
		view.draw(GL_LINE_STRIP, top)
		view.draw(GL_LINE_STRIP, base)

		#pts_circle = self.points_on_circle( point,Geom::Vector3d.new(0,1,0), @radius, @sides)
		#view.draw(GL_LINE_STRIP, pts_circle)
		
		#pts_circle = self.points_on_circle( point,Geom::Vector3d.new(1,0,0), @radius, @sides)
		#view.draw(GL_LINE_STRIP, pts_circle)
		
		@drawn = true
    elsif( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
		ptr = @ip.position		
		return if (ptr == @pts[0])
		
		view.draw(GL_LINE_STRIP, @pts[0], ptr)
		view.drawing_color = "blue"		
		v1 = ptr - @pts[0]
		@radius = v1.length
		pts_circle = self.points_on_circle( @pts[0],Geom::Vector3d.new(0,0,1), @radius, @sides)
		view.draw(GL_LINE_STRIP, pts_circle)
		#view.drawing_color = "red"
		base = []
		top = []	
		for i in 0..@sides
			va = pts_circle[i] - @pts[0]
			vdot = va.axes[0].reverse!
			#pdot = @pts[0].offset(vdot, 200)
			#view.draw_lines(@pts[0], pdot )			
			pts_circle2 = self.points_on_semic( @pts[0],vdot, @radius, @sides)
			view.draw(GL_LINE_STRIP, pts_circle2)
			base.push(pts_circle2[0])
			top.push(pts_circle2.pop)
			#view.drawing_color = "green"			
		end
		view.draw(GL_LINE_STRIP, top)
		view.draw(GL_LINE_STRIP, base)
		@pts[1] = top[0]
		@pts[2] = base[0]
		#view.drawing_color = "red"
		#pts_circle = self.points_on_circle( @pts[0],Geom::Vector3d.new(0,1,0), @radius, @sides)
		#view.draw(GL_LINE_STRIP, pts_circle)
		
		
        @drawn = true
    end
end

def points_on_semic(center, normal, radius, numseg)
    # Get the x and y axes
    axes = Geom::Vector3d.new(normal).axes
    #center = Geom::Point3d.new(center)
    xaxis = axes[0]
    yaxis = axes[1]
    
    xaxis.length = radius
    yaxis.length = radius

	#numseg = numseg*2 if (numseg % 2 == 1)	
	
    # compute the points
    da = (Math::PI * 2) / (numseg)
    pts = []
    for i in -(numseg/4)..numseg/4 do
        angle = i * da
        cosa = Math.cos(angle)
        sina = Math.sin(angle)
        vec = Geom::Vector3d.linear_combination(cosa, xaxis, sina, yaxis)
        pts.push(center + vec)
    end
   
    pts
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
	
    
    case @state
    when 0
		return if ( text.to_i < 4 )
		@sides = text.to_i
		view.invalidate
    when 1
	    # update the radius
		@radius = value
		view.invalidate		
        self.increment_state
    end
end

def create_geometry

	model = Sketchup.active_model
	if @sides > 12
		model.start_operation $exStrings.GetString("Sphere")
	else 
		model.start_operation $exStrings.GetString("Polyhedron")
	end
	
    # Create the base
	ent = model.active_entities 
    pts_circle = self.points_on_circle( @pts[0],Geom::Vector3d.new(0,0,1), @radius, @sides)
	circle = ent.add_circle( @pts[0],Geom::Vector3d.new(0,0,1), @radius, @sides)

	base = []
	top = []
	pts_circle2 = []
	for i in 0..@sides
		va = pts_circle[i] - @pts[0]
		vdot = va.axes[0].reverse!
		pts_circle2[i] = self.points_on_semic( @pts[0],vdot, @radius, @sides)
		base.push(pts_circle2[i][0])
		top.push(pts_circle2[i][pts_circle2[i].size-1])
	end
	
	s = pts_circle2[i].size
	draw_top = draw_base = true
	for x in 0...(s-1)
		for i in 0...@sides		
			a = pts_circle2[i][x]
			b = pts_circle2[(i+1)%@sides][x]
			c = pts_circle2[(i+1)%@sides][(x+1)%s]
			d = pts_circle2[i][(x+1)%s]
			if ( a == b )
				face = ent.add_face a, c, d
				draw_top = false
			elsif ( c == d )
				face = ent.add_face a, b, d
				draw_base = false
			else
				face = ent.add_face a, b, c, d
			end
			if (@sides >= 12)
				ed = face.edges 
				ed[0].soft=true
				ed[1].soft=true
				ed[2].soft=true
				ed[0].smooth=true
				ed[1].smooth=true
				ed[2].smooth=true
			end
		end
	end	
	ent.add_face(top)  if draw_top 
	ent.add_face(base) if draw_base

	model.commit_operation	
    self.reset
end

def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        need_draw = @ip.valid? || @drawn
    end

    view.invalidate if need_draw
end

def getExtents
    bb = Sketchup.active_model.bounds
    case @state
    when 0
        # We are getting the first point
        if( @ip.valid? )
            bb.add @ip.position
        end
    when 1
        bb.add @pts
        if( @ip.valid? )
            bb.add @ip.position
        end
	end
    bb
end


end # SphereTool

#=============================================================================

# Add a menu choice
if( not $shapestool_menu_loaded )
    add_separator_to_menu("Draw")
	new_menu = UI.menu("Draw").add_submenu("3D Shapes")	
	new_toolbar = UI::Toolbar.new("3D Shapes Tool")
	plugins = Sketchup.find_support_file("Plugins")	
	imgdir = File.join(plugins, "3DShapesToolbar")

	cmd = UI::Command.new("Cone or Pyramid")  {Sketchup.active_model.select_tool ConeTool.new  }
	cmd.small_icon = cmd.large_icon = File.join(imgdir, "cone_icon.png")
	cmd.tooltip = "Cone or Pyramid"
	cmd.menu_text = "Cone or Pyramid"
	
    new_menu.add_item(cmd)
    new_toolbar.add_item(cmd)

	cmd = UI::Command.new("Sphere")  {Sketchup.active_model.select_tool SphereTool.new  }
	cmd.small_icon = cmd.large_icon = File.join(imgdir, "sphere_icon.png")
	cmd.tooltip = "Sphere"
	cmd.menu_text = "sphere"
	
    new_menu.add_item(cmd)
    new_toolbar.add_item(cmd)
	
	cmd = UI::Command.new("Rotated Pyramid")  {Sketchup.active_model.select_tool PyramidTool.new }
	cmd.small_icon = cmd.large_icon = File.join(imgdir, "rot_pyramid_icon.png")
	cmd.tooltip = "Rotated Pyramid"
	cmd.menu_text = "Rotated Pyramid"
	
	new_menu.add_item(cmd)	
	new_toolbar.add_item(cmd)	

	
	new_toolbar.show

	$shapestool_menu_loaded = true
end

