# Copyright 2005, TIG
#
# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided the above
# copyright notice appear in all copies.
#
# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
#-----------------------------------------------------------------------------
# Name        : ContourMaker
#
# Description : A tool to make contours
#
# Menu Item   : Plugins -> Contours
#
# Context Menu: None
#
# Author      : TIG
#
# Usage       : First select a Group or Component that has faces that could have 
#		Contours added, then use the Plugins Menu Item 'Contours'.
#
#		A dialog asks for the vertical spacing (Z-blue) in current units.  
#		The Contours always start their vertical spacing from Z=0.
#
#		It then makes a new layer for the contour-group.  The layer is 
#		named CONT-nnnnnnnn (where nnnnnnnn is based on the date/time).  
#
#		It then calculates a series of horizontal slices through the 
#		selection's faces to make the Contours as specified.  
#		
# Date        : 12/2005
#
# Type        : Tool
#
# Verison     :	1.0	17/12/05	First issue.
#		1.1	18/12/05	Circle def moved into Class.
#		1.2	19/12/05`Cutting disc amde very BIG for very large sites.
#
#-----------------------------------------------------------------------------

require 'sketchup.rb'

###

class Contours

###---------------------------------------

def Contours::calculate
 
model = Sketchup.active_model
entities = model.entities
model.start_operation "undo"
ss = model.selection
view = model.active_view
sfix = Time::now.to_i.to_s[3..-1]

  if ss.empty?
     UI.beep
     UI.messagebox("NO Selection !")
     return nil
  end#if
  if ss[1]
     UI.beep
     UI.messagebox("Selection MUST be ONE Group or Component !")
     return nil
  end#if
  if ss[0].typename != "Group" and ss[0].typename != "ComponentInstance"
     UI.beep
     UI.messagebox("Selection is NOT a Group or Component !")
  end#if

###
selected = ss[0]
###

### unselected...
model.selection.clear

# -------------- set up @spacing in def dialog for later...
# -------------- make cutting disc face at xy bounds
###
###
bb = selected.bounds
xmin = bb.min.x
ymin = bb.min.y
xmax = bb.max.x
ymax = bb.max.y
zmin = bb.min.z
zmax = bb.max.z
###
# ----------- dialog ----------------
### show VCB and status info
  Sketchup::set_status_text(("CONTOUR SPACING..." ), SB_PROMPT)
  Sketchup::set_status_text(" ", SB_VCB_LABEL)
  Sketchup::set_status_text(" ", SB_VCB_VALUE)
  return nil if not Contours.dialog
###
###
pnt1 = [xmin,ymin,0]
pnt2 = [xmax,ymax,0]
ctr = Geom.linear_combination(0.5,pnt1,0.5,pnt2)
rad = pnt1.distance pnt2 ### make a LOT bigger than bb
###
disc = entities.add_group
discentities = disc.entities
disccircle = Contours.circle(ctr,[0,0,-1],rad,24)
discentities.add_face disccircle
###
###
# ------------------------ do each of slice
contours = entities.add_group
contoursentities = contours.entities
contours.name= "Contours-"+sfix
###
### intersect disc and selected
###
### The intersect_with method is used to intersect entities, a component instance, or group
### with an entities object.
###
### entities.intersect_with recurse, transformation1, entities1, transformation2, hidden, entities2
###
### recurse - true if you want this entities object to be recursed 
###   (intersection lines will be put inside of groups and components within this entities object)
### transformation1 - the transformation for this entities object
### entities1 - the entities (group etc) where you want the intersection lines to appear
### transformation2 - the transformation for entities1
### hidden - true if you want hidden geometry in this entities object to be used in the intersection
###   (it's false in section cuts)
### entities2 - an entities object (or an array of entities ?)
###

### loop through all possible slices - bottom up
n = ((zmax-zmin)/@spacing).ceil
c = 0
z = 0
while z < zmax+@spacing
  
### show VCB and status info
  Sketchup::set_status_text(("MAKING CONTOURS - #{c} of #{n.to_i}" ), SB_PROMPT)
  Sketchup::set_status_text(" ", SB_VCB_LABEL)
  Sketchup::set_status_text(" ", SB_VCB_VALUE)
### do cut
  disc.move! Geom::Transformation.new [0,0,z] ### first placing near top
  discentities.intersect_with true, disc.transformation, contours, contours.transformation, true, selected
  z = z+@spacing
  c = c+1

end#while

### delete disc
disc.erase!

#----------- sort Layer settings etc
layercontours = ("CONT-" + sfix)
layercontours = model.layers.add(layercontours)
layercontours.page_behavior=(LAYER_IS_HIDDEN_ON_NEW_PAGES | LAYER_HIDDEN_BY_DEFAULT)
layercontours.visible=true
for f in contoursentities
  f.layer= "Layer0"
end#for f
contours.layer=layercontours
###

### update view
view.invalidate

# ---------------------- Close/commit group
model.commit_operation

#-----------------------
end#def

###################################################
def Contours::dialog

### get current units ----------------------
currentunits1=Sketchup.format_length 120000
currentunits2=Sketchup.format_length 1
units = "???" ### default
if currentunits1.to_s.split(".")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "120000"
   units = "inches"
end
if currentunits1.to_s.split(".")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "10000"
   units = "feet"
end
if currentunits1.to_s.split(".")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "3048000"
   units = "mm"
end
if currentunits1.to_s.split(".")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "304800"
   units = "cm"
end
if currentunits1.to_s.split(".")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "3048"
   units = "m"
end
if currentunits2 == "1\""
   units = "inches or ' \" "
end
### ---------------
if units[-1,1] == "m" ### metric units so these are in metres
         xspacing = 1.0.m
      else ### not metric so these are in feet
         xspacing = 5.0.feet
      end #if
### get spacing
   prompts = ["Spacing ("+units+"): "]
   types = [""]
   title = "Contours Spacing"
   @spacing = xspacing if not @spacing
   values = [@spacing]
   popups = [""]
   results = inputbox( prompts, values, popups, title )
   return nil if not results

### do processing of results
@spacing = results[0]

true

end #def dialog
###########################
# --- Function for generating points on a circle
def Contours::circle(center,normal,radius,numseg)
    # Get the x and y axes
    axes = Geom::Vector3d.new(normal).axes
    center = Geom::Point3d.new(center)
    xaxis = axes[0]
    yaxis = axes[1]
    xaxis.length = radius
    yaxis.length = radius
    # compute the points
    da = (Math::PI * 2) / numseg
    pts = []
    for i in 0...numseg do
        angle = i * da
        cosa = Math.cos(angle)
        sina = Math.sin(angle)
        vec = Geom::Vector3d.linear_combination(cosa,xaxis,sina,yaxis)
        pts.push(center + vec)
    end
    # close the circle
    pts.push(pts[0].clone)
    pts
end#def
##################

#-----------------------
end#class
#-----------------------


#--------- menu -----------------------------
if( not file_loaded?("ContourMaker.rb") )
###if( not file_loaded?("18-ContourMaker.rb") )

   add_separator_to_menu("Plugins")
   UI.menu("Plugins").add_item("Contours") { Contours.calculate }
   ###$submenu1.add_item("Contours [on Faces]") { Contours.calculate }

end#if
###file_loaded("18-ContourMaker.rb")
file_loaded("ContourMaker.rb")
#---------------------------------

