###-------------------------------------------------------------------------------------------
#
#	 Name : MatrixProximity.rb
#
# Description : Provides 2 Component Copy/Manipulation Tools 'Matrix' & 'Proximity' 
#
#      Author : TIG (c) 11/2005
#
#       Usage : 
#		Matrix:
#
#		This tool makes multiple copis of selected Component to set rules:
#		Select an instance of a Component - then pick 'Matrix' off the menu.
#		The first dialog has 5 options each with 3 sub-options...
#		X Y Z (red/green/blue) spacings for the copies.
#		X Y Z numbers of copies, the X Y Z +/- % randomizing of spacing,
#		% rotation & % scaling of each copy.  Pick OK to make copies.
#
#		If it's a 2D group (Z=0) a 2nd dialog asks if you want the copies 
#		to drop down onto Faces - Yes to drop, No to leave at same Z.
#		This is useful for dropping the copies onto a 3D ground surface.
#		The drop is always onto the closest Face, if there's no down-face
#		then a message tells you how many were missed.  To ensure they all 
#		drop onto the desired surfaces make the matrix copies well above.
#
#		Choose different combinations of X Y Z settings you get different
#		results.  Play around to see what effects are possible...
#		Giving a spacing to only one of the X Y Z gives a linear array
#		(e.g. use for lines of trees, posts etc)
#		leaving only the Z spacing at 0 gives a 2D array of objects
#		(e.g. use for groups of trees, people etc)
#		randomizing % spacing gives a more natural feeling to groups of
#		people, trees etc, randomizing % Z rotation also assists with this -
#		% rotation with XY gives strange but sometimes useful results,
#		randomizing % scaling (in Z) can also give a less formal appearance -
#		setting a small % scaling (default = 1) in XY makes half of copies 
#		handed - setting at 0 gives no handing - note there's no 'Z handing'.
#		The defaults assume no copies in the Z - i.e. a 2D group matrix...
#		The random % settings are as a range in proportion to the whole -
#		so % spacing is based on distances entered, > 50% and the copies 
#		might sometimes clash with each other as they jostle around; 
#		% rotation is likely to be best at 100% (0 to 360 degrees) and 
#		usually only in Z; % scaling is based on original sizes - so 33% 
#		gives +/- 33% on original sizes, > 100% will cause some flipped
#		copies, which also won't often drop properly, so avoid them unless 
#		you want that effect...
#
#		If you have used a 'Proximity Component' (e.g. tree-000-) with the 
#		Matrix Tool AND there are some other'family members' available 
#		(e.g. tree-001-, tree-002- etc) either the model or in the 'Proximity 
#		Components' folder in the 'Components' folder, then there'll be a
#		3rd dialog telling you that they are a available and asking if you 
#		want to randomly scatter these into the Matrix you have just made -
#		it's useful to give crowds or groups of trees a natural look...
#
#		One Undo reverts to previous model, if there have been 'family'
#		scatterings then it's two Undos - first to remove scattering and
#		second to revert as normal.
#
#
#		Proximity:
#
#		This tool sets level of detail in special Components depending on their 
#		'proximity' (nearness) to the camera/viewer in the current page/view.
#		It requires use of special 'Proximity Components' in the model or
#		in the 'Proximity Components' folder within the 'Components' folder.
#		These consists of a trio, they must share the same name prefix
#		(e.g. tree-), with their suffix ending ( - ~ + ) to tell which to use
#		at a proximity, each containing more detail the closer they are - 
#		e.g. tree-001-, tree-001~ and tree-001+, are respectively for use far 
#		away, mid distance and close to - and would typically be a simple 
#		2D face-me tree, a simple 3D tree and a detailed 3D tree with branches, 
#		leaves etc - obviously the three should also look similar in form, 
#		origin and size.  There are some standard trio-sets supplied, but making 
#		your own is pretty simple.  Make one type and then in two new models use 
#		it as a template to get the other two types appropriately located and 
#		sized etc.  Remember to purge unused layers, blocks and materials from
#		the component models to avoid confusion and unnecessary data transfer.
#
#		Proximity Components can also belong to 'families' - 
#		e.g. tree-001-/...~/...+, tree-002-/...~/...+ etc, these are useful 
#		when large groupings are made with the Matrix Tool as a % substitution 
#		from the family is possible. Organise your sub-set families logically, 
#		e.g. tree-..., conifer-..., shrub-..., people-..., car-... etc
#		so you don't get unexpected pine trees in your deciduous forest !
#
#		To use 'Proximity' pick it off the menu and in the resulting dialog set 
#		the distance ranges for Proximity Components from the camera for the 
#		Near Range (+) and Mid Range (~) - e.g. (assuming your units are metres) 
#		if you want the Near to versions within 10m for the '+' enter 10, and the 
#		Mid Range from 10m to 25m, for the '-' enter 25. If you only want the 
#		simple 'Far' versions enter 0 for both + and ~ (this is useful if you 
#		want to pan zoom acomplex model).  The Near value entered can't be larger
#		than the Mid, so Mid is changed to equal Near if you enter them wrongly, 
#		and a dialog tells you so...
#
#		Proximity then calculates the distance between the Camera and each placed 
#		Proximity Component.  Depending on the distance, their type (+/~/_) and 
#		the Near and Mid Ranges chosen these Proximity Components are automatically 
#		swapped around with either the less or more complex ones in the trio.  
#		Thus only the most complex components are placed near to the camera where 
#		their detail is required, whilst further away they will get simpler - this 
#		optimises the 'poly-count' and speeds up model manipulation etc.  If a 
#		particular Proximity Component has the required trio counterpart in the
#		model that will be used in the swap, however if the trio counterpart is not
#		loaded but it is available in the Proximity Components folder then it will
#		be automatically loaded and swapped to suit.  In this way you can model 
#		with a relatively light set of components (with the simple Far (-) versions) 
#		and only later when the view is set use Proximity to swap them over 
#		automatically with their trio counterparts.
#
#		One Undo reverts to previous model.
#
#		The distribution of Proximity Components depends on the Camera position, 
#		so if you have several Pages you'll need to run Proximity for each one in 
#		turn as you make images - unfortunately this then excludes movie making with
#		levels iof detail - however you can use Proximity to reset to a Far or Mid
#		components for the whole model - set Near/Mid to 0 for All = Far, or Near
#		to 0 and Mid to a very large distance for All = Mid.
#
#		As you make more trios of Proximity Components save them into the Proximity
#		Components folder in the Components folder - and share them...
#
#
#        Type : Tool
#
#     Version : 1.0  21/11/05	First release.
#		1.1  23/11/05	Math::PI sub for PI T'Bd on the safe side.
#				Both defs and all @vars made 'distinct'.
#				Current Units detected and reported in dialogs and
#				dialog wording improved.
#		1.2  26/11/05	Occasional glitch with rotation and scaling at 0 fixed.
#		1.3  03/12/05	Class structure added, $vars --> @vars
#		1.4  06/12/05	Fatal typo fixed in menu section !
#		1.5  19/12/05	Decimal . and , now allowed.
#		1.6  20/12/05 Ditto - corrected...
#   1.7  09/02/08 Glitch with Proximity Components directory finding 
#                 now fixed (Thanks to GreyHead)
#
###--------------------------------------------------------------------------------------------

require 'sketchup.rb'

### #################################################################

class Matrixproximity

def Matrixproximity::matrix

   model = Sketchup.active_model
   model.start_operation "matrix"
   entities = model.entities
   ss = model.selection

###
   if ss.empty?
      Sketchup::set_status_text("Matrix... NO SELECTION ! ", SB_PROMPT)
      UI.messagebox("No Selection ! ")
      return nil
   end
###
   if ss.length > 1
      Sketchup::set_status_text("Matrix... MULTIPLE SELECTION ! ", SB_PROMPT)
      UI.messagebox("Multiple Selection !\nSelect just ONE Component Instance ! ")
      return nil
   end
###
   if ss[0].typename != "ComponentInstance"
      Sketchup::set_status_text("Matrix... SELECTION IS NOT A COMPONENT ! ", SB_PROMPT)
      UI.messagebox("Select just ONE Component Instance ! ")
      return nil
   end

### get component instance details
compo = ss[0]
cdefn = compo.definition
cname = cdefn.name
cinsert = cdefn.insertion_point
ctrans = compo.transformation
cpoint = cinsert.transform ctrans
compx = cpoint.x
compy = cpoint.y
compz = cpoint.z

### get current units
currentunits1=Sketchup.format_length 120000
currentunits2=Sketchup.format_length 1
units = "???" ### default
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "120000"
   units = "inches"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "10000"
   units = "feet"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "3048000"
   units = "mm"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "304800"
   units = "cm"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "3048"
   units = "m"
end
if currentunits2 == "1\""
   units = "inches or ' \" "
end

### show VCB and status info
   Sketchup::set_status_text(("Matrix... PARAMETERS (" + units + ")..." ), SB_PROMPT)
   Sketchup::set_status_text(" ", SB_VCB_LABEL)
   Sketchup::set_status_text(" ", SB_VCB_VALUE)

### defaults and 1st dialog
   if not @xspace
      if units[-1,1] == "m" ### metric units so these are in metres
         @xspace = 10.0.m
         @yspace = 10.0.m
         @zspace = 0.0.m
      else ### not metric so these are in feet
         @xspace = 30.0.feet
         @yspace = 30.0.feet
         @zspace = 0.0.feet
      end #if
      @xcopies = 9
      @ycopies = 9
      @zcopies = 0
      @xrandom = 33.0
      @yrandom = 33.0
      @zrandom = 0.0
      @xrotate = 0.0
      @yrotate = 0.0
      @zrotate = 100.0
      @xscale = 1.0
      @yscale = 1.0
      @zscale = 33.0
   end
   prompts = [ ("X Spacing (" + units + ") :  "), ("Y Spacing (" + units + ") :  "), ("Z Spacing (" + units + ") :  "), "X Number of Copies :  ", "Y Number of Copies :  ", "Z Number of Copies :  ", "X Randomized Spacing % :   ", "Y Randomized Spacing % :   ", "Z Randomized Spacing % :   ", "X Randomized Rotation % :   ", "Y Randomized Rotation % :   ", "Z Randomized Rotation % :   ", "X Randomized Scaling % :   ", "Y Randomized Scaling % :   ", "Z Randomized Scaling % :   " ]
   value =   [ @xspace, @yspace, @zspace, @xcopies, @ycopies, @zcopies, @xrandom, @yrandom, @zrandom, @xrotate, @yrotate, @zrotate, @xscale, @yscale, @zscale ]
   results = inputbox prompts, value, ("Matrix Parameters (" + units + ")")
   return if not results
   @xspace = results[0]
   @yspace= results[1]
   @zspace= results[2]
   @xcopies = results[3]
   @ycopies = results[4]
   @zcopies = results[5]
   @xrandom = results[6]
   @yrandom = results[7]
   @zrandom = results[8]
   @xrotate = results[9]
   @yrotate = results[10]
   @zrotate = results[11]
   @xscale = results[12]
   @yscale = results[13]
   @zscale = results[14]

if @xspace == 0.0.m
  @xcopies = 0
end
if @yspace == 0.0.m
  @ycopies = 0
end
if @zspace == 0.0.m
  @zcopies = 0
end
if @xcopies == 0
  @xspace = 0.0.m
end
if @ycopies == 0
  @yspace = 0.0.m
end
if @zcopies == 0
  @zspace = 0.0.m
end

### loop through to make copies
copies = []

x = 0.0

0.upto(@xcopies) do |this|

 y = 0.0

 0.upto(@ycopies) do |that|

  z = 0.0

  0.upto(@zcopies) do |the_other|

   if rand > 0.5
      d = 1
   else
      d = -1
   end
   xx = (compx + x) + (@xspace * rand * d * @xrandom / 100)
   yy = (compy + y) + (@yspace * rand * d * @yrandom / 100)
   zz = (compz + z) + (@zspace * rand * d * @zrandom / 100)
   point = Geom::Point3d.new xx,yy,zz
   transform = Geom::Transformation.new point
   e = entities.add_instance cdefn,transform

   z = z + @zspace

   if rand > 0.5 ### +/- rotations
      d = 1
   else
      d = -1
   end
   p = Geom::Point3d.new cinsert.transform e.transformation
   v = Geom::Vector3d.new 1,0,0
   r = Math::PI * (2 * @xrotate / 100 * rand * d ) ###v1.2
   rotate = Geom::Transformation.rotation p,v,r
    e.transform! rotate

   v = Geom::Vector3d.new 0,1,0
   r = Math::PI * (2 * @yrotate / 100 * rand * d ) ###v1.2
   rotate = Geom::Transformation.rotation p,v,r
   e.transform! rotate

   v = Geom::Vector3d.new 0,0,1
   r = Math::PI * (2 * @zrotate / 100 * rand * d ) ###v1.2
   rotate = Geom::Transformation.rotation p,v,r
   e.transform! rotate


   dx = 1 
   dy = 1 
   dz = 1 
   if rand > 0.5 and @xscale != 0 ### +/- handing  ### v1.2
      dx = -1
   end
   if rand > 0.5 and @yscale != 0 ### +/- handing ### v1.2
      dy = -1
   end
   if rand > 0.5 and @zscale != 0 ### +/- handing ### v1.2
      dz = -1
   end
   sx = (1 + (@xscale / 100 * rand)) * dx ### v1.2
   sy = (1 + (@yscale / 100 * rand)) * dy ### v1.2
   sz = (1 + (@zscale / 100 * rand))### we don't normally want things flipped in Z !!!  * dz ### v1.2
   scaler = Geom::Transformation.scaling p,sx,sy,sz
   e.transform! scaler

   copies = copies.push(e)

  end #upto

  y = y + @yspace

 end #upto

 x = x + @xspace

end #upto

### remove 'spare' component
compo.erase!

### snap onto ground
fail = 0
if @zspace == 0
   ### show VCB and status info
   Sketchup::set_status_text("Matrix... DROP ONTO NEAREST FACES ? ...", SB_PROMPT) 
   Sketchup::set_status_text(" ", SB_VCB_LABEL)
   Sketchup::set_status_text(" ", SB_VCB_VALUE)
   yn = (UI.messagebox("Drop onto Nearest Faces ?", MB_YESNO)) 
   if yn == 6
      copies.each do |e|
         position = e.transformation.origin
         rt = model.raytest(position,[0,0,-1]) ### try moving it down
         if rt != nil
            pt = rt[0] 
            level = pt[2]
            position = e.transformation.to_a
            position[14] = level
            e.transformation= position
         else
            fail = fail + 1 ### else it doesn't move at all
         end
         model.active_view.invalidate
      end #each
   end #if
   if fail > 0
      ### show VCB and status info
      Sketchup::set_status_text("Matrix... DROP FAILURE...", SB_PROMPT) 
      Sketchup::set_status_text(" ", SB_VCB_LABEL)
      Sketchup::set_status_text(" ", SB_VCB_VALUE)
      UI.messagebox("#{fail} NOT Dropped:\nNO Face below ! ") 
   end
end #if

### check proximity family - ~ + at end
if cname[-1,1] == "-" or cname[-1,1] == "~" or cname[-1,1] == "+"
   ### look within model for others in family group
   cprefix = cname.split("-")[0]
   csuffix = cname[-1,1]
   ### list every ComponentDefinition in model
   definitions = model.definitions
   cdefs = []
   cnames = []
   0.upto(definitions.length - 1) do |c|
      if not definitions[c].group?
         cdefs = cdefs.push(definitions[c])
         cnames = cnames.push(definitions[c].name)
      end
   end #upto
   ### limit components to only the possibles
   cposs = []
   0.upto(cnames.length - 1) do |n|
      if cnames[n].split("-")[0] == cprefix and cnames[n][-1,1] == csuffix
         cposs = cposs.push(cdefs[n])
      end
   end #upto

### ditto for other proximity components in prox comps folder
   proxlist = []
   proxdir1 = (Sketchup.find_support_file "","Components/Proximity Components/")###v1.7
   proxdir=proxdir1+"/Components/Proximity Components/"###v1.7
   proxlist = Dir.entries(proxdir)
   skplist = [] ### list of prox's skp files
   0.upto(proxlist.length - 1) do |f|
      if (proxlist[f].split("-")[0] == cprefix) and (proxlist[f][-5,5] == (csuffix + ".skp"))
         rep = 0
         0.upto(cposs.length - 1) do |c|
            if cposs[c].name == proxlist[f][0..-5] ### - ".skp"
               rep = 1
            end
         end #upto
         if rep == 0  
            skplist = skplist.push(proxlist[f])
         end
      end
   end #upto
   cs = cposs + skplist
   cslength = cs.length
   if cslength > 1 ### i.e. there's more than the original compo...
      ### show VCB and status info
      Sketchup::set_status_text("Matrix... FAMILY COMPONENTS AVAILABLE TO SCATTER WITHIN MATRIX COPIES...", SB_PROMPT)
      Sketchup::set_status_text(" ", SB_VCB_LABEL)
      Sketchup::set_status_text(" ", SB_VCB_VALUE)
      yn = (UI.messagebox("There are #{cslength-1} other 'Proximity' Family Components  (#{cprefix}-***#{csuffix}) available. \nDo you want them scattering randomly within this Matrix ?  ", MB_YESNO))
      if yn == 6
         ### those with .skp ending NOT already in drawing
         0.upto(skplist.length - 1) do |s|
            cc = definitions.load (proxdir + skplist[s])
            cs = cposs.push(cc)
         end #upto
         ###
         model.start_operation "swap"
         ### swap random components in approx. equal %
         for c in copies
            nr = (rand * cslength).floor
            c.definition= cs[nr]
         end #for
         model.active_view.invalidate
         model.commit_operation ### swap undo
      end #if
   end #if
end #if proximity check

### commit for total undo
   model.commit_operation

end #def



### #################################################################

def Matrixproximity::proximity 

   model = Sketchup.active_model
   entities = model.entities
   view = model.active_view
   camera = view.camera
   eye = camera.eye

### get current units
currentunits1=Sketchup.format_length 120000
currentunits2=Sketchup.format_length 1
units = "???" ### default
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "120000"
   units = "inches"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "10000"
   units = "feet"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "3048000"
   units = "mm"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "304800"
   units = "cm"
end
if currentunits1.to_s.split(".")[0].split(",")[0].split("'")[0].split("\"")[0].split("c")[0].split("m")[0] == "3048"
   units = "m"
end
if currentunits2 == "1\""
   units = "inches or ' \" "
end

### show VCB and status info
   Sketchup::set_status_text(("Proximity... PARAMETERS ( in " + units + " )..." ), SB_PROMPT)
   Sketchup::set_status_text(" ", SB_VCB_LABEL)
   Sketchup::set_status_text(" ", SB_VCB_VALUE)

### do dialog
if not @proximity_Near
   if units[-1,1] == "m" ### metric units so these are in metres
      @proximity_Near = 10.0.m
      @proximity_Mid = 50.0.m
   else ### not metric so these are in feet
      @proximity_Near = 30.0.feet
      @proximity_Mid = 150.0.feet
   end #if
end #if
   prompts = [ ("Near ( + ) Range, from 0 to  :   "), ("Mid ( ~ ) Range, from 'Near' to  :   ") ]
   value =   [ @proximity_Near, @proximity_Mid ]
   results = inputbox prompts, value, ("Proximity Parameters ( in " + units + " )")
   return if not results
   @proximity_Near = results[0]
   @proximity_Mid = results[1]

### check relative distances...
   if @proximity_Near > @proximity_Mid
      @proximity_Mid = @proximity_Near
      ### show VCB and status info
      Sketchup::set_status_text("Proximity... ADJUSTING ENTERED PARAMETERS...", SB_PROMPT)
      Sketchup::set_status_text(" ", SB_VCB_LABEL)
      Sketchup::set_status_text(" ", SB_VCB_VALUE)
      UI.messagebox("Near (+) Range can't be larger than Mid (~) Range ! \nSo Mid (~) Range reset to equal Near (+) Range... ")
   end #if

### look within model for all prox comps even if not used
   ### list every ComponentDefinition in model
   definitions = model.definitions
   proxdefs = []
   0.upto(definitions.length - 1) do |c|
      cname = definitions[c].name
      if not definitions[c].group?
         if (cname[-1,1] == "-" or cname[-1,1] == "~" or cname[-1,1] == "+")
            proxdefs = proxdefs.push(definitions[c])
         end
      end
   end #upto
### now have a list of all prox comps in model's browser
### ditto for other proximity components in prox comps folder
   proxlist = []
   proxdir1 = (Sketchup.find_support_file "","Components/Proximity Components/")###v1.7
   proxdir=proxdir1+"/Components/Proximity Components/"###v1.7
   proxlist = Dir.entries(proxdir)
   skplist = [] ### list of prox's skp files
   0.upto(proxlist.length - 1) do |f|
      if proxlist[f][-4,4] == ".skp"
         rep = 0
         0.upto(proxdefs.length - 1) do |c|
            if proxdefs[c].name == proxlist[f][0..-5] ### - ".skp"
               rep = 1
            end
         end #upto
         if rep == 0  
            skplist = skplist.push(proxlist[f])
         end
      end
   end #upto
### now have model prox comp defs and available prox comp folder prox coms as lists
### proxdefs and skplist

### get all proxcomps in model
   proxcomps = []
   for e in entities
      etype = e.typename
      if (etype == "ComponentInstance")
         ename = e.definition.name
         if (ename[-1,1] == "-" or ename[-1,1] == "~" or ename[-1,1] == "+")
            proxcomps = proxcomps.push(e)
         end #if
      end #if
   end #for

### all are now listed in proxcomps so we process them... ###########

cswaps = []
   ###for c in proxcomps
0.upto(proxcomps.length - 1) do |c|
   ### get info...
cdefinition = nil
cdefn = proxcomps[c].definition
cname = cdefn.name
cinsert = cdefn.insertion_point
ctrans = proxcomps[c].transformation
cpoint = cinsert.transform ctrans
disteye = eye.distance cpoint

### 1st we check if already loaded...
### it's near but NOT a near (+) component
if (disteye <= @proximity_Near) and (cname[-1,1] != "+")
   ### now check if available in proxdefs
   0.upto(proxdefs.length - 1) do |d|
      if proxdefs[d].name == (cname[0..-2] + "+") ## end off and replaced - available
         cdefinition = proxdefs[d]
      end
   end #upto
end #if
### it's mid but NOT a mid (~) component
if (disteye <= @proximity_Mid) and (disteye > @proximity_Near) and (cname[-1,1] != "~")
   ### now check if available in proxdefs
   0.upto(proxdefs.length - 1) do |d|
      if proxdefs[d].name == (cname[0..-2] + "~") ## end off and replaced - available
         cdefinition = proxdefs[d]
      end
   end #upto
end #if
### it's far but NOT a far (-) component
if (disteye > @proximity_Mid) and (cname[-1,1] != "-")
   ### now check if available in proxdefs
   0.upto(proxdefs.length - 1) do |d|
      if proxdefs[d].name == (cname[0..-2] + "-") ## end off and replaced - available
         cdefinition = proxdefs[d]
      end
   end #upto
end #if
### 2nd we check if available in prox comp folder
### it's near but NOT a near (+) component
if (disteye <= @proximity_Near) and (cname[-1,1] != "+")
   ### now check if available in skplist
   0.upto(skplist.length - 1) do |s|
      if skplist[s] == (cname[0..-2] + "+.skp") ## end off and replaced and '.skp' - available
         cdefinition = (definitions.load (proxdir + skplist[s]))
      end
   end #upto
end #if
### it's mid but NOT a mid (~) component
if (disteye <= @proximity_Mid) and (disteye > @proximity_Near) and (cname[-1,1] != "~")
   ### now check if available in skplist
   0.upto(skplist.length - 1) do |s|
      if skplist[s] == (cname[0..-2] + "~.skp") ## end off and replaced and '.skp' - available
         cdefinition = (definitions.load (proxdir + skplist[s]))
      end
   end #upto
end #if
### it's far but NOT a far (-) component
if (disteye > @proximity_Mid) and (cname[-1,1] != "-")
   ### now check if available in skplist
   0.upto(skplist.length - 1) do |s|
      if skplist[s] == (cname[0..-2] + "-.skp") ## end off and replaced and '.skp' - available
         cdefinition = (definitions.load (proxdir + skplist[s]))
      end
   end #upto
end #if

if cdefinition 
   cswaps = cswaps.push(cdefinition)
else
   cswaps = cswaps.push("skip")
end #if

   end # for c loop

### start undo
   model.start_operation "proximity_swap"

### do swap
0.upto(proxcomps.length - 1) do |p|
   if cswaps[p] != "skip"
      proxcomps[p].definition= cswaps[p]
   end
end #upto

### commit undo
   model.commit_operation
### update view
   view.invalidate

end #def

end #class

### menu bits  #################################################################

if( not file_loaded?("MatrixProximity.rb") )###
###if( not file_loaded?("57-MatrixProximity.rb") )###

   UI.menu("Plugins").add_separator
   UI.menu("Plugins").add_item("Matrix") { Matrixproximity.matrix } ###v1.3
   UI.menu("Plugins").add_item("Proximity") { Matrixproximity.proximity } ###v1.3
   ###$submenu5.add_item("Matrix") { Matrixproximity.matrix } ###v1.3
   ###$submenu5.add_item("Proximity") { Matrixproximity.proximity } ###v1.3

end

###
file_loaded("MatrixProximity.rb")###
###file_loaded("57-MatrixProximity.rb")###v1.4
### #################################################################
