#SketchyFFD
#Copyright 2008 Chris Phillips
#

require 'sketchup.rb'

#Right click UI menu.
UI.add_context_menu_handler { |menu|
    if(!Sketchup.active_model.selection.empty? && Sketchup.active_model.selection[0].typename=="Group")
        menu.add_item("2x2 FFD"){startFFD(Sketchup.active_model.selection[0],[2,2,2])}
        menu.add_item("3x3 FFD"){startFFD(Sketchup.active_model.selection[0],[3,3,3])}
        menu.add_item("NxN FFD"){
            prompts = ["Width","Depth","Height"]
            values = [4, 4, 4]
            results = inputbox prompts, values, "FFD Dimensions"
            if (results)
                startFFD(Sketchup.active_model.selection[0],results) 
            end
        }
    end
    menu.add_item("Update FFD"){updateFFD}
    }

#This class provides events when the user manipulates the control points
#NOTE. The events work but the hooks have been commented out. 
#Search for observer to enable these events.
class LatticeObserver
    def onElementAdded(entity, xx)
    end
 	def onContentsModified (entity)
	end
        #called when lattice point is moved.
 	def onChangeEntity(entity)
        #puts "changed "+entity.to_s
        if(entity.class==Sketchup::ComponentDefinition)
            entity.set_attribute("controlLattice","currentTransformation",entity.transformation.to_a)    
        elsif(entity.class==Sketchup::ConstructionPoint && $latticeGroup!=nil)
            updateFFD
        end
        
        #if(entity.get_attribute("controlLattice","currentTransformation",nil)!=nil)
            #puts entity.get_attribute("controlLattice","currentTransformation",nil)
        #end
        #~ parent= Sketchup.active_model.active_entities.parent
        #~ if parent.class==Sketchup::ComponentDefinition
           #~ if parent.group?
               #~ trans=parent.instances[0].transformation
               #~ p "in group #{parent.name}"
               #~ p  trans.origin
           #~ end
        #~ end

        
    end
        #called when lattice group is opened
    def onOpen(instance)
        instance.set_attribute("controlLattice","isOpen",true)
    end
        #called when lattice group is closed
    def onClose(instance)
        instance.set_attribute("controlLattice","isOpen",false)
    end
end    


#Create a control lattice and calculate the vertex weights for a group.
def startFFD(grp,numControlPoints)
    $ffdGroup=grp

    #convert and sanity check.
    numControlPoints[0]=(numControlPoints[0]-1).to_f
    numControlPoints[1]=(numControlPoints[1]-1).to_f
    numControlPoints[2]=(numControlPoints[2]-1).to_f

        #remove previous lattice (if any)
    if($latticeGroup!=nil)
        begin
            $latticeGroup.erase!
        rescue
            #group was already deleted.
        end
    end
    
        #create lattice
    $latticeGroup=createControlLattice(grp,numControlPoints)
    
    Sketchup.active_model.start_operation "Start FFD"
    initFFD(Sketchup.active_model.selection[0],numControlPoints)   
    Sketchup.active_model.commit_operation
end

#deform the group based on the position changes of the control points.
def updateFFD
    if($latticeGroup==nil || $latticeGroup.deleted?)
        $latticeGroup==nil
        return
    end
    
    Sketchup.active_model.start_operation "FFD"
    analizeLattice($latticeGroup,$ffdGroup)
    Sketchup.active_model.commit_operation
end
    

#called from startFFD to calculate the groups vertex weights.    
def initFFD(grp,numControlPoints)
    size=grp.bounds.max-grp.bounds.min
        #make sure this group is unique or chaos insues.    
    grp.make_unique

        #global. needs to be made a class varible somehow.
    $allVerts=[]
    grp.entities.each{|ent|
        if(ent.typename == "Edge" && ent.curve!=nil)
            ent.explode_curve #all curves need to be exploded for deform to work right.
        end
        }
    grp.entities.each{|ent|
        begin#guard against entities that dont have verts.
            ent.vertices.each{|vert|
                $allVerts.push(vert)
            }
        rescue
            puts "Warning:entities of type #{ent.typename} can't be deformed."
        end
    }
        #allverts now contains redundant verts. remove duplicates.
    $allVerts.uniq!
    #puts $allVerts.length
    
        #Global to hold the calculated weight for each vertex per control point.
    $allVertWeights=[]

        #this loop could be dramaticly optimized.
    vi=0        
    $allVerts.each{|vert|
        stuv=vert.position
        stuv.x=stuv.x/size[0];stuv.y=stuv.y/size[1];stuv.z=stuv.z/size[2]
            #calc weights.
        weights=[]
        0.upto(numControlPoints[0]){|x|
            bx=calcBernstein(x,numControlPoints[0],stuv.x)
            0.upto(numControlPoints[1]){|y|
                bxy=bx*calcBernstein(y,numControlPoints[1],stuv.y)
                0.upto(numControlPoints[2]){|z|
                    weights.push(bxy * calcBernstein(z,numControlPoints[2],stuv.z))
                }
            }
        }
        $allVertWeights.push(weights)
        vi=vi+1
        Sketchup.set_status_text ("Weighing  #{vi} of #{$allVerts.length}") if(vi%100==0)

    }
end

#deform the mesh based on the change in position of each control point.
#deltas is an array. 
#each element in the deltas array is an array with 2 elements. 
#the first element is the index of the control point.
#the second element is the change in postition of that point. (a array of 3 floats).
#example:
#deltas=[]
#deltas.push([0,[0.0,0.0,1.0]])#move control point 0 "up" 1.0
#deltas.push([5,[0.0,0.0,-2.0]])#move control point 5 "down" 2.0
#etc

def applyMultipleFFD(grp,deltas)
    
    #grp.entities.transform_entities(Geom::Transformation.new([0.0,0.0,0.001]),grp.entities.to_a)
    vi=0;#used to loop through $allVertWeights in the same order as $allVerts
    $allVerts.each{|vert|
        weights=$allVertWeights[vi] #get pre-calculated array of vertex weights (one per control point)
        vi=vi+1
        
            #accumulate the change in position of this vertex based on 
            #each control points weighted movement.
        dvect=[0.0,0.0,0.0] 
        deltas.each{|delta|
            weight=weights[delta[0]] #delta[0] = control point index.
            vect=delta[1]#delta[1]=delta vector for control point
            dvect[0]+=vect[0]*weight
            dvect[1]+=vect[1]*weight
            dvect[2]+=vect[2]*weight
        }
        #move the vertex.
        begin
        grp.entities.transform_entities(Geom::Transformation.new(dvect),vert)
        rescue
        end
        #display progress
        Sketchup.set_status_text ("Deforming #{vi} of #{$allVerts.length}") if(vi%100==0)
        #puts("Deforming #{vi} of #{$allVerts.length}") if(vi%100==0)
    }
end

#deform by a single moved control point
#not used anymore since applyMultipleFFD is much faster for multiple moved cp's
def applyFFD(grp,index,vect)
    vi=0;
    $allVerts.each{|vert|
        weights=$allVertWeights[vi] ##weights=vert.get_attribute("FFD","weights",nil)
        weight=weights[index]
        grp.entities.transform_entities(Geom::Transformation.new([vect[0]*weight,vect[1]*weight,vect[2]*weight]),vert)
        vi=vi+1
    }
end

#helper function to create an array containing 
#the position of each control point based 
#on the size of the bounding box.
def calculateControlLattice(size,numControlPoints)
    #puts numControlPoints
    xstep=size.x.to_f/numControlPoints[0]
    ystep=size.y.to_f/numControlPoints[1]    
    zstep=size.z.to_f/numControlPoints[2]

    controlPoints=[]
    0.upto(numControlPoints[0]){|x|
        0.upto(numControlPoints[1]){|y|
            0.upto(numControlPoints[2]){|z|
                #NOTE: 0.001+ is because bug in sketchup will not allow construction point at 0,0,0
                controlPoints.push([0.001+(x*xstep),0.001+(y*ystep),0.001+(z*zstep)])
            }
        }
    }
    return(controlPoints)
end

#create a group to act as a control lattice.
def createControlLattice(grp,numControlPoints)
    #calc min and max bounding
    size=grp.bounds.max-grp.bounds.min
    origin=grp.bounds.min
    ents=grp.entities
    
    controlPoints=calculateControlLattice(size,numControlPoints)
    Sketchup.active_model.start_operation("create control lattice")
    latticeGroup=Sketchup.active_model.active_entities.add_group

        #create control lattice group.
    entities=latticeGroup.entities

        #create an observer for events. Not used yet.
    observer=LatticeObserver.new
    
    #create construction points.
    index=0
    controlPoints.each{|lcp|
        cpt=entities.add_cpoint(Geom::Point3d.new([lcp[0],lcp[1],lcp[2]]))
        
        cpt.set_attribute("controlPoint","originalPosition",cpt.position.to_a)
        cpt.set_attribute("controlPoint","index",index)
        index=index+1
        
        #trigger event any time the user moves a point. not used yet.
        cpt.add_observer(observer) 
    }

    #move lattice group to match ffd group
    latticeGroup.transformation=grp.transformation

    latticeGroup.set_attribute("controlLattice","currentTransformation",latticeGroup.transformation.to_a)
    
    #monitor group open/close event. not used yet.
    latticeGroup.add_observer(observer)
    
    Sketchup.active_model.commit_operation

    return(latticeGroup)
end

#determine which (if any) control points have been moved and deform the verts in the ffd group.
def analizeLattice(latticeGroup,ffdGroup)
    
    xform=Geom::Transformation.new()
    if(latticeGroup.get_attribute("controlLattice","isOpen",false))
        #if group is open transform by inverse of saved xform (the real xform).
        xform=Geom::Transformation.new(latticeGroup.get_attribute("controlLattice","currentTransformation",nil))
        xform.invert!
    end
    #puts entity.get_attribute("controlLattice","currentTransformation",nil)
    
    deltas=[]
    latticeGroup.entities[0].parent.entities.each{|cpt|
        op=cpt.get_attribute("controlPoint","originalPosition",nil)
        if(op!=nil)#ent is a control point.
            cp=cpt.position.transform(xform).to_a#get current position
            if(cp!=op)#point moved?
                delta=[]#calculate change
                delta[0]=cp[0]-op[0]
                delta[1]=cp[1]-op[1]
                delta[2]=cp[2]-op[2]
                cpt.set_attribute("controlPoint","originalPosition",cp)#update point so user can tweak after inital ffd.
                index=cpt.get_attribute("controlPoint","index",nil)
                deltas.push([index,delta])#add to array of changes.
            end
        end
    }
    #update mesh    
    applyMultipleFFD(ffdGroup,deltas) if (deltas.length>0)
end

#used by calcBernstein. could be replaced by a table since in 
#this application n is never more than 3 (number of control points).
def factorial(n)
    sum = 1
    sum.upto(n) { |i| sum *= i }
    sum
end

#TODO:For this application this function could be replaced by a lookup since
#i=0 to number of control points (<3),n=number of control points (never changes)
#u=float that cant be predicted.
def calcBernstein (i, n, u) #Bernstein Polynomial
    binomial = factorial(n).to_f / (factorial(n - i).to_f * factorial(i).to_f)
    bernstein = binomial * (u**i) * ((1-u)**(n-i))
    return(bernstein)
end

#end