=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed April / July 2008 by Fredo6

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   EraserOnSurface.rb
# Original Date	:   04 June 2008 - version 1.2
#					12 Jul 2008 - version 1.3
# Type			:   Sketchup Tools
# Description	:   Erase lines and construction lines on a surface
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module SUToolsOnSurface

#Constants for LineOnSurface Module (do not translate)	
TOS_ICON_ERASER = "Eraser"
TOS_CURSOR_ERASER = "Eraser_Line"
TOS_CURSOR_ERASER_NO = "Eraser_Line_NO"
TOS_CURSOR_ERASER_C = "Eraser_Cline"
TOS_CURSOR_ERASER_NO_C = "Eraser_CLine_NO"

STR_Eraser_Title = ["Erase Contours on Surface",
	              "|FR| Effacement de contours sur une surface"]					 
MSG_Eraser_Origin = ["Click and drag over edges or construction lines / points",
                   "|FR| Cliquer et S\lectionner des segments ou lignes / points de construction"]

				 
#--------------------------------------------------------------------------------------------------------------
# Top Calling functions: create the classes and launch the tools
#--------------------------------------------------------------------------------------------------------------			 				   

def SUToolsOnSurface.launch_eraser
	HELP.check_older_scripts
	@tool_eraser = TOSToolEraser.new true unless @tool_eraser
	Sketchup.active_model.select_tool @tool_eraser
end

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# TOSToolEraser: Tool to erase lines (plain or construction) on a surface
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
					
class TOSToolEraser

def initialize(linemode)
	@linemode = linemode
	
	#Loading strings and cursors
	Traductor.load_translation SUToolsOnSurface, /MSG_/, binding, "@msg_"
	
	cursorfamily = Traductor::CursorFamily.new TOS_DIR, "TOS_cursor_"
	@idcursor_eraser = cursorfamily.create_cursor TOS_CURSOR_ERASER, 5, 27
	@idcursor_eraser_no = cursorfamily.create_cursor TOS_CURSOR_ERASER_NO, 5, 27
	@idcursor_eraser_C = cursorfamily.create_cursor TOS_CURSOR_ERASER_C, 5, 27
	@idcursor_eraser_no_C = cursorfamily.create_cursor TOS_CURSOR_ERASER_NO_C, 5, 27

	#initializing variables
	@ip = Sketchup::InputPoint.new
		
end

def activate
	@model = Sketchup.active_model
	@selection = @model.selection
	@entities = @model.active_entities
		
	@enter_down = false
	reset_selection
	@button_down = false
	
	info_show
end

def reset_selection
	@list_lines = []
	@list_valid = []
	@list_contour = []
	@selection.clear
end

def deactivate(view)
	view.invalidate
end

#Perform the reasing of selected edges
def erase_edges
	@model.start_operation Traductor[STR_Eraser_Title]
	lst_erase = []
	if @linemode
		@list_lines.each { |e| lst_erase.push e unless OFSG.restore_polyline_param(e) }
		lst_erase.each { |e| @entities.erase_entities e if e.valid?}
	else
		@entities.erase_entities @list_lines
	end 
	@model.commit_operation
end

def onCancel(flag, view)
	#User did an Undo
	case flag
	when 1, 2	#Undo or reselect the tool
		activate
		return
	when 0	#user pressed Escape
		reset_selection
	end
end

def onSetCursor
	if @linemode
		UI::set_cursor (@ok) ? @idcursor_eraser : @idcursor_eraser_no
	else	
		UI::set_cursor (@ok) ? @idcursor_eraser_C : @idcursor_eraser_no_C
	end	
end

def getMenu(menu)
	if (@list_lines.length > 0)
		menu.add_item(@msg_MnuDone) { erase_edges }
		menu.add_separator
	end	
	option_contextual_menu menu
	true
end

#Populate the options in the Contextual menu
def option_contextual_menu(menu)	
	txcur = @msg_MnuCurrent
	text = @msg_MnuLineMode + " #{txcur} " + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine]  + 
	       ") --> " + TOS___LineMode
	menu.add_item(text) { self.change_option_linemode }
end

def change_option_linemode
	@linemode = !@linemode
	reset_selection
	onSetCursor
end

def onLButtonDown(flags, x, y, view)
	@button_down = true
	onMouseMove(flags, x, y, view)
end

def onLButtonUp(flags, x, y, view)
	erase_edges
	reset_selection
	@button_down = false
end

#Key Up
def onKeyUp(key, rpt, flags, view)
	key = Traductor.check_key key, flags, true

	case key
		#Toggling between fixed and variable length
		when COPY_MODIFIER_KEY
			change_option_linemode if @control_down
			view.invalidate
			info_show
	end	
	@control_down = false
end

#Key down
def onKeyDown(key, rpt, flags, view)
	key = Traductor.check_key key, flags, false

	case key			
		#Calling options
		when CONSTRAIN_MODIFIER_KEY
			onMouseMove 2, @x, @y, view if @button_down

		when COPY_MODIFIER_KEY
			@control_down = true
			return
		when TABLE_FKEY[TOS___LineMode] 
			change_option_linemode
			
		when 13
			@enter_down = true
			@control_down = false
			return
		else
			@control_down = false
			return
			
	end	
	@control_down = false
	
	view.invalidate
	info_show
end

#Check if the edges are already part of the selection
def already_selected(ledge)
	status = false
	ledge.each do |e|
		if @list_lines.include?(e) 
			status = true
		elsif OFSG.get_polyline_attribute(e)
			@list_lines.push e
			@selection.add e
			status = true
		end	
	end
	status
end

#Check if edges are already known
def already_valid(ledge)
	status = false
	ledge.each do |e|
		if @list_valid.include?(e) 
			status = true
		elsif OFSG.get_polyline_attribute(e)
			@list_valid.push e
			status = true
		end	
	end
	status
end

#check if contour already known
def already_contour(ledge)
	status = false
	le = []
	ledge.each do |e|
		if @list_contour.include?(e) 
			status = true
		elsif OFSG.get_polyline_attribute(e)
			le.push e
			status = true
		end	
	end
	return status unless le.length > 0
	
	pe = PolyEdit.new
	pe.set_list_edge le, nil
	ll = pe.get_list_edge
	ll.each do |l|
		l.each do |e|
			unless @list_lines.include?(e)
				@list_lines.push e
				@selection.add e
				@list_contour.push e
			end	
		end	
	end	
	true
end

#Check and mark edge or edges at vertex for selection
def mark_edge_for_delete(edge, vertex)
	le = (vertex) ? vertex.edges : [edge]
	(@button_down) ? already_selected(le) : already_valid(le)
end

#Check and mark edge or edges at vertex for selection with full contour option
def mark_contour_for_delete(edge, vertex)
	le = (vertex) ? vertex.edges : [edge]
	return false unless already_valid(le)
	(@button_down) ? already_contour(le) : true
end

def mark_cline_for_delete(cline)
	if OFSG.get_polyline_attribute(cline) 
		if @button_down && ! @list_lines.include?(cline)
			@list_lines.push cline
			@selection.add cline
		end	
		status = true
	end	
	status
end

#Mouse Move method
def onMouseMove(flags, x, y, view)
	@x = x
	@y = y
	@ok = false
	ph = view.pick_helper
	ph.do_pick x, y
	edge = ph.best_picked	
	if @linemode
		return unless edge && edge.class == Sketchup::Edge
		@ip.pick view, x, y
		if flags > 1
			@ok = mark_contour_for_delete edge, @ip.vertex
		else
			@ok = mark_edge_for_delete edge, @ip.vertex
		end	
		if (@ok) && edge && OFSG.get_polyline_attribute(edge)
			@pk_edge = edge
		else
			@pk_edge = nil
		end	
	else	
		return unless edge && (edge.class == Sketchup::ConstructionLine || edge.class == Sketchup::ConstructionPoint)
		@ok = mark_cline_for_delete edge
	end	
	onSetCursor
	view.invalidate
end	

#Draw method for Polyline tool
def draw(view)	
	if @button_down == false && @pk_edge && @pk_edge.valid?
		view.drawing_color = 'orange'
		view.line_stipple = ""
		view.line_width = 4
		view.draw GL_LINE_STRIP, @pk_edge.start.position, @pk_edge.end.position
	end
end

#display information in the Sketchup status bar
def info_show
	msg = Traductor[STR_Eraser_Title] 	
	msg += " [" + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine] + "] -- "
	msg += @msg_Eraser_Origin
	
	Sketchup.set_status_text msg	
end

end	#End Class TOSToolEraser
	
end	#End Module SUToolsOnSurface
