=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed April / July 2008 by Fredo6

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   FreehandOnSurface.rb
# Original Date	:   12 Jul 2008 - version 1.3
# Revisions		:	
# Type			:   Sketchup Tools
# Description	:   Suite of Tools to draw on a surface
# Menu Items	:   Tools --> "Line on Surface"
# Toolbar		:   Name = "Surface Operations"
# Context Menu	:   All options
# Usage			:   See Tutorial and Quick Ref Card in PDF format
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module SUToolsOnSurface

#Constants for LineOnSurface Module (do not translate)
TOS_ICON_FREEHAND = "Freehand"	
TOS_CURSOR_FREEHAND_LINE = "Freehand_Line"
TOS_CURSOR_FREEHAND_CLINE = "Freehand_Cline"

STR_Freehand_Title = ["Free hand on Surface",
	                  "|FR| Main lev\e sur une surface"]					 
MSG_Pixel_Precision = ["Pixel pace", "|FR| Pas en pixel"]
MSG_Freehand_Origin = ["Enter Origin",
                       "|FR| Point d'origine"]
MSG_Freehand_End = ["Freehand Drawing (keep Ctrl down to pause input, Shift down to use inference)",
                    "|FR| Dessiner \ main lev\e (Ctrl enfonc\ pour interrompre saisie, Maj enfonc\ pour forcer inf\rence)"]

				 
#--------------------------------------------------------------------------------------------------------------
# Top Calling functions: create the classes and launch the tools
#--------------------------------------------------------------------------------------------------------------			 				   

def SUToolsOnSurface.launch_freehand
	HELP.check_older_scripts
	@tool_freehand = TOSToolFreehand.new true unless @tool_freehand
	Sketchup.active_model.select_tool @tool_freehand
end

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# TOSToolLine: Tool to draw line (plain or construction) on a surface
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
					
class TOSToolFreehand

def initialize(linemode)
	@linemode = linemode
	@title = Traductor[STR_Freehand_Title]
	
	#Loading strings and cursors
	Traductor.load_translation SUToolsOnSurface, /MSG_/, binding, "@msg_"
	
	cursorfamily = Traductor::CursorFamily.new TOS_DIR, "TOS_cursor_"
	@idcursor_line = cursorfamily.create_cursor TOS_CURSOR_FREEHAND_LINE, 0, 32
	@idcursor_cline = cursorfamily.create_cursor TOS_CURSOR_FREEHAND_CLINE, 0, 32

	#initializing variables
	@ip_origin = Sketchup::InputPoint.new
	@ip = Sketchup::InputPoint.new
	@px_precision = TOS_DEFAULT_Freehand_Precision
	@time_delta = TOS_DEFAULT_Freehand_Time
	@normaldef = Z_AXIS
	@planedef = [ORIGIN, @normaldef]
	
	#Initializing parameters
	@ofsgrp = CommonGroup.create
	
	@option_group = TOS_DEFAULT_Group
	@option_cpoint = (linemode) ? TOS_DEFAULT_CPoint_L : TOS_DEFAULT_CPoint_C
	@option_nocurves = ! TOS_DEFAULT_GenCurve

end

def activate
	@model = Sketchup.active_model
	@selection = @model.selection
	@entities = @model.active_entities
	@bb = @model.bounds	
	
	@option_group = @ofsgrp.get_option_group	
	
	@enter_down = false
	@pts = []
	@distance = 0
	@parcours = []
	@edges = nil
	set_state STATE_ORIGIN
end

def reset
	@lst_points2d = []
	@lst_marks = []
	@lst_parcours = []
	@bigparcours = []
end

def deactivate(view)
	view.invalidate
end

#Return bounding box	
def getExtents
	return @bb if @state == STATE_ORIGIN	
	@bb = @bb.add @mark_last.pt if @mark_last
    @bb
end

def onCancel(flag, view)
	#User did an Undo
	case flag
	when 1, 2	#Undo or reselect the tool
		activate
		return
	when 0	#user pressed Escape
		return  if (@state == STATE_ORIGIN)  #Exiting the tool
		undo_last view
	end
end

def undo_last(view)
	if @lst_marks.length < 2
		set_state STATE_ORIGIN
	else
		@lst_marks[-1..-1] = []
		@mark_last = @lst_marks.last
		view.invalidate
		info_show
	end	
end

def onSetCursor
	UI::set_cursor (@linemode) ? @idcursor_line : @idcursor_cline
end

#Compute all parameters of the contour for drawing or creating the curve
def compute_contour
	@pts_contour = []
	nb = @lst_marks.length - 2
	return if nb <= 0
	@bigparcours = [@lst_marks.first]
	for i in 0..nb
		mk1 = @lst_marks[i]
		mk2 = @lst_marks[i+1]
		parcours = Junction.calculate mk1, mk2
		@bigparcours += parcours[1..-1] if parcours.length > 0 
		pts = []
		parcours.each { |mk| pts.push mk.pt }
		@pts_contour += pts[0..-1]
	end	
	#@pts_contour.push @mark_last.pt
end

#Creation method for the line
def execute_drawing
	return if @lst_marks.length < 2
	
	@model.start_operation @title

	#identifying the Group if needed
	if @option_group
		grp = @ofsgrp.get_current
		unless grp
			@model.abort_operation
			return
		end	
		entities = grp.entities
	else
		entities = @entities
	end
	
	#Creating the complete path
	compute_contour
	lst_vert = []
	@lst_marks.each { |mk| lst_vert.push mk.pt }
	
	#creating the new edges or construction lines
	pts = @pts_contour
	attr = '-'
	list_coseg = []
	OFSG.compute_coseg(@bigparcours, list_coseg) if @linemode && @bigparcours.length > 1
	if @linemode
		edges = []
		OFSG.commit_line(entities, pts, attr, @option_nocurves, list_coseg, edges, lst_vert)
	else
		nb = pts.length - 2
		for i in 0..nb
			cline = entities.add_cline pts[i], pts[i+1]
			OFSG.set_cline_attribute cline, attr
		end
	end	
	if @option_cpoint
		pts.each do |pt|
			cpoint = entities.add_cpoint pt 
			OFSG.set_cline_attribute cpoint, attr
		end
	end	
	@model.commit_operation	
end

#Control the states of the tool
def set_state(state)
	@state = state
	case @state
	when STATE_EXECUTION 
		execute
	when STATE_ORIGIN
		reset
	when STATE_END
		
	end
	info_show
end

def execute
	execute_drawing
	set_state STATE_ORIGIN
end

def getMenu(menu)
	if (@state >= STATE_END)
		menu.add_item(@msg_MnuDone) { execute }
		menu.add_separator
	end	
	option_contextual_menu menu
	true
end

#Populate the options in the Contextual menu
def option_contextual_menu(menu)	
	txcur = @msg_MnuCurrent
	text = @msg_MnuGroup + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_group) ? 'Y' : 'N']] + 
	       ") --> " + TOS___Group
	menu.add_item(text) { self.change_option_group }
	
	text = @msg_MnuNoCurves + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_nocurves) ? 'N' : 'Y']] + 
		   ") --> " + TOS___NoCurves
	menu.add_item(text) { self.change_option_nocurves }
	
	text = @msg_MnuCPoint + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_cpoint) ? 'Y' : 'N']] + 
	       ") --> " + TOS___CPoint
	menu.add_item(text) { self.change_option_cpoint }
		
	text = @msg_MnuLineMode + " #{txcur} " + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine]  + 
	       ") --> " + TOS___LineMode
	menu.add_item(text) { self.change_option_linemode }
end

def change_option_linemode
	@linemode = !@linemode
	onSetCursor
end

def change_option_group
	@option_group = @ofsgrp.set_option_group(!@option_group)
end

def change_option_cpoint
	@option_cpoint = !@option_cpoint
end
	
def change_option_nocurves
	@option_nocurves = !@option_nocurves
end

#Button Down - Start input of End point
def onLButtonDown(flags, x, y, view)
	@time_mouse_down = Time.now
	@time_move = 0
	@xdown = x
    @ydown = y
	@button_down = true
	if (@state == STATE_END)
		#return if close_to_last_point x, y
	end	
	set_state @state + 1
end

def close_to_last_point(x, y)
	return false if @lst_points2d.length == 0
	pt = @lst_points2d.last
	((pt.x - x).abs < 3) && ((pt.y - y).abs < 3)
end

#Button Up - execute if move has happened, otherwise ignore
def onLButtonUp(flags, x, y, view)
	return if @state == STATE_ORIGIN
	return if Time.now - @time_mouse_down < 0.2
	return if @button_down && ((@xdown - x).abs < 2) && ((@ydown - y) < 2)
	if (@lst_points2d.length > 1)
		set_state @state + 1
	end	
	@button_down = false
end

#Double Click to repeat with same length
def onLButtonDoubleClick(flags, x, y, view)
	
end

#Key Up
def onKeyUp(key, rpt, flags, view)
	key = Traductor.check_key key, flags, true

	case key
		#Toggling between fixed and variable length
		when COPY_MODIFIER_KEY
			if @control_down
				@control_down = false
				onMouseMove 0, @xmove, @ymove, view
				return if (Time.now - @time_ctrl_down) > 0.2
				change_option_linemode
				view.invalidate
				info_show
			end	
			
		when CONSTRAIN_MODIFIER_KEY
			view.invalidate
			
		when 13
			set_state @state + 1 unless @enter_down
			@enter_down = false
	end	
	@control_down = false
end

#Key down
def onKeyDown(key, rpt, flags, view)
	key = Traductor.check_key key, flags, false

	case key			
		#Calling options
		when COPY_MODIFIER_KEY
			@control_down = true
			@time_ctrl_down = Time.now
			return
			
		when CONSTRAIN_MODIFIER_KEY
			flags = CONSTRAIN_MODIFIER_MASK
		when TABLE_FKEY[TOS___Group]
			change_option_group
		when TABLE_FKEY[TOS___NoCurves]
			change_option_nocurves
		when TABLE_FKEY[TOS___LineMode] 
			change_option_linemode
		when TABLE_FKEY[TOS___CPoint]
			change_option_cpoint
			
		else
			@control_down = false
			return
			
	end	
	@control_down = false
	
	view.invalidate
	info_show
end

def replace_point?(pt2d)
	return 1 if @lst_points2d.length < 2

	#checking for time
	delta = Time.now - @time_move
	return 0 if delta.to_f < @time_delta
	
	#checking for distance in pixel
	ptlast = @lst_points2d.last
	return -1 if @pt2d_last.distance(ptlast) < @px_precision
	
	return 1
end

#Accept a new input point and map it on the surface
def accept_point(view, flags, pt2d)
	#getting the next marks
	if @inference_on
		@ip.pick view, pt2d.x, pt2d.y
		mark = OFSG.mark_from_inputpoint view, @ip, pt2d.x, pt2d.y, [@mark_last.pt, @normaldef]
		#pt2d = view.screen_coords mark.pt
		replace = replace_point?(pt2d)
		dof = @ip.degrees_of_freedom
		if (dof == 0)
			replace = 1 
			@lst_marks[-1..-1] = []
		end	
	else
		@lst_points2d.each { |pt| pt2d = pt if pt2d.distance(pt) < 5 }
		mark = compute_mark view, pt2d
		replace = replace_point?(pt2d)
	end	
	return unless mark
	
	case replace
	when -1, 0
		@lst_points2d[-1..-1] = [pt2d]
		@lst_marks[-1..-1] = [mark]
	when 1
		@lst_points2d.push pt2d
		@lst_marks.push mark
		@pt2d_last = pt2d
		@mark_last = mark
		@time_move = Time.now
	end	
end

#Compute the mark corresponding to the point 2d
def compute_mark(view, pt2d)
	ray = view.pickray pt2d.x, pt2d.y
	ph = view.pick_helper 
	ph.do_pick pt2d.x, pt2d.y
	
	#finding the face
	face = nil
	edge = nil
	picked = ph.all_picked
	picked.each do |e|
		face = e if e.class == Sketchup::Face
		edge = e if e.class == Sketchup::Edge
	end
	
	unless face
		@ip.pick view, pt2d.x, pt2d.y
		face = @ip.face
	end
	
	#No face
	unless face
		plane = [@mark_last.pt, @normaldef]
		pt = Geom.intersect_line_plane ray, plane
		return OFSG.mark(pt, nil, nil, nil, nil)
	end
	
	#Face - Sepcial treatment for making sure the point is on the face
	plane = face.plane
	pt = Geom.intersect_line_plane ray, plane
	return OFSG.mark(pt, face, nil, edge, nil) if pt && OFSG.within_face_extended?(face, pt)

	face.vertices.each do |v|
		v.faces.each do |f|
			next if f == face
			pt = Geom.intersect_line_plane ray, f.plane
			next unless pt
			return OFSG.mark(pt, f, nil, nil, nil) if OFSG.within_face_extended?(f, pt)
		end
	end
	return nil
end

#Input of length in the VCB
def onUserText(text, view) 
	@enter_down = true
	
	begin
		px = text.to_i
		return UI.beep if px < 10 || px > 200
	rescue
		return UI.beep
	end	
	
	@px_precision = px
	
	view.invalidate
	info_show
end

#Mouse Move method
def onMouseMove(flags, x, y, view)
	@xmove = x
	@ymove = y
	
	#Origin Point
    if (@state == STATE_ORIGIN)
		@ip.pick view, x, y
		@ip_origin.copy! @ip
		mark = OFSG.mark_from_inputpoint(view, @ip, x, y, @planedef)
		view.tooltip = @ip.tooltip
		@pt2d_last = view.screen_coords @ip.position
		@lst_points2d = [@pt2d_last]
		@lst_marks = [mark]
		@mark_last = mark
		@time_move = Time.now
		
	#End Point	
	elsif (@state == STATE_END)
		@skip_on = Traductor.ctrl_mask?(flags)
		@inference_on = Traductor.shift_mask?(flags)
		return if @skip_on
		accept_point view, flags, Geom::Point3d.new(x, y)
		view.tooltip = (@inference_on) ? @ip.tooltip : ""
	end
	
	view.invalidate
	info_show
end	

#Draw method for tool
def draw(view)
	#drawing the origin and end points
	@ip_origin.draw view
	return if @state < STATE_END
	@ip.draw view if @inference_on
	
	#Drawing the Line contour
	if @linemode
		if (@option_group)
			color = 'darkred'
			stipple = "-.-"
		else	
			color = 'red' 
			stipple = ""
		end	
	else
		color = (@option_group) ? 'orange' : 'red'
		stipple = "_"
	end
	width = 1
	view.line_width = width
	view.line_stipple = stipple
	view.drawing_color = color
	
	compute_contour
	view.draw GL_LINE_STRIP, @pts_contour if @pts_contour.length > 1
		
	#Drawing marks at control points
	if @lst_marks.length > 2
		@lst_marks[1..-2].each do |mk|
			view.draw_points mk.pt, 6, 2, "orange"
		end	
	end	
end

#display information in the Sketchup status bar
def info_show
	msg = "[" + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine] + "] "

	case @state
	when STATE_ORIGIN
		msg += @msg_Freehand_Origin
	when STATE_END
		msg += @msg_Freehand_End
	when STATE_EXECUTION
		msg += @title
	end
	
	nb = @bigparcours.length

	msg += " [" + @msg_Group + "]" if (@option_group)
	msg += " [" + @msg_NoCurves + "]" if (@option_nocurves)
	msg += " [" + @msg_CPoint + "]" if (@option_cpoint)
	
	msg += " -- " + @msg_Pixel_Precision + " = #{@px_precision}"
	
	txvalue = @msg_Edges + "=#{nb.to_s} px=#{@px_precision}"

	Sketchup.set_status_text msg	
	Sketchup.set_status_text @msg_Edges, SB_VCB_LABEL
	Sketchup.set_status_text txvalue, SB_VCB_VALUE
end

end	#End Class TOSToolFreehand

end	#End Module SUToolsOnSurface
