=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed April / July 2008 by Fredo6

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   LineOnSurface.rb
# Original Date	:   14 May 2008 - version 1.1
# Revisions		:	04 Jun 2008 - version 1.2
#					11 Jul 2008 - version 1.3
# Type			:   Sketchup Tools
# Description	:   Draw lines on a surface
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module SUToolsOnSurface

#Constants for LineOnSurface Module (do not translate)	
TOS_ICON_LINE = "Line"
TOS_ICON_CLINE = "Cline"
TOS_CURSOR_LINE = "Line"
TOS_CURSOR_CLINE = "Cline"

STR_Line_Title = ["Line on Surface",
	              "|FR| Lignes sur une surface"]					 
MSG_Line_Origin = ["Enter Origin",
                   "|FR| Point d'origine"]
MSG_Line_End = ["Enter End Point",
                "|FR| Point de Fin"]

				 
#--------------------------------------------------------------------------------------------------------------
# Top Calling functions: create the classes and launch the tools
#--------------------------------------------------------------------------------------------------------------			 				   

def SUToolsOnSurface.launch_line
	HELP.check_older_scripts
	@tool_line = TOSToolLine.new true unless @tool_line
	Sketchup.active_model.select_tool @tool_line
end

def SUToolsOnSurface.launch_cline
	@tool_cline = TOSToolLine.new false unless @tool_cline
	Sketchup.active_model.select_tool @tool_cline
end

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# TOSToolLine: Tool to draw line (plain or construction) on a surface
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
					
class TOSToolLine

def initialize(linemode)
	@linemode = linemode
	@title = Traductor[STR_Line_Title]
	
	#Loading strings and cursors
	Traductor.load_translation SUToolsOnSurface, /MSG_/, binding, "@msg_"
	
	cursorfamily = Traductor::CursorFamily.new TOS_DIR, "TOS_cursor_"
	@idcursor_line = cursorfamily.create_cursor TOS_CURSOR_LINE, 3, 31
	@idcursor_cline = cursorfamily.create_cursor TOS_CURSOR_CLINE, 3, 31

	#initializing variables
	@ip_origin = Sketchup::InputPoint.new
	@ip_end = Sketchup::InputPoint.new
	@ip = Sketchup::InputPoint.new
	
	#Initializing parameters
	@ofsgrp = CommonGroup.create
	@linepicker = LinePicker.new
	
	@option_group = TOS_DEFAULT_Group
	@option_cpoint = (linemode) ? TOS_DEFAULT_CPoint_L : TOS_DEFAULT_CPoint_C
	@option_nocurves = ! TOS_DEFAULT_GenCurve
	@option_protractor = false
	@prev_dist = 0
	@prev_dist0 = 0
	@prev_vec = nil
	@prev_vec0 = nil
	@angle_prev = nil

end

def activate
	@model = Sketchup.active_model
	@selection = @model.selection
	@entities = @model.active_entities
	@bb = @model.bounds
	
	@option_group = @ofsgrp.get_option_group	
	
	@enter_down = false
	@pts = []
	@distance = 0
	@parcours = []
	@edges = nil
	@moved = false
	
	set_state STATE_ORIGIN
end

def deactivate(view)
	view.invalidate
end

#Return bounding box	
def getExtents
    return @bb if @state == STATE3_ORIGIN	
	@bb = @linepicker.bounds_add @bb
    @bb
end

def onCancel(flag, view)
	#User did an Undo
	case flag
	when 1, 2	#Undo or reselect the tool
		activate
		return
	when 0	#user pressed Escape
		return  if (@state == STATE_ORIGIN)  #Exiting the tool
		set_state @state - 1
	end
end

def onSetCursor
	UI::set_cursor (@linemode) ? @idcursor_line : @idcursor_cline
end

#Compute the path of the line, both for interactive session and final construction

def compute_path(store=false)
	parcours = @linepicker.parcours
	@linepicker.set_prev_direction parcours[-2..-1] if store && parcours
	return @pts = [] unless parcours
	@pts = []
	parcours.each { |mk| @pts.push mk.pt }
	compute_distance
	if store
		@list_coseg = []
		OFSG.compute_coseg(parcours, @list_coseg) if @linemode
		@parcours = parcours 
	end	
	return @pts
end
	
#Computing distance
def compute_distance
	return 0 if @pts.length < 2
	nb = @pts.length - 2
	@distance = 0.0.to_l
	for i in 0..nb
		@distance += @pts[i].distance @pts[i+1]
	end
end

#Finalization of the line drawing
def compute_junction
	#calculating the parcours
	compute_path true
	
	#Drawing the parcours
	execute_drawing

	#Chaining with next point
	prepare_chaining
end

#Used for Redo with imposed distance
def execute_to_distance(mark, vecdir, len)
	@parcours = Junction.to_distance mark, vecdir, len
	@pts = []
	@parcours.each { |mk| @pts.push mk.pt }
	@linepicker.set_prev_direction @parcours[-2..-1]
	
	#Drawing the parcours
	execute_drawing	
end

#Creation method for the line
def execute_drawing
	return if @pts.length < 2
	
	#Storing distance for further Redo
	push_previous @pts

	@model.start_operation @title

	#identifying the Group if needed
	if @option_group
		grp = @ofsgrp.get_current
		unless grp
			@model.abort_operation
			return
		end	
		entities = grp.entities
	else
		entities = @entities
	end
	
	#creating the new edges or construction lines
	attr = '-'
	if @linemode
		@edges = []
		OFSG.commit_line(entities, @pts, attr, @option_nocurves, @list_coseg, @edges, [@pts.first, @pts.last])
	else
		nb = @pts.length - 2
		for i in 0..nb
			cline = entities.add_cline @pts[i], @pts[i+1]
			OFSG.set_cline_attribute cline, attr
		end
	end	
	if @option_cpoint
		@pts.each do |pt|
			cpoint = entities.add_cpoint pt 
			OFSG.set_cline_attribute cpoint, attr
		end
	end	
	@model.commit_operation
	
end

#Perfrom correction of distance after operation
def correct_previous(len)
	#Checking if this is applicable
	return false unless @edges
	@edges.each { |e| return false unless e.valid? }
	return false unless @parcours && @parcours.length >= 2
	
	#Computing the origin
	mark = @parcours[0]
	vecdir = mark.pt.vector_to @parcours[1].pt
		
	Sketchup.undo
	execute_to_distance mark, vecdir, len
	prepare_chaining
	true
end

#Switch the end and origin for drawing the next segment in continuity
def prepare_chaining(view=nil)
	return set_state(STATE_ORIGIN) unless @chaining
	return unless @parcours.length > 0
	view = @model.active_view unless view
	@linepicker.chain_origin view, @parcours.last
	set_state STATE_END
end

#Control the states of the tool
def set_state(state)
	@state = state
	case @state
	when STATE_EXECUTION 
		compute_junction
	when STATE_ORIGIN
		@linepicker.reset
	when STATE_END
		
	end
	info_show
end

def getMenu(menu)
	if (@state >= STATE_END)
		menu.add_item(@msg_MnuDone) { compute_junction }
		menu.add_separator
	end	
	if (@prev_dist0 > 0)
		tx = Traductor[MSG_MnuRedo, @prev_dist0.to_l]
		menu.add_item(tx) { menu_redo }
		menu.add_separator
	end
	option_contextual_menu menu
	true
end

#Populate the options in the Contextual menu
def option_contextual_menu(menu)	
	txcur = @msg_MnuCurrent
	text = @msg_MnuGroup + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_group) ? 'Y' : 'N']] + 
	       ") --> " + TOS___Group
	menu.add_item(text) { self.change_option_group }
	
	text = @msg_MnuNoCurves + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_nocurves) ? 'N' : 'Y']] + 
		   ") --> " + TOS___NoCurves
	menu.add_item(text) { self.change_option_nocurves }
	
	text = @msg_MnuCPoint + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_cpoint) ? 'Y' : 'N']] + 
	       ") --> " + TOS___CPoint
	menu.add_item(text) { self.change_option_cpoint }
	
	text = @msg_MnuProtractor + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_protractor) ? 'Y' : 'N']] + 
	       ") --> " + TOS___Protractor
	menu.add_item(text) { self.change_option_protractor }
	
	text = @msg_MnuLineMode + " #{txcur} " + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine]  + 
	       ") --> " + TOS___LineMode
	menu.add_item(text) { self.change_option_linemode }
end

def change_option_linemode
	@linemode = !@linemode
	onSetCursor
end

def change_option_group
	@option_group = @ofsgrp.set_option_group(!@option_group)
end

def change_option_cpoint
	@option_cpoint = !@option_cpoint
end

def change_option_protractor
	@option_protractor = !@option_protractor
	@linepicker.set_protractor_on @option_protractor
end
	
def change_option_nocurves
	@option_nocurves = !@option_nocurves
end

#Button Down - Start input of End point
def onLButtonDown(flags, x, y, view)
	@time_mouse_down = Time.now
	@xdown = x
    @ydown = y
	if (@state == STATE_END)
		return if @linepicker.close_to_origin(x, y)
	end	
	set_state @state + 1
	@chaining = true
end

#Button Up - execute if move has happened, otherwise ignore
def onLButtonUp(flags, x, y, view)
	return if Time.now - @time_mouse_down < 0.2
	return if (@xdown - x).abs < 2 && (@ydown - y).abs < 2
	if (@linepicker.mark_end && @linepicker.moved?)
		@chaining = false
		set_state @state + 1
	end	
end

#Double Click to repeat with same length
def onLButtonDoubleClick(flags, x, y, view)
	redo_previous
end

def push_previous(pts)
	nb = pts.length - 2
	d = 0
	if (nb >= 0)
		d = 0
		for i in 0..nb
			d += pts[i].distance pts[i+1]
		end	
	end	
	return if d == 0
	
	#swapping the stored values	
	@prev_dist = @prev_dist0
	@prev_vec = (@prev_vec0) ? @prev_vec0.clone : nil
	@prev_dist0 = d
	@prev_vec0 = pts[0].vector_to pts[1]
end

#Correction of length after operation
def redo_previous
	mark_origin = @linepicker.mark_origin
	mark_end = @linepicker.mark_end
	moved = @linepicker.moved?
	return UI.beep if mark_end
	if (moved && @prev_dist > 0)
		correct_previous @prev_dist
	elsif (!moved && @prev_vec0 && @prev_dist0 > 0)
		execute_to_distance(mark_origin, @prev_vec0, @prev_dist0)
		prepare_chaining
	else
		UI.beep
	end	
end

#Redo, when called from contextual menu
def menu_redo
	mark_origin = @linepicker.mark_origin
	mark_end = @linepicker.mark_end
	moved = @linepicker.moved?
	if (mark_end && moved && @prev_dist0 > 0)
		vecdir = @pts[0].vector_to @pts[1]
		execute_to_distance(mark_origin, vecdir, @prev_dist0)
		prepare_chaining
	elsif (mark_end == nil && @prev_vec0 && @prev_dist0 > 0)
		execute_to_distance(mark_origin, @prev_vec0, @prev_dist0)
		prepare_chaining
	else
		UI.beep
	end	
end

#Set the default axis via arrows
def check_arrow_keys(key)
	case key
	when VK_RIGHT
		axisdef = X_AXIS
	when VK_LEFT
		axisdef = Y_AXIS
	when VK_UP
		axisdef = Z_AXIS
	when VK_DOWN
		axisdef = nil
	else
		return false
	end
	@linepicker.set_forced_axis axisdef
	return true
end

#Key Up
def onKeyUp(key, rpt, flags, view)
	key = Traductor.check_key key, flags, true

	case key
		#Toggling between fixed and variable length
		when COPY_MODIFIER_KEY
			if @control_down
				@control_down = false
				return if (Time.now - @time_ctrl_down) > 0.2
				change_option_linemode
				@linepicker.simulate_move_end flags, view if (@state >= STATE_END)
				view.invalidate
				info_show
			end	
			
		when CONSTRAIN_MODIFIER_KEY
			@linepicker.end_forced
			view.invalidate
			
		when 13
			set_state @state + 1 unless @enter_down
			@enter_down = false
	end	
	@control_down = false
end

#Key down
def onKeyDown(key, rpt, flags, view)
	key = Traductor.check_key key, flags, false
	
	if check_arrow_keys(key)
		@linepicker.simulate_move_end flags, view if (@state >= STATE_END)
		view.invalidate
		info_show
		@control_down = false
		return
	end

	case key			
		#Calling options
		when COPY_MODIFIER_KEY
			@control_down = true
			@time_ctrl_down = Time.now
			return
			
		when CONSTRAIN_MODIFIER_KEY
			flags = CONSTRAIN_MODIFIER_MASK
		when TABLE_FKEY[TOS___Group]
			change_option_group
		when TABLE_FKEY[TOS___NoCurves]
			change_option_nocurves
		when TABLE_FKEY[TOS___LineMode] 
			change_option_linemode
		when TABLE_FKEY[TOS___Protractor]
			change_option_protractor
		when TABLE_FKEY[TOS___CPoint]
			change_option_cpoint
			
		else
			@control_down = false
			return
			
	end	
	@control_down = false
	
	@linepicker.simulate_move_end flags, view if (@state >= STATE_END)
	view.invalidate
	info_show
end

#Input of length in the VCB
def onUserText(text, view) 
	@enter_down = true
	
	#Parsing the text
	ldist = []
	langle = []
	return UI.beep unless parse_VCB(text, ldist, langle)
	
	#Modifying parameters of the line
	if ldist.length == 0 && langle.length == 0			#No change
		return
	elsif ldist.length > 0 && langle.length == 0		#Change of length only
		modify_length view, ldist.last
	elsif ldist.length == 0 && langle.length > 0		#Modify direction only
		@angle_prev = langle.last
		@linepicker.set_force_angle_direction langle.last	
	else												#modify both length and angle
		@angle_prev = langle.last
		@linepicker.set_angle_direction langle.last
		modify_length view, ldist.last	
	end
	view.invalidate
	info_show
end

#Modify length of line
def modify_length(view, len)
	mark_origin = @linepicker.mark_origin
	mark_end = @linepicker.mark_end
	unless mark_end
		UI.beep unless correct_previous len
	else
		vecdir = mark_origin.pt.vector_to mark_end.pt
		if vecdir.length > 0.0
			execute_to_distance mark_origin, vecdir, len
			prepare_chaining view
		else
			UI.beep
		end	
	end	
end

#Parse the VCB text
def parse_VCB(text, ldist, langle)
	@input_error = nil
	nbeep = 0
	parse_text text, ldist, langle, nbeep
	if (nbeep > 0) 
		@input_error = text
		return false
	end	
	true
end
#Recursive method to parse input text
def parse_text(text, ldist, langle, nbeep)
	#end of text
	text = text.strip
	return if text.length == 0
		
	#Chunk of text separated by space or semi-column
	if text =~ /\s+/ || text =~ /;+/
		parse_text $`, ldist, langle, nbeep
		parse_text $', ldist, langle, nbeep

	#Angle
	elsif text =~ /d/i
		parse_angle $`, langle

	#Length
	else
		parse_distance text, ldist, nbeep
	end
end

#Parse angle fom VCB
def parse_angle(text, langle)
	if text == ""
		langle.push @angle_prev if @angle_prev != nil
		return
	end	
	angle = text.to_f.degrees
	angle = angle.modulo (DEUX_PI)
	angle = angle + DEUX_PI if angle < 0
	langle.push angle
end

def parse_distance(text, ldist, nbeep)
	begin
		d = text.to_l
		if d == 0.0 
			nbeep += 1
		else
			ldist.push d
		end
	rescue
		nbeep += 1
	end
end

#Mouse Move method
def onMouseMove(flags, x, y, view)
	#Origin Point
    if (@state == STATE_ORIGIN)
		@mark_beg = @linepicker.onMouseMove_origin flags, x, y, view
		
	#End Point	
	elsif (@state == STATE_END)
		@linepicker.onMouseMove_end flags, x, y, view
	end
	
	@input_error = nil
	view.tooltip = @linepicker.tooltip
	view.invalidate
	info_show
end	

#Draw method for tool
def draw(view)
	#drawing the origin and end points
	@linepicker.draw view
	
	#Drawing the Line contour
	if (@state >= STATE_END)
		pts = compute_path
		#@linepicker.draw_inference_mark view, pts
		factor = @linepicker.inference_factor
		if @linemode
			if (@option_group)
				#view.drawing_color = 'darkred'
				stipple = "-.-"
				width = 2 * factor
			else	
				#view.drawing_color = 'black' 
				stipple = ""
				width = 1 * factor
			end	
		else
			#view.drawing_color = (@option_group) ? 'orange' : 'black'
			stipple = "_"
			width = 1 * factor
		end
		
		@linepicker.set_drawing_parameters 'black', width, stipple
		@linepicker.draw_line view, true
		#view.drawing_color = @linepicker.color
		#view.draw GL_LINE_STRIP, pts if pts.length > 1
		view.line_stipple = ""
		view.draw_points pts[1..-2], 10, 3, 'black' if @option_cpoint && pts.length > 2
	end	
	#Watchlist.draw view
end

#display information in the Sketchup status bar
def info_show
	msg = "[" + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine] + "] "

	case @state
	when STATE_ORIGIN
		msg += @msg_Line_Origin
	when STATE_END
		msg += @msg_Line_End
	when STATE_EXECUTION
		msg += @title
	end
	
	compute_distance
	d = @distance

	msg += " [" + @msg_Group + "]" if (@option_group)
	msg += " [" + @msg_NoCurves + "]" if (@option_nocurves)
	msg += " [" + @msg_CPoint + "]" if (@option_cpoint)

	msg += "   #{@msg_Error} {#{@input_error}}" if @input_error
	
	txvalue = d.to_l.to_s
	angle = @linepicker.get_angle_direction
	txvalue += " ; " + sprintf("%3.1f ", angle.radians) + "\" if angle

	Sketchup.set_status_text msg	
	Sketchup.set_status_text @msg_Distance, SB_VCB_LABEL
	Sketchup.set_status_text txvalue, SB_VCB_VALUE
end

end	#End Class TOSToolLine

end	#End Module SUToolsOnSurface
