=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed April / July 2008 by Fredo6

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   ShapeOnSurface.rb
# Original Date	:   12 July 2008 - version 1.3
# Type			:   Sketchup Tools
# Description	:   Polygon, Ellipes and Arcs on Surface (part of the suite of Tools to draw on a surface)
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module SUToolsOnSurface

#Constants for PolygonOnSurface Module (do not translate)	
MSG_Shape_Origin = ["Pick Initial Point",
	                  "|FR| Cliquer Point Initial"]
MSG_Shape_End = ["Pick End Point",
	             "|FR| Cliquer Extr\mit\"]
MSG_Axis = ["Axis",
            "|FR| Axe"]
MSG_ShapeError = ["ERROR:",
                  "|FR| ERREUR:"]

NAME_RECTANGLE = ["Rectangle", "|FR| Rectangle"]	
NAME_PARALLELOGRAM = ["Parallelogram", "|FR| Parall\logramme"]	
NAME_ELLIPSE = ["Ellipse", "|FR| Ellipse"]	
NAME_ARC = ["Arc", "|FR| Arc"]	
NAME_POLYGON = ["Polygon", "|FR| Polygone"]	
NAME_CIRCLE = ["Circle", "|FR| Cercle"]	
NAME_SECTOR = ["Sector (Pie)", "|FR| Secteur"]	
NAME_CIRCLE3P = ["Circle (3 Points)", "|FR| Cercle (3 Points)"]	

STR_AXE_LENGTH = ["Length", "|FR| Longueur"]
STR_AXE_WIDTH = ["Width", "|FR| Largeur"]
STR_AXE_RADIUS = ["Radius", "|FR| Rayon"]
STR_AXE_DIAMETER = ["Diameter", "|FR| Diam\tre"]
STR_AXE_CHORD = ["Chord", "|FR| Corde"]
STR_AXE_SAGITTA = ["Sagitta", "|FR| Fl\che"]
STR_AXE_AXIS = ["Axis", "|FR| Axe"]
STR_AXE_FIRST_CHORD = ["First Chord", "|FR| Premi\re corde"]
STR_AXE_SECOND_CHORD = ["Second Chord", "|FR| Seconde corde"]
STR_TRIGO = "Trigonometric sense", "|FR| Sens trigonom\trique"


#Hash Table defining each shape (do not translate text)
HSHAPE_RECTANGLE = { 
	'Type' => CODE_Rectangle,
	'NameConv' => 'Rectangle',
	'Name' => NAME_RECTANGLE,
	'HotX' => 3,
	'HotY' => 31,
	'NbSegDef' => 4,
	'NbFixed' => true,
	'Ortho' => true,
	'Axe1' => STR_AXE_LENGTH,
	'Axe2' => STR_AXE_WIDTH,
	} 
	
HSHAPE_ELLIPSE = { 
	'Type' => CODE_Ellipse,
	'NameConv' => 'Ellipse',
	'Name' => NAME_ELLIPSE,
	'NbSegDef' => TOS_DEFAULT_CircleSegments,
	'NbFixed' => false,
	'Ortho' => true,
	'Radial' => true
	} 
	
HSHAPE_PARALLELOGRAM = { 
	'Type' => CODE_Parallelogram,
	'NameConv' => 'Parallelogram',
	'Name' => NAME_PARALLELOGRAM,
	'NbSegDef' => 4,
	'NbFixed' => true,
	'Ortho' => false
	} 

HSHAPE_ARC = { 
	'Type' => CODE_Arc,
	'NameConv' => 'Arc',
	'Name' => NAME_ARC,
	'NbSegDef' => TOS_DEFAULT_CircleSegments / 2,
	'NbSegMin' => 2,
	'NbFixed' => false,
	'Arc' => true,
	'Ortho' => true,
	'Axe1' => STR_AXE_CHORD,
	'Axe2' => STR_AXE_SAGITTA
	} 

HSHAPE_SECTOR = { 
	'Type' => CODE_Sector,
	'NameConv' => 'Sector',
	'Name' => NAME_SECTOR,
	'NbSegDef' => TOS_DEFAULT_CircleSegments,
	'NbFixed' => false,
	'Ortho' => false,
	'1/2 Axe1' => STR_AXE_RADIUS,
	'1/2 Axe2' => STR_AXE_RADIUS
	} 
	
HSHAPE_POLYGON = { 
	'Type' => CODE_Polygon,
	'Single' => true,
	'NameConv' => 'Polygon',
	'Name' => NAME_POLYGON,
	'NbSegDef' => TOS_DEFAULT_PolygonSegments,
	'NbFixed' => false,
	'Radial' => true,
	'Axe1' => STR_AXE_DIAMETER,
	'1/2 Axe1' => STR_AXE_RADIUS
	} 

HSHAPE_CIRCLE = { 
	'Type' => CODE_Circle,
	'Single' => true,
	'NameConv' => 'Circle',
	'Name' => NAME_CIRCLE,
	'NbSegDef' => TOS_DEFAULT_CircleSegments,
	'NbFixed' => false,
	'Radial' => true,
	'Axe1' => STR_AXE_DIAMETER,
	'1/2 Axe1' => STR_AXE_RADIUS
	} 

HSHAPE_CIRCLE3P = { 
	'Type' => CODE_Circle3P,
	'NameConv' => 'Circle3P',
	'Name' => NAME_CIRCLE3P,
	'NbSegDef' => TOS_DEFAULT_CircleSegments,
	'NbFixed' => false,
	'Radial' => true,
	'Axe1' => STR_AXE_FIRST_CHORD,
	'Axe2' => STR_AXE_SECOND_CHORD,
	'IniAxe1' => true,
	'IniAxe2' => true
	} 
	
STATE3_ORIGIN = 0
STATE3_AXE1 = 1
STATE3_AXE2 = 2
STATE3_EXECUTION = 3

TOS_ShapeInput = Struct.new("TOS_ShapeInput", :nbseg, :ldist, :ldelta, :angle, :beep, :changed) 
TOS_ShapeSaver = Struct.new("TOS_ShapeSaver", :time, :distance1, :distance2, :vecpar1, :vecpar2,
                                              :angle, :mark_origin, :param_axe1, :param_axe2, :angle_sector) 
TOS_ShapeRay = Struct.new("TOS_ShapeRay", :x, :y, :angle, :distance, :dfactor) 
TOS_ShapeDraw = Struct.new("TOS_ShapeDraw", :lmk_contour, :pts_contour, :pts_vertices, :lmk_vertices, 
                                            :delta, :face) 
	
#--------------------------------------------------------------------------------------------------------------
# Top Calling functions: create the classes and launch the tools
#--------------------------------------------------------------------------------------------------------------			 				   

def SUToolsOnSurface.load_one_shape(family, iconconv, hshape)
	#text = Traductor[hshape['Name']] + " " + Traductor[STR_OnSurface]
	text = Traductor[hshape['Name']]
	icon = hshape['NameConv']
	family.add_command(text, text, icon, iconconv, nil) { SUToolsOnSurface.launch_shape hshape['Type'], hshape }
end

def SUToolsOnSurface.load_shapes(family, iconconv)
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_RECTANGLE
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_CIRCLE
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_POLYGON
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_ELLIPSE 
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_PARALLELOGRAM
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_ARC
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_CIRCLE3P
	SUToolsOnSurface.load_one_shape family, iconconv, HSHAPE_SECTOR
end

def SUToolsOnSurface.launch_shape(id, hshape)
	HELP.check_older_scripts
	@hall_tools = {} unless @hall_tools
	tool = @hall_tools[id]
	unless tool
		tool = TOSShapeTool.new hshape
		@hall_tools[id] = tool
	end	
	Sketchup.active_model.select_tool tool
end

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# TOSToolPolygon: Tool to mimic Sketchup Polygon Mesure on surface
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
					
class TOSShapeTool

def initialize(hshape)
	@linemode = true
	@shape = TOSShape.new hshape
	
	#Loading parameters according to type
	@name = @shape.name
	@title = @shape.title
	@type = @shape.type
	@nbseg = @shape.nbsegdef
	@ortho = @shape.ortho
	@single = @shape.single
	cursorfamily = Traductor::CursorFamily.new TOS_DIR, NAMECONV_CURSOR
	@idcursor_line = @shape.get_id_cursor false
	@idcursor_cline = @shape.get_id_cursor true
	
	#Loading strings and cursors
	Traductor.load_translation SUToolsOnSurface, /MSG_/, binding, "@msg_"
		
	#initializing variables
	@ip_origin = Sketchup::InputPoint.new
	@ip_end = Sketchup::InputPoint.new
	@ip = Sketchup::InputPoint.new
	@prev_dist = 0
	@prev_dist0 = 0
	@prev_vec = nil
	@prev_vec0 = nil
	
	#Initializing parameters
	@option_group = TOS_DEFAULT_Group
	@option_cpoint = TOS_DEFAULT_CPoint_L
	@option_nocurves = ! TOS_DEFAULT_GenCurve
	@option_nofaces = ! TOS_DEFAULT_GenFaces
	
	#Group handling
	@ofsgrp = CommonGroup.create
	@linepicker1 = LinePicker.new
	@linepicker1.set_drawing_parameters "black", 1, "_"
	@linepicker2 = LinePicker.new
	@linepicker2.set_drawing_parameters "black", 1, "_"
	@shape.set_line_pickers @linepicker1, @linepicker2
	if @shape.type == CODE_Sector
		@linepicker2.set_protractor_on
		@option_protractor = true
	else
		@option_protractor = false
	end
end

def activate
	@model = Sketchup.active_model
	@selection = @model.selection
	@entities = @model.active_entities
	@bb = @model.bounds
			
	@option_group = @ofsgrp.get_option_group		
	@enter_down = false
	
	@correction_possible = false
	set_state STATE3_ORIGIN
	info_show
end

def deactivate(view)
	view.invalidate
end
	
#Return bounding box	
def getExtents
    return @bb if @state == STATE3_ORIGIN
	@bb = @shape.get_bounds(@bb)	
	@bb
end
	
#Generate the shape in the model	
def execute_drawing
	#Preparing the context of the model
	@model.start_operation @title
	
	#identifying the Group if needed
	if @option_group
		grp = @ofsgrp.get_current
		unless grp
			@model.abort_operation
			return
		end	
		entities = grp.entities
	else
		entities = @entities
	end
	
	if @shape.execute_all_shapes_drawing entities, @linemode, @option_nocurves, @option_nofaces, @option_cpoint
		@model.commit_operation
		@correction_possible = true
	else
		@model.abort_operation
	end
end
	
#Top routine to calculate polygon shape		
def execute_shape
	execute_drawing
	set_state STATE3_ORIGIN
	@time_execute = Time.now.to_f
end
	
def onCancel(flag, view)
	#User did an Undo
	case flag
	when 1, 2	#Undo or reselect the tool
		activate
		return
	when 0	#user pressed Escape
		return if (@state == STATE3_ORIGIN)
		set_state @state - 1
	end
end

#setting the right cursor
def onSetCursor
	UI::set_cursor (@linemode) ? @idcursor_line : @idcursor_cline
end

#Control the states of the tool
def set_state(state)
	state = 0 if state < 0 
	@state = state
	
	#Other cases
	if @state == STATE3_EXECUTION || (@state == STATE3_AXE2 && @single)
		info_show
		return execute_shape
	elsif state == STATE3_ORIGIN
		@linepicker1.reset
		@linepicker2.reset
		@linepicker = @linepicker1
		@shape.reset
	elsif state == STATE3_AXE1
		@linepicker2.reset
		@shape.set_parcours2 nil
	elsif state == STATE3_AXE2
		chain_origin
	end	
	
	#Resetting
	@linepicker = (@state < STATE3_AXE2) ? @linepicker1 : @linepicker2
	info_show
end

#Contextual menu
def getMenu(menu)
	if (@state >= STATE3_AXE2)
		menu.add_item(@msg_MnuDone) { execute_shape }
		menu.add_separator
	end	
	if (@shape.redo_possible?)
		text = @shape.redo_menu_text
		tx = Traductor[MSG_MnuRedo, text]
		menu.add_item(tx) { redo_geometry }
		menu.add_separator
	end
	option_contextual_menu menu
	true
end

#Populate the options in the Contextual menu
def option_contextual_menu(menu)	
	txcur = @msg_MnuCurrent
	
	case @type
	when CODE_Sector, CODE_Parallelogram, CODE_Circle3P
		text = @msg_MnuProtractor + " #{txcur} " + Traductor[DLG_EnumYesNo[(@option_protractor) ? 'Y' : 'N']] + 
	       ") --> " + TOS___Protractor
		menu.add_item(text) { self.change_option_protractor }
	end
	
	if @type = CODE_Sector
		text = @msg_MnuTrigo + " #{txcur} " + OFSG.yes_no(@shape.trigo_sense) + ") --> " + TOS___InputAxes
		menu.add_item(text) { self.toggle_trigo }
	elsif @shape.toggle_axes_valid?
		text = @msg_MnuInputAxes + " #{txcur} " + string_input_axes + ") --> " + TOS___InputAxes
		menu.add_item(text) { self.toggle_option_axes }
	end	
	
	text = @msg_MnuGroup + " #{txcur} " + OFSG.yes_no(@option_group) + ") --> " + TOS___Group
	menu.add_item(text) { self.change_option_group }

	text = @msg_MnuCPoint + " #{txcur} " + OFSG.yes_no(@option_cpoint) + ") --> " + TOS___CPoint
	menu.add_item(text) { self.change_option_cpoint }
	
	unless @type == CODE_Arc
		text = @msg_MnuNoFaces + " #{txcur} " + OFSG.yes_no(@option_nofaces) + ") --> " + TOS___NoFaces
		menu.add_item(text) { self.change_option_nofaces }
	end	
	
	text = @msg_MnuNoCurves + " #{txcur} " + OFSG.yes_no(@option_nocurves) + ") --> " + TOS___NoCurves
	menu.add_item(text) { self.change_option_nocurves }
	
end

def change_option_protractor
	@option_protractor = !@option_protractor
	@linepicker.set_protractor_on @option_protractor
end

def change_option_linemode
	@linemode = !@linemode
	onSetCursor
end

def change_option_group
	@option_group = @ofsgrp.set_option_group(!@option_group)
end

def change_option_nocurves
	@option_nocurves = !@option_nocurves
end

def change_option_nofaces
	@option_nofaces = !@option_nofaces
end

def change_option_cpoint
	@option_cpoint = !@option_cpoint
end

def toggle_option_axes(view=nil)
	return UI.beep unless @shape.toggle_axes_valid?
	@shape.toggle_axes
	chain_origin view
end

def toggle_trigo(view=nil)
	@shape.toggle_trigo
	view = Sketchup.active_model.active_view unless view
	view.invalidate
	info_show
end

def chain_origin(view=nil)
	return unless @state > STATE3_AXE1
	view = @model.active_view unless view
	@shape.chain_origin view 
	view.invalidate
end
		
#Button Down - Start input of End point
def onLButtonDown(flags, x, y, view)
	@time_mouse_down = Time.now.to_f
	@xdown = x
    @ydown = y
	if @state > STATE3_ORIGIN
		return if @linepicker.close_to_origin(x, y)
	end	
	set_state @state + 1
end
	
#Button Up - execute if move has happened, otherwise ignore
def onLButtonUp(flags, x, y, view)
	return if Time.now.to_f - @time_mouse_down < 0.2
	return if (@xdown - x).abs < 5 && (@ydown - y).abs < 5
	set_state @state + 1 if (@linepicker.mark_end && @linepicker.moved?)
end

#Key Up
def onKeyUp(key, rpt, flags, view)
	key = Traductor.check_key key, flags, true

	case key
		#Toggling between fixed and variable length
		when COPY_MODIFIER_KEY
			if @control_down
				@control_down = false
				return if (Time.now.to_f - @time_ctrl_down) > 0.5
				change_option_linemode
				onMouseMove(flags, @xmove, @ymove, view) if (@state >= STATE3_AXE1)
			end	
		when CONSTRAIN_MODIFIER_KEY
			@linepicker.end_forced
			view.invalidate
	
		when 13
			set_state @state + 1 unless @enter_down
			@enter_down = false
	end	
	@control_down = false
end

#Set the default axis via arrows
def check_arrow_keys(key)
	case key
	when VK_RIGHT
		@axisdef = X_AXIS
	when VK_LEFT
		@axisdef = Y_AXIS
	when VK_UP
		@axisdef = Z_AXIS
	when VK_DOWN
		@axisdef = nil
	else
		return false
	end
	axis = @shape.set_plane_def @axisdef
	@linepicker1.set_plane_def axis
	@linepicker2.set_plane_def axis
	return true
end

#Key down
def onKeyDown(key, rpt, flags, view)
	key = Traductor.check_key key, flags, false

	#Option keys
	case key
		#checking arrows for default axis
		when VK_UP, VK_DOWN, VK_LEFT, VK_RIGHT
			check_arrow_keys(key)
			
		#Toggling between line and Cline mode
		when COPY_MODIFIER_KEY
			@control_down = true
			@time_ctrl_down = Time.now.to_f
			return
			
		#forcing inference	
		when CONSTRAIN_MODIFIER_KEY
			flags = CONSTRAIN_MODIFIER_MASK
			force_plane_def view, flags if @state == STATE3_ORIGIN
			
		#Calling options
		when TABLE_FKEY[TOS___Group]
			change_option_group
		when TABLE_FKEY[TOS___CPoint]
			change_option_cpoint
		when TABLE_FKEY[TOS___NoCurves]
			change_option_nocurves
		when TABLE_FKEY[TOS___NoFaces]
			change_option_nofaces
		when TABLE_FKEY[TOS___InputAxes]
			if @type == CODE_Sector
				toggle_trigo view
			else
				toggle_option_axes view
			end	
		when TABLE_FKEY[TOS___Protractor]
			case @type
			when CODE_Sector, CODE_Parallelogram, CODE_Circle3P
				change_option_protractor
			end	
		when TABLE_FKEY[TOS___LineMode]
			change_option_linemode
		else
			@control_down = false
			return
			
	end	
	@control_down = false
	
	onMouseMove(flags, @xmove, @ymove, view) if (@state >= STATE3_AXE1)
end

#Double Click to repeat with same length
def onLButtonDoubleClick(flags, x, y, view)
	redo_geometry
end

#Reexecuting the shape by reusing previous parameters
def redo_geometry
	return UI.beep unless @shape.redo_possible?
	Sketchup.undo if (Time.now.to_f - @time_execute) < 0.5
	@shape.redo_geometry
	execute_shape
end

#Correcting the geometry after entry in the VCB
def correct_geometry(pinput)
	unless @correction_possible && @shape.correction_valid?
		UI.beep
		return false
	end	
		
	Sketchup.undo
	@shape.correct_geometry pinput
	execute_shape
	true
end

#entry in the VCB: Distance and nb of segments
def onUserText(text, view) 
	@enter_down = true
	pinput = @shape.parse_VCB text
	unless pinput
		info_show
		return UI.beep
	end	
	
	if (@state == STATE3_ORIGIN)
		correct_geometry(pinput)
		return
	end
	
	finish = @shape.apply_pinput pinput
	if finish == 1
		return set_state(@state + 1)
	elsif finish == 2
		return execute_shape
	else	
		onMouseMove(0, @xmove, @ymove, view) if (@state >= STATE3_AXE1)
	end	
end

#Force default plane for shapes drawn with 'free' origin
def force_plane_def(view, flags)
	return unless @state == STATE3_ORIGIN
	face = @mark_origin.face
	return unless face && Traductor.shift_mask?(flags)
	normal = face.normal
	@shape.force_plane_def normal
	unless @axisdef
		@linepicker1.set_plane_def normal
		@linepicker2.set_plane_def normal
	end	
	@face_hilight = face
	view.invalidate
	UI.start_timer(5) { @face_hilight = nil }
end

#Mouse Move method
def onMouseMove(flags, x, y, view)
	@xmove = x
	@ymove = y
	@shape.set_camera view
	
	#Move the various points
	if @state == STATE3_ORIGIN
		@mark_origin = @linepicker1.onMouseMove_origin flags, x, y, view
		@shape.set_mark_origin @mark_origin
		force_plane_def view, flags
		tt = ""
	elsif @state == STATE3_AXE1
		@mark_end1 = @linepicker1.onMouseMove_end flags, x, y, view
		@shape.set_parcours1 @linepicker1.parcours
		tt = @linepicker.tooltip + " "
	elsif @state == STATE3_AXE2 && @single == false
		@mark_end2 = @linepicker2.onMouseMove_end flags, x, y, view
		@shape.set_parcours2 @linepicker2.parcours
		tt = @linepicker.tooltip + " "
	end
	view.tooltip = tt + "[#{@shape.nbseg} #{@msg_Segment}]"
	view.invalidate
	info_show
end	

#Draw method for tool
def draw(view)
	#Drawing the axes
	#if @state <= STATE3_AXE1
		#@linepicker1.draw_line view, true
	#else
		#@linepicker1.draw_line view, false
		#@linepicker2.draw_line view, true
	#end
	
	#drawing highlight faces (for setting default normal plane)
	if @face_hilight
		pts = []
		@face_hilight.outer_loop.vertices.each { |v| pts.push v.position }
		pts.push pts.first
		view.drawing_color = 'blue'
		view.line_width = 4
		view.line_stipple = ""
		view.draw GL_LINE_STRIP, pts
	end	
		
	#drawing the polygon
	if (@option_group)
		color = TOS_COLOR_Group
		width = 4
		cwidth = 2
	else
		color = TOS_COLOR_Normal
		width = 2
		cwidth = 1
	end	
	if (@linemode)
		width = width
		stipple = (@option_group) ? "-.-" : ""
	else
		width = cwidth
		stipple = "_"
	end
	
	#Computing and Drawing the shape
	@shape.calculate_all_shapes
	@shape.draw_center view
	@shape.draw_shape view, width, stipple, 5, 8, @option_cpoint	
	
	#Drawing the axes
	if @state <= STATE3_AXE1
		@linepicker1.draw_line view, true
	else
		@linepicker1.draw_line view, false
		@linepicker2.draw_line view, true
	end
	
end

def string_input_axes
	if @type == CODE_Sector
		return @msg_MnuTrigo + " = " + OFSG.yes_no(@shape.trigo_sense)
	end
	str_axe1 = @shape.get_label_axe 1
	str_axe2 = @shape.get_label_axe 2
	(@single) ? str_axe1 : (str_axe1 + ", " + str_axe2)
end

#display information in the Sketchup status bar
def info_show
	msg = @name + " [" + Traductor[(@linemode) ? STR_ModeLine : STR_ModeCLine] + "] -- "
	msg += " #{@shape.nbseg} #{@msg_Segment} -- " unless @shape.nbfixed
	
	case @state
	when STATE3_ORIGIN
		msg += @msg_Shape_Origin
	when STATE3_AXE1
		msg += @msg_Shape_End + " 1"
	when STATE3_AXE2
		msg += (@single) ? @title : @msg_Shape_End + " 2"
	else
		msg += @title
	end	
	msg += " [" + string_input_axes + "]"
	
	angle = nil
	if @state == STATE3_ORIGIN
		d = @last_d
		angle = @last_a
		d = 0.0 unless d
		label = ""
	elsif @single
		label = @shape.get_label_axe 1
		d = @shape.distance1 
	else
		angle = @shape.angle_sector
		if @state > STATE3_AXE1
			d = @shape.distance2 
			label = @shape.get_label_axe 2
		else
			d = @shape.distance1 
			label = @shape.get_label_axe 1
		end
	end
	@last_d = d unless d == 0.0
	if @last_d && d == 0.0
		d = @last_d
		angle = @last_a if @last_a
	end
	
	msg += " -- [" + @msg_Group + "]" if (@option_group)
	msg += " -- [" + @msg_NoCurves + "]" if (@option_nocurves)
	msg += " -- [" + @msg_NoFaces + "]" if (@option_nofaces)
	msg += " -- [" + @msg_CPoint + "]" if (@option_cpoint)
	
	tx_error = @shape.input_error
	msg += "   #{@msg_ShapeError} {#{tx_error}}" if tx_error
	
	txvalue = d.to_l.to_s
	@last_a = angle
	txvalue += " ; " + sprintf("%3.1f ", angle.radians) + "\" if angle
	
	Sketchup.set_status_text msg	
	Sketchup.set_status_text label, SB_VCB_LABEL
	Sketchup.set_status_text txvalue, SB_VCB_VALUE
end

end	#Class TOSShapeTool

#--------------------------------------------------------------------------------------------------------------
# Class TOSShape: hold logic and data for shapes on surface
#--------------------------------------------------------------------------------------------------------------			 				   
					   
class TOSShape

attr_reader :name, :title, :type, :nbsegdef, :nbseg, :ortho, :nbfixed, :single, :arc,
			:distance1, :distance2, :parcours1, :parcours2, :input_error,
			:param_axe1, :param_axe2, :angle_at_origin, :angle_sector, :trigo_sense

#Create the shape accoding to its properties	
def initialize(hshape)
	@hshape = hshape
	@type = hshape['Type']
	@nameconv = hshape['NameConv']
	@name = Traductor[hshape['Name']]
	@title = @name + " " + Traductor[STR_OnSurface]
	@nbsegdef = hshape['NbSegDef']
	@nbsegmin = hshape['NbSegMin']
	@nbsegmin = 3 unless @nbsegmin
	@nbsegmax = hshape['NbSegMax']
	@nbsegmax = 150 unless @nbsegmax
	@nbseg = @nbsegdef
	@ortho = (hshape['Ortho']) ? true : false
	@nbfixed = (hshape['NbFixed']) ? true : false
	@single = (hshape['Single']) ? true : false
	@radial = (hshape['Radial']) ? true : false
	@arc = (hshape['Arc']) ? true : false
	@cursorfamily = Traductor::CursorFamily.new TOS_DIR, "TOS_cursor_"
	@param_axe1 = (hshape['IniAxe1']) ? true : false
	@param_axe2 = (hshape['IniAxe2']) ? true : false
	@trigo_sense = true
	compute_axes_strings
	@lst_rings = []
	@default_axis = Z_AXIS
	@default_normal = Z_AXIS
	@lst_savers = []
end
	
def reset
	@parcours1 = nil
	@parcours2 = nil
	@mkcenter = nil
	@lst_shapedraw = []
	@distance1 = 0.0
	@distance2 = 0.0
	@vecref1 = nil
	@vecref2 = nil
	@input_error = nil
	@angle_sector = nil
end
	
def get_id_cursor(cline)
	name = @hshape['NameConv'] + ((cline) ? '_Cline' : '_Line') 
	hotx = @hshape['HotX']
	hotx = 3 unless hotx
	hoty = @hshape['HotY']
	hoty = 31 unless hoty
	@cursorfamily.create_cursor name, hotx, hoty
end

#Set the line Pickers
def set_line_pickers(linepicker1, linepicker2)
	@linepicker1 = linepicker1
	@linepicker1.set_drawing_parameters "black", 1, "_"
	@linepicker2 = linepicker2
	@linepicker2.set_drawing_parameters "black", 1, "_"
end

#Set the default axis via arrows
def set_plane_def(default_axis)
	@default_axis = (default_axis) ? default_axis : @default_normal
end

def force_plane_def(axis)
	@default_normal = axis
	set_plane_def nil
end

def reset_error
	@input_error = nil
end

def get_bounds(bb)
	@lst_shapedraw.each do |sd|
		bb = bb.add sd.pts_contour
	end	
	bb = @linepicker1.bounds_add bb
	bb = @linepicker2.bounds_add bb
	bb
end

def compute_axes_strings
	@str_axe1 = @hshape["Axe1"]
	@str_axe2 = @hshape["Axe2"]
	@str_half_axe1 = @hshape["1/2 Axe1"]
	@str_half_axe2 = @hshape["1/2 Axe2"]
	
	@str_axe1 = Traductor[@str_axe1] if @str_axe1
	@str_axe2 = Traductor[@str_axe2] if @str_axe2
	@str_half_axe1 = Traductor[@str_half_axe1] if @str_half_axe1
	@str_half_axe2 = Traductor[@str_half_axe2] if @str_half_axe2
	
	ssaxe = Traductor[STR_AXE_AXIS]
	
	@str_axe1 = ssaxe + " 1" unless @str_axe1
	@str_axe2 = ssaxe + " 2" unless @str_axe2
	@str_half_axe1 = "1/2 " + @str_axe1 unless @str_half_axe1
	@str_half_axe2 = "1/2 " + @str_axe2 unless @str_half_axe2
end

def get_label_axe(nb)
	if (nb == 1)
		return (@param_axe1) ? @str_axe1 : @str_half_axe1
	else
		return (@param_axe2) ? @str_axe2 : @str_half_axe2
	end	
end

def toggle_axes_valid?
	@type != CODE_Arc && @type != CODE_Sector && @type != CODE_Circle3P
end

#Change the input mode for input of full or 1/2 axis
#Return true if full axis 2, or false if 1/2 axis2
def toggle_axes
	#Arcs, Sectors and Circle 3P
	return unless toggle_axes_valid?
	
	#Single polygon
	if @single
		@param_axe1 = !@param_axe1
	
	#multiple polygon
	else
		if !@param_axe1 && !@param_axe2
			@param_axe1 = true
		elsif @param_axe1 && !@param_axe2	
			@param_axe2 = true
		elsif @param_axe1 && @param_axe2	
			@param_axe1 = false
		else	
			@param_axe1 = false
			@param_axe2 = false
		end
	end
	
	#returning mode for first axis
	return @param_axis1
end
	
def set_camera(view)
	@vcamera = view.camera.direction
end
	
def set_mark_origin(mark_origin)
	@mark_origin = mark_origin
end
	
def set_parcours1(parcours)
	@parcours1 = parcours
	@distance1 = OFSG.compute_parcours_length parcours
	@mark_origin = (parcours) ? parcours.first : nil
	@vecpar1 = @parcours1[0].pt.vector_to @parcours1[1].pt if parcours
end

def set_parcours2(parcours)
	if (parcours && parcours.length > 1)
		@parcours2 = parcours
		@distance2 = OFSG.compute_parcours_length parcours
		@vecpar2 = @parcours2[0].pt.vector_to @parcours2[1].pt
		@angle_sector = @linepicker2.get_angle_direction
	else
		@parcours2 = nil
		@distance2 = 0.0
		@vecpar2 = nil
		@angle_sector = nil
	end	
end
	
#Compute the mark where to chain the next origin for second axis	
def mark_for_chain_origin
	if @arc
		lres = mark_within_parcours(@parcours1, @distance1 * 0.5)
		mark = lres[0]
		@vec_at_origin = lres[1]
	elsif @param_axe1
		mark = @parcours1.last
		@vec_at_origin = @parcours1[-2].pt.vector_to mark.pt
	else
		mark = @parcours1.first	
		@vec_at_origin = mark.pt.vector_to @parcours1[1].pt
	end
	return mark
end
	
def chain_origin(view)
	@linepicker2.chain_origin view, mark_for_chain_origin 
	@linepicker2.impose_length @distance1 if @type == CODE_Sector
	if @ortho
		@linepicker2.set_imposed_normal @vec_at_origin
	else	
		@linepicker2.set_prev_direction @parcours1
	end	
end
	
#Save the parameters of the geometry	
def save_geometry
	saver = TOS_ShapeSaver.new
	saver.time = Time.now.to_f
	saver.mark_origin = @parcours1.first
	saver.distance1 = @distance1
	saver.param_axe1 = param_axe1
	saver.param_axe2 = param_axe2
	saver.vecpar1 = @parcours1.first.pt.vector_to @parcours1[1].pt
	saver.angle_sector = @angle_sector
	@angle_sector_prev = @angle_sector
	if (@parcours2)
		saver.distance2 = @distance2
		saver.vecpar2 = @parcours2.first.pt.vector_to @parcours2[1].pt
		lres = mark_for_chain_origin
		saver.angle = @vec_at_origin.angle_between saver.vecpar2
	else
		saver.vecpar2 = nil
		saver.distance2 = 0.0
		saver.angle = 0		
	end	
	@lst_savers[0..-2] = [] if @lst_savers.length >= 2
	@lst_savers.push saver
end

def reinterpret_distance(current_axe, saved_axe)
	return 1.0 if saved_axe == current_axe
	return 2.0 if current_axe && !saved_axe
	return 0.5 
end

#compute the center of a Circle defined by 3 points
def compute_center_circle3P
	mk1 = @mark_origin
	mk2 = @parcours1.last
	mk3 = @parcours2.last
	
	vec12 = mk2.pt.vector_to @parcours1[-2].pt
	vec23 = mk2.pt.vector_to @parcours2[1].pt
	normalref = vec12 * vec23
	angle = vec12.angle_between vec23
	sinus = Math.sin angle
	cosinus = Math.cos angle
	return degraded_center if sinus.abs < 0.1		# Points are collinear

	yc = 0.5 * (@distance2 - @distance1 * cosinus) / sinus
	
	@vecref1 = nil
	vec1 = @parcours1.first.pt.vector_to @parcours1[1].pt
	parcours = Junction.to_distance mk1, vec1, @distance1 * 0.5
	mkmid = parcours.last
	v1mid = parcours[-2].pt.vector_to mkmid.pt
	normal = compute_normal_at_mark mkmid
	normal = normal.reverse unless normal % normalref < 0
	
	parcours = Junction.to_distance mkmid, v1mid * normal, -yc
	@mkcenter = parcours.last
	parcours = Junction.calculate mk1, @mkcenter
	@radius1 = OFSG.compute_parcours_length parcours
	@vecref1 = @mkcenter.pt.vector_to parcours[-2].pt
	
	@radius2 = @radius1
	parcours = Junction.calculate @mkcenter, mk2
	@vecref2 = parcours.last.pt.vector_to parcours[-2].pt
end
			
#compute the origin for an arc
def compute_center_arc()	
	#computing the radius (Credit to Stephen La Rocque)
	c = @distance1	#chord length
	s = @distance2	#sagitta length
	c2 = c * c
	y0 = (s - c2 * 0.25 / s)
	r = Math.sqrt(c2 + y0 * y0) * 0.5
	
	vec = @parcours2.last.pt.vector_to @parcours2[-2].pt
	parcours = Junction.to_distance @parcours2.last, vec, r
	@mkcenter = parcours.last
	@radius1 = @radius2 = r
	@vecref1 = parcours.last.pt.vector_to parcours[-2].pt
	@vecref2 = @parcours2[0].pt.vector_to @parcours2[1].pt
end
	
def degraded_center()
	if @param_axe1 
		@radius1 = @distance1 * 0.5
		lres = mark_within_parcours @parcours1, @radius1 
		@mkcenter = lres[0]
		@vecref1 = lres[1]
	else
		@mkcenter = @parcours1[0]
		@radius1 = @distance1
		@vecref1 = @parcours1[0].pt.vector_to @parcours1[1].pt			
	end
	@radius2 = @radius1
end
	
def compute_angle_at_origin
	if @ortho || @single || @parcours2 == nil
		@angle_at_origin = nil
	else
		vec2 = @parcours2[0].pt.vector_to @parcours2[1].pt
		@angle_at_origin = @vec_at_origin.angle_between vec2
	end
end
	
#Compute the origin of the polygon		
def compute_center()

	#Compute angle at origin
	compute_angle_at_origin
	
	#Single or double unfinished
	if (@parcours2 == nil || @parcours2.length < 2)
		degraded_center
		return
	end

	#Compute angle at drawing origin
	
	#Special Computation for arcs, circle 3Points and Sectors
	return compute_center_arc if @type == CODE_Arc
	return compute_center_circle3P if @type == CODE_Circle3P
	

	#Default vectrors for directions
	@vecref1 = @parcours1[0].pt.vector_to @parcours1[1].pt
	@vecref2 = @parcours2[0].pt.vector_to @parcours2[1].pt
	
	#Origin = Center
	if !@param_axe1 && !@param_axe2
		@mkcenter = @parcours1[0]
		@radius1 = @distance1
		@radius2 = @distance2
		return
	end
		
	#Grand Axe 1 and Grand Axe2
	if @param_axe1 && @param_axe2
		@radius1 = @distance1 * 0.5
		@radius2 = @distance2 * 0.5
		lres = mark_within_parcours @parcours1, @radius1 
		mkmid = lres[0]
		@vecref1 = lres[1]
		parcours = Junction.to_distance mkmid, @vecref2, @radius2
		@mkcenter = parcours.last
		@vecref2 = parcours[-2].pt.vector_to @mkcenter.pt
		return
	end	
	
	#!/2 Axe1 and Grand Axe2
	if !@param_axe1 && @param_axe2
		@radius1 = @distance1
		@radius2 = @distance2 * 0.5
		parcours = Junction.to_distance @parcours1.first, @vecref2, @radius2
		@mkcenter = parcours.last
		@vecref2 = parcours[-2].pt.vector_to @mkcenter.pt
		return
	end	

	#Grand Axe1 and 1/2 Axe2
	if @param_axe1 && !@param_axe2
		@radius1 = @distance1 * 0.5
		@radius2 = @distance2
		lres = mark_within_parcours @parcours1, @radius1 
		@mkcenter = lres[0]
		@vecref1 = lres[1]
		return
	end		
end
		
#Computing the center of a parcours	
def mark_within_parcours(parcours, distance)
	mkorigin = parcours[0]
	vec = mkorigin.pt.vector_to parcours[1].pt
	parc = Junction.to_distance mkorigin, vec, distance
	#vecref = parc.last.pt.vector_to parc[-2].pt
	vecref = parc[-2].pt.vector_to parc.last.pt
	[parc.last, vecref]
end

#Compute the normal direction at a Mark	
def compute_normal_at_mark(mark)
	face = mark.face
	return face.normal if face
	return @default_axis if @single || @vecpar2 == nil
	
	normal = nil
	normal = @vecpar1 * @vecpar2 if @vecref1 == nil || @vecref2 == nil || @vecref2.parallel?(@vecref1)
	#normal = @vecref1 * @vecref2 unless normal && normal.valid? 
	normal = @default_axis unless normal && normal.valid? 
	
	normal
end

def absolute_normal_at_mark(mark)	
	face = mark.face
	if face
		normal = face.normal
	elsif @single || @vecpar2 == nil
		normal = @default_axis
	else		
		normal = nil
		normal = @vecpar1 * @vecpar2 if @vecref1 == nil || @vecref2 == nil || @vecref2.parallel?(@vecref1)
		normal = @vecref1 * @vecref2 unless normal && normal.valid? 
		normal = @default_axis unless normal && normal.valid? 
	end
	
	view = Sketchup.active_model.active_view
	vcamera = view.camera.direction
	return (normal % vcamera > 0) ? normal.reverse : normal
end

#Generate the outer contour by joining the vertices
def generate_shapedraw(lst_vert, delta)
	#computing the contour
	lmk_contour = [lst_vert.first]
	nb = lst_vert.length - 2
	for i in 0..nb
		parcours = Junction.calculate lst_vert[i], lst_vert[i+1]
		parcours[1..-1].each { |mk| lmk_contour.push mk } if parcours.length > 1
		lmk_contour.last.signature = true
	end
	
	#computing the points in the polygon contour
	pts_contour = []
	lmk_contour.each { |mk| pts_contour.push mk.pt }
	pts_vertices = []
	lst_vert.each { |mk| pts_vertices.push mk.pt }	

	#Generating the structure
	shapedraw = TOS_ShapeDraw.new
	shapedraw.lmk_vertices = lst_vert
	shapedraw.lmk_contour = lmk_contour
	shapedraw.pts_contour = pts_contour
	shapedraw.pts_vertices = pts_vertices
	shapedraw.delta = delta
	shapedraw.face = nil
	return shapedraw
end

#method to draw the center of a shape, to be called from the Tool draw() method
def draw_center(view, size=nil, color=nil, mark=nil)
	return unless @mkcenter
	color = 'red' unless color
	mark = 3 unless mark
	size = 12 unless size
	view.line_stipple = ""
	view.draw_points @mkcenter.pt, size, mark, color
end	

#method to draw a shape, to be called from the Tool draw() method
def draw_shape(view, width, stipple, mark, size, option_cpoint)
	view.line_stipple = stipple
	color = TOS_COLOR_Normal
	
	@lst_shapedraw.each do |shapedraw|
		view.drawing_color = color
		view.line_width = width
		
		#Drawing contour
		pts_contour = shapedraw.pts_contour
		return unless pts_contour
		pts_vertices = shapedraw.pts_vertices
		view.draw GL_LINE_STRIP, pts_contour if pts_contour.length > 1
		
		#drawing marks on main contour
		view.line_stipple = ""
		view.draw_points pts_vertices, size, mark, color if pts_vertices && pts_vertices.length > 2
		
		#Darwing optional Construction points
		pts = []
		pts_contour.each { |pt| pts.push pt unless pts_vertices.include?(pt) }
		view.draw_points pts, 10, 3, 'black' if option_cpoint && pts.length > 2
		
		color = TOS_COLOR_Secondary
		width = 2
		mark = 0
	end	
end

#create all shapes in the model
def execute_all_shapes_drawing(entities, linemode, option_nocurves, option_nofaces, option_cpoint)	
	#Computing the shapes
	calculate_all_shapes
	
	#drawing each shape
	@edges = []
	attr = @type + " " + ((linemode) ? 'L' : 'C') + " ---" + Time.now.to_i.to_s
	@lst_shapedraw.each do |shapedraw|	
		unless execute_single_shape_drawing shapedraw, entities, linemode, option_nocurves, option_cpoint, attr
			return false
		end	
	end
	
	#Drawing the center of the shape
	cpoint = entities.add_cpoint @mkcenter.pt		#center of the polygon
	OFSG.set_cline_attribute cpoint, attr

	#Generating faces optionally
	generate_all_faces entities unless option_nofaces || linemode == false
	
	#Saving the geometry for future correction or redo
	save_geometry
	
	return true
end

#Create a single shape in the model
def execute_single_shape_drawing(shapedraw, entities, linemode, option_nocurves, option_cpoint, attr)
	
	#Parameters of the shape
	lmk_contour = shapedraw.lmk_contour
	pts_contour = shapedraw.pts_contour
	pts_vertices = shapedraw.pts_vertices
	return false if pts_contour.length < 2
	
	list_coseg = []
	OFSG.compute_coseg(lmk_contour, list_coseg) if linemode	
	
	#Creating the shapes
	if linemode
		edges = []
		OFSG.commit_line entities, pts_contour, attr, option_nocurves, list_coseg, edges, pts_vertices
		@edges += edges
	else
		nb = pts_contour.length - 2
		for i in 0..nb
			cline = entities.add_cline pts_contour[i], pts_contour[i+1]
			OFSG.set_cline_attribute cline, attr
		end
	end	
	if option_cpoint
		pts_contour.each do |pt| 
			cpoint = entities.add_cpoint pt 
			OFSG.set_cline_attribute cpoint, attr
		end
	end	
	
	#Operation successful
	return true
end

#Manage generation of faces for shapes
def generate_all_faces(entities)
	#Arcs have no face
	return if @type == CODE_Arc
	
	#Generate the faces
	@lst_shapedraw.each do |sd|
		begin
			sd.face = entities.add_face sd.pts_contour
		rescue
			sd.face = nil
		end
	end
	
	#For sectors, Shapes are already built. 
	return if @type == CODE_Sector
	
	#For others, need to eliminate some faces to manage rings
	nb = @lst_shapedraw.length - 1
	return if nb == 0
	lsd = @lst_shapedraw.sort { |s1, s2| -s1.delta <=> -s2.delta }
	i = 1
	while (i <= nb)
		face = lsd[i].face
		entities.erase_entities face if face
		i += 2
	end	
end

#Initialize a structure for getting input parameters
def init_pinput
	pinput = TOS_ShapeInput.new
	pinput.nbseg = @nbseg
	pinput.ldist = []
	pinput.ldelta = []
	pinput.angle = nil
	@lst_rings.each { |d| pinput.ldelta.push d }
	pinput.beep = 0
	return pinput
end

#Apply passively the new parameters to the shape
def apply_pinput(pinput)
	finish = 0
	
	#Modifying number of segments
	if pinput.nbseg != @nbseg
		@nbseg = pinput.nbseg
	end

	#modifying rings
	if pinput.ldelta != @lst_rings
		@lst_rings = []
		pinput.ldelta.each do |r| 
			if r == 0.0
				@lst_rings = []
			else	
				@lst_rings.push r
			end	
		end	
	end
	
	#Modifying angle
	if pinput.angle
		finish = (@type == CODE_Sector) ? 1 : 1
	end
	
	#Modifying length
	ldist = pinput.ldist
	nb = ldist.length
	if nb > 0
		if @single
			@distance1 = ldist.last
			finish = 2
		elsif @vecpar2 == nil	
			@distance1 = ldist.last
			finish = 1
		elsif nb == 1 
			@distance2 = ldist.last
			finish = 2
		else 
			@distance1 = ldist[-2]
			@distance2 = ldist.last
			finish = 2
		end
		if @type == CODE_Sector		#sector
			@distance1 = @distance2 = ldist.last
		end			
		#reconstruct_geometry(pinput)
		#@linepicker1.set_parcours @parcours1 if (finish == 1)
	end
	if finish > 0
		reconstruct_geometry(pinput)
		@linepicker1.set_parcours @parcours1 if (finish == 1)
		if pinput.angle && finish == 1
			@linepicker2.set_angle_direction pinput.angle
			#@linepicker2.set_parcours @parcours2
		end	
	end	
	return finish
end

def reconstruct_geometry(pinput)
	@parcours1 = Junction.to_distance @mark_origin, @vecpar1, @distance1
	mark = mark_for_chain_origin
	unless @single || @vecpar2 == nil
		if pinput.angle
			@angle_sector = pinput.angle
			@vecpar2 = OFSG.rotate_vector @mark_origin, @vecpar1, pinput.angle, @default_normal
		end	
		@parcours2 = Junction.to_distance mark, @vecpar2, @distance2
	end	
end

#Check if correction can be done
def correction_valid?
	return false unless @edges
	@edges.each { |e| return false unless e.valid? }
	true
end

#Correct geometry after execution, based on VCB inputs
def correct_geometry(pinput=nil)
	return false unless @lst_savers.length > 0
	saver = @lst_savers.last
	@mark_origin = saver.mark_origin
	@distance1 = saver.distance1
	@distance2 = saver.distance2
	@vecpar1 = saver.vecpar1
	@vecpar2 = saver.vecpar2
	@angle_sector = saver.angle_sector
	apply_pinput pinput if pinput
	reconstruct_geometry(pinput)
	true
end

#check if a Redo is possible
def redo_possible?
	@lst_savers.length > 0
end

#Compute text for redoing
def redo_menu_text
	return nil unless @lst_savers.length > 0
	saver = @lst_savers.last
	s1 = get_label_axe 1
	text = "#{s1} = #{saver.distance1.to_l}"
	unless @single
		s2 = get_label_axe 2
		text += ", #{s2} = #{saver.distance2.to_l}"
	end
	text	
end

#Toggle sense trgo for sectors
def toggle_trigo
	@trigo_sense = !@trigo_sense
end

#Reset shape for redoing, based on last savings
def redo_geometry
	#Retrieving the right saver (because of double-click)
	saverlast = @lst_savers.last
	recent = (Time.now.to_f - saverlast.time) < 0.5
	saverfirst = (recent) ? @lst_savers.first : saverlast
	if (recent && !@single)
		factor1 = reinterpret_distance(@param_axe1, saverlast.param_axe1)
		@distance1 = saverlast.distance1 * factor1
	else
		factor1 = reinterpret_distance(@param_axe1, saverfirst.param_axe1)
		@distance1 = saverfirst.distance1 * factor1
	end
	
	if (@parcours1)
		@vecpar1 = @parcours1.first.pt.vector_to @parcours1[1].pt
	else
		@vecpar1 = saverlast.vecpar1
	end	
	
	#Computing the first axis
	@parcours1 = Junction.to_distance @mark_origin, @vecpar1, @distance1
	return if @single
	
	#recomputing the second vector
	factor2 = reinterpret_distance(@param_axe2, saverfirst.param_axe2)
	@distance2 = saverfirst.distance2 * factor2
	mark2 = mark_for_chain_origin
	if @parcours2
		@vecpar2 = @parcours2.first.pt.vector_to parcours2[1].pt
		@parcours2 = Junction.to_distance mark2, @vecpar2, @distance2
		return
	end	
	
	angle = (@ortho) ? Math::PI * 0.5 : saverlast.angle
	normal = compute_normal_at_mark mark2
	t = Geom::Transformation.rotation mark2.pt, normal, angle
	@vecpar2 = @vecpar1.transform t
	@parcours2 = Junction.to_distance mark2, @vecpar2, @distance2
	@angle_sector = (@ortho) ? nil : angle
end

#Parse the VCB text
def parse_VCB(text)
	pinput = init_pinput
	@input_error = nil
	parse_text text, pinput
	if (pinput.beep > 0) 
		@input_error = text
		return nil
	else
		return pinput
	end	
end

#Recursive method to parse input text
def parse_text(text, pinput)
	#end of text
	text = text.strip
	return if text.length == 0
	
	#Isolating blocks for Delta
	if text =~ /\(.*\)x/i || text =~ /\[.*\]x/i
		parse_text $`, pinput
		parse_delta_block $&, pinput
		parse_text $', pinput
	
	#Chunk of text separated by space or semi-column
	elsif text =~ /\s+/ || text =~ /;+/
		parse_text $`, pinput
		parse_text $', pinput

	#number of segments
	elsif text =~ /s/i
		parse_segments $`, pinput

	#Angle
	elsif text =~ /d/i
		parse_angle $`, pinput

	#delta
	elsif text =~ /x/i
		if $` == ""
			pinput.ldelta = []
		else	
			parse_delta $`, pinput
		end	

	#Length
	else
		parse_distance text, pinput
	end
end

#Parse a Ring block in the form [...]x
def parse_delta_block(text, pinput)
	text = text.slice(1..-3) 
	pinput.ldelta = []
	text = text.strip
	return if text == ""
	
	while true
		text = text.strip
		if text =~ /\s/ || text =~ /;/
			parse_delta $`, pinput
			text = $'
		else
			parse_delta text, pinput
			break
		end
	end	
end

#Parse a single value of ring distance
def parse_delta(text, pinput)
	f = text.to_l
	begin
		pinput.ldelta.push f #unless f == 0.0
	rescue
	end
end

#Parse number of segments
def parse_segments(text, pinput)
	if @nbfixed == true
		pinput.beep += 1
		return
	elsif text == ""
		nb = @nbsegdef
	else 
		nb = text.to_i
	end
	
	if (nb < @nbsegmin || nb > @nbsegmax)
		pinput.beep += 1
	elsif nb != @nbseg	
		pinput.nbseg = nb
	end	
end

#Parse angle (for sectors and parallelograms)
def parse_angle(text, pinput)
	if @single || @ortho
		pinput.beep += 1
		return
	elsif text == ""
		pinput.angle = @angle_sector_prev if @angle_sector_prev
		return
	else 
		angle = text.to_f.degrees
	end
	angle = angle.modulo (DEUX_PI)
	angle = angle + DEUX_PI if angle < 0
	if (angle == 0)
		pinput.beep += 1
	else 	
		angle = (@trigo_sense) ? angle : DEUX_PI - angle
		pinput.angle = angle
	end	
end

def parse_distance(text, pinput)
	begin
		f = text.to_l
		if f <= 0.0
			pinput.beep += 1
		else
			pinput.ldist.push f
		end
	rescue
		pinput.beep += 1
	end
end

#--------------------------------------
#Calculation of geometry
#--------------------------------------			 				   
	
#Compute all shapes to be drawn as <shaepdraw> structure
def calculate_all_shapes()
	#Initialization
	@lst_shapedraw = []
	@mkcenter = nil
	return unless @parcours1
	
	#computing the origin
	compute_center
	
	#Double polygon unfinished
	return unless @single || @parcours2
	
	case @type		
		when CODE_Rectangle, CODE_Parallelogram	#Rectnagle, Parallelogram
			quadri_calculate_shapes		
		else	#Circle, Polygon, Ellipse, Circle 3P
			radial_calculate_shapes			
	end
end


#Compute the whole shape for Quadri shapes	
def quadri_calculate_shapes()	

	#Calculating angle for parallelogram
	compute_angle_sector if @type == CODE_Parallelogram
	
	#generating all shapes in rings
	([0.0] + @lst_rings).each do |delta|
		lst_vert =  quadri_calculate_single_shape delta
		next unless lst_vert
		shapedraw = generate_shapedraw(lst_vert, delta)
		@lst_shapedraw.push shapedraw
	end
end

#Compute a shape with 4 sides (Rectangle, parallelogram)
def quadri_calculate_single_shape(delta)	
	return nil if (@radius1 + delta < 0) || (@radius2 + delta < 0)
	
	angle = (@ortho) ? Math::PI * 0.5 : @angle_sector
	r1 = @radius1 + delta
	r2 = @radius2 + delta
	d1 = 2.0 * r1
	d2 = 2.0 * r2
	
	#Origin = Center
	parcours = Junction.to_distance @mkcenter, @vecref1, r1
	mkmid1 = parcours.last
	parcours = Junction.to_distance mkmid1, @vecref2.reverse, r2
	mkcorner1 = parcours.last	
		
	parcours = Junction.to_distance mkcorner1, @vecref2, d2
	mkcorner2 = parcours.last
		
	parcours = Junction.to_distance mkcorner2, @vecref1.reverse, d1
	mkcorner3 = parcours.last
		
	parcours = Junction.to_distance mkcorner3, @vecref2.reverse, d2
	mkcorner4 = parcours.last
	
	#Adjustments for forcing the input points
	return [mkcorner1, mkcorner2, mkcorner3, mkcorner4, mkcorner1]
	mkcorner1 = @parcours1.last if @param_axe2
	mkcorner2 = @parcours2.last if @param_axe1
	mkcorner4 = @parcours1.first if @param_axe1 && @param_axe2
	
	if !@param_axe1
		parcours = Junction.calculate mkcorner2, @parcours2.last
		vec = parcours[-2].pt.vector_to parcours.last.pt
		parcours = Junction.to_distance @parcours2.last, vec, r1
		mkcorner3 = parcours.last
	end
		
	return [mkcorner1, mkcorner2, mkcorner3, mkcorner4, mkcorner1]
end

#Calculate all shapes with Radial pattern
def radial_calculate_shapes()
	radial_shapes_from_center()
end

def radial_list_rays(nbseg, angle_deb, angle_total, ratio)
	angle_unit = angle_total / nbseg
	lst_ray = []
	for i in 0..nbseg
		angle = angle_deb + angle_unit * i
		x = Math.cos(angle)
		y = Math.sin(angle)
		dfactor = 1.0
		if (ratio != 1.0)
			y = y * ratio
			dfactor = dfactor * Math.sqrt(x * x + y * y)
			angle = Math.atan2(y, x)
		end	
		ray = TOS_ShapeRay.new
		ray.dfactor = dfactor
		ray.angle = angle
		ray.x = x
		ray.y = y
		lst_ray.push ray
	end
	lst_ray
end

#Compute the vertice when mark is imposed
def radial_force_mark (mark, delta)
	return mark if (delta == 0.0)
	parcours = Junction.calculate @mkcenter, mark	
	vec = parcours[-2].pt.vector_to parcours.last.pt
	parcours = Junction.to_distance mark, vec, delta
	parcours.last		
end

#Substitute imposed extremities
def radial_impose_extremities(lst_vert, delta, markdeb, markend)
	forced_mkbeg = radial_force_mark markdeb, delta
	forced_mkend = radial_force_mark markend, delta
	
	mkvert = lst_vert.first
	d1 = mkvert.pt.distance forced_mkbeg.pt		
	d2 = mkvert.pt.distance forced_mkend.pt	
	mkforced = (d1 <= d2) ? forced_mkbeg : forced_mkend
	lst_vert[0..0] = [mkforced]

	mkvert = lst_vert.last
	d1 = mkvert.pt.distance forced_mkbeg.pt		
	d2 = mkvert.pt.distance forced_mkend.pt	
	mkforced = (d1 <= d2) ? forced_mkbeg : forced_mkend
	lst_vert[-1..-1] = [mkforced]
end

def compute_angle_sector
	angle_total = @angle_at_origin
	v1 = @mark_origin.pt.vector_to @parcours1.last.pt
	v2 = @mark_origin.pt.vector_to @parcours2.last.pt
	ps = (v1 * v2) % @vcamera
	angle_total = (2 * Math::PI - angle_total) if (v1 * v2) % @vcamera > 0
	@angle_sector = angle_total
end

#Compute the whole shape for Radial shapes	
def radial_shapes_from_center()	
	#drawing arcs
	case @type
	when CODE_Arc	#Arcs
		angle_total = 2.0 * Math.atan2(@distance1 * 0.5, @radius1 - @distance2)
		angle_deb = -angle_total * 0.5
		ratio = 1.0
		markdeb = @parcours1.first
		markend = @parcours1.last
		nb = @nbseg
		if (@mkcenter.face && markdeb.face && markend.face) ||
		   (@mkcenter.face == nil && markdeb.face == nil && markend.face == nil)
			method_center = false
		else
			method_center = false
		end	


	when CODE_Sector	#Sectors
		#compute_angle_sector
		if (@trigo_sense)
			angle_total = @angle_sector
			angle_deb = 0
		else	
			angle_total = DEUX_PI - @angle_sector
			angle_deb = -angle_total
		end	
		markdeb = @parcours1.last
		markend = @parcours2.last
		ratio = 1.0
		nb = @nbseg * angle_total.abs / DEUX_PI
		nb = nb.to_i + 1
		method_center = true
		
	else	#circle by center, ellipse, polygon, circle 3P
		angle_total = 2 * Math::PI
		ratio = (@single || @distance2 == 0) ? 1.0 : (@radius2 / @radius1)
		angle_deb = 0.0
		markdeb = nil
		markend = nil
		nb = @nbseg
		method_center = true
	end	
	
	#Computing rays to vertices
	#lst_ray = radial_list_rays nb, angle_deb, angle_total, ratio
	
	#generating all shapes in rings
	if (method_center)
		radial_special_center_OK nb, angle_deb, angle_total, markdeb, markend
	else
		lst_ray = radial_list_rays nb, angle_deb, angle_total, ratio
		radial_special_center_nil lst_ray, markdeb, markend, angle_total, nb
	end	

	#Special treatment for Sectors to join borders
	arrange_sectors if @type == CODE_Sector

	return
end

#Compute the circular shapes when center has a face	
def radial_special_center_OK(nb, angle_deb, angle_total, markdeb, markend)	

	#generating all shapes in rings
	normal = compute_normal_at_mark @mkcenter
	normal = normal.reverse if normal % @vcamera > 0
	origin = @mkcenter.pt
	distance = @radius1
	([0.0] + @lst_rings).each do |delta|
		radius = distance + delta
		ratio = (@single || @distance2 == 0) ? 1.0 : ((@radius2 + delta) / (@radius1 + delta))
		next if radius <= 0.0
		lst_ray = radial_list_rays nb, angle_deb, angle_total, ratio
		lst_vert = []
		lst_ray.each do |ray|
			t = Geom::Transformation.rotation origin, normal, ray.angle
			vec = @vecref1.transform t
			parcours = Junction.to_distance @mkcenter, vec, ray.dfactor * radius
			mkvert = parcours.last
			lst_vert.push mkvert
		end	
		
		#Imposing vertices at extremities
		radial_impose_extremities lst_vert, delta, markdeb, markend if markdeb && markend
		
		#generating the Shapedraw structure
		shapedraw = generate_shapedraw lst_vert, delta
		@lst_shapedraw.push shapedraw
	end
end	

def oldradial_special_center_OK(lst_ray, markdeb, markend)	

	#generating all shapes in rings
	normal = compute_normal_at_mark @mkcenter
	normal = normal.reverse if normal % @vcamera > 0
	origin = @mkcenter.pt
	distance = @radius1
	([0.0] + @lst_rings).each do |delta|
		radius = distance + delta
		next if radius <= 0.0
		lst_vert = []
		lst_ray.each do |ray|
			t = Geom::Transformation.rotation origin, normal, ray.angle
			vec = @vecref1.transform t
			parcours = Junction.to_distance @mkcenter, vec, ray.dfactor * radius
			mkvert = parcours.last
			lst_vert.push mkvert
		end	
		
		#Imposing vertices at extremities
		radial_impose_extremities lst_vert, delta, markdeb, markend if markdeb && markend
		
		#generating the Shapedraw structure
		shapedraw = generate_shapedraw lst_vert, delta
		@lst_shapedraw.push shapedraw
	end
end	

#Compute the circular shapes when center has no face	
def radial_special_center_nil(lst_ray, markdeb, markend, angle_total, nb)	
	([0.0] + @lst_rings).each do |delta|
		#Computing the initial point
		radius = @radius1 + delta
		next if radius <= 0.0		#ring too small
		vecdir = @parcours2[-2].pt.vector_to @parcours2.last.pt
		if (delta != 0.0)
			parcours = Junction.to_distance @parcours2.last, vecdir, delta
			topmark = parcours.last
		else
			topmark = @parcours2.last
		end	
		
		angle_unit = angle_total / nb
		d0 = radius * angle_unit
		
		#Right quadran
		prev_mark = topmark
		prev_normal = absolute_normal_at_mark prev_mark
		prev_vec = vecdir * prev_normal
		lvert1 = []
		angledeb = angle_unit * 0.5
		nb2 = (nb / 2) - 1
		for i in 0..nb2
			t = Geom::Transformation.rotation prev_mark.pt, prev_normal, angledeb - angle_unit
			vec = prev_vec.transform t
			parcours = Junction.to_distance prev_mark, vec, d0
			prev_mark = parcours.last
			prev_normal = absolute_normal_at_mark prev_mark
			prev_vec = parcours[-2].pt.vector_to prev_mark.pt
			lvert1.push prev_mark
			angledeb = 0
		end	

		#Left quadrant
		prev_mark = topmark
		prev_normal = absolute_normal_at_mark prev_mark
		prev_vec = prev_normal * vecdir
		lvert2 = []
		angledeb = -angle_unit * 0.5
		nb2 = (nb / 2) - 1
		for i in 0..nb2
			t = Geom::Transformation.rotation prev_mark.pt, prev_normal, angledeb + angle_unit
			vec = prev_vec.transform t
			parcours = Junction.to_distance prev_mark, vec, d0
			prev_mark = parcours.last
			prev_normal = absolute_normal_at_mark prev_mark
			prev_vec = parcours[-2].pt.vector_to prev_mark.pt
			lvert2.push prev_mark
			angledeb = 0
		end	
		
		#Joining the parcours
		lst_vert = lvert2.reverse + [topmark] + lvert1
		radial_impose_extremities lst_vert, delta, markdeb, markend if delta == 0 && markdeb && markend
		
		#generating the Shapedraw structure
		shapedraw = generate_shapedraw lst_vert, delta
		@lst_shapedraw.push shapedraw
	end
end	

#Create Sectors
def arrange_sectors
	#Sorting the list of sectors
	lsd = @lst_shapedraw.sort { |s1, s2| -s1.delta <=> -s2.delta }
	
	#Joining the borders
	lsdnew = []
	nb = lsd.length - 1
	i = 0
	while (i <= nb)
		lsd1 = lsd[i].lmk_vertices
		lsd2 = (i == nb) ? [@mkcenter] : lsd[i+1].lmk_vertices
		lst_vert = lsd1 + lsd2.reverse + [lsd1.first]
		sd = generate_shapedraw lst_vert, lsd[i].delta
		lsdnew.push sd
		i += 2
	end	
	@lst_shapedraw = lsdnew	
end

end	#class TOSShape

	
end	#End Module SUToolsOnSurface
