/*
**      Source listing for the DEMO program.
**      Copyright Oracle, 1993-2008, all rights reserved.
**
**      This file implements all the Control related messages handling.
**      Part 1
*/

#include "stdafx.h"
#include <time.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#include "demo.h"
#include "define.h"
#include "demodoc.h"
#include "demoview.h"
#include "diags.h"
#include "msghndl.h"
#include "outwnd.h"

#undef MoveTo

#include "errortrc.h"

/*------------------------------------------------------------------------------
** Display a description of the given error and message.
*/
void CDemoView::ErrorTrace(int err, UINT msg)
{
	char* out_msg;

	if ((msg < PM_CTL_first) || (msg > PM_CTL_last)) {
		out_msg = "Unknown control message";
	} else {
		out_msg = msgType[msg - PM_CTL_first];
	}
	m_pWndOut->SendToOutput("ERROR %s: %s", out_msg, errType[err]);
}

/*------------------------------------------------------------------------------
** Send the given message to the currently selected region.
*/
BOOL CDemoView::SendMsg(UINT msg, WPARAM wParam, LPARAM lParam)
{
	int err = (int) ::SendMessage(m_hCurCtl, msg, wParam, lParam);
	if (err != PAN_CTLERRNONE) {
		ErrorTrace(err, msg);
		return FALSE;
	} else {
		char *out_msg;

		if ((msg < PM_CTL_first) || (msg > PM_CTL_last)) {
			out_msg = "Unknown control message";
			m_pWndOut->SendToOutput("SendMessage(%x) returns %x", msg, err);
		} else {
			out_msg = msgType[msg - PM_CTL_first];
			m_pWndOut->SendToOutput("SendMessage(%s) returns %x", out_msg, err);
		}
	}
	return TRUE;
}

/*------------------------------------------------------------------------------
** Retrieve the control handle of the currently selected region.
*/
BOOL CDemoView::Validate()
{
	CDemoDoc* pDoc = GetDocument();
	CPanCtRgn* pRgn = pDoc->GetCurrRgn();

	if (pRgn == NULL) {
		m_pWndOut->SendToOutput("No control selected!");
		return FALSE;
	}

	m_hCurCtl = pRgn->GetCtlHandle();
	if (m_hCurCtl == NULL) {
		m_pWndOut->SendToOutput("Selected control was not initialized!");
		return FALSE;
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Call functions for each PAN Control message

// --------------------------------------------------------------------------
// Clear all selections in the current file.
//
void CDemoView::OnSendmessagePmctlclearsels()
{
	if (! Validate()) {
		return;
	}

	if (! SendMsg(PM_CTLCLEARSELS, 0, 0)) {
		return;
	}
	m_pWndOut->SendToOutput("PM_CTLCLEARSELS: Done.");
}

// --------------------------------------------------------------------------
// Display the world coordinates (i.e. the position in the current file)
// corresponding to the given client area coordinates (i.e. relative to the
// topleft corner of the client area of the control window).
//
void CDemoView::OnSendmessagePmctlclienttoworld()
{
	if (! Validate()) {
		return;
	}

	PAN_CtlPos cc;
	{
		CPxyzDiag dlg;
		if (dlg.DoModal() != IDOK) {
			return;
		}
	    cc.x    = dlg.m_posx;
		cc.y    = dlg.m_posy;
		cc.z    = dlg.m_posz;
	}

	if (! SendMsg(PM_CTLCLIENTTOWORLD, 0, (LPARAM) (PAN_CtlPos FAR*) &cc)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLCLIENTTOWORLD: x = %.3lf y = %.3lf", cc.x, cc.y);
}

// --------------------------------------------------------------------------
// Create a duplicate of the control as an independent window.
// NOTE: This control CANNOT receive messages from the application.
//
void CDemoView::OnSendmessagePmctlclonecontrol()
{
	if (! Validate()) {
		return;
	}

	HWND handle = NULL;
	if (! SendMsg(PM_CTLCLONECONTROL, 0, (LPARAM)(HWND FAR*) &handle)) {
		return;
	}

	::ShowWindow(handle, SW_SHOW);
	m_pWndOut->SendToOutput("PM_CTLCLONECONTROL: Done.");
}

// --------------------------------------------------------------------------
// Copy the current selections to the clipboard using the clipboard formats
// specified in the given mask.
//
// For this message to work, a selection must be available in the PAN control
// This routine is called by the Control notification procedure and by
// OnSendmessagePmctlcopy().
//
void CDemoView::CopyToClpbrd()
{
	DWORD	clpbrdOptions = 0;

	if (!Validate()) {
		return;
	}

	{
		CClpbDiag	dlg;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		if (dlg.m_dib)          clpbrdOptions |= PAN_CTLCLPBRDDIB;
		if (dlg.m_bmap)         clpbrdOptions |= PAN_CTLCLPBRDBITMAP;
		if (dlg.m_pal)          clpbrdOptions |= PAN_CTLCLPBRDPALETTE;
		if (dlg.m_meta)         clpbrdOptions |= PAN_CTLCLPBRDMETAFILE;
		if (dlg.m_text)         clpbrdOptions |= PAN_CTLCLPBRDTEXT;
		if (dlg.m_rtf)          clpbrdOptions |= PAN_CTLCLPBRDRTF;
		if (dlg.m_heading)      clpbrdOptions |= PAN_CTLCLPBRDHEADINGS;
	}

	if (! SendMsg(PM_CTLCOPY, 0, clpbrdOptions)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLCOPY: Done.");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlcopy()
{
	if (! Validate()) {
		return;
	}

	CopyToClpbrd();
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetbasefont()
//
// Display the base font of the current file.
{
	if (! Validate()) {
		return;
	}

	LOGFONT fnt;
	if (! SendMsg(PM_CTLGETBASEFONT, 0, (LPARAM)(LPLOGFONT) &fnt)) {
		return;
	}
	m_pWndOut->SendToOutput("PM_CTLGETBASEFONT: font = %s / %d", fnt.lfFaceName, abs(fnt.lfHeight));
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetblock()
//
// Display the active block of the current file.
{
	if (!Validate()) {
		return;
	}

	LONG act_block;
	if (! SendMsg(PM_CTLGETBLOCK, 0, (LPARAM)(LONG FAR *) &act_block)) {
		return;
	}
	m_pWndOut->SendToOutput("PM_CTLGETBLOCK: block = %ld", act_block);
}

// --------------------------------------------------------------------------

void CDemoView::OnSendmessagePmctlgetblocknames()
//
// Display the names of the blocks set by the decoder.
{
	LONG num_blocks = 0;
	LONG i = 0;

	if (!Validate()) {
		return;
	}

	LPPAN_BLOCK blocks = new PAN_BLOCK[PAN_CTLMAXBLOCKS];
	num_blocks = (LONG) ::SendMessage(m_hCurCtl, PM_CTLGETBLOCKNAMES,
		PAN_CTLMAXBLOCKS, (LPARAM) blocks);

	m_pWndOut->SendToOutput("PM_CTLGETBLOCKNAMES: number of blocks = %ld", num_blocks);

	if (num_blocks > 0) {
		for (i = 0; i < num_blocks; i++) {
			m_pWndOut->SendToOutput("block %ld: %lx %s\r",
					i, (blocks + i)->id,
					(blocks + i)->name);
		}
	}
	delete blocks;
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetcaps()
//
// Display the currently enabled capabilities of the control.
{
	DWORD caps;

	if (! Validate()) {
		return;
	}
	if (! SendMsg(PM_CTLGETCAPS, 0, (LPARAM)(LPDWORD) &caps)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETCAPS:");
	m_pWndOut->SendToOutput("Zoom   : %s", (caps & PAN_CTLCAPSZOOM)    == PAN_CTLCAPSZOOM    ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Copy   : %s", (caps & PAN_CTLCAPSCOPY)    == PAN_CTLCAPSCOPY    ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Search : %s", (caps & PAN_CTLCAPSSEARCH)  == PAN_CTLCAPSSEARCH  ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Page   : %s", (caps & PAN_CTLCAPSPAGE)    == PAN_CTLCAPSPAGE    ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Size   : %s", (caps & PAN_CTLCAPSSIZE)    == PAN_CTLCAPSSIZE    ? "ON" : "OFF");
	m_pWndOut->SendToOutput("HScroll: %s", (caps & PAN_CTLCAPSHSCROLL) == PAN_CTLCAPSHSCROLL ? "ON" : "OFF");
	m_pWndOut->SendToOutput("VScroll: %s", (caps & PAN_CTLCAPSVSCROLL) == PAN_CTLCAPSVSCROLL ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Mouse  : %s", (caps & PAN_CTLCAPSMOUSE)   == PAN_CTLCAPSMOUSE   ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Keybd  : %s", (caps & PAN_CTLCAPSKEYBD)   == PAN_CTLCAPSKEYBD   ? "ON" : "OFF");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetclpbrdfmts()
//
// Display a description of each clipboard format supported by the control.
{
	if (! Validate()) {
		return;
	}

	PAN_CtlClpbrdFmtList fl;
	if (!SendMsg(PM_CTLGETCLPBRDFMTS, 0, (LPARAM)(PAN_CtlClpbrdFmtList FAR*) &fl)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETCLPBRDFMTS");
	m_pWndOut->SendToOutput("-------------------");

	PAN_CtlClpbrdFmt FAR* cf = (PAN_CtlClpbrdFmt FAR*) GlobalLock(fl.hFmts);
	for (WORD i = 0; i < fl.nFmts; i++) {
		char desc[PAN_CTLMAXDESC];

		_fstrcpy(desc, cf[i].desc);
		if (strcmp(desc,"")!=0) m_pWndOut->SendToOutput("%s", desc);
	}
	GlobalUnlock(fl.hFmts);
	GlobalFree(fl.hFmts);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetcolwidth()
//
// Display the width (in pixels) of the given field/column in the
// current page.
{
	LONG	num;
	WORD	unitCode;

	if (!Validate()) {
		return;
	}

	// Get field/colum value.
	{
		CRowCol	dlg;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		num = (LONG)dlg.m_entityNumber;
		unitCode = (WORD)dlg.unitCode;
	}
	WORD ret = (WORD)::SendMessage(m_hCurCtl, PM_CTLGETCOLWIDTH,
							(WPARAM) unitCode, (LPARAM)num);
	if (ret == -1) {
		ErrorTrace(PAN_GetCtlErrorCode(), PM_CTLGETCOLWIDTH);
		return;
	} else {
		m_pWndOut->SendToOutput("PM_CTLGETCOLWIDTH: width = %d", ret);
	}
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetfgbgcolor()
//
// Display the current foreground / background color.
{
	if (! Validate()) {
		return;
	}

	COLORREF fg, bg;
	if (! SendMsg(PM_CTLGETFGBGCOLOR, 0, (LPARAM)(COLORREF FAR*) &fg)) {
		return;
	}
	if (! SendMsg(PM_CTLGETFGBGCOLOR, 1, (LPARAM)(COLORREF FAR*) &bg)) {
		return;
	}

	BYTE r[2], g[2], b[2];
	r[0] = GetRValue(fg); r[1] = GetRValue(bg);
	g[0] = GetGValue(fg); g[1] = GetGValue(bg);
	b[0] = GetBValue(fg); b[1] = GetBValue(bg);

	m_pWndOut->SendToOutput("PM_CTLGETFGBGCOLOR:");
	m_pWndOut->SendToOutput("foreground = (%3d, %3d, %3d)", r[0], g[0], b[0]);
	m_pWndOut->SendToOutput("background = (%3d, %3d, %3d)", r[1], g[1], b[1]);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetfile()
//
// Display information about the current file.
{
	if (! Validate()) {
		return;
	}

	PAN_CtlFileInfo fi;
	if (! SendMsg(PM_CTLGETFILE, 0, (LPARAM)(PAN_CtlFileInfo FAR*) &fi)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETFILE:\r");

	switch (fi.type) {
	  case PAN_RasterFile      : m_pWndOut->SendToOutput("type: RASTER FILE");      break;
	  case PAN_VectorFile      : m_pWndOut->SendToOutput("type: VECTOR FILE");      break;
	  case PAN_DatabaseFile    : m_pWndOut->SendToOutput("type: DATABASE FILE");    break;
	  case PAN_SpreadsheetFile : m_pWndOut->SendToOutput("type: SPREADSHEET FILE"); break;
	  case PAN_DocumentFile    : m_pWndOut->SendToOutput("type: DOCUMENT FILE");    break;
	  case PAN_ArchiveFile     : m_pWndOut->SendToOutput("type: ARCHIVE FILE");     break;
	} /* switch */

	m_pWndOut->SendToOutput("name: %s",  fi.name);
	m_pWndOut->SendToOutput("size: %ld", fi.size);

	time_t currentDate = fi.date;
	m_pWndOut->SendToOutput("date: %s",  ctime(&currentDate));
	m_pWndOut->SendToOutput("desc: %s",  fi.desc);

	// display the dimensions
	PAN_CtlRange* rg = &fi.dimensions;
	switch (fi.type) {
	   case PAN_VectorFile:
		m_pWndOut->SendToOutput("dim : %.2lf x %.2lf x %.2lf\r",
				rg->max.x - rg->min.x,  rg->max.y - rg->min.y, rg->max.z - rg->min.z);
		break;
	   case PAN_RasterFile:
		m_pWndOut->SendToOutput("dim : %.2lf x %.2lf\r",
				rg->max.x - rg->min.x,  rg->max.y - rg->min.y);
		break;
	   default:
		m_pWndOut->SendToOutput("dim : %.2lf x %.2lf\r",
				rg->max.x - rg->min.x , rg->max.y - rg->min.y);
	}

	m_pWndOut->SendToOutput("colr: %d (bits)", fi.colorDepth);
	m_pWndOut->SendToOutput("nbPg: %d", fi.nPages);
	if ( fi.type == PAN_RasterFile ) {
		m_pWndOut->SendToOutput("tiles: %d x %d", fi.tilex, fi.tiley);
	}
	if ( fi.ins.offset.x !=0 || fi.ins.offset.y !=0 || fi.ins.offset.z != 0 ||
		 fi.ins.scale.x !=0 || fi.ins.scale.y !=0 || fi.ins.scale.z != 0
	   ) {
		m_pWndOut->SendToOutput("Insertion data:\n");
		m_pWndOut->SendToOutput("     Point: (%.2lf, %.2lf, %.2lf)",
				fi.ins.offset.x, fi.ins.offset.y, fi.ins.offset.z);
		m_pWndOut->SendToOutput("     Scaling: (%.2lf, %.2lf, %.2lf)",
				fi.ins.scale.x, fi.ins.scale.y, fi.ins.scale.z);
		m_pWndOut->SendToOutput("     Resolution: (%.2lf, %.2lf, %.2lf)",
				fi.ins.dpi.x, fi.ins.dpi.y, fi.ins.dpi.z);
		m_pWndOut->SendToOutput("     Rotation: %.2lf radians",
				fi.ins.rot);
	}
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetfilefmts()
//
// Display a description of each file format supported by the control.
{
	if (! Validate()) {
		return;
	}

	PAN_CtlFileFmtList fl;
	if (! SendMsg(PM_CTLGETFILEFMTS, 0, (LPARAM)(PAN_CtlFileFmtList FAR*) &fl)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETFILEFMTS");
	m_pWndOut->SendToOutput("-----------------");

	PAN_CtlFileFmt FAR* f = (PAN_CtlFileFmt FAR*) GlobalLock(fl.hFmts);
	for (WORD i = 0; i < fl.nFmts; i++) {
		char desc[PAN_CTLMAXDESC];
		char exts[PAN_CTLMAXEXTS];

		_fstrcpy(desc, f[i].desc);
		_fstrcpy(exts, f[i].exts);
		m_pWndOut->SendToOutput("%s (%s)", desc, exts);
	}
	GlobalUnlock(fl.hFmts);
	GlobalFree(fl.hFmts);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetimageex0()
//
// Display information about the image options being used by a monochrome
// raster control.
{
	if (! Validate()) {
		return;
	}

	int     val;
	if (!SendMsg(PM_CTLGETIMAGEEX, PAN_IMAGE_CONTRAST, (LPARAM)(LPINT) &val)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETIMAGEEX: contrast = %d\r", val);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetimageex1()
//
// Display information about the image options being used by a monochrome
// raster control.
{
	if (! Validate()) {
		return;
	}

	BOOL    val;
	if (!SendMsg(PM_CTLGETIMAGEEX, PAN_IMAGE_ANTIALIAS, (LPARAM)(BOOL FAR *) &val)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETIMAGEEX: antialias %s\r",
		(val ? "ON" : "OFF"));
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetimageex2()
//
// Display information about the image options being used by a monochrome
// raster control.
{
	if (! Validate()) {
		return;
	}

	BOOL    val;
	if (!SendMsg(PM_CTLGETIMAGEEX, PAN_IMAGE_INVERT, (LPARAM)(BOOL FAR *) &val)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETIMAGEEX: invert %s\r",
		(val ? "ON" : "OFF"));
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetinfo()
//
// Display information about the control.
{
	if (! Validate()) {
		return;
	}

	PAN_CtlInfo fi;
	if (!SendMsg(PM_CTLGETINFO, 0, (LPARAM)(PAN_CtlFileInfo FAR*) &fi)) {
		return;
	}

	switch (fi.type) {
	  case PAN_RasterFile    :
		m_pWndOut->SendToOutput("PM_CTLGETINFO: type: RASTER FILE, version: %d", fi.version);
		break;
	  case PAN_VectorFile    :
		m_pWndOut->SendToOutput("PM_CTLGETINFO: type: VECTOR FILE, version: %d", fi.version);
		break;
	  case PAN_DatabaseFile   :
		m_pWndOut->SendToOutput("PM_CTLGETINFO: type: DATABASE FILE, version: %d", fi.version);
		break;
	  case PAN_SpreadsheetFile:
		m_pWndOut->SendToOutput("PM_CTLGETINFO: type: SPREADSHEET FILE, version: %d", fi.version);
		break;
	  case PAN_DocumentFile   :
		m_pWndOut->SendToOutput("PM_CTLGETINFO: type: DOCUMENT FILE, version: %d", fi.version);
		break;
	  case PAN_ArchiveFile  :
		m_pWndOut->SendToOutput("PM_CTLGETINFO: type: ARCHIVE FILE, version: %d", fi.version);
		break;
	} /* switch */
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetlayerstate()
//
// Display the states of the layers set by the decoder.
{
	if (!Validate()) {
		return;
	}

	// First ask number of layers
	LONG num_layers = 0;
	num_layers = (LONG) ::SendMessage(m_hCurCtl, PM_CTLGETLAYERSTATE, 0, NULL);

	if (num_layers < 0) {
		m_pWndOut->SendToOutput("PM_CTLGETLAYERSTATE: No layers used in file format");
		return;
	}

	// Get all layers
	PAN_LAYER FAR *layers = NULL;

	if (num_layers > 0) {
		layers = new PAN_LAYER[num_layers];
	}

	LONG i = 0;

	num_layers = (LONG) ::SendMessage(m_hCurCtl, PM_CTLGETLAYERSTATE,
		num_layers, (LPARAM) layers);

	m_pWndOut->SendToOutput("PM_CTLGETLAYERSTATE: number of layers = %ld", num_layers);

	if (num_layers > 0) {
		for (i = 0; i < num_layers; i++) {
			m_pWndOut->SendToOutput("layer %ld: %s(id %lx) %s, (color %lx)\r",
					i, (layers + i)->name, (layers + i)->id,
					((layers + i)->bState ? "ON" : "OFF"),
					(LONG) (layers + i)->color);
		}
	}
	delete layers;
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetlmbaction()
//
// Display the action taken when clicking and dragging the with left mouse.
{
	if (! Validate()) {
		return;
	}

	WORD lmba;
	if (! SendMsg(PM_CTLGETLMBACTION, 0, (LPARAM)(LPWORD) &lmba)) {
		return;
	}

	if (lmba == PAN_CTLLMBNONE)     m_pWndOut->SendToOutput("PM_CTLGETLMBACTION: none");   else
	if (lmba == PAN_CTLLMBSELECT)   m_pWndOut->SendToOutput("PM_CTLGETLMBACTION: select"); else
	if (lmba == PAN_CTLLMBZOOM)     m_pWndOut->SendToOutput("PM_CTLGETLMBACTION: zoom");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetmode()
//
// Display the currently enabled modes of the control.
{
	if (! Validate()) {
		return;
	}

	DWORD mode;
	if (!SendMsg(PM_CTLGETMODE, 0, (LPARAM)(LPDWORD) &mode)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETMODE:");
	m_pWndOut->SendToOutput("Opaque   : %s", (mode & PAN_CTLMODEOPAQUE)       != 0 ? "ON" : "OFF");
	m_pWndOut->SendToOutput("NoRedraw : %s", (mode & PAN_CTLMODENOREDRAW)     != 0 ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Ex Scrl  : %s", (mode & PAN_CTLMODEEXCESSSCROLL) != 0 ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Aniso    : %s", (mode & PAN_CTLMODEANISOTROPIC)  != 0 ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Drg&drp  : %s", (mode & PAN_CTLMODEDRAGDROP)     != 0 ? "ON" : "OFF");
	m_pWndOut->SendToOutput("Interrupt: %s", (mode & PAN_CTLMODEINTERRUPTIBLE)!= 0 ? "ON" : "OFF");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetnumpages()
//
// Display the number of pages in the current file.
{
	if (! Validate()) {
		return;
	}

	int	pages;
	if (! SendMsg(PM_CTLGETNUMPAGES, 0, (LPARAM)(LPWORD) &pages)) {
		return;
	}
	m_pWndOut->SendToOutput("PM_CTLGETNUMPAGES: pages = %d", pages);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetnumsels()
//
// Display the current number of selections.
{
	if (! Validate()) {
		return;
	}

	WORD numsel;
	if (! SendMsg(PM_CTLGETNUMSELS, 0, (LPARAM)(LPWORD) &numsel)) {
		return;
	}
	m_pWndOut->SendToOutput("PM_CTLGETNUMSELS: #sels = %d", numsel);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetoffset()
//
// Display the offset of the origin for the current file in world coord.
{
	if (! Validate()) {
		return;
	}

	PAN_CtlPos fp;
	if (! SendMsg(PM_CTLGETOFFSET, 0, (LPARAM)(PAN_CtlPos FAR*) &fp)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETOFFSET: x=%.3lf y=%.3lf\r", fp.x, fp.y);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetpage()
{
	if (! Validate()) {
		return;
	}

	WORD page;
	if (! SendMsg(PM_CTLGETPAGE, 0, (LPARAM)(LPWORD) &page)) {
		return;
	}
	m_pWndOut->SendToOutput("PM_CTLGETPAGE: page = %d", page);
}

// --------------------------------------------------------------------------
// Display the size of the current page in world coordinates.
//
void CDemoView::OnSendmessagePmctlgetpagesize()
{
	int		nPage;
	if (!Validate()) {
		return;
	}

	{
        CPageDiag dlg;

		// Get current page.
        if (!SendMsg(PM_CTLGETPAGE, 0, (LPARAM) &dlg.m_page)) {
                return;
        }

        if (dlg.DoModal() != IDOK) {
                return;
        }

        nPage = dlg.m_page;
	}

	PAN_CtlRange	rg;

	if (!SendMsg(PM_CTLGETPAGESIZE,
		nPage, (LPARAM)(PAN_CtlRange FAR*) &rg)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETPAGESIZE: page # %d - (%.3lf, %.3lf) x (%.3lf, %.3lf)\r",
		nPage, rg.min.x, rg.min.y, rg.max.x, rg.max.y);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetresourceinfostate()
//
// Display the states of the layers set by the decoder.
{
	if (!Validate()) {
		return;
	}

	LONG num_resinfo = 0;
	LONG i = 0;

	num_resinfo = (LONG) ::SendMessage(m_hCurCtl, PM_CTLGETRESOURCEINFOSTATE,
		(WPARAM) 0, (LPARAM) NULL);

	if (num_resinfo <= 0) {
		m_pWndOut->SendToOutput("PM_CTLGETRESOURCEINFOSTATE: no info");
		return;
	}

	PAN_RESOURCEINFO _huge *resInfo = new PAN_RESOURCEINFO[num_resinfo];

	::SendMessage(m_hCurCtl, PM_CTLGETRESOURCEINFOSTATE,
		(WPARAM) num_resinfo, (LPARAM) resInfo);

	for (i = 0; i < num_resinfo; i++) {
		char *typeName;

		switch ((resInfo+i)->resourceType) {
		case RI_TEXTFONT:
			typeName = "Text Font";
			break;

		case RI_LINESTYLE:
			typeName = "Line Style";
			break;

		case RI_SHAPE:
			typeName = "Shape File";
			break;

		case RI_VECTOR_EXTERNAL_REFERENCE:
			typeName = "External Reference File";
			break;

		case RI_RASTER_EXTERNAL_REFERENCE:
			typeName = "External Reference File";
			break;

		default:
			typeName = "Unknown";
			break;
		}
		m_pWndOut->SendToOutput("%s: %s -> %s, (%s)\n",
					typeName,
					(resInfo + i)->szResourceID,
					*((resInfo + i)->szResolvedResource) == '\0' ?
					(resInfo + i)->szResource :
					(resInfo + i)->szResolvedResource,
					((resInfo + i)->fOK ? ((resInfo + i)->fOK == 1 ? "LOCATED" : "SUBSTITUTED") : "MISSING"));
	}
	delete resInfo;
}

// --------------------------------------------------------------------------
// Display the height in pixels of the given record/row in the current page
//
void CDemoView::OnSendmessagePmctlgetrowheight()
{
	LONG	num;
	WORD	unitCode;

	if (!Validate()) {
		return;
	}

	// Get record/row value
	{
		CRowCol	 dlg;

		if (dlg.DoModal()!=IDOK) {
			return;
		}

		num = (LONG) dlg.m_entityNumber;
		unitCode = (WORD) dlg.unitCode;
	}

	WORD ret = (WORD)::SendMessage(m_hCurCtl, PM_CTLGETROWHEIGHT,
								(WPARAM)unitCode, (LPARAM)num);
	if (ret == -1) {
		ErrorTrace(PAN_GetCtlErrorCode(), PM_CTLGETROWHEIGHT);
		return;
	} else {
		m_pWndOut->SendToOutput("PM_CTLGETROWHEIGHT: height = %d", ret);
	}
}

// --------------------------------------------------------------------------
// Display the list of current selections in the file.
//
void CDemoView::OnSendmessagePmctlgetsels()
{
	if (! Validate()) {
		return;
	}

	PAN_CtlSelList	sl;

	if (! SendMsg(PM_CTLGETSELS, 0, (LPARAM)(PAN_CtlSelList FAR*) &sl)) {
		return;
	}

	if (sl.nSels == 0) {
		m_pWndOut->SendToOutput("PM_CTLGETSELS: No selections available.");
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETSELS");
	m_pWndOut->SendToOutput("-------------");

	PAN_CtlSel FAR* s = (PAN_CtlSel FAR*) GlobalLock(sl.hSels);
	for (WORD i = 0; i < sl.nSels; i++) {
		if (s[i].selType == PAN_CTLSELECTION_VIEW) {
			PAN_CtlRange FAR* rg = &s[i].range.vwRange;
			m_pWndOut->SendToOutput("sel: %u x = %.3lf y = %.3lf  w = %.3lf h = %.3lf", i,
				rg->min.x, rg->min.y, rg->max.x - rg->min.x, rg->max.y - rg->min.y);
		} else
		if (s[i].selType == PAN_CTLSELECTION_CARET) {
			PAN_CtlCaretRange FAR* rg = &s[i].range.ctRange;
			m_pWndOut->SendToOutput("sel: %u  FROM page = %d flow=%ld offs=%ld",
						i, rg->from.page, rg->from.flow, rg->from.offset);
			m_pWndOut->SendToOutput("         TO   page = %d flow=%ld offs=%ld",
						rg->to.page, rg->to.flow, rg->to.offset);
		}
		else {
			m_pWndOut->SendToOutput("Unrecognized type!\r");
		}
	} // for

	GlobalUnlock(sl.hSels);
	GlobalFree(sl.hSels);
}

// --------------------------------------------------------------------------
// Display the current status of the control.
//
void CDemoView::OnSendmessagePmctlgetstatus()
{
	if (! Validate()) {
		return;
	}

	DWORD	stat;

	if (! SendMsg(PM_CTLGETSTATUS, 0, (LPARAM)(LPDWORD) &stat)) {
		return;
	}

	if (stat & PAN_CTLSTATUSPROCESSING)
		m_pWndOut->SendToOutput("PM_CTLGETSTATUS: Processing...");

	else if (stat & PAN_CTLSTATUSREFRESHING)
		m_pWndOut->SendToOutput("PM_CTLGETSTATUS: Refreshing...");

	else if (stat & PAN_CTLSTATUSREGENERATING)
		m_pWndOut->SendToOutput("PM_CTLGETSTATUS: Reading...");

	else
		m_pWndOut->SendToOutput("PM_CTLGETSTATUS: Idle...");
}

// --------------------------------------------------------------------------
// Display the current view of the vector file being displayed.
//
void CDemoView::OnSendmessagePmctlgetview()
{
	if (! Validate()) {
		return;
	}

	PAN_VIEW	cur_view;

	if (! SendMsg(PM_CTLGETVIEW, 0, (LPARAM)(PAN_VIEW FAR *) &cur_view)) {
		return;
	}

	m_pWndOut->SendToOutput("GETVIEW: %s(%lx) min pt = (%.3lf %.3lf) max pt = (%.3lf h=%.3lf)\r",
			cur_view.name, cur_view.id,
			cur_view.vmin.x, cur_view.vmin.y,
			cur_view.vmax.x, cur_view.vmax.y);
}

// --------------------------------------------------------------------------
// Display the current view extents of the file in world coordinates.
//
void CDemoView::OnSendmessagePmctlgetviewextents()
{
	if (! Validate()) {
		return;
	}

	PAN_CtlRange	rg;

	if (! SendMsg(PM_CTLGETVIEWEXTENTS, 0, (LPARAM)(PAN_CtlRange FAR*) &rg)) {
		return;
	}

	m_pWndOut->SendToOutput("GETVIEWEXTENTS:");
	m_pWndOut->SendToOutput("     min = (%.3lf, %.3lf)  max = (%.3lf, %.3lf)\r",
			rg.min.x, rg.min.y, rg.max.x , rg.max.y);
	m_pWndOut->SendToOutput("     w=%.3lf h=%.3lf\r",
			rg.max.x - rg.min.x, rg.max.y - rg.min.y);
}

// --------------------------------------------------------------------------
// Display the names of the views found by the decoder.
//
void CDemoView::OnSendmessagePmctlgetviewnames()
{
	if (!Validate()) {
		return;
	}

	LPPAN_VIEW	views = new PAN_VIEW[PAN_CTLMAXVIEWS];
	LONG		num_views = 0;
	LONG		i = 0;

	num_views = (LONG) ::SendMessage(m_hCurCtl, PM_CTLGETVIEWNAMES,
		PAN_CTLMAXVIEWS, (LPARAM) views);

	m_pWndOut->SendToOutput("PM_CTLGETVIEWNAMES: number of views = %ld", num_views);

	if (num_views > 0) {
		for (i = 0; i < num_views; i++) {
			m_pWndOut->SendToOutput("view %ld: %s(id %lx) min pt = (%.3lf %.3lf) max pt = (%.3lf h=%.3lf)\r",
					i, (views + i)->name, (views + i)->id,
					(views + i)->vmin.x, (views + i)->vmin.y,
					(views + i)->vmax.x, (views + i)->vmax.y);
		}
	}

	delete views;
}

// --------------------------------------------------------------------------
// Display the zoom factor for the current file
//
void CDemoView::OnSendmessagePmctlgetzoom()
{
	if (!Validate()) {
		return;
	}

	LPDWORD	fact = new DWORD[2];

	if (!SendMsg(PM_CTLGETZOOM, PAN_CTLZOOMX, (LPARAM) fact)) return;
	if (!SendMsg(PM_CTLGETZOOM, PAN_CTLZOOMY, (LPARAM) (fact + 1))) return;

	m_pWndOut->SendToOutput("PM_CTLGETZOOM:\r");
	m_pWndOut->SendToOutput("zoom X : %.3f...\r", ((float) (*fact) / 1000.0));
	m_pWndOut->SendToOutput("zoom Y : %.3f...\r", ((float) *(fact + 1) / 1000.0));
	delete fact;
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlprint()
{
	if (!Validate())
		return;

	m_pWndOut->SendToOutput("PRINT: printing...\r");
	OnFilePrintctl();
}


// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlregen()
{
	if (!Validate())
		return;

	if (!SendMsg(PM_CTLREGEN, 0, (LPARAM) 0))
		return;

	CDemoDoc *doc = (CDemoDoc *) GetDocument();
	if (doc->autoRepaint)
			SendMsg(PM_CTLPAINT, (WPARAM)0, (LPARAM)0);

	m_pWndOut->SendToOutput("REGEN: regenerating display...\r");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsearch()
{
	m_pWndOut->SendToOutput("PM_CTLSEARCH...");
	OnEditSearch();
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetstring()
{
	m_pWndOut->SendToOutput("PM_CTLGETSTRING...");
	OnEditGetString();
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetentity()
{
	m_pWndOut->SendToOutput("PM_CTLGETENTITY...");
	OnEditGetEntity();
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetbasefont()
{
	if (!Validate()) {
		return;
	}

	LOGFONT	fnt;

	if (!SendMsg(PM_CTLGETBASEFONT, 0, (LPARAM)(LPLOGFONT) &fnt)) {
		return;
	}

	CFontDialog	dlg(&fnt);
	if (dlg.DoModal() != IDOK) {
		return;
	}

	if (! SendMsg(PM_CTLSETBASEFONT, 0, (LPARAM)(LOGFONT FAR*) &fnt)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETBASEFONT: Done.");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetcaps()
//
// Enable the capabilities specified by the user from the dialog.
{
	if (! Validate()) {
		return;
	}

	DWORD	caps;
	if (!SendMsg(PM_CTLGETCAPS, 0, (LPARAM) (LPDWORD) &caps)) {
		return;
	}

	{
		CCapsDiag	dlg;

		dlg.m_copy      = (caps & PAN_CTLCAPSCOPY)    != 0;
		dlg.m_hscroll   = (caps & PAN_CTLCAPSHSCROLL) != 0;
		dlg.m_keybd     = (caps & PAN_CTLCAPSKEYBD)   != 0;
		dlg.m_mouse     = (caps & PAN_CTLCAPSMOUSE)   != 0;
		dlg.m_page      = (caps & PAN_CTLCAPSPAGE)    != 0;
		dlg.m_search    = (caps & PAN_CTLCAPSSEARCH)  != 0;
		dlg.m_size      = (caps & PAN_CTLCAPSSIZE)    != 0;
		dlg.m_vscroll   = (caps & PAN_CTLCAPSVSCROLL) != 0;
		dlg.m_zoom      = (caps & PAN_CTLCAPSZOOM)    != 0;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		if (dlg.m_copy)         caps |= PAN_CTLCAPSCOPY;        else caps &= ~PAN_CTLCAPSCOPY;
		if (dlg.m_hscroll)      caps |= PAN_CTLCAPSHSCROLL;     else caps &= ~PAN_CTLCAPSHSCROLL;
		if (dlg.m_keybd)        caps |= PAN_CTLCAPSKEYBD;       else caps &= ~PAN_CTLCAPSKEYBD;
		if (dlg.m_mouse)        caps |= PAN_CTLCAPSMOUSE;       else caps &= ~PAN_CTLCAPSMOUSE;
		if (dlg.m_page)         caps |= PAN_CTLCAPSPAGE;        else caps &= ~PAN_CTLCAPSPAGE;
		if (dlg.m_search)       caps |= PAN_CTLCAPSSEARCH;      else caps &= ~PAN_CTLCAPSSEARCH;
		if (dlg.m_size)         caps |= PAN_CTLCAPSSIZE;        else caps &= ~PAN_CTLCAPSSIZE;
		if (dlg.m_vscroll)      caps |= PAN_CTLCAPSVSCROLL;     else caps &= ~PAN_CTLCAPSVSCROLL;
		if (dlg.m_zoom)         caps |= PAN_CTLCAPSZOOM;        else caps &= ~PAN_CTLCAPSZOOM;
	}

	if (! SendMsg(PM_CTLSETCAPS, 0, caps)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETCAPS: Done.");

	CPanCtRgn* pRgn = GetDocument()->GetCurrRgn();
	pRgn->SetCurCaps();
}

// --------------------------------------------------------------------------
// Set the color values for the current control and all subsequent
// created controls. The values set here remains valid until the
// next call of this function.
//
void CDemoView::OnSendmessagePmctlsetfgbgcolor()
{
	if (! Validate()) {
		return;
	}

	COLORREF	fgColor;

	if (! SendMsg(PM_CTLGETFGBGCOLOR, 0, (LPARAM) (COLORREF FAR*) &fgColor)) {
		return;
	}

	COLORREF	bgColor;

	if (! SendMsg(PM_CTLGETFGBGCOLOR, 1, (LPARAM) (COLORREF FAR*) &bgColor)) {
		return;
	}

	// Dialog scope
	{
		CColrDiag dlg;
		dlg.m_fgR = GetRValue(fgColor);
		dlg.m_fgG = GetGValue(fgColor);
		dlg.m_fgB = GetBValue(fgColor);
		dlg.m_bgR = GetRValue(bgColor);
		dlg.m_bgG = GetGValue(bgColor);
		dlg.m_bgB = GetBValue(bgColor);

		if (dlg.DoModal() != IDOK) {
			return;
		}

		fgColor = RGB(dlg.m_fgR, dlg.m_fgG, dlg.m_fgB);
		bgColor = RGB(dlg.m_bgR, dlg.m_bgG, dlg.m_bgB);
	}

	if (! SendMsg(PM_CTLSETFGBGCOLOR, 0, (LPARAM)(COLORREF FAR*) &fgColor)) {
		return;
	}

	if (! SendMsg(PM_CTLSETFGBGCOLOR, 1, (LPARAM)(COLORREF FAR*) &bgColor)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETFGBGCOLOR: Done.");

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM)0, (LPARAM)0);
}

// --------------------------------------------------------------------------
// Load a new file in the current control.
// This function make a call to the GetFile() procedure defined in the
// demoview.cpp file.
//
void CDemoView::OnSendmessagePmctlsetfile()
{
	if (! Validate()) {
		return;
	}

	static char fileName[MAX_FILENAME];

	if (GetFile(fileName)) {
		CDemoDoc*	pDoc = GetDocument();
		CPanCtRgn*	pRgn = pDoc->FindRgn(m_hCurCtl);

		if (pRgn == NULL) {
			return;
		}

		pRgn->LoadCtrlData(fileName);
		if (PAN_GetCtlErrorCode() == PAN_CTLERRNONE) {
			m_pWndOut->SendToOutput("PM_CTLSETFILE: Done.");
		}

        /*
        ** 21.02.1995: Select the region and set focus
        */
		pDoc->Select(pRgn);

		/*
		**	05.05.1995: Try to paint after focus setting
		*/
		CDemoDoc	*doc = (CDemoDoc *) GetDocument();

		if (doc->autoRepaint)
			SendMsg(PM_CTLPAINT, (WPARAM)0, (LPARAM)0);
	}
}

/*------------------------------------------------------------------------------
** Set the action taken when clicking and dragging with the left mouse
** button in a control window.
**
** NOTE: These settings can be overriden by other application functions.
*/
void CDemoView::OnSendmessagePmctlsetlmbaction()
{
	if (! Validate()) {
		return;
	}

	WORD	lmba;
	if (!SendMsg(PM_CTLGETLMBACTION, 0, (LPARAM) (LPWORD) &lmba)) {
		return;
	}

	int def = 0;

	switch (lmba) {
	case PAN_CTLLMBNONE:
	default:
		def = 0;
		break;

	case PAN_CTLLMBSELECT:
		def = 1;
		break;

	case PAN_CTLLMBZOOM:
		def = 2;
		break;
	}

	{
		CLmbDiag	dlg;

		dlg.Default(def);

		if (dlg.DoModal() != IDOK) {
			return;
		}

		switch (dlg.lmb) {
		case 0:
			lmba = PAN_CTLLMBNONE;
			break;

		case 1:
			lmba = PAN_CTLLMBSELECT;
			break;

		case 2:
			lmba = PAN_CTLLMBZOOM;
			break;
		}
	}

	if (!SendMsg(PM_CTLSETLMBACTION, lmba, 0L)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETLMBACTION: Done.");

	CPanCtRgn* pRgn = GetDocument()->GetCurrRgn();
	pRgn->SetCurLMB();
}


/*----------------------------------------------------------------------------*/
void CDemoView::OnSendmessagePmctlmultisearch()
{
#if 0
// This is obsolete code since the PM_CTLSEARCH now support the MultiSearch.
/*
	if (! Validate()) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLMULTISEARCH...");

	CSrchDiag                       dlg;
	static PAN_CtlSearchInfo        si;
	PAN_CtlFileInfo                 fi;

	SendMsg(PM_CTLGETFILE, 0, (LPARAM)(PAN_CtlFileInfo FAR*) &fi);
	if (fi.type == PAN_RasterFile || fi.type == PAN_VectorFile) {
		m_pWndOut->SendToOutput("Current Control does not support string search!");
		return;
	}

	dlg.m_direction_up      = si.fDown == FALSE;
	dlg.m_wrap              = si.fWrap;
	dlg.m_matchCase         = si.fCase;
	dlg.m_matchWord         = si.fWord;

	// get search string and initialize search structure
	if (dlg.DoModal() == IDOK) {
		::UpdateWindow(m_hCurCtl);

		memset(&si.startPos, 0, sizeof(PAN_CtlCaretPos));
		SendMsg(PM_CTLGETCARETPOS, 0, (LPARAM)(PAN_CtlCaretPos FAR*)&si.startPos);

		si.fDown = dlg.m_direction_up == FALSE; // search downward
		si.fWrap = dlg.m_wrap;                  // wrap around file
		si.fCase = dlg.m_matchCase;             // case sensitive
		si.fWord = dlg.m_matchWord;             // whole word
		si.string = dlg.m_txt.GetBuffer(35);
		SendMsg(PM_CTLMULTISEARCH, 0, (LPARAM)(PAN_CtlSearchInfo FAR*) &si);

		if (si.fFound == 0) {
			m_pWndOut->SendToOutput("String not found!");
			return;
		}

		// Display all the founded positions
		PAN_CtlCaretPos FAR* s = (PAN_CtlCaretPos FAR*) GlobalLock(si.hFoundPos);
		for (WORD i = 0; i < si.fFound; i++) {
			m_pWndOut->SendToOutput("[ %s ] found at: page %d, flow= %ld  offs=%ld",
					si.string, s[i].page, s[i].flow,
					s[i].offset);
		}

		GlobalUnlock(si.hFoundPos);
		GlobalFree(si.hFoundPos);
		si.hFoundPos = 0;
	}
*/
#endif
}
