/*
**		Source listing for the DEMO program.
**		Copyright Oracle, 1993-2008, all rights reserved.
**
**		This file implements all the Control related messages handling.
**		Part 2
*/

#include "stdafx.h"
#include <time.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>

#include "demo.h"
#include "define.h"
#include "demodoc.h"
#include "demoview.h"
#include "diags.h"
#include "msghndl.h"
#include "outwnd.h"

#include "pctl.h"			// Control library public header file
#undef MoveTo

// --------------------------------------------------------------------------
// Enable the modes of the control specified by the user (from a dialog)
//
void CDemoView::OnSendmessagePmctlsetmode()
{
	if (!Validate()) {
		return;
	}

	DWORD	flg;

	if (!SendMsg(PM_CTLGETMODE, 0, (LPARAM) (LPDWORD) &flg)) {
		return;
	}

	{
		CModeDiag	dlg;

		dlg.m_aniso = (flg & PAN_CTLMODEANISOTROPIC) != 0;
		dlg.m_drag = (flg & PAN_CTLMODEDRAGDROP) != 0;
		dlg.m_exscr = (flg & PAN_CTLMODEEXCESSSCROLL) != 0;
		dlg.m_noredr = (flg & PAN_CTLMODENOREDRAW) != 0;
		dlg.m_opaq = (flg & PAN_CTLMODEOPAQUE) != 0;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		if (dlg.m_aniso)
			flg |= PAN_CTLMODEANISOTROPIC;
		else
			flg &= ~PAN_CTLMODEANISOTROPIC;

		if (dlg.m_drag)
			flg |= PAN_CTLMODEDRAGDROP;
		else
			flg &= ~PAN_CTLMODEDRAGDROP;

		if (dlg.m_exscr)
			flg |= PAN_CTLMODEEXCESSSCROLL;
		else
			flg &= ~PAN_CTLMODEEXCESSSCROLL;

		if (dlg.m_noredr)
			flg |= PAN_CTLMODENOREDRAW;
		else
			flg &= ~PAN_CTLMODENOREDRAW;

		if (dlg.m_opaq)
			flg |= PAN_CTLMODEOPAQUE;
		else
			flg &= ~PAN_CTLMODEOPAQUE;
	}

	if (!SendMsg(PM_CTLSETMODE, 0, flg)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETMODE: Done.");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetoffset()
{
	if (!Validate()) {
		return;
	}

	PAN_CtlFileInfo FAR	*fi = new PAN_CtlFileInfo;

	if (!SendMsg(PM_CTLGETFILE, 0, (LPARAM) fi)) {
		delete fi;
		return;
	}

	WORD	pg;

	if (!SendMsg(PM_CTLGETPAGE, 0, (LPARAM) (LPWORD) &pg)) {
		delete fi;
		return;
	}

	DWORD	oldCaps;

	if (!SendMsg(PM_CTLGETCAPS, 0, (LPARAM) (LPDWORD) &oldCaps)) {
		delete fi;
		return;
	}

	DWORD tmpCaps = oldCaps | (PAN_CTLCAPSHSCROLL | PAN_CTLCAPSVSCROLL);

	PAN_CtlPos	po;

	if (!SendMsg(PM_CTLGETOFFSET, 0, (LPARAM) (PAN_CtlPos FAR*) &po)) {
		delete fi;
		return;
	}

	{
		CPxyzDiag	dlg;

		dlg.m_posx = po.x;
		dlg.m_posy = po.y;
		dlg.m_posz = po.z;

		if (dlg.DoModal() != IDOK) {
			delete(fi);
			return;
		}

		po.x = dlg.m_posx;
		po.y = dlg.m_posy;
		po.z = dlg.m_posz;
	}

	if (!SendMsg(PM_CTLSETOFFSET, 0, (LPARAM) (PAN_CtlPos FAR*) &po)) {
		delete fi;
		return;
	}

	if (!SendMsg(PM_CTLSETCAPS, 0, oldCaps)) {
		delete fi;
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLSETOFFSET: Done.");
	delete fi;
}

// --------------------------------------------------------------------------
// Set the current page
//
void CDemoView::OnSendmessagePmctlsetpage()
{
	int		nPage;

	if (!Validate()) {
		return;
	}

	{
		CPageDiag	dlg;

		if (!SendMsg(PM_CTLGETPAGE, 0, (LPARAM) &dlg.m_page)) {
			return;
		}

		if (dlg.DoModal() != IDOK) {
			return;
		}

		nPage = dlg.m_page;
	}

	if (!SendMsg(PM_CTLSETPAGE, nPage, 0)) {
		return;
	}

	CDemoDoc *doc = (CDemoDoc *) GetDocument();
	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLSETPAGE: Done.");
}

// --------------------------------------------------------------------------
// Select the entire control.
//
void CDemoView::OnSendmessagePmctlsetsel()
{
	int				nPage;
	PAN_CtlRange	rg;
	BOOL			clrFlag;

	if (!Validate()) {
		return;
	}

	{
		CSelDiag	dlg;

		dlg.m_clrFlg = FALSE;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		nPage = dlg.m_page;
		rg.min.x = dlg.m_selstartx;
		rg.min.y = dlg.m_selstarty;
		rg.min.z = 0;
		rg.max.x = dlg.m_selendx;
		rg.max.y = dlg.m_selendy;
		rg.max.z = 0;
		clrFlag = dlg.m_clrFlg;
	}

	PAN_CtlFileInfo FAR	*fi = new(PAN_CtlFileInfo);

	if (!SendMsg(PM_CTLGETFILE, 0, (LPARAM) fi)) {
		delete fi;
		return;
	}

	if (!SendMsg(PM_CTLSETSEL, !clrFlag, (LPARAM) (PAN_CtlRange FAR *) &rg)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETSEL: Done.");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetviewextents()
{
	if (!Validate()) {
		return;
	}

	PAN_CtlFileInfo FAR	*fi = new PAN_CtlFileInfo;

	if (!SendMsg(PM_CTLGETFILE, 0, (LPARAM) fi)) {
		delete fi;
		return;
	}

	WORD pg;
	if (!SendMsg(PM_CTLGETPAGE, 0, (LPARAM) (LPWORD) &pg)) {
		delete fi;
		return;
	}

	DWORD	oldCaps;

	if (!SendMsg(PM_CTLGETCAPS, 0, (LPARAM) (LPDWORD) &oldCaps)) {
		delete fi;
		return;
	}

	DWORD	tmpCaps = oldCaps | (PAN_CTLCAPSHSCROLL | PAN_CTLCAPSVSCROLL);

	PAN_CtlRange	rg;

	if (!SendMsg(PM_CTLGETVIEWEXTENTS, 0, (LPARAM) (PAN_CtlRange FAR*) &rg)) {
		delete fi;
		return;
	}

	fi->dimensions.min.z = rg.min.z;
	fi->dimensions.max.z = rg.max.z;

	{
		CSzeDiag	dlg;
		dlg.m_x = rg.min.x;
		dlg.m_y = rg.min.y;
		dlg.m_x1 = rg.max.x;
		dlg.m_y1 = rg.max.y;

		if (dlg.DoModal() != IDOK) {
			delete fi;
			return;
		}
		rg.min.z = fi->dimensions.min.z;
		rg.max.z = fi->dimensions.max.z;
		rg.min.x = dlg.m_x;
		rg.min.y = dlg.m_y;
		rg.max.x = dlg.m_x1;
		rg.max.y = dlg.m_y1;
	}

	if (!SendMsg(PM_CTLSETVIEWEXTENTS, 0, (LPARAM) (PAN_CtlRange FAR*) &rg)) {
		delete fi;
		return;
	}

	if (!SendMsg(PM_CTLSETCAPS, 0, oldCaps)) {
		delete fi;
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();
	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLSETVIEWEXTENTS: Done.");
	delete fi;
}

// --------------------------------------------------------------------------
// Set the zoom factor for the current file
//
void CDemoView::OnSendmessagePmctlsetzoom()
{
	WORD	flg = 0;
	DWORD	fct;

	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLGETZOOM, PAN_CTLZOOMBOTH, (LPARAM) &fct)) {
		return;
	}

	{
		CZoomDiag	dlg;
		dlg.m_both = 1;
		dlg.m_zoom = (UINT) ((fct / 1000.0) + 0.0005);

		if (dlg.DoModal() != IDOK) {
			return;
		}

		if (dlg.m_both)
			flg |= PAN_CTLZOOMBOTH;

		if (dlg.m_x)
			flg |= PAN_CTLZOOMX;

		if (dlg.m_y)
			flg |= PAN_CTLZOOMY;

		fct = ((DWORD) dlg.m_zoom) * 1000;
	}

	if (!SendMsg(PM_CTLSETZOOM, flg, fct)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();
	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLSETZOOM: Done.");
}

// --------------------------------------------------------------------------
// Return the client area coordinates (i.e. relative to the top,left corner
// of the client area of the control window) corrisponding to the given
// world coordinates (i.e. the position in the current file).
//
void CDemoView::OnSendmessagePmctlworldtoclient()
{
	PAN_CtlPos	po;

	if (!Validate()) {
		return;
	}

	{
		CPxyzDiag	dlg;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		po.x = dlg.m_posx;
		po.y = dlg.m_posy;
		po.z = dlg.m_posz;
	}

	if (!SendMsg(PM_CTLWORLDTOCLIENT, 0, (LPARAM) (PAN_CtlPos FAR*) &po)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLWORLDTOCLIENT: x = %.3lf y = %.3lf",
			po.x, po.y);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlflipNone()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLFLIP, PAN_CTLFLIPNONE, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLFLIP: (none)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlflipX()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLFLIP, PAN_CTLFLIPX, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLFLIP: (X)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlflipXandy()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLFLIP, PAN_CTLFLIPXY, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLFLIP: (X and Y)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlflipY()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLFLIP, PAN_CTLFLIPY, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLFLIP: (Y)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlrotate0deg()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLROTATE, 0, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLROTATE: (0 degrees)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlrotate180degrees()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLROTATE, 180, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLROTATE: (180 degrees)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlrotate270degrees()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLROTATE, 270, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLROTATE: (270 degrees)");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlrotate90degrees()
{
	if (!Validate()) {
		return;
	}

	if (!SendMsg(PM_CTLROTATE, 90, 0)) {
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	m_pWndOut->SendToOutput("PM_CTLROTATE: (90 degrees)");
}

// --------------------------------------------------------------------------
// Tell the control to repaint it's window area.
// This will result in a blanking of the window.
//
void CDemoView::OnSendmessagePmctlpaint()
{
	if (!Validate()) {
		return;
	}

	CDemoDoc*	pDoc = GetDocument();
	CPanCtRgn*	pRgn = pDoc->GetCurrRgn();

	if (pRgn == NULL) {
		return;
	}

	HWND	hCtl = pRgn->GetCtlHandle();

	if (hCtl == NULL) {
		return;
	}

	::InvalidateRect(hCtl, 0, 0);
	if (!SendMsg(PM_CTLPAINT, 0, 0)) {
		return;
	}

	::ValidateRect(hCtl, 0);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlctldestroy()
{
	m_pWndOut->SendToOutput(
		"PM_CTLDESTROY: Not implemented in the DEMO program!");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctldestroydescendants()
{
	m_pWndOut->SendToOutput(
		"PM_CTLDESTROYDESCENDANTS: Not implemented in the DEMO program!");
}


// --------------------------------------------------------------------------
// Get the file type of a user defined file. This function will not load
// any file in the control.
//
void CDemoView::OnSendmessagePmctlgetfiletype()
{
	if (!Validate()) {
		return;
	}

	char	fname[MAX_FILENAME];

	if (GetFile(fname)) {
		LONG	result = (LONG) ::SendMessage(m_hCurCtl, PM_CTLGETFILETYPE,
									0, (LPARAM) (LPCSTR) fname);
		int		ftype = LOWORD(result);
		int		findex = HIWORD(result);
		char	*lpType;

		switch (ftype) {
		case PAN_RasterFile:
			lpType = "PAN_RasterFile";
			break;

		case PAN_VectorFile:
			lpType = "PAN_VectorFile";
			break;

		case PAN_DocumentFile:
			lpType = "PAN_DocumentFile";
			break;

		case PAN_DatabaseFile:
			lpType = "PAN_DatabaseFile";
			break;

		case PAN_SpreadsheetFile:
			lpType = "PAN_SpreadsheetFile";
			break;

		case PAN_ArchiveFile:
			lpType = "PAN_ArchiveFile";
			break;
		}

		m_pWndOut->SendToOutput("PM_CTLGETFILETYPE: %s, index = %d",
				lpType, findex);
	}
}

// --------------------------------------------------------------------------
// Get the palette of the control and display all the color entries.
//
void CDemoView::OnSendmessagePmctlgetpalette()
{
	if (!Validate()) {
		return;
	}

	// get the palette size
	WORD	colorDepth = (WORD) ::SendMessage(m_hCurCtl, PM_CTLGETPALETTE,
										0, NULL);

	if (colorDepth == (WORD) -1) {
		return; // something went wrong!
	}

	WORD nbColors = 2 << (colorDepth - 1);

	// allocate the palette
	HGLOBAL lPal = GlobalAlloc(GHND,
		sizeof (LOGPALETTE) + (nbColors - 1) * sizeof (PALETTEENTRY));

	if (lPal == NULL) {
		return;
	}

	LOGPALETTE FAR* logPalette = (LOGPALETTE FAR*) GlobalLock(lPal);
	if (logPalette == NULL) {
		GlobalFree(lPal);
		return;
	}

	logPalette->palNumEntries = nbColors;
	if (-1 == (int) ::SendMessage(m_hCurCtl, PM_CTLGETPALETTE,
						0, (LPARAM) (LOGPALETTE FAR*) logPalette)) {
		GlobalUnlock(lPal);
		GlobalFree(lPal);
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETPALETTE: nb of colors = %d",
			logPalette->palNumEntries);

	for (WORD i = 0; i < logPalette->palNumEntries; i++) {
		PALETTEENTRY FAR* e = &logPalette->palPalEntry[i];

		m_pWndOut->SendToOutput("%3d (%3d, %3d, %3d)", i,
				e->peRed, e->peGreen, e->peBlue);
	}

	// free memory
	GlobalUnlock(lPal);
	GlobalFree(lPal);
}

// --------------------------------------------------------------------------
// NOTE: this message must be sent ONLY in response to a WM_PALETTECHANGED
//
void CDemoView::OnSendmessagePmctlpalettechanged()
{
	if (!Validate()) {
		return;
	}

	HWND	appHwnd = ((CDemoApp*) AfxGetApp())->m_pMainWnd->GetSafeHwnd();

	if (!SendMsg(PM_CTLPALETTECHANGED, (WPARAM) (HWND) appHwnd, 0)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLPALETTECHANGED: Done.");
}

// --------------------------------------------------------------------------
// NOTE: this message must be sent ONLY in response to a WM_QUERYNEWPALETTE
//
void CDemoView::OnSendmessagePmctlquerynewpalette()
{
	if (!Validate()) {
		return;
	}

	HWND	appHwnd = ((CDemoApp*) AfxGetApp())->m_pMainWnd->GetSafeHwnd();

	if (!SendMsg(PM_CTLQUERYNEWPALETTE, (WPARAM) (HWND) appHwnd, 0)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLQUERYNEWPALETTE: Done.");
}

static void ConvertPhysicalPoints(HDC hdc, LPPHYSPOINT points,
	WORD nPoints, WORD srcUnits, WORD dstUnits)
/*-----------------------------------------------------------------------------
	Convert an array of physical points from one set of units to another.
	The srcUnits and dstUnits parameters must be one of CTLUNIT_*.
------------------------------------------------------------------------------*/
{
	int		xdpi, ydpi;
	WORD	i;
#define INCH_TO_MM		(25.4)

	xdpi = GetDeviceCaps(hdc, LOGPIXELSX);
	if (xdpi == 0) {
		xdpi = 1;
	}

	ydpi = GetDeviceCaps(hdc, LOGPIXELSY);
	if (ydpi == 0) {
		ydpi = 1;
	}

	for (i = 0; i < nPoints; ++i) {
		if (srcUnits == CTLUNIT_PIXEL) {
			if (dstUnits == CTLUNIT_INCH) {
				points[i].x /= (double) xdpi;
				points[i].y /= (double) ydpi;
			} else if (dstUnits == CTLUNIT_MM) {
				points[i].x *= INCH_TO_MM / (double) xdpi;
				points[i].y *= INCH_TO_MM / (double) ydpi;
			}
		} else if (srcUnits == CTLUNIT_INCH) {
			if (dstUnits == CTLUNIT_PIXEL) {
				points[i].x *= (double) xdpi;
				points[i].y *= (double) ydpi;
			} else if (dstUnits == CTLUNIT_MM) {
				points[i].x *= INCH_TO_MM;
				points[i].y *= INCH_TO_MM;
			}
		} else if (srcUnits == CTLUNIT_MM) {
			if (dstUnits == CTLUNIT_PIXEL) {
				points[i].x *= (double) xdpi / INCH_TO_MM;
				points[i].y *= (double) ydpi / INCH_TO_MM;
			} else if (dstUnits == CTLUNIT_INCH) {
				points[i].x /= INCH_TO_MM;
				points[i].y /= INCH_TO_MM;
			}
		}
	}
}

/*
**	Set up the mapping modes before rendering onto a DC.
*/
int DemoSetHDCMode(HDC hdc,
	HWND hwndCtl, LPRECT lpRect, int type, PAN_CtlRange FAR *rg)
{
	RECT	clientRect;
	RECT	clipRect;
	RECT	windowRect;

	HDC		hdcScr;

	if ((hdcScr = ::GetDC(0)) == 0) {
		return(0);
	} else {
		PHYSRECT	clientPhysRect;
		HRGN		clipRgn;

		/*
		**	Clip to target rectangle without exceeding view rectangle.
		**	(I don't think that this is doing anything, AN)
		*/
		::GetClientRect(hwndCtl, &clientRect);

		/* Convert client rectangle to printer device units. */
		clientPhysRect.left = clientRect.left;
		clientPhysRect.top = clientRect.top;
		clientPhysRect.right = clientRect.right;
		clientPhysRect.bottom = clientRect.bottom;

		ConvertPhysicalPoints(hdcScr, (LPPHYSPOINT) &clientPhysRect,
			2, CTLUNIT_PIXEL, CTLUNIT_INCH);

		ConvertPhysicalPoints(hdc, (LPPHYSPOINT) &clientPhysRect,
			2, CTLUNIT_INCH, CTLUNIT_PIXEL);

		clientRect.left = (int) (clientPhysRect.left + 0.5);
		clientRect.top = (int) (clientPhysRect.top + 0.5);
		clientRect.right = (int) (clientPhysRect.right + 0.5);
		clientRect.bottom = (int) (clientPhysRect.bottom + 0.5);

		::OffsetRect(&clientRect, lpRect->left, lpRect->top);
		::IntersectRect(&clipRect, &clientRect, lpRect);

		::SetMapMode(hdc, MM_ISOTROPIC);

		windowRect.left = (int) rg->min.x;
		windowRect.top = (int) rg->min.y;
		windowRect.right = (int) (rg->max.x - rg->min.x);
		windowRect.bottom = (int) (rg->max.y - rg->min.y);

		::SetWindowExtEx(hdc, windowRect.right, windowRect.bottom, NULL);
		::SetWindowOrgEx(hdc, windowRect.left, windowRect.top, NULL);

		::SetViewportExtEx(hdc,
			clipRect.right - clipRect.left,
			clipRect.bottom - clipRect.top, NULL);
		::SetViewportOrgEx(hdc, clipRect.left, clipRect.top, NULL);

		/* Set clipping region. */
		clipRgn = ::CreateRectRgn(clipRect.left, clipRect.top,
							clipRect.right, clipRect.bottom);
		::SelectClipRgn(hdc, clipRgn);

		/* Clean up. */
		::DeleteObject((HGDIOBJ)clipRgn);

		::ReleaseDC(0, hdcScr);
	}

	return(1);
}

//-----------------------------------------------------------------------------
// Render the control selection on the desktop window.
//
void CDemoView::OnSendmessagePmctlrenderontodc()
{
	PAN_CtlRange			reg;
	PAN_CtlRenderOptions	ro;

	if (!Validate()) {
		return;
	}

//	CDC*	deskTopDC;
	HDC		deskTopDC;

//	deskTopDC = ((CDemoApp*) AfxGetApp())->m_pMainWnd->GetDesktopWindow()->GetDC();
	deskTopDC = ::GetDC(0);
	if (deskTopDC == NULL) {
		return;
	}

	m_pWndOut->SendToOutput("PM_RENDERONTODC: Page setting will have no effect");

	// Get Region to render
	{
		CSelDiag	dlg;

		if (!SendMsg(PM_CTLGETPAGE, 0, (LPARAM) &dlg.m_page)) {
			return;
		}

		if (!SendMsg(PM_CTLGETVIEWEXTENTS, 0,
				(LPARAM) (PAN_CtlRange FAR*) &reg)) {
			return;
		}

		PAN_CtlSelList	sl;
		PAN_CtlSel FAR	*s;

		if (!SendMsg(PM_CTLGETSELS, 0, (LPARAM) (PAN_CtlSelList FAR*) &sl)) {
			return;
		}

		if (sl.nSels != 0) {
			s = (PAN_CtlSel FAR*) GlobalLock(sl.hSels);
			if (s[0].selType == PAN_CTLSELECTION_VIEW) {
				reg = s[0].range.vwRange;
			}
			GlobalUnlock(sl.hSels);
		}

		dlg.m_selstartx = (float) reg.min.x;
		dlg.m_selstarty = (float) reg.min.y;
		dlg.m_selendx = (float) reg.max.x;
		dlg.m_selendy = (float) reg.max.y;
		if (dlg.DoModal() != IDOK) {
			return;
		}

		/*
		** 16.09.1994: GETPAGE - SETPAGE combination changes view for documents
		**		::SendMessage(m_hCurCtl, PM_CTLSETPAGE, dlg.m_page, 0);
		*/
		LONG	mode;

		::SendMessage(m_hCurCtl, PM_CTLGETMODE, 0, (LPARAM) &mode);
		mode &= PAN_CTLMODEANISOTROPIC;

		// Setup the render options structure.
		_fmemset(&ro, 0, sizeof (PAN_CtlRenderOptions));

//		ro.hdc = deskTopDC->m_hDC;
		ro.hdc = deskTopDC;
		ro.mode = PAN_CTLMODEOPAQUE | mode;
		::GetClientRect(m_hCurCtl, &ro.devRect);
		ro.source.min.x = dlg.m_selstartx;
		ro.source.min.y = dlg.m_selstarty;
		ro.source.max.x = dlg.m_selendx;
		ro.source.max.y = dlg.m_selendy;
	}

	m_pWndOut->SendToOutput("PM_CTLRENDERONTODC: Rendering to Desktop window");

	::SendMessage(m_hCurCtl, PM_CTLRENDERONTODC,
			0, (LPARAM) (PAN_CtlRenderOptions FAR*) &ro);

//	((CDemoApp*) AfxGetApp())->m_pMainWnd->GetDesktopWindow()->ReleaseDC(deskTopDC);
	::ReleaseDC(0, deskTopDC);
}


// --------------------------------------------------------------------------
// Set the control palette to the system default palette.
//
void CDemoView::OnSendmessagePmctlsetpalette()
{
	if (!Validate()) {
		return;
	}

	// get the palette size
	WORD	colorDepth = (WORD) ::SendMessage(m_hCurCtl, PM_CTLGETPALETTE,
										0, NULL);

	if (colorDepth == (WORD) -1) {
		return;		// something went wrong!
	}

#define MAXNUMBER		256

	WORD nbColors = min(2 << (colorDepth - 1), MAXNUMBER);

	// allocate the palette
	HGLOBAL lPal = GlobalAlloc(GHND, sizeof(LOGPALETTE) +
						(nbColors - 1) * sizeof(PALETTEENTRY));

	if (lPal == NULL) {
		return;
	}

	LOGPALETTE FAR*	logPalette = (LOGPALETTE FAR*) GlobalLock(lPal);

	if (logPalette == NULL) {
		GlobalFree(lPal);
	}

	// initialize the palette with random values
	PALETTEENTRY	pe[MAXNUMBER];

	_fmemset(pe, 0, MAXNUMBER * sizeof(PALETTEENTRY));

	for (int i = 0; i < (int) nbColors; i++) {
		pe[i].peFlags = 0;
		pe[i].peRed = (int) ((((float) rand()) / RAND_MAX) * 255);
		pe[i].peGreen = (int) ((((float) rand()) / RAND_MAX) * 255);
		pe[i].peBlue = (int) ((((float) rand()) / RAND_MAX) * 255);
	}

	_fmemcpy(logPalette->palPalEntry, pe, nbColors*sizeof(PALETTEENTRY));
	logPalette->palVersion = 0x300;
	logPalette->palNumEntries = nbColors;

	if (!SendMsg(PM_CTLSETPALETTE, 0, (LPARAM) (LOGPALETTE FAR*) logPalette)) {
		GlobalUnlock(lPal);
		GlobalFree(lPal);
		return;
	}

	CDemoDoc	*doc = (CDemoDoc *) GetDocument();

	if (doc->autoRepaint)
		SendMsg(PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

	::UpdateWindow(m_hCurCtl);			// display changes immediately

	// free memory
	GlobalUnlock(lPal);
	GlobalFree(lPal);

	m_pWndOut->SendToOutput("PM_CTLSETPALETTE: Done.");
}

// --------------------------------------------------------------------------
// Prompt user to specify a scroll code (as defined for the WM_VSCROLL message
// and send it to the control.
//
void CDemoView::OnSendmessagePmctlhscroll()
{
	if (!Validate()) {
		return;
	}

	wndscrl dlg;
	if (dlg.DoModal() == IDOK) {
#if TARGET == WIN32S || defined(_WIN64)
		if (!SendMsg(PM_CTLHSCROLL, MAKELONG(dlg.scrollCode, dlg.m_thmbpos), 0)) {
#else
		if (!SendMsg(PM_CTLHSCROLL, dlg.scrollCode, MAKELONG(dlg.m_thmbpos, 0))) {
#endif
			return;
		}
		m_pWndOut->SendToOutput("PM_CTLHSCROLL: Done.");
	}
}

// --------------------------------------------------------------------------
// Prompt user to specify a scroll code (as defined for the WM_VSCROLL message
// and send it to the control.
//
void CDemoView::OnSendmessagePmctlvscroll()
{
	if (!Validate()) {
		return;
	}

	wndscrl	dlg;

	if (dlg.DoModal() == IDOK) {
#if TARGET == WIN32S || defined(_WIN64)
		if (!SendMsg(PM_CTLVSCROLL, MAKELONG(dlg.scrollCode, dlg.m_thmbpos), 0)) {
#else
		if (!SendMsg(PM_CTLVSCROLL, dlg.scrollCode, MAKELONG(dlg.m_thmbpos, 0))) {
#endif
			return;
		}

		m_pWndOut->SendToOutput("PM_CTLVSCROLL: Done.");
	}
}

// --------------------------------------------------------------------------
// Prompt the user for a new width and height of the control window and
// send the message so the control window will be resized to those new
// values.
//
void CDemoView::OnSendmessagePmctlsize()
{
	if (!Validate()) {
		return;
	}

	wndsze	dlg;

	if (dlg.DoModal() == IDOK) {
		if (!SendMsg(PM_CTLSIZE, SIZE_RESTORED,
				MAKELONG(dlg.m_width, dlg.m_height))) {
			return;
		}

		m_pWndOut->SendToOutput("PM_CTLSIZE: Done.");

		// Resize handles border.
		CDemoDoc*	pDoc = GetDocument();
		CPanCtRgn*	pRgn = GetDocument()->GetCurrRgn();

		if (pRgn != NULL) {
			pDoc->Deselect(pRgn, FALSE);

			CRect wndRect;
			::GetWindowRect(m_hCurCtl, &wndRect);
			::MapWindowPoints(HWND_DESKTOP, GetSafeHwnd(),
					(LPPOINT) (LPRECT) wndRect, 2);
			pRgn->SetInnerRect(wndRect);

			pDoc->Select(pRgn);
		}
	}
}

// --------------------------------------------------------------------------
// Ask the control to validate the specified invalid margins.
// Since this message requires a valid printer hDC we need to correctly fill the
// PrintOptions structure (at least to get the printer DC).
//
void CDemoView::OnSendmessagePmctlvalidatemargins()
{
	if (!Validate()) {
		return;
	}

	// fill structure with 0
	PAN_CtlPrintOptions printOptions;
	_fmemset(&printOptions, 0, sizeof(PAN_CtlPrintOptions));

	// initialize default settings for the printer
	PRINTDLG	printDlg;
	_fmemset(&printDlg, 0, sizeof(PRINTDLG));

	printDlg.lStructSize = sizeof(PRINTDLG);
	printDlg.Flags = PD_RETURNDC | PD_RETURNDEFAULT;

	if (PrintDlg(&printDlg) == 0) {
		::MessageBox((HWND) 0, "Unable to initialize the printer!",
				"INFO", MB_ICONINFORMATION);
		return;
	}

	printOptions.printDlg = &printDlg;

	// initialize with invalid margins
	printOptions.margins.units = CTLUNIT_INCH;
	printOptions.margins.top = -1.00;
	printOptions.margins.left = -1.00;
	printOptions.margins.bottom = 150.00;
	printOptions.margins.right = 500.00;

	m_pWndOut->SendToOutput("PM_CTLVALIDATEMARGINS:\r");
	m_pWndOut->SendToOutput("	invalid margins: top=%6.2f left=%6.2f bottom=%6.2f right=%6.2f INCHES\r",
					printOptions.margins.top, printOptions.margins.left,
					printOptions.margins.bottom, printOptions.margins.right);

	// now validate the margins
	if (!SendMsg(PM_CTLVALIDATEMARGINS, 0, (LPARAM) (PAN_CtlPrintOptions FAR*) &printOptions)) {
		// Clean up
		DeleteDC(printDlg.hDC);
		return;
	}

	m_pWndOut->SendToOutput(" validated margins: top=%6.2f left=%6.2f bottom=%6.2f right=%6.2f INCHES\r",
					printOptions.margins.top, printOptions.margins.left,
					printOptions.margins.bottom, printOptions.margins.right);

	// Clean up
	DeleteDC(printDlg.hDC);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlprintpreview()
{
	PRINTDLG			printDlg;
	PAN_CtlPrintOptions	printOptions;
	static CPrnDiag		pDlg;

	if (!Validate()) {
		return;
	}

	// fill structure with 0
	_fmemset(&printOptions, 0, sizeof(PAN_CtlPrintOptions));

	// initialize the LOGFONT structure
	strcpy(printOptions.headers.font.lfFaceName, "Arial");
	printOptions.headers.font.lfHeight = -10;

	// initialize default settings for the printer
	_fmemset(&printDlg, 0, sizeof(PRINTDLG));
	printDlg.lStructSize = sizeof(PRINTDLG);
	printDlg.Flags = PD_RETURNDC | PD_RETURNDEFAULT;

	if (PrintDlg(&printDlg) == 0) {
		::MessageBox((HWND) 0, "Unable to initialize the printer!",
			"INFO", MB_ICONINFORMATION);
		return;
	}

	printDlg.Flags = PD_RETURNDC | PD_RETURNIC;
	printDlg.nCopies = 1;
	printDlg.nFromPage = 1;
	printDlg.nToPage = 0x7FFF;
	printDlg.nMinPage = 0x0001;
	printDlg.nMaxPage = 0x7FFF;
	printOptions.printDlg = &printDlg;

	printOptions.mode = 0;

	printOptions.units = CTLUNIT_PIXEL;
	printOptions.nImageUnits = 1;
	printOptions.nPaperUnits = 1;

	printOptions.margins.units = CTLUNIT_INCH;
	printOptions.margins.top = 0.75;
	printOptions.margins.left = 0.75;
	printOptions.margins.bottom= 0.75;
	printOptions.margins.right = 0.75;

	// initialize dialog
	pDlg.margins.units = INCHES;
	pDlg.margins.m_bottom = printOptions.margins.bottom;
	pDlg.margins.m_top = printOptions.margins.top;
	pDlg.margins.m_right = printOptions.margins.right;
	pDlg.margins.m_left = printOptions.margins.left;
	pDlg.hdrFont = &printOptions.headers.font;
	pDlg.prnDlg = &printDlg;

	if (pDlg.DoModal() == IDOK) {
		// fill the printing structure
		if (pDlg.fit == TRUE) {
			// a page fit require these values at 0
			printOptions.nImageUnits = 0;
			printOptions.nPaperUnits = 0;
		} else {
			printOptions.units = (pDlg.units == PIXELS) ? CTLUNIT_PIXEL :
								((pDlg.units == INCHES) ? CTLUNIT_INCH :
								CTLUNIT_MM);
			// WARNING! no checks are made to validate user inputs
			printOptions.nImageUnits = pDlg.m_graphics;
			printOptions.nPaperUnits = pDlg.m_paper;
		}

		PAN_CtlFileInfo FAR *fi = new PAN_CtlFileInfo;

		if (!SendMsg(PM_CTLGETFILE, 0, (LPARAM) fi)) {
			delete fi;
			return;
		}

		printOptions.source = fi->dimensions;
		delete fi;

		if (pDlg.marginsChanged) {
			// set the margins only if user changed something
			printOptions.margins.units = (pDlg.margins.units == INCHES) ?
											CTLUNIT_INCH :
											CTLUNIT_MM;

			printOptions.margins.top = pDlg.margins.m_top;
			printOptions.margins.left = pDlg.margins.m_left;
			printOptions.margins.bottom= pDlg.margins.m_bottom;
			printOptions.margins.right = pDlg.margins.m_right;

			if (!SendMsg(PM_CTLVALIDATEMARGINS, 0,  (LPARAM) (PAN_CtlPrintOptions FAR*) &printOptions)) {
				// Clean up
				DeleteDC(printDlg.hDC);
				return;
			}
		}

		printOptions.headers.topLeftText = pDlg.m_hdTopLeft;
		printOptions.headers.topCtrText = pDlg.m_hdTopCent;
		printOptions.headers.topRightText= pDlg.m_hdTopRight;
		printOptions.headers.botLeftText = pDlg.m_hdBotLeft;
		printOptions.headers.botCtrText = pDlg.m_hdBotCent;
		printOptions.headers.botRightText= pDlg.m_hdBotRight;

		printOptions.outputFileName = 0;

		// get the print preview information
		if (!SendMsg(PM_CTLPRINTPREVIEW, (WPARAM) -1, (LPARAM) (PAN_CtlPrintOptions FAR*) &printOptions)) {
			// Clean up
			DeleteDC(printDlg.hDC);
			return;
		}

		// display it
		CPrntPrvw dlg;
		dlg.printPreview = (PAN_CtlPrintPreview FAR *)
								&printOptions.printPreview;
		dlg.DoModal();
	} /* if IDOK */

	// Clean up
	DeleteDC(printDlg.hDC);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlcarettoworld()
{
	PAN_CtlCaretPos	cp;
	PAN_CtlPos FAR*	wp;

	if (!Validate()) {
		return;
	}

	{
		CFlowDiag	dlg;

		dlg.m_flow = 1;
		if (dlg.DoModal() != IDOK) {
			return;
		}

		cp.page = 1;
		cp.flow = (DWORD) dlg.m_flow;
		cp.offset = 0;
	}

	wp = (PAN_CtlPos FAR*) ::SendMessage(m_hCurCtl, PM_CTLCARETTOWORLD,
									0, (LPARAM) (PAN_CtlCaretPos FAR*) &cp);

	if (wp)
		m_pWndOut->SendToOutput("PM_CTLCARETTOWORLD: x: %lf y: %lf\r",
			wp->x, wp->y);
	else
		m_pWndOut->SendToOutput("PM_CTLCARETTOWORLD: No caret position!\r");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetcaretpos()
{
	PAN_CtlCaretPos cp;

	if (!Validate()) {
		return;
	}

	if (SendMsg(PM_CTLGETCARETPOS, 0, (LPARAM) (PAN_CtlCaretPos FAR *) &cp))
		m_pWndOut->SendToOutput("PM_CTLGETCARETPOS: page %d, flow: %ld, offset %ld\r",
			cp.page, cp.flow, cp.offset);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlgetdims()
{
	PAN_CtlDimensions dims;

	if (!Validate()) {
		return;
	}

	if(!SendMsg(PM_CTLGETDIMS, 0, (LPARAM) (PAN_CtlDimensions FAR*) &dims)) {
		return;
	}

	m_pWndOut->SendToOutput("File dimensions: width=%.3lf height=%.3lf depth=%.3lf\r",
					dims.DimWidth, dims.DimHeight, dims.DimDepth);
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetcaretpos()
{
	PAN_CtlCaretPos cp;

	if (!Validate()) {
		return;
	}

	{
		CFlowDiag	dlg;

		dlg.m_flow = 1;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		cp.page = 1;
		cp.flow = (DWORD) dlg.m_flow;
		cp.offset = 0;
	}

	if (!SendMsg(PM_CTLSETCARETPOS, 0,
				(LPARAM) (PAN_CtlCaretPos FAR *) &cp)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLGETCARETPOS: done!\r");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetselcaret()
{
	PAN_CtlCaretRange	cr;
	BOOL				clrFlag;

	if (!Validate()) {
		return;
	}

	{
		CCrSelDiag	dlg;

		dlg.m_clrFlg = FALSE;
		dlg.m_fromPage = 1; dlg.m_toPage = 1;
		dlg.m_fromFlow = 1; dlg.m_toFlow = 1;
		dlg.m_fromOffset = 0; dlg.m_toOffset = 0;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		cr.from.page = dlg.m_fromPage;
		cr.from.flow = dlg.m_fromFlow;
		cr.from.offset = dlg.m_fromOffset;
		cr.to.page = dlg.m_toPage;
		cr.to.flow = dlg.m_toFlow;
		cr.to.offset = dlg.m_toOffset;
		clrFlag = dlg.m_clrFlg;
	}

	if (!SendMsg(PM_CTLSETSELCARET, !clrFlag,
				(LPARAM) (PAN_CtlCaretRange FAR *) &cr)) {
		return;
	}

	m_pWndOut->SendToOutput("PM_CTLSETSELCARET: done!\r");
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsort()
{
	if (!Validate()) {
		return;
	}

	CSortDlg	dlg;

	if (dlg.DoModal() == IDOK) {
		int		page;
		PAN_CtlSortInfo	si;

		SendMsg(PM_CTLGETPAGE, 0, (LPARAM) &page);
		si.sortRange.from.page = page;
		si.sortRange.from.offset = 0;
		si.sortRange.from.flow = dlg.m_fromRange;
		si.sortRange.to.page = page;
		si.sortRange.to.offset = 0;
		si.sortRange.to.flow = dlg.m_toRange;

		si.sortBy = dlg.m_sortBy;

		si.sortKey1.page = page;
		si.sortKey1.offset = 0;
		si.sortKey1.flow = dlg.m_key1Flow;
		si.sortDir1 = dlg.m_key1Dir;

		si.sortKey2.page = page;
		si.sortKey2.offset = 0;
		si.sortKey2.flow = dlg.m_key2Flow;
		si.sortDir2 = dlg.m_key2Dir;

		si.sortKey3.page = page;
		si.sortKey3.offset = 0;
		si.sortKey3.flow = dlg.m_key3Flow;
		si.sortDir3 = dlg.m_key3Dir;

		SendMsg(PM_CTLSORT, 0, (LPARAM) (PAN_CtlSortInfo FAR *) &si);
		m_pWndOut->SendToOutput("PM_CTLSORT: done!\r");
	}
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlworldtocaret()
{
	PAN_CtlCaretPos FAR*	cp;
	PAN_CtlPos				wp;

	if (!Validate()) {
		return;
	}

	{
		CPxyzDiag	dlg;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		wp.x = dlg.m_posx;
		wp.y = dlg.m_posy;
		wp.z = dlg.m_posz;
	}

	cp = (PAN_CtlCaretPos FAR*) ::SendMessage(m_hCurCtl, PM_CTLWORLDTOCARET,
										0, (LPARAM) (PAN_CtlPos FAR*) &wp);

	if (cp)
		m_pWndOut->SendToOutput("PM_CTLWORDLTOCARET: flow: %ld\r", cp->flow);
	else
		m_pWndOut->SendToOutput("PM_CTLWORLDTOCARET: Caret not applicable!\r");

}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetblock()
{
	WORD	x;

	if (!Validate()) {
		return;
	}

	{
		CSetBlockDlg	dlg;

		dlg.m_x = 0;

		if (dlg.DoModal() != IDOK) {
			return;
		}

		x = dlg.m_x;
	}
	if (SendMsg(PM_CTLSETBLOCK, (WPARAM) x, 0)) {
		// Force update of drawing.
		SendMsg(PM_CTLREGEN, 0, 0);
		// Do a zoom fit.
		if (!SendMsg(PM_CTLSETZOOM, PAN_CTLZOOMBOTH, 100000)) {//zoom 100%
			return;
		}
		m_pWndOut->SendToOutput("PM_CTLSETBLOCK: done!\r");
	} else {
		m_pWndOut->SendToOutput("PM_CTLSETBLOCK: Returned error!\r");
	}
}

//--------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetimageex()
{
	CImgExDlg	dlg;

	if (!Validate()) {
		return;
	}

	//set initial values
	SendMsg(PM_CTLGETIMAGEEX, PAN_IMAGE_CONTRAST, (LPARAM)(LPINT) &(dlg.m_x));
	SendMsg(PM_CTLGETIMAGEEX, PAN_IMAGE_ANTIALIAS, (LPARAM)(BOOL FAR *) &(dlg.m_antiAlias));
	SendMsg(PM_CTLGETIMAGEEX, PAN_IMAGE_INVERT,	(LPARAM)(BOOL FAR *) &(dlg.m_invert));

	//to make sure in proper handling SetCheck for Check Buttons
	if (dlg.m_antiAlias) {
	    dlg.m_antiAlias = 1;
	}

	if (dlg.m_invert) {
		dlg.m_invert = 1;
	}

	if (dlg.DoModal() == IDOK) {
		if (SendMsg(PM_CTLSETIMAGEEX, PAN_IMAGE_CONTRAST, (LPARAM) dlg.m_x))

			if (SendMsg(PM_CTLSETIMAGEEX, PAN_IMAGE_ANTIALIAS, (LPARAM) dlg.m_antiAlias))

				if (SendMsg(PM_CTLSETIMAGEEX, PAN_IMAGE_INVERT,	(LPARAM) dlg.m_invert)) {
					// Force update of drawing.
//					SendMsg(PM_CTLREGEN, 0, 0);

					m_pWndOut->SendToOutput("PM_CTLSETIMAGEEX: done!\r");
					return;
				}
		m_pWndOut->SendToOutput("PM_CTLSETIMAGEEX: Returned error!\r");
	}
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetlayerstate()
{
	CSetLayerDlg	dlg;

	if (!Validate()) {
		return;
	}

	dlg.m_hwnd = m_hCurCtl;

	if (dlg.DoModal() == IDOK) {
		if (dlg.m_layers) {
			if (SendMsg(PM_CTLSETLAYERSTATE, dlg.m_numLayers,
					(LPARAM) ((PAN_LAYER FAR *) dlg.m_layers))) {

				// Force update of drawing.
				// No longer needed as of June 2001.
				//SendMsg(PM_CTLREGEN, 0, 0);

				m_pWndOut->SendToOutput("PM_CTLSETLAYERSTATE: done!\r");
			} else {
				m_pWndOut->SendToOutput("PM_CTLSETLAYERSTATE: Returned error!\r");
			}
		} else {
			m_pWndOut->SendToOutput("PM_CTLSETLAYERSTATE: Returned error!\r");
		}
	}
}

// --------------------------------------------------------------------------
void CDemoView::OnSendmessagePmctlsetview()
{
	int		curSel;
	CSetViewDlg	dlg;

	if (!Validate()) {
		return;
	}

	dlg.m_hwnd = m_hCurCtl;

	if (dlg.DoModal() == IDOK) {
		curSel = dlg.m_curSel - 1;
		if (SendMsg(PM_CTLSETVIEW, curSel, (LPARAM) 0)) {
			// Force update of drawing.
			SendMsg(PM_CTLREGEN, PAN_CLEARLAYERS | PAN_CLEARVIEWEXTENTS, 0);
			// Set the view-extents only if it is not the default view
			if (curSel != -1) {
				PAN_VIEW	cur_view;
				if (SendMsg(PM_CTLGETVIEW, 0, (LPARAM)(PAN_VIEW FAR *) &cur_view)) {
					PAN_CtlRange	viewExt = {	{cur_view.vmin.x, cur_view.vmin.y, cur_view.vmin.z},
										{cur_view.vmax.x, cur_view.vmax.y, cur_view.vmax.z}};

					// Set view extents only if they we explicitely specified by
					// the decoder (i.e. not null).
					if (Fabs(viewExt.max.x-viewExt.min.x) > ExtremelyVerySmall &&
						Fabs(viewExt.max.y-viewExt.min.y) > ExtremelyVerySmall)
					{
						SendMsg(PM_CTLSETVIEWEXTENTS, 0, (LPARAM) (PAN_CtlRange FAR*) &viewExt);
					}
				}
			}
			m_pWndOut->SendToOutput("PM_CTLSETVIEW: done!\r");
		} else {
			m_pWndOut->SendToOutput("PM_CTLSETVIEW: Returned error!\r");
		}
	}
}
