/*
**	Source listing for the DEMO program.
**	Copyright Oracle, 1993-2008, all rights reserved.
**
**	All the objects used by the Document class are declared from these
**	classes.
** panctrgn.cpp : implementation of the CPanCtRgn and CPanCtRgnLst classes
*/
#include "stdafx.h"
#include "demo.h"
#include "define.h"
#include "demodoc.h"
#include "demoview.h"
#include "msghndl.h"
#include "diags.h"
#include "outwnd.h"

#include "pctl.h"				// Control library public header file
#undef MoveTo

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#if	TARGET==WIN16
	/*
	**	16-bit MSVC (v 1.5) gives an internal compiler error
	**	if this file is compiled with all optimisations on.
	*/
#	pragma 	optimize("", off)
#endif

//------------------------------------------------------------------------------
/*
** CHndlRgn Class Member Functions
*/

//------------------------------------------------------------------------------
CHndlRgn::CHndlRgn()
{
	// Initialize private member variables.
	m_rcInnerRect.SetRectEmpty();
	m_rcOldInnerRect.SetRectEmpty();

	m_nActiveHandle	= IDNONE;
	m_nSelected	= FALSE;
	m_nDrawMode	= TRUE;
}

//------------------------------------------------------------------------------
CHndlRgn::~CHndlRgn()
{
}

//------------------------------------------------------------------------------
int CHndlRgn::IsPickPointOverHandles(CPoint pt)
{
	int result = IDNONE;

	CRect rcInner = GetInnerRect();
	CRect rcOuter = GetOuterRect();

	if (rcOuter.PtInRect(pt)) {
		if (rcInner.PtInRect(pt)) {
			result = IDCONTROL;
		} else {
			result = IDALL;

			for (int i = 0; i < 8; ++i) {
				CRect	rc = GetHandle(i);

				if (rc.PtInRect(pt)) {
					result = i;
					break;
				}
			}
		}
	}

	return (result);
}

//------------------------------------------------------------------------------
CRect CHndlRgn::GetOuterRect()
{
	CRect	rcOuter;

	rcOuter.SetRect(
		m_rcInnerRect.left - HNDLSIZE, m_rcInnerRect.top - HNDLSIZE,
		m_rcInnerRect.right + HNDLSIZE, m_rcInnerRect.bottom + HNDLSIZE);

	return (rcOuter);
}

//------------------------------------------------------------------------------
CRect CHndlRgn::GetHandle(int nHandle)
{
	CRect	rcIn	= GetInnerRect();
	CRect	rcOut	= GetOuterRect();
	CRect	rc;
	int		nTemp;

	switch (nHandle) {
	case 0:
		rc.SetRect(rcOut.left, rcOut.top, rcIn.left, rcIn.top);
		break;

	case 1:
		nTemp = rcIn.left + (rcIn.right - rcIn.left - HNDLSIZE) / 2;
		rc.SetRect(nTemp, rcOut.top, nTemp + HNDLSIZE, rcIn.top);
		break;

	case 2:
		rc.SetRect(rcIn.right, rcOut.top, rcOut.right, rcIn.top);
		break;

	case 3:
		nTemp = rcIn.top + (rcIn.bottom - rcIn.top - HNDLSIZE) / 2;
		rc.SetRect(rcOut.left, nTemp, rcIn.left, nTemp + HNDLSIZE);
		break;

	case 4:
		nTemp = rcIn.top + (rcIn.bottom - rcIn.top - HNDLSIZE) / 2;
		rc.SetRect(rcIn.right, nTemp, rcOut.right, nTemp + HNDLSIZE);
		break;

	case 5:
		rc.SetRect(rcOut.left, rcIn.bottom, rcIn.left, rcOut.bottom);
		break;

	case 6:
		nTemp = rcIn.left + (rcIn.right - rcIn.left - HNDLSIZE) / 2;
		rc.SetRect(nTemp, rcIn.bottom, nTemp + HNDLSIZE, rcOut.bottom);
		break;

	case 7:
		rc.SetRect(rcIn.right, rcIn.bottom, rcOut.right, rcOut.bottom);
		break;
	}

	return (rc);
}

//------------------------------------------------------------------------------
/*
** CPanCtRgn Class Member Functions
*/

//------------------------------------------------------------------------------
CPanCtRgn::CPanCtRgn(CView* pView, WORD ID, CRect* rc)
{
	// Initialize private member variables.
	m_hCtl = NULL;
	m_wCtlId = 0;
	m_nHasFile = FALSE;
	m_curCaps = 0;
	m_curLMB = PAN_CTLLMBNONE;
	memset(&m_si, 0, sizeof(PAN_CtlSearchInfo));

	// Create CSI Multi-Format Control.
	// Function PAN_CreateControl expects window dimensions as origin & extents.
	CRect	rcCtl;

	rcCtl.SetRect(rc->left, rc->top, rc->right - rc->left, rc->bottom - rc->top);

	HWND	hParent	= pView->GetSafeHwnd();
	HWND	hCtl = PAN_CreateControl(hParent, ID, rcCtl, NULL, lpfnNotifyProc);

	if (hCtl == NULL) {
		GetCurrentView()->m_pWndOut->SendToOutput(
			"ERROR PAN_CreateControl(): %i", PAN_GetCtlErrorCode());

	} else {

		// Save control attributes.
		m_hCtl = hCtl;
		m_wCtlId = ID;
		m_rcInnerRect = *rc;

		// Save current control capabilities and mouse action.
		SetCurCaps();
		SetCurLMB();

		// Set control mode.
		DWORD	mode;

		::SendMessage(hCtl, PM_CTLGETMODE, 0, (LPARAM) &mode);

		mode |= (PAN_CTLMODEEXCESSSCROLL | PAN_CTLMODEDRAGDROP);
		::SendMessage(hCtl, PM_CTLSETMODE, 0, (LPARAM) mode);
	}
}

//------------------------------------------------------------------------------
CPanCtRgn::~CPanCtRgn()
{
	if (m_hCtl != NULL) {
		::DestroyWindow(m_hCtl);
	}
}

//------------------------------------------------------------------------------
/*
** Mark the region as selected/deselected and activate/deactivate the control
** window of the region.
*/
void CPanCtRgn::SetSelected(BOOL fSelected)
{
	if (m_hCtl == NULL) {
		return;
	}

	m_nSelected = fSelected;
	if (m_nSelected) {
		::SetWindowPos(m_hCtl, HWND_TOP, 0, 0, 0, 0,
			SWP_SHOWWINDOW | SWP_NOMOVE | SWP_NOSIZE);

		::SetFocus(m_hCtl);
	}
}

//------------------------------------------------------------------------------
/*
** Load the file with the given name into the control.
*/
void CPanCtRgn::LoadCtrlData(char* fname)
{
	if (m_hCtl == NULL) {
		return;
	}

	int		ctlErr = (int) ::SendMessage(m_hCtl, PM_CTLSETFILE,
								(WPARAM) -1, (LPARAM) (LPCSTR) fname);

	if (ctlErr != PAN_CTLERRNONE) {
		GetCurrentView()->m_pWndOut->SendToOutput("ERROR PM_CTLSETFILE: %i", ctlErr);

	} else {

		m_nHasFile = TRUE;

		// Save current control capabilities and mouse action.
		SetCurCaps();
		SetCurLMB();
		memset(&m_si, 0, sizeof(PAN_CtlSearchInfo));
		m_si.fDown = TRUE;

		// Load the file
		::SendMessage(m_hCtl, PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);
	}
}

//------------------------------------------------------------------------------
/*
** Abort procedure for printer.
*/
static BOOL CALLBACK PrintAbortProc(HDC hdc, int error)
{
	if (error != 0 && error != SP_OUTOFDISK) {
		// Unrecoverable error.
		return (FALSE);
	} else {
		MSG		msg;

		while (::PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
			::TranslateMessage(&msg);
			::DispatchMessage(&msg);
		}
	}
	return (TRUE);
}

//------------------------------------------------------------------------------
/*
** Print the contents of the control.
*/
void CPanCtRgn::Print()
{
	if (m_hCtl == NULL) {
		return;
	}

	::SendMessage(m_hCtl, PM_CTLPAINT, (WPARAM)0, (LPARAM)0);

	HWND			handle = m_hCtl;
	PRINTDLG		printDlg;
	PAN_CtlPrintOptions	printOptions;
	CPrnDiag		pDlg;

	// fill structure with 0
	_fmemset(&printOptions, 0, sizeof(PAN_CtlPrintOptions));

	// initialize the LOGFONT structure
	strcpy(printOptions.headers.font.lfFaceName, "Arial");
	printOptions.headers.font.lfHeight = -10;

	// initialize default settings for the printer
	_fmemset(&printDlg, 0, sizeof(PRINTDLG));
	printDlg.lStructSize = sizeof(PRINTDLG);
	printDlg.Flags = PD_RETURNDC | PD_RETURNDEFAULT;
	if (PrintDlg(&printDlg) == 0) {
		::MessageBox((HWND)0, "Unable to initialize the printer!",
				"INFO", MB_ICONINFORMATION);
		return;
	}

	printDlg.Flags = PD_RETURNDC | PD_RETURNIC;
	printDlg.nCopies = 1;
	printDlg.nFromPage 	= 1;
	printDlg.nToPage = 0x7FFF;
	printDlg.nMinPage = 0x0001;
	printDlg.nMaxPage = 0x7FFF;
	printOptions.printDlg = &printDlg;

	printOptions.mode = 0;

	printOptions.units = CTLUNIT_PIXEL;
	printOptions.nImageUnits = 1;
	printOptions.nPaperUnits = 1;

	printOptions.margins.units = CTLUNIT_INCH;
	printOptions.margins.top = 0.75;
	printOptions.margins.left = 0.75;
	printOptions.margins.bottom= 0.75;
	printOptions.margins.right = 0.75;

	// initialize dialog
	pDlg.margins.units = INCHES;
	pDlg.margins.m_bottom = printOptions.margins.bottom;
	pDlg.margins.m_top	= printOptions.margins.top;
	pDlg.margins.m_right = printOptions.margins.right;
	pDlg.margins.m_left = printOptions.margins.left;
	pDlg.hdrFont = &printOptions.headers.font;
	pDlg.prnDlg = &printDlg;

	// create a clone control
	if (::SendMessage(m_hCtl, PM_CTLCLONECONTROL, 999,
				(LPARAM)(HWND FAR*) &handle) != PAN_CTLERRNONE) {
		GetCurrentView()->m_pWndOut->SendToOutput("PRINT: Clone creation failed\n");
		return;
	}

	COLORREF	cWhite = RGB(255, 255, 255);
	::SendMessage(handle, PM_CTLSETFGBGCOLOR,
			1, (LPARAM) (COLORREF FAR*) &cWhite);

	if (pDlg.DoModal() == IDOK) {
		::UpdateWindow(m_hCtl);

		// fill the printing structure
		if (pDlg.fit == TRUE) {
			// a page fit require these values at 0
			printOptions.nImageUnits = 0;
			printOptions.nPaperUnits = 0;
		} else {
			printOptions.units = (pDlg.units == PIXELS) ? CTLUNIT_PIXEL :
								 ((pDlg.units==INCHES) ? CTLUNIT_INCH :
										CTLUNIT_MM);

			// WARNING! no checks are made to validate user inputs
			printOptions.nImageUnits = pDlg.m_graphics;
			printOptions.nPaperUnits = pDlg.m_paper;
		}

		PAN_CtlFileInfo	fi;
		if (::SendMessage(handle, PM_CTLGETFILE, 0,
				(LPARAM) (PAN_CtlFileInfo FAR*) &fi) != PAN_CTLERRNONE) {
			GetCurrentView()->m_pWndOut->SendToOutput("PRINT: Clone creation failed\n");
			return;
		}
		switch (fi.type) {
			case PAN_DocumentFile :
				if (::SendMessage(handle, PM_CTLGETPAGESIZE, 1,
				(LPARAM)  &fi.dimensions) != PAN_CTLERRNONE) {
					GetCurrentView()->m_pWndOut->SendToOutput("PRINT: Clone creation failed\n");
				return;
				}

				break;
			case PAN_VectorFile :			
				// George 24/01/2006: Bug # 52930
				// Set Extents to zero to indicate Zoom Fit for every page
				_fmemset(&(fi.dimensions), 0, sizeof(PAN_CtlRange));
				break;
			default:
				break;
		}
		printOptions.source = fi.dimensions;

		if (pDlg.marginsChanged) {
			// set the margins only if user changed something
			printOptions.margins.units = (pDlg.margins.units==INCHES) ? CTLUNIT_INCH : CTLUNIT_MM;
			printOptions.margins.top = pDlg.margins.m_top;
			printOptions.margins.left = pDlg.margins.m_left;
			printOptions.margins.bottom= pDlg.margins.m_bottom;
			printOptions.margins.right = pDlg.margins.m_right;

			::SendMessage(handle, PM_CTLVALIDATEMARGINS, 0, (LPARAM)(PAN_CtlPrintOptions FAR*)&printOptions);
		}

		printOptions.headers.topLeftText = pDlg.m_hdTopLeft;
		printOptions.headers.topCtrText = pDlg.m_hdTopCent;
		printOptions.headers.topRightText= pDlg.m_hdTopRight;
		printOptions.headers.botLeftText = pDlg.m_hdBotLeft;
		printOptions.headers.botCtrText = pDlg.m_hdBotCent;
		printOptions.headers.botRightText= pDlg.m_hdBotRight;

		printOptions.outputFileName = 0;

		// Set printer abort procedure.
#if TARGET == WIN32S || defined(_WIN64)
		ABORTPROC abortProc = (ABORTPROC) PrintAbortProc;
#else
		ABORTPROC abortProc = (ABORTPROC) ::MakeProcInstance(
			(FARPROC) PrintAbortProc, AfxGetInstanceHandle());
#endif

		if (abortProc == 0) {
			::SendMessage(handle, PM_CTLDESTROY, 0, 0);
			::DestroyWindow(handle);
			return;
		}

		if (SetAbortProc(printOptions.printDlg->hDC, abortProc) <= 0) {
#if TARGET != WIN32S && !defined(_WIN64)
			::FreeProcInstance((FARPROC) abortProc);
#endif
			::SendMessage(handle, PM_CTLDESTROY, 0, 0);
			::DestroyWindow(handle);
			return;
		}

		// Print control.
		::SendMessage(handle, PM_CTLPRINT, (WORD) -1,
			(LPARAM) (PAN_CtlPrintOptions FAR*) &printOptions);

		// Clean up.
#if TARGET != WIN32S && !defined(_WIN64)
		::FreeProcInstance((FARPROC) abortProc);
#endif

	} /* if IDOK */

	::SendMessage(handle, PM_CTLDESTROY, 0, 0);
	::DestroyWindow(handle);

	// Clean up
	DeleteDC(printDlg.hDC);
}

//------------------------------------------------------------------------------
/*
**	Call the PANCNVRT.EXE program to convert the contents of the control.
**	*** To be replaced.
*/
void CPanCtRgn::Convert()
{
	if (m_hCtl == NULL) {
		return;
	}

	PAN_CtlFileInfo fi;
	char cmd[1024];
	int	 err;

	err = (int) ::SendMessage(m_hCtl, PM_CTLGETFILE,
					0, (LPARAM) (PAN_CtlFileInfo FAR*) &fi);

	strcpy(cmd, "pancnvrt.exe ");		 // make cmd line
	strcat(cmd, fi.name);
	err = ::WinExec(cmd, SW_SHOW);		// execute
	if (err < 32) {
		::MessageBox((HWND) 0, "WinExec Error", "INFO", MB_ICONINFORMATION);
	}
}

//------------------------------------------------------------------------------
/*
** Get a search string from the user and send it to the control.
** This function does nothing if the control does not have search capability.
*/
void CPanCtRgn::Search()
{
	if (m_hCtl == NULL) {
		return;
	}

	if (!AcceptCaps(PAN_CTLCAPSSEARCH)) {
		GetCurrentView()->m_pWndOut->SendToOutput("Control does not support this command!");
		return;
	}

	CSrchDiag 		 dlg;

	dlg.m_direction_up = (m_si.fDown == FALSE);
	dlg.m_wrap = m_si.fWrap;
	dlg.m_matchCase = m_si.fCase;
	dlg.m_matchWord = m_si.fWord;
	dlg.m_findAll = m_si.fMulti;

	PAN_CtlInfo	info;

	::SendMessage(m_hCtl, PM_CTLGETINFO, 0, (LPARAM) (PAN_CtlInfo FAR *) &info);

	if (info.type == PAN_VectorFile) {
		::SendMessage(m_hCtl, PM_CTLGETPAGE, 0,
				(LPARAM) (LPWORD) &(m_si.startPos.page));

		m_si.startPos.flow = m_si.foundPos.flow;
		m_si.startPos.offset = m_si.foundPos.offset;

		if (! m_si.startPos.page) {
			m_si.startPos.page = 1;
		}
		if (! m_si.startPos.flow) {
			m_si.startPos.flow = 1L;
		}

	} else {
		if (::SendMessage(m_hCtl, PM_CTLGETCARETPOS, 0,
				(LPARAM) (PAN_CtlCaretPos FAR*) &m_si.startPos) != PAN_CTLERRNONE) {
			GetCurrentView()->m_pWndOut->SendToOutput("SEARCH: Cannot get current caret position");
			return;
		}
	}

	dlg.m_page = (DWORD) m_si.startPos.page;
	dlg.m_flow = m_si.startPos.flow;
	dlg.m_offs = m_si.startPos.offset;

	if (dlg.DoModal() == IDOK) {
		::UpdateWindow(m_hCtl);

		m_si.fDown = (dlg.m_direction_up == FALSE);		// search downward ?
		m_si.fWrap = dlg.m_wrap;						// wrap around file	?
		m_si.fCase = dlg.m_matchCase;					// case sensitive ?
		m_si.fWord = dlg.m_matchWord;					// whole word ?
		m_si.fMulti = dlg.m_findAll;					// multi search ?
		m_si.string = dlg.m_txt.GetBuffer(35);

		m_si.startPos.page = (int)dlg.m_page;
		m_si.startPos.flow = dlg.m_flow;
		m_si.startPos.offset = dlg.m_offs;

		int		err = (int) ::SendMessage(m_hCtl, PM_CTLSEARCH, 0,
						(LPARAM) (PAN_CtlSearchInfo FAR*) &m_si);

		if (err != PAN_CTLERRNONE) {
			GetCurrentView()->m_pWndOut->SendToOutput("SEARCH: Failed");
			GetCurrentView()->ErrorTrace(err, PM_CTLSEARCH);
			return;
		}

		if (m_si.fFound == 0) {
			GetCurrentView()->m_pWndOut->SendToOutput("String not found!");
		} else {
			if (m_si.fMulti == FALSE) {
				if (info.type == PAN_VectorFile) {
					PAN_CtlRange rg;
					Real		 extx, exty;

					// zoom on the found text string.
					rg.min = m_si.foundBBox[0];
					rg.max = m_si.foundBBox[2];

					if (rg.min.x > rg.max.x) {
						Real	tmpx = rg.max.x;

						rg.max.x = rg.min.x;
						rg.min.x = tmpx;
					}

					if (rg.min.y > rg.max.y) {
						Real	tmpy = rg.max.y;

						rg.max.y = rg.min.y;
						rg.min.y = tmpy;
					}

					extx = rg.max.x - rg.min.x;
					exty = rg.max.y - rg.min.y;

					rg.min.x -= extx;
					rg.max.x += extx;
					rg.min.y -= exty;
					rg.max.y += exty;

					::SendMessage(m_hCtl, PM_CTLSETVIEWEXTENTS, 0,
							(LPARAM) (PAN_CtlRange FAR*) &rg);

					::SendMessage(m_hCtl, PM_CTLPAINT, (WPARAM) 0, (LPARAM) 0);

					GetCurrentView()->m_pWndOut->SendToOutput(
						"[ %s ] found at: page %d, flow= %ld offs= %ld, handle = 0x%lx:%lx",
						m_si.string, m_si.foundPos.page, m_si.foundPos.flow, m_si.foundPos.offset,
						m_si.foundHandle.high, m_si.foundHandle.low);

					for (WORD i = 0; i < 4; i++) {
						GetCurrentView()->m_pWndOut->SendToOutput(
							"	Bounding box point %d = %lf, %lf, %lf",
							i+1, m_si.foundBBox[i].x, m_si.foundBBox[i].y, m_si.foundBBox[i].z);
					}
				} else {
					/* Taken care of in the control
					**	::SendMessage(m_hCtl, PM_CTLSETCARETPOS,
					**		0, (LPARAM)(PAN_CtlCaretPos FAR*)&m_si.foundPos);
					*/
					GetCurrentView()->m_pWndOut->SendToOutput(
						"[ %s ] found at: page %d, flow= %ld offs= %ld",
						m_si.string, m_si.foundPos.page, m_si.foundPos.flow, m_si.foundPos.offset);
				}
			} else {
				if (info.type == PAN_VectorFile) {
					// Display all the found positions and the bounding boxes.
					PAN_CtlCaretPos 	FAR* s = (PAN_CtlCaretPos FAR*) GlobalLock(m_si.hFoundPos);
					PAN_CtlPos		 	FAR* box = (PAN_CtlPos FAR*) GlobalLock(m_si.hFoundBBox);
					PAN_CtlHandle 		FAR* enthdl = NULL;

					/*
					** handle will be 0 for vector formats that do not support
					** handles.
					*/
					if (m_si.hFoundHandle) {
						enthdl = (PAN_CtlHandle FAR*) GlobalLock(m_si.hFoundHandle);
					}

					GetCurrentView()->m_pWndOut->SendToOutput("Found %d occurrences", m_si.fFound);
					for (WORD i = 0; i < m_si.fFound; i++) {
						if (m_si.hFoundHandle && enthdl) {
							GetCurrentView()->m_pWndOut->SendToOutput("[ %s ] found at: page %d, flow= %ld offs=%ld, handle = 0x%lx:%lx",
								m_si.string, s[i].page, s[i].flow, s[i].offset,
								enthdl[i].high, enthdl[i].low);
						} else {
							GetCurrentView()->m_pWndOut->SendToOutput("[ %s ] found at: page %d, flow= %ld offs=%ld",
								m_si.string, s[i].page, s[i].flow, s[i].offset);
						}

						for (WORD j = 0; j < 4; j++) {
							GetCurrentView()->m_pWndOut->SendToOutput(
								"	Bounding box point %d = %lf, %lf, %lf",
								j+1, box[4*i+j].x, box[4*i+j].y, box[4*i+j].z);
						}
					} /* for */

					GlobalUnlock(m_si.hFoundPos);
					GlobalFree(m_si.hFoundPos);
					m_si.hFoundPos = 0;
					if (m_si.hFoundHandle) {
						GlobalUnlock(m_si.hFoundHandle);
						GlobalFree(m_si.hFoundHandle);
						m_si.hFoundHandle = 0;
					}
					GlobalUnlock(m_si.hFoundBBox);
					GlobalFree(m_si.hFoundBBox);
					m_si.hFoundBBox = 0;
				} else {
					// Display all the found positions
					PAN_CtlCaretPos FAR* s = (PAN_CtlCaretPos FAR*) GlobalLock(m_si.hFoundPos);

					GetCurrentView()->m_pWndOut->SendToOutput("Found %d occurrences", m_si.fFound);
					for (WORD i = 0; i < m_si.fFound; i++) {
						GetCurrentView()->m_pWndOut->SendToOutput("[ %s ] found at: page %d, flow= %ld offs=%ld",
							m_si.string, s[i].page, s[i].flow, s[i].offset);
					} /* for */

					GlobalUnlock(m_si.hFoundPos);
					GlobalFree(m_si.hFoundPos);
					m_si.hFoundPos = 0;
				}
			}
		}
	}
}

//------------------------------------------------------------------------------
/*
** Get number of strings or a string value from an index.
** This function does nothing if the control does not have search capability.
*/
void CPanCtRgn::GetString()
{
	if (m_hCtl == NULL) {
		return;
	}

	if (!AcceptCaps(PAN_CTLCAPSSEARCH)) {
		GetCurrentView()->m_pWndOut->SendToOutput("Control does not support this command!");
		return;
	}

	CGetStringDlg	dlg;

	dlg.m_x = 0;

	if (dlg.DoModal() == IDOK) {
		DWORD	wParam = dlg.m_x;

		if (wParam == -1) {
			DWORD count = 0;

			int err = ::SendMessage(m_hCtl, PM_CTLGETSTRING, wParam,
					(LPARAM)(LPDWORD) &count);

			if (err != PAN_CTLERRNONE) {
				GetCurrentView()->m_pWndOut->SendToOutput("GET STRING COUNT: Failed");
				GetCurrentView()->ErrorTrace(err, PM_CTLGETSTRING);
				return;
			}

			GetCurrentView()->m_pWndOut->SendToOutput("String count = %ld", count);

		} else {
			char	szbuf[PAN_CTLMAXSTRING];

			*szbuf = '\0';

			int err = ::SendMessage(m_hCtl, PM_CTLGETSTRING, wParam,
					(LPARAM)(LPSTR) &szbuf);

			if (err != PAN_CTLERRNONE) {
				GetCurrentView()->m_pWndOut->SendToOutput("GET STRING: Failed");
				GetCurrentView()->ErrorTrace(err, PM_CTLGETSTRING);
				return;
			}
			GetCurrentView()->m_pWndOut->SendToOutput("String %d = %s",
				wParam, szbuf);
		}
	}
}

//------------------------------------------------------------------------------
/*
** Get handles of entities that intersect a certain region
** This function does nothing if the control does not have search capability.
*/
void CPanCtRgn::GetEntity()
{
	if (m_hCtl == NULL) {
		return;
	}

	if (!AcceptCaps(PAN_CTLCAPSSEARCH)) {
		GetCurrentView()->m_pWndOut->SendToOutput("Control does not support this command!");
		return;
	}

	PAN_CtlSelList sl;

	if (::SendMessage(m_hCtl, PM_CTLGETSELS, 0,
		(LPARAM)(PAN_CtlSelList FAR*) &sl) != PAN_CTLERRNONE) {
		return;
	}
//:
	if (sl.nSels == 0) {
		GetCurrentView()->m_pWndOut->SendToOutput("PM_CTLGETSELS: No selections available.");
		return;
	}

	PAN_CtlSel FAR* s = (PAN_CtlSel FAR*) GlobalLock(sl.hSels);

	if (s[0].selType == PAN_CTLSELECTION_VIEW) {
		PAN_CtlGetEntityInfo	GetEntityInfo;

		GetEntityInfo.bbox = s[0].range.vwRange;
//:		GetEntityInfo.bbox.max = GetEntityInfo.bbox.min;
		GetEntityInfo.iThreshold = 6;
		GetEntityInfo.nFound = 0L;
		GetEntityInfo.hFound = (HGLOBAL) 0;

		GlobalUnlock(sl.hSels);
		GlobalFree(sl.hSels);

		int err = ::SendMessage(m_hCtl, PM_CTLGETENTITY, 0,
					(LPARAM) (PAN_CtlGetEntityInfo FAR *) &GetEntityInfo);

		if (err != PAN_CTLERRNONE) {
			GetCurrentView()->m_pWndOut->SendToOutput("GET ENTITY: Failed");
			GetCurrentView()->ErrorTrace(err, PM_CTLGETENTITY);
			return;
		}

		if (GetEntityInfo.nFound != 0L) {
			DWORD				entcount = 0L;
			int					hdlcount = 0;
			int					inslevel;
			PAN_CtlHandle		Handle;
			char		huge	*entPtr = NULL;

			if (GetEntityInfo.hFound) {
				if ((entPtr = (char huge *) GlobalLock(GetEntityInfo.hFound)) == NULL) {
					GlobalFree(GetEntityInfo.hFound);
					GetEntityInfo.hFound = 0;
					return;
				}
			} else {
				return;
			}

			GetCurrentView()->m_pWndOut->SendToOutput("Found %d entities", GetEntityInfo.nFound);

			for (WORD i = 0; i < GetEntityInfo.nFound; i++) {
				hmemcpy(&inslevel, entPtr, sizeof(int));
				entPtr += sizeof(int);

				for (WORD j = 0; j < inslevel+1; j++) {
					hmemcpy(&Handle, entPtr, sizeof(PAN_CtlHandle));
					entPtr += sizeof(PAN_CtlHandle);

					if (!j) {
						GetCurrentView()->m_pWndOut->SendToOutput("Entity %d, Insertion level = %d, Entity handle = 0x%lx:%lx",
							i+1, inslevel, Handle.high, Handle.low);
					} else {
						GetCurrentView()->m_pWndOut->SendToOutput("		Insertion handle %d = 0x%lx:%lx",
							j, Handle.high, Handle.low);
					}
				}
			}
			GlobalUnlock(GetEntityInfo.hFound);
			GlobalFree(GetEntityInfo.hFound);
			GetEntityInfo.hFound = 0;
		}
	} else {
		GlobalUnlock(sl.hSels);
		GlobalFree(sl.hSels);
	}

	GetCurrentView()->m_pWndOut->SendToOutput("Done!");
}

//------------------------------------------------------------------------------
/*
** Disable the given capabilities.
*/
void CPanCtRgn::Disable(DWORD caps)
{
	if (m_hCtl == NULL) {
		return;
	}

	DWORD	allCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 1, (LPARAM) &allCaps);
	caps &= allCaps;

	DWORD	curCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 0, (LPARAM) &curCaps);
	curCaps &= ~caps;

	int ctlErr = (int) ::SendMessage(m_hCtl, PM_CTLSETCAPS, 0, (LPARAM) curCaps);
	if (ctlErr != PAN_CTLERRNOFILESET && ctlErr != PAN_CTLERRNONE) {
		GetCurrentView()->m_pWndOut->SendToOutput("ERROR PM_CTLSETCAPS: %i", ctlErr);
	}
}

//------------------------------------------------------------------------------
/*
** Enable the given capabilities.
*/
void CPanCtRgn::Enable(DWORD caps)
{
	if (m_hCtl == NULL) {
		return;
	}

	DWORD	allCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 1, (LPARAM) &allCaps);
	caps &= allCaps;

	DWORD	curCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 0, (LPARAM) &curCaps);
	curCaps |= caps;

	int		ctlErr = (int) ::SendMessage(m_hCtl, PM_CTLSETCAPS, 0,
								(LPARAM) curCaps);

	if ((ctlErr != PAN_CTLERRNOFILESET) && (ctlErr != PAN_CTLERRNONE)) {
		GetCurrentView()->m_pWndOut->SendToOutput("ERROR PM_CTLSETCAPS: %i",
				ctlErr);
	}
}

//------------------------------------------------------------------------------
/*
** Set the given mouse action.
*/
void CPanCtRgn::ChangeLMB(WORD lmb)
{
	if (m_hCtl == NULL) {
		return;
	}

	DWORD	curCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 0, (LPARAM) &curCaps);

	switch (lmb) {
	case PAN_CTLLMBZOOM:
		if (!(curCaps & PAN_CTLCAPSZOOM)) {
			lmb = PAN_CTLLMBNONE;
		}
		break;

	case PAN_CTLLMBSELECT:
		if (!(curCaps & PAN_CTLCAPSCOPY)) {
			lmb = PAN_CTLLMBNONE;
		}
		break;

	default:
		lmb = PAN_CTLLMBNONE;
		break;
	}

	int		ctlErr = (int) ::SendMessage(m_hCtl, PM_CTLSETLMBACTION,
								(WPARAM) lmb, 0);

	if ((ctlErr != PAN_CTLERRNOFILESET) && (ctlErr != PAN_CTLERRNONE)) {
		GetCurrentView()->m_pWndOut->SendToOutput("ERROR PM_CTLSETLMBACTION: %i",
				ctlErr);
	}
}

//------------------------------------------------------------------------------
/*
** Return TRUE if the control supports the given capabilities.
*/
BOOL CPanCtRgn::AcceptCaps(DWORD caps)
{
	if (m_hCtl == NULL) {
		return (FALSE);
	}

	DWORD	allCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 1, (LPARAM) &allCaps);

	return ((allCaps & caps) == caps);
}

//------------------------------------------------------------------------------
/*
** Return TRUE if the control supports the given mouse action.
*/
BOOL CPanCtRgn::AcceptLMB(WORD lmb)
{
	if (m_hCtl == NULL) {
		return (FALSE);
	}

	DWORD	curCaps;

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 0, (LPARAM) &curCaps);

	BOOL	result = FALSE;

	switch (lmb) {
	case PAN_CTLLMBZOOM:
		if (curCaps & PAN_CTLCAPSZOOM) {
			result = TRUE;
		}
		break;

	case PAN_CTLLMBSELECT:
		if (curCaps & PAN_CTLCAPSCOPY) {
			result = TRUE;
		}
		break;

	case PAN_CTLLMBNONE:
		result = TRUE;
		break;
	}
	return (result);
}

//------------------------------------------------------------------------------
/*
** Save the current control capabilities.
*/
void CPanCtRgn::SetCurCaps()
{
	if (m_hCtl == NULL) {
		return;
	}

	::SendMessage(m_hCtl, PM_CTLGETCAPS, 0, (LPARAM) &m_curCaps);
}

//------------------------------------------------------------------------------
/*
** Save the current control mouse action.
*/
void CPanCtRgn::SetCurLMB()
{
	if (m_hCtl == NULL) {
		return;
	}

	::SendMessage(m_hCtl, PM_CTLGETLMBACTION, 0, (LPARAM) &m_curLMB);
}

/*----------------------------------------------------------------------------*/

#if	TARGET==WIN16
#	pragma 	optimize("", on)
#endif
