// Markup Control Wrapper For AutoVue Module Header File, Message Translator 

#ifndef __MRKPCTL_H__
#define __MRKPCTL_H__

#include "stdafx.h"
#include "pctl.h"
#include "mrkupctl.h"

#define LPENTHANDLE LPMRKENTHANDLE
#define ENTHANDLE	MRKENTHANDLE

#define	COLORERROR		(COLORREF)-2

#pragma warning (disable :4135) //Conversion between different integral types

class CVCETControl;

class CMarkupControl : public CWnd
{
	DECLARE_DYNCREATE(CMarkupControl)

private:
	BOOL m_fLogicallyDestroyed;
	//	TRUE after control window has been logically destroyed,
	//	i.e. PM_CTLDESTROY was sent and succeeded

public:
	LRESULT SendMessage(UINT message, WPARAM wParam = 0, LPARAM lParam = 0);
	//	Sends the given message to the control window

//	Static functions
private:
#	if TARGET==WIN16
	static	LRESULT __loadds CALLBACK
#else
	static	LRESULT CALLBACK
#endif
		DefNotifProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
	//	Default VCET notification procedure
	
#	if TARGET==WIN16
	static	int	 __loadds CALLBACK
#else
	static	int CALLBACK
#endif
		DefEventProc(HWND hWnd);
	//	Default VCET event procedure

protected:
	virtual LRESULT OnNotification(UINT msg, WPARAM wParam, LPARAM lParam);
	//	Called by DefNotifProc() to notify the controlling application
	//	This function is meant to be overridden by the controlling application
	//	to handle the various VCET notification messages
	virtual int OnEvent();
	//	Called by DefEventProc()
	//	This function is meant to be overridden by the controlling application

public:
	CMarkupControl()
	// Constructs the control
	:	m_fLogicallyDestroyed(FALSE)
	{
		m_lpfnNotifyProc = NULL;
		m_lpfnEventProc = NULL;
		m_fValid = FALSE;
		m_nActiveLayer = 0;
		m_fIsNotModified = FALSE;
	}

	//	Destroys the control
	virtual ~CMarkupControl()
	{
		if (m_hWnd && ::IsWindow(m_hWnd)) {
			DestroyWindow();
		}

		if (m_lpfnNotifyProc != 0) {
			FreeProcInstance((FARPROC) m_lpfnNotifyProc);
		}
		m_lpfnNotifyProc = 0;

		if (m_lpfnEventProc != 0) {
			FreeProcInstance((FARPROC) m_lpfnEventProc);
		}
		m_lpfnEventProc = 0;
	}


	//	Creates the control
	BOOL	Create(const RECT& rect, CWnd* pParentWnd, HWND hCtl);

	virtual BOOL DestroyWindow();
	//	Destroys the control window

	BOOL	IsValid() const
	{
		return (m_fValid && !m_fLogicallyDestroyed);
	}

//
//
// Markup API interface messages wrapper
//
//
	BOOL SetPage(int nPage)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_SETPAGE, (WPARAM) nPage);
	}
	
	int GetPage()
	{
		ASSERT_VALID(this);
		int nPage;
		SendMessage(MRK_GETPAGE, 0, (LPARAM)(LPINT)&nPage);
		return nPage;
	}

	int GetNumPages()
	{
		ASSERT_VALID(this);
		return (int) SendMessage(MRK_GETNUMPAGES);
	}
	
	BOOL GetPageSize(PAN_CtlRange FAR* psize)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_GETPAGESIZE, 0, (LPARAM)psize);
	}

	BOOL SetBaseExtents(const PAN_CtlRange* pExtents)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_SETBASEEXTENTS, 0, (LPARAM) pExtents);
	}

	BOOL GetBaseExtents(PAN_CtlRange* pExtents)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_GETBASEEXTENTS, 0, (LPARAM) pExtents);
	}    

	void ClientToWorld(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_CLIENTTOWORLD, 0, (LPARAM) pPos);
	}

	void WorldToClient(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_WORLDTOCLIENT, 0, (LPARAM) pPos);
	}

	int SetEntity(int nEntityID, LPCSTR lpszUDEName = NULL)
	{
		ASSERT_VALID(this);
		return (int)SendMessage(MRK_SETENTITY, nEntityID, (LPARAM)lpszUDEName);
	}

	int GetEntity(LPSTR *lpszUDEName = NULL)
	{
		ASSERT_VALID(this);
		int nEntity = (int) SendMessage(MRK_GETENTITY, 0, (LPARAM)lpszUDEName);
		return nEntity;
	}

	int SetSel(int nEntityID, const PAN_CtlRange* pRegion)
	{
		ASSERT_VALID(this);
		return (int) SendMessage(MRK_SETSEL, (WPARAM) nEntityID, (LPARAM) pRegion);
	}

	int GetSels(int MaxSize, LPENTHANDLE lpHandles, int EntType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);
		
		MRK_GetEntities	GetEntStruct;
		memset(&GetEntStruct, 0, sizeof(GetEntStruct));
		
		GetEntStruct.Type = EntType;
		GetEntStruct.nMaxSize = MaxSize;
		GetEntStruct.pHandles = lpHandles;
		
		return SendMessage(MRK_GETENTITIES, GET_SELECTED, (LPARAM)&GetEntStruct);
	}

	int GetSelCount(int EntType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);
		
		MRK_GetEntities	GetEntStruct;
		memset(&GetEntStruct, 0, sizeof(GetEntStruct));
		GetEntStruct.Type = EntType;
		
		return SendMessage(MRK_GETENTITIES, GET_SELECTED, (LPARAM)&GetEntStruct);
	}
	
	BOOL EditEntity(ENTHANDLE handle)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_EDITENTITY, 0, (LPARAM)handle);
	}
	
	void ClearSel()
	{
		ASSERT_VALID(this);
		SendMessage(MRK_CLEARSEL);
	}
	
	void EnumEntities(FARPROC lpEntEnumProc, int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_EnumEntities	EnumStruct;
		memset(&EnumStruct, 0, sizeof(EnumStruct));
		
		EnumStruct.Type = nEntityType;
		EnumStruct.pCallbackFn = lpEntEnumProc;
		
		SendMessage(MRK_ENUMENTITIES, ENUM_ALLENTITIES, (LPARAM)&EnumStruct);
	}

	void EnumSelEntities(FARPROC lpSelEnumProc, int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_EnumEntities	EnumStruct;
		memset(&EnumStruct, 0, sizeof(EnumStruct));
		
		EnumStruct.Type = nEntityType;
		EnumStruct.pCallbackFn = lpSelEnumProc;
		
		SendMessage(MRK_ENUMENTITIES, ENUM_SELECTED, (LPARAM)&EnumStruct);
	}

	void EnumChildren(FARPROC lpEnumProc, ENTHANDLE Handle, int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_EnumEntities	EnumStruct;
		memset(&EnumStruct, 0, sizeof(EnumStruct));

		EnumStruct.Handle = Handle;
		EnumStruct.Type = nEntityType;
		EnumStruct.pCallbackFn = lpEnumProc;
		
		SendMessage(MRK_ENUMENTITIES, ENUM_ENTCHILDREN, (LPARAM)&EnumStruct);
	}

	int GetNumUdes()
	{
		ASSERT_VALID(this);
		return (int) SendMessage(MRK_GETUDEINFO, 0, 0);
	}
	
	void GetUdeInfo(int nUdeNum, LPUDEINFO lpUdeInfo)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_GETUDEINFO, nUdeNum, (LPARAM) lpUdeInfo);
	}


	void SetLayerColor(COLORREF color, int nLayer = -1)
	{
		ASSERT_VALID(this);
		ASSERT(nLayer < GetNumLayers());

		LPPAN_LAYER pLayers;
		if ((pLayers = new PAN_LAYER[GetNumLayers()]) == NULL) {
			ASSERT(0);
			return;
		}
		int nLayerNum = nLayer >= 0 ? nLayer : GetActiveLayer();
		pLayers[nLayerNum].color = color;
		SetLayers(GetNumLayers(), pLayers);
		delete [] pLayers;
	}
	
	void SetCurrentColor(COLORREF color)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SET_EPROP, ME_COLOR, (LPARAM)color);
	}
	
	void SetFgBgColor(BOOL fBackgroundColor, COLORREF color)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SETFGBGCOLOR, fBackgroundColor, (LPARAM)color);
	}
	
	COLORREF GetCurrentColor()
	{
		ASSERT_VALID(this);
		COLORREF	color;
		SendMessage(MRK_GET_EPROP, ME_COLOR, (LPARAM) &color);
		return color;
	}

	void SetPalette(int nEntries, const LPPALETTEENTRY lpPaletteEntry)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SETPALETTE, (WPARAM)nEntries, (LPARAM)lpPaletteEntry);
	}
	
	int GetPalette(LPPALETTEENTRY lpPaletteEntry, int nEntries = 0)
	{
		PALETTEENTRY	palette;
		ASSERT_VALID(this);
		int result = SendMessage(MRK_GETPALETTE, (WPARAM)nEntries, (LPARAM)&palette);	
		return result;
	}

	int	GetCurrentFillType()
	{
		ASSERT_VALID(this);
		int FillType = MRK_FILLNONE;
		SendMessage(MRK_GET_EPROP, ME_FILLTYPE, (LPARAM) &FillType);
		return FillType;
	}
	void SetCurrentFillType(int FillType)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SET_EPROP, ME_FILLTYPE, (LPARAM) FillType);
	}

	COLORREF GetCurrentFillColor()
	{
		ASSERT_VALID(this);
		COLORREF FillColor = (COLORREF)-1;
		SendMessage(MRK_GET_EPROP, ME_FILLCOLOR, (LPARAM) &FillColor);
		return FillColor;
	}
	void SetCurrentFillColor(COLORREF FillColor)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SET_EPROP, ME_FILLCOLOR, (LPARAM) FillColor);
	}
	
	void SetViewExtents(PAN_CtlRange FAR* lprgExtents, BOOL fEraseBgrnd = FALSE)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SETVIEWEXTENTS, fEraseBgrnd, (LPARAM)lprgExtents);
	}	
	
	void GetViewExtents(PAN_CtlRange FAR* lprgExtents)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_GETVIEWEXTENTS, 0, (LPARAM)lprgExtents);
	}
	
	void SetFont(const CMrkFont& font
				 ) {
		ASSERT_VALID(this);
		SendMessage(MRK_SET_EPROP, ME_FONT, (LPARAM)&font);
	}
	
	void GetFont(CMrkFont& font
				 ) {
		ASSERT_VALID(this);
		SendMessage(MRK_GET_EPROP, ME_FONT, (LPARAM)&font);
	}

	void SetAction(int nActionCode, DWORD dwParameter = 0)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SETACTION, (WPARAM)nActionCode, (LPARAM)dwParameter);
	}
	
	int GetAction(LPINT lpnAction = NULL)
	{
		ASSERT_VALID(this);
		int	nMrkpAction;
		SendMessage(MRK_GETACTION, 0, (LPARAM)(LPINT) &nMrkpAction);
		if (lpnAction) {
			*lpnAction = (int) nMrkpAction;
		}
		return nMrkpAction;
	}
	
	void DeleteAllEntities()
	{
		ASSERT_VALID(this);
		SendMessage(MRK_REDDELALL);
	}
	
	void Quit()
	{
		ASSERT_VALID(this);
		SendMessage(MRK_QUIT);
		m_fLogicallyDestroyed = TRUE;
	}
	
	void SetLineWidth(int nLineWidth)
	{
		ASSERT_VALID(this);
		// Force to be pixel-units
		if (nLineWidth > 0) {
			nLineWidth = -nLineWidth;
		}
		double width = nLineWidth;
		SendMessage(MRK_SET_EPROP, ME_PENWIDTH, (LPARAM)&width);
	}
	
	int GetLineWidth()
	{
		ASSERT_VALID(this);
		double width;
		SendMessage(MRK_GET_EPROP, ME_PENWIDTH, (LPARAM)&width);

		if (width < 0.0) {
			return (int) -width;
		} else {
			return 0;
		}
	}

	int	MarkupPenStyle(int nPenStyle)
	{
		return	nPenStyle == PS_SOLID		? MRK_PENSTYLE_SOLID		:
				nPenStyle == PS_DASH		? MRK_PENSTYLE_DASH			:
				nPenStyle == PS_DOT			? MRK_PENSTYLE_DOT			:
				nPenStyle == PS_DASHDOT		? MRK_PENSTYLE_DASHDOT		:
				nPenStyle == PS_DASHDOTDOT	? MRK_PENSTYLE_DASHDOTDOT	:
				nPenStyle == PS_NULL		? MRK_PENSTYLE_HOLLOW		:
								MRK_PENSTYLE_SOLID;
	}

	int	WindowsPenStyle(int nPenStyle)
	{
		return	nPenStyle == MRK_PENSTYLE_SOLID			? PS_SOLID		:
				nPenStyle == MRK_PENSTYLE_DASH			? PS_DASH		:
				nPenStyle == MRK_PENSTYLE_DOT			? PS_DOT		:
				nPenStyle == MRK_PENSTYLE_DASHDOT		? PS_DASHDOT	:
				nPenStyle == MRK_PENSTYLE_DASHDOTDOT	? PS_DASHDOTDOT	:
				nPenStyle == MRK_PENSTYLE_HOLLOW		? PS_NULL		:
								PS_SOLID;
	}

	void SetPenStyle(int nPenStyle)
	{
		ASSERT_VALID(this);

		// Map to markup defined PenStyles.
		nPenStyle = MarkupPenStyle(nPenStyle);		
						
		SendMessage(MRK_SET_EPROP, ME_PENSTYLE, (LPARAM)nPenStyle);
	}
	
	int GetPenStyle()
	{
		ASSERT_VALID(this);
		int nPenStyle;
		SendMessage(MRK_GET_EPROP, ME_PENSTYLE, (LPARAM)&nPenStyle);

		// Map from markup defined PenStyles.
		nPenStyle = WindowsPenStyle(nPenStyle);
		
		return nPenStyle;
	}

	BOOL IsModified()
	{
		// Make more natural looking function
		ASSERT_VALID(this);
		int nResult;
		SendMessage(MRK_GET_MOPROP, MO_ISMODIFIED, (LPARAM)&nResult);
		return nResult != 0;
	}
	
	void SetIsModified(BOOL fModified = TRUE)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SET_MOPROP, MO_ISMODIFIED, (LPARAM) fModified);
	}

	int GetNumEntities(int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_GetEntities	GetEntStruct;
		memset(&GetEntStruct, 0, sizeof(GetEntStruct));
		
		GetEntStruct.Type = nEntityType;
		
		return SendMessage(MRK_GETENTITIES, GET_ALLENTITIES, (LPARAM)&GetEntStruct);
	}	

	int GetEntities(int MaxSize, LPENTHANDLE pHandleBuf, int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_GetEntities	GetEntStruct;
		memset(&GetEntStruct, 0, sizeof(GetEntStruct));
		
		GetEntStruct.Type = nEntityType;
		GetEntStruct.nMaxSize = MaxSize;
		GetEntStruct.pHandles = pHandleBuf;
		
		return SendMessage(MRK_GETENTITIES, GET_ALLENTITIES, (LPARAM)&GetEntStruct);
	}

	int GetNumChildren(ENTHANDLE Handle, int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_GetEntities	GetEntStruct;
		memset(&GetEntStruct, 0, sizeof(GetEntStruct));

		GetEntStruct.Handle = Handle;
		GetEntStruct.Type = nEntityType;
		
		return SendMessage(MRK_GETENTITIES, GET_ENTCHILDREN, (LPARAM)&GetEntStruct);
	}	

	int GetChildren(ENTHANDLE Handle, int MaxSize, LPENTHANDLE pHandleBuf, int nEntityType = ENTTYPE_NULL)
	{
		ASSERT_VALID(this);

		MRK_GetEntities	GetEntStruct;
		memset(&GetEntStruct, 0, sizeof(GetEntStruct));

		GetEntStruct.Handle = Handle;
		GetEntStruct.Type = nEntityType;
		GetEntStruct.nMaxSize = MaxSize;
		GetEntStruct.pHandles = pHandleBuf;
		
		return SendMessage(MRK_GETENTITIES, GET_ENTCHILDREN, (LPARAM)&GetEntStruct);
	}

	BOOL LockEntity(LPMRK_EntitySpec pEntity)
	{
		ASSERT_VALID(this);
		if (SendMessage(MRK_LOCKENTITY, 0, (LPARAM)pEntity)) {
			pEntity->Com.PenStyle = WindowsPenStyle(pEntity->Com.PenStyle);
			return TRUE;
		}
		return FALSE;
	}

	BOOL UnlockEntity(LPMRK_EntitySpec pEntity, BOOL fApplyChanges)
	{
		ASSERT_VALID(this);
		pEntity->Com.PenStyle = MarkupPenStyle(pEntity->Com.PenStyle);
		return (BOOL) SendMessage(MRK_UNLOCKENTITY, fApplyChanges, (LPARAM)pEntity);
	}

	ENTHANDLE AddEntity(const LPMRK_EntitySpec lpEntitySpec)
	{
		ASSERT_VALID(this);
		return (ENTHANDLE)SendMessage(MRK_ADDENTITY, 0, (LPARAM)lpEntitySpec);
	}
	
	void RenderOntoDC(const MRK_RenderOptions FAR* lpcRenderOptions)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_RENDERONTODC, 0, (LPARAM)lpcRenderOptions);
	}

	void RenderOntoDC(const PAN_CtlRenderOptions FAR* lpcRenderOptions)
	{
		ASSERT_VALID(this);
		
		ASSERT(lpcRenderOptions != NULL);

		MRK_RenderOptions 	Options;
		memset(&Options, 0, sizeof(MRK_RenderOptions));

		Options.hdc = lpcRenderOptions->hdc;
		Options.source = lpcRenderOptions->source;
		Options.devRect = lpcRenderOptions->devRect;
		
		SendMessage(MRK_RENDERONTODC, 0, (LPARAM)&Options);
	}
	
	BOOL AddNewMarkup()
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_NEW);
	}

	BOOL DeleteMarkup(int nIndex)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_DELETE, nIndex);
	}

	BOOL CopyLayer(int nDestMarkup, int nDestLayer, int nSrcMarkup, int nSrcLayer)
	{
		ASSERT_VALID(this);
		MRK_CopyLayer			copyStruct;
		copyStruct.nDestMO		= nDestMarkup;
		copyStruct.nDestLayer	= nDestLayer;
		copyStruct.nSrcMO		= nSrcMarkup;
		copyStruct.nSrcLayer	= nSrcLayer;
		return (BOOL) SendMessage(MRK_COPYLAYER, 0, (LPARAM) &copyStruct);
	}

	BOOL Save(LPCSTR lpcszMarkupFileName)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_SAVE, 0, (LPARAM)lpcszMarkupFileName);
	}
	
	BOOL Read(LPCSTR lpcszMarkupFileName)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_READ, 0, (LPARAM)lpcszMarkupFileName);
	}
	
	BOOL ShowMarkup(BOOL fShow)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_SET_MOPROP, MO_VISIBLE, fShow);
	}
	
	void SetCurrentMarkupInfo(LPCSTR lpszMarkupInfo)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SET_MOPROP, MO_INFO, (LPARAM)lpszMarkupInfo);
	}

	void GetCurrentMarkupInfo(LPSTR lpszMarkupInfo)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_GET_MOPROP, MO_INFO, (LPARAM)lpszMarkupInfo);
	}

	void SetBaseInfo(LPMRK_BaseInfo lpBaseInfo)
	{
		ASSERT_VALID(this);
		ASSERT(lpBaseInfo != NULL);
		
		SendMessage(MRK_SET_MOPROP, MO_INS, (LPARAM)lpBaseInfo);
	}

	void GetBaseInfo(LPMRK_BaseInfo lpBaseInfo)
	{
		ASSERT_VALID(this);
		ASSERT(lpBaseInfo != NULL);
		
		SendMessage(MRK_GET_MOPROP, MO_INS, (LPARAM)lpBaseInfo);
	}
	
	HCURSOR SetMarkupCursor(HCURSOR hCursor)
	{
		ASSERT_VALID(this);
		HCURSOR	hOldCursor = NULL;
		SendMessage(MRK_GET_MOPROP, MO_CURSOR, (LPARAM) &hOldCursor);
		SendMessage(MRK_SET_MOPROP, MO_CURSOR, (LPARAM) (LPCSTR)hCursor);
		return hOldCursor;
	}

	BOOL IsMarkupVisible()
	{
		ASSERT_VALID(this);
		int reslt;
		SendMessage(MRK_GET_MOPROP, MO_VISIBLE, (LPARAM)&reslt);
		return reslt != 0;
	}

	BOOL HasEntities(int nEntityType = 0)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_GETENTITIES, nEntityType);
	}
	
	BOOL IsMarkupFile(LPCSTR lpcszMarkupFileName)
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_ISMRKFILE, 0, (LPARAM)lpcszMarkupFileName);
	}
	
	int GetNumMarkups()
	{
		ASSERT_VALID(this);
		return SendMessage(MRK_GETNUMMARKUPS);
	}

	int GetNumLayers()
	{
		ASSERT_VALID(this);
		return GetLayers(0, NULL);
	}

	void SetLayers(int nLayers, const LPPAN_LAYER lpLayers)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SETLAYERS, nLayers, (LPARAM)lpLayers);
	}
	
	int GetLayers(int nLayers, LPPAN_LAYER lpLayers)
	{
		ASSERT_VALID(this);
		return SendMessage(MRK_GETLAYERS, nLayers, (LPARAM)lpLayers);
	}	
	
	void SetActiveLayer(int nLayer)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_SET_EPROP, ME_LAYER, nLayer);
	}
	
	int GetActiveLayer()
	{
		ASSERT_VALID(this);
		int nLayerNum;
		SendMessage(MRK_GET_EPROP, ME_LAYER, (LPARAM)&nLayerNum);
		return nLayerNum;
	}

	int GetActiveMarkup()
	{
		return SendMessage(MRK_GETACTIVE);
	}
	
	BOOL SetActiveMarkup(int nMarkup)
	{
		return SendMessage(MRK_SETACTIVE, nMarkup);
	}

	void Undo()
	{
		ASSERT_VALID(this);
		SendMessage(MRK_UNDO);
	}
	
	void Redo(int fRepeat = 0)
	{
		ASSERT_VALID(this);
		SendMessage(MRK_REDO, fRepeat);
	}
	
	BOOL IsRedoAvailable()
	{
		ASSERT_VALID(this);
		return SendMessage(MRK_REDO, 1);
	}
	
	BOOL IsUndoAvailable()
	{
		ASSERT_VALID(this);
		return SendMessage(MRK_UNDO, 1);
	}

	int		GetRotation(BOOL fRelative = FALSE)
	{
		double	degrees = 0.0;
		SendMessage(MRK_GETROTATE, fRelative ? 1 : 0, (LPARAM) &degrees );
		return (int) ( degrees + 0.5 );		// rounding value for proper results
	}
	int		SetRotation(int rotation = 0, BOOL fRelative = FALSE)
	{
		double	degrees = rotation;
		return SendMessage(MRK_SETROTATE, fRelative ? 1 : 0, (LPARAM)&degrees);
	}

	int		GetFlip(BOOL fRelative = FALSE)
	{
		int	flipping = 0;
		SendMessage( MRK_GETFLIP, fRelative ? 1 : 0, (LPARAM) &flipping );
		return flipping;
	}
	int		SetFlip(int flipping = FLIP_NULL, BOOL fRelative = FALSE)
	{
		return SendMessage(MRK_SETFLIP, fRelative == FALSE ? 0 : 1, flipping);
	}
	void	GetMarkupInfo(
				CString &strUser,
				CString &strDepartment,
				CString &strCompany,
				CString &strLocation,
				CString &strTelephone);

	void	SetMarkupInfo(
				const CString &strUser,
				const CString &strDepartment,
				const CString &strCompany,
				const CString &strLocation,
				const CString &strTelephone);

	int NotePreparePrint(LPMRK_NotePrint pStruct)
	{
		ASSERT_VALID(this);
		ASSERT(pStruct != NULL);
		return (int) SendMessage(MRK_NOTEPREPAREPRINT, 0, (LPARAM)pStruct);
	}

	BOOL NoteEndPrint()
	{
		ASSERT_VALID(this);
		return (BOOL) SendMessage(MRK_NOTEENDPRINT, 0, 0L);
	}

	BOOL NoteRenderPage(LPMRK_NoteRender pStruct)
	{
		ASSERT_VALID(this);
		ASSERT(pStruct != NULL);
		return (BOOL) SendMessage(MRK_NOTERENDERPAGE, 0, (LPARAM) pStruct);
	}

	BOOL NoteBatchPrint(LPMRK_NotePrint pStruct)
	{
		ASSERT_VALID(this);
		ASSERT(pStruct != NULL);
		return (BOOL) SendMessage(MRK_NOTEBATCHPRINT, 0, (LPARAM) pStruct);
	}

	// Utility functions to get/set attributes of selected entities
	void		GetSelectedFont(CMrkFont& font);
	COLORREF	GetSelectedColor();
	int			GetSelectedPenStyle();
	int			GetSelectedPenWidth();
	void		SetSelectedFont(const CMrkFont& font);
	void		SetSelectedColor(COLORREF col);
	void		SetSelectedFillColor(COLORREF col);
	void		SetSelectedPenStyle(int penStyle);
	void		SetSelectedPenWidth(int penWidth);

	CString		GetMarkupFileName()	{ return m_strMarkupFile; }
	void		SetMarkupFileName(CString &str)	{ m_strMarkupFile = str; }
	CString		GetShortID() { return CString("001"); }
	CString		GetLongID() { return CString("001"); }

private:
	WNDPROC	m_lpfnNotifyProc;
	FARPROC	m_lpfnEventProc;
	CString	m_strMarkupFile;

protected:
	BOOL	m_fValid;
	int		m_nActiveLayer;
	BOOL	m_fIsNotModified;

#ifdef _DEBUG
public:
	void AssertValid() const;
#endif

	// Generated message map functions
protected:
	//{{AFX_MSG(CMarkupControl)
		// NOTE - the ClassWizard will add and remove member functions here.
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()

};

class CMrkpLibrary
{
public:
	static BOOL Initialize()
	{
		if (! fInitialized) {
			if (!LoadMarkupControl()) {
				return FALSE;
			}
			fInitialized = TRUE;
		}
		return TRUE;
	}

	static void CleanUp()
	{
		if (fInitialized) {
			FreeMarkupControl();
			fInitialized = FALSE;
		}
	}
	static	BOOL	IsInitialized() 	{return(fInitialized);}
private:
	static	BOOL	fInitialized;
};

// Wrappers for markup entites.
class CEntityInfo : public CObject
{
public:
	CEntityInfo(MRK_EntitySpec *pEntitySpec = NULL)
	{
		if (pEntitySpec != NULL) {
			m_EntitySpec = *pEntitySpec;
		} else {
			memset(&m_EntitySpec, 0, sizeof(MRK_EntitySpec));
		}
	}

public:
	MRK_EntitySpec		m_EntitySpec;

public:
	virtual ENTHANDLE	GetHandle()	const
	{
		return m_EntitySpec.Com.Handle;
	}
};

class CLinkInfo : public CEntityInfo
{
public:
	CLinkInfo(MRK_EntitySpec *pEntitySpec = NULL)
		: CEntityInfo(pEntitySpec)
	{
	}
	
	// Add class specific members.
};

class CNoteInfo : public CEntityInfo
{
public:
	CNoteInfo(MRK_EntitySpec *pEntitySpec = NULL)
		: CEntityInfo(pEntitySpec)
	{
	}
	
	// Add class specific members.
};

#endif //__MRKPCTL_H__


