//	vcet.h - Public library interface

#ifndef vcet_h
#define vcet_h

#ifndef __AFXWIN_H__
#error Include 'stdafx.h' before including this file
#endif

class CVCETLibrary
//	Encapsulates library management functions
{
//	Static variables
private:
	static UINT m_version;
	//	Library version number
	//	The two least significant digits represent the minor version
	//	The remaining digits represent the major version
	//	E.g. 1230 means version 12.30

//	Static functions
public:
	static BOOL Initialize(LPCSTR profileName, LPCSTR regName);
	//	Initializes the library

	static void CleanUp();
	//	Frees resources allocated by the library

	static UINT GetVersion()
	//	Returns the library version number
	{
		return m_version;
	}
};


class CVCETControl : public CWnd
//	Defines a C++ interface to a VCET Control
{
	DECLARE_DYNCREATE(CVCETControl)

//	Variables
private:
	BOOL m_fLogicallyDestroyed;
	//	TRUE after control window has been logically destroyed,
	//	i.e. PM_CTLDESTROY was sent and succeeded

//	Static functions
private:
#	if TARGET==WIN16
	static	LRESULT __loadds CALLBACK
#else
	static	LRESULT CALLBACK
#endif
		DefNotifProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
	//	Default VCET notification procedure
	
#	if TARGET==WIN16
	static	int	 __loadds CALLBACK
#else
	static	int CALLBACK
#endif
		DefEventProc(HWND hWnd);
	//	Default VCET event procedure

//	Functions
private:
	BOOL Create(LPCSTR className, LPCSTR windowName, DWORD style,
		const RECT& rect, CWnd* pParentWnd, UINT childID, CCreateContext* pContext)
	//	DISABLED FUNCTION
	//	Use public version instead
	{
		ASSERT(FALSE); return FALSE;
	}

public:
	LRESULT SendMessage(UINT message, WPARAM wParam = 0, LPARAM lParam = 0);
	//	Sends the given message to the control window
protected:

	virtual LRESULT OnNotification(UINT msg, WPARAM wParam, LPARAM lParam);
	//	Called by DefNotifProc() to notify the controlling application
	//	This function is meant to be overridden by the controlling application
	//	to handle the various VCET notification messages
	virtual int OnEvent();
	//	Called by DefEventProc()
	//	This function is meant to be overridden by the controlling application

public:
	CVCETControl()
	//	Constructs the control,
		: m_fLogicallyDestroyed(FALSE)
	{
		m_wRotate = 0;
		m_wFlip   = PAN_CTLFLIPNONE;
		m_lpfnNotifyProc = (WNDPROC)MakeProcInstance((FARPROC)DefNotifProc, AfxGetInstanceHandle());
		m_lpfnEventProc = MakeProcInstance((FARPROC)DefEventProc, AfxGetInstanceHandle());
	}

	//	Creates the control
	void Create(const RECT& rect, CWnd* pParentWnd,
		UINT ctlID, FARPROC pEventProc, WNDPROC pNotifyProc);

	//	Creates the control with no notification procs
	void Create(const RECT& rect, CWnd* pParentWnd,	UINT ctlID);

	//	Destroys the control
	virtual ~CVCETControl()
	{
		if (::IsWindow(this->GetSafeHwnd())) {
			DestroyWindow();
		}
		if (m_lpfnNotifyProc != 0) {
			FreeProcInstance((FARPROC) m_lpfnNotifyProc);
		}
		m_lpfnNotifyProc = 0;

		if (m_lpfnEventProc != 0) {
			FreeProcInstance((FARPROC) m_lpfnEventProc);
		}
		m_lpfnEventProc = 0;
	}

	CVCETControl* Clone(CWnd* pParentWnd, UINT ctlID);
	//	Clones the control

	virtual BOOL DestroyWindow();
	//	Destroys the control window
	//	This function must be called until it returns TRUE

	void GetInfo(PAN_CtlInfo* pInfo)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETINFO, 0, (LPARAM) pInfo);
	}

	DWORD GetStatus()
	{
		ASSERT_VALID(this);
		DWORD status;
		SendMessage(PM_CTLGETSTATUS, 0, (LPARAM) &status);
		return status;
	}

	void SetMode(BOOL fAllModes, DWORD mode)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETMODE, (WPARAM) fAllModes, (LPARAM) mode);
	}

	DWORD GetMode(BOOL fAllModes)
	{
		ASSERT_VALID(this);
		DWORD mode;
		SendMessage(PM_CTLGETMODE, (WPARAM) fAllModes, (LPARAM) &mode);
		return mode;
	}

	void SetCaps(BOOL fAllCaps, DWORD caps)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETCAPS, (WPARAM) fAllCaps, (LPARAM) caps);
	}

	DWORD GetCaps(BOOL fAllCaps)
	{
		ASSERT_VALID(this);
		DWORD caps;
		SendMessage(PM_CTLGETCAPS, (WPARAM) fAllCaps, (LPARAM) &caps);
		return caps;
	}

	void SetLMBAction(WORD action)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETLMBACTION, (WPARAM) action, 0);
	}

	void SetRMBAction(WORD action)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETLMBACTION, (WPARAM) action, 1);
	}

	WORD GetLMBAction()
	{
		ASSERT_VALID(this);
		WORD action = 0;
		SendMessage(PM_CTLGETLMBACTION, 0, (LPARAM) &action);
		return action;
	}

	WORD GetRMBAction()
	{
		ASSERT_VALID(this);
		WORD action = 0;
		SendMessage(PM_CTLGETLMBACTION, 1, (LPARAM) &action);
		return action;
	}

	void GetFileFmts(PAN_CtlFileFmtList* pList)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETFILEFMTS, 0, (LPARAM) pList);
	}

	LONG GetFileType(LPCSTR fileName)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETFILETYPE, 0, (LPARAM) fileName);
		return (LONG) result;
	}

	void SetFile(LPCSTR fileName, WORD baseFmtIndex = -1)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETFILE, (WPARAM) baseFmtIndex, (LPARAM) fileName);
	}

	void GetFile(PAN_CtlFileInfo* pInfo)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETFILE, 0, (LPARAM) pInfo);
	}

	int GetNumPages()
	{
		ASSERT_VALID(this);
		int nPages;
		SendMessage(PM_CTLGETNUMPAGES, 0, (LPARAM) &nPages);
		return nPages;
	}

	void GetPageSize(PAN_CtlRange* pRange, WORD index = 0)
	{
		ASSERT_VALID(this);
		if (index == 0) {
			index = GetPage();
		}
		SendMessage(PM_CTLGETPAGESIZE, (WPARAM) index, (LPARAM) pRange);
	}

	void SetPage(WORD pageNum)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETPAGE, (WPARAM) pageNum);
	}

	WORD GetPage()
	{
		ASSERT_VALID(this);
		WORD	pageNum = 0;
		if (m_hWnd && IsWindow(m_hWnd)) {
			SendMessage( PM_CTLGETPAGE, 0, (LPARAM) &pageNum);
		}
		// Work aroung for bug in DC control: If you set a file and
		// then immediately getpage, get page 0
		if (pageNum == 0) {
			pageNum = 1;
		}
		return pageNum;
	}

	void Paint()
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLPAINT);
	}

	void Regen(WORD wParam = 0)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLREGEN, wParam);
	}

	void RenderOntoDC(PAN_CtlRenderOptions* pOptions)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLRENDERONTODC, 0, (LPARAM) pOptions);
	}

	void HScroll(WORD code, WORD thumbPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLHSCROLL, (WPARAM) code, (LPARAM) thumbPos);
	}

	void VScroll(WORD code, WORD thumbPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLVSCROLL, (WPARAM) code, (LPARAM) thumbPos);
	}

	void Size(WORD width, WORD height)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSIZE, 0, MAKELPARAM(width, height));
	}

	void SetViewExtents(PAN_CtlRange* pRange)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETVIEWEXTENTS, 0, (LPARAM) pRange);
	}

	void GetViewExtents(PAN_CtlRange* pRange)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETVIEWEXTENTS, 0, (LPARAM) pRange);
	}

	void SetOffset(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETOFFSET, 0, (LPARAM) pPos);
	}

	void GetOffset(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETOFFSET, 0, (LPARAM) pPos);
	}

	void SetZoom(WORD flag, DWORD factor)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETZOOM, (WPARAM) flag, (LPARAM) factor);
	}

	DWORD GetZoom(WORD flag)
	{
		ASSERT_VALID(this);
		DWORD factor = 1;
		if (m_hWnd && IsWindow(m_hWnd)) {
			SendMessage( PM_CTLGETZOOM, (WPARAM) flag, (LPARAM) &factor);
		}
		return factor;
	}

	void SetCaretPos(PAN_CtlCaretPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETCARETPOS, 0, (LPARAM) pPos);
	}

	void GetCaretPos(PAN_CtlCaretPos* pCaretPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETCARETPOS, 0, (LPARAM) pCaretPos);
	}

	PAN_CtlPos* CaretToWorld(PAN_CtlCaretPos* pCaretPos)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLCARETTOWORLD, 0, (LPARAM) pCaretPos);
		return (PAN_CtlPos*) result;
	}

	PAN_CtlCaretPos* WorldToCaret(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		LRESULT result =
		SendMessage(PM_CTLWORLDTOCARET, 0, (LPARAM) pPos);
		return (PAN_CtlCaretPos*) result;
	}

	void ClientToWorld(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLCLIENTTOWORLD, 0, (LPARAM) pPos);
	}

	void WorldToClient(PAN_CtlPos* pPos)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLWORLDTOCLIENT, 0, (LPARAM) pPos);
	}

	void SetFgBgColor(BOOL fBgColor, const COLORREF pColor)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETFGBGCOLOR, (WPARAM) fBgColor, (LPARAM) &pColor);
	}
	void SetFgBgColor(BOOL fBgColor, const COLORREF *pColor)
	{
		ASSERT_VALID(this);
		ASSERT(pColor != 0);
		SendMessage(PM_CTLSETFGBGCOLOR, (WPARAM) fBgColor, (LPARAM) pColor);
	}

	COLORREF GetFgBgColor(BOOL fBgColor)
	{
		ASSERT_VALID(this);
		COLORREF	pColor = RGB(192,192,192);
		SendMessage(PM_CTLGETFGBGCOLOR, (WPARAM) fBgColor, (LPARAM) &pColor);
		return(pColor);
	}
	void GetFgBgColor(BOOL fBgColor, COLORREF *pColor)
	{
		ASSERT_VALID(this);
		ASSERT(pColor != 0);
		SendMessage(PM_CTLGETFGBGCOLOR, (WPARAM) fBgColor, (LPARAM) pColor);
	}

	LONG QueryNewPalette(CWnd* pWnd)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLQUERYNEWPALETTE, 0, pWnd ? (LPARAM) (LPVOID)pWnd->GetSafeHwnd() : 0);
		return (LONG) result;
	}

	LONG PaletteChanged(CWnd* pWnd)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLPALETTECHANGED, 0, pWnd ? (LPARAM) (LPVOID)pWnd->GetSafeHwnd() : 0);
		return (LONG) result;
	}

	void SetPalette(LOGPALETTE* pLogPal)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETPALETTE, 0, (LPARAM) pLogPal);
	}

	WORD GetPalette(LOGPALETTE* pLogPal)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETPALETTE, 0, (LPARAM) pLogPal);
		return (WORD) result;
	}

	void SetBaseFont(const LOGFONT* pLogFont)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETBASEFONT, 0, (LPARAM) pLogFont);
	}

	void GetBaseFont(LOGFONT* pLogFont)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETBASEFONT, 0, (LPARAM) pLogFont);
	}

	void GetDims(PAN_CtlDimensions* pDims)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETDIMS, 0, (LPARAM) pDims);
	}

	LONG GetColWidth(WORD units, LONG col)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETCOLWIDTH, (WPARAM) units, (LPARAM) col);
		/*
		**  Workaround for VCET bug: The TWIPS dimensions change with
		**  the zoom factor. Compensate for this:
		*/
		if (units == CTLUNIT_TWIPS) {
			DWORD	dwZoom = GetZoom(PAN_CTLZOOMBOTH);
			if (dwZoom) {
				result = (LRESULT)(((Real)result * 100000.0) / (Real)dwZoom);
			}
		}
		return (LONG) result;
	}

	LONG GetRowHeight(WORD units, LONG row)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETROWHEIGHT, (WPARAM) units, (LPARAM) row);
		/*
		**  Workaround for VCET bug: The TWIPS dimensions change with
		**  the zoom factor. Compensate for this:
		*/
		if (units == CTLUNIT_TWIPS) {
			DWORD	dwZoom = GetZoom(PAN_CTLZOOMBOTH);
			if (dwZoom) {
				result = (LRESULT)(((Real)result * 100000.0) / (Real)dwZoom);
			}
		}
		return (LONG) result;
	}

	int	GetResourceInfoState(LPPAN_RESOURCEINFO lpResourceInfo, int nResourceInfoEntries)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETRESOURCEINFOSTATE, (WPARAM) nResourceInfoEntries, (LPARAM) lpResourceInfo);
		return (int) result;
	}

	void SetLayerState(PAN_LAYER* pLayers, WORD nLayers)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETLAYERSTATE, (WPARAM) nLayers, (LPARAM) pLayers);
	}

	WORD GetLayerState(PAN_LAYER* pLayers, WORD nLayers)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETLAYERSTATE, (WPARAM) nLayers, (LPARAM) pLayers);
		return (WORD) result;
	}

	void SetXRefState(PAN_XREF* pXRefs, WORD nXRefs)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETXREFSTATE, (WPARAM) nXRefs, (LPARAM) pXRefs);
	}

	WORD GetXRefState(PAN_XREF* pXRefs, WORD nXRefs)	
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETXREFSTATE, (WPARAM) nXRefs, (LPARAM) pXRefs);
		return (WORD) result;
	}

	WORD GetBlockNames(PAN_BLOCK* pBlocks, WORD nBlocks)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETBLOCKNAMES, (WPARAM) nBlocks, (LPARAM) pBlocks);
		return (WORD) result;
	}

	void SetBlock(WORD iBlock)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETBLOCK, (WPARAM) iBlock);
	}

	LONG GetBlock()
	{
		ASSERT_VALID(this);
		LONG iBlock;
		SendMessage(PM_CTLGETBLOCK, 0, (LPARAM) &iBlock);
		return iBlock;
	}

	WORD GetViewNames(PAN_VIEW* pViews, WORD nViews)
	{
		ASSERT_VALID(this);
		LRESULT result = SendMessage(PM_CTLGETVIEWNAMES, (WPARAM) nViews, (LPARAM) pViews);
		return (WORD) result;
	}

	void SetView(WORD iView, PAN_VIEW* pView)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETVIEW, (WPARAM) iView, (LPARAM) pView);
	}

	void GetView(PAN_VIEW* pView)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETVIEW, 0, (LPARAM) pView);
	}

	void Rotate(WORD nDegrees)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLROTATE, (WPARAM) nDegrees);
		m_wRotate = nDegrees;
	}

	void Flip(WORD flag)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLFLIP, (WPARAM) flag);
		m_wFlip = flag;
	}

	void SetImageEx(WORD flag, LONG value)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETIMAGEEX, (WPARAM) flag, (LPARAM) value);
	}

	LONG GetImageEx(WORD flag)
	{
		ASSERT_VALID(this);
		LONG value;
		SendMessage(PM_CTLGETIMAGEEX, (WPARAM) flag, (LPARAM) &value);
		return value;
	}

	void Sort(PAN_CtlSortInfo* pInfo)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSORT, 0, (LPARAM) pInfo);
	}

	void Search(PAN_CtlSearchInfo* pInfo)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSEARCH, 0, (LPARAM) pInfo);
	}

	void GetString(WPARAM wParam, LPARAM lParam)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETSTRING, wParam, lParam);
	}

	void GetClpbrdFmts(PAN_CtlClpbrdFmtList* pList)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETCLPBRDFMTS, 0, (LPARAM) pList);
	}

	WORD GetNumSels()
	{
		ASSERT_VALID(this);
		WORD nSels;
		SendMessage(PM_CTLGETNUMSELS, 0, (LPARAM) &nSels);
		return nSels;
	}

	void SetSel(BOOL fSel, PAN_CtlCaretRange* pRange)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETSEL, (WPARAM) fSel, (LPARAM) pRange);
	}

	void SetSelCaret(PAN_CtlCaretRange* pRange)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLSETSELCARET, 0, (LPARAM) pRange);
	}

	void GetSels(PAN_CtlSelList* pList)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLGETSELS, 0, (LPARAM) pList);
	}

	void ClearSels()
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLCLEARSELS);
	}

	void Copy(DWORD format)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLCOPY, 0, (LPARAM) format);
	}

	void ValidateMargins(PAN_CtlPrintOptions* pOptions)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLVALIDATEMARGINS, 0, (LPARAM) pOptions);
	}

	void PrintPreview(PAN_CtlPrintOptions* pOptions)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLPRINTPREVIEW, 0, (LPARAM) pOptions);
	}

	void Print(WORD pageNum, PAN_CtlPrintOptions* pOptions)
	{
		ASSERT_VALID(this);
		SendMessage(PM_CTLPRINT, (WPARAM) pageNum, (LPARAM) pOptions);
	}

	FARPROC Convert(WORD procType)
	{
		ASSERT_VALID(this);
		FARPROC pfn;
		SendMessage(PM_CTLCONVERT, (WPARAM) procType, (LPARAM) &pfn);
		return pfn;
	}

	void	RotateFlipRange(PAN_CtlRange FAR *pRect, BOOL bUndo)
	{
		ASSERT_VALID(this);
		SendMessage(PM_XFRMRECT, (WPARAM) (bUndo ? XFRM_FNC_RECIPROCAL : XFRM_FNC), (LPARAM) pRect);
	}

	// Useful functions
	DWORD EnableCaps(DWORD caps, BOOL fEnable)
	{
		// Add or remove the given caps
		DWORD	dwCaps	=  GetCaps(FALSE);
		DWORD	ret = dwCaps;
		if (fEnable) {
			dwCaps |=  caps;
		} else {
			dwCaps &= ~caps;
		}
		SetCaps(FALSE, dwCaps);
		return ret;		// return original caps
	}
	DWORD EnableModes(DWORD modes, BOOL fEnable)
	{
		// Add or remove the given modes
		DWORD	dwModes	=  GetMode(FALSE);
		DWORD	ret = dwModes;
		if (fEnable) {
			dwModes |=  modes;
		} else {
			dwModes &= ~modes;
		}
		SetMode(FALSE, dwModes);
		return ret;		// return original modes
	}
	BOOL	IsXAxisRight()
	{
		PAN_CtlRange	rcPan;
		_fmemset(&rcPan, 0, sizeof(rcPan));
		GetPageSize(&rcPan);
		WorldToClient(&rcPan);

		return (rcPan.min.x <= rcPan.max.x) ? TRUE : FALSE;
	}
	BOOL	IsYAxisDown()
	{
		PAN_CtlRange rcPan;
		_fmemset(&rcPan, 0, sizeof(rcPan));
		GetPageSize(&rcPan);
		WorldToClient(&rcPan);

		return (rcPan.min.y <= rcPan.max.y) ? TRUE : FALSE;
	}

	BOOL	IsDead()
	{
		return m_fLogicallyDestroyed;
	}

	void	ClientToWorld(PAN_CtlRange *pRect)
	{
		ClientToWorld(&pRect->min);
		ClientToWorld(&pRect->max);
	}

	void	WorldToClient(PAN_CtlRange *pRect)
	{
		WorldToClient(&pRect->min);
		WorldToClient(&pRect->max);
	}	

	// The Controls have no GetRotate and GetFlip messages, so do it ourselves
	WORD	GetRotate()	{return m_wRotate;}
	WORD	GetFlip()	{return m_wFlip;}

private:
	WORD	m_wRotate;
	WORD	m_wFlip;
	WNDPROC	m_lpfnNotifyProc;
	FARPROC	m_lpfnEventProc;

public:
#ifdef _DEBUG
	virtual void AssertValid() const;
#endif
};

/*
**  C++  Encapsulation of PAN_CtlRange
*/
class CPAN_CtlRange : public tagPAN_Rect
{
public:

//	Constructors
	CPAN_CtlRange()		{memset(this, 0, sizeof(*this));}
	CPAN_CtlRange(const PAN_CtlRange &srcRect)	{*this = srcRect;}

	// Attributes (in addition to RECT members)
	Real	Width()  const	{return this->max.x - this->min.x;}
	Real	Height() const	{return this->max.y - this->min.y;}
	void	Expand(Real fact)
	{
		// Scales the rect, keeping the the min point constant;
		Real	w = Width(),
				h = Height();
		this->max.x = this->min.x + fact*w;
		this->max.y = this->min.y + fact*h;
	}

	// convert to PAN_CtlRange (no need for &)
 	operator PAN_CtlRange *() const		{return (PAN_CtlRange *)this;}

	// convert to CRect
	operator CRect() const
	{
		return CRect((int)this->min.x, (int)this->min.y, (int)this->max.x, (int)this->max.y);
	}

	// Equate a PAN_CtlRange
	void operator=(const PAN_CtlRange &srcRect)	{*this = srcRect;}

	// Equate a RECT
	void operator=(const RECT &srcRect)
	{
		this->min.x = srcRect.left;
		this->min.y = srcRect.top;
		this->min.z = 0.0;
		this->max.x = srcRect.right;
		this->max.y = srcRect.bottom;
		this->max.z = 0.0;
	}

	// Equate a CRect
	void operator=(const CRect &srcRect)
	{
		this->min.x = srcRect.left;
		this->min.y = srcRect.top;
		this->min.z = 0.0;
		this->max.x = srcRect.right;
		this->max.y = srcRect.bottom;
		this->max.z = 0.0;
	}

	void operator*=(const Real fact)
	{
		// Scales about the center
		Real	w = Width(),
				h = Height(),
				midx = this->min.x + 0.5 * w,
				midy = this->min.y + 0.5 * h;
		this->min.x = midx - 0.5*fact*w;
		this->max.x = midx + 0.5*fact*w;
		this->min.y = midy - 0.5*fact*h;
		this->max.y = midy + 0.5*fact*h;
	}
};



#endif

//	vcet.h
